/*********************************************************************
*
* Copyright 2017-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  stats_example.c
*
* @purpose   Routing ARP Statistics APIs Example.
*
* @component OPEN
*
* @comments
*
* @create    1/20/2017
*
* @end
*
**********************************************************************/
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <stdlib.h>
#include <ctype.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <getopt.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_routing_stats.h"

const char *protocolToName(uint32_t proto, OPEN_AF_t family)
{
  typedef struct protomap_s
  {
    uint32_t protocol;
    char *name;
  } protomap_t;

  static protomap_t ipv4map[] = { 
    { OPEN_RTO_LOCAL, "LOCAL" }, 
    { OPEN_RTO_KERNEL, "KERNEL" },
    { OPEN_RTO_STATIC, "STATIC" },
    { OPEN_RTO_MPLS, "MPLS" },
    { OPEN_RTO_OSPF_INTRA_AREA, "OSPF_INTRA_AREA" },
    { OPEN_RTO_OSPF_INTER_AREA, "OSPF_INTER_AREA" },
    { OPEN_RTO_OSPF_TYPE1_EXT, "OSPF_TYPE1_EXT" },
    { OPEN_RTO_OSPF_TYPE2_EXT, "OSPF_TYPE2_EXT" },
    { OPEN_RTO_OSPF_NSSA_TYPE1_EXT, "OSPF_NSSA_TYPE1_EXT" },
    { OPEN_RTO_OSPF_NSSA_TYPE2_EXT, "OSPF_NSSA_TYPE2_EXT" },
    { OPEN_RTO_OSPFV3_INTRA_AREA, "OSPFV3_INTRA_AREA" },
    { OPEN_RTO_OSPFV3_INTER_AREA, "OSPFV3_INTER_AREA" },
    { OPEN_RTO_OSPFV3_TYPE1_EXT, "OSPFV3_TYPE1_EXT" },
    { OPEN_RTO_OSPFV3_TYPE2_EXT, "OSPFV3_TYPE2_EXT" },
    { OPEN_RTO_OSPFV3_NSSA_TYPE1_EXT, "OSPFV3_NSSA_TYPE1_EXT" },
    { OPEN_RTO_OSPFV3_NSSA_TYPE2_EXT, "OSPFV3_NSSA_TYPE2_EXT" },
    { OPEN_RTO_RIP, "RIP" },
    { OPEN_RTO_DEFAULT, "DEFAULT" },
    { OPEN_RTO_IBGP, "IBGP" },
    { OPEN_RTO_EBGP, "EBGP" },
    { OPEN_RTO_LBGP, "LBGP" },
    { OPEN_RTO_EXTERNAL, "EXTERNAL" },
    { OPEN_RTO_UNNUMBERED, "UNNUMBERED" },
    { OPEN_RTO_NET_PROTOTYPE, "NET_PROTOTYPE" }
  };

  static protomap_t ipv6map[] = {
    { OPEN_RTO6_LOCAL, "LOCAL" },
    { OPEN_RTO6_KERNEL, "KERNEL" },
    { OPEN_RTO6_STATIC, "STATIC" },
    { OPEN_RTO6_MPLS, "MPLS" },
    { OPEN_RTO6_OSPFV3_INTRA_AREA, "OSPFV3_INTRA_AREA" },
    { OPEN_RTO6_OSPFV3_INTER_AREA, "OSPFV3_INTER_AREA" },
    { OPEN_RTO6_OSPFV3_TYPE1_EXT, "OSPFV3_TYPE1_EXT" },
    { OPEN_RTO6_OSPFV3_TYPE2_EXT, "OSPFV3_TYPE2_EXT" },
    { OPEN_RTO6_OSPFV3_NSSA_TYPE1_EXT, "OSPFV3_NSSA_TYPE1_EXT" },
    { OPEN_RTO6_OSPFV3_NSSA_TYPE2_EXT, "OSPFV3_NSSA_TYPE2_EXT" },
    { OPEN_RTO6_RIP, "RIP)" },
    { OPEN_RTO6_DEFAULT, "DEFAULT" },
    { OPEN_RTO6_IBGP, "IBGP" },
    { OPEN_RTO6_EBGP, "EBGP" },
    { OPEN_RTO6_LBGP, "LBGP" },
    { OPEN_RTO6_EXTERNAL, "EXTERNAL" },
    { OPEN_RTO6_6TO4, "6TO4" },
    { OPEN_RTO6_NET_PROTOTYPE, "NET_PROTOTYPE" }
  };

  protomap_t *p;
  char *name = (char *) NULL;
  int size;
  int i;

  if (family == OPEN_AF_INET) 
  {
    p = ipv4map;
    size = sizeof(ipv4map) / sizeof(protomap_t);
  } 
  else
  {
    p = ipv6map;
    size = sizeof(ipv6map) / sizeof(protomap_t);
  }

  for (i = 0; i < size; i++)
  {
    if (p->protocol == proto)
    {
      name = p->name;
      break;
    }
    p++;
  }
  return(name);
}
  

void dumpStats(const openRouteStats_t *stats, open_buffdesc *numRoutes, open_buffdesc *ecmpHisto, open_buffdesc *prefixesNo, open_buffdesc *ecmpCounts, const bool ipv4)
{
  uint32_t *p = numRoutes->pstart;
  uint32_t total, totalBgp=0, totalOspf=0;
  uint32_t count;
	uint32_t ct1, ct2, ct3, ct4;
  uint32_t *histo;
	bool bgpPresent=false, ospfPresent=false;
  int index;
  int i;
  openEcmpRouteProtocolCount_t *pEcmpCount;

  total = 0;
  index = ipv4 ? OPEN_RTO_LOCAL : OPEN_RTO6_LOCAL;
  if (stats->mask & (1 << index))
  {
    count = p[index];
    total += count;
    printf("Connected Routes............................... %u\n", count);
  }
  index = ipv4 ? OPEN_RTO_STATIC : OPEN_RTO6_STATIC;
  if (stats->mask & (1 << index))
  {
    count = p[index];
    total += count;
    printf("Static Routes.................................. %u\n", count);
  }
  index = ipv4 ? OPEN_RTO_KERNEL : OPEN_RTO6_KERNEL;
  if (stats->mask & (1 << index))
  {
    count = p[index];
    total += count;
    printf("Kernel Routes.................................. %u\n", count);
  }
  if (ipv4 == false && stats->mask & (1 << OPEN_RTO6_6TO4))
  {
    count = p[OPEN_RTO6_6TO4];
    total += count;
    printf("6To4 Routes.................................... %u\n", count);
  }
  if (ipv4 && stats->mask & (1 << OPEN_RTO_UNNUMBERED))
  {
    count = p[OPEN_RTO_UNNUMBERED];
    total += count;
    printf("Unnumbered Peer Routes......................... %u\n", count);
  }
	ct1 = ct2 = ct3 = 0;
  index = ipv4 ? OPEN_RTO_EBGP : OPEN_RTO6_EBGP;
  if (stats->mask & (1 << index))
  {
    ct1 = count = p[index];
    total += count;
		totalBgp += ct1;
		bgpPresent = true;
  }
  index = ipv4 ? OPEN_RTO_IBGP : OPEN_RTO6_IBGP;
  if (stats->mask & (1 << index))
  {
    ct2 = count = p[index];
    total += count;
		totalBgp += ct2;
		bgpPresent = true;
  }
  index = ipv4 ? OPEN_RTO_LBGP : OPEN_RTO6_LBGP;
  if (stats->mask & (1 << index))
  {
    ct3 = count = p[index];
    total += count;
		totalBgp += ct3;
		bgpPresent = true;
  }
	if (bgpPresent == true)
	{
		printf("BGP Routes..................................... %u\n", totalBgp);
		printf("  BGP External................................. %u\n", ct1);
		printf("  BGP Internal................................. %u\n", ct2);
		printf("  BGP Local.................................... %u\n", ct3);
	}
	ct1 = ct2 = ct3 = ct4 = 0;
  index = ipv4 ? OPEN_RTO_OSPF_INTRA_AREA : OPEN_RTO6_OSPFV3_INTRA_AREA;
  if (stats->mask & (1 << index))
  {
    ct1 = count = p[index];
    total += count;
		totalOspf += ct1;
		ospfPresent = true;
  }
  index = ipv4 ? OPEN_RTO_OSPF_INTER_AREA : OPEN_RTO6_OSPFV3_INTER_AREA;
  if (stats->mask & (1 << index))
  {
    ct2 = count = p[index];
    total += count;
		totalOspf += ct2;
		ospfPresent = true;
  }
  index = ipv4 ? OPEN_RTO_OSPF_TYPE1_EXT : OPEN_RTO6_OSPFV3_TYPE1_EXT;
  if (stats->mask & (1 << index))
  {
    ct3 = count = p[index];
    total += count;
		totalOspf += ct3;
		ospfPresent = true;
  }
  index = ipv4 ? OPEN_RTO_OSPF_TYPE2_EXT : OPEN_RTO6_OSPFV3_NSSA_TYPE2_EXT;
  if (stats->mask & (1 << index))
  {
    ct4 = count = p[index];
    total += count;
		totalOspf += ct4;
		ospfPresent = true;
  }
	if (ospfPresent == true)
	{
		printf("OSPF Routes.................................... %u\n", totalOspf);
		printf("  OSPF Intra Area Routes....................... %u\n", ct1);
		printf("  OSPF Inter Area Routes....................... %u\n", ct2);
		printf("  OSPF External Type-1 Routes.................. %u\n", ct3);
		printf("  OSPF External Type-2 Routes.................. %u\n", ct4);
	}
  if (stats->rejectRoutesValid == OPEN_TRUE)
  {
    printf("Reject Routes.................................. %u\n", stats->rejectRoutes);
  }
  index = ipv4 ? OPEN_RTO_NET_PROTOTYPE : OPEN_RTO6_NET_PROTOTYPE;
  if (stats->mask & (1 << index))
  {
    count = p[index];
    total += count;
    printf("Net Prototype Routes........................... %u\n", count);
  }
  printf("Total routes................................... %u\n", total);
  printf("\n");
  printf("Best Routes (High)............................. %u (%u)\n",
          stats->bestRoutes, stats->bestRoutesHigh);
  printf("Alternate Routes............................... %u\n", 
          stats->altRoutes);
  if (stats->leakedRoutesValid == OPEN_TRUE)
  {
    printf("Leaked Routes.................................. %u\n", 
            stats->leakedRoutes);
  }
  if (stats->mplsRoutesValid == OPEN_TRUE)
  {
    printf("MPLS Routes.................................... %u\n", 
           stats->mplsRoutes);
  }
  if (stats->rfc5549RoutesValid)
  {
    printf("RFC5549 Routes - IPv4 with IPv6 nexthop........ %u\n", 
           stats->rfc5549Routes);
  }
  printf("Route Adds..................................... %u\n", stats->adds);
  printf("Route Modifies................................. %u\n", stats->mods);
  printf("Route Deletes.................................. %u\n", stats->dels);
  printf("Unresolved Route Adds.......................... %u\n", stats->unresAdds);
  printf("Invalid Route Adds............................. %u\n", stats->invalidAdds);
  printf("Failed Route Adds.............................. %u\n", stats->failedAdds);
  printf("Failed Kernel Route Adds....................... %u\n", stats->kernelFailedAdds);
  printf("Hardware Failed Route Adds..................... %u\n", stats->hwFailureRoutes);
  printf("Reserved Locals................................ %u\n", stats->resvLocals);
  printf("\n");
  printf("Unique Next Hops (High)........................ %u (%u)\n", 
         stats->nextHops, stats->nextHopsHigh);
  printf("Next Hop Groups (High)......................... %u (%u)\n", 
         stats->nextHopGroups, stats->nhGroupsHigh);
  printf("ECMP Groups (High)............................. %u (%u)\n",
         stats->ecmpGroups, stats->ecmpGroupsHigh);
  printf("ECMP Routes.................................... %u\n", stats->ecmpRoutes);
  printf("Truncated ECMP Routes.......................... %u\n", stats->ecmpTruncs);
  printf("ECMP Retries................................... %u\n", stats->ecmpRetries);
  histo = (uint32_t *) ecmpHisto->pstart;
  for (i = 0; i < ecmpHisto->size / sizeof(uint32_t); i++)
  {
    if (histo[i] != 0)
    {
      printf("Routes with %d Next Hop%s......................... %u\n", 
              i, (i != 1) ? "s" : "", histo[i]);
    }
  }
  pEcmpCount = (openEcmpRouteProtocolCount_t *) ecmpCounts->pstart;

  for (i = 0; i < ecmpCounts->size / sizeof(*pEcmpCount); i++)
  {
    const char *p = protocolToName(pEcmpCount->protocol, pEcmpCount->family);
    if (p != (char *) NULL) 
    {
      printf("Protocol %s number of %s ECMP routes................ %u\n", 
              p, pEcmpCount->family == OPEN_AF_INET ? "IPv4" : "IPv6", pEcmpCount->numEcmpRoutes);
    }
    pEcmpCount++;
  }
}

void cleanupRouteStats(open_buffdesc *nRoutes, open_buffdesc *ecmp, open_buffdesc *prefixes, open_buffdesc *ecmpCounts)
{
  if (nRoutes && nRoutes->pstart)
  {
    free(nRoutes->pstart);
    nRoutes->pstart = 0;
  }
  if (ecmp && ecmp->pstart)
  {
    free(ecmp->pstart);
    ecmp->pstart = 0;
  }
  if (prefixes && prefixes->pstart)
  {
    free(prefixes->pstart);
    prefixes->pstart = 0;
  }
  if (ecmpCounts && ecmpCounts->pstart)
  {
    free(ecmpCounts->pstart);
    ecmpCounts->pstart = 0;
  }
}

bool initRouteStats(openapiClientHandle_t *client_handle, openRouteStats_t *p, open_buffdesc *nRoutes, open_buffdesc *ecmp, open_buffdesc *prefixes, open_buffdesc *ecmpCounts, bool ipv4)
{
  open_error_t rc;
  uint32_t count;

  memset(p, 0, sizeof(openRouteStats_t));
  if (ipv4 == true)
  {
    rc = openapiIpv4NumRoutingProtocolsGet(client_handle, &count);
    if (rc != OPEN_E_NONE)
    {
      printf("%s openapiIpv4NumRoutingProtocolsGet failed (%d)\n", __FUNCTION__, rc);
      return false;
    }
  }
  else
  {
    rc = openapiIpv6NumRoutingProtocolsGet(client_handle, &count);
    if (rc != OPEN_E_NONE)
    {
      printf("%s openapiIpv6NumRoutingProtocolsGet failed (%d)\n", __FUNCTION__, rc);
      return false;
    }
  }

  nRoutes->pstart = malloc(count * sizeof(uint32_t));
  if (!nRoutes->pstart)
  {
    printf("%s unable to allocate numRoutes\n", __FUNCTION__);
    return false;
  }
  nRoutes->size = count * sizeof(uint32_t);

  rc = openapiNumEcmpRoutesGet(client_handle, &count);
  if (rc != OPEN_E_NONE)
  {
    cleanupRouteStats(ecmp, nRoutes, prefixes, ecmpCounts);
    printf("%s openapiNumEcmpRoutesGet failed (%d)\n", __FUNCTION__, rc);
    return false;
  }

  ecmp->pstart = calloc(count, sizeof(uint32_t));
  if (!ecmp->pstart)
  {
    cleanupRouteStats(ecmp, nRoutes, prefixes, ecmpCounts);
    printf("%s unable to allocate ecmpHisto\n", __FUNCTION__);
    return false;
  }
  ecmp->size = count * sizeof(uint32_t);

  ecmpCounts->pstart = calloc(count, sizeof(openEcmpRouteProtocolCount_t));
  if (!ecmpCounts->pstart)
  {
    cleanupRouteStats(ecmp, nRoutes, prefixes, ecmpCounts);
    printf("%s unable to allocate ecmpHisto\n", __FUNCTION__);
    return false;
  }
  ecmpCounts->size = count * sizeof(openEcmpRouteProtocolCount_t);

  if (ipv4 == false)
  {
    rc = openapiNumIpv6PrefixesGet(client_handle, &count);
    if (rc != OPEN_E_NONE)
    {
      cleanupRouteStats(ecmp, nRoutes, prefixes, ecmpCounts);
      printf("%s openapiNumIpv6PrefixesGet failed (%d)\n", __FUNCTION__, rc);
      return false;
    }

    prefixes->pstart = malloc(count * sizeof(uint32_t));
    if (!prefixes->pstart)
    {
      cleanupRouteStats(ecmp, nRoutes, prefixes, ecmpCounts);
      printf("%s unable to allocate prefixesNo\n", __FUNCTION__);
      return false;
    }
    prefixes->size = count * sizeof(uint32_t);
  }
  return true;
}

/*********************************************************************
* @purpose  Show Dynamic Routes
*
* @param    client_handle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/

void showDynamicRoutes(openapiClientHandle_t *client_handle, uint32_t *numRoutes)
{
  open_buffdesc protoName;
  open_buffdesc routeTypeName;
  uint32_t protoId;
  uint32_t protoTotal;
  uint32_t routeType;
  uint32_t openRoutes = 0;
  uint32_t protoNameLen;
  uint32_t routeTypeNameLen;

  if (openapiIpMapProtoNameLenGet(client_handle, &protoNameLen) != OPEN_E_NONE)
  {
    printf("%s: unable to get protocol name length\n", __FUNCTION__);
    return;
  }

  protoName.pstart = malloc(protoNameLen * sizeof(char));

  if (!protoName.pstart)
  {
    printf("%s: unable to allocated protoName buffer\n", __FUNCTION__);
    return;
  }

  if (openapiIpMapRouteTypeNameLenGet(client_handle, &routeTypeNameLen) != OPEN_E_NONE)
  {
    printf("%s: unable to get route type name length\n", __FUNCTION__);
    free(protoName.pstart);
    return;
  }

  routeTypeName.pstart = malloc(routeTypeNameLen * sizeof(char));

  if (!routeTypeName.pstart)
  {
    printf("%s: unable to allocated routeTypename buffer\n", __FUNCTION__);
    free(protoName.pstart);
    return;
  }

  protoId = 0;
  while (openapiIpMapDynamicProtoNextGet(client_handle, &protoId) == OPEN_E_NONE)
  {
    protoTotal = 0;

    /* First get the total number of routes for this protocol */
    routeType = 0;
    while (openapiIpMapProtoRouteTypeNextGet(client_handle, protoId, &routeType) == OPEN_E_NONE)
    {
      protoTotal += numRoutes[routeType];
      openRoutes += numRoutes[routeType];
    }
    protoName.size = protoNameLen;
    if (openapiIpMapProtoNameGet(client_handle, protoId, &protoName) == OPEN_E_NONE)
    {
      printf("Protocol: %s\n", (char*) protoName.pstart);

      /* Now print count for each route type within this protocol */
      routeType = 0;
      while (openapiIpMapProtoRouteTypeNextGet(client_handle, protoId, &routeType) == OPEN_E_NONE)
      {
        routeTypeName.size = routeTypeNameLen;
        if (openapiIpMapRouteTypeNameGet(client_handle, routeType, &routeTypeName) == OPEN_E_NONE)
        {
          printf("%s: %u\n", (char *) routeTypeName.pstart, numRoutes[routeType]);
        }
      }
    }
  }
  free(routeTypeName.pstart);
  free(protoName.pstart);
}

/*********************************************************************
* @purpose  Emulate cli "show ip route summary" 
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    name             @b{(input)}   vrf name, or OPEN_DEFAULT_VRF_NAME
* @param    bestRoutes       @b{(input)}   if OPEN_TRUE, only best routes
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/

void showIPv4RouteSummary(openapiClientHandle_t *client_handle, const char *name, const OPEN_BOOL_t bestRoutes)
{
  open_error_t rc;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1];
  open_buffdesc vrfName;
  open_buffdesc numRoutes;
  open_buffdesc ecmpHisto;
  open_buffdesc ecmpCounts;
  openRouteStats_t routeStats;
  bool ret;
  
  if (strlen(name) > OPEN_VRF_MAX_NAME_LEN)
  {
    printf("VRF name %s is too long\n", name);
    return;
  }

  ret = initRouteStats(client_handle, &routeStats, &numRoutes, &ecmpHisto, 0, &ecmpCounts, true);
  if (ret == false)
  {
    return;
  }

  strncpy(vrfNameStr, name, OPEN_VRF_MAX_NAME_LEN);
  vrfName.pstart = vrfNameStr;
  vrfName.size = sizeof(vrfNameStr) - 1;

  rc = openapiIpv4RouteTableStatsGet(client_handle, &vrfName, bestRoutes, &routeStats, &numRoutes, &ecmpHisto);

  if (rc == OPEN_E_NONE)
  {
    rc = openapiIpv4EcmpRouteProtocolCountsGet(client_handle, &vrfName, &ecmpCounts);
  }
  else
  {
    printf("%s openapiIpv4EcmpRouteProtocolCountsGet failed (%d)\n", __FUNCTION__, rc);
  }
  if (rc == OPEN_E_NONE)
  {
    dumpStats(&routeStats, &numRoutes, &ecmpHisto, 0, &ecmpCounts, true);
    showDynamicRoutes(client_handle, numRoutes.pstart);
  }
  else
  {
    printf("%s openapiIpv4RouteTableStatsGet failed (%d)\n", __FUNCTION__, rc);
  }
  cleanupRouteStats(&numRoutes, &ecmpHisto, 0, &ecmpCounts);
}

/*********************************************************************
* @purpose  Emulate cli "show ipv6 route summary"
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    bestRoutes       @b{(input)}   if OPEN_TRUE, only best routes
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/

void showIPv6RouteSummary(openapiClientHandle_t *client_handle, const OPEN_BOOL_t bestRoutes)
{
  open_error_t rc;
  open_buffdesc numRoutes;
  open_buffdesc ecmpHisto;
  open_buffdesc prefixesNo;
  open_buffdesc ecmpCounts;
  openRouteStats_t routeStats;
  bool ret;

  ret = initRouteStats(client_handle, &routeStats, &numRoutes, &ecmpHisto, &prefixesNo, &ecmpCounts, false);

  if (ret == false)
  {
    return;
  }

  rc = openapiIpv6RouteTableStatsGet(client_handle, bestRoutes, &routeStats, &numRoutes, &ecmpHisto, &prefixesNo);
  if (rc == OPEN_E_NONE)
  {
    rc = openapiIpv6EcmpRouteProtocolCountsGet(client_handle, &ecmpCounts);
  }
  else
  {
    printf("%s openapiIpv6EcmpRouteProtocolCountsGet failed (%d)\n", __FUNCTION__, rc);
  }
  if (rc == OPEN_E_NONE)
  {
    dumpStats(&routeStats, &numRoutes, &ecmpHisto, &prefixesNo, &ecmpCounts, false);
  }
  else
  {
    printf("%s openapiIpv6RouteTableStatsGet failed (%d)\n", __FUNCTION__, rc);
  }
  cleanupRouteStats(&numRoutes, &ecmpHisto, &prefixesNo, &ecmpCounts);
}

void print_usage(const char *name)
{
  printf("%s [-b][-v vrfname]\n", name);
  exit(0);
}

int main(int argc, char *argv[])
{
  openapiClientHandle_t clientHandle;
  char vrfName[OPEN_VRF_MAX_NAME_LEN + 1];
  int option = 0;
  OPEN_BOOL_t bestRoutes = OPEN_FALSE;
  int rc;

  vrfName[0] = '\0';

  while ((option = getopt(argc, argv,"bv:")) != -1) {
    switch (option) {
      case 'v' : 
        strncpy(vrfName, optarg, sizeof(vrfName));
        break;
      case 'b' : 
        bestRoutes = OPEN_TRUE;
        break;
      default: 
        print_usage(argv[0]); 
        break;
    }
  }

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((rc = openapiClientRegister("routing_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", rc);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  printf("show ip route summary\n\n");
  showIPv4RouteSummary(&clientHandle, vrfName, bestRoutes);
  printf("\n\nshow ipv6 route summary\n\n");
  showIPv6RouteSummary(&clientHandle, bestRoutes);
  
  return 0;
}


