
/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  snmp_example.c
*
* @purpose   Simple Network Management Protocol (SNMP) Example
*
* @component OPEN
*
* @comments
*
* @create    09/23/2013
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_snmp.h"

#define PRINTBADRESULT(result, msg, feat) \
if (result==OPEN_E_UNAVAIL) { printf("Feature not supported - %s.\n", msg); } \
else if (result==OPEN_E_NOT_FOUND) { printf("Test Skipped (not found) - %s %s.\n", msg, feat); } \
else if (result!=OPEN_E_NONE) { printf("Test Failure - %s (err %d).\n", msg, result); }

bool fVerbose = true;

/*******************************************************************
*
* @brief  This function prints the SNMP Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printSNMPAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Sets the system name: %s 1 <sysName>\n", name);
  printf("Test 2: Gets the system name: %s 2\n", name);
  printf("Test 3: Sets the system location: %s 3 <sysLocation>\n", name);
  printf("Test 4: Gets the system location: %s 4\n", name);
  printf("Test 5: Sets the system contact: %s 5 <sysContact>\n", name);
  printf("Test 6: Gets the system contact: %s 6\n", name);
  printf("Test 7: Sets the SNMP source interface: %s 7 <intf>\n", name);
  printf("Test 8: Gets the SNMP source interface: %s 8\n", name);
  printf("Test 9: Sets the SNMP trap flag mode: %s 9 <trapFlag> <mode>\n", name);
  printf("Test 10: Gets the SNMP trap flag mode: %s 10 <trapFlag>\n", name);
  printf("Test 11: Sets the SNMP Engine ID: %s 11 <engineId>\n", name);
  printf("Test 12: Gets the SNMP Engine ID: %s 12\n", name);
  printf("Test 13: Deletes the SNMP Engine ID: %s 13\n", name);
  printf("Test 14: Creates the SNMP user: %s 14 <userName> <groupName> <authProto> "
         "<authPwd> <authKey> <privProto> <privPwd> <privKey> <engineId>\n", name);
  printf("Test 15: Gets the SNMP user details configured for a given user name: %s 15 <userName>\n", name);
  printf("Test 16: Gets all the SNMP users configured: %s 16\n", name);
  printf("Test 17: Deletes the SNMP user: %s 17 <userName> <engineId>\n", name);
  printf("Test 18: Creates the SNMP group: %s 18 <groupName> <securityModel> "
         "<securityLevel> <contextPrefix> <readView> <writeView> <notifyView>\n", name);
  printf("Test 19: Gets the SNMP group details for a given group: %s 19 <groupName>\n", name);
  printf("Test 20: Gets all the SNMP groups configured: %s 20\n", name);
  printf("Test 21: Deletes the SNMP group: %s 21 <groupName> <securityModel> <securityLevel> <contextPrefix>\n", name);
  printf("Test 22: Creates the SNMP filter: %s 22 <filterName> <oidTree> <filterType>\n", name);
  printf("Test 23: Gets the SNMP filter details for a given filter: %s 23 <filterName>\n", name);
  printf("Test 24: Gets all the SNMP filters configured: %s 24\n", name);
  printf("Test 25: Deletes the SNMP filter: %s 25 <filterName> <oidTree>\n", name);
  printf("Test 26: Creates the SNMP view: %s 26 <viewName> <oidTree> <viewType>\n", name);
  printf("Test 27: Gets the SNMP view details for a given view: %s 27 <viewName>\n", name);
  printf("Test 28: Gets all the SNMP views configured: %s 28\n", name);
  printf("Test 29: Deletes the SNMP view: %s 29 <viewName> <oidTree>\n", name);
  printf("Test 30: Creates the SNMP host: %s 30 <host> <serverPort> <securityName> "
         "<securityModel> <securityLevel> <notifyType> <timeout> <retries> <filter>\n", name);
  printf("Test 31: Gets the SNMP host details for a given host: %s 31 <host>\n", name);
  printf("Test 32: Gets all the SNMP hosts configured: %s 32\n", name);
  printf("Test 33: Deletes the SNMP host: %s 33 <host> <notifyType>\n", name);
  printf("Test 34: Creates the SNMP community: %s 34 <communityName> <accessType> <viewName> <addr>\n", name);
  printf("Test 35: Creates the SNMP community along with group: %s 35 <communityName> <groupName> <addr>\n", name);
  printf("Test 36: Gets the SNMP community details for a given community: %s 36 <communityName>\n", name);
  printf("Test 37: Gets all the SNMP communities configured: %s 37\n", name);
  printf("Test 38: Deletes the SNMP community: %s 38 <communityName>\n", name);

  /* Non CONF API Examples */
  printf("Test 39: Creates the SNMP community with IP address and mask: %s 39 <communityName> "
          "<accessType> <ipAddr> <ipMask>\n", name);
  printf("Test 40: Creates the SNMP community with IP address, mask and viewName: %s 40 <communityName> "
         "<accessType> <ipAddr> <ipMask> <viewName>\n", name);
  printf("Test 41: Creates the SNMP community with IPv6 address and ipv6PrefixLen: %s 41 <communityName> "
         "<accessType> <ipAddr> <ipv6PrefixLen>\n", name);
  printf("Test 42: Gets the SNMP community details with IP address, IP mask and viewName for a given community: %s 42 <communityName>\n", name);
  printf("Test 43: Gets the next SNMP community details with IP address for the given community : %s 43 <communityName>\n", name);
  printf("Test 44: Deletes the SNMP community and IP Address: %s 44 <communityName>\n", name);

  printf("Test 45: Creates the SNMP Trap configuration for Non CONF API: %s 45 <communityName> <ipAddr> <version>\n", name);
  printf("Test 46: Gets the SNMP trap entry for a given community and IP address: %s 46 <communityName> <ipAddr>\n", name);
  printf("Test 47: Gets the next SNMP trap entry configured: %s 47 <communityName> <ipAddr>\n", name);
  printf("Test 48: Deletes the SNMP trap entry: %s 48 <communityName> <ipAddr>\n", name);
  /*Non CONF API examples end */

  printf("Test 49: Run API sanity checks: %s 49\n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSnmpSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc zeroLenBuf, badBufdescPointer;
  open_buffdesc communityDesc, nextCommunityDesc, groupDesc, viewDesc;
  open_buffdesc filterDesc, oidDesc, nextFilterDesc, nextOidDesc;
  open_buffdesc contextDesc, readDesc, writeDesc, notifyDesc, nextGroupDesc, nextContextDesc;
  open_buffdesc userName, engineId, nextUserName, nextEngineId, authPwd;
  open_buffdesc authKey, privPwd, privKey, host, securityName, nextHost, nextOidTree;
  open_buffdesc sysName, nextViewName, sysLocation, sysContact, oidString, oidVal;
  open_inet_addr_t ipAddr;
  char buffer[128] = {0}, ipStr[128] = {0}, contextStr[30+1] = {0}, readStr[30+1] = {0};
  char writeStr[30+1] = {0}, notifyStr[30+1] = {0}, nextGroupStr[30+1], nextContextStr[30+1];
  char communityStr[20+1] ={0}, nextCommunityStr[20+1] ={0}, viewStr[30+1] = {0}, nextViewStr[30+1] = {0};
  char groupStr[30+1] = {0}, filterStr[30+1] = {0}, oidStr[128+1] = {0};
  char userNameStr[30+1] = {0}, engineStr[32], nextUserNameStr[30+1], nextEngineStr[32];
  char nextOidStr[128+1] = {0}, nextFilterStr[30+1] = {0};
  char aPwdStr[32+1] = {0}, aKeyStr[48+1] = {0}, pPwdStr[32+1] = {0}, pKeyStr[32+1];
  char str[255+1] = {0}, oidStringStr[128+1] = {0}, oidValStr[128+1]= {0};
  char hostStr[158+1] = {0}, securityStr[30+1] = {0}, nextHostStr[158+1] = {0};
  OPEN_SNMP_COMMUNITY_ACCESS_TYPE_t accessType = OPEN_SNMP_COMMUNITY_ACCESS_RO;
  OPEN_SNMP_NOTIFY_FILTER_TYPE_t filterType = OPEN_SNMP_NOTIFY_FILTER_TYPE_INCLUDED;
  OPEN_SNMP_SECURITY_MODEL_t securityModel = OPEN_SNMP_SECURITY_MODEL_ANY;
  OPEN_SNMP_SECURITY_LEVEL_t securityLevel = OPEN_SNMP_SECURITY_LEVEL_NOAUTHNOPRIV;
  OPEN_USM_USER_AUTH_PROTOCOL_t authProto = OPEN_USM_USER_AUTH_PROTOCOL_NONE;
  OPEN_USM_USER_PRIV_PROTOCOL_t privProto = OPEN_USM_USER_PRIV_PROTOCOL_NONE;
  OPEN_SNMP_VIEW_TYPE_t viewType = OPEN_SNMP_VIEW_TYPE_INCLUDED;
  OPEN_SNMP_NOTIFY_TYPE_t notifyType = OPEN_SNMP_NOTIFY_TYPE_TRAP;
  OPEN_SNMP_VACM_TREE_FAMILY_TYPE_t treeType = OPEN_SNMP_VIEW_TREE;
  OPEN_SNMP_COMMUNITY_t community;
  uint32_t serverPort = 162, timeout = 3, retries = 1, trapFlag = 1, intf, type;
  bool mode = 1;

  badBufdescPointer.pstart = (void *) NULL;       
  badBufdescPointer.size = sizeof(buffer);        
  badBufdescPointer.size = 0;

  zeroLenBuf.pstart = buffer;
  zeroLenBuf.size = 0;

  sysName.pstart = str;
  sysName.size = sizeof(str);

  sysLocation.pstart = str;
  sysLocation.size = sizeof(str);

  sysContact.pstart = str;
  sysContact.size = sizeof(str);

  host.pstart = hostStr;
  host.size = sizeof(hostStr);

  securityName.pstart = securityStr;
  securityName.size = sizeof(securityStr);

  nextHost.pstart = nextHostStr;
  nextHost.size = sizeof(nextHostStr);

  memset(oidStringStr, 0, sizeof(oidStringStr));
  oidString.pstart = oidStringStr;
  oidString.size = sizeof(oidStringStr);

  memset(oidValStr, 0, sizeof(oidValStr));
  oidVal.pstart = oidValStr;
  oidVal.size = sizeof(oidValStr);

  nextViewName.pstart = nextViewStr;
  nextViewName.size = sizeof(nextViewStr);

  nextOidTree.pstart = nextOidStr;
  nextOidTree.size = sizeof(nextOidStr);

  userName.pstart = userNameStr;
  userName.size = sizeof(userNameStr);

  engineId.pstart = engineStr;
  engineId.size = sizeof(engineStr);
  
  memset(nextGroupStr, 0, sizeof(nextGroupStr));
  nextGroupDesc.pstart = nextGroupStr;
  nextGroupDesc.size = sizeof(nextGroupStr);

  nextUserName.pstart = nextUserNameStr;
  nextUserName.size = sizeof(nextUserNameStr);
  
  nextEngineId.pstart = nextEngineStr;
  nextEngineId.size = sizeof(nextEngineStr);

  authPwd.pstart = aPwdStr;
  authPwd.size = sizeof(aPwdStr);

  authKey.pstart = aKeyStr;
  authKey.size = sizeof(aKeyStr);

  privPwd.pstart = pPwdStr;
  privPwd.size = sizeof(pPwdStr);

  privKey.pstart = pKeyStr;
  privKey.size = sizeof(pKeyStr);

  filterDesc.pstart = filterStr;
  filterDesc.size = sizeof(filterStr);

  oidDesc.pstart = oidStr;
  oidDesc.size = sizeof(oidStr);

  contextDesc.pstart = contextStr;
  contextDesc.size = sizeof(contextStr);

  memset(nextContextStr, 0, sizeof(nextContextStr));
  nextContextDesc.pstart = nextContextStr;
  nextContextDesc.size = sizeof(nextContextStr);

  memset(readStr, 0, sizeof(readStr));
  readDesc.pstart = readStr;
  readDesc.size = sizeof(readStr);

  memset(writeStr, 0, sizeof(writeStr));
  writeDesc.pstart = writeStr;
  writeDesc.size = sizeof(writeStr);

  memset(notifyStr, 0, sizeof(notifyStr));
  notifyDesc.pstart = notifyStr;
  notifyDesc.size = sizeof(notifyStr);

  communityDesc.pstart = communityStr;
  communityDesc.size = sizeof(communityStr);

  nextCommunityDesc.pstart = nextCommunityStr;
  nextCommunityDesc.size = sizeof(nextCommunityStr);

  groupDesc.pstart = groupStr;
  groupDesc.size = strlen(groupStr) + 1;

  viewDesc.pstart = viewStr;
  viewDesc.size = strlen(viewStr)+1;

  memset(nextFilterStr, 0, sizeof(nextFilterStr));
  nextFilterDesc.pstart = nextFilterStr;
  nextFilterDesc.size = sizeof(nextFilterStr);

  memset(nextOidStr, 0, sizeof(nextOidStr));
  nextOidDesc.pstart = nextOidStr;
  nextOidDesc.size = sizeof(nextOidStr);

  memset(&ipAddr, 0, sizeof(ipAddr));
  memset(ipStr, 0, sizeof(ipStr));
  snprintf(ipStr, sizeof(ipStr), "%s", "0.0.0.0");
  inet_pton(AF_INET, ipStr, (void*)&(ipAddr.addr.ipv4));
  ipAddr.family = OPEN_AF_INET;

  memset(&community, 0, sizeof(community));

  printf("Testing SNMP OpEN APIs sanity:\n\n");

  printf("Testing openapiSnmpCommunityCreate():\n");
  result = openapiSnmpCommunityCreate(NULL, &communityDesc, accessType, &viewDesc, ipAddr);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpCommunityCreate(clientHandle, &badBufdescPointer, accessType, &viewDesc, ipAddr);
  printf("NULL community name argument 2:(result = %d)\n", result);
  result = openapiSnmpCommunityCreate(clientHandle, &zeroLenBuf, accessType, &viewDesc, ipAddr);
  printf("Buffdesc 0 length for community name argument 2:(result = %d)\n", result);
  result = openapiSnmpCommunityCreate(clientHandle, &communityDesc, OPEN_SNMP_COMMUNITY_ACCESS_NONE - 1, &viewDesc, ipAddr);
  printf("less than minimum value access type argument 3:(result = %d)\n", result);
  result = openapiSnmpCommunityCreate(clientHandle, &communityDesc, OPEN_SNMP_COMMUNITY_ACCESS_SU + 1, &viewDesc, ipAddr);
  printf("greater than maximum value access type argument 3:(result = %d)\n", result);
  result = openapiSnmpCommunityCreate(clientHandle, &communityDesc, accessType, &badBufdescPointer, ipAddr);
  printf("Null view name argument 4:(result = %d)\n", result);
  result = openapiSnmpCommunityCreate(clientHandle, &communityDesc, accessType, &zeroLenBuf, ipAddr);
  printf("Buffdesc 0 length for view name argument 4:(result = %d)\n", result);
  result = openapiSnmpCommunityCreate(clientHandle, &communityDesc, accessType, &viewDesc, ipAddr);
  printf("NULL IP address argument 5:(result = %d)\n", result);
  printf("Testing openapiSnmpCommunityCreate() done.\n\n");

  printf("Testing openapiSnmpCommunityDelete():\n");
  result = openapiSnmpCommunityDelete(NULL, &communityDesc);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpCommunityDelete(clientHandle, &badBufdescPointer);
  printf("NULL community name argument 2:(result = %d)\n", result);
  result = openapiSnmpCommunityDelete(clientHandle, &zeroLenBuf);
  printf("Buffdesc 0 length for community name argument 2:(result = %d)\n", result);
  printf("Testing openapiSnmpCommunityDelete() done.\n\n");

  printf("Testing openapiSnmpCommunityGetNext():\n");
  result = openapiSnmpCommunityGetNext(NULL, &communityDesc, &accessType, &viewDesc, &ipAddr, &nextCommunityDesc);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpCommunityGetNext(clientHandle, &badBufdescPointer, &accessType, &viewDesc, &ipAddr, &nextCommunityDesc);
  printf("NULL community name argument 2:(result = %d)\n", result);
  result = openapiSnmpCommunityGetNext(clientHandle, &zeroLenBuf, &accessType, &viewDesc, &ipAddr, &nextCommunityDesc);
  printf("Buffdesc 0 length for community name argument 2:(result = %d)\n", result);
  result = openapiSnmpCommunityGetNext(clientHandle, &communityDesc, NULL, &viewDesc, &ipAddr, &nextCommunityDesc);
  printf("NULL acccess type argument 3:(result = %d)\n", result);
  result = openapiSnmpCommunityGetNext(clientHandle, &communityDesc, &accessType, &badBufdescPointer, &ipAddr, &nextCommunityDesc);
  printf("NULL view name argument 4:(result = %d)\n", result);
  result = openapiSnmpCommunityGetNext(clientHandle, &communityDesc, &accessType, &zeroLenBuf, &ipAddr, &nextCommunityDesc);
  printf("Buffdesc 0 length for view name argument 4:(result = %d)\n", result);
  result = openapiSnmpCommunityGetNext(clientHandle, &communityDesc, &accessType, &viewDesc, NULL, &nextCommunityDesc);
  printf("NULL IP address argument 5:(result = %d)\n", result);
  result = openapiSnmpCommunityGetNext(clientHandle, &communityDesc, &accessType, &viewDesc, &ipAddr, &badBufdescPointer);
  printf("NULL next community name argument 6:(result = %d)\n", result);
  result = openapiSnmpCommunityGetNext(clientHandle, &communityDesc, &accessType, &viewDesc, &ipAddr, &zeroLenBuf);
  printf("Buffdesc 0 length for next community name argument 6:(result = %d)\n", result);
  printf("Testing openapiSnmpCommunityGetNext() done.\n\n");

  printf("Testing openapiSnmpCommunityGroupCreate():\n");
  result = openapiSnmpCommunityGroupCreate(NULL, &communityDesc, &groupDesc, ipAddr);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpCommunityGroupCreate(clientHandle, &badBufdescPointer, &groupDesc, ipAddr);
  printf("NULL community name argument 2:(result = %d)\n", result);
  result = openapiSnmpCommunityGroupCreate(clientHandle, &zeroLenBuf, &groupDesc, ipAddr);
  printf("Buffdesc 0 length for community name argument 2:(result = %d)\n", result);
  result = openapiSnmpCommunityGroupCreate(clientHandle, &communityDesc, &badBufdescPointer, ipAddr);
  printf("NULL group name argument 3:(result = %d)\n", result);
  result = openapiSnmpCommunityGroupCreate(clientHandle, &communityDesc, &zeroLenBuf, ipAddr);
  printf("Buffdesc 0 length for group name argument 3:(result = %d)\n", result);
  result = openapiSnmpCommunityGroupCreate(clientHandle, &communityDesc, &groupDesc, ipAddr);
  printf("NULL IP address argument 4:(result = %d)\n", result);
  printf("Testing openapiSnmpCommunityGroupCreate() done.\n\n");

  printf("Testing openapiSnmpFilterCreate():\n");
  result = openapiSnmpFilterCreate(NULL, &filterDesc, &oidDesc, filterType);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpFilterCreate(clientHandle, &badBufdescPointer, &oidDesc, filterType);
  printf("NULL filter name argument 2:(result = %d)\n", result);
  result = openapiSnmpFilterCreate(clientHandle, &zeroLenBuf, &oidDesc, filterType);
  printf("Buffdesc 0 length for filter name argument 2:(result = %d)\n", result);
  result = openapiSnmpFilterCreate(clientHandle, &filterDesc, &badBufdescPointer, filterType);
  printf("NULL OID argument 3:(result = %d)\n", result);
  result = openapiSnmpFilterCreate(clientHandle, &filterDesc, &zeroLenBuf, filterType);
  printf("Buffdesc 0 length for OID argument 3:(result = %d)\n", result);
  result = openapiSnmpFilterCreate(clientHandle, &filterDesc, &oidDesc, OPEN_SNMP_NOTIFY_FILTER_TYPE_INCLUDED - 1);
  printf("less than minimum value filter type argument 4:(result = %d)\n", result);
  result = openapiSnmpFilterCreate(clientHandle, &filterDesc, &oidDesc, OPEN_SNMP_NOTIFY_FILTER_TYPE_EXCLUDED + 1);
  printf("greater than maximum value filter type argument 4:(result = %d)\n", result);
  printf("Testing openapiSnmpFilterCreate() done.\n\n");

  printf("Testing openapiSnmpFilterDelete():\n");
  result = openapiSnmpFilterDelete(NULL, &filterDesc, &oidDesc);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpFilterDelete(clientHandle, &badBufdescPointer, &oidDesc);
  printf("NULL filter name argument 2:(result = %d)\n", result);
  result = openapiSnmpFilterDelete(clientHandle, &zeroLenBuf, &oidDesc);
  printf("Buffdesc 0 length for filter name argument 2:(result = %d)\n", result);
  result = openapiSnmpFilterDelete(clientHandle, &filterDesc, &badBufdescPointer);
  printf("NULL OID argument 3:(result = %d)\n", result);
  result = openapiSnmpFilterDelete(clientHandle, &filterDesc, &zeroLenBuf);
  printf("Buffdesc 0 length for OID argument 3:(result = %d)\n", result);
  printf("Testing openapiSnmpFilterDelete() done.\n\n");

  printf("Testing openapiSnmpFilterGetNext():\n");
  result = openapiSnmpFilterGetNext(NULL, &filterDesc, &oidDesc,
                                    &filterType, &nextFilterDesc, &nextOidDesc);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpFilterGetNext(clientHandle, &badBufdescPointer, &oidDesc,
                                    &filterType, &nextFilterDesc, &nextOidDesc);
  printf("NULL filter name argument 2:(result = %d)\n", result);
  result = openapiSnmpFilterGetNext(clientHandle, &zeroLenBuf, &oidDesc,
                                    &filterType, &nextFilterDesc, &nextOidDesc);
  printf("Buffdesc 0 length for filter name argument 2:(result = %d)\n", result);
  result = openapiSnmpFilterGetNext(clientHandle, &filterDesc, &badBufdescPointer,
                                    &filterType, &nextFilterDesc, &nextOidDesc);
  printf("NULL OID argument 3:(result = %d)\n", result);
  result = openapiSnmpFilterGetNext(clientHandle, &filterDesc, &zeroLenBuf,
                                    &filterType, &nextFilterDesc, &nextOidDesc);
  printf("Buffdesc 0 length for OID argument 3:(result = %d)\n", result);
  result = openapiSnmpFilterGetNext(clientHandle, &filterDesc, &oidDesc,
                                    NULL, &nextFilterDesc, &nextOidDesc);
  printf("NULL filter type argument 4:(result = %d)\n", result);
  result = openapiSnmpFilterGetNext(clientHandle, &filterDesc, &oidDesc,
                                    &filterType, &badBufdescPointer, &nextOidDesc);
  printf("NULL next filter name argument 5:(result = %d)\n", result);
  result = openapiSnmpFilterGetNext(clientHandle, &filterDesc, &oidDesc,
                                    &filterType, &zeroLenBuf, &nextOidDesc);
  printf("Buffdesc 0 length for next filter name argument 5:(result = %d)\n", result);
  result = openapiSnmpFilterGetNext(clientHandle, &filterDesc, &oidDesc,
                                    &filterType, &nextFilterDesc, &badBufdescPointer);
  printf("NULL next OID argument 6:(result = %d)\n", result);
  result = openapiSnmpFilterGetNext(clientHandle, &filterDesc, &oidDesc,
                                    &filterType, &nextFilterDesc, &zeroLenBuf);
  printf("Buffdesc 0 length for next OID argument 6:(result = %d)\n", result);
  printf("Testing openapiSnmpFilterGetNext() done.\n\n");

  printf("Testing openapiSnmpGroupCreate():\n");
  result = openapiSnmpGroupCreate(NULL, &groupDesc, securityModel, securityLevel,
                                  &contextDesc, &readDesc, &writeDesc, &notifyDesc);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpGroupCreate(clientHandle, &badBufdescPointer, securityModel, securityLevel,
                                  &contextDesc, &readDesc, &writeDesc, &notifyDesc);
  printf("NULL group name argument 2:(result = %d)\n", result);
  result = openapiSnmpGroupCreate(clientHandle, &zeroLenBuf, securityModel, securityLevel,
                                  &contextDesc, &readDesc, &writeDesc, &notifyDesc);
  printf("Buffdesc 0 length for group name argument 2:(result = %d)\n", result);
  result = openapiSnmpGroupCreate(clientHandle, &groupDesc, OPEN_SNMP_SECURITY_MODEL_ANY - 1, securityLevel,
                                  &contextDesc, &readDesc, &writeDesc, &notifyDesc);
  printf("less than minimum value security model argument 3:(result = %d)\n", result);
  result = openapiSnmpGroupCreate(clientHandle, &groupDesc, OPEN_SNMP_SECURITY_MODEL_USM + 1, securityLevel,
                                  &contextDesc, &readDesc, &writeDesc, &notifyDesc);
  printf("greater than maximum value security model argument 3:(result = %d)\n", result);
  result = openapiSnmpGroupCreate(clientHandle, &groupDesc, securityModel, OPEN_SNMP_SECURITY_LEVEL_NOAUTHNOPRIV - 1,
                                  &contextDesc, &readDesc, &writeDesc, &notifyDesc);
  printf("less than minimum value security level argument 4:(result = %d)\n", result);
  result = openapiSnmpGroupCreate(clientHandle, &groupDesc, securityModel,  OPEN_SNMP_SECURITY_LEVEL_AUTHPRIV + 1,
                                  &contextDesc, &readDesc, &writeDesc, &notifyDesc);
  printf("greater than maximum value security level argument 4:(result = %d)\n", result);
  result = openapiSnmpGroupCreate(clientHandle, &groupDesc, securityModel, securityLevel,
                                  &badBufdescPointer, &readDesc, &writeDesc, &notifyDesc);
  printf("NULL context argument 5:(result = %d)\n", result);
  result = openapiSnmpGroupCreate(clientHandle, &groupDesc, securityModel, securityLevel,
                                  &zeroLenBuf, &readDesc, &writeDesc, &notifyDesc);
  printf("Buffdesc 0 length for context argument 5:(result = %d)\n", result);
  result = openapiSnmpGroupCreate(clientHandle, &groupDesc, securityModel, securityLevel,
                                  &contextDesc, &badBufdescPointer, &writeDesc, &notifyDesc);
  printf("NULL readview argument 6:(result = %d)\n", result);
  result = openapiSnmpGroupCreate(clientHandle, &groupDesc, securityModel, securityLevel,
                                  &contextDesc, &zeroLenBuf, &writeDesc, &notifyDesc);
  printf("Buffdesc 0 length for readview argument 6:(result = %d)\n", result);
  result = openapiSnmpGroupCreate(clientHandle, &groupDesc, securityModel, securityLevel,
                                  &contextDesc, &readDesc, &badBufdescPointer, &notifyDesc);
  printf("NULL writeview argument 7:(result = %d)\n", result);
  result = openapiSnmpGroupCreate(clientHandle, &groupDesc, securityModel, securityLevel,
                                  &contextDesc, &readDesc, &zeroLenBuf, &notifyDesc);
  printf("Buffdesc 0 length for writeview argument 7:(result = %d)\n", result);
  result = openapiSnmpGroupCreate(clientHandle, &groupDesc, securityModel, securityLevel,
                                  &contextDesc, &readDesc, &writeDesc, &badBufdescPointer);
  printf("NULL notifyview argument 8:(result = %d)\n", result);
  result = openapiSnmpGroupCreate(clientHandle, &groupDesc, securityModel, securityLevel,
                                  &contextDesc, &readDesc, &writeDesc, &zeroLenBuf);
  printf("Buffdesc 0 length for notifyview argument 8:(result = %d)\n", result);
  printf("Testing openapiSnmpGroupCreate() done.\n\n");


  printf("Testing openapiSnmpGroupDelete():\n");
  result = openapiSnmpGroupDelete(NULL, &groupDesc, securityModel,
                                  securityLevel, &contextDesc);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpGroupDelete(clientHandle, &badBufdescPointer, securityModel,
                                  securityLevel, &contextDesc);
  printf("NULL group name argument 2:(result = %d)\n", result);
  result = openapiSnmpGroupDelete(clientHandle, &zeroLenBuf, securityModel,
                                  securityLevel, &contextDesc);
  printf("Buffdesc 0 length for group name argument 2:(result = %d)\n", result);
  result = openapiSnmpGroupDelete(clientHandle, &groupDesc, OPEN_SNMP_SECURITY_MODEL_ANY - 1,
                                  securityLevel, &contextDesc);
  printf("less than minimum value security model argument 3:(result = %d)\n", result);
  result = openapiSnmpGroupDelete(clientHandle, &groupDesc, OPEN_SNMP_SECURITY_MODEL_USM + 1,
                                  securityLevel, &contextDesc);
  printf("greater than maximum value security model argument 3:(result = %d)\n", result);
  result = openapiSnmpGroupDelete(clientHandle, &groupDesc, securityModel,
                                  OPEN_SNMP_SECURITY_LEVEL_NOAUTHNOPRIV - 1, &contextDesc);
  printf("less than minimum value security level argument 4:(result = %d)\n", result);
  result = openapiSnmpGroupDelete(clientHandle, &groupDesc, securityModel,
                                  OPEN_SNMP_SECURITY_LEVEL_AUTHPRIV + 1, &contextDesc);
  printf("greater than maximum value security level argument 4:(result = %d)\n", result);
  result = openapiSnmpGroupDelete(clientHandle, &groupDesc, securityModel,
                                  securityLevel, &badBufdescPointer);
  printf("NULL context argument 5:(result = %d)\n", result);
  result = openapiSnmpGroupDelete(clientHandle, &groupDesc, securityModel,
                                  securityLevel, &zeroLenBuf);
  printf("Buffdesc 0 length for context argument 5:(result = %d)\n", result);
  printf("Testing openapiSnmpGroupDelete() done.\n\n");

  printf("Testing openapiSnmpGroupGetNext():\n");
  result = openapiSnmpGroupGetNext(NULL, &groupDesc, &securityModel, &securityLevel,
                                   &contextDesc, &readDesc, &writeDesc, &notifyDesc,
                                   &nextGroupDesc, &nextContextDesc);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpGroupGetNext(clientHandle, &badBufdescPointer, &securityModel, &securityLevel,
                                   &contextDesc, &readDesc, &writeDesc, &notifyDesc,
                                   &nextGroupDesc, &nextContextDesc);
  printf("NULL group name argument 2:(result = %d)\n", result);
  result = openapiSnmpGroupGetNext(clientHandle, &zeroLenBuf, &securityModel, &securityLevel,
                                   &contextDesc, &readDesc, &writeDesc, &notifyDesc,
                                   &nextGroupDesc, &nextContextDesc);
  printf("Buffdesc 0 length for group name argument 2:(result = %d)\n", result);
  result = openapiSnmpGroupGetNext(clientHandle, &groupDesc, NULL, &securityLevel,
                                   &contextDesc, &readDesc, &writeDesc, &notifyDesc,
                                   &nextGroupDesc, &nextContextDesc);
  printf("NULL security model argument 3:(result = %d)\n", result);
  result = openapiSnmpGroupGetNext(clientHandle, &groupDesc, &securityModel, NULL,
                                   &contextDesc, &readDesc, &writeDesc, &notifyDesc,
                                   &nextGroupDesc, &nextContextDesc);
  printf("NULL security level argument 4:(result = %d)\n", result);
  result = openapiSnmpGroupGetNext(clientHandle, &groupDesc, &securityModel, &securityLevel,
                                   &badBufdescPointer, &readDesc, &writeDesc, &notifyDesc,
                                   &nextGroupDesc, &nextContextDesc);
  printf("NULL context argument 5:(result = %d)\n", result);
  result = openapiSnmpGroupGetNext(clientHandle, &groupDesc, &securityModel, &securityLevel,
                                   &zeroLenBuf, &readDesc, &writeDesc, &notifyDesc,
                                   &nextGroupDesc, &nextContextDesc);
  printf("Buffdesc 0 length for context argument 5:(result = %d)\n", result);
  result = openapiSnmpGroupGetNext(clientHandle, &groupDesc, &securityModel, &securityLevel,
                                   &contextDesc, &badBufdescPointer, &writeDesc, &notifyDesc,
                                   &nextGroupDesc, &nextContextDesc);
  printf("NULL readview argument 6:(result = %d)\n", result);
  result = openapiSnmpGroupGetNext(clientHandle, &groupDesc, &securityModel, &securityLevel,
                                   &contextDesc, &zeroLenBuf, &writeDesc, &notifyDesc,
                                   &nextGroupDesc, &nextContextDesc);
  printf("Buffdesc 0 length for readview argument 6:(result = %d)\n", result);
  result = openapiSnmpGroupGetNext(clientHandle, &groupDesc, &securityModel, &securityLevel,
                                   &contextDesc, &readDesc, &badBufdescPointer, &notifyDesc,
                                   &nextGroupDesc, &nextContextDesc);
  printf("NULL writeview argument 7:(result = %d)\n", result);
  result = openapiSnmpGroupGetNext(clientHandle, &groupDesc, &securityModel, &securityLevel,
                                   &contextDesc, &readDesc, &zeroLenBuf, &notifyDesc,
                                   &nextGroupDesc, &nextContextDesc);
  printf("Buffdesc 0 length for writeview argument 7:(result = %d)\n", result);
  result = openapiSnmpGroupGetNext(clientHandle, &groupDesc, &securityModel, &securityLevel,
                                   &contextDesc, &readDesc, &writeDesc, &badBufdescPointer,
                                   &nextGroupDesc, &nextContextDesc);
  printf("NULL notifyview argument 8:(result = %d)\n", result);
  result = openapiSnmpGroupGetNext(clientHandle, &groupDesc, &securityModel, &securityLevel,
                                   &contextDesc, &readDesc, &writeDesc, &zeroLenBuf,
                                   &nextGroupDesc, &nextContextDesc);
  printf("Buffdesc 0 length for notifyview argument 8:(result = %d)\n", result);
  result = openapiSnmpGroupGetNext(clientHandle, &groupDesc, &securityModel, &securityLevel,
                                   &contextDesc, &readDesc, &writeDesc, &notifyDesc,
                                   &badBufdescPointer, &nextContextDesc);
  printf("NULL next group name argument 9:(result = %d)\n", result);
  result = openapiSnmpGroupGetNext(clientHandle, &groupDesc, &securityModel, &securityLevel,
                                   &contextDesc, &readDesc, &writeDesc, &notifyDesc,
                                   &zeroLenBuf, &nextContextDesc);
  printf("Buffdesc 0 length for next group name argument 9:(result = %d)\n", result);
  result = openapiSnmpGroupGetNext(clientHandle, &groupDesc, &securityModel, &securityLevel,
                                   &contextDesc, &readDesc, &writeDesc, &notifyDesc,
                                   &nextGroupDesc, &badBufdescPointer);
  printf("NULL next context argument 10:(result = %d)\n", result);
  result = openapiSnmpGroupGetNext(clientHandle, &groupDesc, &securityModel, &securityLevel,
                                   &contextDesc, &readDesc, &writeDesc, &notifyDesc,
                                   &nextGroupDesc, &zeroLenBuf);
  printf("Buffdesc 0 length for next context argument 10:(result = %d)\n", result);
  printf("Testing openapiSnmpGroupGetNext() done.\n\n");

  printf("Testing openapiSnmpLocalEngineIdDelete():\n");
  result = openapiSnmpLocalEngineIdDelete(NULL);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  printf("Testing openapiSnmpLocalEngineIdDelete() done.\n\n");

  printf("Testing openapiSnmpLocalEngineIdGet():\n");
  result = openapiSnmpLocalEngineIdGet(NULL, &engineId);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpLocalEngineIdGet(clientHandle, &badBufdescPointer);
  printf("NULL engine id argument 2:(result = %d)\n", result);
  result = openapiSnmpLocalEngineIdGet(clientHandle, &zeroLenBuf);
  printf("Buffdesc 0 length for engine id argument 2:(result = %d)\n", result);
  printf("Testing openapiSnmpLocalEngineIdGet() done.\n\n");

  printf("Testing openapiSnmpLocalEngineIdSet():\n");
  result = openapiSnmpLocalEngineIdSet(NULL, &engineId);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpLocalEngineIdSet(clientHandle, &badBufdescPointer);
  printf("NULL engine id argument 2:(result = %d)\n", result);
  result = openapiSnmpLocalEngineIdSet(clientHandle, &zeroLenBuf);
  printf("Buffdesc 0 length for engine id argument 2:(result = %d)\n", result);
  printf("Testing openapiSnmpLocalEngineIdSet() done.\n\n");

  printf("Testing openapiSnmpUserCreate():\n");
  result = openapiSnmpUserCreate(NULL, &userName, &groupDesc, authProto,
                                 &authPwd, &authKey, privProto, &privPwd,
                                 &privKey, &engineId);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpUserCreate(clientHandle, &badBufdescPointer, &groupDesc, authProto,
                                 &authPwd, &authKey, privProto, &privPwd,
                                 &privKey, &engineId);
  printf("NULL user name argument 2:(result = %d)\n", result);
  result = openapiSnmpUserCreate(clientHandle, &zeroLenBuf, &groupDesc, authProto,
                                 &authPwd, &authKey, privProto, &privPwd,
                                 &privKey, &engineId);
  printf("Buffdesc 0 length for user name argument 2:(result = %d)\n", result);
  result = openapiSnmpUserCreate(clientHandle, &userName, &badBufdescPointer, authProto,
                                 &authPwd, &authKey, privProto, &privPwd,
                                 &privKey, &engineId);
  printf("NULL group name argument 3:(result = %d)\n", result);
  result = openapiSnmpUserCreate(clientHandle, &userName, &zeroLenBuf, authProto,
                                 &authPwd, &authKey, privProto, &privPwd,
                                 &privKey, &engineId);
  printf("Buffdesc 0 length for group name argument 3:(result = %d)\n", result);
  result = openapiSnmpUserCreate(clientHandle, &userName, &groupDesc, OPEN_USM_USER_AUTH_PROTOCOL_NONE - 1,
                                 &authPwd, &authKey, privProto, &privPwd,
                                 &privKey, &engineId);
  printf("less than minimum value auth protocol argument 4:(result = %d)\n", result);
  result = openapiSnmpUserCreate(clientHandle, &userName, &groupDesc, OPEN_USM_USER_AUTH_PROTOCOL_SHA512 + 1,
                                 &authPwd, &authKey, privProto, &privPwd,
                                 &privKey, &engineId);
  printf("greater than maximum value auth protocol argument 4:(result = %d)\n", result);
  result = openapiSnmpUserCreate(clientHandle, &userName, &groupDesc, authProto,
                                 &badBufdescPointer, &authKey, privProto, &privPwd,
                                 &privKey, &engineId);
  printf("NULL auth password argument 5:(result = %d)\n", result);
  result = openapiSnmpUserCreate(clientHandle, &userName, &groupDesc, authProto,
                                 &zeroLenBuf, &authKey, privProto, &privPwd,
                                 &privKey, &engineId);
  printf("Buffdesc 0 length for auth password argument 5:(result = %d)\n", result);
  result = openapiSnmpUserCreate(clientHandle, &userName, &groupDesc, authProto,
                                 &authPwd, &badBufdescPointer, privProto, &privPwd,
                                 &privKey, &engineId);
  printf("NULL auth key argument 6:(result = %d)\n", result);
  result = openapiSnmpUserCreate(clientHandle, &userName, &groupDesc, authProto,
                                 &authPwd, &zeroLenBuf, privProto, &privPwd,
                                 &privKey, &engineId);
  printf("Buffdesc 0 length for auth key argument 6:(result = %d)\n", result);
  result = openapiSnmpUserCreate(clientHandle, &userName, &groupDesc, authProto,
                                 &authPwd, &authKey, OPEN_USM_USER_PRIV_PROTOCOL_NONE - 1, &privPwd,
                                 &privKey, &engineId);
  printf("less than minimum value priv protocol argument 7:(result = %d)\n", result);
  result = openapiSnmpUserCreate(clientHandle, &userName, &groupDesc, authProto,
                                 &authPwd, &authKey, OPEN_USM_USER_PRIV_PROTOCOL_AES256 + 1, &privPwd,
                                 &privKey, &engineId);
  printf("greater than maximum value priv protocol argument 7:(result = %d)\n", result);
  result = openapiSnmpUserCreate(clientHandle, &userName, &groupDesc, authProto,
                                 &authPwd, &authKey, privProto, &badBufdescPointer,
                                 &privKey, &engineId);
  printf("NULL priv password argument 8:(result = %d)\n", result);
  result = openapiSnmpUserCreate(clientHandle, &userName, &groupDesc, authProto,
                                 &authPwd, &authKey, privProto, &zeroLenBuf,
                                 &privKey, &engineId);
  printf("Buffdesc 0 length for priv password argument 8:(result = %d)\n", result);
  result = openapiSnmpUserCreate(clientHandle, &userName, &groupDesc, authProto,
                                 &authPwd, &authKey, privProto, &privPwd,
                                 &badBufdescPointer, &engineId);
  printf("NULL priv key argument 9:(result = %d)\n", result);
  result = openapiSnmpUserCreate(clientHandle, &userName, &groupDesc, authProto,
                                 &authPwd, &authKey, privProto, &privPwd,
                                 &zeroLenBuf, &engineId);
  printf("Buffdesc 0 length for priv key argument 9:(result = %d)\n", result);
  result = openapiSnmpUserCreate(clientHandle, &userName, &groupDesc, authProto,
                                 &authPwd, &authKey, privProto, &privPwd,
                                 &privKey, &badBufdescPointer);
  printf("NULL engine id argument 10:(result = %d)\n", result);
  result = openapiSnmpUserCreate(clientHandle, &userName, &groupDesc, authProto,
                                 &authPwd, &authKey, privProto, &privPwd,
                                 &privKey, &zeroLenBuf);
  printf("Buffdesc 0 length for engine id argument 10:(result = %d)\n", result);
  printf("Testing openapiSnmpUserCreate() done.\n\n");

  printf("Testing openapiSnmpUserDelete():\n");
  result = openapiSnmpUserDelete(NULL, &userName, &engineId);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpUserDelete(clientHandle, &badBufdescPointer, &engineId);
  printf("NULL user name argument 2:(result = %d)\n", result);
  result = openapiSnmpUserDelete(clientHandle, &zeroLenBuf, &engineId);
  printf("Buffdesc 0 length for user name argument 2:(result = %d)\n", result);
  result = openapiSnmpUserDelete(clientHandle, &userName, &badBufdescPointer);
  printf("NULL engine id argument 3:(result = %d)\n", result);
  result = openapiSnmpUserDelete(clientHandle, &userName, &zeroLenBuf);
  printf("Buffdesc 0 length for engine id argument 3:(result = %d)\n", result);
  printf("Testing openapiSnmpUserDelete() done.\n\n");

  printf("Testing openapiSnmpUserGetNext():\n");
  result = openapiSnmpUserGetNext(NULL, &userName, &engineId, &groupDesc,
                                  &authProto, &privProto, &nextUserName, &nextEngineId);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpUserGetNext(clientHandle, &badBufdescPointer, &engineId, &groupDesc,
                                  &authProto, &privProto, &nextUserName, &nextEngineId);
  printf("NULL user name argument 2:(result = %d)\n", result);
  result = openapiSnmpUserGetNext(clientHandle, &zeroLenBuf, &engineId, &groupDesc,
                                  &authProto, &privProto, &nextUserName, &nextEngineId);
  printf("Buffdesc 0 length for user name argument 2:(result = %d)\n", result);
  result = openapiSnmpUserGetNext(clientHandle, &userName, &badBufdescPointer, &groupDesc,
                                  &authProto, &privProto, &nextUserName, &nextEngineId);
  printf("NULL engine id argument 3:(result = %d)\n", result);
  result = openapiSnmpUserGetNext(clientHandle, &userName, &zeroLenBuf, &groupDesc,
                                  &authProto, &privProto, &nextUserName, &nextEngineId);
  printf("Buffdesc 0 length for engine id argument 3:(result = %d)\n", result);
  result = openapiSnmpUserGetNext(clientHandle, &userName, &engineId, &badBufdescPointer,
                                  &authProto, &privProto, &nextUserName, &nextEngineId);
  printf("NULL group name argument 4:(result = %d)\n", result);
  result = openapiSnmpUserGetNext(clientHandle, &userName, &engineId, &zeroLenBuf,
                                  &authProto, &privProto, &nextUserName, &nextEngineId);
  printf("Buffdesc 0 length for group name argument 4:(result = %d)\n", result);
  result = openapiSnmpUserGetNext(clientHandle, &userName, &engineId, &groupDesc, 
                                  NULL, &privProto, &nextUserName, &nextEngineId);
  printf("NULL auth protocol argument 5:(result = %d)\n", result);
  result = openapiSnmpUserGetNext(clientHandle, &userName, &engineId, &groupDesc,
                                  &authProto, NULL, &nextUserName, &nextEngineId);
  printf("NULL priv protocol argument 6:(result = %d)\n", result);
  result = openapiSnmpUserGetNext(clientHandle, &userName, &engineId, &groupDesc,
                                  &authProto, &privProto, &badBufdescPointer, &nextEngineId);
  printf("NULL next user name argument 7:(result = %d)\n", result);
  result = openapiSnmpUserGetNext(clientHandle, &userName, &engineId, &groupDesc,
                                  &authProto, &privProto, &zeroLenBuf, &nextEngineId);
  printf("Buffdesc 0 length for next user name argument 7:(result = %d)\n", result);
  result = openapiSnmpUserGetNext(clientHandle, &userName, &engineId, &groupDesc,
                                  &authProto, &privProto, &nextUserName, &badBufdescPointer);
  printf("NULL next engine id argument 8:(result = %d)\n", result);
  result = openapiSnmpUserGetNext(clientHandle, &userName, &engineId, &groupDesc,
                                  &authProto, &privProto, &nextUserName, &zeroLenBuf);
  printf("Buffdesc 0 length for next engine id argument 8:(result = %d)\n", result);
  printf("Testing openapiSnmpUserGetNext() done.\n\n");

  printf("Testing openapiSnmpHostCreate():\n");
  result = openapiSnmpHostCreate(NULL, &host, serverPort, &securityName, securityModel,
                                 securityLevel, notifyType, timeout, retries, &filterDesc);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpHostCreate(clientHandle, &badBufdescPointer, serverPort,
                                 &securityName, securityModel, securityLevel,
                                 notifyType, timeout, retries, &filterDesc);
  printf("NULL host argument 2:(result = %d)\n", result);
  result = openapiSnmpHostCreate(clientHandle, &zeroLenBuf, serverPort, &securityName,
                                 securityModel, securityLevel, notifyType,
                                 timeout, retries, &filterDesc);
  printf("Buffdesc 0 length for host argument 2:(result = %d)\n", result);
  result = openapiSnmpHostCreate(clientHandle, &host, 0, &securityName,
                                 securityModel, securityLevel, notifyType,
                                 timeout, retries, &filterDesc);
  printf("less than minimum value port argument 3:(result = %d)\n", result);
  result = openapiSnmpHostCreate(clientHandle, &host, 65537, &securityName, securityModel,
                                 securityLevel, notifyType, timeout, retries, &filterDesc);
  printf("greater than maximum value port argument 3:(result = %d)\n", result);
  result = openapiSnmpHostCreate(clientHandle, &host, serverPort, &badBufdescPointer, 
                                 securityModel, securityLevel, notifyType,
                                 timeout, retries, &filterDesc);
  printf("NULL security name argument 4:(result = %d)\n", result);
  result = openapiSnmpHostCreate(clientHandle, &host, serverPort, &zeroLenBuf,
                                 securityModel, securityLevel, notifyType,
                                 timeout, retries, &filterDesc);
  printf("Buffdesc 0 length for security name argument 4:(result = %d)\n", result);
  result = openapiSnmpHostCreate(clientHandle, &host, serverPort, &securityName,
                                 OPEN_SNMP_SECURITY_MODEL_ANY - 1, securityLevel, notifyType,
                                 timeout, retries, &filterDesc);
  printf("less than minimum value security model argument 5:(result = %d)\n", result);
  result = openapiSnmpHostCreate(clientHandle, &host, serverPort, &securityName,
                                 OPEN_SNMP_SECURITY_MODEL_USM + 1, securityLevel, notifyType,
                                 timeout, retries, &filterDesc);
  printf("greater than maximum value security model argument 5:(result = %d)\n", result);
  result = openapiSnmpHostCreate(clientHandle, &host, serverPort, &securityName,
                                 securityModel, OPEN_SNMP_SECURITY_LEVEL_NOAUTHNOPRIV - 1, notifyType,
                                 timeout, retries, &filterDesc);
  printf("less than minimum value security level argument 6:(result = %d)\n", result);
  result = openapiSnmpHostCreate(clientHandle, &host, serverPort, &securityName,
                                 securityModel, OPEN_SNMP_SECURITY_LEVEL_AUTHPRIV + 1, notifyType,
                                 timeout, retries, &filterDesc);
  printf("greater than maximum value security level argument 6:(result = %d)\n", result);
  result = openapiSnmpHostCreate(clientHandle, &host, serverPort, &securityName,
                                 securityModel, securityLevel, OPEN_SNMP_NOTIFY_TYPE_TRAP - 1,
                                 timeout, retries, &filterDesc);
  printf("less than minimum value notify type argument 7:(result = %d)\n", result);
  result = openapiSnmpHostCreate(clientHandle, &host, serverPort, &securityName,
                                 securityModel, securityLevel, OPEN_SNMP_NOTIFY_TYPE_INFORM + 1,
                                 timeout, retries, &filterDesc);
  printf("greater than maximum value notify type argument 7:(result = %d)\n", result);
  result = openapiSnmpHostCreate(clientHandle, &host, serverPort, &securityName,
                                 securityModel, securityLevel, notifyType,
                                 0, retries, &filterDesc);
  printf("less than minimum value timeout argument 8:(result = %d)\n", result);
  result = openapiSnmpHostCreate(clientHandle, &host, serverPort, &securityName,
                                 securityModel, securityLevel, notifyType,
                                 301, retries, &filterDesc);
  printf("greater than maximum value timeout argument 8:(result = %d)\n", result);
  result = openapiSnmpHostCreate(clientHandle, &host, serverPort, &securityName,
                                 securityModel, securityLevel, notifyType,
                                 timeout, 0, &filterDesc);
  printf("less than minimum value retries argument 9:(result = %d)\n", result);
  result = openapiSnmpHostCreate(clientHandle, &host, serverPort, &securityName,
                                 securityModel, securityLevel, notifyType,
                                 timeout, 256, &filterDesc);
  printf("greater than maximum value retries argument 9:(result = %d)\n", result);
  result = openapiSnmpHostCreate(clientHandle, &host, serverPort, &securityName,
                                 securityModel, securityLevel, notifyType,
                                 timeout, retries, &badBufdescPointer);
  printf("NULL filter argument 10:(result = %d)\n", result);
  result = openapiSnmpHostCreate(clientHandle, &host, serverPort, &securityName,
                                 securityModel, securityLevel, notifyType,
                                 timeout, retries, &zeroLenBuf);
  printf("Buffdesc 0 length for filter argument 10:(result = %d)\n", result);
  printf("Testing openapiSnmpHostCreate() done.\n\n");


  printf("Testing openapiSnmpHostDelete():\n");
  result = openapiSnmpHostDelete(NULL, &host, notifyType);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpHostDelete(clientHandle, &badBufdescPointer, notifyType);
  printf("NULL host argument 2:(result = %d)\n", result);
  result = openapiSnmpHostDelete(clientHandle, &zeroLenBuf, notifyType);
  printf("Buffdesc 0 length for host argument 2:(result = %d)\n", result);
  result = openapiSnmpHostDelete(clientHandle, &host, OPEN_SNMP_NOTIFY_TYPE_TRAP - 1);
  printf("less than minimum value notify type argument 3:(result = %d)\n", result);
  result = openapiSnmpHostDelete(clientHandle, &host, OPEN_SNMP_NOTIFY_TYPE_INFORM + 1);
  printf("greater than maximum value notify type argument 3:(result = %d)\n", result);
  printf("Testing openapiSnmpHostDelete() done.\n\n");

  printf("Testing openapiSnmpHostGetNext():\n");
  result = openapiSnmpHostGetNext(NULL, &host, &serverPort, &securityName,
                                  &securityModel, &securityLevel, &notifyType,
                                  &timeout, &retries, &filterDesc, &nextHost);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpHostGetNext(clientHandle, &badBufdescPointer, &serverPort,
                                  &securityName, &securityModel, &securityLevel,
                                  &notifyType, &timeout, &retries, &filterDesc, &nextHost);
  printf("NULL host argument 2:(result = %d)\n", result);
  result = openapiSnmpHostGetNext(clientHandle, &zeroLenBuf, &serverPort, &securityName,
                                  &securityModel, &securityLevel, &notifyType,
                                  &timeout, &retries, &filterDesc, &nextHost);
  printf("Buffdesc 0 length for host argument 2:(result = %d)\n", result);
  result = openapiSnmpHostGetNext(clientHandle, &host, NULL, &securityName,
                                  &securityModel, &securityLevel, &notifyType,
                                  &timeout, &retries, &filterDesc, &nextHost);
  printf("NULL port argument 3:(result = %d)\n", result);
  result = openapiSnmpHostGetNext(clientHandle, &host, &serverPort, &badBufdescPointer,
                                  &securityModel, &securityLevel, &notifyType, &timeout,
                                  &retries, &filterDesc, &nextHost);
  printf("NULL security name argument 4:(result = %d)\n", result);
  result = openapiSnmpHostGetNext(clientHandle, &host, &serverPort, &zeroLenBuf,
                                  &securityModel, &securityLevel, &notifyType,
                                  &timeout, &retries, &filterDesc, &nextHost);
  printf("Buffdesc 0 length for security name argument 4:(result = %d)\n", result);
  result = openapiSnmpHostGetNext(clientHandle, &host, &serverPort, &securityName,
                                  NULL, &securityLevel, &notifyType, &timeout,
                                  &retries, &filterDesc, &nextHost);
  printf("NULL security model argument 5:(result = %d)\n", result);
  result = openapiSnmpHostGetNext(clientHandle, &host, &serverPort, &securityName,
                                  &securityModel, NULL, &notifyType, &timeout,
                                  &retries, &filterDesc, &nextHost);
  printf("NULL security level argument 6:(result = %d)\n", result);
  result = openapiSnmpHostGetNext(clientHandle, &host, &serverPort, &securityName,
                                  &securityModel, &securityLevel, NULL, &timeout,
                                  &retries, &filterDesc, &nextHost);
  printf("NULL notify type argument 7:(result = %d)\n", result);
  result = openapiSnmpHostGetNext(clientHandle, &host, &serverPort, &securityName,
                                  &securityModel, &securityLevel, &notifyType, NULL,
                                  &retries, &filterDesc, &nextHost);
  printf("NULL timeout argument 8:(result = %d)\n", result);
  result = openapiSnmpHostGetNext(clientHandle, &host, &serverPort, &securityName,
                                  &securityModel, &securityLevel, &notifyType, &timeout,
                                  NULL, &filterDesc, &nextHost);
  printf("NULL retries  argument 9:(result = %d)\n", result);
  result = openapiSnmpHostGetNext(clientHandle, &host, &serverPort, &securityName,
                                  &securityModel, &securityLevel, &notifyType, &timeout,
                                  &retries, &badBufdescPointer, &nextHost);
  printf("NULL filter argument 10:(result = %d)\n", result);
  result = openapiSnmpHostGetNext(clientHandle, &host, &serverPort, &securityName,
                                  &securityModel, &securityLevel, &notifyType, &timeout,
                                  &retries, &zeroLenBuf, &nextHost);
  printf("Buffdesc 0 length for filter argument 10:(result = %d)\n", result);
  result = openapiSnmpHostGetNext(clientHandle, &host, &serverPort, &securityName,
                                  &securityModel, &securityLevel, &notifyType, &timeout,
                                  &retries, &filterDesc, &badBufdescPointer);
  printf("NULL next host argument 11:(result = %d)\n", result);
  result = openapiSnmpHostGetNext(clientHandle, &host, &serverPort, &securityName,
                                  &securityModel, &securityLevel, &notifyType, &timeout,
                                  &retries, &filterDesc, &zeroLenBuf);
  printf("Buffdesc 0 length for next host argument 11:(result = %d)\n", result);
  printf("Testing openapiSnmpHostGetNext() done.\n\n");

  printf("Testing openapiSnmpTrapFlagGet():\n");
  result = openapiSnmpTrapFlagGet(NULL, trapFlag, &mode);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpTrapFlagGet(clientHandle, OPEN_SNMP_ACL_TRAP_FLAG - 1, &mode);
  printf("less than minimum value trap flag argument 2:(result = %d)\n", result);
  result = openapiSnmpTrapFlagGet(clientHandle, OPEN_SNMP_VIOLATION_TRAP_FLAG + 1, &mode);
  printf("greater than maximum value trap flag argument 2:(result = %d)\n", result);
  result = openapiSnmpTrapFlagGet(clientHandle, trapFlag, (void *)NULL);
  printf("NULL trap mode argument 3:(result = %d)\n", result);
  printf("Testing openapiSnmpTrapFlagGet() done.\n\n");

  printf("Testing openapiSnmpTrapFlagSet():\n");
  result = openapiSnmpTrapFlagSet(NULL, trapFlag, mode);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpTrapFlagSet(clientHandle, OPEN_SNMP_ACL_TRAP_FLAG - 1, mode);
  printf("less than minimum value trap flag argument 2:(result = %d)\n", result);
  result = openapiSnmpTrapFlagSet(clientHandle, OPEN_SNMP_VIOLATION_TRAP_FLAG + 1, mode);
  printf("greater than maximum value trap flag argument 2:(result = %d)\n", result);
  result = openapiSnmpTrapFlagSet(clientHandle, trapFlag, 3);
  printf("invalid trap mode argument 3:(result = %d)\n", result);
  printf("Testing openapiSnmpTrapFlagSet() done.\n\n");

  printf("Testing openapiSnmpTrapSrcIntfGet():\n");
  result = openapiSnmpTrapSrcIntfGet(NULL, &intf, &type, &ipAddr);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpTrapSrcIntfGet(clientHandle, NULL, &type, &ipAddr);
  printf("NULL source interface argument 1:(result = %d)\n", result);
  result = openapiSnmpTrapSrcIntfGet(clientHandle, &intf, NULL, &ipAddr);
  printf("NULL source interface type argument 2:(result = %d)\n", result);
  result = openapiSnmpTrapSrcIntfGet(clientHandle, &intf, &type, NULL);
  printf("NULL source address argument 3:(result = %d)\n", result);
  printf("Testing openapiSnmpTrapSrcIntfGet() done.\n\n");

  printf("Testing openapiSnmpTrapSrcIntfSet():\n");
  result = openapiSnmpTrapSrcIntfSet(NULL, intf);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpTrapSrcIntfSet(clientHandle, 0);
  printf("less than minimum value source interface argument 2:(result = %d)\n", result);
  result = openapiSnmpTrapSrcIntfSet(clientHandle, 123456788);
  printf("greater than maximum value source interface argument 2:(result = %d)\n", result);
  printf("Testing openapiSnmpTrapSrcIntfSet() done.\n\n");

  printf("Testing openapiSnmpViewCreate():\n");
  result = openapiSnmpViewCreate(NULL, &viewDesc, &oidDesc, viewType);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpViewCreate(clientHandle, &badBufdescPointer, &oidDesc, viewType);
  printf("NULL view name argument 2:(result = %d)\n", result);
  result = openapiSnmpViewCreate(clientHandle, &zeroLenBuf, &oidDesc, viewType);
  printf("Buffdesc 0 length for view name argument 2:(result = %d)\n", result);
  result = openapiSnmpViewCreate(clientHandle, &viewDesc, &badBufdescPointer, viewType);
  printf("NULL oid argument 3:(result = %d)\n", result);
  result = openapiSnmpViewCreate(clientHandle, &viewDesc, &zeroLenBuf, viewType);
  printf("Buffdesc 0 length for oid argument 3:(result = %d)\n", result);
  result = openapiSnmpViewCreate(clientHandle, &viewDesc, &oidDesc, (OPEN_SNMP_VIEW_TYPE_INCLUDED -1));
  printf("less than minimum value view type argument 4:(result = %d)\n", result);
  result = openapiSnmpViewCreate(clientHandle, &viewDesc, &oidDesc, (OPEN_SNMP_VIEW_TYPE_EXCLUDED +1));
  printf("greater than maximum value view type argument 3:(result = %d)\n", result);
  printf("Testing openapiSnmpViewCreate() done.\n\n");

  printf("Testing openapiSnmpViewDelete():\n");
  result = openapiSnmpViewDelete(NULL, &viewDesc, &oidDesc);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpViewDelete(clientHandle, &badBufdescPointer, &oidDesc);
  printf("NULL view name argument 2:(result = %d)\n", result);
  result = openapiSnmpViewDelete(clientHandle, &zeroLenBuf, &oidDesc);
  printf("Buffdesc 0 length for view name argument 2:(result = %d)\n", result);
  result = openapiSnmpViewDelete(clientHandle, &viewDesc, &badBufdescPointer);
  printf("NULL oid argument 3:(result = %d)\n", result);
  result = openapiSnmpViewDelete(clientHandle, &viewDesc, &zeroLenBuf);
  printf("Buffdesc 0 length for oid argument 3:(result = %d)\n", result);
  printf("Testing openapiSnmpViewDelete() done.\n\n");

  printf("Testing openapiSnmpViewGetNext():\n");
  result = openapiSnmpViewGetNext(NULL, &viewDesc, &oidDesc, &viewType, &nextViewName, &nextOidTree);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpViewGetNext(clientHandle, &badBufdescPointer, &oidDesc, &viewType, &nextViewName, &nextOidTree);
  printf("NULL view name argument 2:(result = %d)\n", result);
  result = openapiSnmpViewGetNext(clientHandle, &zeroLenBuf, &oidDesc, &viewType, &nextViewName, &nextOidTree);
  printf("Buffdesc 0 length for view name argument 2:(result = %d)\n", result);
  result = openapiSnmpViewGetNext(clientHandle, &viewDesc, &badBufdescPointer, &viewType, &nextViewName, &nextOidTree);
  printf("NULL oid argument 3:(result = %d)\n", result);
  result = openapiSnmpViewGetNext(clientHandle, &viewDesc, &zeroLenBuf, &viewType, &nextViewName, &nextOidTree);
  printf("Buffdesc 0 length for oid argument 3:(result = %d)\n", result);
  result = openapiSnmpViewGetNext(clientHandle, &viewDesc, &oidDesc, NULL, &nextViewName, &nextOidTree);
  printf("NULL view type argument 4:(result = %d)\n", result);
  result = openapiSnmpViewGetNext(clientHandle, &viewDesc, &oidDesc, &viewType, &badBufdescPointer, &nextOidTree);
  printf("NULL next view name argument 5:(result = %d)\n", result);
  result = openapiSnmpViewGetNext(clientHandle, &viewDesc, &oidDesc, &viewType, &zeroLenBuf, &nextOidTree);
  printf("Buffdesc 0 length for next view name argument 5:(result = %d)\n", result);
  result = openapiSnmpViewGetNext(clientHandle, &viewDesc, &oidDesc, &viewType, &nextViewName, &badBufdescPointer);
  printf("NULL next oid argument 6:(result = %d)\n", result);
  result = openapiSnmpViewGetNext(clientHandle, &viewDesc, &oidDesc, &viewType, &nextViewName, &zeroLenBuf);
  printf("Buffdesc 0 length for next oid argument 6:(result = %d)\n", result);
  printf("Testing openapiSnmpViewGetNext() done.\n\n");

  printf("Testing openapiSnmpTreeFamilyOidStringGet():\n");
  result = openapiSnmpTreeFamilyOidStringGet(NULL, &viewDesc, &oidDesc, treeType, &oidString);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpTreeFamilyOidStringGet(clientHandle, &badBufdescPointer, &oidDesc, treeType, &oidString);
  printf("NULL view name argument 2:(result = %d)\n", result);
  result = openapiSnmpTreeFamilyOidStringGet(clientHandle, &zeroLenBuf, &oidDesc, treeType, &oidString);
  printf("Buffdesc 0 length for view name argument 2:(result = %d)\n", result);
  result = openapiSnmpTreeFamilyOidStringGet(clientHandle, &viewDesc, &badBufdescPointer, treeType, &oidString);
  printf("NULL oid argument 3:(result = %d)\n", result);
  result = openapiSnmpTreeFamilyOidStringGet(clientHandle, &viewDesc, &zeroLenBuf, treeType, &oidString);
  printf("Buffdesc 0 length for oid argument 3:(result = %d)\n", result);
  result = openapiSnmpTreeFamilyOidStringGet(clientHandle, &viewDesc, &oidDesc, OPEN_SNMP_VIEW_TREE - 1, &oidString);
  printf("less than minimum value tree type argument 4:(result = %d)\n", result);
  result = openapiSnmpTreeFamilyOidStringGet(clientHandle, &viewDesc, &oidDesc, OPEN_SNMP_FILTER_TREE + 1, &oidString);
  printf("greater than maximum value tree type argument 4:(result = %d)\n", result);
  result = openapiSnmpTreeFamilyOidStringGet(clientHandle, &viewDesc, &oidDesc, treeType, &badBufdescPointer);
  printf("NULL next oid string argument 5:(result = %d)\n", result);
  result = openapiSnmpTreeFamilyOidStringGet(clientHandle, &viewDesc, &oidDesc, treeType, &zeroLenBuf);
  printf("Buffdesc 0 length for oid string argument 5:(result = %d)\n", result);
  printf("Testing openapiSnmpTreeFamilyOidStringGet() done.\n\n");

  printf("Testing openapiSnmpTreeFamilyOidValGet():\n");
  result = openapiSnmpTreeFamilyOidValGet(NULL, &viewDesc, &oidDesc, treeType, &oidVal);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpTreeFamilyOidValGet(clientHandle, &badBufdescPointer, &oidDesc, treeType, &oidVal);
  printf("NULL view name argument 2:(result = %d)\n", result);
  result = openapiSnmpTreeFamilyOidValGet(clientHandle, &zeroLenBuf, &oidDesc, treeType, &oidVal);
  printf("Buffdesc 0 length for view name argument 2:(result = %d)\n", result);
  result = openapiSnmpTreeFamilyOidValGet(clientHandle, &viewDesc, &badBufdescPointer, treeType, &oidVal);
  printf("NULL oid argument 3:(result = %d)\n", result);
  result = openapiSnmpTreeFamilyOidValGet(clientHandle, &viewDesc, &zeroLenBuf, treeType, &oidVal);
  printf("Buffdesc 0 length for oid argument 3:(result = %d)\n", result);
  result = openapiSnmpTreeFamilyOidValGet(clientHandle, &viewDesc, &oidDesc, OPEN_SNMP_VIEW_TREE - 1, &oidVal);
  printf("less than minimum value tree type argument 4:(result = %d)\n", result);
  result = openapiSnmpTreeFamilyOidValGet(clientHandle, &viewDesc, &oidDesc, OPEN_SNMP_FILTER_TREE + 1, &oidVal);
  printf("greater than maximum value tree type argument 4:(result = %d)\n", result);
  result = openapiSnmpTreeFamilyOidValGet(clientHandle, &viewDesc, &oidDesc, treeType, &badBufdescPointer);
  printf("NULL next oid value argument 5:(result = %d)\n", result);
  result = openapiSnmpTreeFamilyOidValGet(clientHandle, &viewDesc, &oidDesc, treeType, &zeroLenBuf);
  printf("Buffdesc 0 length for oid value argument 5:(result = %d)\n", result);
  printf("Testing openapiSnmpTreeFamilyOidValGet() done.\n\n");

  printf("Testing openapiSnmpSysContactGet():\n");
  result = openapiSnmpSysContactGet((void *)NULL, &sysContact);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpSysContactGet(clientHandle, &badBufdescPointer);
  printf("NULL sys contact argument 2:(result = %d)\n", result);
  result = openapiSnmpSysContactGet(clientHandle, &zeroLenBuf);
  printf("Buffdesc 0 length for sys contact argument 2:(result = %d)\n", result);
  printf("Testing openapiSnmpSysContactGet() done.\n\n");

  printf("Testing openapiSnmpSysContactSet():\n");
  result = openapiSnmpSysContactSet((void *)NULL, &sysContact);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpSysContactSet(clientHandle, &badBufdescPointer);
  printf("NULL sys contact argument 2:(result = %d)\n", result);
  result = openapiSnmpSysContactSet(clientHandle, &zeroLenBuf);
  printf("Buffdesc 0 length for sys contact argument 2:(result = %d)\n", result);
  printf("Testing openapiSnmpSysContactSet() done.\n\n");

  printf("Testing openapiSnmpSysNameGet():\n");
  result = openapiSnmpSysNameGet((void *)NULL, &sysName);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpSysNameGet(clientHandle, &badBufdescPointer);
  printf("NULL sys name argument 2:(result = %d)\n", result);
  result = openapiSnmpSysNameGet(clientHandle, &zeroLenBuf);
  printf("Buffdesc 0 length for sys name argument 2:(result = %d)\n", result);
  printf("Testing openapiSnmpSysNameGet() done.\n\n");

  printf("Testing openapiSnmpSysNameSet():\n");
  result = openapiSnmpSysNameSet((void *)NULL, &sysName);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpSysNameSet(clientHandle, &badBufdescPointer);
  printf("NULL sys name id argument 2:(result = %d)\n", result);
  result = openapiSnmpSysNameSet(clientHandle, &zeroLenBuf);
  printf("Buffdesc 0 length for sys name argument 2:(result = %d)\n", result);
  printf("Testing openapiSnmpSysNameSet() done.\n\n");

  printf("Testing openapiSnmpSysLocationGet():\n");
  result = openapiSnmpSysLocationGet((void *)NULL, &sysLocation);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpSysLocationGet(clientHandle, &badBufdescPointer);
  printf("NULL sys location argument 2:(result = %d)\n", result);
  result = openapiSnmpSysLocationGet(clientHandle, &zeroLenBuf);
  printf("Buffdesc 0 length for sys location argument 2:(result = %d)\n", result);
  printf("Testing openapiSnmpSysLocationGet() done.\n\n");

  printf("Testing openapiSnmpSysLocationSet():\n");
  result = openapiSnmpSysLocationSet(NULL, &sysLocation);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpSysLocationSet(clientHandle, (void *)NULL);
  printf("NULL sys location argument 2:(result = %d)\n", result);
  result = openapiSnmpSysLocationSet(clientHandle, (void *)NULL);
  printf("Buffdesc 0 length for sys location 2:(result = %d)\n", result);
  printf("Testing openapiSnmpSysLocationSet() done.\n\n");

  printf("Testing openapiSnmpCommunityAndIPAddrCreate():\n");
  result = openapiSnmpCommunityAndIPAddrCreate(NULL, &communityDesc, community);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpCommunityAndIPAddrCreate(clientHandle, &badBufdescPointer, community);
  printf("NULL  community name argument 2:(result = %d)\n", result);
  result = openapiSnmpCommunityAndIPAddrCreate(clientHandle, &zeroLenBuf, community);
  printf("Buffdesc 0 length for community name  argument 2:(result = %d)\n", result);
  result = openapiSnmpCommunityAndIPAddrCreate(clientHandle, &communityDesc, community);
  printf("NULL community(OPEN_SNMP_COMMUNITY_t) argument 3:(result = %d)\n", result);
  printf("Testing openapiSnmpCommunityAndIPAddrCreate() done.\n\n");

  printf("Testing openapiSnmpCommunityAndIPAddrDelete():\n");
  result = openapiSnmpCommunityAndIPAddrDelete(NULL, &communityDesc);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpCommunityAndIPAddrDelete(clientHandle, &badBufdescPointer);
  printf("NULL  community name argument 2:(result = %d)\n", result);
  result = openapiSnmpCommunityAndIPAddrDelete(clientHandle, &zeroLenBuf);
  printf("Buffdesc 0 length for community name  argument 2:(result = %d)\n", result);
  printf("Testing openapiSnmpCommunityAndIPAddrDelete() done.\n\n");

  printf("Testing openapiSnmpCommunityAndIPAddrGet():\n");
  result = openapiSnmpCommunityAndIPAddrGet(NULL, &communityDesc, &community);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpCommunityAndIPAddrGet(clientHandle, &badBufdescPointer, &community);
  printf("NULL  community name argument 2:(result = %d)\n", result);
  result = openapiSnmpCommunityAndIPAddrGet(clientHandle, &zeroLenBuf, &community);
  printf("Buffdesc 0 length for community name  argument 2:(result = %d)\n", result);
  result = openapiSnmpCommunityAndIPAddrGet(clientHandle, &communityDesc, &community);
  printf("NULL community(OPEN_SNMP_COMMUNITY_t) argument 3:(result = %d)\n", result);
  printf("Testing openapiSnmpCommunityAndIPAddrGet() done.\n\n");

  printf("Testing openapiSnmpCommunityAndIPAddrGetNext():\n");
  result = openapiSnmpCommunityAndIPAddrGetNext(NULL, &communityDesc, &nextCommunityDesc);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpCommunityAndIPAddrGetNext(clientHandle, &badBufdescPointer, &nextCommunityDesc);
  printf("NULL  community name argument 2:(result = %d)\n", result);
  result = openapiSnmpCommunityAndIPAddrGetNext(clientHandle, &zeroLenBuf, &nextCommunityDesc);
  printf("Buffdesc 0 length for community name  argument 2:(result = %d)\n", result);
  result = openapiSnmpCommunityAndIPAddrGetNext(clientHandle, &communityDesc, &badBufdescPointer);
  printf("NULL  community name argument 3:(result = %d)\n", result);
  result = openapiSnmpCommunityAndIPAddrGetNext(clientHandle, &communityDesc, &zeroLenBuf);
  printf("Buffdesc 0 length for community name  argument 3:(result = %d)\n", result);
  printf("Testing openapiSnmpCommunityAndIPAddrGetNext() done.\n\n");

  printf("Testing openapiSnmpUserConfigGet():\n");
  result = openapiSnmpUserConfigGet(NULL, &userName, &engineId, &groupDesc,
                                    &authProto, &authKey, &privProto, &privKey);
  printf("NULL client handle argument 1:(result = %d)\n", result);
  result = openapiSnmpUserConfigGet(clientHandle, &badBufdescPointer, &engineId, &groupDesc,
                                    &authProto, &authKey, &privProto, &privKey);
  printf("NULL user name argument 2:(result = %d)\n", result);
  result = openapiSnmpUserConfigGet(clientHandle, &zeroLenBuf, &engineId, &groupDesc,
                                    &authProto, &authKey, &privProto, &privKey);
  printf("Buffdesc 0 length for user name argument 2:(result = %d)\n", result);
  result = openapiSnmpUserConfigGet(clientHandle, &userName, &badBufdescPointer, &groupDesc,
                                    &authProto, &authKey, &privProto, &privKey);
  printf("NULL engine id argument 3:(result = %d)\n", result);
  result = openapiSnmpUserConfigGet(clientHandle, &userName, &zeroLenBuf, &groupDesc,
                                    &authProto, &authKey, &privProto, &privKey);
  printf("Buffdesc 0 length for engine id argument 3:(result = %d)\n", result);
  result = openapiSnmpUserConfigGet(clientHandle, &userName, &engineId, &badBufdescPointer,
                                    &authProto, &authKey, &privProto, &privKey);
  printf("NULL group name argument 4:(result = %d)\n", result);
  result = openapiSnmpUserConfigGet(clientHandle, &userName, &engineId, &zeroLenBuf,
                                    &authProto, &authKey, &privProto, &privKey);
  printf("Buffdesc 0 length for group name argument 4:(result = %d)\n", result);
  result = openapiSnmpUserConfigGet(clientHandle, &userName, &engineId, &groupDesc, 
                                    NULL, &authKey, &privProto, &privKey);
  printf("NULL auth protocol argument 5:(result = %d)\n", result);
  result = openapiSnmpUserConfigGet(clientHandle, &userName, &engineId, &groupDesc,
                                    &authProto, &badBufdescPointer, &privProto, &privKey);
  printf("NULL auth key argument 6:(result = %d)\n", result);
  result = openapiSnmpUserConfigGet(clientHandle, &userName, &engineId, &groupDesc,
                                    &authProto, &zeroLenBuf, &privProto, &privKey);
  printf("Buffdesc 0 length for auth key argument 6:(result = %d)\n", result);
  result = openapiSnmpUserConfigGet(clientHandle, &userName, &engineId, &groupDesc,
                                    &authProto, &authKey, NULL, &privKey);
  printf("NULL priv protocol argument 7:(result = %d)\n", result);
  result = openapiSnmpUserConfigGet(clientHandle, &userName, &engineId, &groupDesc,
                                  &authProto, &authKey, &privProto, &badBufdescPointer);
  printf("NULL priv key argument 8:(result = %d)\n", result);
  result = openapiSnmpUserConfigGet(clientHandle, &userName, &engineId, &groupDesc,
                                  &authProto, &authKey, &privProto, &zeroLenBuf);
  printf("Buffdesc 0 length for priv key argument 8:(result = %d)\n", result);
  printf("Testing openapiSnmpUserConfigGet() done.\n\n");

  printf("SNMP OpEN APIs sanity done.\n");

  return;
}

/*********************************************************************
 * @purpose  Get the Trap Flag type for SNMP
 *
 * @param    type  @b{(input)}  Trap flag Type
 * 
 * @returns The Trap Flag Type Description
 *
 * @notes
 *
 * @end
 *********************************************************************/
char* getTrapFlagType(OPEN_SNMP_TRAP_FLAGS_t type)
{
  switch (type)
  {
    case OPEN_SNMP_ACL_TRAP_FLAG:
      return "ACL";
    case OPEN_SNMP_BOXS_FAN_STATUS_TRAP_FLAG:   
      return "Fan status";
    case OPEN_SNMP_BOXS_POWER_STATUS_TRAP_FLAG:
      return "Power supply status";
    case OPEN_SNMP_BOXS_TEMP_STATUS_TRAP_FLAG:
      return "Temperature status";
    case OPEN_SNMP_BGP_TRAP_FLAG:
      return "BGP";
    case OPEN_SNMP_LINK_MODE_TRAP_FLAG:
      return "Link up/down";
    case OPEN_SNMP_MULTI_USERS_TRAP_FLAG:
      return "multiple logins";
    case OPEN_SNMP_OSPFV2_TRAP_FLAG:
      return "OSPFv2";
    case OPEN_SNMP_OSPFV3_TRAP_FLAG:
      return "OSPFv3";
    case OPEN_SNMP_STP_MODE_TRAP_FLAG:
      return "Spanning Tree";
    case OPEN_SNMP_USER_AUTH_TRAP_FLAG:
      return "User authentication";
    case OPEN_SNMP_VIOLATION_TRAP_FLAG:   
      return "Port MAC locking security violations";
  }
  return "";
}

/*********************************************************************
 * @purpose  Get the filter type for SNMP
 *
 * @param    type  @b{(input)}  Filter Type
 * 
 * @returns The Filter Type Description
 *
 * @notes
 *
 * @end
 *********************************************************************/
char* getFilterType(OPEN_SNMP_NOTIFY_FILTER_TYPE_t type)
{
  switch (type)
  {
    case OPEN_SNMP_NOTIFY_FILTER_TYPE_INCLUDED:
      return "Included";
    case OPEN_SNMP_NOTIFY_FILTER_TYPE_EXCLUDED:
      return "Excluded";
  }
  return "";
}

/*********************************************************************
 * @purpose  Get the view type for SNMP
 *
 * @param    type  @b{(input)}  View Type
 * 
 * @returns The View Type Description
 *
 * @notes
 *
 * @end
 *********************************************************************/
char* getViewType(OPEN_SNMP_VIEW_TYPE_t type)
{
  switch (type)
  {
    case OPEN_SNMP_VIEW_TYPE_INCLUDED:
      return "Included";
    case OPEN_SNMP_VIEW_TYPE_EXCLUDED:
      return "Excluded";
  }
  return "";
}

/*********************************************************************
 * @purpose  Get the SNMP security level
 *
 * @param    secLevel  @b{(input)} security level
 * 
 * @returns The security level Description
 *
 * @notes
 *
 * @end
 *********************************************************************/
char* getSecurityLevel(OPEN_SNMP_SECURITY_LEVEL_t secLevel)
{
  switch (secLevel)
  {
    case OPEN_SNMP_SECURITY_LEVEL_NOAUTHNOPRIV:
      return "NoAuthNoPriv";
    case OPEN_SNMP_SECURITY_LEVEL_AUTHNOPRIV:
      return "AuthNoPriv";
    case OPEN_SNMP_SECURITY_LEVEL_AUTHPRIV:
      return "AuthPriv";
  }
  return "";
}
/*********************************************************************
 * @purpose  Get the SNMP version
 *
 * @param    secModel  @b{(input)} version
 * 
 * @returns The version Description
 *
 * @notes
 *
 * @end
 *********************************************************************/
char* getVersion(OPEN_SNMP_SECURITY_MODEL_t secModel)
{
  switch (secModel)
  {
    case OPEN_SNMP_SECURITY_MODEL_ANY:
      return "Any";
    case OPEN_SNMP_SECURITY_MODEL_SNMPV1:
      return "SNMPv1";
    case OPEN_SNMP_SECURITY_MODEL_SNMPV2C:
      return "SNMPv2c";
    case OPEN_SNMP_SECURITY_MODEL_USM:
      return "SNMPv3";
  }
  return "";
}

/*********************************************************************
 * @purpose  Get the Notify type for SNMP hosts
 *
 * @param    type  @b{(input)}  Notify Type
 * 
 * @returns The Notify Type Description
 *
 * @notes
 *
 * @end
 *********************************************************************/
char* getNotifyType(OPEN_SNMP_NOTIFY_TYPE_t type)
{
  switch (type)
  {
    case OPEN_SNMP_NOTIFY_TYPE_TRAP:
      return "Trap";
    case OPEN_SNMP_NOTIFY_TYPE_INFORM:
      return "Inform";
  }
  return "";
}

/*********************************************************************
 * @purpose  Get the Access type for community
 *
 * @param    type  @b{(input)}  Access Type
 * 
 * @returns The Access Type Description
 *
 * @notes
 *
 * @end
 *********************************************************************/
char* getAccessType(OPEN_SNMP_COMMUNITY_ACCESS_TYPE_t type)
{
  switch (type)
  {
    case OPEN_SNMP_COMMUNITY_ACCESS_NONE:
      return "None";
    case OPEN_SNMP_COMMUNITY_ACCESS_RO:
      return "Read-Only";
    case OPEN_SNMP_COMMUNITY_ACCESS_RW:
      return "Read-Write";
    case OPEN_SNMP_COMMUNITY_ACCESS_SU:
      return "Super-User";
  }
  return "";
}

/*********************************************************************
 * @purpose  Get the authentication type for SNMPv3 user
 *
 * @param    type  @b{(input)}  Authentication Type
 * 
 * @returns The Authentication Type Description
 *
 * @notes
 *
 * @end
 *********************************************************************/
char* getAuthType(OPEN_USM_USER_AUTH_PROTOCOL_t type)
{
  switch (type)
  {
    case OPEN_USM_USER_AUTH_PROTOCOL_NONE:
      return "None";
    case OPEN_USM_USER_AUTH_PROTOCOL_SHA:
      return "SHA";
    case OPEN_USM_USER_AUTH_PROTOCOL_SHA512:
      return "SHA512";
    case OPEN_USM_USER_AUTH_PROTOCOL_MD5:
      return "MD5";
  }
  return "";
}

/*********************************************************************
 * @purpose  Get the encryption type for SNMPv3 user      
 *
 * @param    type  @b{(input)} Encryption Type
 * 
 * @returns The Encryption Type Description
 *
 * @notes
 *
 * @end
 *********************************************************************/
char* getPrivType(OPEN_USM_USER_PRIV_PROTOCOL_t type)
{
  switch (type)
  {
    case OPEN_USM_USER_PRIV_PROTOCOL_NONE:
      return "None";
    case OPEN_USM_USER_PRIV_PROTOCOL_DES:
      return "DES";
    case OPEN_USM_USER_PRIV_PROTOCOL_3DES:
      return "3DES";
    case OPEN_USM_USER_PRIV_PROTOCOL_AES128:
      return "AES128";
    case OPEN_USM_USER_PRIV_PROTOCOL_AES192:
      return "AES192";
    case OPEN_USM_USER_PRIV_PROTOCOL_AES256:
      return "AES256";
  }
  return "";
}

/*********************************************************************
* @purpose  Get the interface description for the particular interface type
*          
* @param    type  @b{(input)}  Interface Type
* 
* @returns The Interface Type Description
*
* @notes
*
* @end
*********************************************************************/
char* getInterfaceType(uint32_t type)
{
  switch (type)
  {
  case OPEN_INTF_TYPE_PHY:
    return "Port";
  case OPEN_INTF_TYPE_VLAN:
    return "VLAN";
  case OPEN_INTF_TYPE_LOOPBACK:
    return "Loop";
  case OPEN_INTF_TYPE_TUNNEL:
    return "Tnnl";
  }
  return "";
}

/*****************************************************************************
* @purpose  Delete the SNMP engine id on the local device. The engine id is
*           actually reset to the default id which is automatically generated
*           based on the local MAC address.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
****************************************************************************/
void testSnmpLocalEngineIdDelete(openapiClientHandle_t *clientHandle)
{
  open_error_t result = OPEN_E_NONE;

  result = openapiSnmpLocalEngineIdDelete(clientHandle);

  if (OPEN_E_NONE == result)
  {
    printf("\nSNMP Engine ID is deleted successfully.\n");
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/*****************************************************************************
* @purpose  Get the SNMP engine id on the local device.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    engineId            @b{(output)} local engine id for comparison
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes
* 
* @end
****************************************************************************/
void testSnmpLocalEngineIdGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc desc;
  char str[32+1];

  memset(str, 0, sizeof(str));
  desc.pstart = str;
  desc.size = sizeof(str);

  result = openapiSnmpLocalEngineIdGet(clientHandle, &desc);
  if (OPEN_E_NONE == result)
  {
    unsigned char *ptr = desc.pstart;
    uint32_t idx;

    if (fVerbose)
    {
      printf("Engine Id : ");
      for (idx=0; idx<desc.size; idx++)
      {
        printf("%02x", (unsigned char) *ptr++);
      }
      printf("\n");
    }
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/*****************************************************************************
* @purpose  Set the SNMP engine id on the local device.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    engineId            @b{(input)}  local engine id
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
****************************************************************************/
void testSnmpLocalEngineIdSet(openapiClientHandle_t *clientHandle,
                              char *engineId)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc desc;
  char str[32+1];

  if (strcmp(engineId, "default") == 0) { strcpy(str, "1"); }
  else { strcpy(str, engineId); }

  desc.pstart = str;
  desc.size = strlen(str) + 1;

  result = openapiSnmpLocalEngineIdSet(clientHandle, &desc);
  if (OPEN_E_NONE == result)
  {
    printf("\nSNMP Engine Id '%s' is configured successfully.\n", engineId);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }
  
  return;
}

/*****************************************************************************
* @purpose  Get the system name.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    sysName             @b{(output)} system name
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
*
* @notes
* 
* @end
****************************************************************************/
void testSnmpSysNameGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc desc;
  char str[255+1];

  memset(str, 0, sizeof(str));
  desc.pstart = str;
  desc.size = sizeof(str);

  result = openapiSnmpSysNameGet(clientHandle, &desc);
  if (OPEN_E_NONE == result)
  {
    printf("\nSystem Name:%s \n", str);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/*****************************************************************************
* @purpose  Set the SNMP system name. An administratively-assigned name for
*           this managed node. By convention, this is the node's fully-qualified
*           domain name.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    sysName             @b{(input)}  system name
*
* @returns  OPEN_E_NONE         Set is successful
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
****************************************************************************/
void testSnmpSysNameSet(openapiClientHandle_t *clientHandle, 
                                char *sysName)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc desc;
  char str[255+1];

  strcpy(str, sysName);
  desc.pstart = str;
  desc.size = strlen(str) + 1;

  result = openapiSnmpSysNameSet(clientHandle, &desc);
  if (OPEN_E_NONE == result)
  {
    printf("\nSystem name '%s' is configured successfully.\n", sysName);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/*****************************************************************************
* @purpose  Get the system contact information.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    sysContact          @b{(output)} system contact
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
*
* @notes
* 
* @end
****************************************************************************/
void testSnmpSysContactGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc desc;
  char str[255+1];

  memset(str, 0, sizeof(str));
  desc.pstart = str;
  desc.size = sizeof(str);

  result = openapiSnmpSysContactGet(clientHandle, &desc);
  if (OPEN_E_NONE == result)
  {
    printf("\nSystem Contact:%s \n", str);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/*****************************************************************************
* @purpose  Set the SNMP system contact. The textual identification of the
*           contact person for this managed node, together with information
*           on how to contact this person.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    sysContact          @b{(input)}  system contact
*
* @returns  OPEN_E_NONE         Set is successful
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
****************************************************************************/
void testSnmpSysContactSet(openapiClientHandle_t *clientHandle, 
                           char *sysContact)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc desc;
  char str[255+1];

  strcpy(str, sysContact);
  desc.pstart = str;
  desc.size = strlen(str) + 1;

  result = openapiSnmpSysContactSet(clientHandle, &desc);
  if (OPEN_E_NONE == result)
  {
    printf("\nSystem contact '%s' is configured successfully.\n", sysContact);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/*****************************************************************************
* @purpose  Get the system location information.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    sysLocation         @b{(output)} system location
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
*
* @notes
* 
* @end
****************************************************************************/
void testSnmpSysLocationGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc desc;
  char str[255+1];

  memset(str, 0, sizeof(str));
  desc.pstart = str;
  desc.size = sizeof(str);

  result = openapiSnmpSysLocationGet(clientHandle, &desc);
  if (OPEN_E_NONE == result)
  {
    printf("\nSystem Location:%s \n", str);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/*****************************************************************************
* @purpose  Set the SNMP system location. The physical location of this node,
*           for example; 'telephone closet' or '3rd floor'.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    sysLocation         @b{(input)}  system location
*
* @returns  OPEN_E_NONE         Set is successful
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
****************************************************************************/
void testSnmpSysLocationSet(openapiClientHandle_t *clientHandle, 
                            char *sysLocation)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc desc;
  char str[255+1];

  strcpy(str, sysLocation);
  desc.pstart = str;
  desc.size = strlen(str) + 1;

  result = openapiSnmpSysLocationSet(clientHandle, &desc);
  if (OPEN_E_NONE == result)
  {
    printf("\nSystem location '%s' is configured successfully.\n", sysLocation);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/*********************************************************************
* @purpose  Create a new SNMP user
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* @param    userName            @b{(input)}  user name
* @param    groupName           @b{(input)}  group name
* @param    authProto           @b{(input)}  authentication protocol type
* @param    authPwd             @b{(input)}  HMAC-MD5-96 or HMAC-SHA-96 authentication password
* @param    authKey             @b{(input)}  pre-generated key
* @param    privProto           @b{(input)}  privilege protocol type
* @param    privPwd             @b{(input)}  CBC-DES symmetric encryption password
* @param    privKey             @b{(input)}  pre-generated key
* @param    engineId            @b{(input)}  remote engine id
* 
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void testSnmpUserCreate(openapiClientHandle_t *clientHandle, 
                        char *userName,
                        char *groupName,
                        OPEN_USM_USER_AUTH_PROTOCOL_t authProto,
                        char *authPwd,
                        char *authKey,
                        OPEN_USM_USER_PRIV_PROTOCOL_t privProto,
                        char *privPwd,
                        char *privKey,
                        char *engineId)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc userDesc;
  open_buffdesc groupDesc;
  open_buffdesc aPwdDesc;
  open_buffdesc aKeyDesc;
  open_buffdesc pPwdDesc;
  open_buffdesc pKeyDesc;
  open_buffdesc engineDesc;
  char userNameStr[30+1];
  char gStr[30+1];
  char aPwdStr[32+1];
  char aKeyStr[128+1];
  char pPwdStr[32+1];
  char pKeyStr[128+1];
  char eStr[32+1];

  strcpy(userNameStr, userName);
  userDesc.pstart = userNameStr;
  userDesc.size = strlen(userNameStr) + 1;

  strcpy(gStr, groupName);
  groupDesc.pstart = gStr;
  groupDesc.size = strlen(gStr) + 1;

  strcpy(aPwdStr, authPwd);
  aPwdDesc.pstart = aPwdStr;
  aPwdDesc.size = strlen(aPwdStr) + 1;

  strcpy(aKeyStr, authKey);
  aKeyDesc.pstart = aKeyStr;
  aKeyDesc.size = strlen(aKeyStr) + 1;

  strcpy(pPwdStr, privPwd);
  pPwdDesc.pstart = pPwdStr;
  pPwdDesc.size = strlen(pPwdStr) + 1;

  strcpy(pKeyStr, privKey);
  pKeyDesc.pstart = pKeyStr;
  pKeyDesc.size = strlen(pKeyStr) + 1;

  strcpy(eStr, engineId);
  engineDesc.pstart = eStr;
  engineDesc.size = strlen(eStr) + 1;

  result = openapiSnmpUserCreate(clientHandle,
                                 &userDesc, &groupDesc,
                                 authProto, &aPwdDesc, &aKeyDesc,
                                 privProto, &pPwdDesc, &pKeyDesc,
                                 &engineDesc);
  if (OPEN_E_NONE == result)
  {
    printf("\nSNMPv3 user '%s' is configured successfully.\n", userName);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/*********************************************************************
* @purpose  Delete a new SNMP user
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* @param    userName            @b{(input)}  user name
* @param    engineId            @b{(input)}  remote engine id
* 
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void testSnmpUserDelete(openapiClientHandle_t *clientHandle, 
                        char *userName,
                        char *engineId)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc userDesc;
  open_buffdesc engineDesc;
  char userNameStr[30+1];
  char eStr[32+1];

  strcpy(userNameStr, userName);
  userDesc.pstart = userNameStr;
  userDesc.size = strlen(userNameStr) + 1;

  strcpy(eStr, engineId);
  engineDesc.pstart = eStr;
  engineDesc.size = strlen(eStr) + 1;

  result = openapiSnmpUserDelete(clientHandle,
                                 &userDesc, 
                                 &engineDesc);
  if (OPEN_E_NONE == result)
  {
    printf("\nSNMPv3 user '%s' is deleted successfully.\n", userName);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/*********************************************************************
* @purpose  Demonstrate how to loop through and retrieve the existing
*           SNMP users. This function simply stops at the first userName
*           match and displays its contents.
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* @param    userName            @b{(input)}  user name
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes
* 
* @end
*********************************************************************/
void testSnmpUserGet(openapiClientHandle_t *clientHandle, 
                     char *userName)
{
  OPEN_USM_USER_AUTH_PROTOCOL_t authProto;
  OPEN_USM_USER_PRIV_PROTOCOL_t privProto;
  open_buffdesc userDesc;
  open_buffdesc engineDesc;
  open_buffdesc groupDesc;
  open_buffdesc nextUserDesc;
  open_buffdesc nextEngineDesc;
  char userNameStr[30+1];
  char groupNameStr[30+1];
  char engineStr[32];
  char nextUserNameStr[30+1];
  char nextEngineStr[32];
  bool fFound = false;
  bool displayAllEntries = false;
  uint32_t idx;
  open_error_t result;
  char authKeyStr[128+1];
  char privKeyStr[128+1];
  char engineIdStr[64];
  char engine[32];
  open_buffdesc authKey;
  open_buffdesc privKey;
  open_buffdesc engineIdDesc;
  unsigned char *ptr = NULL;

  if ((NULL != userName) && 
      (0 == strlen(userName)))
  {
    displayAllEntries = true;
  }

  memset(userNameStr, 0, sizeof(userNameStr));
  userDesc.pstart = userNameStr;
  userDesc.size = sizeof(userNameStr);

  memset(groupNameStr, 0, sizeof(groupNameStr));
  groupDesc.pstart = groupNameStr;
  groupDesc.size = sizeof(groupNameStr);

  memset(engineStr, 0, sizeof(engineStr));
  engineDesc.pstart = engineStr;
  engineDesc.size = sizeof(engineStr);

  memset(nextUserNameStr, 0, sizeof(nextUserNameStr));
  nextUserDesc.pstart = nextUserNameStr;
  nextUserDesc.size = sizeof(nextUserNameStr);

  memset(nextEngineStr, 0, sizeof(nextEngineStr));
  nextEngineDesc.pstart = nextEngineStr;
  nextEngineDesc.size = sizeof(nextEngineStr);

  memset(engineIdStr, 0, sizeof(engineIdStr));
  memset(engine, 0, sizeof(engine));
  engineIdDesc.pstart = engineIdStr;
  engineIdDesc.size = sizeof(engineIdStr);

  while (openapiSnmpUserGetNext(clientHandle,
                                &userDesc, 
                                &engineDesc, 
                                &groupDesc,
                                &authProto, 
                                &privProto,
                                &nextUserDesc, 
                                &nextEngineDesc) == OPEN_E_NONE)
  {
    memset(authKeyStr, 0, sizeof(authKeyStr));
    authKey.pstart = authKeyStr;
    authKey.size = sizeof(authKeyStr);

    memset(privKeyStr, 0, sizeof(privKeyStr));
    privKey.pstart = privKeyStr;
    privKey.size = sizeof(privKeyStr);

    memset(engineIdStr, 0, sizeof(engineIdStr));
    engineIdDesc.size = sizeof(engineIdStr);

    memset(groupNameStr, 0, sizeof(groupNameStr));
    groupDesc.pstart = groupNameStr;
    groupDesc.size = sizeof(groupNameStr);

    ptr = nextEngineDesc.pstart;

    for (idx=0; idx<nextEngineDesc.size; idx++)
    {
      memset(engine, 0x0, sizeof(engine));
      snprintf(engine, sizeof(engine), "%02x", (unsigned char) *ptr++);
      strncat(engineIdStr, engine, (sizeof(engineIdStr)-strlen(engineIdStr)-1));
    }
    engineIdDesc.size = strlen(engineIdStr) + 1;
    
    result = openapiSnmpUserConfigGet(clientHandle,&nextUserDesc,
                                      &engineIdDesc, &groupDesc, &authProto,
                                      &authKey, &privProto, &privKey);
    
    if (strcmp(userName, nextUserDesc.pstart)==0)
    {
      fFound = true;
      break;
    }
    
    if (true == displayAllEntries)
    {
      ptr = nextEngineDesc.pstart;
      printf("\n");
      if (fVerbose)
      {
        printf("User name      : %s\n"
               "Group name     : %s\n"
               "Authentication : %d (%s)\n"
               "Privilege      : %d (%s)\n"
               "Engine Id      : ",
               (char *) nextUserDesc.pstart,
               (char *) groupDesc.pstart,
               authProto, getAuthType(authProto),
               privProto, getPrivType(privProto));
        for (idx=0; idx<nextEngineDesc.size; idx++)
        {
          printf("%02x", (unsigned char) *ptr++);
        }

        if (OPEN_E_NONE == result)
        {
          printf("\nAuthentication Key      : %s\n"
                 "Privilege Key           : %s\n",
                 (char *) authKey.pstart,
                 (char *) privKey.pstart);
        }

        printf("\n");
      }
    }

    /* Setup for next iteration */
    memset(userNameStr, 0, sizeof(userNameStr));
    strncpy(userDesc.pstart, nextUserDesc.pstart, nextUserDesc.size);
    userDesc.size = nextUserDesc.size;

    memset(engineStr, 0, sizeof(engineStr));
    memcpy(engineDesc.pstart, nextEngineDesc.pstart, nextEngineDesc.size);
    engineDesc.size = nextEngineDesc.size;

    /* Necessary to reset size for next user record */
    memset(nextEngineStr, 0, sizeof(nextEngineStr));
    memset(groupNameStr, 0, sizeof(groupNameStr));
    memset(nextUserNameStr, 0, sizeof(nextUserNameStr));
    groupDesc.size = sizeof(groupNameStr);
    nextUserDesc.size = sizeof(nextUserNameStr);
    nextEngineDesc.size = sizeof(nextEngineStr);
  }

  if (fFound)
  {
    ptr = nextEngineDesc.pstart;

    if (fVerbose)
    {
      printf("User name      : %s\n"
             "Group name     : %s\n"
             "Authentication : %d (%s)\n"
             "Privilege      : %d (%s)\n"
             "Engine Id      : ",
             (char *) nextUserDesc.pstart,
             (char *) groupDesc.pstart,
             authProto, getAuthType(authProto),
             privProto, getPrivType(privProto));
      for (idx=0; idx<nextEngineDesc.size; idx++)
      {
        printf("%02x", (unsigned char) *ptr++);
      }

      if (OPEN_E_NONE == result)
      {
        printf("\nAuthentication Key      : %s\n"
               "Privilege Key           : %s\n",
               (char *) authKey.pstart,
               (char *) privKey.pstart);
      }

      printf("\n");
    }

  }

  return;
}

/*********************************************************************
* @purpose  Create a new SNMP group.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    groupName           @b{(input)}  group name
* @param    securityModel       @b{(input)}  version number
* @param    securityLevel       @b{(input)}  security level (valid only for securityModel @ref OPEN_SNMP_SECURITY_MODEL_USM)
* @param    contextPrefix       @b{(input)}  optional prefix name to associate the group with
* @param    readView            @b{(input)}  optional, enables user to view the contents of the agent
* @param    writeView           @b{(input)}  optional, enables user to enter data and configure the contents of the agent
* @param    notifyView          @b{(input)}  optional, enables user to specify an inform or a trap
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_EXITS        if SNMP group already exists
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch.
* 
* @end
*********************************************************************/
void testSnmpGroupCreate(openapiClientHandle_t *clientHandle, 
                         char *groupName,
                         OPEN_SNMP_SECURITY_MODEL_t securityModel,
                         OPEN_SNMP_SECURITY_LEVEL_t securityLevel,
                         char *contextPrefix,
                         char *readView,
                         char *writeView,
                         char *notifyView)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc groupDesc;
  open_buffdesc contextDesc;
  open_buffdesc readDesc;
  open_buffdesc writeDesc;
  open_buffdesc notifyDesc;
  char groupStr[30+1];
  char contextStr[30+1];
  char readStr[30+1];
  char writeStr[30+1];
  char notifyStr[30+1];

  strcpy(groupStr, groupName);
  groupDesc.pstart = groupStr;
  groupDesc.size = strlen(groupStr) + 1;

  strcpy(contextStr, contextPrefix);
  contextDesc.pstart = contextStr;
  contextDesc.size = strlen(contextStr) + 1;

  strcpy(readStr, readView);
  readDesc.pstart = readStr;
  readDesc.size = strlen(readStr) + 1;

  strcpy(writeStr, writeView);
  writeDesc.pstart = writeStr;
  writeDesc.size = strlen(writeStr) + 1;

  strcpy(notifyStr, notifyView);
  notifyDesc.pstart = notifyStr;
  notifyDesc.size = strlen(notifyStr) + 1;

  result = openapiSnmpGroupCreate(clientHandle,
                                  &groupDesc, securityModel, securityLevel,
                                  &contextDesc, &readDesc, &writeDesc, &notifyDesc);
  if (OPEN_E_NONE == result)
  {
    printf("\nSNMP group '%s' is configured successfully.\n", groupName);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }


  return;
}

/*********************************************************************
* @purpose  Delete an existing SNMP group.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    groupName           @b{(input)}  group name
* @param    securityModel       @b{(input)}  version number
* @param    securityLevel       @b{(input)}  security level (valid only for securityModel @ref OPEN_SNMP_SECURITY_MODEL_USM)
* @param    contextPrefix       @b{(input)}  optional prefix name to associate the group with
*
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch.
* 
* @end
*********************************************************************/
void testSnmpGroupDelete(openapiClientHandle_t *clientHandle, 
                         char *groupName,
                         OPEN_SNMP_SECURITY_MODEL_t securityModel,
                         OPEN_SNMP_SECURITY_LEVEL_t securityLevel,
                         char *contextPrefix)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc groupDesc;
  open_buffdesc contextDesc;
  char groupStr[30+1];
  char contextStr[30+1];

  strcpy(groupStr, groupName);
  groupDesc.pstart = groupStr;
  groupDesc.size = strlen(groupStr) + 1;

  strcpy(contextStr, contextPrefix);
  contextDesc.pstart = contextStr;
  contextDesc.size = strlen(contextStr) + 1;

  result = openapiSnmpGroupDelete(clientHandle,
                                  &groupDesc,
                                  securityModel,
                                  securityLevel,
                                  &contextDesc);
  if (OPEN_E_NONE == result)
  {
    printf("\nSNMP group '%s' is deleted successfully.\n", groupName);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/********************************************************************* 
* @purpose  Interate through the existing SNMP groups until a match for
*           group is found. If found, display the associated group parameters.
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    host                @b{(input)}   IPv4 or IPv6 address
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes
* 
* @end
*********************************************************************/
void testSnmpGroupGet(openapiClientHandle_t *clientHandle, 
                      char *group)
{
  OPEN_SNMP_SECURITY_MODEL_t securityModel;
  OPEN_SNMP_SECURITY_LEVEL_t securityLevel;
  open_buffdesc groupDesc;
  open_buffdesc contextDesc;
  open_buffdesc readDesc;
  open_buffdesc writeDesc;
  open_buffdesc notifyDesc;
  open_buffdesc nextGroupDesc;
  open_buffdesc nextContextDesc;
  char groupStr[30+1];
  char contextStr[30+1];
  char readStr[30+1];
  char writeStr[30+1];
  char notifyStr[30+1];
  char nextGroupStr[30+1];
  char nextContextStr[30+1];
  bool displayAllEntries = false;

  if ((NULL != group) &&
      (0 == strlen(group)))
  {
    displayAllEntries = true;
  }

  memset(groupStr, 0, sizeof(groupStr));
  groupDesc.pstart = groupStr;
  groupDesc.size = sizeof(groupStr);

  memset(nextGroupStr, 0, sizeof(nextGroupStr));
  nextGroupDesc.pstart = nextGroupStr;
  nextGroupDesc.size = sizeof(nextGroupStr);

  memset(contextStr, 0, sizeof(contextStr));
  contextDesc.pstart = contextStr;
  contextDesc.size = sizeof(contextStr);

  memset(nextContextStr, 0, sizeof(nextContextStr));
  nextContextDesc.pstart = nextContextStr;
  nextContextDesc.size = sizeof(nextContextStr);

  memset(readStr, 0, sizeof(readStr));
  readDesc.pstart = readStr;
  readDesc.size = sizeof(readStr);

  memset(writeStr, 0, sizeof(writeStr));
  writeDesc.pstart = writeStr;
  writeDesc.size = sizeof(writeStr);

  memset(notifyStr, 0, sizeof(notifyStr));
  notifyDesc.pstart = notifyStr;
  notifyDesc.size = sizeof(notifyStr);

  while (openapiSnmpGroupGetNext(clientHandle,
                                 &groupDesc,
                                 &securityModel,
                                 &securityLevel,
                                 &contextDesc,
                                 &readDesc,
                                 &writeDesc,
                                 &notifyDesc,
                                 &nextGroupDesc,
                                 &nextContextDesc) == OPEN_E_NONE)
  {
    if (true == displayAllEntries)
    {
      if (fVerbose)
      {
        printf("\n");
        printf("Group name     : %s\n"
               "Security model : %d (%s)\n"
               "Security level : %d (%s)\n"
               "Context name   : %s\n"
               "Read view      : %s\n"
               "Write view     : %s\n"
               "Notify view    : %s\n",
               (char *) nextGroupDesc.pstart,
               securityModel, getVersion(securityModel),
               securityLevel, getSecurityLevel(securityLevel),
               (char *) nextContextDesc.pstart,
               (char *) readDesc.pstart,
               (char *) writeDesc.pstart,
               (char *) notifyDesc.pstart);
        printf("\n");
      }
    }
    else
    {
      if (strcmp(group, nextGroupDesc.pstart)==0)
      {
        if (fVerbose)
        {
          printf("\n");
          printf("Group name     : %s\n"
                 "Security model : %d (%s)\n"
                 "Security level : %d (%s)\n"
                 "Context name   : %s\n"
                 "Read view      : %s\n"
                 "Write view     : %s\n"
                 "Notify view    : %s\n",
                 (char *) nextGroupDesc.pstart,
                 securityModel, getVersion(securityModel),
                 securityLevel, getSecurityLevel(securityLevel),
                 (char *) nextContextDesc.pstart,
                 (char *) readDesc.pstart,
                 (char *) writeDesc.pstart,
                 (char *) notifyDesc.pstart);
          printf("\n");
        }
      }
    }

    /* Setup for next iteration */
    memset(readStr,    0, sizeof(readStr));
    memset(writeStr,   0, sizeof(writeStr));
    memset(notifyStr,  0, sizeof(notifyStr));
    strncpy(groupDesc.pstart, nextGroupDesc.pstart, nextGroupDesc.size);
    groupDesc.size = nextGroupDesc.size;
    strncpy(contextDesc.pstart, nextContextDesc.pstart, nextContextDesc.size);
    contextDesc.size = nextContextDesc.size;

    /* Necessary to reset size for next group record */
    memset(nextGroupStr, 0, sizeof(nextGroupStr));
    memset(nextContextStr, 0, sizeof(nextContextStr));
    nextGroupDesc.size = sizeof(nextGroupStr);
    nextContextDesc.size = sizeof(nextContextStr);
  }

  return;
}

/********************************************************************* 
* @purpose  Create a new, or edit an existing SNMP view.
*           If the view name already exists, the associated MIBS
*           (vacmViewTreeFamilySubtree and vacmViewTreeFamilyMask) are updated
*           according to the excluded flag.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    viewName            @b{(input)}  view name
* @param    oidTree             @b{(input)}  ASN.1 subtree OID to be included/excluded from the view
* @param    viewType            @b{(input)}  included or excluded from the MIB view
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void testSnmpViewCreate(openapiClientHandle_t *clientHandle, 
                        char *view,
                        char *oid,
                        OPEN_SNMP_VIEW_TYPE_t type)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc viewDesc;
  open_buffdesc oidDesc;
  char viewStr[30+1];
  char oidStr[128+1];

  strcpy(viewStr, view);
  viewDesc.pstart = viewStr;
  viewDesc.size = strlen(viewStr)+1;

  strcpy(oidStr, oid);
  oidDesc.pstart = oidStr;
  oidDesc.size = strlen(oidStr)+1;

  result = openapiSnmpViewCreate(clientHandle, &viewDesc, &oidDesc, type);
  if (OPEN_E_NONE == result)
  {
    printf("\nSNMP view '%s' is configured successfully.\n", view);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/********************************************************************* 
* @purpose  Interate through the existing SNMP views until a match for
*           view and oid is found. If found, display the associated view parameters.
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    viewName            @b{(input)}  view name
* @param    oidTree             @b{(input)}  ASN.1 subtree OID to be included/excluded from the view
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes
* 
* @end
*********************************************************************/
void testSnmpViewGet(openapiClientHandle_t *clientHandle, 
                     char *view)
{
  OPEN_SNMP_VIEW_TYPE_t type;
  open_buffdesc viewDesc;
  open_buffdesc oidDesc;
  open_buffdesc oidValDesc;
  open_buffdesc oidStringDesc;
  open_buffdesc nextViewDesc;
  open_buffdesc nextOidDesc;
  char viewStr[30+1];
  char oidStr[128+1];
  char oidValStr[128+1];
  char oidStringStr[128+1];
  char nextViewStr[30+1];
  char nextOidStr[128+1];
  bool displayAllEntries = false;

  if ((NULL != view) &&
      (0 == strlen(view)))
  {
    displayAllEntries = true;
  }

  memset(viewStr, 0, sizeof(viewStr));
  viewDesc.pstart = viewStr;
  viewDesc.size = sizeof(viewStr);

  memset(oidStr, 0, sizeof(oidStr));
  oidDesc.pstart = oidStr;
  oidDesc.size = sizeof(oidStr);

  memset(nextViewStr, 0, sizeof(nextViewStr));
  nextViewDesc.pstart = nextViewStr;
  nextViewDesc.size = sizeof(nextViewStr);

  memset(nextOidStr, 0, sizeof(nextOidStr));
  nextOidDesc.pstart = nextOidStr;
  nextOidDesc.size = sizeof(nextOidStr);

  memset(oidValStr, 0, sizeof(oidValStr));
  oidValDesc.pstart = oidValStr;
  oidValDesc.size = sizeof(oidValStr);

  memset(oidStringStr, 0, sizeof(oidStringStr));
  oidStringDesc.pstart = oidStringStr;
  oidStringDesc.size = sizeof(oidStringStr);

  strcpy(viewStr, view);

  while (openapiSnmpViewGetNext(clientHandle,
                                &viewDesc,
                                &oidDesc,
                                &type,
                                &nextViewDesc,
                                &nextOidDesc) == OPEN_E_NONE)
  {
    openapiSnmpTreeFamilyOidStringGet(clientHandle, &nextViewDesc, &nextOidDesc,
                                      OPEN_SNMP_VIEW_TREE, &oidStringDesc);
    openapiSnmpTreeFamilyOidValGet(clientHandle, &nextViewDesc, &nextOidDesc,
                                   OPEN_SNMP_VIEW_TREE, &oidValDesc);

    if (true == displayAllEntries)
    {
      if (fVerbose)
      {
        printf("\n");
        printf("View name  : %s\n"
               "OID Tree   : %s\n"
               "OID String : %s\n"
               "OID Val      : %s\n"
               "View Type  : %d (%s)\n",
               (char *) nextViewDesc.pstart,
               (char *) nextOidDesc.pstart,
               (char *) oidStringDesc.pstart,
               (char *) oidValDesc.pstart,
               type, getViewType(type));
        printf("\n");
      }
    }
    else
    {
      if (strcmp(view, nextViewDesc.pstart)==0)
      {
        if (fVerbose)
        {
          printf("\n");
          printf("View name  : %s\n"
                 "OID Tree   : %s\n"
                 "OID String : %s\n"
                 "OID Val      : %s\n"
                 "View Type  : %d (%s)\n",
                 (char *) nextViewDesc.pstart,
                 (char *) nextOidDesc.pstart,
                 (char *) oidStringDesc.pstart,
                 (char *) oidValDesc.pstart,
                 type, getViewType(type));
          printf("\n");
        }
      }
    }

    /* Setup for next iteration */
    strncpy(viewDesc.pstart, nextViewDesc.pstart, nextViewDesc.size);
    viewDesc.size = nextViewDesc.size;
    strncpy(oidDesc.pstart, oidStringDesc.pstart, oidStringDesc.size);
    oidDesc.size = oidStringDesc.size;

    /* Necessary to reset size for next group record */
    memset(nextViewStr, 0, sizeof(nextViewStr));
    memset(nextOidStr, 0, sizeof(nextOidStr));
    memset(oidStringStr, 0, sizeof(oidStringStr));
    memset(oidValStr, 0, sizeof(oidValStr));

    nextViewDesc.size = sizeof(nextViewStr);
    nextOidDesc.size = sizeof(nextOidStr);
    oidValDesc.size = sizeof(oidValStr);
    oidStringDesc.size = sizeof(oidStringStr);
  }

  return;
}

/********************************************************************* 
* @purpose  Delete an existing SNMP view.
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    viewName            @b{(input)}  view name
* @param    oidTree             @b{(input)}  ASN.1 subtree OID to be included/excluded from the view
*
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void testSnmpViewDelete(openapiClientHandle_t *clientHandle, 
                        char *view,
                        char *oid)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc viewDesc;
  open_buffdesc oidDesc;
  char viewStr[30+1];
  char oidStr[128+1];

  strcpy(viewStr, view);
  viewDesc.pstart = viewStr;
  viewDesc.size = strlen(viewStr)+1;

  strcpy(oidStr, oid);
  oidDesc.pstart = oidStr;
  oidDesc.size = strlen(oidStr)+1;

  result = openapiSnmpViewDelete(clientHandle, &viewDesc, &oidDesc);
  if (OPEN_E_NONE == result)
  {
    printf("\nSNMP view '%s' is deleted successfully.\n", view);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/********************************************************************* 
* @purpose  Create a new, or edit an existing SNMP filter.
*           If the filter name already exists, the associated MIBS
*           (vacmFilterTreeFamilySubtree and vacmFilterTreeFamilyMask) are updated
*           according to the excluded flag.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    filterName          @b{(input)}  filter name
* @param    oidTree             @b{(input)}  ASN.1 subtree OID to be included/excluded from the view
* @param    viewType            @b{(input)}  included or excluded from the MIB view
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void testSnmpFilterCreate(openapiClientHandle_t *clientHandle, 
                          char *filter,
                          char *oid,
                          OPEN_SNMP_NOTIFY_FILTER_TYPE_t type)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc filterDesc;
  open_buffdesc oidDesc;
  char filterStr[30+1];
  char oidStr[128+1];

  strcpy(filterStr, filter);
  filterDesc.pstart = filterStr;
  filterDesc.size = strlen(filterStr)+1;

  strcpy(oidStr, oid);
  oidDesc.pstart = oidStr;
  oidDesc.size = strlen(oidStr)+1;

  result = openapiSnmpFilterCreate(clientHandle, &filterDesc, &oidDesc, type);
  if (OPEN_E_NONE == result)
  {
    printf("\nSNMP filter '%s' is configured successfully.\n", filter);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/********************************************************************* 
* @purpose  Delete an existing SNMP filter.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    filterName          @b{(input)}  filter name
* @param    oidTree             @b{(input)}  ASN.1 subtree OID to be included/excluded from the filter
*
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void testSnmpFilterDelete(openapiClientHandle_t *clientHandle, 
                          char *filter,
                          char *oid)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc filterDesc;
  open_buffdesc oidDesc;
  char filterStr[30+1];
  char oidStr[128+1];

  strcpy(filterStr, filter);
  filterDesc.pstart = filterStr;
  filterDesc.size = strlen(filterStr)+1;

  strcpy(oidStr, oid);
  oidDesc.pstart = oidStr;
  oidDesc.size = strlen(oidStr)+1;

  result = openapiSnmpFilterDelete(clientHandle, &filterDesc, &oidDesc);
  if (OPEN_E_NONE == result)
  {
    printf("\nSNMP filter '%s' is deleted successfully.\n", filter);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/********************************************************************* 
* @purpose  Interate through the existing SNMP filters until a match for
*           filter and oid is found. If found, display the associated filter parameters.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    filterName          @b{(input)}  filter name
* @param    oidTree             @b{(input)}  ASN.1 subtree OID to be included/excluded from the filter
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes
* 
* @end
*********************************************************************/
void testSnmpFilterGet(openapiClientHandle_t *clientHandle, 
                       char *filter)
{
  OPEN_SNMP_NOTIFY_FILTER_TYPE_t type;
  open_buffdesc filterDesc;
  open_buffdesc oidDesc;
  open_buffdesc oidValDesc;
  open_buffdesc oidStringDesc;
  open_buffdesc nextFilterDesc;
  open_buffdesc nextOidDesc;
  char filterStr[30+1];
  char oidStr[128+1];
  char oidValStr[128+1];
  char oidStringStr[128+1];
  char nextFilterStr[30+1];
  char nextOidStr[128+1];
  bool fFound = false;
  bool displayAllEntries = false;

  if ((NULL != filter) &&
      (0 == strlen(filter)))
  {
    displayAllEntries = true;
  }

  memset(filterStr, 0, sizeof(filterStr));
  filterDesc.pstart = filterStr;
  filterDesc.size = sizeof(filterStr);

  memset(oidStr, 0, sizeof(oidStr));
  oidDesc.pstart = oidStr;
  oidDesc.size = sizeof(oidStr);

  memset(nextFilterStr, 0, sizeof(nextFilterStr));
  nextFilterDesc.pstart = nextFilterStr;
  nextFilterDesc.size = sizeof(nextFilterStr);

  memset(nextOidStr, 0, sizeof(nextOidStr));
  nextOidDesc.pstart = nextOidStr;
  nextOidDesc.size = sizeof(nextOidStr);

  memset(oidValStr, 0, sizeof(oidValStr));
  oidValDesc.pstart = oidValStr;
  oidValDesc.size = sizeof(oidValStr);

  memset(oidStringStr, 0, sizeof(oidStringStr));
  oidStringDesc.pstart = oidStringStr;
  oidStringDesc.size = sizeof(oidStringStr);

  strcpy(filterStr, filter);

  while (openapiSnmpFilterGetNext(clientHandle,
                                  &filterDesc,
                                  &oidDesc,
                                  &type,
                                  &nextFilterDesc,
                                  &nextOidDesc) == OPEN_E_NONE)
  {
    openapiSnmpTreeFamilyOidStringGet(clientHandle, &nextFilterDesc, &nextOidDesc,
                                      OPEN_SNMP_FILTER_TREE, &oidStringDesc);
    openapiSnmpTreeFamilyOidValGet(clientHandle, &nextFilterDesc, &nextOidDesc,
                                   OPEN_SNMP_FILTER_TREE, &oidValDesc);

    if (strcmp(filter, nextFilterDesc.pstart)==0)
    {
      fFound = true;
      break;
    }

    if (true == displayAllEntries)
    {
      if (fVerbose)
      {
        printf("\n");
        printf("Filter name  : %s\n"
               "OID Tree     : %s\n"
               "OID String   : %s\n"
               "OID Val      : %s\n"
               "View Type    : %d (%s)\n",
               (char *) nextFilterDesc.pstart,
               (char *) nextOidDesc.pstart,
               (char *) oidStringDesc.pstart,
               (char *) oidValDesc.pstart,
               type, getFilterType(type));
        printf("\n");
      }
    }

    /* Setup for next iteration */
    strncpy(filterDesc.pstart, nextFilterDesc.pstart, nextFilterDesc.size);
    filterDesc.size = nextFilterDesc.size;
    strncpy(oidDesc.pstart, oidStringDesc.pstart, oidStringDesc.size);
    oidDesc.size = oidStringDesc.size;

    /* Necessary to reset size for next group record */
    memset(oidValStr, 0, sizeof(oidValStr));
    memset(oidStringStr, 0, sizeof(oidStringStr));
    memset(nextFilterStr, 0, sizeof(nextFilterStr));
    memset(nextOidStr, 0, sizeof(nextOidStr));

    nextFilterDesc.size = sizeof(nextFilterStr);
    nextOidDesc.size = sizeof(nextOidStr);
    oidStringDesc.size = sizeof(oidStringStr);
    oidValDesc.size = sizeof(oidValStr);
  }

  if (fFound)
  {
    if (fVerbose)
    {
      printf("Filter name  : %s\n"
             "OID Tree     : %s\n"
             "OID String   : %s\n"
             "OID Val      : %s\n"
             "View Type    : %d (%s)\n",
             (char *) nextFilterDesc.pstart,
             (char *) nextOidDesc.pstart,
             (char *) oidStringDesc.pstart,
             (char *) oidValDesc.pstart,
             type, getFilterType(type));
    }
  }

  return;
}

/********************************************************************* 
* @purpose  Create an SNMPv1/2 or v3 host. This includes the target address 
*           object and its various parameters, including an optional notification 
*           filter.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    host                @b{(input)}  IPv4 or IPv6 address
* @param    serverPort          @b{(input)}  optional host receiver port number.
* @param    securityName        @b{(input)}  context name (community or user name)
* @param    securityModel       @b{(input)}  version number
* @param    securityLevel       @b{(input)}  security level 
* @param    notifyType          @b{(input)}  notification type
* @param    timeout             @b{(input)}  optional timeout in milliseconds for @ref OPEN_SNMP_NOTIFY_TYPE_INFORM trap type
* @param    retries             @b{(input)}  optional retries for @ref OPEN_SNMP_NOTIFY_TYPE_INFORM trap type
* @param    filter              @b{(input)}  optional notification filter name
* 
* @returns  OPEN_E_NONE         Set is successful
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch 
* 
* @end
*********************************************************************/
void testSnmpHostCreate(openapiClientHandle_t *clientHandle, 
                        char *host,
                        uint32_t serverPort,
                        char *securityName,
                        OPEN_SNMP_SECURITY_MODEL_t securityModel,
                        OPEN_SNMP_SECURITY_LEVEL_t securityLevel,
                        OPEN_SNMP_NOTIFY_TYPE_t notifyType,
                        uint32_t timeout,
                        uint32_t retries,
                        char *filter)

{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc hostDesc;
  open_buffdesc securityDesc;
  open_buffdesc filterDesc;
  char hostStr[158+1];
  char securityStr[30+1];
  char filterStr[30+1];

  strcpy(hostStr, host);
  hostDesc.pstart = hostStr;
  hostDesc.size = strlen(hostStr)+1;

  strcpy(securityStr, securityName);
  securityDesc.pstart = securityStr;
  securityDesc.size = strlen(securityStr)+1;

  strcpy(filterStr, filter);
  filterDesc.pstart = filterStr;
  filterDesc.size = strlen(filterStr)+1;

  result = openapiSnmpHostCreate(clientHandle,
                                 &hostDesc, serverPort,
                                 &securityDesc, securityModel,
                                 securityLevel, notifyType,
                                 timeout, retries, 
                                 &filterDesc);
  if (OPEN_E_NONE == result)
  {
    printf("\nSNMP host '%s' is configured successfully.\n", host);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/********************************************************************* 
* @purpose  Delete an existing SNMPv1/2 or v3 host entry.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    host                @b{(input)}  IPv4 or IPv6 address
*
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void testSnmpHostDelete(openapiClientHandle_t *clientHandle, 
                        char *host,
                        OPEN_SNMP_NOTIFY_TYPE_t notifyType)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc hostDesc;
  char hostStr[158+1];

  strcpy(hostStr, host);
  hostDesc.pstart = hostStr;
  hostDesc.size = strlen(hostStr)+1;

  result = openapiSnmpHostDelete(clientHandle, &hostDesc, notifyType);
  if (OPEN_E_NONE == result)
  {
    printf("\nSNMP host '%s' is deleted successfully.\n", host);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/********************************************************************* 
* @purpose  Create, or update an existing community, then retrieve it
*           for verification.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    community           @b{(input)}  community name
* @param    accessType          @b{(input)}  optional community access
* @param    viewName            @b{(input)}  optional view name
* @param    ipStr               @b{(input)}  optional IPv4 address, for allowed access
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void testSnmpCommunity(openapiClientHandle_t *clientHandle, 
                       char *community,
                       OPEN_SNMP_COMMUNITY_ACCESS_TYPE_t accessType,
                       char *viewName,
                       char *ipStr)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc communityDesc;
  open_buffdesc viewDesc;
  open_inet_addr_t ipAddr;
  char communityStr[20+1];
  char viewStr[30+1];

  strcpy(communityStr, community);
  communityDesc.pstart = communityStr;
  communityDesc.size = strlen(communityStr)+1;

  strcpy(viewStr, viewName);
  viewDesc.pstart = viewStr;
  viewDesc.size = strlen(viewStr)+1;

  memset(&ipAddr, 0, sizeof(ipAddr));
  inet_pton(AF_INET, ipStr, (void*)&(ipAddr.addr.ipv4));
  ipAddr.family = OPEN_AF_INET;

  result = openapiSnmpCommunityCreate(clientHandle, &communityDesc, accessType, &viewDesc, ipAddr);
  if (OPEN_E_NONE == result)
  {
    printf("\nSNMP community '%s' is configured successfully.\n", community);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }


  return;
}

/********************************************************************* 
* @purpose  Delete an existing community entry.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    community           @b{(input)}  community name
*
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void testSnmpCommunityDelete(openapiClientHandle_t *clientHandle, 
                             char *community)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc communityDesc;
  char communityStr[20+1];

  strcpy(communityStr, community);
  communityDesc.pstart = communityStr;
  communityDesc.size = strlen(communityStr)+1;

  result = openapiSnmpCommunityDelete(clientHandle, &communityDesc);
  if (OPEN_E_NONE == result)
  {
    printf("\nSNMP community '%s' is deleted successfully.\n", community);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/********************************************************************* 
* @purpose  Interate through the existing SNMP hosts until a match for
*           host is found. If found, display the associated host parameters.
*
* @param    client_handle       @b{(input)} client handle from registration API
* @param    community           @b{(input)} community name
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes
* 
* @end
*********************************************************************/
void testSnmpCommunityGet(openapiClientHandle_t *clientHandle, 
                          char *community)
{
  open_error_t result = OPEN_E_NONE;
  OPEN_SNMP_COMMUNITY_ACCESS_TYPE_t accessType;
  open_inet_addr_t addr;
  char ipAddrStr[32];
  open_buffdesc communityDesc;
  open_buffdesc viewDesc;
  open_buffdesc nextCommunityDesc;
  char communityStr[20+1];
  char viewStr[30+1];
  char nextCommunityStr[20+1];
  bool fFound = false;
  bool displayAllEntries = false;

  if ((NULL != community) &&
      (0 == strlen(community)))
  {
    displayAllEntries = true;
  }

  memset(communityStr, 0, sizeof(communityStr));
  communityDesc.pstart = communityStr;
  communityDesc.size = sizeof(communityStr);

  memset(viewStr, 0, sizeof(viewStr));
  viewDesc.pstart = viewStr;
  viewDesc.size = sizeof(viewStr);

  memset(nextCommunityStr, 0, sizeof(nextCommunityStr));
  nextCommunityDesc.pstart = nextCommunityStr;
  nextCommunityDesc.size = sizeof(nextCommunityStr);

  while (openapiSnmpCommunityGetNext(clientHandle,
                                     &communityDesc,
                                     &accessType,
                                     &viewDesc,
                                     &addr,
                                     &nextCommunityDesc) == OPEN_E_NONE)
  {
    if (strcmp(community, nextCommunityDesc.pstart)==0)
    {
      fFound = true;
      break;
    }

    if (true == displayAllEntries)
    {
      if (addr.family == OPEN_AF_INET)
      {
        memset(&ipAddrStr, 0, sizeof(ipAddrStr));
        if (inet_ntop(AF_INET, (void*)&(addr.addr.ipv4), ipAddrStr, sizeof(ipAddrStr)) == NULL)
        {
           result = OPEN_E_PARAM;
        }
      }
      else
      {
        strcpy(ipAddrStr, "All");
      }
      PRINTBADRESULT(result, "inet_ntop", "");
      
      if (fVerbose)
      {
        printf("\n");
        printf("Community   : %s\n"
               "Access Type : %d (%s)\n"
               "View Name   : %s\n"
               "IP Address  : %s\n",
               (char *) nextCommunityDesc.pstart,
               accessType, getAccessType(accessType),
               (char *) viewDesc.pstart,
               ipAddrStr);
        printf("\n");
      }
    }

    /* Setup for next iteration */
    memset(&ipAddrStr, 0, sizeof(ipAddrStr));

    memset(viewStr, 0, sizeof(viewStr));
    viewDesc.size = sizeof(viewStr);

    strncpy(communityDesc.pstart, nextCommunityDesc.pstart, nextCommunityDesc.size);
    communityDesc.size = nextCommunityDesc.size;

    /* Necessary to reset size for next host record */
    nextCommunityDesc.size = sizeof(nextCommunityStr);
    memset(nextCommunityStr, 0, sizeof(nextCommunityStr));
  }

  if (fFound)
  {
    if (addr.family == OPEN_AF_INET)
    {
      memset(&ipAddrStr, 0, sizeof(ipAddrStr));
      if (inet_ntop(AF_INET, (void*)&(addr.addr.ipv4), ipAddrStr, sizeof(ipAddrStr)) == NULL)
      {
        result = OPEN_E_PARAM;
      }
    }
    else
    {
      strcpy(ipAddrStr, "All");
    }
    PRINTBADRESULT(result, "inet_ntop", "");
      
    if (fVerbose)
    {
      printf("Community   : %s\n"
             "Access Type : %d (%s)\n"
             "View Name   : %s\n"
             "IP Address  : %s\n",
             (char *) nextCommunityDesc.pstart,
             accessType,getAccessType(accessType),
             (char *) viewDesc.pstart,
             ipAddrStr);
    }

    result = OPEN_E_NONE;
  }


  return;
}

/********************************************************************* 
* @purpose  Create, or update an existing community, then retrieve it
*           for verification.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    community           @b{(input)}  community name
* @param    group               @b{(input)}  group name
* @param    ipStr               @b{(input)}  optional IPv4 address, for allowed access
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void testSnmpCommunityGroup(openapiClientHandle_t *clientHandle, 
                            char *community,
                            char *group,
                            char *ipStr)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc communityDesc;
  open_buffdesc groupDesc;
  open_inet_addr_t ipAddr;
  char communityStr[20+1];
  char groupStr[30+1];

  strcpy(communityStr, community);
  communityDesc.pstart = communityStr;
  communityDesc.size = strlen(communityStr)+1;

  strcpy(groupStr, group);
  groupDesc.pstart = groupStr;
  groupDesc.size = strlen(groupStr) + 1;

  memset(&ipAddr, 0, sizeof(ipAddr));
  inet_pton(AF_INET, ipStr, (void*)&(ipAddr.addr.ipv4));
  ipAddr.family = OPEN_AF_INET;

  result = openapiSnmpCommunityGroupCreate(clientHandle, &communityDesc, &groupDesc, ipAddr);
  if (OPEN_E_NONE == result)
  {
    printf("\nSNMP community '%s' is configured successfully.\n", community);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/********************************************************************* 
* @purpose  Interate through the existing SNMP hosts until a match for
*           host is found. If found, display the associated host parameters.
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    host                @b{(input)}   IPv4 or IPv6 address
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes
* 
* @end
*********************************************************************/
void testSnmpHostGet(openapiClientHandle_t *clientHandle, 
                     char *host)
{
  OPEN_SNMP_SECURITY_MODEL_t securityModel;
  OPEN_SNMP_SECURITY_LEVEL_t securityLevel;
  OPEN_SNMP_NOTIFY_TYPE_t notifyType;
  open_buffdesc hostDesc;
  open_buffdesc securityDesc;
  open_buffdesc filterDesc;
  open_buffdesc nextHostDesc;
  char hostStr[30+1];
  char securityStr[30+1];
  char filterStr[30+1];
  char nextHostStr[30+1];
  uint32_t serverPort = 0;
  uint32_t timeout = 0;
  uint32_t retries = 0;
  bool fFound = false;
  bool displayAllEntries = false;

  if ((NULL != host) &&
      (0 == strlen(host)))
  {
    displayAllEntries = true;
  }

  memset(hostStr, 0, sizeof(hostStr));
  hostDesc.pstart = hostStr;
  hostDesc.size = sizeof(hostStr);

  memset(securityStr, 0, sizeof(securityStr));
  securityDesc.pstart = securityStr;
  securityDesc.size = sizeof(securityStr);

  memset(filterStr, 0, sizeof(filterStr));
  filterDesc.pstart = filterStr;
  filterDesc.size = sizeof(filterStr);

  memset(nextHostStr, 0, sizeof(nextHostStr));
  nextHostDesc.pstart = nextHostStr;
  nextHostDesc.size = sizeof(nextHostStr);

  while (openapiSnmpHostGetNext(clientHandle,
                                &hostDesc,
                                &serverPort,
                                &securityDesc,
                                &securityModel,
                                &securityLevel,
                                &notifyType,
                                &timeout,
                                &retries,
                                &filterDesc,
                                &nextHostDesc) == OPEN_E_NONE)
  {
    if (strcmp(host, nextHostDesc.pstart)==0)
    {
      fFound = true;
      break;
    }

    if (true == displayAllEntries)
    {
      if (fVerbose)
      {
        printf("\n");
        printf("Address:port   : %s:%d\n"
               "Security name  : %s\n"
               "Security model : %d (%s)\n"
               "Security level : %d (%s)\n"
               "Notify type    : %d (%s)\n"
               "Timeout        : %d\n"
               "Retries        : %d\n"
               "Filter         : %s\n",
               (char *) nextHostDesc.pstart, serverPort,
               (char *) securityDesc.pstart,
               securityModel, getVersion(securityModel),
               securityLevel, getSecurityLevel(securityLevel),
               notifyType, getNotifyType(notifyType),
               timeout,
               retries,
               (char *) filterDesc.pstart);
        printf("\n");
      }
    }

    /* Setup for next iteration */
    memset(securityStr, 0, sizeof(securityStr));
    strncpy(hostDesc.pstart, nextHostDesc.pstart, nextHostDesc.size);
    hostDesc.size = nextHostDesc.size;

    /* Necessary to reset size for next host record */
    memset(nextHostStr, 0, sizeof(nextHostStr));
    nextHostDesc.size = sizeof(nextHostStr);
  }

  if (fFound)
  {
    if (fVerbose)
    {
      printf("Address:port   : %s:%d\n"
             "Security name  : %s\n"
             "Security model : %d (%s)\n"
             "Security level : %d (%s)\n"
             "Notify type    : %d (%s)\n"
             "Timeout        : %d\n"
             "Retries        : %d\n"
             "Filter         : %s\n",
             (char *) nextHostDesc.pstart, serverPort,
             (char *) securityDesc.pstart,
             securityModel, getVersion(securityModel),
             securityLevel,  getSecurityLevel(securityLevel),
             notifyType, getNotifyType(notifyType),
             timeout,
             retries,
             (char *) filterDesc.pstart);
    }
  }

  return;
}

/*****************************************************************************
* @purpose  Enable or disable the traps identified via the trapFlags selection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    flagBits            @b{(input)}  selected traps, a bitmask of @ref OPEN_SNMP_TRAP_FLAGS_t
* @param    mode                @b{(input)}  Set on or off
*
* @returns  OPEN_E_NONE         Set is successful
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_ERROR        If the set request was invalid
* @returns  OPEN_E_UNAVAIL      If one or more trap sets failed due to the feature availability.
* 
* @notes
* 
* @end
*****************************************************************************/
void testSnmpTrapFlagSet(openapiClientHandle_t *clientHandle,
                         uint32_t flagBits,
                         bool mode)
{
  open_error_t result = OPEN_E_NONE;

  result = openapiSnmpTrapFlagSet(clientHandle, flagBits, mode);
  if (OPEN_E_NONE == result)
  {
    printf("\nSNMP trap flag %d (%s) is configured successfully with mode %d.\n",
           flagBits, getTrapFlagType(flagBits), mode);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/*****************************************************************************
* @purpose  Retrieve the trap mode for a single trap.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    flagBits            @b{(input)}  selected trap from @ref OPEN_SNMP_TRAP_FLAGS_t
* @param    mode                @b{(output)} Set on or off
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes
* 
* @end
****************************************************************************/
void testSnmpTrapFlagGet(openapiClientHandle_t *clientHandle,
                         uint32_t flagBits)
{
  open_error_t result = OPEN_E_NONE;
  bool tmpMode = 0;

  result = openapiSnmpTrapFlagGet(clientHandle, flagBits, &tmpMode);
  if (result == OPEN_E_NONE)
  {
    printf("\nSNMP trap flag:%d (%s)  mode:%d\n", flagBits, getTrapFlagType(flagBits), tmpMode);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/***************************************************************************** 
* @purpose  Set the source interface which SNMP trap originates.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    type                @b{(input)}  type of interface to retrieve
*
* @returns  OPEN_E_NONE         Set is successful
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* @notes    The source interface must be a valid routing, loopback, tunnel, or VLAN interface.
* 
* @end
****************************************************************************/
void testSnmpTrapSourceInterfaceSet(openapiClientHandle_t *clientHandle,
                                    uint32_t intf)
{
  open_error_t result = OPEN_E_NONE;

  result = openapiSnmpTrapSrcIntfSet(clientHandle, intf);
  if (OPEN_E_NONE == result)
  {
    printf("\nSNMP  Source interface  '%d' is configured successfully.\n", intf);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/***************************************************************************** 
* @purpose  Get the source interface which SNMP trap originates.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    type                @b{(input)}  type of interface to retrieve
*
* @returns  OPEN_E_NONE         Set is successful
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* @notes    The source interface must be a valid routing, loopback, tunnel, or VLAN interface.
* 
* @end
****************************************************************************/
void testSnmpTrapSourceInterfaceGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result = OPEN_E_NONE;
  OPEN_INTF_TYPE_t tmpType;
  uint32_t tmpIntf;
  open_inet_addr_t addr;
  char msg[256]= {0};
  char ipAddr[32];

  memset(&ipAddr, 0, sizeof(ipAddr));

  result = openapiSnmpTrapSrcIntfGet(clientHandle, &tmpIntf, &tmpType, &addr);
  PRINTBADRESULT(result, "openapiSnmpTrapSrcIntfGet", msg);
  if (OPEN_E_NONE == result)
  {
    if (addr.family == OPEN_AF_INET)
    {
      if (inet_ntop(AF_INET, (void*)&(addr.addr.ipv4), ipAddr, sizeof(ipAddr)) == NULL)
      {
        result = OPEN_E_PARAM;
      }
    }
    else
    {
      if (inet_ntop(AF_INET6, (void*)&(addr.addr.ipv6), ipAddr, sizeof(ipAddr)) == NULL)
      {
        result = OPEN_E_PARAM;
      }
    }

    if (result == OPEN_E_NONE)
    {
      if (fVerbose)
      {
        printf("Interface:%d Type:%d Address:%s\n", tmpIntf, tmpType, ipAddr);
      }
    }
    else
    {
      PRINTBADRESULT(result, "inet_ntop", msg);
    }
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }


  return;
}

/********************************************************************* 
* @purpose  Create, or update an existing community.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    accessType          @b{(input)}  optional community access
* @param    ipAddr              @b{(input)}  optional IPv4 address, for allowed access
* @param    ipMask              @b{(input)}  optional IPv4 mask, for allowed access
* @param    status              @b{(input)}  community status
*
* @returns  OPEN_E_NONE         Set is successful 
* @returns  OPEN_E_FAIL         Set failed        
* @returns  OPEN_E_ERROR        The set request is invalid
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end 
*********************************************************************/
void testSnmpCommunityWithIPAddrCreate(openapiClientHandle_t *clientHandle,
                                               char *communityName,                               
                                               OPEN_SNMP_COMMUNITY_ACCESS_TYPE_t accessType,      
                                               char *ipAddr,                                      
                                               char *ipMask,
                                               OPEN_SNMP_COMMUNITY_STATUS_t status)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc buffDesc;                         
  OPEN_SNMP_COMMUNITY_t community;
  char str[OPENAPI_SNMP_COMMUNITY_NAME_MAX];      
  char ipAddrStr[40];
  char ipMaskStr[40];
  open_buffdesc ipAddrBufd;
  open_buffdesc ipMaskBufd;

  memset(ipAddrStr, 0x0, sizeof(ipAddrStr));
  strncpy(ipAddrStr, ipAddr, sizeof(ipAddrStr));
  ipAddrBufd.pstart = ipAddrStr;
  ipAddrBufd.size = strlen(ipAddrStr) + 1;

  memset(ipMaskStr, 0x0, sizeof(ipMaskStr));
  strncpy(ipMaskStr, ipMask, sizeof(ipMaskStr));
  ipMaskBufd.pstart = ipMaskStr;
  ipMaskBufd.size = strlen(ipMaskStr) + 1;

  strncpy(str, communityName, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(&community, 0, sizeof(community));
  community.type = accessType;

  if (OPEN_E_NONE != openapiInetAddrGet(clientHandle, &ipAddrBufd, &(community.ipAddr)))
  {
    printf("\nUnable to convert IP address:%s to INET address format for community:%s\n",
           ipAddr, communityName);
    return;
  }
  if (OPEN_AF_INET == community.ipAddr.family)
  {
    if (OPEN_E_NONE != openapiInetAddrGet(clientHandle, &ipMaskBufd, &(community.ipMask)))
    {
      printf("\nUnable to convert IP mask:%s to INET address format for community:%s\n",
             ipMask, communityName);
      return;
    }
    community.ipAddr.addr.ipv4 = ntohl(community.ipAddr.addr.ipv4); 
    community.ipMask.addr.ipv4 = ntohl(community.ipMask.addr.ipv4); 
  }
  else if (OPEN_AF_INET6 == community.ipAddr.family)
  {
    community.ipv6PrefixLen = atoi(ipMask);
  }
  else
  {
    printf("\nIncorrect family for IP address:%s community:%s\n", ipAddr, communityName);
    return;
  }
  
  community.status = status;

  result = openapiSnmpCommunityAndIPAddrCreate(clientHandle,
                                               &buffDesc,
                                               community);

  if (OPEN_E_NONE == result)
  {
    printf("\nSNMP community with IP address and mask has been create succcessfully.\n");
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }

  return;
}

/********************************************************************* 
* @purpose  Delete an existing community.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
*
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_ERROR        The delete request is invalid
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void testSnmpCommunityAndIPAddrDelete(openapiClientHandle_t *clientHandle,
                                              char *communityName)
{
  open_error_t result = OPEN_E_NONE;
  char str[OPENAPI_SNMP_COMMUNITY_NAME_MAX];
  open_buffdesc buffDesc;  

  strncpy(str, communityName, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  result = openapiSnmpCommunityAndIPAddrDelete(clientHandle, &buffDesc);
  
  if (OPEN_E_NONE == result)
  {
    printf("\nSNMP community '%s' is deleted successfully.\n", communityName);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }
  return;
}

/********************************************************************* 
* @purpose  Get an existing community.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
*
* @returns  OPEN_E_NONE         Get is successful
* @returns  OPEN_E_FAIL         Get failed
* @returns  OPEN_E_ERROR        The get request is invalid
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    
* 
* @end
*********************************************************************/
void testSnmpCommunityAndIPAddrGet(openapiClientHandle_t *clientHandle,
                                           char *communityName)
{
  open_error_t result = OPEN_E_NONE;
  OPEN_SNMP_COMMUNITY_t community;
  char str[OPENAPI_SNMP_COMMUNITY_NAME_MAX];
  open_buffdesc buffDesc;  
  char ipAddr[OPEN_IP_ADDR_DISP_LENGTH];
  char ipMask[OPEN_IP_ADDR_DISP_LENGTH];
  open_buffdesc ipAddrBufd;
  open_buffdesc ipMaskBufd;

  strncpy(str, communityName, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(ipAddr, 0x0, sizeof(ipAddr));
  ipAddrBufd.pstart = ipAddr;                  
  ipAddrBufd.size = sizeof(ipAddr);

  memset(ipMask, 0x0, sizeof(ipMask));
  ipMaskBufd.pstart = ipMask;                  
  ipMaskBufd.size = sizeof(ipMask);

  memset(&community, 0, sizeof(community));

  result = openapiSnmpCommunityAndIPAddrGet(clientHandle, &buffDesc, &community);

  if (result == OPEN_E_NONE)
  {
    if (community.ipAddr.family == OPEN_AF_INET)
    {
      openapiOpenIPtoStringGet(clientHandle, community.ipAddr, &ipAddrBufd);
      openapiOpenIPtoStringGet(clientHandle, community.ipMask, &ipMaskBufd);
    }
    else
    {
      if (inet_ntop(AF_INET6, (void*)&(community.ipAddr.addr.ipv6), ipAddr, sizeof(ipAddr)) == NULL)
      {
        result = OPEN_E_PARAM;
      }
    }
  }


  if (result == OPEN_E_NONE)
  {
    printf("SNMP community get is successful :%s \n", communityName);

    printf("Community Name: %s \n", str);
    printf("Access Type: %d \n", community.type);
    printf("IP address: %s \n", ipAddr);
    if (OPEN_AF_INET == community.ipAddr.family)
    {
      printf("IP address mask: %s \n", ipMask);
    }
    else if (OPEN_AF_INET6 == community.ipAddr.family)
    {
      printf("IPv6 prefix length:%d \n", community.ipv6PrefixLen);
    }
    printf("View Name: %s \n", community.viewName);
    printf("Status: %d \n", community.status);
  }
  else
  {
    PRINTBADRESULT(result, __func__, "");
  }
  return;
}

/********************************************************************* 
* @purpose  Get next community.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
*
* @returns  OPEN_E_NONE         Get is successful
* @returns  OPEN_E_FAIL         Get failed
* @returns  OPEN_E_ERROR        The get request is invalid
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    
* 
* @end
*********************************************************************/
void testSnmpCommunityAndIPAddrGetNext(openapiClientHandle_t *clientHandle,
                                               char *communityName)
{
  open_error_t result = OPEN_E_NONE;
  char str[OPENAPI_SNMP_COMMUNITY_NAME_MAX];
  open_buffdesc buffDesc;  

  strncpy(str, communityName, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  result = openapiSnmpCommunityAndIPAddrGetNext(clientHandle, &buffDesc, &buffDesc);

  if (result == OPEN_E_NONE)
  {
    printf("SNMP community get next is successful :%s next community :%s \n", communityName, str);
  }  
  else
  {
    PRINTBADRESULT(result, __func__,"");
  }
  return;
}

/********************************************************************* 
* @purpose  Create, or update an existing trap entry.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    ipAddr              @b{(input)}  IPv4 address, for allowed access
* @param    version             @b{(input)}  community version
* @param    status              @b{(input)}  community status
*
* @returns  OPEN_E_NONE         Set is successful 
* @returns  OPEN_E_FAIL         Set failed        
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end 
*********************************************************************/
void testSnmpTrapManagerConfigCreate(openapiClientHandle_t *clientHandle,
                                             char *communityName,                               
                                             char *ipAddr,                                      
                                             OPEN_SNMP_TRAP_VERSION_t version,
                                             OPEN_SNMP_COMMUNITY_STATUS_t status)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t temp = 0;
  open_buffdesc buffDesc;                         
  char str[OPENAPI_SNMP_COMMUNITY_NAME_MAX];
  OPEN_SNMP_TRAP_CONFIG_t trapConfig; 
  open_inet_addr_t addr;

  memset(str, 0, sizeof(str));
  strncpy(str, communityName, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(&trapConfig, 0, sizeof(trapConfig));
 
  memset(&addr, 0, sizeof(addr));  
  inet_pton(AF_INET, ipAddr, (void*)&temp);
  addr.family = OPEN_AF_INET;
  addr.addr.ipv4 = ntohl(temp);

  trapConfig.version = version;
  trapConfig.status = status;

  result = openapiTrapManagerConfigCreate(clientHandle,
                                          &buffDesc,
                                          addr,
                                          trapConfig);

  PRINTBADRESULT(result, __FUNCTION__, "");
  return;
}

/*****************************************************************************
* @purpose  Deletes an existing SNMP Trap entry and related entries if 
*           SNMP config API is not supported.     
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    ipAddr              @b{(input)}  community IP address
*
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed     
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature or SNMP config API is not supported 
*                               on this platform. 
*
* @notes    Calling this API will change the running configuration of the switch
*
* @end 
****************************************************************************/
void testSnmpTrapManagerConfigDelete(openapiClientHandle_t *clientHandle,
                                             char *communityName,                      
                                             char *ipAddr)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t temp = 0;
  open_buffdesc buffDesc;                         
  char str[OPENAPI_SNMP_COMMUNITY_NAME_MAX];
  open_inet_addr_t addr;

  memset(str, 0, sizeof(str));
  strncpy(str, communityName, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(&addr, 0, sizeof(addr));  
  inet_pton(AF_INET, ipAddr, (void*)&temp);
  addr.family = OPEN_AF_INET;
  addr.addr.ipv4 = ntohl(temp);

  result = openapiTrapManagerConfigDelete(clientHandle,
                                          &buffDesc,
                                          addr);

  PRINTBADRESULT(result, __FUNCTION__, "");
  return;
}

/*****************************************************************************
* @purpose  Get SNMP Trap information.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    ipAddr              @b{(input)}  community IP address
*
* @returns  OPEN_E_NONE         Get successful
* @returns  OPEN_E_FAIL         Get failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature or SNMP config API is not supported 
*                               on this platform.
*
* @notes
*
* @end
****************************************************************************/
void testSnmpTrapManagerConfigGet(openapiClientHandle_t *clientHandle,
                                          char *communityName,
                                          char *ipAddr)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t temp = 0;
  open_buffdesc buffDesc;                         
  char str[OPENAPI_SNMP_COMMUNITY_NAME_MAX];
  open_inet_addr_t addr;
  OPEN_SNMP_TRAP_CONFIG_t trapConfig;

  memset(str, 0, sizeof(str));
  strncpy(str, communityName, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(&addr, 0, sizeof(addr));  
  inet_pton(AF_INET, ipAddr, (void*)&temp);
  addr.family = OPEN_AF_INET;
  addr.addr.ipv4 = ntohl(temp);

  memset(&trapConfig, 0, sizeof(trapConfig));

  result = openapiTrapManagerConfigGet(clientHandle,
                                       &buffDesc,
                                       addr,
                                       &trapConfig);

  if (result == OPEN_E_NONE)
  {
    printf("Get successful for SNMP trap entry. \n");
    printf("Community :%s \n", str);
    printf("IP address :%s [%x] \n", ipAddr, addr.addr.ipv4);
    printf("Version :%d \n", trapConfig.version);
    printf("Status :%d \n", trapConfig.status);
  }  
     
  PRINTBADRESULT(result, __FUNCTION__, "");
  return;
}

/*****************************************************************************
* @purpose  Get next SNMP Trap information.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    ipAddr              @b{(input)}  community IP address
*
* @returns  OPEN_E_NONE         Get successful
* @returns  OPEN_E_FAIL         Get failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature or SNMP config API is not supported 
*                               on this platform.
*
* @notes
*
* @end
****************************************************************************/
void testSnmpTrapManagerConfigGetNext(openapiClientHandle_t *clientHandle,
                                              char *communityName,
                                              char *ipAddr)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t temp = 0;
  open_buffdesc buffDesc;                         
  open_buffdesc tmpBuf;                         
  char str[OPENAPI_SNMP_COMMUNITY_NAME_MAX];
  char tmp[OPENAPI_SNMP_COMMUNITY_NAME_MAX];
  open_inet_addr_t addr;
  char ipAddrStr[OPEN_IP_ADDR_LENGTH];

  memset(str, 0, sizeof(str));
  strncpy(str, communityName, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(tmp, 0, sizeof(tmp));
  tmpBuf.pstart = tmp;
  tmpBuf.size = sizeof(tmp);

  memset(&addr, 0, sizeof(addr));  
  inet_pton(AF_INET, ipAddr, (void*)&temp);
  addr.family = OPEN_AF_INET;
  addr.addr.ipv4 = ntohl(temp);

  result = openapiTrapManagerConfigCommunityGetNext(clientHandle,
                                                    &buffDesc,
                                                    addr,
                                                    &tmpBuf);

  if (result == OPEN_E_NONE)
  {
    printf("Get next successful for SNMP trap entry. \n");
    printf("Community :%s \n", communityName);
    printf("IP address :%s [%x] \n", ipAddr, addr.addr.ipv4);
    printf("Next Community :%s \n", tmp);
  }

  result = openapiTrapManagerConfigIPGetNext(clientHandle,
                                             &buffDesc,
                                             addr,
                                             &addr);

  if (result == OPEN_E_NONE)
  {
    if (addr.family == OPEN_AF_INET)
    {
      memset(&ipAddrStr, 0, sizeof(ipAddrStr));
      inet_ntop(AF_INET, (void*)&(addr.addr.ipv4), ipAddrStr, sizeof(ipAddrStr)); 
    }
    printf("Next IP address :%s [%x] \n", ipAddrStr, addr.addr.ipv4);
  }  

  PRINTBADRESULT(result, __FUNCTION__, "");
  return;
}

/*********************************************************************
* @purpose  Create, or update an existing community.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    accessType          @b{(input)}  optional community access
* @param    ipAddr              @b{(input)}  optional IPv4 address, for allowed access
* @param    ipMask              @b{(input)}  optional IPv4 mask, for allowed access
* @param    ipv6PrefixLen       @b{(input)}  optional IPv6 prefix length, for allowed access
* @param    viewName            @b{(input)}  optional view name
* @param    status              @b{(input)}  community status
*
* @returns  OPEN_E_NONE         Set is successful
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_ERROR        The set request is invalid
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
*
* @notes    Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void testSnmpCommunityWithIPAddrAndViewNameCreate(openapiClientHandle_t *clientHandle,
                                                          char *communityName,
                                                          OPEN_SNMP_COMMUNITY_ACCESS_TYPE_t accessType,
                                                          char *ipAddr,
                                                          char *ipMask,
                                                          uint32_t ipv6PrefixLen,
                                                          char *viewName,
                                                          OPEN_SNMP_COMMUNITY_STATUS_t status)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc buffDesc;
  OPEN_SNMP_COMMUNITY_t community;
  char str[OPENAPI_SNMP_COMMUNITY_NAME_MAX];
  char view[31];

  memset(str, 0, sizeof(str));
  strncpy(str, communityName, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(view, 0, sizeof(view));
  strncpy(view, viewName, sizeof(view));

  memset(&community, 0, sizeof(community));
  community.type = accessType;
  strncpy((char *)community.viewName, viewName, sizeof(community.viewName));

  if (inet_pton(AF_INET, ipAddr, (void*)&(community.ipAddr.addr.ipv4)) > 0)
  {
    community.ipAddr.family = OPEN_AF_INET;
  }
  else if (inet_pton(AF_INET6, ipAddr, (void*)&(community.ipAddr.addr.ipv6)) > 0)
  {
    community.ipAddr.family = OPEN_AF_INET6;
  }
  else
  {
    printf("Bad return code trying to convert ip address.\n");
    result = OPEN_E_PARAM;
  }

  if (OPEN_AF_INET == community.ipAddr.family)
  {
    if (inet_pton(AF_INET, ipMask, (void*)&(community.ipMask.addr.ipv4)) > 0)
    {
      community.ipMask.family = OPEN_AF_INET;
    }
    else
    {
      printf("Bad return code trying to convert ip address.\n");
      result = OPEN_E_PARAM;
    }
  }
  else if (OPEN_AF_INET6 == community.ipAddr.family)
  {
    community.ipMask.family = OPEN_AF_INET6;
    community.ipv6PrefixLen = ipv6PrefixLen;
  }

  community.status = status;

  if (OPEN_E_NONE == result)
  {
    result = openapiSnmpCommunityAndIPAddrCreate(clientHandle,
                                                 &buffDesc,
                                                 community);
  }
  if (OPEN_E_NONE == result)
  {
    printf("\nSNMP Community %s with IP Address and viewName has been created successfully.\n",communityName); 
  }
  else
  {
    PRINTBADRESULT(result, "testSnmpCommunityWithIPAddrCreate", "");
  }

  return;
}

/*********************************************************************
* @purpose  Get an existing community.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
*
* @returns  OPEN_E_NONE         Get is successful
* @returns  OPEN_E_FAIL         Get failed
* @returns  OPEN_E_ERROR        The get request is invalid
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes
*
* @end
*********************************************************************/
void testSnmpCommunityAndIPAddrAndViewnameGet(openapiClientHandle_t *clientHandle,
                                                      char *communityName)
{
  open_error_t result = OPEN_E_NONE;
  OPEN_SNMP_COMMUNITY_t community;
  char str[OPENAPI_SNMP_COMMUNITY_NAME_MAX];
  open_buffdesc buffDesc;
  open_buffdesc ipDesc;
  open_buffdesc maskDesc;
  char ipDescStr[46];
  char maskDescStr[46];

  ipDesc.pstart = ipDescStr;
  ipDesc.size = sizeof(ipDescStr)-1;
  maskDesc.pstart = maskDescStr;
  maskDesc.size = sizeof(maskDescStr)-1;

  strncpy(str, communityName, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(&community, 0, sizeof(community));

  result = openapiSnmpCommunityAndIPAddrGet(clientHandle, &buffDesc, &community);

  if (result == OPEN_E_NONE)
  {
    printf("SNMP community get is successful :%s \n", communityName);

    printf("Community Name: %s \n", str);
    printf("Access Type: %d \n", community.type);

    openapiOpenIPtoStringGet(clientHandle, community.ipAddr, &ipDesc);
    if (OPEN_AF_INET == community.ipAddr.family)
    {
      printf("IP address: %s \n", ipDescStr);

      openapiOpenIPtoStringGet(clientHandle, community.ipMask, &maskDesc);
      printf("IP address mask: %s \n", maskDescStr);
    }
    else if (OPEN_AF_INET6 == community.ipAddr.family)
    {
      printf("IPv6 address/Prefix Length: %s/%d \n", ipDescStr, community.ipv6PrefixLen);
    }
    printf("View Name: %s \n", community.viewName);
    printf("Status: %d \n", community.status);
  }

  PRINTBADRESULT(result, "testSnmpCommunityAndIPAddrGet", "");
  return;
}

/*******************************************************************
*
* @brief  This is the main function that will demonstrate 
*         SNMP OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main (int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switchOsRevision;
  char switchOsRevisionStr[100];

  if (argc < 2)
  {
    printSNMPAppMenu(argv[0]);
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("snmp_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (OPEN_E_NONE != openapiConnectivityCheck(&clientHandle))
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting SNMP API example application");

  printf("\n");
  switchOsRevision.pstart = switchOsRevisionStr;
  switchOsRevision.size = sizeof(switchOsRevisionStr);
  if (OPEN_E_NONE == openapiNetworkOSVersionGet(&clientHandle, &switchOsRevision))
  {
    printf("Network OS version = %s\n", switchOsRevisionStr);
  }
  else
  {
    printf("Network OS version retrieve error\n");
  }

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 3)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpSysNameSet(&clientHandle, argv[2]);
      break;

    case 2:
      if (argc != 2)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpSysNameGet(&clientHandle);
      break;

    case 3:
      if (argc != 3)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpSysLocationSet(&clientHandle, argv[2]);
      break;

    case 4:
      if (argc != 2)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpSysLocationGet(&clientHandle);
      break;

    case 5:
      if (argc != 3)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpSysContactSet(&clientHandle, argv[2]);
      break;

    case 6:
      if (argc != 2)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpSysContactGet(&clientHandle);
      break;

    case 7:
      if (argc != 3)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpTrapSourceInterfaceSet(&clientHandle, atoi(argv[2]));
      break;

    case 8:
      if (argc != 2)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpTrapSourceInterfaceGet(&clientHandle);
      break;

    case 9:
      if (argc != 4)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpTrapFlagSet(&clientHandle, atoi(argv[2]), atoi(argv[3]));
      break;

    case 10:
      if (argc != 3)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpTrapFlagGet(&clientHandle, atoi(argv[2]));
      break;

    case 11:
      if (argc != 3)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpLocalEngineIdSet(&clientHandle, argv[2]);
      break;

    case 12:
      if (argc != 2)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpLocalEngineIdGet(&clientHandle);
      break;

    case 13:
      if (argc != 2)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpLocalEngineIdDelete(&clientHandle);
      break;

    case 14:
      if (argc != 11)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpUserCreate(&clientHandle, argv[2], argv[3], atoi(argv[4]), argv[5], argv[6],
                         atoi(argv[7]), argv[8], argv[9], argv[10]);
      break;

    case 15:
      if (argc != 3)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpUserGet(&clientHandle, argv[2]);
      break;

    case 16:
      if (argc != 2)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpUserGet(&clientHandle, "");
      break;

    case 17:
      if (argc != 4)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpUserDelete(&clientHandle, argv[2], argv[3]);
      break;

    case 18:
      if (argc != 9)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpGroupCreate(&clientHandle, argv[2], atoi(argv[3]), atoi(argv[4]), argv[5], argv[6],
                          argv[7], argv[8]);
      break;

    case 19:
      if (argc != 3)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpGroupGet(&clientHandle, argv[2]);
      break;

    case 20:
      if (argc != 2)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpGroupGet(&clientHandle, "");
      break;

    case 21:
      if (argc != 6)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpGroupDelete(&clientHandle, argv[2], atoi(argv[3]), atoi(argv[4]), argv[5]);
      break;

    case 22:
      if (argc != 5)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpFilterCreate(&clientHandle, argv[2], argv[3], atoi(argv[4]));
      break;

    case 23:
      if (argc != 3)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpFilterGet(&clientHandle, argv[2]);
      break;

    case 24:
      if (argc != 2)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpFilterGet(&clientHandle, "");
      break;

    case 25:
      if (argc != 4)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpFilterDelete(&clientHandle, argv[2], argv[3]);
      break;

    case 26:
      if (argc != 5)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpViewCreate(&clientHandle, argv[2], argv[3], atoi(argv[4]));
      break;

    case 27:
      if (argc != 3)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpViewGet(&clientHandle, argv[2]);
      break;

    case 28:
      if (argc != 2)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpViewGet(&clientHandle, "");
      break;

    case 29:
      if (argc != 4)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpViewDelete(&clientHandle, argv[2], argv[3]);
      break;

    case 30:
      if (argc != 11)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpHostCreate(&clientHandle, argv[2], atoi(argv[3]), argv[4],
                         atoi(argv[5]), atoi(argv[6]), atoi(argv[7]),
                         atoi(argv[8]), atoi(argv[9]), argv[10]);
      break;

    case 31:
      if (argc != 3)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpHostGet(&clientHandle, argv[2]);
      break;

    case 32:
      if (argc != 2)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpHostGet(&clientHandle, "");
      break;

    case 33:
      if (argc != 4)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpHostDelete(&clientHandle, argv[2], atoi(argv[3]));
      break;

    case 34:
      if (argc != 6)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpCommunity(&clientHandle, argv[2], atoi(argv[3]), argv[4], argv[5]);
      break;

    case 35:
      if (argc != 5)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpCommunityGroup(&clientHandle, argv[2], argv[3], argv[4]);
      break;

    case 36:
      if (argc != 3)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpCommunityGet(&clientHandle, argv[2]);
      break;

    case 37:
      if (argc != 2)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpCommunityGet(&clientHandle, "");
      break;

    case 38:
      if (argc != 3)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpCommunityDelete(&clientHandle, argv[2]);
      break;
    /* Non CONF API Examples */
    case 39:
      if (argc != 6)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpCommunityWithIPAddrCreate(&clientHandle, argv[2], atoi(argv[3]), argv[4], argv[5], OPEN_SNMP_COMMUNITY_STATUS_VALID);
      break;

    case 40:
      if (argc != 7)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpCommunityWithIPAddrAndViewNameCreate(&clientHandle, argv[2], atoi(argv[3]), argv[4], argv[5], 0, argv[6], OPEN_SNMP_COMMUNITY_STATUS_VALID);
      break;

    case 41:
      if (argc != 6)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpCommunityWithIPAddrAndViewNameCreate(&clientHandle, argv[2], atoi(argv[3]), argv[4], "", atoi(argv[5]), "",OPEN_SNMP_COMMUNITY_STATUS_VALID);
      break;

    case 42:
      if (argc != 3)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpCommunityAndIPAddrGet(&clientHandle, argv[2]);
      break;

    case 43:
      if (argc != 3)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpCommunityAndIPAddrGetNext(&clientHandle, argv[2]);
      break;

    case 44:
      if (argc != 3)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpCommunityAndIPAddrDelete(&clientHandle, argv[2]);
      break;

    case 45:
      if (argc != 5)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpTrapManagerConfigCreate(&clientHandle, argv[2], argv[3], atoi(argv[4]), OPEN_SNMP_COMMUNITY_STATUS_VALID);
      break;

    case 46:
      if (argc != 4)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpTrapManagerConfigGet(&clientHandle, argv[2], argv[3]);
      break;

    case 47:
      if (argc != 4)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpTrapManagerConfigGetNext(&clientHandle, argv[2], argv[3]);
      break;

    case 48:
      if (argc != 4)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      testSnmpTrapManagerConfigDelete(&clientHandle, argv[2], argv[3]);
      break;

    /* Non CONF API Examples end */

    case 49:
      if (argc != 2)
      {
        printSNMPAppMenu(argv[0]);
        exit(1);
      }
      runSnmpSanity(&clientHandle);
      break;

    default:
      printSNMPAppMenu(argv[0]);
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping SNMP API example application");

  (void) openapiClientTearDown(&clientHandle);

  return 0;
}

