/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  dot1s_example.c
*
* @purpose   Spanning Tree APIs Example.
*
* @component OPEN
*
* @comments
*
* @create    19/10/2012
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_dot1s.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the Spanning Tree Configuration Example 
*         Application Menu.
*
* @param    none
*
* @returns  none
*
*********************************************************************/
void printDot1sAppMenu()
{
  printf("Usage: dot1s_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: Set Spanning Tree Operational Mode: dot1s_example 1 <mode> \n");
  printf("Test 2: Get Spanning Tree Operational Mode: dot1s_example 2 \n");
  printf("Test 3: Set Spanning Tree Interface BPDU Filter Mode: dot1s_example 3 <interface> <mode> \n");
  printf("Test 4: Get Spanning Tree Interface BPDU Filter Mode: dot1s_example 4 <interface> \n");
  printf("Test 5: Set Spanning Tree Interface BPDU Flood Mode: dot1s_example 5 <interface> <mode> \n");
  printf("Test 6: Get Spanning Tree Interface BPDU Flood Mode: dot1s_example 6 <interface> \n");
  printf("Test 7: Set Spanning Tree BPDU Guard Mode: dot1s_example 7 <mode> \n");
  printf("Test 8: Get Spanning Tree BPDU Guard Mode: dot1s_example 8 \n");
  printf("Test 9: Set Spanning Tree Configuration Name: dot1s_example 9 <name> \n");
  printf("Test 10: Get Spanning Tree Configuration Name: dot1s_example 10 \n");
  printf("Test 11: Set Spanning Tree Configuration Revision: dot1s_example 11 <revision> \n");
  printf("Test 12: Get Spanning Tree Configuration Revision: dot1s_example 12 \n");
  printf("Test 13: Set Spanning Tree Configuration Force Version: dot1s_example 13 <version> \n");
  printf("Test 14: Get Spanning Tree Configuration Force Version: dot1s_example 14 \n");
  printf("Test 15: Set Spanning Tree Interface Guard Mode: dot1s_example 15 <interface> <mode> \n");
  printf("Test 16: Get Spanning Tree Interface Guard Mode: dot1s_example 16 <interface> \n");
  printf("Test 17: Create Spanning Tree Instance: dot1s_example 17 <instance id> \n");
  printf("Test 18: Delete Spanning Tree Instance: dot1s_example 18 <instance id> \n");
  printf("Test 19: Get All Spanning Tree Instances: dot1s_example 19 \n");
  printf("Test 20: Set Spanning Tree MST Instance Priority: dot1s_example 20 <instance id> <priority> \n");
  printf("Test 21: Get Spanning Tree MST Instance Priority: dot1s_example 21 <instance id> \n");
  printf("Test 22: Add Spanning Tree MST Instance - VLAN Association: dot1s_example 22 <instance id> <vlanid> \n");
  printf("Test 23: Delete Spanning Tree MST Instance - VLAN Association: dot1s_example 23 <instance id> <vlanid> \n");
  printf("Test 24: Get All VLANs associated with the Spanning Tree Instance: dot1s_example 24 <instance id> \n");
  printf("Test 25: Get All Forwarding Database Ids associated with the Spanning Tree Instance: dot1s_example 25 <instance id> \n");
  printf("Test 26: Set Spanning Tree Interface Mode: dot1s_example 26 <interface> <mode> \n");
  printf("Test 27: Get Spanning Tree Interface Mode: dot1s_example 27 <interface> \n");
  printf("Test 28: Get Spanning Tree Interface Forwarding State: dot1s_example 28 <interface> <instance id> \n");
  printf("Test 29: Get Spanning Tree Interface Mode: dot1s_example 29 <interface> <instance id> \n");
  printf("Test 30: Set Spanning Tree Interface Edge Port Mode: dot1s_example 30 <interface> <mode> \n");
  printf("Test 31: Get Spanning Tree Interface Edge Port Mode: dot1s_example 31 <interface> \n");
  printf("Test 32: Get Spanning Tree Interface Port Loop Inconsistent State: dot1s_example 32 <interface> <instance id> \n");
  printf("Test 33: Get the Spanning Tree MST Bridge Identifier for a specific instance: dot1s_example 33 <instance id>\n");
  printf("Test 34: Get the Spanning Tree MST Time since the last Topology Change for a specific instance: dot1s_example 34 <instance id>\n");
  printf("Test 35: Get the Spanning Tree MST count of Topology Changes in a specific instance: dot1s_example 35 <instance id>\n");
  printf("Test 36: Get if a topology change is in progress for any port assigned to a specific Spanning Tree MST instance: dot1s_example 36 <instance id>\n");
  printf("Test 37: Get the Spanning Tree MST Designated Root Identifier in a specific instance: dot1s_example 37 <instance id>\n");
  printf("Test 38: Get the Spanning Tree MST Designated Root Path Cost parameter in a specific instance: dot1s_example 38 <instance id>\n");
  printf("Test 39: Get the Spanning Tree Port Root Port ID in a specific instance: dot1s_example 39 <instance id>\n");
  printf("Test 40: Get the max age for the CIST: dot1s_example 40\n");
  printf("Test 41: Get Max Hop count for the device: dot1s_example 41\n");
  printf("Test 42: Get the Tx Hold count value for the device: dot1s_example 42\n");
  printf("Test 43: Get the CIST Bridge Forward Delay: dot1s_example 43\n");
  printf("Test 44: Get the CIST Hello Time for a specific port: dot1s_example 44 <interface>\n");
  printf("Test 45: Get the CIST Bridge Hold Time: dot1s_example 45\n");
  printf("Test 46: Get the CIST Regional Root Identifier: dot1s_example 46\n");
  printf("Test 47: Get the CIST Regional Root Path Cost: dot1s_example 47\n");
  printf("Test 48: Get Spanning Tree MST Interface port details: dot1s_example 48 <instance id> <interface> \n");
  printf("Test 49: Set Spanning Tree MST Interface priority: dot1s_example 49 <instance id> <interface> <priority> \n");
  printf("Test 50: Get Spanning Tree MST Interface priority: dot1s_example 50 <instance id> <interface> \n");
  printf("Test 51: Get Spanning Tree MST Interface Path Cost Mode: dot1s_example 51 <instance id> <interface> \n");
  printf("Test 52: Set Spanning Tree MST Interface Cost: dot1s_example 52 <instance id> <interface> <priority> \n");
  printf("Test 53: Get Spanning Tree MST Interface Operational Cost: dot1s_example 53 <instance id> <interface> \n");
  printf("Test 54: Get Spanning Tree MST Interface Designated Root ID: dot1s_example 54 <instance id> <interface> \n");
  printf("Test 55: Get Spanning Tree MST Interface Designated Cost: dot1s_example 55 <instance id> <interface> \n");
  printf("Test 56: Get Spanning Tree MST Interface Designated Bridge ID: dot1s_example 56 <instance id> <interface> \n");
  printf("Test 57: Get Spanning Tree MST Interface Designated Port ID: dot1s_example 57 <instance id> <interface> \n");
  printf("Test 58: Get Spanning Tree auto calculate port path cost mode: dot1s_example 58 <interface> \n");
  printf("Test 59: Get Spanning Tree External Port Path Cost: dot1s_example 59 <interface> \n");
  printf("Test 60: Set Spanning Tree External Port Path Cost: dot1s_example 60 <interface> <cost> \n");
  printf("Test 61: Get Spanning Tree Topology Change ACK: dot1s_example 61 <interface> \n");
  printf("Test 62: Get Spanning Tree Hello Time: dot1s_example 62 \n");
  printf("Test 63: Get Spanning Tree Operational Edge Status: dot1s_example 63 <interface> \n");
  printf("Test 64: Get Spanning Tree Point-To-Point MAC Status: dot1s_example 64 <interface> \n");
  printf("Test 65: Get Spanning Tree Interface Regional Root ID: dot1s_example 65 <interface> \n");
  printf("Test 66: Get Spanning Tree Interface Port Path Cost: dot1s_example 66 <interface> \n");
  printf("Test 67: Get Spanning Tree MST Port Loop Inconsistant State: dot1s_example 67 <instance id> <interface> \n");
  printf("Test 68: Get Spanning Tree MST Transitions Into Loop Inconsistant: dot1s_example 68 <instance id> <interface> \n");
  printf("Test 69: Get Spanning Tree MST Transitions Out of Loop Inconsistant: dot1s_example 69 <instance id> <interface> \n");
  printf("Test 70: Clear Spanning Tree Interface Statistics: dot1s_example 70 <interface> \n");
  printf("Test 71: Get Spanning Tree Interface Hello Time: dot1s_example 71 <interface> \n");
  printf("Test 72: Get Spanning Tree Interface BPDU Guard Effect: dot1s_example 72 <interface> \n");
  printf("Test 73: Set Spanning Tree Interface Restricted TCN Parameter: dot1s_example 73 <interface> <mode> \n");
  printf("Test 74: Get Spanning Tree Interface Restricted TCN Parameter: dot1s_example 74 <interface> \n");
  printf("Test 75: Get Spanning Tree Interface Auto Edge Status: dot1s_example 75 <interface> \n");
  printf("Test 76: Set Spanning Tree Interface Auto Edge Status: dot1s_example 76 <interface> <mode> \n");
  printf("Test 77: Get Spanning Tree Interface Up Time: dot1s_example 77 <interface> \n");
  printf("Test 78: Get Spanning Tree Interface STP BPDUs Sent: dot1s_example 78 <interface> \n");
  printf("Test 79: Get Spanning Tree Interface STP BPDUs Received: dot1s_example 79 <interface> \n");
  printf("Test 80: Get Spanning Tree Interface RSTP BPDUs Sent: dot1s_example 80 <interface> \n");
  printf("Test 81: Get Spanning Tree Interface RSTP BPDUs Received: dot1s_example 81 <interface> \n");
  printf("Test 82: Get Spanning Tree Interface MSTP BPDUs Sent: dot1s_example 82 <interface> \n");
  printf("Test 83: Get Spanning Tree Interface MSTP BPDUs Received: dot1s_example 83 <interface> \n");
  printf("Test 84: Get the MSTP Bridge Max Age parameter for the CIST: dot1s_example 84 \n");
  printf("Test 85: Set the MSTP Bridge Max Age parameter for the CIST: dot1s_example 85 <maxAge>\n");
  printf("Test 86: Get the MSTP Bridge hello time parameter for the CIST: dot1s_example 86 \n");
  printf("Test 87: Get the MSTP Bridge forward delay for the CIST: dot1s_example 87 \n");
  printf("Test 88: Set the MSTP Bridge forward delay for the CIST: dot1s_example 88 <fwdDelay>\n");
  printf("Test 89: Set the MSTP max hop count parameter for the device: dot1s_example 89 <maxHopCount>\n");
  printf("Test 90: Set the MSTP transmit hold count parameter for the device: dot1s_example 90 <txHoldCount>\n");
  printf("Test 91: Get the MSTP root port forward delay for the CIST: dot1s_example 91 \n");
  printf("Test 92: Get the port ID that triggered topology change: dot1s_example 92 \n");
  printf("Test 93: Get the MSTP port path cost in a specific instance: dot1s_example 93 <mstId> <ifNum> \n");
  printf("Test 94: Test Spanning Tree OpEN APIs sanity: dot1s_example 94 <interface> \n");
  printf("Test 95: Get the MSTP Configuration Digest Key: dot1s_example 95\n");
  printf("Test 96: Get the MSTP Time since the last Topology Change in a specific instance: dot1s_example 96 <mstID>\n");
  printf("Test 97: Get the next port participating in MSTP associated with the specified MSTP instance ID: dot1s_example 97 <mstID><prevIntIfNum>\n");


  return;
}

/*********************************************************************
* @purpose  Enables/Disables Spanning Tree Operational Mode
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    mode            @b{(input)}   spanning tree operational mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiDot1SModeSet(clientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree Operational Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Operational Mode set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree Operational Mode.
*
* @param    clientHandle   @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t mode;
  
  if ((result = openapiDot1SModeGet(clientHandle, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Operational Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Operational Mode: %u \n", mode);
  }
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree BPDU Filter mode for an interface
*
* @param    clientHandle           @b{(input)}   client handle from registration API
* @param    ifNum                   @b{(input)}   ID of the interface
* @param    mode                    @b{(input)}   spanning tree BPDU filter mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SIntfBPDUFilterModeSet(openapiClientHandle_t *clientHandle, uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiDot1SIntfBPDUFilterModeSet(clientHandle, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree Interface BPDU Filter Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface BPDU Filter Mode set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree BPDU Filter mode for an interface
*
* @param    clientHandle           @b{(input)}   client handle from registration API
* @param    ifNum                   @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SIntfBPDUFilterModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t mode;
  
  if ((result = openapiDot1SIntfBPDUFilterModeGet(clientHandle, ifNum, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Interface BPDU Filter Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface BPDU Filter Mode for interface %u: %u \n", ifNum, mode);
  }
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree BPDU Flood mode for an interface
*
* @param    clientHandle         @b{(input)}   client handle from registration API
* @param    ifNum                 @b{(input)}   ID of the interface
* @param    mode                  @b{(input)}   spanning tree BPDU Flood mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SIntfBPDUFloodModeSet(openapiClientHandle_t *clientHandle, uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiDot1SIntfBPDUFloodModeSet(clientHandle, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree Interface BPDU Flood Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface BPDU Flood Mode set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree BPDU Flood mode for an interface
*
* @param    clientHandle          @b{(input)}   client handle from registration API
* @param    ifNum                  @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SIntfBPDUFloodModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t mode;
  
  if ((result = openapiDot1SIntfBPDUFloodModeGet(clientHandle, ifNum, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Interface BPDU Flood Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface BPDU Flood Mode for interface %u: %u \n", ifNum, mode);
  }
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree BPDU Guard Mode
*
* @param    clientHandle             @b{(input)}   client handle from registration API
* @param    mode                      @b{(input)}   spanning tree BPDU guard
*
* @returns  mode 
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SBPDUGuardModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiDot1SBPDUGuardModeSet(clientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree BPDU Guard Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree BPDU Guard Mode set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the Spanning Tree BPDU Guard Mode
*
* @param    clientHandle             @b{(input)}   client handle from registration API
*
* @returns  mode 
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SBPDUGuardModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t mode;
  
  if ((result = openapiDot1SBPDUGuardModeGet(clientHandle, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree BPDU Guard Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree BPDU Guard Mode: %u \n", mode);
  }
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree Configuration Name
*
* @param    clientHandle         @b{(input)}   client handle from registration API
* @param    name                  @b{(input)}   spanning tree configuration name
*                                               Populate dot1sConfigName:pstart and size with a char buffer
*                                               containing the config name
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SConfigNameSet(openapiClientHandle_t *clientHandle, char *name)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, name, (sizeof(str) - 1));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  if ((result = openapiDot1SConfigNameSet(clientHandle, &buffDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree Configuration Name. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Configuration Name set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the Spanning Tree Configuration Name
*
* @param    clientHandle         @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SConfigNameGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char *str;
  uint32_t dot1SMgmtStrLen;

  if ((result = openapiDot1sMgmtStrLenGet(clientHandle, OPEN_DOT1S_MGMT_STR_CONFIG_NAME, &dot1SMgmtStrLen)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n", result);
    return;
  }

  if ((str = (char*)malloc(dot1SMgmtStrLen)) == NULL) 
  {
    printf("Could not allocate memory.\n");
    return;
  }

  memset(str, 0, dot1SMgmtStrLen);
  buffDesc.pstart = str;
  buffDesc.size = dot1SMgmtStrLen;

  if ((result = openapiDot1SConfigNameGet(clientHandle, &buffDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Configuration Name. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Configuration Name: %s \n", str);
  }

  free(str);
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree Configuration Revision
*
* @param    clientHandle         @b{(input)}   client handle from registration API
* @param    revision              @b{(input)}   spanning tree configuration revision
*
* @returns  none 
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SConfigRevisionSet(openapiClientHandle_t *clientHandle, uint32_t revision)
{
  open_error_t result;
  
  if ((result = openapiDot1SConfigRevisionSet(clientHandle, revision)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree Configuration Revision. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Configuration Revision set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the Spanning Tree Configuration Revision
*
* @param    clientHandle         @b{(input)}   client handle from registration API
*
* @returns  none 
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SConfigRevisionGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t revision;
  
  if ((result = openapiDot1SConfigRevisionGet(clientHandle, &revision)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Configuration Revision. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Configuration Revision: %u \n", revision);
  }
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree Force Revision
*
* @param    clientHandle             @b{(input)}   client handle from registration API
* @param    forceVersion              @b{(input)}   spanning tree force version
*
* @returns  none 
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SForceVersionSet(openapiClientHandle_t *clientHandle, uint32_t forceVersion)
{
  open_error_t result;
  
  if ((result = openapiDot1SForceVersionSet(clientHandle, forceVersion)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree Force Version. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Force Version set successfully \n");
  }

  return;
}

/*********************************************************************
* @purpose  Gets the Spanning Tree Force Revision
*
* @param    clientHandle             @b{(input)}   client handle from registration API
*
* @returns  none 
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SForceVersionGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t forceVersion;
  
  if ((result = openapiDot1SForceVersionGet(clientHandle, &forceVersion)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Force Version. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Force Version: %u \n", forceVersion);
  }
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree Guard mode
*
* @param    clientHandle         @b{(input)}   client handle from registration API
* @param    ifNum                 @b{(input)}   ID of the interface
* @param    mode                  @b{(input)}   spanning tree Guard mode
*
* @returns  none 
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SIntfGuardModeSet(openapiClientHandle_t *clientHandle, uint32_t ifNum, uint32_t mode)
{
  open_error_t result;
  
  if ((result = openapiDot1SIntfGuardModeSet(clientHandle, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree Interface Guard Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface Guard Mode set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree Guard mode
*
* @param    clientHandle          @b{(input)}   client handle from registration API
* @param    ifNum                  @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SIntfGuardModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_DOT1S_PORT_GUARD_t mode;
  
  if ((result = openapiDot1SIntfGuardModeGet(clientHandle, ifNum, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Interface Guard Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface Guard Mode for interface %u: %u \n", ifNum, mode);
  }
  return;
}

/*********************************************************************
* @purpose  Creates/Deletes Spanning Tree MST Instance
*
* @param    clientHandle             @b{(input)}   client handle from registration API
* @param    dot1sMSTInstanceId        @b{(input)}   spanning tree MST instance Id
* @param    action                    @b{(input)}   create/delete instance
*
* @returns  none 
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SMSTInstanceSet(openapiClientHandle_t *clientHandle, uint32_t dot1sMSTInstanceId, OPEN_ACTION_t action)
{
  open_error_t result;
  
  if ((result = openapiDot1SMSTInstanceSet(clientHandle, dot1sMSTInstanceId, action)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure Spanning Tree MST Instance. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Instance set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets all Spanning Tree MST Instances in the switch
*
* @param    clientHandle             @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SMSTInstancesGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t currInstanceId, nextInstanceId;
  
  if ((result = openapiDot1SMSTInstanceFirstGet(clientHandle, &currInstanceId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the first Spanning Tree MST Instance. (result = %d)\n", result);
    return;
  }
  else
  {
    printf("Spanning Tree MST Instances:\n");
    printf("%u", currInstanceId);
  }

  while(openapiDot1SMSTInstanceNextGet(clientHandle, currInstanceId, &nextInstanceId) == OPEN_E_NONE) 
  {
    printf(", %u", nextInstanceId);
    currInstanceId = nextInstanceId;
  }
  printf("\n");
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree MST Instance priority
*
* @param    clientHandle              @b{(input)}   client handle from registration API
* @param    dot1sMSTInstance          @b{(input)}   spanning tree MST instance
* @param    priority                  @b{(input)}   instance priority
*
* @returns  none  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SMSTInstancePrioritySet(openapiClientHandle_t *clientHandle, uint32_t dot1sMSTInstance, uint32_t priority)
{
  open_error_t result;
  
  if ((result = openapiDot1SMSTInstancePrioritySet(clientHandle, dot1sMSTInstance, priority)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree Instance Priority. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Instance Priority set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree MST Instance priority
*
* @param    clientHandle              @b{(input)}   client handle from registration API
* @param    dot1sMSTInstance          @b{(input)}   spanning tree MST instance
*
* @returns  none  
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SMSTInstancePriorityGet(openapiClientHandle_t *clientHandle, uint32_t dot1sMSTInstance)
{
  open_error_t result;
  uint16_t priority;
  
  if ((result = openapiDot1SMSTInstancePriorityGet(clientHandle, dot1sMSTInstance, &priority)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Instance Priority. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Instance Priority for instance %u: %u \n", dot1sMSTInstance, priority);
  }
  return;
}

/*********************************************************************
* @purpose  Adds/Deletes Spanning Tree MST Instance-VLAN association
*
* @param    clientHandle              @b{(input)}   client handle from registration API
* @param    dot1sMSTInstanceId        @b{(input)}   spanning tree MST instance
* @param    vlanId                    @b{(input)}   VLAN
* @param    action                    @b{(input)}   create/delete association
*
* @returns  none  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SMSTInstanceVLANSet(openapiClientHandle_t *clientHandle, uint32_t dot1sMSTInstanceId, 
                             uint32_t vlanId, OPEN_ACTION_t action)
{
  open_error_t result;
  uint32_t vlanMax;

  result = openapiDot1VlanMaxGet(clientHandle, &vlanMax);

  if (result == OPEN_E_NONE)
  {
    if ((vlanId > vlanMax))
    {
      printf("Invalid VLAN id. The maximum id is %d.\n", vlanMax);
      result = OPEN_E_PARAM;
    }
  }

  if (result == OPEN_E_NONE)
  {
    if ((result = openapiDot1SMSTInstanceVLANSet(clientHandle, dot1sMSTInstanceId, vlanId, action)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to configure Spanning Tree MST Instance - VLAN association. (result = %d)\n", result);
    }
    else
    {
      printf("Spanning Tree MST Instance - VLAN association set successfully \n");
    }
  }

  return;
}

/*********************************************************************
* @purpose  Gets all VLAN Ids for the Spanning Tree MST Instance
*
* @param    clientHandle              @b{(input)}   client handle from registration API
* @param    dot1sMSTInstanceId        @b{(input)}   spanning tree MST instance
*
* @returns  none
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SMSTInstanceVLANsGet(openapiClientHandle_t *clientHandle, uint32_t dot1sMSTInstanceId)
{
  open_error_t result;
  uint32_t currVLANId, nextVLANId;
  
  if ((result = openapiDot1SMSTInstanceFirstVLANGet(clientHandle, dot1sMSTInstanceId, &currVLANId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the first Spanning Tree MST VLAN. (result = %d)\n", result);
    return;
  }
  else
  {
    printf("Spanning Tree MST VLANs:\n");
    printf("%u", currVLANId);
  }

  while(openapiDot1SMSTInstanceNextVLANGet(clientHandle, dot1sMSTInstanceId, currVLANId, &nextVLANId) == OPEN_E_NONE) 
  {
    printf(", %u", nextVLANId);
    currVLANId = nextVLANId;
  }
  printf("\n");
  return;
}

/*********************************************************************
* @purpose  Gets all Forwarding Database Ids for the Spanning Tree MST Instance
*
* @param    clientHandle              @b{(input)}   client handle from registration API
* @param    dot1sMSTInstanceId        @b{(input)}   spanning tree MST instance
*
* @returns  none  
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SMSTInstanceFIdsGet(openapiClientHandle_t *clientHandle, uint32_t dot1sMSTInstanceId)
{
  open_error_t result;
  uint32_t currFId, nextFId;
  
  if ((result = openapiDot1SMSTInstanceFirstFIdGet(clientHandle, dot1sMSTInstanceId, &currFId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the first Spanning Tree MST Forwarding Database Id. (result = %d)\n", result);
    return;
  }
  else
  {
    printf("Spanning Tree MST FIds:\n");
    printf("%u", currFId);
  }

  while(openapiDot1SMSTInstanceNextFIdGet(clientHandle, dot1sMSTInstanceId, currFId, &nextFId) == OPEN_E_NONE) 
  {
    printf(", %u", nextFId);
    currFId = nextFId;
  }

  printf("\n");
  return;
}

/*********************************************************************
* @purpose  Enables/Disables Spanning Tree Admin Mode on an interface
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    ifNum           @b{(input)}   ID of the interface
* @param    mode            @b{(input)}   spanning tree mode
*
* @returns  none 
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SIntfModeSet(openapiClientHandle_t *clientHandle, uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiDot1SIntfModeSet(clientHandle, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree Interface Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface Mode set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree Admin Mode on an interface
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    ifNum           @b{(input)}   ID of the interface
*
* @returns  none  
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SIntfModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t mode;
  
  if ((result = openapiDot1SIntfModeGet(clientHandle, ifNum, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface Mode for interface %u: %u \n", ifNum, mode);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree Port Forwarding State
*
* @param    clientHandle          @b{(input)}   client handle from registration API
* @param    ifNum                  @b{(input)}   ID of the interface
* @param    dot1sInstance          @b{(input)}   spanning tree MST instance
*
* @returns  none 
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SIntfPortFwdStateGet(openapiClientHandle_t *clientHandle, uint32_t ifNum, uint32_t dot1sInstance)
{
  open_error_t result;
  OPEN_DOT1S_PORT_FWD_STATE_t dot1sPortFwdState;
  
  if ((result = openapiDot1SIntfPortFwdStateGet(clientHandle, ifNum, dot1sInstance, &dot1sPortFwdState)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Port Forwarding State. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface Port Forwarding State for interface %u: %u \n", ifNum, dot1sPortFwdState);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree Port Role
*
* @param    clientHandle          @b{(input)}   client handle from registration API
* @param    ifNum                  @b{(input)}   ID of the interface
* @param    dot1sInstance          @b{(input)}   spanning tree MST instance
*
* @returns  none  
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SIntfPortRoleGet(openapiClientHandle_t *clientHandle, uint32_t ifNum, uint32_t dot1sInstance)
{
  open_error_t result;
  OPEN_DOT1S_PORT_ROLE_t dot1sPortRole;
  
  if ((result = openapiDot1SIntfPortRoleGet(clientHandle, ifNum, dot1sInstance, &dot1sPortRole)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Port Role. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Port Role for interface %u: %u \n", ifNum, dot1sPortRole);
  }
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree Edge Port mode for an interface
*
* @param    clientHandle           @b{(input)}   client handle from registration API
* @param    ifNum                  @b{(input)}   ID of the interface
* @param    mode                   @b{(input)}   spanning tree edge port mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SIntfEdgePortModeSet(openapiClientHandle_t *clientHandle, uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiDot1SIntfEdgePortModeSet(clientHandle, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree Interface Edge Port Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface Edge Port Mode set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree Edge Port mode for an interface
*
* @param    clientHandle           @b{(input)}   client handle from registration API
* @param    ifNum                  @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SIntfEdgePortModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t mode;
  
  if ((result = openapiDot1SIntfEdgePortModeGet(clientHandle, ifNum, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Interface Edge Port Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface Edge Port Mode for interface %u: %u \n", ifNum, mode);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree Port Loop Inconsistent State
*
* @param    clientHandle           @b{(input)}   client handle from registration API
* @param    ifNum                  @b{(input)}   ID of the interface
* @param    dot1sInstance          @b{(input)}   spanning tree MST instance
*
* @returns  none 
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SIntfPortLoopInconsistentStateGet(openapiClientHandle_t *clientHandle, uint32_t ifNum, uint32_t dot1sInstance)
{
  open_error_t result;
  OPEN_BOOL_t loopInconsistentState;
  
  if ((result = openapiDot1SIntfPortLoopInconsistentStateGet(clientHandle, ifNum, dot1sInstance, &loopInconsistentState)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Port Loop Inconsistent State. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface Port Loop Inconsistent State for interface %u: %u \n", ifNum, loopInconsistentState);
  }
  return;
}

/*********************************************************************
* @purpose  Convert a bridge ID to a string of hex digits.
*
* @param    id               @b{(input)}  instance of OPEN_DOT1S_BRIDGEID_t
* @param    idStr            @b{(output)} resulting string
* @param    size             @b{(input)}  size of idStr
*
* @returns  pointer to idStr
*
* @end
*********************************************************************/
static char * idToStr(OPEN_DOT1S_BRIDGEID_t *id, char *idStr, int size)
{
  unsigned char *p = (unsigned char *) id;
  snprintf(idStr, size, "%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X",
           p[0], p[1], p[2], p[3], p[4], p[5], p[6], p[7]);
  return idStr;
}

/*********************************************************************
* @purpose  Gets the Spanning Tree MST Bridge Identifier for a specific instance.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mstId               @b{(input)}  the MSTID for the desired MSTI
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sMSTInstanceBridgeIdentifierGet(openapiClientHandle_t *client_handle, uint32_t mstId)
{
  open_error_t result;
  OPEN_DOT1S_BRIDGEID_t id;
  char idStr[25];

  if ((result = openapiDot1sMSTInstanceBridgeIdentifierGet(client_handle, mstId, &id)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST Bridge Identifier for a specific instance. (result = %d)\n", result);
  }
  else
  {
    printf("MST Instance Bridge Identifier for mstId %d: %s\n", mstId, idToStr(&id, idStr, sizeof(idStr)));
  }
  return;
}

/*********************************************************************
* @purpose  Gets the Spanning Tree MST Time since the last Topology Change for a specific instance.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mstId               @b{(input)}  the MSTID for the desired MSTI
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sMSTInstanceTimeSinceTopologyChangeGet(openapiClientHandle_t *client_handle, uint32_t mstId)
{
  open_error_t result;
  uint32_t tVal;

  if ((result = openapiDot1sMSTInstanceTimeSinceTopologyChangeGet(client_handle, mstId, &tVal)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST Time since the last Topology Change for a specific instance. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Time since the last Topology Change for mstId %d: %u seconds\n", mstId, tVal);
  }
  return;
}

/*********************************************************************
* @purpose  Get the Spanning Tree MST count of Topology Changes in a specific instance.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mstId               @b{(input)}  the MSTID for the desired MSTI
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sMSTInstanceTopologyChangeCountGet(openapiClientHandle_t *client_handle, uint32_t mstId)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiDot1sMSTInstanceTopologyChangeCountGet(client_handle, mstId, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST count of Topology Changes in a specific instance. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST count of Topology Changes for mstId %d: %u\n", mstId, count);
  }
  return;
}

/*********************************************************************
* @purpose  Get the Spanning Tree MST Topology Change in progress state.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mstId               @b{(input)}  the MSTID for the desired MSTI
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sMSTInstanceTopologyChangeStateGet(openapiClientHandle_t *client_handle, uint32_t mstId)
{
  open_error_t result;
  OPEN_BOOL_t state;

  if ((result = openapiDot1sMSTInstanceTopologyChangeStateGet(client_handle, mstId, &state)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST Topology Change state in a specific instance. (result = %d)\n", result);
  }
  else
  {
    printf("A topology change is %s for some port in the MST instance %d.\n", (state == OPEN_TRUE ? "in progress":"not in progress"), mstId);
  }
  return;
}

/*********************************************************************
* @purpose  Get the Spanning Tree MST Designated Root Identifier in a specific instance.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mstId               @b{(input)}  the MSTID for the desired MSTI
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sMSTInstanceDesignatedRootIDGet(openapiClientHandle_t *client_handle,uint32_t mstId)
{
  open_error_t result;
  OPEN_DOT1S_BRIDGEID_t id;
  char idStr[25];

  if ((result = openapiDot1sMSTInstanceDesignatedRootIDGet(client_handle, mstId, &id)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST Designated Root Identifier in a specific instance. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Designated Root Identifier for mstId %d: %s\n", mstId, idToStr(&id, idStr, sizeof(idStr)));
  }
  return;
}

/*********************************************************************
* @purpose  Get the Spanning Tree MST Designated Root Path Cost parameter in a specific instance
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mstId               @b{(input)}  the MSTID for the desired MSTI
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sMSTInstanceRootPathCostGet(openapiClientHandle_t *client_handle, uint32_t mstId)
{
  open_error_t result;
  uint32_t cost;

  if ((result = openapiDot1sMSTInstanceRootPathCostGet(client_handle, mstId, &cost)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST Designated Root Path Cost parameter in a specific instance. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Designated Root path cost for mstId %d: %u\n", mstId, cost);
  }
  return;
}

/*********************************************************************
* @purpose  Get the Spanning Tree MST Root Port ID in a specific instance.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mstId               @b{(input)}  the MSTID for the desired MSTI
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sMSTInstanceRootPortIDGet(openapiClientHandle_t *client_handle, uint32_t mstId)
{
  open_error_t result;
  uint32_t port;

  if ((result = openapiDot1sMSTInstanceRootPortIDGet(client_handle, mstId, &port)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST Root Port ID in a specific instance. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST root port ID for mstId %d: %u\n", mstId, port);
  }
  return;
}

/*********************************************************************
* @purpose  Get the max age for the CIST.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sCISTMaxAgeGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  uint32_t age;

  if ((result = openapiDot1sCISTMaxAgeGet(client_handle, &age)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the max age for the CIST. (result = %d)\n", result);
  }
  else
  {
    printf("Max age for the CIST: %u\n", age);
  }
  return;
}

/*********************************************************************
* @purpose  Get the Max Hop count for the bridge.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sBridgeMaxHopGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiDot1sBridgeMaxHopGet(client_handle, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Max Hop count for the bridge. (result = %d)\n", result);
  }
  else
  {
    printf("Max hop count for the bridge: %u\n", count);
  }
  return;
}

/*********************************************************************
* @purpose  Get the Spanning Tree MST Tx Hold count parameter for the bridge.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sBridgeTxHoldCountGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiDot1sBridgeTxHoldCountGet(client_handle, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST Tx Hold count for the bridge. (result = %d)\n", result);
  }
  else
  {
    printf("Tx hold count for the bridge: %u\n", count);
  }
  return;
}

/*********************************************************************
* @purpose  Get the Spanning Tree MST Bridge Forward Delay for the CIST.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sCISTRootFwdDelayGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  uint32_t delay;

  if ((result = openapiDot1sCISTRootFwdDelayGet(client_handle, &delay)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST Bridge Forward Delay for the CIST. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Bridge Forward Delay for the CIST: %u\n", delay);
  }
  return;
}


/*********************************************************************
* @purpose  Get the Spanning Tree MST Hello Time of a specific port for the CIST.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ifNum               @b{(input)}  interface number
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sCISTHelloTimeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result;
  uint32_t helloTime;

  if ((result = openapiDot1SIntfHelloTimeGet(client_handle, ifNum, &helloTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST Hello Time of a specific port for the CIST. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Hello Time of port %u for the CIST: %u\n", ifNum, helloTime);
  }
  return;
}

/*********************************************************************
* @purpose  Get the Spanning Tree MST Bridge Hold Time for the CIST.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sCISTBridgeHoldTimeGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  uint32_t holdTime;

  if ((result = openapiDot1sCISTBridgeHoldTimeGet(client_handle, &holdTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST Bridge Hold Time for the CIST. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Bridge Hold Time for the CIST: %u\n", holdTime);
  }
  return;
}

/*********************************************************************
* @purpose  Get the Spanning Tree MST CIST Regional Root Identifier.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sCISTRegionalRootIDGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  OPEN_DOT1S_BRIDGEID_t rootId;
  char idStr[25];

  if ((result = openapiDot1sCISTRegionalRootIDGet(client_handle, &rootId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST CIST Regional Root Identifier. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST CIST Regional Root Identifier: %s\n", idToStr(&rootId, idStr, sizeof(idStr)));
  }
  return;
}

/*********************************************************************
* @purpose  Get the Spanning Tree MST CIST Regional Root Path Cost.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sCISTRegionalRootPathCostGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  uint32_t cost;

  if ((result = openapiDot1sCISTRegionalRootPathCostGet(client_handle, &cost)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST CIST Regional Root Path Cost. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST CIST Regional Root Path Cost: %u\n", cost);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree MST port details for an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    dot1sMSTInstance   @b{(input)}   spanning tree MST instance
* @param    ifNum              @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SMSTInstanceIntfPortIDGet(openapiClientHandle_t *clientHandle, 
                                   uint32_t dot1sMSTInstance,
                                   uint32_t ifNum)
{
  open_error_t   result;
  open_buffdesc  buffdesc;
  char           str[256];

  buffdesc.pstart = str;
  buffdesc.size = sizeof(str);

  if ((result = openapiDot1SMSTIntfPortIDGet(clientHandle, 
                                             dot1sMSTInstance, 
                                             ifNum, &buffdesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree MST Instance Port identifier fo interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Instance Port ID for interface %u %u: %s \n", dot1sMSTInstance, ifNum, str);
  }
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree MST priority for an interface
*
* @param    clientHandle           @b{(input)}   client handle from registration API
* @param    dot1sMSTInstance       @b{(input)}   spanning tree MST instance
* @param    ifNum                  @b{(input)}   ID of the interface
* @param    priority               @b{(input)}   spanning tree MST port priority 
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SMSTInstanceIntfPrioritySet(openapiClientHandle_t *clientHandle, 
                                     uint32_t dot1sMSTInstance,
                                     uint32_t ifNum, uint32_t priority)
{
  open_error_t result;
  
  if ((result = openapiDot1SMstiPortPrioritySet(clientHandle, dot1sMSTInstance, 
                                                ifNum, priority)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree MST priority for an Interface . (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface MST Port Priority set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree MST port priority for an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    dot1sMSTInstance   @b{(input)}   spanning tree MST instance
* @param    ifNum              @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SMSTInstanceIntfPriorityGet(openapiClientHandle_t *clientHandle, 
                                     uint32_t dot1sMSTInstance,
                                     uint32_t ifNum)
{
  open_error_t   result;
  uint32_t       priority = 0;

  if ((result = openapiDot1SMstiPortPriorityGet(clientHandle, 
                                                dot1sMSTInstance, 
                                                ifNum, &priority)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree MST Instance Priority for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Instance Priority for interface %u %u: %u \n", dot1sMSTInstance, ifNum, priority);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree MST port path cost mode for an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    dot1sMSTInstance   @b{(input)}   spanning tree MST instance
* @param    ifNum              @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SMSTInstancePortPathCostModeGet(openapiClientHandle_t *clientHandle, 
                                         uint32_t dot1sMSTInstance,
                                         uint32_t ifNum)
{
  open_error_t   result;
  OPEN_CONTROL_t mode;

  if ((result = openapiDot1SMstiPortPathCostModeGet(clientHandle, 
                                                    dot1sMSTInstance, 
                                                    ifNum, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree MST Instance path cost mode for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Instance Path Cost Mode for interface %u %u: %u \n", dot1sMSTInstance, ifNum, mode);
  }
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree MST path cost for an interface
*
* @param    clientHandle           @b{(input)}   client handle from registration API
* @param    dot1sMSTInstance       @b{(input)}   spanning tree MST instance
* @param    ifNum                  @b{(input)}   ID of the interface
* @param    cost                   @b{(input)}   spanning tree MST port path cost 
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SMSTInstancePortPathCostSet(openapiClientHandle_t *clientHandle, 
                                     uint32_t dot1sMSTInstance,
                                     uint32_t ifNum, uint32_t cost)
{
  open_error_t result;
  
  if ((result = openapiDot1SMstiPortPathCostSet(clientHandle, dot1sMSTInstance, 
                                                ifNum, cost)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree MST Path Cost for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface MST Port Path Cost set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree MST Operational port path cost for an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    dot1sMSTInstance   @b{(input)}   spanning tree MST instance
* @param    ifNum              @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SMSTInstanceOperPortPathCostGet(openapiClientHandle_t *clientHandle, 
                                         uint32_t dot1sMSTInstance,
                                         uint32_t ifNum)
{
  open_error_t   result;
  uint32_t       cost = 0;

  if ((result = openapiDot1SMstiOperPortPathCostGet(clientHandle, 
                                                    dot1sMSTInstance, 
                                                    ifNum, &cost)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree MST Instance Operatinal Path Cost for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Instance Operational Path Cost for interface %u %u: %u \n", dot1sMSTInstance, ifNum, cost);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree MST Designated Root ID for an interface
*
* @param    clientHandle       @b{(input)} client handle from registration API
* @param    dot1sMSTInstance   @b{(input)} spanning tree MST instance
* @param    ifNum              @b{(input)} ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SMstiPortDesignatedRootIDGet(openapiClientHandle_t *clientHandle, 
                                      uint32_t dot1sMSTInstance,
                                      uint32_t ifNum)
{
  open_error_t   result;
  open_buffdesc  buffdesc;
  char           str[256];

  buffdesc.pstart = str;
  buffdesc.size = sizeof(str);

  if ((result = openapiDot1SMstiPortDesignatedRootIDGet(clientHandle, 
                                                        dot1sMSTInstance, 
                                                        ifNum, &buffdesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree MST Instance Designationed Root ID for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Instance Designated Root ID for interface %u %u: %s \n", dot1sMSTInstance, ifNum, str);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree MST designated port path cost for an interface
*
* @param    clientHandle       @b{(input)} client handle from registration API
* @param    dot1sMSTInstance   @b{(input)} spanning tree MST instance
* @param    ifNum              @b{(input)} ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SMstiPortDesignatedCostGet(openapiClientHandle_t *clientHandle, 
                                    uint32_t dot1sMSTInstance,
                                    uint32_t ifNum)
{
  open_error_t   result;
  uint32_t       cost = 0;

  if ((result = openapiDot1SMstiDesignatedCostGet(clientHandle, 
                                                  dot1sMSTInstance, 
                                                  ifNum, &cost)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree MST Instance Designated Port Path Cost for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Instance Designated Port Path Cost for interface %u %u: %u \n", dot1sMSTInstance, ifNum, cost);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree MST Designated Bridge ID for an interface
*
* @param    clientHandle       @b{(input)} client handle from registration API
* @param    dot1sMSTInstance   @b{(input)} spanning tree MST instance
* @param    ifNum              @b{(input)} ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SMstiPortDesignatedBridgeIDGet(openapiClientHandle_t *clientHandle, 
                                        uint32_t dot1sMSTInstance,
                                        uint32_t ifNum)
{
  open_error_t   result;
  open_buffdesc  buffdesc;
  char           str[256];

  buffdesc.pstart = str;
  buffdesc.size = sizeof(str);

  if ((result = openapiDot1SMstiDesignatedBridgeIDGet(clientHandle, 
                                                      dot1sMSTInstance, 
                                                      ifNum, &buffdesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree MST Instance Designationed Bridge ID for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Instance Designated Bridge ID for interface %u %u: %s \n", dot1sMSTInstance, ifNum, str);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree MST Designated Port ID for an interface
*
* @param    clientHandle       @b{(input)} client handle from registration API
* @param    dot1sMSTInstance   @b{(input)} spanning tree MST instance
* @param    ifNum              @b{(input)} ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SMstiPortDesignatedPortIDGet(openapiClientHandle_t *clientHandle, 
                                      uint32_t dot1sMSTInstance,
                                      uint32_t ifNum)
{
  open_error_t   result;
  open_buffdesc  buffdesc;
  char           str[256];

  buffdesc.pstart = str;
  buffdesc.size = sizeof(str);

  if ((result = openapiDot1SMstiDesignatedPortIDGet(clientHandle, 
                                                    dot1sMSTInstance, 
                                                    ifNum, &buffdesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree MST Instance Designationed Port ID for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Instance Designated Port ID for interface %u %u: %s \n", dot1sMSTInstance, ifNum, str);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree CIST external port path cost mode for an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    ifNum              @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SCistExtPortPathCostModeGet(openapiClientHandle_t *clientHandle, 
                                     uint32_t ifNum)
{
  open_error_t   result;
  OPEN_CONTROL_t mode;

  if ((result = openapiDot1SCistExtPortPathCostModeGet(clientHandle, 
                                                       ifNum, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree CIST External Path Cost Mode for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree CIST External Path Cost Mode for interface %u: %u \n", ifNum, mode);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree CIST external port path cost for an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    ifNum              @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SCistExtPortPathCostGet(openapiClientHandle_t *clientHandle, 
                                 uint32_t ifNum)
{
  open_error_t   result;
  uint32_t       cost = 0;

  if ((result = openapiDot1SCistExtPortPathCostGet(clientHandle, 
                                                   ifNum, &cost)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree CIST External Path Cost for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree CIST External Path Cost for interface %u: %u \n", ifNum, cost);
  }
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree CIST external path cost for an interface
*
* @param    clientHandle    @b{(input)} client handle from registration API
* @param    ifNum           @b{(input)} ID of the interface
* @param    cost            @b{(input)} spanning tree CIST external port path cost 
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SCistExtPortPathCostSet(openapiClientHandle_t *clientHandle, 
                                 uint32_t ifNum, uint32_t cost)
{
  open_error_t result;
  
  if ((result = openapiDot1SCistExtPortPathCostSet(clientHandle, 
                                                   ifNum, cost)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree CIST External Path Cost for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree CIST External Port Path Cost set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree CIST Topology Change ACK for an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    ifNum              @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SCistPortTopologyChangeAckGet(openapiClientHandle_t *clientHandle, 
                                       uint32_t ifNum)
{
  open_error_t   result;
  OPEN_BOOL_t    val = 0;

  if ((result = openapiDot1SCistPortTopologyChangeAckGet(clientHandle, 
                                                         ifNum, &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree CIST Topology Change ACK for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree CIST Topology Change ACK for interface %u : %u \n", ifNum, val);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree CIST Hello time 
*
* @param    clientHandle       @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SCistHelloTimeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t   result;
  uint32_t       val = 0;

  if ((result = openapiDot1SCistHelloTimeGet(clientHandle, &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree CIST Hello time. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree CIST Hello time %u \n", val);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree CIST Operational Edge Port status for an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    ifNum              @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SCistPortOperEdgeGet(openapiClientHandle_t *clientHandle, 
                              uint32_t ifNum)
{
  open_error_t   result;
  OPEN_BOOL_t    val;

  if ((result = openapiDot1SCistPortOperEdgeGet(clientHandle, 
                                                ifNum, &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree CIST Operational Edge Port Status for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree CIST Operational Edge Port Status for interface %u : %u \n", ifNum, val);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree CIST Point-to-Point MAC status for an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    ifNum              @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SCistPortOperPointToPointGet(openapiClientHandle_t *clientHandle, 
                                      uint32_t ifNum)
{
  open_error_t   result;
  OPEN_BOOL_t    val;

  if ((result = openapiDot1SCistPortOperPointToPointGet(clientHandle, 
                                                        ifNum, &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree CIST Point-to-Point MAC status for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree CIST Point-to-Point MAC status for interface %u : %u \n", ifNum, val);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree CIST Regional Root ID for an interface
*
* @param    clientHandle       @b{(input)} client handle from registration API
* @param    ifNum              @b{(input)} ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SCistPortRegionalRootIDGet(openapiClientHandle_t *clientHandle, 
                                    uint32_t ifNum)
{
  open_error_t   result;
  open_buffdesc  buffdesc;
  char           str[256];

  buffdesc.pstart = str;
  buffdesc.size = sizeof(str);

  if ((result = openapiDot1SCistPortRegionalRootIDGet(clientHandle, 
                                                      ifNum, &buffdesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree CIST Regional Root ID for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree CIST Regional Root ID for interface %u: %s \n", ifNum, str);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree CIST Internal Root path cost for an interface
*
* @param    clientHandle       @b{(input)} client handle from registration API
* @param    ifNum              @b{(input)} ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SCistPortPathCostGet(openapiClientHandle_t *clientHandle, 
                              uint32_t ifNum)
{
  open_error_t   result;
  uint32_t       cost = 0;

  if ((result = openapiDot1SCistPortPathCostGet(clientHandle, 
                                                ifNum, &cost)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree CIST Internal Root path cost for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree CIST Internal Root path cost for interface %u: %u \n", ifNum, cost);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree MST Loop Inconsistent State for an interface
*
* @param    clientHandle       @b{(input)} client handle from registration API
* @param    dot1sMSTInstance   @b{(input)} spanning tree MST instance
* @param    ifNum              @b{(input)} ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SMstiPortLoopInconsistentStateGet(openapiClientHandle_t *clientHandle, 
                                           uint32_t dot1sMSTInstance,
                                           uint32_t ifNum)
{
  open_error_t   result;
  OPEN_BOOL_t    val;

  if ((result = openapiDot1SMstiPortLoopInconsistentStateGet(clientHandle, 
                                                             dot1sMSTInstance, 
                                                             ifNum, &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree MST Loop Inconsistent State for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Loop Inconsistent State for interface %u %u: %u \n", dot1sMSTInstance, ifNum, val);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree MST transitions into loop inconsistent state for an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    dot1sMSTInstance   @b{(input)}   spanning tree MST instance
* @param    ifNum              @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SPortStatsTransitionsIntoLoopInconsistentStateGet(openapiClientHandle_t *clientHandle, 
                                                           uint32_t dot1sMSTInstance,
                                                           uint32_t ifNum)
{
  open_error_t   result;
  uint32_t       count = 0;

  if ((result = openapiDot1SPortStatsTransitionsIntoLoopInconsistentStateGet(clientHandle, 
                                                                             dot1sMSTInstance, 
                                                                             ifNum, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree MST transitions into loop inconsistent state for an interface. (result = %d)\n",     result);
  }
  else
  {
    printf("Spanning Tree MST transitions into loop inconsistent state for interface %u %u: %u \n", 
           dot1sMSTInstance, ifNum, count);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree MST transitions out of loop inconsistent state for an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    dot1sMSTInstance   @b{(input)}   spanning tree MST instance
* @param    ifNum              @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SPortStatsTransitionsOutOfLoopInconsistentStateGet(openapiClientHandle_t *clientHandle, 
                                                            uint32_t dot1sMSTInstance,
                                                            uint32_t ifNum)
{
  open_error_t   result;
  uint32_t       count = 0;

  if ((result = openapiDot1SPortStatsTransitionsOutOfLoopInconsistentStateGet(clientHandle, 
                                                                              dot1sMSTInstance, 
                                                                              ifNum, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree MST transitions out of loop inconsistent state for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST transitions out of loop inconsistent state for interface %u %u: %u \n", 
           dot1sMSTInstance, ifNum, count);
  }
  return;
}
 
/*********************************************************************
* @purpose  Clears Spanning Tree CIST statistics for an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    ifNum              @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SPortStatsClear(openapiClientHandle_t *clientHandle, 
                         uint32_t ifNum)
{
  open_error_t   result;

  if ((result = openapiDot1SPortStatsClear(clientHandle, 
                                           ifNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to clear Spanning CIST statistics for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree CIST statistics cleared for interface %u\n", ifNum);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree Hello Time for an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    ifNum              @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SIntfHelloTimeGet(openapiClientHandle_t *clientHandle, 
                           uint32_t ifNum)
{
  open_error_t   result;
  uint32_t       val = 0;

  if ((result = openapiDot1SIntfHelloTimeGet(clientHandle, 
                                             ifNum, &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Hello Time for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Hello Time for interface %u : %u \n", ifNum, val);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree BPDU Guard effect for an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    ifNum              @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SIntfBPDUGuardEffectGet(openapiClientHandle_t *clientHandle, 
                                 uint32_t ifNum)
{
  open_error_t   result;
  OPEN_CONTROL_t val;

  if ((result = openapiDot1SIntfBPDUGuardEffectGet(clientHandle, 
                                                   ifNum, &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree BPDU Guard effect for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Administrative BPDU Guard effect for interface %u: %u \n", ifNum, val);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree Restricted TCN parameter for an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    ifNum              @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SPortRestrictedTcnGet(openapiClientHandle_t *clientHandle, 
                               uint32_t ifNum)
{
  open_error_t   result;
  OPEN_BOOL_t val;

  if ((result = openapiDot1SPortRestrictedTcnGet(clientHandle, 
                                                 ifNum, &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Restricted TCN parameterfor an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Restricted TCN parameter for interface %u: %u \n", ifNum, val);
  }
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree Restricted TCN parameter for an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    ifNum              @b{(input)}   ID of the interface
* @param    val                @b{(input)}   value
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SPortRestrictedTcnSet(openapiClientHandle_t *clientHandle, 
                               uint32_t ifNum, OPEN_BOOL_t val)
{
  open_error_t   result;

  if ((result = openapiDot1SPortRestrictedTcnSet(clientHandle, 
                                                 ifNum, val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree Restricted TCN parameter for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Restricted TCN parameter set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree Auto Edge value for an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    ifNum              @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SPortAutoEdgeGet(openapiClientHandle_t *clientHandle, 
                          uint32_t ifNum)
{
  open_error_t   result;
  OPEN_BOOL_t val;

  if ((result = openapiDot1SPortAutoEdgeGet(clientHandle, 
                                            ifNum, &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Auto Edge Value for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Auto Edge Value for interface %u: %u \n", ifNum, val);
  }
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree Auto Edge value for an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    ifNum              @b{(input)}   ID of the interface
* @param    val                @b{(input)}   value
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SPortAutoEdgeSet(openapiClientHandle_t *clientHandle, 
                          uint32_t ifNum, OPEN_BOOL_t val)
{
  open_error_t   result;

  if ((result = openapiDot1SPortAutoEdgeSet(clientHandle, 
                                            ifNum, val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree Auto Edge Value for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Auto Edge Value set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree Up Time for an interface
*
* @param    clientHandle       @b{(input)} client handle from registration API
* @param    ifNum              @b{(input)} ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SPortUpTimeGet(openapiClientHandle_t *clientHandle, 
                        uint32_t ifNum)
{
  open_error_t   result;
  open_buffdesc  buffdesc;
  char           str[256];

  buffdesc.pstart = str;
  buffdesc.size = sizeof(str);

  if ((result = openapiDot1SPortUpTimeGet(clientHandle, 
                                          ifNum, &buffdesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Up Time for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Up Time for interface %u: %s \n", ifNum, str);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree STP BPDUs sent on an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    ifNum              @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SPortStatsSTPBPDUsSentGet(openapiClientHandle_t *clientHandle, 
                                   uint32_t ifNum)
{
  open_error_t   result;
  uint32_t       val = 0;

  if ((result = openapiDot1SPortStatsSTPBPDUsSentGet(clientHandle, 
                                                     ifNum, 
                                                     &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get STP BPDUs sent on an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree STP BPDUs sent on interface %u: %u \n", ifNum, val);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree STP BPDUs received on an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    ifNum              @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SPortStatsSTPBPDUsReceivedGet(openapiClientHandle_t *clientHandle, 
                                       uint32_t ifNum)
{
  open_error_t   result;
  uint32_t       val = 0;

  if ((result = openapiDot1SPortStatsSTPBPDUsReceivedGet(clientHandle, 
                                                         ifNum, 
                                                         &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get STP BPDUs received on an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree STP BPDUs received on interface %u: %u \n", ifNum, val);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree RSTP BPDUs sent on an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    ifNum              @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SPortStatsRSTPBPDUsSentGet(openapiClientHandle_t *clientHandle, 
                                    uint32_t ifNum)
{
  open_error_t   result;
  uint32_t       val = 0;

  if ((result = openapiDot1SPortStatsRSTPBPDUsSentGet(clientHandle, 
                                                      ifNum, 
                                                      &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get RSTP BPDUs sent on an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree RSTP BPDUs sent on interface %u: %u \n", ifNum, val);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree RSTP BPDUs received on an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    ifNum              @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SPortStatsRSTPBPDUsReceivedGet(openapiClientHandle_t *clientHandle, 
                                        uint32_t ifNum)
{
  open_error_t   result;
  uint32_t       val = 0;

  if ((result = openapiDot1SPortStatsRSTPBPDUsReceivedGet(clientHandle, 
                                                          ifNum, 
                                                          &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get RSTP BPDUs received on an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree RSTP BPDUs received on interface %u: %u \n", ifNum, val);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree MSTP BPDUs sent on an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    ifNum              @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SPortStatsMSTPBPDUsSentGet(openapiClientHandle_t *clientHandle, 
                                    uint32_t ifNum)
{
  open_error_t   result;
  uint32_t       val = 0;

  if ((result = openapiDot1SPortStatsMSTPBPDUsSentGet(clientHandle, 
                                                      ifNum, 
                                                      &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get MSTP BPDUs sent on an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MSTP BPDUs sent on interface %u: %u \n", ifNum, val);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree MSTP BPDUs received on an interface
*
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    ifNum              @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void dot1SPortStatsMSTPBPDUsReceivedGet(openapiClientHandle_t *clientHandle, 
                                        uint32_t ifNum)
{
  open_error_t   result;
  uint32_t       val = 0;

  if ((result = openapiDot1SPortStatsMSTPBPDUsReceivedGet(clientHandle, 
                                                          ifNum, 
                                                          &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get MSTP BPDUs received on an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MSTP BPDUs received on interface %u: %u \n", ifNum, val);
  }
  return;
}

/*********************************************************************
* @purpose  Get the MSTP Bridge Max Age parameter for the CIST.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    *maxAge       @b{(output)} MSTP Bridge Max Age value for the CIST
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sCistBridgeMaxAgeGet(openapiClientHandle_t *client_handle, uint32_t *maxAge)
{
  open_error_t result;

  if ((result = openapiDot1sCistBridgeMaxAgeGet(client_handle, maxAge)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the MSTP Bridge Max Age parameter for the CIST. (result = %d)\n", result);
  }
  else
  {
    printf("Bridge max age for CIST is %u.\n", *maxAge);
  }
  return;
}

/*********************************************************************
* @purpose  Set the MSTP Bridge Max Age parameter for the CIST.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    maxAge        @b{(input)}  MSTP Bridge Max Age value for the CIST
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sCistBridgeMaxAgeSet(openapiClientHandle_t *client_handle, uint32_t maxAge)
{
  open_error_t result;

  if ((result = openapiDot1sCistBridgeMaxAgeSet(client_handle, maxAge)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the MSTP Bridge Max Age parameter for the CIST. (result = %d)\n", result);
  }
  else
  {
    printf("Bridge max age for CIST set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the MSTP Bridge hello time parameter for the CIST.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    *helloTime       @b{(output)}  MSTP Bridge hello time value for the CIST
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sCistBridgeHelloTimeGet(openapiClientHandle_t *client_handle, uint32_t *helloTime)
{
  open_error_t result;

  if ((result = openapiDot1sCistBridgeHelloTimeGet(client_handle, helloTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the MSTP Bridge hello time parameter for the CIST. (result = %d)\n", result);
  }
  else
  {
    printf("Bridge hello time for CIST is %u.\n", *helloTime);
  }
  return;
}

/*********************************************************************
* @purpose  Get the MSTP Bridge forward delay for the CIST.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    *fwdDelay       @b{(output)}  MSTP Bridge forward delay value for the CIST
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sCistBridgFwdDelayGet(openapiClientHandle_t *client_handle, uint32_t *fwdDelay)
{
  open_error_t result;

  if ((result = openapiDot1sCistBridgFwdDelayGet(client_handle, fwdDelay)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the MSTP Bridge forward delay for the CIST. (result = %d)\n", result);
  }
  else
  {
    printf("Bridge forward delay for CIST is %u.\n", *fwdDelay);
  }
  return;
}

/*********************************************************************
* @purpose  Set the MSTP Bridge forward delay for the CIST.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    fwdDelay       @b{(input)}  MSTP Bridge forward delay value for the CIST
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sCistBridgeFwdDelaySet(openapiClientHandle_t *client_handle, uint32_t fwdDelay)
{
  open_error_t result;

  if ((result = openapiDot1sCistBridgeFwdDelaySet(client_handle, fwdDelay)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the MSTP Bridge forward delay for the CIST. (result = %d)\n", result);
  }
  else
  {
    printf("Bridge forward delay for CIST set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Set the MSTP max hop count parameter for the device.
*
* @param    client_handle     @b{(input)}  client handle from registration API
* @param    maxHopCount       @b{(input)}  MSTP Bridge max hop count value for the device
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sBridgeMaxHopSet(openapiClientHandle_t *client_handle, uint32_t maxHopCount)
{
  open_error_t result;

  if ((result = openapiDot1sBridgeMaxHopSet(client_handle, maxHopCount)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the MSTP max hop count parameter for the device. (result = %d)\n", result);
  }
  else
  {
    printf("Bridge max hop count set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Set the MSTP transmit hold count parameter for the device.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    txHoldCount    @b{(input)}  MSTP Bridge transmit hold count parameter for the device
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sBridgeTxHoldCountSet(openapiClientHandle_t *client_handle, uint32_t txHoldCount)
{
  open_error_t result;

  if ((result = openapiDot1sBridgeTxHoldCountSet(client_handle, txHoldCount)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the MSTP transmit hold count parameter for the device. (result = %d)\n", result);
  }
  else
  {
    printf("Bridge transmit hold count set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the MSTP root port forward delay for the CIST.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    *fwdDelay       @b{(output)}  MSTP root port forward delay for the CIST
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sCistRootFwdDelayGet(openapiClientHandle_t *client_handle, uint32_t *fwdDelay)
{
  open_error_t result;

  if ((result = openapiDot1sCistRootFwdDelayGet(client_handle, fwdDelay)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the MSTP root port forward delay for the CIST. (result = %d)\n", result);
  }
  else
  {
    printf("Bridge root forward delay for CIST is %u.\n", *fwdDelay);
  }
  return;
}

/*********************************************************************
* @purpose  Get the port ID that triggered topology change.
*
* @param    client_handle @b{(input)}   client handle from registration API
* @param    *portId       @b{(output)}  Port ID that triggered topology change
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sTopologyChangePortIDGet(openapiClientHandle_t *client_handle, uint32_t *portId)
{
  open_error_t result;

  if ((result = openapiDot1sTopologyChangePortIDGet(client_handle, portId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the port ID that triggered topology change. (result = %d)\n", result);
  }
  else
  {
    printf("Port ID that triggered TC is %u.\n", *portId);
  }
  return;
}

/*********************************************************************
* @purpose  Get the MSTP port path cost in a specific instance.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    mstId         @b{(input)}  The MSTID for the desired MST instance
* @param    ifNum         @b{(input)}  interface
* @param    *portPathCost @b{(output)} MSTP port path cost in a specific instance
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sMstiPortPathCostGet(openapiClientHandle_t *client_handle, uint32_t mstId, uint32_t ifNum, uint32_t *portPathCost)
{
  open_error_t result;

  if ((result = openapiDot1sMstiPortPathCostGet(client_handle, mstId, ifNum, portPathCost)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the MSTP port path cost in a specific instance. (result = %d)\n", result);
  }
  else
  {
    printf("Port path cost for interface %u in mst instance %u is %u.\n", ifNum, 
           mstId, *portPathCost);
  }
  return;
}

/*********************************************************************
* @purpose  Get the MSTP Configuration Digest Key.
*
* @param    client_handle @b{(input)}  client handle from regn API
* @param    buf           @b{(output)} MSTP configuration digest key

*
* @returns  none
*
* @end
*********************************************************************/
void dot1sConfigDigestKeyGet(openapiClientHandle_t *client_handle, 
                             open_buffdesc *buf)
{
  open_error_t result;
  unsigned char digestKey[16] = {0};

  if ((result = openapiDot1sConfigDigestKeyGet(client_handle, buf)) != 
      OPEN_E_NONE)
  {
    printf("Bad return code trying to get the MSTP Configuration Digest Key. "
           "(result = %d)\n", result);
  }
  else
  {
    memcpy(digestKey, (unsigned char*) buf->pstart, 16);
    printf("Digest key: 0x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x"
           "%02x%02x%02x\n", 
           digestKey[0], digestKey[1], digestKey[2], digestKey[3], digestKey[4],
           digestKey[5], digestKey[6], digestKey[7], digestKey[8], digestKey[9],
           digestKey[10], digestKey[11], digestKey[12], digestKey[13], 
           digestKey[14], digestKey[15]);
  }

  return;
}

/*********************************************************************
* @purpose  Get the MSTP Time since last Topology Change in an instance.
*
* @param    client_handle @b{(input)}   client handle from registration API
* @param    mstID         @b{(input)}   MSTID for the desired MTSI
* @param    days          @b{(output)}  time since last TCN - days
* @param    hours         @b{(output)}  time since last TCN - hours
* @param    minutes       @b{(output)}  time since last TCN - minutes
* @param    seconds       @b{(output)}  time since last TCN - seconds

*
* @returns  none
*
* @end
*********************************************************************/
void dot1sMstiTimeSinceTopologyChangeGet(openapiClientHandle_t *client_handle, 
                                         uint32_t mstID, uint32_t *days, 
                                         uint32_t *hours, uint32_t *minutes, 
                                         uint32_t *seconds)
{
  open_error_t result;

  if ((result = openapiDot1sMstiTimeSinceTopologyChangeGet(client_handle, mstID, 
      days, hours, minutes, seconds)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the MSTP Time since the last Topology "
           "Change in a specific instance. (result = %d)\n", result);
  }
  else
  {
    printf("Time since last topology change: %d days %d hours %d minutes "
           "%d seconds\n", *days, *hours, *minutes, *seconds);
  }

  return;
}


/*********************************************************************
* @purpose  Get the next port participating in MSTP associated with 
*           the specified MSTP instance ID.
*
* @param    client_handle  @b{(input)}  client handle from regn API
* @param    mstID          @b{(input)}  MSTID of the desired MSTI
* @param    prevIntIfNum   @b{(input)}  previous interface number
* @param    intIfNum       @b{(output)} next interface number

*
* @returns  none
*
* @end
*********************************************************************/
void dot1sPortNextGet(openapiClientHandle_t *client_handle, uint32_t mstID, 
                      uint32_t prevIntIfNum, uint32_t *intIfNum)
{
  open_error_t result;

  if ((result = openapiDot1sPortNextGet(client_handle, mstID, prevIntIfNum, 
      intIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the next port participating in MSTP "
           "associated with the specified MSTP instance ID. (result = %d)\n", 
           result);
  }
  else
  {
    printf("Next interface: %d\n", *intIfNum);
  }

  return;
}

/*********************************************************************
* @purpose  This function sanity checks all DOT1S OpEN APIs. All 
*           DOT1S OpEN APIs are called with possible NULL and 
*           invalid parameters to check the API robustness.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}   ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void dot1SOpENAPIsTestSanity(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t controlMode;
  uint32_t invalidControlMode = 10;
  uint32_t dot1SMgmtStrLen, strLen, dot1sConfigRev, instanceId, nextInstanceId;
  uint32_t vlan, nextVlan, FId, nextFId;
  uint16_t priority;
  OPEN_DOT1S_FORCE_VERSION_t dot1sForceVersion;
  OPEN_DOT1S_PORT_GUARD_t dot1sGuardMode;
  open_buffdesc buffDesc;
  char *str, buf[256];
  OPEN_DOT1S_PORT_FWD_STATE_t dot1sPortFwdState;
  OPEN_DOT1S_PORT_ROLE_t dot1sPortRole;
  OPEN_BOOL_t loopInconsistentState;
  uint32_t var1;
  uint32_t    port_priority = 0, val = 0;
  OPEN_BOOL_t bool_val;
  uint32_t mstid = 0;
  uint32_t badMstid = 9999;
  OPEN_BOOL_t var2;
  uint32_t maxHopCount = 6;
  uint32_t maxAge = 6;
  uint32_t txHoldCount = 1;
  uint32_t fwdDelay = 4;
  OPEN_DOT1S_BRIDGEID_t id;
  open_buffdesc bufDesc;
  open_buffdesc zeroLenBuf;
  open_buffdesc badBufdescPointer;
  unsigned char buffer[16] = {0};
  uint32_t days, hours, minutes, seconds;
  uint32_t mstID = 1, prevIntIfNum = 1, intIfNum;

  badBufdescPointer.pstart = (void *) NULL;
  badBufdescPointer.size = sizeof(buffer);
  bufDesc.pstart = buffer;
  bufDesc.size = sizeof(buffer);
  zeroLenBuf.pstart = buffer;
  zeroLenBuf.size = 0;


  printf("Testing Spanning Tree OpEN APIs sanity:\n\n");

  if ((result = openapiDot1sMgmtStrLenGet(clientHandle, OPEN_DOT1S_MGMT_STR_CONFIG_NAME, &dot1SMgmtStrLen)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n", result);
    return;
  }

  if ((str = (char*)malloc(dot1SMgmtStrLen)) == NULL) 
  {
    printf("Could not allocate memory.\n");
    return;
  }

  memset(str, 0, dot1SMgmtStrLen);

  /* openapiDot1sMgmtStrLenGet() */
  printf("Testing openapiDot1sMgmtStrLenGet():\n");

  result = openapiDot1sMgmtStrLenGet(NULL, OPEN_DOT1S_MGMT_STR_CONFIG_NAME, &strLen);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1sMgmtStrLenGet(clientHandle, 1, &strLen);
  printf("Invalid string type:(result = %d)\n", result);

  result = openapiDot1sMgmtStrLenGet(clientHandle, OPEN_DOT1S_MGMT_STR_CONFIG_NAME, NULL);
  printf("NULL String Length:(result = %d)\n", result);

  printf("openapiDot1sMgmtStrLenGet() sanity successful\n\n");

  /* openapiDot1SModeSet() */
  printf("Testing openapiDot1SModeSet():\n");

  result = openapiDot1SModeSet(NULL, OPEN_ENABLE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SModeSet(clientHandle, invalidControlMode);
  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiDot1SModeSet() sanity successful\n\n");

  /* openapiDot1SModeGet() */
  printf("Testing openapiDot1SModeGet():\n");

  result = openapiDot1SModeGet(NULL, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SModeGet(clientHandle, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiDot1SModeGet() sanity successful\n\n");

  /* openapiDot1SIntfBPDUFilterModeSet() */
  printf("Testing openapiDot1SIntfBPDUFilterModeSet():\n");

  result = openapiDot1SIntfBPDUFilterModeSet(NULL, ifNum, OPEN_ENABLE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfBPDUFilterModeSet(clientHandle, ifNum, invalidControlMode);
  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiDot1SIntfBPDUFilterModeSet() sanity successful\n\n");

  /* openapiDot1SIntfBPDUFilterModeGet() */
  printf("Testing openapiDot1SIntfBPDUFilterModeGet():\n");

  result = openapiDot1SIntfBPDUFilterModeGet(NULL, ifNum, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfBPDUFilterModeGet(clientHandle, ifNum, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiDot1SIntfBPDUFloodModeSet() sanity successful\n\n");

  /* openapiDot1SIntfBPDUFloodModeSet() */
  printf("Testing openapiDot1SIntfBPDUFloodModeSet():\n");

  result = openapiDot1SIntfBPDUFloodModeSet(NULL, ifNum, OPEN_ENABLE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfBPDUFloodModeSet(clientHandle, ifNum, invalidControlMode);
  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiDot1SIntfBPDUFloodModeSet() sanity successful\n\n");

  /* openapiDot1SIntfBPDUFloodModeGet() */
  printf("Testing openapiDot1SIntfBPDUFloodModeGet():\n");

  result = openapiDot1SIntfBPDUFloodModeGet(NULL, ifNum, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfBPDUFloodModeGet(clientHandle, ifNum, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiDot1SIntfBPDUFloodModeGet() sanity successful\n\n");

  /* openapiDot1SBPDUGuardModeSet() */
  printf("Testing openapiDot1SBPDUGuardModeSet():\n");

  result = openapiDot1SBPDUGuardModeSet(NULL, OPEN_ENABLE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SBPDUGuardModeSet(clientHandle, invalidControlMode);
  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiDot1SBPDUGuardModeSet() sanity successful\n\n");

  /* openapiDot1SBPDUGuardModeGet() */
  printf("Testing openapiDot1SBPDUGuardModeGet():\n");

  result = openapiDot1SBPDUGuardModeGet(NULL, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SBPDUGuardModeGet(clientHandle, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiDot1SBPDUGuardModeGet() sanity successful\n\n");

  /* openapiDot1SConfigNameSet() */
  printf("Testing openapiDot1SConfigNameSet():\n");

  buffDesc.pstart = str;
  buffDesc.size = dot1SMgmtStrLen;

  result = openapiDot1SConfigNameSet(NULL, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SConfigNameSet(clientHandle, NULL);
  printf("NULL Config Name:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiDot1SConfigNameSet(clientHandle, &buffDesc);
  printf("NULL Config Name buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = dot1SMgmtStrLen + 10;

  result = openapiDot1SConfigNameSet(clientHandle, &buffDesc);
  printf("NULL Config Name buffer length:(result = %d)\n", result);

  printf("openapiDot1SConfigNameSet() sanity successful\n\n");

  /* openapiDot1SConfigNameGet() */
  printf("Testing openapiDot1SConfigNameGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = dot1SMgmtStrLen;

  result = openapiDot1SConfigNameGet(NULL, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SConfigNameGet(clientHandle, NULL);
  printf("NULL Config Name:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiDot1SConfigNameGet(clientHandle, &buffDesc);
  printf("NULL Config Name buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiDot1SConfigNameGet(clientHandle, &buffDesc);
  printf("NULL Config Name buffer length:(result = %d)\n", result);

  printf("openapiDot1SConfigNameGet() sanity successful\n\n");

  /* openapiDot1SConfigRevisionSet() */
  printf("Testing openapiDot1SConfigRevisionSet():\n");

  result = openapiDot1SConfigRevisionSet(NULL, dot1sConfigRev);
  printf("NULL Client Handle:(result = %d)\n", result);

  printf("openapiDot1SConfigRevisionSet() sanity successful\n\n");

  /* openapiDot1SConfigRevisionGet() */
  printf("Testing openapiDot1SConfigRevisionGet():\n");

  result = openapiDot1SConfigRevisionGet(NULL, &dot1sConfigRev);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SConfigRevisionGet(clientHandle, NULL);
  printf("NULL Config Revision:(result = %d)\n", result);

  printf("openapiDot1SConfigRevisionGet() sanity successful\n\n");

  /* openapiDot1SForceVersionSet() */
  printf("Testing openapiDot1SForceVersionSet():\n");

  result = openapiDot1SForceVersionSet(NULL, OPEN_DOT1S_FORCE_VERSION_DOT1D);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SForceVersionSet(clientHandle, invalidControlMode);
  printf("Invalid Force Version:(result = %d)\n", result);

  printf("openapiDot1SForceVersionSet() sanity successful\n\n");

  /* openapiDot1SForceVersionGet() */
  printf("Testing openapiDot1SForceVersionGet():\n");

  result = openapiDot1SForceVersionGet(NULL, &dot1sForceVersion);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SForceVersionGet(clientHandle, NULL);
  printf("NULL Force Version:(result = %d)\n", result);

  printf("openapiDot1SForceVersionGet() sanity successful\n\n");

  /* openapiDot1SIntfGuardModeSet() */
  printf("Testing openapiDot1SIntfGuardModeSet():\n");

  result = openapiDot1SIntfGuardModeSet(NULL, ifNum, OPEN_DOT1S_PORT_GUARD_LOOP);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfGuardModeSet(clientHandle, ifNum, invalidControlMode);
  printf("Invalid Guard Mode:(result = %d)\n", result);

  printf("openapiDot1SIntfGuardModeSet() sanity successful\n\n");

  /* openapiDot1SIntfGuardModeGet() */
  printf("Testing openapiDot1SIntfGuardModeGet():\n");

  result = openapiDot1SIntfGuardModeGet(NULL, ifNum, &dot1sGuardMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfGuardModeGet(clientHandle, ifNum, NULL);
  printf("NULL Guard Mode:(result = %d)\n", result);

  printf("openapiDot1SIntfGuardModeGet() sanity successful\n\n");

  /* openapiDot1SMSTInstanceSet() */
  printf("Testing openapiDot1SMSTInstanceSet():\n");

  result = openapiDot1SMSTInstanceSet(NULL, 10, OPEN_ACTION_ADD);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMSTInstanceSet(clientHandle, 10, invalidControlMode);
  printf("Invalid Action:(result = %d)\n", result);

  printf("openapiDot1SMSTInstanceSet() sanity successful\n\n");

  /* openapiDot1SMSTInstanceFirstGet() */
  printf("Testing openapiDot1SMSTInstanceFirstGet():\n");

  result = openapiDot1SMSTInstanceFirstGet(NULL, &instanceId);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMSTInstanceFirstGet(clientHandle, NULL);
  printf("NULL Instance:(result = %d)\n", result);

  printf("openapiDot1SMSTInstanceFirstGet() sanity successful\n\n");

  /* openapiDot1SMSTInstanceNextGet() */
  printf("Testing openapiDot1SMSTInstanceNextGet():\n");

  result = openapiDot1SMSTInstanceNextGet(NULL, instanceId, &nextInstanceId);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMSTInstanceNextGet(clientHandle, instanceId, NULL);
  printf("NULL Next Instance:(result = %d)\n", result);

  printf("openapiDot1SMSTInstanceNextGet() sanity successful\n\n");

  /* openapiDot1SMSTInstancePrioritySet() */
  printf("Testing openapiDot1SMSTInstancePrioritySet():\n");

  result = openapiDot1SMSTInstancePrioritySet(NULL, 10, 10);
  printf("NULL Client Handle:(result = %d)\n", result);

  printf("openapiDot1SMSTInstancePrioritySet() sanity successful\n\n");

  /* openapiDot1SMSTInstancePriorityGet() */
  printf("Testing openapiDot1SMSTInstancePriorityGet():\n");

  result = openapiDot1SMSTInstancePriorityGet(NULL, 10, &priority);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMSTInstancePriorityGet(clientHandle, 10, NULL);
  printf("NULL Priority:(result = %d)\n", result);

  printf("openapiDot1SMSTInstancePriorityGet() sanity successful\n\n");

  /* openapiDot1SMSTInstanceVLANSet() */
  printf("Testing openapiDot1SMSTInstanceVLANSet():\n");

  result = openapiDot1SMSTInstanceVLANSet(NULL, 10, 10, OPEN_ACTION_ADD);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMSTInstanceVLANSet(clientHandle, 10, 10, invalidControlMode);
  printf("NULL Action:(result = %d)\n", result);

  printf("openapiDot1SMSTInstanceVLANSet() sanity successful\n\n");

  /* openapiDot1SMSTInstanceFirstVLANGet() */
  printf("Testing openapiDot1SMSTInstanceFirstVLANGet():\n");

  result = openapiDot1SMSTInstanceFirstVLANGet(NULL, 10, &vlan);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMSTInstanceFirstVLANGet(clientHandle, 10, NULL);
  printf("NULL VLAN:(result = %d)\n", result);

  printf("openapiDot1SMSTInstanceFirstVLANGet() sanity successful\n\n");

  /* openapiDot1SMSTInstanceNextVLANGet() */
  printf("Testing openapiDot1SMSTInstanceNextVLANGet():\n");

  result = openapiDot1SMSTInstanceNextVLANGet(NULL, 10, 10, &nextVlan);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMSTInstanceNextVLANGet(clientHandle, 10, 10, NULL);
  printf("NULL Next Vlan:(result = %d)\n", result);

  printf("openapiDot1SMSTInstanceNextVLANGet() sanity successful\n\n");

  /* openapiDot1SMSTInstanceFirstFIdGet() */
  printf("Testing openapiDot1SMSTInstanceFirstFIdGet():\n");

  result = openapiDot1SMSTInstanceFirstFIdGet(NULL, 10, &FId);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMSTInstanceFirstFIdGet(clientHandle, 10, NULL);
  printf("NULL FId:(result = %d)\n", result);

  printf("openapiDot1SMSTInstanceFirstFIdGet() sanity successful\n\n");

  /* openapiDot1SMSTInstanceNextFIdGet() */
  printf("Testing openapiDot1SMSTInstanceNextFIdGet():\n");

  result = openapiDot1SMSTInstanceNextFIdGet(NULL, 10, 10, &nextFId);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMSTInstanceNextFIdGet(clientHandle, 10, 10, NULL);
  printf("NULL Next FId:(result = %d)\n", result);

  printf("openapiDot1SMSTInstanceNextFIdGet() sanity successful\n\n");

  /* openapiDot1SIntfModeSet() */
  printf("Testing openapiDot1SIntfModeSet():\n");

  result = openapiDot1SIntfModeSet(NULL, ifNum, OPEN_ENABLE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfModeSet(clientHandle, ifNum, invalidControlMode);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiDot1SIntfModeSet() sanity successful\n\n");

  /* openapiDot1SIntfModeGet() */
  printf("Testing openapiDot1SIntfModeGet():\n");

  result = openapiDot1SIntfModeGet(NULL, ifNum, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfModeGet(clientHandle, ifNum, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiDot1SIntfModeGet() sanity successful\n\n");

  /* openapiDot1SIntfModeGet() */
  printf("Testing openapiDot1SIntfModeGet():\n");

  result = openapiDot1SIntfModeGet(NULL, ifNum, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfModeGet(clientHandle, ifNum, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiDot1SIntfModeGet() sanity successful\n\n");

  /* openapiDot1SIntfPortFwdStateGet() */
  printf("Testing openapiDot1SIntfPortFwdStateGet():\n");

  result = openapiDot1SIntfPortFwdStateGet(NULL, ifNum, 10, &dot1sPortFwdState);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfPortFwdStateGet(clientHandle, ifNum, 10, NULL);
  printf("NULL FwdState:(result = %d)\n", result);

  printf("openapiDot1SIntfPortFwdStateGet() sanity successful\n\n");

  /* openapiDot1SIntfPortRoleGet() */
  printf("Testing openapiDot1SIntfPortRoleGet():\n");

  result = openapiDot1SIntfPortRoleGet(NULL, ifNum, 10, &dot1sPortRole);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfPortRoleGet(clientHandle, ifNum, 10, NULL);
  printf("NULL PortRole:(result = %d)\n", result);

  printf("openapiDot1SIntfPortRoleGet() sanity successful\n\n");

  /* openapiDot1SIntfEdgePortModeSet() */
  printf("Testing openapiDot1SIntfEdgePortModeSet():\n");

  result = openapiDot1SIntfEdgePortModeSet(NULL, ifNum, OPEN_ENABLE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfEdgePortModeSet(clientHandle, ifNum, invalidControlMode);
  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiDot1SIntfEdgePortModeSet() sanity successful\n\n");

  /* openapiDot1SIntfEdgePortModeGet() */
  printf("Testing openapiDot1SIntfEdgePortModeGet():\n");

  result = openapiDot1SIntfEdgePortModeGet(NULL, ifNum, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfEdgePortModeGet(clientHandle, ifNum, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiDot1SIntfEdgePortModeGet() sanity successful\n\n");

  /* openapiDot1SIntfPortLoopInconsistentStateGet() */
  printf("Testing openapiDot1SIntfPortLoopInconsistentStateGet():\n");

  result = openapiDot1SIntfPortLoopInconsistentStateGet(NULL, ifNum, 10, &loopInconsistentState);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfPortLoopInconsistentStateGet(clientHandle, ifNum, 10, NULL);
  printf("NULL Loop Inconsistent State:(result = %d)\n", result);

  printf("openapiDot1SIntfPortLoopInconsistentStateGet() sanity successful\n\n");

  buffDesc.pstart = buf;
  buffDesc.size = sizeof(buf);

  result = openapiDot1SMSTIntfPortIDGet(NULL, 5, ifNum, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMSTIntfPortIDGet(clientHandle, 20, ifNum, &buffDesc);
  printf("Invalid MST instance:(result = %d)\n", result);

  result = openapiDot1SMSTIntfPortIDGet(clientHandle, 5, 0, &buffDesc);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SMSTIntfPortIDGet(clientHandle, 5, ifNum, NULL);
  printf("NULL pointer to string:(result = %d)\n", result);

  printf("openapiDot1SMSTIntfPortIDGet() sanity successful\n\n");

  /* openapiDot1SMstiPortPrioritySet() */
  printf("Testing openapiDot1SMstiPortPrioritySet():\n");

  port_priority = 64;
  result = openapiDot1SMstiPortPrioritySet(NULL, 5, ifNum, port_priority);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMstiPortPrioritySet(clientHandle, 20, ifNum, port_priority);
  printf("Invalid MST instance:(result = %d)\n", result);

  result = openapiDot1SMstiPortPrioritySet(clientHandle, 5, 0, port_priority);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SMstiPortPrioritySet(clientHandle, 5, ifNum, 61450);
  printf("Invalid priority:(result = %d)\n", result);

  printf("openapiDot1SMstiPortPrioritySet() sanity successful\n\n");

  /* openapiDot1SMstiPortPathCostModeGet() */
  printf("Testing openapiDot1SMstiPortPathCostModeGet():\n");

  result = openapiDot1SMstiPortPathCostModeGet(NULL, 5, ifNum, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMstiPortPathCostModeGet(clientHandle, 20, ifNum, &controlMode);
  printf("Invalid MST instance:(result = %d)\n", result);

  result = openapiDot1SMstiPortPathCostModeGet(clientHandle, 5, 0, &controlMode);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SMstiPortPathCostModeGet(clientHandle, 5, ifNum, NULL);
  printf("NULL control mode:(result = %d)\n", result);

  printf("openapiDot1SMstiPortPathCostModeGet() sanity successful\n\n");

  /* openapiDot1SMstiPortPathCostSet() */
  printf("Testing openapiDot1SMstiPortPathCostSet():\n");

  result = openapiDot1SMstiPortPathCostSet(NULL, 5, ifNum, 200);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMstiPortPathCostSet(clientHandle, 20, ifNum, 200);
  printf("Invalid MST instance:(result = %d)\n", result);

  result = openapiDot1SMstiPortPathCostSet(clientHandle, 5, 0, 200);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SMstiPortPathCostSet(clientHandle, 5, ifNum, 200000010);
  printf("Invalid cost:(result = %d)\n", result);
 
  printf("openapiDot1SMstiPortPathCostSet() sanity successful\n\n");

  /* openapiDot1SMstiOperPortPathCostGet() */
  printf("Testing openapiDot1SMstiOperPortPathCostGet():\n");

  result = openapiDot1SMstiOperPortPathCostGet(NULL, 5, ifNum, &val);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMstiOperPortPathCostGet(clientHandle, 20, ifNum, &val);
  printf("Invalid MST instance:(result = %d)\n", result);

  result = openapiDot1SMstiOperPortPathCostGet(clientHandle, 5, 0, &val);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SMstiOperPortPathCostGet(clientHandle, 5, ifNum, NULL);
  printf("NULL cost:(result = %d)\n", result);

  printf("openapiDot1SMstiOperPortPathCostGet() sanity successful\n\n");

  /* openapiDot1SMstiPortDesignatedRootIDGet() */
  printf("Testing openapiDot1SMstiPortDesignatedRootIDGet():\n");

  buffDesc.pstart = buf;
  buffDesc.size = sizeof(buf);

  result = openapiDot1SMstiPortDesignatedRootIDGet(NULL, 5, ifNum, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMstiPortDesignatedRootIDGet(clientHandle, 20, ifNum, &buffDesc);
  printf("Invalid MST instance:(result = %d)\n", result);

  result = openapiDot1SMstiPortDesignatedRootIDGet(clientHandle, 5, 0, &buffDesc);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SMstiPortDesignatedRootIDGet(clientHandle, 5, ifNum, NULL);
  printf("NULL string:(result = %d)\n", result);

  printf("openapiDot1SMstiPortDesignatedRootIDGet() sanity successful\n\n");

  /* openapiDot1SMstiDesignatedCostGet() */
  printf("Testing openapiDot1SMstiDesignatedCostGet():\n");

  result = openapiDot1SMstiDesignatedCostGet(NULL, 5, ifNum, &val);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMstiDesignatedCostGet(clientHandle, 20, ifNum, &val);
  printf("Invalid MST instance:(result = %d)\n", result);

  result = openapiDot1SMstiDesignatedCostGet(clientHandle, 5, 0, &val);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SMstiDesignatedCostGet(clientHandle, 5, ifNum, NULL);
  printf("NULL cost:(result = %d)\n", result);

  printf("openapiDot1SMstiDesignatedCostGet() sanity successful\n\n");

  /* openapiDot1SMstiDesignatedPortIDGet() */
  printf("Testing openapiDot1SMstiDesignatedPortIDGet():\n");

  buffDesc.pstart = buf;
  buffDesc.size = sizeof(buf);

  result = openapiDot1SMstiDesignatedPortIDGet(NULL, 5, ifNum, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMstiDesignatedPortIDGet(clientHandle, 20, ifNum, &buffDesc);
  printf("Invalid MST instance:(result = %d)\n", result);

  result = openapiDot1SMstiDesignatedPortIDGet(clientHandle, 5, 0, &buffDesc);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SMstiDesignatedPortIDGet(clientHandle, 5, ifNum, NULL);
  printf("NULL string:(result = %d)\n", result);

  printf("openapiDot1SMstiDesignatedPortIDGet() sanity successful\n\n");

  /* openapiDot1SMstiPortLoopInconsistentStateGet() */
  printf("Testing openapiDot1SMstiPortLoopInconsistentStateGet():\n");

  result = openapiDot1SMstiPortLoopInconsistentStateGet(NULL, 5, ifNum, &bool_val);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMstiPortLoopInconsistentStateGet(clientHandle, 20, ifNum, &bool_val);
  printf("Invalid MST instance:(result = %d)\n", result);

  result = openapiDot1SMstiPortLoopInconsistentStateGet(clientHandle, 5, 0, &bool_val);
  printf("NULL interface:(result = %d)\n", result);
  
  result = openapiDot1SMstiPortLoopInconsistentStateGet(clientHandle, 5, ifNum, NULL);
  printf("NULL boolean value:(result = %d)\n", result);

  printf("openapiDot1SMstiPortLoopInconsistentStateGet() sanity successful\n\n");

  /* openapiDot1SPortStatsTransitionsIntoLoopInconsistentStateGet() */
  printf("Testing openapiDot1SPortStatsTransitionsIntoLoopInconsistentStateGet():\n");

  result = openapiDot1SPortStatsTransitionsIntoLoopInconsistentStateGet(NULL, 5, ifNum, &val);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SPortStatsTransitionsIntoLoopInconsistentStateGet(clientHandle, 20, ifNum, &val);
  printf("Invalid MST instance:(result = %d)\n", result);

  result = openapiDot1SPortStatsTransitionsIntoLoopInconsistentStateGet(clientHandle, 5, 0, &val);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SPortStatsTransitionsIntoLoopInconsistentStateGet(clientHandle, 5, ifNum, NULL);
  printf("NULL boolean value:(result = %d)\n", result);

  printf("openapiDot1SPortStatsTransitionsIntoLoopInconsistentStateGet() sanity successful\n\n");

  /* openapiDot1SPortStatsTransitionsOutOfLoopInconsistentStateGet() */
  printf("Testing openapiDot1SPortStatsTransitionsOutOfLoopInconsistentStateGet():\n");

  result = openapiDot1SPortStatsTransitionsOutOfLoopInconsistentStateGet(NULL, 5, ifNum, &val);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SPortStatsTransitionsOutOfLoopInconsistentStateGet(clientHandle, 20, ifNum, &val);
  printf("Invalid MST instance:(result = %d)\n", result);

  result = openapiDot1SPortStatsTransitionsOutOfLoopInconsistentStateGet(clientHandle, 5, 0, &val);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SPortStatsTransitionsOutOfLoopInconsistentStateGet(clientHandle, 5, ifNum, NULL);
  printf("NULL boolean value:(result = %d)\n", result);

  printf("openapiDot1SPortStatsTransitionsOutOfLoopInconsistentStateGet() sanity successful\n\n");

  /* openapiDot1SCistExtPortPathCostModeGet() */
  printf("Testing openapiDot1SCistExtPortPathCostModeGet():\n");

  result = openapiDot1SCistExtPortPathCostModeGet(NULL, ifNum, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SCistExtPortPathCostModeGet(clientHandle, 0, &controlMode);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SCistExtPortPathCostModeGet(clientHandle, ifNum, NULL);
  printf("NULL controlMode:(result = %d)\n", result);

  printf("openapiDot1SCistExtPortPathCostModeGet() sanity successful\n\n");

  /* openapiDot1SCistExtPortPathCostGet() */
  printf("Testing openapiDot1SCistExtPortPathCostGet():\n");

  result = openapiDot1SCistExtPortPathCostGet(NULL, ifNum, &val);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SCistExtPortPathCostGet(clientHandle, 0, &val);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SCistExtPortPathCostGet(clientHandle, ifNum, NULL);
  printf("NULL cost:(result = %d)\n", result);

  printf("openapiDot1SCistExtPortPathCostGet() sanity successful\n\n");

  /* openapiDot1SCistExtPortPathCostSet() */
  printf("Testing openapiDot1SCistExtPortPathCostSet():\n");

  result = openapiDot1SCistExtPortPathCostSet(NULL, ifNum, 200);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SCistExtPortPathCostSet(clientHandle, 0, 200);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SCistExtPortPathCostSet(clientHandle, ifNum, 200000010);
  printf("Invalid cost:(result = %d)\n", result);

  printf("openapiDot1SCistExtPortPathCostSet() sanity successful\n\n");

  /* openapiDot1SCistPortTopologyChangeAckGet() */
  printf("Testing openapiDot1SCistPortTopologyChangeAckGet():\n");

  result = openapiDot1SCistPortTopologyChangeAckGet(NULL, ifNum, &bool_val);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SCistPortTopologyChangeAckGet(clientHandle, 0, &bool_val);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SCistPortTopologyChangeAckGet(clientHandle, ifNum, NULL);
  printf("NULL boolean mode:(result = %d)\n", result);

  printf("openapiDot1SCistPortTopologyChangeAckGet() sanity successful\n\n");

  /* openapiDot1SCistHelloTimeGet() */
  printf("Testing openapiDot1SCistHelloTimeGet():\n");

  result = openapiDot1SCistHelloTimeGet(NULL, &val);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SCistHelloTimeGet(clientHandle, NULL);
  printf("NULL value:(result = %d)\n", result);

  printf("openapiDot1SCistHelloTimeGet() sanity successful\n\n");

  /* openapiDot1SCistPortOperEdgeGet() */
  printf("Testing openapiDot1SCistPortOperEdgeGet():\n");

  result = openapiDot1SCistPortOperEdgeGet(NULL, ifNum, &bool_val);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SCistPortOperEdgeGet(clientHandle, 0, &bool_val);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SCistPortOperEdgeGet(clientHandle, ifNum, NULL);
  printf("NULL boolean mode:(result = %d)\n", result);

  printf("openapiDot1SCistPortOperEdgeGet() sanity successful\n\n");

  /* openapiDot1SCistPortOperPointToPointGet() */
  printf("Testing openapiDot1SCistPortOperPointToPointGet():\n");

  result = openapiDot1SCistPortOperPointToPointGet(NULL, ifNum, &bool_val);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SCistPortOperPointToPointGet(clientHandle, 0, &bool_val);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SCistPortOperPointToPointGet(clientHandle, ifNum, NULL);
  printf("NULL boolean mode:(result = %d)\n", result);

  printf("openapiDot1SCistPortOperPointToPointGet() sanity successful\n\n");

  /* openapiDot1SCistPortRegionalRootIDGet() */
  printf("Testing openapiDot1SCistPortRegionalRootIDGet():\n");

  buffDesc.pstart = buf;
  buffDesc.size = sizeof(buf);

  result = openapiDot1SCistPortRegionalRootIDGet(NULL, ifNum, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SCistPortRegionalRootIDGet(clientHandle, 0, &buffDesc);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SCistPortRegionalRootIDGet(clientHandle, ifNum, NULL);
  printf("NULL string:(result = %d)\n", result);

  printf("openapiDot1SCistPortRegionalRootIDGet() sanity successful\n\n");

  /* openapiDot1SCistPortPathCostGet() */
  printf("Testing openapiDot1SCistPortPathCostGet():\n");

  result = openapiDot1SCistPortPathCostGet(NULL, ifNum, &val);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SCistPortPathCostGet(clientHandle, 0, &val);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SCistPortPathCostGet(clientHandle, ifNum, NULL);
  printf("NULL cost:(result = %d)\n", result);

  printf("openapiDot1SCistPortPathCostGet() sanity successful\n\n");

  /* openapiDot1SPortStatsClear() */
  printf("Testing openapiDot1SPortStatsClear():\n");

  result = openapiDot1SPortStatsClear(NULL, ifNum);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SPortStatsClear(clientHandle, 0);
  printf("NULL interface:(result = %d)\n", result);

  printf("openapiDot1SPortStatsClear() sanity successful\n\n");

  /* openapiDot1SIntfHelloTimeGet() */
  printf("Testing openapiDot1SIntfHelloTimeGet():\n");

  result = openapiDot1SIntfHelloTimeGet(NULL, ifNum, &val);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfHelloTimeGet(clientHandle, 0, &val);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SIntfHelloTimeGet(clientHandle, ifNum, NULL);
  printf("NULL val:(result = %d)\n", result);

  printf("openapiDot1SIntfHelloTimeGet() sanity successful\n\n");

  /* openapiDot1SIntfBPDUGuardEffectGet() */
  printf("Testing openapiDot1SIntfBPDUGuardEffectGet():\n");

  result = openapiDot1SIntfBPDUGuardEffectGet(NULL, ifNum, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfBPDUGuardEffectGet(clientHandle, 0, &controlMode);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SIntfBPDUGuardEffectGet(clientHandle, ifNum, NULL);
  printf("NULL val:(result = %d)\n", result);

  printf("openapiDot1SIntfBPDUGuardEffectGet() sanity successful\n\n");

  /* openapiDot1SPortRestrictedTcnSet() */
  printf("Testing openapiDot1SPortRestrictedTcnSet():\n");

  bool_val = OPEN_TRUE;
  result = openapiDot1SPortRestrictedTcnSet((void *)NULL, ifNum, bool_val);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SPortRestrictedTcnSet(clientHandle, 0, bool_val);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SPortRestrictedTcnSet(clientHandle, ifNum, 5);
  printf("Invalid boolean val:(result = %d)\n", result);
  
  printf("openapiDot1SPortRestrictedTcnSet() sanity successful\n\n");

  /* openapiDot1SPortRestrictedTcnGet() */
  printf("Testing openapiDot1SPortRestrictedTcnGet():\n");

  result = openapiDot1SPortRestrictedTcnGet((void *)NULL, ifNum, &bool_val);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SPortRestrictedTcnGet(clientHandle, 0, &bool_val);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SPortRestrictedTcnGet(clientHandle, ifNum, NULL);
  printf("NULL val:(result = %d)\n", result);
  
  printf("openapiDot1SPortRestrictedTcnGet() sanity successful\n\n");
  
  /* openapiDot1SPortAutoEdgeGet() */
  printf("Testing openapiDot1SPortAutoEdgeGet():\n");

  result = openapiDot1SPortAutoEdgeGet((void *)NULL, ifNum, &bool_val);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SPortAutoEdgeGet(clientHandle, 0, &bool_val);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SPortAutoEdgeGet(clientHandle, ifNum, NULL);
  printf("NULL val:(result = %d)\n", result);
  
  printf("openapiDot1SPortAutoEdgeGet() sanity successful\n\n");

  /* openapiDot1SPortAutoEdgeSet() */
  printf("Testing openapiDot1SPortAutoEdgeSet():\n");

  bool_val = OPEN_TRUE;
  result = openapiDot1SPortAutoEdgeSet((void *)NULL, ifNum, bool_val);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SPortAutoEdgeSet(clientHandle, 0, bool_val);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SPortAutoEdgeSet(clientHandle, ifNum, 5);
  printf("Invalid boolean val:(result = %d)\n", result);
  
  printf("openapiDot1SPortAutoEdgeSet() sanity successful\n\n");

  /* openapiDot1SPortUpTimeGet() */
  printf("Testing openapiDot1SPortUpTimeGet():\n");

  memset(buf, 0, sizeof(buf));
  buffDesc.pstart = buf;
  buffDesc.size = sizeof(buf);

  result = openapiDot1SPortUpTimeGet((void *)NULL, ifNum, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SPortUpTimeGet(clientHandle, 0, &buffDesc);
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SPortUpTimeGet(clientHandle, ifNum, NULL); 
  printf("NULL val:(result = %d)\n", result);
  
  printf("openapiDot1SPortUpTimeGet() sanity successful\n\n");
  
  /* openapiDot1SPortStatsSTPBPDUsSentGet() */
  printf("Testing openapiDot1SPortStatsSTPBPDUsSentGet():\n");

  result = openapiDot1SPortStatsSTPBPDUsSentGet((void *)NULL, ifNum, &val);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiDot1SPortStatsSTPBPDUsSentGet(clientHandle, 0, &val); 
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SPortStatsSTPBPDUsSentGet(clientHandle, ifNum, NULL);
  printf("NULL val:(result = %d)\n", result);

  printf("openapiDot1SPortStatsSTPBPDUsSentGet() sanity successful\n\n");
 
  /* openapiDot1SPortStatsSTPBPDUsReceivedGet() */
  printf("Testing openapiDot1SPortStatsSTPBPDUsReceivedGet():\n");

  result = openapiDot1SPortStatsSTPBPDUsReceivedGet((void *)NULL, ifNum, &val);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiDot1SPortStatsSTPBPDUsReceivedGet(clientHandle, 0, &val); 
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SPortStatsSTPBPDUsReceivedGet(clientHandle, ifNum, NULL);
  printf("NULL val:(result = %d)\n", result);

  printf("openapiDot1SPortStatsSTPBPDUsReceivedGet() sanity successful\n\n");
 
  /* openapiDot1SPortStatsRSTPBPDUsSentGet() */
  printf("Testing openapiDot1SPortStatsRSTPBPDUsSentGet():\n");

  result = openapiDot1SPortStatsRSTPBPDUsSentGet((void *)NULL, ifNum, &val);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiDot1SPortStatsRSTPBPDUsSentGet(clientHandle, 0, &val); 
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SPortStatsRSTPBPDUsSentGet(clientHandle, ifNum, NULL);
  printf("NULL val:(result = %d)\n", result);

  printf("openapiDot1SPortStatsRSTPBPDUsSentGet() sanity successful\n\n");
 
  /* openapiDot1SPortStatsRSTPBPDUsReceivedGet() */
  printf("Testing openapiDot1SPortStatsRSTPBPDUsReceivedGet():\n");

  result = openapiDot1SPortStatsRSTPBPDUsReceivedGet((void *)NULL, ifNum, &val);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiDot1SPortStatsRSTPBPDUsReceivedGet(clientHandle, 0, &val); 
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SPortStatsRSTPBPDUsReceivedGet(clientHandle, ifNum, NULL);
  printf("NULL val:(result = %d)\n", result);

  printf("openapiDot1SPortStatsRSTPBPDUsReceivedGet() sanity successful\n\n");
 
  /* openapiDot1SPortStatsMSTPBPDUsSentGet() */
  printf("Testing openapiDot1SPortStatsMSTPBPDUsSentGet():\n");

  result = openapiDot1SPortStatsMSTPBPDUsSentGet((void *)NULL, ifNum, &val);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiDot1SPortStatsMSTPBPDUsSentGet(clientHandle, 0, &val); 
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SPortStatsMSTPBPDUsSentGet(clientHandle, ifNum, NULL);
  printf("NULL val:(result = %d)\n", result);

  printf("openapiDot1SPortStatsMSTPBPDUsSentGet() sanity successful\n\n");
 
  /* openapiDot1SPortStatsMSTPBPDUsReceivedGet() */
  printf("Testing openapiDot1SPortStatsMSTPBPDUsReceivedGet():\n");

  result = openapiDot1SPortStatsMSTPBPDUsReceivedGet((void *)NULL, ifNum, &val);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiDot1SPortStatsMSTPBPDUsReceivedGet(clientHandle, 0, &val); 
  printf("NULL interface:(result = %d)\n", result);

  result = openapiDot1SPortStatsMSTPBPDUsReceivedGet(clientHandle, ifNum, NULL);
  printf("NULL val:(result = %d)\n", result);

  printf("openapiDot1SPortStatsMSTPBPDUsReceivedGet() sanity successful\n\n");
 
  free(str);

  mstid = 0;
  printf("Testing openapiDot1sMSTInstanceBridgeIdentifierGet():\n");
  result = openapiDot1sMSTInstanceBridgeIdentifierGet(NULL, mstid, &id);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceBridgeIdentifierGet():\n");
  result = openapiDot1sMSTInstanceBridgeIdentifierGet(clientHandle, badMstid, &id);
  printf("Invalid MST ID:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceBridgeIdentifierGet():\n");
  result = openapiDot1sMSTInstanceBridgeIdentifierGet(clientHandle, mstid, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("openapiDot1sMSTInstanceBridgeIdentifierGet() sanity successful\n\n");

  printf("Testing openapiDot1sMSTInstanceTimeSinceTopologyChangeGet():\n");
  result = openapiDot1sMSTInstanceTimeSinceTopologyChangeGet(NULL, mstid, &var1);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceTimeSinceTopologyChangeGet():\n");
  result = openapiDot1sMSTInstanceTimeSinceTopologyChangeGet(clientHandle, badMstid, &var1);
  printf("Invalid MST ID:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceTimeSinceTopologyChangeGet():\n");
  result = openapiDot1sMSTInstanceTimeSinceTopologyChangeGet(clientHandle, mstid, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("openapiDot1sMSTInstanceTimeSinceTopologyChangeGet() sanity successful\n\n");

  printf("Testing openapiDot1sMSTInstanceTopologyChangeCountGet():\n");
  result = openapiDot1sMSTInstanceTopologyChangeCountGet(NULL, mstid, &var1);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceTopologyChangeCountGet():\n");
  result = openapiDot1sMSTInstanceTopologyChangeCountGet(clientHandle, badMstid, &var1);
  printf("Invalid MST ID:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceTopologyChangeCountGet():\n");
  result = openapiDot1sMSTInstanceTopologyChangeCountGet(clientHandle, mstid, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("openapiDot1sMSTInstanceTopologyChangeCountGet() sanity successful\n\n");

  printf("Testing openapiDot1sMSTInstanceTopologyChangeStateGet():\n");
  result = openapiDot1sMSTInstanceTopologyChangeStateGet(NULL, mstid, &var2);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceTopologyChangeStateGet():\n");
  result = openapiDot1sMSTInstanceTopologyChangeStateGet(clientHandle, badMstid, &var2);
  printf("Invalid MST ID:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceTopologyChangeStateGet():\n");
  result = openapiDot1sMSTInstanceTopologyChangeStateGet(clientHandle, mstid, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("openapiDot1sMSTInstanceTopologyChangeStateGet() sanity successful\n\n");
  printf("Testing openapiDot1sMSTInstanceDesignatedRootIDGet():\n");

  result = openapiDot1sMSTInstanceDesignatedRootIDGet(NULL, mstid, &id);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceDesignatedRootIDGet():\n");
  result = openapiDot1sMSTInstanceDesignatedRootIDGet(clientHandle, badMstid, &id);
  printf("Invalid MST ID:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceDesignatedRootIDGet():\n");
  result = openapiDot1sMSTInstanceDesignatedRootIDGet(clientHandle, mstid, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("openapiDot1sMSTInstanceDesignatedRootIDGet() sanity successful\n\n");

  printf("Testing openapiDot1sMSTInstanceRootPathCostGet():\n");
  result = openapiDot1sMSTInstanceRootPathCostGet(NULL, mstid, &var1);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceRootPathCostGet():\n");
  result = openapiDot1sMSTInstanceRootPathCostGet(clientHandle, badMstid, &var1);
  printf("Invalid MST ID:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceRootPathCostGet():\n");
  result = openapiDot1sMSTInstanceRootPathCostGet(clientHandle, mstid, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("openapiDot1sMSTInstanceRootPathCostGet() sanity successful\n\n");

  printf("Testing openapiDot1sMSTInstanceRootPortIDGet():\n");

  result = openapiDot1sMSTInstanceRootPortIDGet(NULL, mstid, &var1);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceRootPortIDGet():\n");
  result = openapiDot1sMSTInstanceRootPortIDGet(clientHandle, badMstid, &var1);
  printf("Invalid MST ID:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceRootPortIDGet():\n");
  result = openapiDot1sMSTInstanceRootPortIDGet(clientHandle, mstid, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("openapiDot1sMSTInstanceRootPortIDGet() sanity successful\n\n");

  printf("Testing openapiDot1sCISTMaxAgeGet():\n");
  result = openapiDot1sCISTMaxAgeGet(NULL, &var1);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sCISTMaxAgeGet():\n");
  result = openapiDot1sCISTMaxAgeGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("openapiDot1sCISTMaxAgeGet() sanity successful\n\n");

  printf("Testing openapiDot1sBridgeMaxHopGet():\n");
  result = openapiDot1sBridgeMaxHopGet(NULL, &var1);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sBridgeMaxHopGet():\n");
  result = openapiDot1sBridgeMaxHopGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("openapiDot1sBridgeMaxHopGet() sanity successful\n\n");

  printf("Testing openapiDot1sBridgeTxHoldCountGet():\n");
  result = openapiDot1sBridgeTxHoldCountGet(NULL, &var1);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sBridgeTxHoldCountGet():\n");
  result = openapiDot1sBridgeTxHoldCountGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("openapiDot1sBridgeTxHoldCountGet() sanity successful\n\n");

  printf("Testing openapiDot1sCISTRootFwdDelayGet():\n");
  result = openapiDot1sCISTRootFwdDelayGet(NULL, &var1);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sCISTRootFwdDelayGet():\n");
  result = openapiDot1sCISTRootFwdDelayGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("openapiDot1sCISTRootFwdDelayGet() sanity successful\n\n");

  printf("Testing openapiDot1sCISTHelloTimeGet():\n");
  result = openapiDot1sCISTHelloTimeGet(NULL, 1, &var1);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sCISTHelloTimeGet():\n");
  result = openapiDot1sCISTHelloTimeGet(clientHandle, 9999, &var1);
  printf("Likely invalid interface number:(result = %d)\n", result);

  printf("Testing openapiDot1sCISTHelloTimeGet():\n");
  result = openapiDot1sCISTHelloTimeGet(clientHandle, 1, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("openapiDot1sCISTHelloTimeGet() sanity successful\n\n");

  printf("Testing openapiDot1sCISTBridgeHoldTimeGet():\n");
  result = openapiDot1sCISTBridgeHoldTimeGet(NULL, &var1);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sCISTBridgeHoldTimeGet():\n");
  result = openapiDot1sCISTBridgeHoldTimeGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("openapiDot1sCISTBridgeHoldTimeGet() sanity successful\n\n");

  printf("Testing openapiDot1sCISTRegionalRootIDGet():\n");
  result = openapiDot1sCISTRegionalRootIDGet(NULL, &id);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sCISTRegionalRootIDGet():\n");
  result = openapiDot1sCISTRegionalRootIDGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("openapiDot1sCISTRegionalRootIDGet() sanity successful\n\n");

  printf("Testing openapiDot1sCISTRegionalRootPathCostGet():\n");
  result = openapiDot1sCISTRegionalRootPathCostGet(NULL, &var1);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sCISTRegionalRootPathCostGet():\n");
  result = openapiDot1sCISTRegionalRootPathCostGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("openapiDot1sCISTRegionalRootPathCostGet() sanity successful\n\n");

  printf("Testing openapiDot1sCistBridgeMaxAgeSet():\n");
  result = openapiDot1sCistBridgeMaxAgeSet(NULL, maxAge);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1sCistBridgeMaxAgeSet(clientHandle, OPEN_DOT1S_BRIDGE_MAXAGE_MIN - 1);
  printf("less than minimum value argument 2:(result = %d)\n", result);
  result = openapiDot1sCistBridgeMaxAgeSet(clientHandle, OPEN_DOT1S_BRIDGE_MAXAGE_MAX + 1);
  printf("greater than maximum value argument 2:(result = %d)\n", result);

  printf("Testing openapiDot1sCistBridgeFwdDelaySet():\n");
  result = openapiDot1sCistBridgeFwdDelaySet(NULL, fwdDelay);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1sCistBridgeFwdDelaySet(clientHandle, OPEN_DOT1S_BRIDGE_FORWARDDELAY_MIN - 1);
  printf("less than minimum value argument 2:(result = %d)\n", result);
  result = openapiDot1sCistBridgeFwdDelaySet(clientHandle, OPEN_DOT1S_BRIDGE_FORWARDDELAY_MAX + 1);
  printf("greater than maximum value argument 2:(result = %d)\n", result);

  printf("Testing openapiDot1sBridgeMaxHopSet():\n");
  result = openapiDot1sBridgeMaxHopSet(NULL, maxHopCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1sBridgeMaxHopSet(clientHandle, OPEN_DOT1S_BRIDGE_MAXHOP_MIN - 1);
  printf("less than minimum value argument 2:(result = %d)\n", result);
  result = openapiDot1sBridgeMaxHopSet(clientHandle, OPEN_DOT1S_BRIDGE_MAXHOP_MAX + 1);
  printf("greater than maximum value argument 2:(result = %d)\n", result);

  printf("Testing openapiDot1sBridgeTxHoldCountSet():\n");
  result = openapiDot1sBridgeTxHoldCountSet(NULL, txHoldCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1sBridgeTxHoldCountSet(clientHandle, OPEN_DOT1S_BRIDGE_HOLDCOUNT_MIN - 1);
  printf("less than minimum value argument 2:(result = %d)\n", result);
  result = openapiDot1sBridgeTxHoldCountSet(clientHandle, OPEN_DOT1S_BRIDGE_HOLDCOUNT_MAX + 1);
  printf("greater than maximum value argument 2:(result = %d)\n", result);

  printf("Testing openapiDot1sConfigDigestKeyGet():\n");
  result = openapiDot1sConfigDigestKeyGet(NULL, &bufDesc);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1sConfigDigestKeyGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);
  result = openapiDot1sConfigDigestKeyGet(clientHandle, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 2:(result = %d)\n", result);
  result = openapiDot1sConfigDigestKeyGet(clientHandle, &zeroLenBuf);
  printf("buffdesc 0 length size argument 2:(result = %d)\n", result);

  printf("Testing openapiDot1sMstiTimeSinceTopologyChangeGet():\n");
  result = openapiDot1sMstiTimeSinceTopologyChangeGet(NULL, mstID, &days, &hours, &minutes, &seconds);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1sMstiTimeSinceTopologyChangeGet(clientHandle, mstID, NULL, &hours, &minutes, &seconds);
  printf("NULL argument 6:(result = %d)\n", result);
  result = openapiDot1sMstiTimeSinceTopologyChangeGet(clientHandle, mstID, &days, NULL, &minutes, &seconds);
  printf("NULL argument 6:(result = %d)\n", result);
  result = openapiDot1sMstiTimeSinceTopologyChangeGet(clientHandle, mstID, &days, &hours, NULL, &seconds);
  printf("NULL argument 6:(result = %d)\n", result);
  result = openapiDot1sMstiTimeSinceTopologyChangeGet(clientHandle, mstID, &days, &hours, &minutes, NULL);
  printf("NULL argument 6:(result = %d)\n", result);

  printf("Testing openapiDot1sPortNextGet():\n");
  result = openapiDot1sPortNextGet(NULL, mstID, prevIntIfNum, &intIfNum);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1sPortNextGet(clientHandle, mstID, prevIntIfNum, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for Spanning Tree Configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum, arg1, arg2, arg3, val;
  open_buffdesc switch_os_revision;
  open_buffdesc mstDigestKey;
  char switch_os_revision_string[100];
  char mstDigestKeyString[16];
  uint32_t days, hours, minutes, seconds, intIfNum;

  if (argc < 2)
  {
    printDot1sAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);
  
  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("dot1s_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Spanning Tree API example application");

  mstDigestKey.pstart = mstDigestKeyString;
  mstDigestKey.size = sizeof(mstDigestKeyString);

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");
  /* exercise various OPEN API Spanning Tree functions */

  switch (testNum)
  {
    case 1:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SModeSet(&clientHandle, arg1);
      break;

    case 2:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1SModeGet(&clientHandle);
      break;

    case 3:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SIntfBPDUFilterModeSet(&clientHandle, arg1, arg2);
      break;

    case 4:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SIntfBPDUFilterModeGet(&clientHandle, arg1);
      break;
    
    case 5:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SIntfBPDUFloodModeSet(&clientHandle, arg1, arg2);
      break;

    case 6:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SIntfBPDUFloodModeGet(&clientHandle, arg1);
      break;
  
    case 7:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SBPDUGuardModeSet(&clientHandle, arg1);
      break;
  
    case 8:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1SBPDUGuardModeGet(&clientHandle);
      break;
  
    case 9:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1SConfigNameSet(&clientHandle, argv[2]);
      break;

    case 10:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1SConfigNameGet(&clientHandle);
      break;

    case 11:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SConfigRevisionSet(&clientHandle, arg1);
      break;

    case 12:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1SConfigRevisionGet(&clientHandle);
      break;

    case 13:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SForceVersionSet(&clientHandle, arg1);
      break;

    case 14:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1SForceVersionGet(&clientHandle);
      break;

    case 15:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SIntfGuardModeSet(&clientHandle, arg1, arg2);
      break;

    case 16:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SIntfGuardModeGet(&clientHandle, arg1);
      break;

    case 17:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SMSTInstanceSet(&clientHandle, arg1, OPEN_ACTION_ADD);
      break;

    case 18:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SMSTInstanceSet(&clientHandle, arg1, OPEN_ACTION_DELETE);
      break;

    case 19:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1SMSTInstancesGet(&clientHandle);
      break;

    case 20:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SMSTInstancePrioritySet(&clientHandle, arg1, arg2);
      break;

    case 21:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SMSTInstancePriorityGet(&clientHandle, arg1);
      break;

    case 22:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SMSTInstanceVLANSet(&clientHandle, arg1, arg2, OPEN_ACTION_ADD);
      break;

    case 23:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SMSTInstanceVLANSet(&clientHandle, arg1, arg2, OPEN_ACTION_DELETE);
      break;

    case 24:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SMSTInstanceVLANsGet(&clientHandle, arg1);
      break;

    case 25:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SMSTInstanceFIdsGet(&clientHandle, arg1);
      break;

    case 26:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SIntfModeSet(&clientHandle, arg1, arg2);
      break;

    case 27:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SIntfModeGet(&clientHandle, arg1);
      break;

    case 28:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SIntfPortFwdStateGet(&clientHandle, arg1, arg2);
      break;

    case 29:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SIntfPortRoleGet(&clientHandle, arg1, arg2);
      break;

    case 30:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SIntfEdgePortModeSet(&clientHandle, arg1, arg2);
      break;

    case 31:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SIntfEdgePortModeGet(&clientHandle, arg1);
      break;

    case 32:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SIntfPortLoopInconsistentStateGet(&clientHandle, arg1, arg2);
      break;

    case 33:
      if (argc != 3)
      { 
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1sMSTInstanceBridgeIdentifierGet(&clientHandle, arg1);      
      break;
    case 34:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1sMSTInstanceTimeSinceTopologyChangeGet(&clientHandle, arg1);
      break;
    case 35:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1sMSTInstanceTopologyChangeCountGet(&clientHandle, arg1);
      break;
    case 36:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1sMSTInstanceTopologyChangeStateGet(&clientHandle, arg1);    
      break;
    case 37:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1sMSTInstanceDesignatedRootIDGet(&clientHandle, arg1);
      break;
    case 38:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1sMSTInstanceRootPathCostGet(&clientHandle, arg1);
      break;
    case 39:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1sMSTInstanceRootPortIDGet(&clientHandle, arg1);
      break;
    case 40:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sCISTMaxAgeGet(&clientHandle);
      break;
    case 41:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sBridgeMaxHopGet(&clientHandle);
      break;
    case 42:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sBridgeTxHoldCountGet(&clientHandle);
      break;
    case 43:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sCISTRootFwdDelayGet(&clientHandle);
      break;
    case 44:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1sCISTHelloTimeGet(&clientHandle, arg1);
      break;
    case 45:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sCISTBridgeHoldTimeGet(&clientHandle);
      break;
    case 46:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sCISTRegionalRootIDGet(&clientHandle);
      break;
    case 47:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sCISTRegionalRootPathCostGet(&clientHandle);
      break;

    case 48:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SMSTInstanceIntfPortIDGet(&clientHandle, arg1, arg2);
      break;
    
    case 49:
      if (argc != 5)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg3 = atoi(argv[4]);
      dot1SMSTInstanceIntfPrioritySet(&clientHandle, arg1, arg2, arg3);
      break;

    case 50:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SMSTInstanceIntfPriorityGet(&clientHandle, arg1, arg2);
      break;
    
    case 51:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SMSTInstancePortPathCostModeGet(&clientHandle, arg1, arg2);
      break;

    case 52:
      if (argc != 5)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg3 = atoi(argv[4]);
      dot1SMSTInstancePortPathCostSet(&clientHandle, arg1, arg2, arg3);
      break;
    
    case 53:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SMSTInstanceOperPortPathCostGet(&clientHandle, arg1, arg2);
      break;

    case 54:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SMstiPortDesignatedRootIDGet(&clientHandle, arg1, arg2);
      break;

    case 55:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SMstiPortDesignatedCostGet(&clientHandle, arg1, arg2);
      break;

    case 56:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SMstiPortDesignatedBridgeIDGet(&clientHandle, arg1, arg2);
      break;

    case 57:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SMstiPortDesignatedPortIDGet(&clientHandle, arg1, arg2);
      break;

    case 58:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SCistExtPortPathCostModeGet(&clientHandle, arg1);
      break;

    case 59:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SCistExtPortPathCostGet(&clientHandle, arg1);
      break;

    case 60:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SCistExtPortPathCostSet(&clientHandle, arg1, arg2);
      break;
    
    case 61:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SCistPortTopologyChangeAckGet(&clientHandle, arg1);
      break;

    case 62:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1SCistHelloTimeGet(&clientHandle);
      break;
            
    case 63:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SCistPortOperEdgeGet(&clientHandle, arg1);
      break;
            
    case 64:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SCistPortOperPointToPointGet(&clientHandle, arg1);
      break;
            
    case 65:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SCistPortRegionalRootIDGet(&clientHandle, arg1);
      break;
            
    case 66:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SCistPortPathCostGet(&clientHandle, arg1);
      break;
            
    case 67:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SMstiPortLoopInconsistentStateGet(&clientHandle, arg1, arg2);
      break;

    case 68:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SPortStatsTransitionsIntoLoopInconsistentStateGet(&clientHandle, arg1, arg2);
      break;

    case 69:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SPortStatsTransitionsOutOfLoopInconsistentStateGet(&clientHandle, arg1, arg2);
      break;

    case 70:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SPortStatsClear(&clientHandle, arg1);
      break;

    case 71:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SIntfHelloTimeGet(&clientHandle, arg1);
      break;

    case 72:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SIntfBPDUGuardEffectGet(&clientHandle, arg1);
      break;
    
    case 73:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SPortRestrictedTcnSet(&clientHandle, arg1, arg2);
      break;

    case 74:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SPortRestrictedTcnGet(&clientHandle, arg1);
      break;
    
    case 75:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SPortAutoEdgeGet(&clientHandle, arg1);
      break;

    case 76:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SPortAutoEdgeSet(&clientHandle, arg1, arg2);
      break;
    
    case 77:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SPortUpTimeGet(&clientHandle, arg1);
      break;

    case 78:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SPortStatsSTPBPDUsSentGet(&clientHandle, arg1);
      break;
    
    case 79:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SPortStatsSTPBPDUsReceivedGet(&clientHandle, arg1);
      break;
    
    case 80:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SPortStatsRSTPBPDUsSentGet(&clientHandle, arg1);
      break;
    
    case 81:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SPortStatsRSTPBPDUsReceivedGet(&clientHandle, arg1);
      break;
    
    case 82:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SPortStatsMSTPBPDUsSentGet(&clientHandle, arg1);
      break;
    
    case 83:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SPortStatsMSTPBPDUsReceivedGet(&clientHandle, arg1);
      break;      
      
     case 84:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sCistBridgeMaxAgeGet(&clientHandle, &val);
      break;      

      case 85:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sCistBridgeMaxAgeSet(&clientHandle, atoi(argv[2]));
      break;      

      case 86:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sCistBridgeHelloTimeGet(&clientHandle, &val);
      break;      

      case 87:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sCistBridgFwdDelayGet(&clientHandle, &val);
      break;      

      case 88:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sCistBridgeFwdDelaySet(&clientHandle, atoi(argv[2]));
      break;      

      case 89:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sBridgeMaxHopSet(&clientHandle, atoi(argv[2]));
      break;      

      case 90:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sBridgeTxHoldCountSet(&clientHandle, atoi(argv[2]));
      break;      

      case 91:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sCistRootFwdDelayGet(&clientHandle, &val);
      break;      

      case 92:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sTopologyChangePortIDGet(&clientHandle, &val);
      break;      

      case 93:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sMstiPortPathCostGet(&clientHandle, atoi(argv[2]), atoi(argv[3]), &val);
      break;      

    case 94:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SOpENAPIsTestSanity(&clientHandle, arg1);
      break;

    case 95:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sConfigDigestKeyGet(&clientHandle, &mstDigestKey);
      break;      

    case 96:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sMstiTimeSinceTopologyChangeGet(&clientHandle, atoi(argv[2]), &days, 
                                          &hours, &minutes, &seconds);
      break;      

    case 97:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sPortNextGet(&clientHandle, atoi(argv[2]), atoi(argv[3]),
                       &intIfNum);
      break;      

    default:
      printDot1sAppMenu();
      break;
  }
                                          
  /* Log goodbye message with OPEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Spanning Tree API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
