/*********************************************************************
*
* Copyright 2016-2022 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  ssh_example.c
*
* @purpose   SSH OpEN APIs Example 
*
* @component OpEN
*
* @comments
*
* @create    29/04/2013
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_ssh.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the ssh
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/
/*******************************************************************
*
* @brief  This function prints the SSH Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printSSHAppMenu()
{
  printf("Usage:  ssh_example <test#> <arg1> <arg2> ... \n");
  printf("Usage:  In below, <key-type> should be set to 0 for RSA, 1 for DSA and 2 for ECDSA\n");
  printf("Usage:  In below, <protocol_version> should be set to 0 for OPEN_SSH_PROTO_VERSION_1, and 1 for OPEN_SSH_PROTO_VERSION_2\n");
  printf("Usage:  In below, <mode> should be set to 0 for OPEN_DISABLE and 1 for OPEN_ENABLE\n");
  printf("Test 1: Generate SSH key: ssh_example 1 <key-type> \n");
  printf("Test 2: Get SSH key status: ssh_example 2 <key-type> \n");
  printf("Test 3: Delete SSH key: ssh_example 3 <key-type>\n");
  printf("Test 4: Set SSH admin mode: ssh_example 4 <mode>\n");
  printf("Test 5: Get SSH admin mode: ssh_example 5 \n");
  printf("Test 6: Set SSH protocol version: ssh_example 6 <protocol-version> \n");
  printf("Test 7: Get SSH protocol version: ssh_example 7 \n");
  printf("Test 8: Set SSH session idle timeout value: ssh_example 8 <timeout>\n");
  printf("Test 9: Get SSH session idle timeout value: ssh_example: ssh_example 9 \n");
  printf("Test 10: Set maximum number of allowed SSH sessions: ssh_example 10 <max-sessions>\n");
  printf("Test 11: Set maximum number of allowed SSH sessions to default: ssh_example 11 \n");
  printf("Test 12: Get maximum number of allowed SSH sessions: ssh_example 12 \n");
  printf("Test 13: Get maximum number of active SSH sessions: ssh_example 13 \n");
  printf("Test 14: SSH OpEN API Sanity: ssh_example 14 \n");
  printf("Test 15: Set SCP Server admin mode: ssh example 15 <mode>\n");
  printf("Test 16: Get SCP server admin mode: ssh_example 16 \n");
  printf("Test 17: SCP server file transfer start: ssh_example 17  <filename> \n");
  printf("Test 18: Set SCP server file transfer complete status: ssh_example 18 \n");
  printf("Test 19: Get the remote server password for a given client socket ID for outbound SSH: ssh_example 19 <clientSockID> \n");
  printf("Test 20: SCP server file upload transfer start: ssh_example 20  <filename> \n");
  printf("Test 21: Set SCP remote transfer complete status: ssh_example 21 <transfer-status> <transfer-type> \n");
  printf("Test 22: Sets the port number to be used for SSH Server: ssh_example 22 <portNum>\n");
  printf("Test 23: Gets ssh server port number: ssh_example 23 \n");
  printf("Test 24: Gets ssh version flags: ssh_example 24\n");
  printf("Test 25: Set the length for SSH Key to be generated: ssh_example 25 <type><length>\n");
  printf("Test 26: Gets the configured key length of SSH key: ssh_example 26 <type>\n");
  printf("Test 27: Gets the length of the existing SSH key: ssh_example 27 <type>\n");
  printf("Test 28: Gets key data for a specific key type: ssh_example 28 <type>\n");
  return;
}

/*********************************************************************
* @purpose  Generate SSH key.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    keyType          @b{(input)}   SSH key type
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void sshKeyGenerate(openapiClientHandle_t *clientHandle, OPEN_SSH_KEY_TYPE_t keyType)
{
  open_error_t result;

  if ((result = openapiSSHKeyGenerate(clientHandle, keyType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to generate SSH key. (result = %d)\n", result);
  }
  else
  {
    if (keyType == OPEN_SSH_KEY_TYPE_RSA)
    {
      printf("RSA");
    }
    else if (OPEN_SSH_KEY_TYPE_DSA == keyType)
    {
      printf("DSA");
    }
    else if (OPEN_SSH_KEY_TYPE_ECDSA == keyType)
    {
      printf("ECDSA");
    }
    printf(" key generated successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the status of the SSH key.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    keyType          @b{(input)}   SSH key type
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void sshKeyStatusGet(openapiClientHandle_t *clientHandle, OPEN_SSH_KEY_TYPE_t keyType)
{
  open_error_t result;
  char str[20];

  if (keyType == OPEN_SSH_KEY_TYPE_RSA)
  {
    strcpy(str, "RSA");
  }
  else if (keyType == OPEN_SSH_KEY_TYPE_DSA)
  {
    strcpy(str, "DSA");
  }
  else if (OPEN_SSH_KEY_TYPE_ECDSA == keyType)
  {
    strcpy(str, "ECDSA");
  }
  else
  {
    memset(str, 0, sizeof(str));
  }
 
  
  result = openapiSSHKeyStatusGet(clientHandle, keyType);

  if (result == OPEN_E_NONE)
  {
    printf(" %s key is present. \n", str);
  }
  else if (result == OPEN_E_FAIL)
  {
    printf(" %s key generation is in progress. \n", str);
  }
  else if (result == OPEN_E_NOT_FOUND)
  {
    printf(" %s key is not found. \n", str);
  }
  else
  {
    printf("Bad return code trying to get SSH keys status. (resul = %d)\n", result);
  }

  
  return;
}

/*********************************************************************
* @purpose  Delete SSH key.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    keyType          @b{(input)}   SSH key type
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void sshKeyDelete(openapiClientHandle_t *clientHandle, OPEN_SSH_KEY_TYPE_t keyType)
{
  open_error_t result;

  if ((result = openapiSSHKeyDelete(clientHandle, keyType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete SSH key. (result = %d)\n", result);
  }
  else
  {
    if (keyType == OPEN_SSH_KEY_TYPE_RSA)
    {
      printf("RSA");
    }
    else if (OPEN_SSH_KEY_TYPE_DSA == keyType)
    {
      printf("DSA");
    }
    else if (OPEN_SSH_KEY_TYPE_ECDSA == keyType)
    {
      printf("ECDSA");
    }
    printf(" key deleted successfully. (result = %d)\n", result);
  }

  return;
} 

/*********************************************************************
* @purpose  Set SSH admin mode.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    mode             @b{(input)}   SSH admin mode
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void sshAdminModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiSSHAdminModeSet(clientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set SSH admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("SSH admin mode set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get SSH admin mode.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void sshAdminModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  if ((result = openapiSSHAdminModeGet(clientHandle, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set SSH admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("SSH admin mode: %d (result = %d)\n", mode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set SSH protocol version.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    protoVersion     @b{(input)}   SSH protocol version
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void sshProtocolVersionSet(openapiClientHandle_t *clientHandle, OPEN_SSH_PROTO_VERSION_t protoVersion)
{
  open_error_t result;

  if ((result = openapiSSHProtocolVersionSet(clientHandle, protoVersion)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set SSH protocol version. (result = %d)\n", result);
  }
  else
  {
    printf("SSH protocol version is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get SSH protocol version.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void sshProtocolVersionGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_SSH_PROTO_VERSION_t protoVersion;

  if ((result = openapiSSHProtocolVersionGet(clientHandle, &protoVersion)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get SSH protocol version. (result = %d)\n", result);
  }
  else
  {
    printf("SSH protocol version: %d (result = %d)\n", protoVersion, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set SSH session timeout value.
* 
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    sshSessionTimeout  @b{(input)}   SSH session timeout value
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void sshSessionTimeoutSet(openapiClientHandle_t *clientHandle, uint32_t sshSessionTimeout)
{
  open_error_t result;

  if ((result = openapiSSHSessionTimeoutSet(clientHandle, sshSessionTimeout)) != OPEN_E_NONE)
  { 
    printf("Bad return code trying to set SSH session timeout. (result = %d)\n", result);
  }
  else
  { 
    printf("SSH session timeout is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get SSH session timeout value
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void sshSessionTimeoutGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t sshSessionTimeout;

  if ((result = openapiSSHSessionTimeoutGet(clientHandle, &sshSessionTimeout)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get SSH session timeout. (result = %d)\n", result);
  }
  else
  {
    printf("SSH session timeout: %d. (result = %d)\n", sshSessionTimeout, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set maximum number of SSH sessions allowed.
* 
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    maxSshSessions     @b{(input)}   Maximum SSH sessions allowed
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void sshMaxSessionsSet(openapiClientHandle_t *clientHandle, uint32_t maxSshSessions)
{
  open_error_t result;

  if ((result = openapiSSHMaxSessionsSet(clientHandle, maxSshSessions)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set maximum SSH sessions allowed. (result = %d)\n", result);
  }
  else
  {
    printf("Maximum number of SSH sessions is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Set maximum number of SSH sessions allowed to default.
* 
* @param    clientHandle       @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void sshMaxSessionsDefaultSet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  if ((result = openapiSSHMaxSessionsDefaultSet(clientHandle)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set maximum SSH sessions allowed to default. (result = %d)\n", result);
  }
  else
  {
    printf("Maximum number of SSH sessions is set to default successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get maximum number of SSH sessions allowed.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void sshMaxSessionsGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t maxSshSessions;

  if ((result = openapiSSHMaxSessionsGet(clientHandle, &maxSshSessions)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get maximum SSH sessions allowed. (result = %d)\n", result);
  }
  else
  {
    printf("Maximum number of SSH sessions: %d. (result = %d)\n", maxSshSessions, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the number of active SSH sessions.
* 
* @param    clientHandle  @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void sshActiveSessionsGet(openapiClientHandle_t *clientHandle) 
{
  open_error_t result;
  uint32_t activeSshSessions;

  if ((result = openapiSSHActiveSessionsGet(clientHandle, &activeSshSessions)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get active SSH sessions. (result = %d)\n", result);
  }
  else
  {
    printf("Number of active SSH sessions: %d. (result = %d)\n", activeSshSessions, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set SCP server admin mode.
*
* @param    clientHandle  @b{(input)}   client handle from registration API
* @param    mode          @b{(input)}   SSH admin mode
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void scpServerAdminModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiScpServerAdminModeSet(clientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set SCP server admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("SCP server admin mode set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get SCP server admin mode.
*
* @param    clientHandle  @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void scpServerAdminModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  if ((result = openapiScpServerAdminModeGet(clientHandle, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set SCP server  admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("SCP server admin mode: %s (result = %d)\n", (mode == OPEN_ENABLE ? "enable" : "disable"), result);
  }

  return;
}

/*********************************************************************
* @purpose  Initialte SCP server file upload transfer.
*
* @param    clientHandle  @b{(input)}  client handle from registration API
* @param    fileParams    @b{(inout)}  contains the source file name
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void scpServerRemoteTransferUploadStart(openapiClientHandle_t *clientHandle,
                                        open_buffdesc *fileParams)
{
  open_error_t result;

  if ((result = openapiScpRemoteTransferUploadStart(clientHandle, getpid(), fileParams)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to start SCP server remote file upload transfer (result = %d)\n", result);
  }
  else
  {
    printf("SCP server start remote file upload transfer : (result = %d)\n", result);
  }
  return;
}

/*********************************************************************
* @purpose  To update appication about the completion of scp remote 
*           file transfer.
*
* @param    clientHandle   @b{(input)} Client handle from registration API.
* @param    transferStatus  @b{(input)} transfer status
* @param    transferType    @b{(input)} transfer type
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void scpServerTransferRemoteComplete(openapiClientHandle_t *clientHandle,
                                     OPEN_TRANSFER_STATUS_t transferStatus,
                                     OPEN_SCP_OPER_t transferType)
{
  open_error_t result;

  if ((result = openapiScpTransferRemoteComplete(clientHandle, transferStatus, transferType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set SCP remote transfer complete (result = %d)\n", result);
  }
  else
  {
    printf("SCP server file remote transfer complete : (result = %d)\n", result);
  }
  return;
}

/*********************************************************************
* @purpose  Initialte SCP server file transfer. 
*
* @param    clientHandle  @b{(input)}  client handle from registration API
* @param    fileParams    @b{(inout)}  contains the download file name 
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void scpServerRemoteTransferStart (openapiClientHandle_t *clientHandle,
                                   open_buffdesc *fileParams)
{
  open_error_t result;

  if ((result = openapiScpRemoteTransferStart(clientHandle, getpid(), fileParams)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to start SCP server remote file transfer (result = %d)\n", result);
  }
  else
  {
    printf("SCP server start remote file  transfer : (result = %d)\n", result);
  }
  return;
}

/*********************************************************************
* @purpose  To update appication about the completion of scp remote file transfer.
*
* @param    clientHandle   @b{(input)} Client handle from registration API.
* @param    status          @b{(input)} SCP remote server transfer complete status.
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void ScpServerRemoteTransferComplete (openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  if ((result = openapiScpRemoteTransferComplete(clientHandle)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set SCP transfer complete (result = %d)\n", result);
  }
  else
  {
    printf("SCP server file transfer complete : (result = %d)\n", result);
  }
  return;
}

/*********************************************************************************
* @purpose  Get the remote server password for a client socket ID for outbound SSH.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    clientSockID        @b{(input)}  client socket ID
* @param    password            @b{(output)} remote server password 
*
* @returns  none
* 
* @end
*********************************************************************/
void sshRemotePasswordGet(openapiClientHandle_t *clientHandle,
                          uint32_t clientSockID,  open_buffdesc *password)
{ 
  open_error_t result = OPEN_E_NONE;

  if ((result = openapiSSHRemotePasswordGet(clientHandle, clientSockID, password)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the password for a client socket ID. (result = %d)\n", result);
  }
  else
  {
    printf("Password is returned successfully : (result = %d)\n", result); 
  }
  return;
}

/*********************************************************************
* @purpose  Sets the port number to be used for SSH Server.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    portNum       @b{(input)}  SSH server port-number

*
* @returns  none
*
* @end
*********************************************************************/
void sshdPortNumSet(openapiClientHandle_t *clientHandle, uint32_t portNum)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSshdPortNumSet(clientHandle, portNum)))
  {
    printf("Bad return code trying to sets the port number to be used for SSH Server. (result = %d)\n", result);
  }
  else
  {
    printf("port-number for SSH Server is succesfully set\n");
  }
  return;
}


/*********************************************************************
* @purpose  Gets ssh server port number.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    pPort       @b{(output)}  telnet access port number

*
* @returns  none
*
* @end
*********************************************************************/
void sshdPortNumGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t port;

  if (OPEN_E_NONE != (result = openapiSshdPortNumGet(clientHandle, &port)))
  {
    printf("Bad return code trying to gets ssh server port number. (result = %d)\n", result);
  }
  else
  {
    printf("ssh server port number is %d\n", port);
  }
  return;
}


/*********************************************************************
* @purpose  Gets ssh version flags.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    pSshV1Flag       @b{(output)}  ssh version 1 flag
* @param    pSshV2Flag       @b{(output)}  ssh version 2 flag

*
* @returns  none
*
* @end
*********************************************************************/
void sshdProtoLevelFlagsGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_BOOL_t sshV1Flag;
  OPEN_BOOL_t sshV2Flag;

  if (OPEN_E_NONE != (result = openapiSshdProtoLevelFlagsGet(clientHandle,
                                  &sshV1Flag, &sshV2Flag)))
  {
    printf("Bad return code trying to gets ssh version flags. (result = %d)\n", result);
  }
  else
  {
    printf("ssh server version1 flag is %s and version2 flag is %s\n", 
        (OPEN_TRUE == sshV1Flag) ? "enabled" : "disabled",
        (OPEN_TRUE == sshV2Flag) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Set the length for SSH Key to be generated.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    type       @b{(input)}  SSH key type
* @param    length       @b{(input)}  key length

*
* @returns  none
*
* @end
*********************************************************************/
void sshdKeyLenSet(openapiClientHandle_t *clientHandle, OPEN_SSH_KEY_TYPE_t type, uint32_t length)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSshdKeyLenSet(clientHandle, type, length)))
  {
    printf("Bad return code trying to set the length for SSH Key to be generated. (result = %d)\n", result);
  }
  else
  {
    printf("length for SSH key generation of type:%d is set to %d\n", type, length);
  }
  return;
}


/*********************************************************************
* @purpose  Gets the configured key length of SSH key.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    type       @b{(input)}  SSH key type
* @param    pLength       @b{(output)}  key length

*
* @returns  none
*
* @end
*********************************************************************/
void sshdCfgKeyLenGet(openapiClientHandle_t *clientHandle, OPEN_SSH_KEY_TYPE_t type)
{
  open_error_t result;
  uint32_t length;

  if (OPEN_E_NONE != (result = openapiSshdCfgKeyLenGet(clientHandle, type, &length)))
  {
    printf("Bad return code trying to gets the configured key length of SSH key. (result = %d)\n", result);
  }
  else
  {
    printf("Configured key length of SSH key is %d\n", length);
  }
  return;
}


/*********************************************************************
* @purpose  Gets the length of the existing SSH key.
*
* @param    clientHandle @b{(input)}  client handle from registration API
* @param    type          @b{(input)}  SSH key type

*
* @returns  none
*
* @end
*********************************************************************/
void sshdKeyLenGet(openapiClientHandle_t *clientHandle, OPEN_SSH_KEY_TYPE_t type)
{
  open_error_t result;
  uint32_t length;

  if (OPEN_E_NONE != (result = openapiSshdKeyLenGet(clientHandle, type, &length)))
  {
    printf("Bad return code trying to gets the length of the existing SSH key. (result = %d)\n", result);
  }
  else
  {
    printf("Configured key length of SSH key is %d\n", length);
  }
  return;
}


/*********************************************************************
* @purpose  Gets key data for a specific key type.
*
* @param    clientHandle @b{(input)}  client handle from registration API
* @param    type          @b{(input)}  SSH key type RSA or DSA.

*
* @returns  none
*
* @end
*********************************************************************/
void sshdKeyDataGet(openapiClientHandle_t *clientHandle, OPEN_SSH_KEY_TYPE_t type)
{
  open_error_t result;
  char key[OPEN_SSHD_KEY_DATA_SIZE_MAX] = {0};
  open_buffdesc keyData = {.pstart = key, .size = sizeof(key)};

  if (OPEN_E_NONE != (result = openapiSshdKeyDataGet(clientHandle, type, &keyData)))
  {
    printf("Bad return code trying to gets key data for a specific key type. (result = %d)\n", result);
  }
  else
  {
    printf("SSH key data for keyType:%d is %s\n", type, key);
  }
  return;
}

/*********************************************************************
* @purpose  SSH OpEN APIs Sanity
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void sshOpENAPISanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_error_t sanityResult = OPEN_E_NONE;
  OPEN_SSH_KEY_TYPE_t keyType = OPEN_SSH_KEY_TYPE_RSA;
  OPEN_CONTROL_t mode = OPEN_DISABLE;
  OPEN_SSH_PROTO_VERSION_t protoVersion = OPEN_SSH_PROTO_VERSION_2; 
  uint32_t activeSshSessions = 0;
  uint32_t maxSshSessions = 0;
  uint32_t sshSessionTimeout = 0;
  open_buffdesc fileParams;
  open_buffdesc password;
  uint32_t clientSockID = 1;
  uint32_t port;
  OPEN_BOOL_t sshV2Flag;
  OPEN_BOOL_t sshV1Flag;
  uint32_t length;
  char buffer[OPEN_SSHD_KEY_DATA_SIZE_MAX];
  open_buffdesc keyData;
  open_buffdesc zeroLenBuf;
  open_buffdesc badBufdescPointer;
  
  badBufdescPointer.pstart = (void *) NULL;
  badBufdescPointer.size = sizeof(buffer);
  keyData.pstart = buffer;
  keyData.size = sizeof(buffer);
  zeroLenBuf.pstart = buffer;
  zeroLenBuf.size = 0;

  /* openapiSSHKeyGenerate() */ 
  printf("\nTesting openapiSSHKeyGenerate(): \n");

  result = openapiSSHKeyGenerate(NULL, keyType);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSSHKeyGenerate(clientHandle, 4);
  printf("Invalid SSH key. (result = %d)\n", result);

  printf("openapiSSHKeyGenerate() sanity successful.\n");

  /* openapiSSHKeyDelete() */ 
  printf("\nTesting openapiSSHKeyDelete(): \n");

  result = openapiSSHKeyDelete(NULL, keyType);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSSHKeyDelete(clientHandle, 4);
  printf("Invalid SSH key. (result = %d)\n", result);

  printf("openapiSSHKeyDelete() sanity successful.\n");

  /* openapiSSHKeyStatusGet() */
  printf("\nTesting openapiSSHKeyStatusGet(): \n");

  openapiSSHKeyStatusGet(NULL, keyType);
  printf("NULL Client Handle. (result = %d)\n", result);
  
  openapiSSHKeyStatusGet(clientHandle, 4);
  printf("Invalid SSH key. (result = %d)\n", result);

  printf("openapiSSHKeyStatusGet() sanity successful.\n");

  /* openapiSSHAdminModeSet() */  
  printf("\nTesting openapiSSHAdminModeSet(): \n");

  result = openapiSSHAdminModeSet(NULL, mode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSSHAdminModeSet(clientHandle, 4);
  printf("Invalid SSH admin mode. (result = %d)\n", result); 

  printf("openapiSSHKeyStatusSet() sanity successful.\n");

  /* openapiSSHAdminModeGet() */
  printf("\nTesting openapiSSHAdminModeGet(): \n");

  result = openapiSSHAdminModeGet(NULL, &mode);
  printf("NULL Client Handle. (result = %d)\n", result);
  
  result = openapiSSHAdminModeGet(clientHandle, NULL);
  printf("SSH admin mode. (result = %d)\n", result);
  
  printf("openapiSSHKeyStatusGet() sanity successful.\n");

  /* openapiSSHProtocolVersionSet() */
  printf("\nTesting openapiSSHProtocolVersionSet(): \n");

  result = openapiSSHProtocolVersionSet(NULL, protoVersion); 
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSSHProtocolVersionSet(clientHandle, 5); 
  printf("Invalid SSH protocol version. (result = %d)\n", result);

  printf("openapiSSHProtocolVersionSet() sanity successful.\n");

  /* openapiSSHProtocolVersionGet() */
  printf("\nTesting openapiSSHProtocolVersionGet(): \n");

  result = openapiSSHProtocolVersionGet(NULL, &protoVersion);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSSHProtocolVersionGet(clientHandle, NULL);
  printf("SSH protocol version. (result = %d)\n", result);

  printf("openapiSSHProtocolVersionGet() sanity successful.\n");

  /* openapiSSHSessionTimeoutSet() */  
  printf("\nTesting openapiSSHSessionTimeoutSet(): \n");

  result = openapiSSHSessionTimeoutSet(NULL, sshSessionTimeout);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSSHSessionTimeoutSet(clientHandle, 200);
  printf("Invalid SSH timeout value. (result = %d)\n", result);

  printf("openapiSSHSessionTimeoutSet() sanity successful.\n");

  /* openapiSSHSessionTimeoutGet() */
  printf("\nTesting openapiSSHSessionTimeoutGet(): \n");

  result = openapiSSHSessionTimeoutGet(NULL, &sshSessionTimeout);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSSHSessionTimeoutGet(clientHandle, NULL);
  printf("NULL SSH timeout value. (result = %d)\n", result);

  printf("openapiSSHSessionTimeoutGet() sanity successful.\n");

  /* openapiSSHMaxSessionsSet() */
  printf("\nTesting openapiSSHMaxSessionsSet(): \n");

  result = openapiSSHMaxSessionsSet(NULL, maxSshSessions);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSSHMaxSessionsSet(clientHandle, 10);
  printf("Invalid maximum SSH sessions. (result = %d)\n", result);

  printf("openapiSSHMaxSessionsSet() sanity successful.\n");

  /* openapiSSHMaxSessionsDefaultSet() */
  printf("\nTesting openapiSSHMaxSessionsDefaultSet(): \n");

  result = openapiSSHMaxSessionsDefaultSet(NULL);
  printf("NULL Client Handle. (result = %d)\n", result);

  printf("openapiSSHMaxSessionsDefaultSet() sanity successful.\n");

  /* openapiSSHMaxSessionsGet() */
  printf("\nTesting openapiSSHMaxSessionsGet(): \n");

  result = openapiSSHMaxSessionsGet(NULL, &maxSshSessions);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSSHMaxSessionsGet(clientHandle, NULL);
  printf("NULL maximum SSH sessions. (result = %d)\n", result);

  printf("openapiSSHMaxSessionsGet() sanity successful.\n");

  /* openapiSSHActiveSessionsGet() */
  printf("\nTesting openapiSSHActiveSessionsGet(): \n");

  result = openapiSSHActiveSessionsGet(NULL, &activeSshSessions);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSSHActiveSessionsGet(clientHandle, NULL);
  printf("NULL active SSH sessions. (result = %d)\n", result);

  printf("openapiSSHActiveSessionsGet() sanity successful.\n");

  /* openapiScpServerAdminModeSet() */
  printf("\nTesting openapiScpServerAdminModeSet(): \n");

  sanityResult = OPEN_E_NONE;

  /* client handle is NULL. The result should be failed */ 
  result = openapiScpServerAdminModeSet(NULL, mode);
  printf("NULL Client Handle. (result = %d)\n", result);
  if (OPEN_E_NONE == result) 
  {
    sanityResult = OPEN_E_FAIL;
  }
  /* Invalid scp admin mode. The result should be failed */ 
  result = openapiScpServerAdminModeSet(clientHandle, 4);
  printf("Invalid SCP Server admin mode. (result = %d)\n", result);
  if (OPEN_E_NONE == result) 
  {
    sanityResult = OPEN_E_FAIL;
  }

  if (OPEN_E_NONE == sanityResult)
  {
    printf("openapiScpServerAdminModeSet() sanity successful.\n");
  }
  else 
  {
    printf("openapiScpServerAdminModeSet() sanity failed.\n");
  }

  /* openapiScpServerAdminModeGet() */
  printf("\nTesting openapiScpServerAdminModeGet(): \n");

  sanityResult = OPEN_E_NONE;

  /* client handle is NULL. The result should be failed */ 
  result = openapiScpServerAdminModeGet(NULL, &mode);
  printf("NULL Client Handle. (result = %d)\n", result);
  if (OPEN_E_NONE == result) 
  {
    sanityResult = OPEN_E_FAIL;
  }
  /* scp admin mode input argument is NULL. The result should be failed */ 
  result = openapiScpServerAdminModeGet(clientHandle, NULL);
  printf("SCP sever admin mode. (result = %d)\n", result);
  if (OPEN_E_NONE == result) 
  {
    sanityResult = OPEN_E_FAIL;
  }

  if (OPEN_E_NONE == sanityResult)
  {
    printf("openapiScpServerAdminModeGet() sanity successful.\n");
  }
  else 
  {
    printf("openapiScpServerAdminModeGet() sanity failed.\n");
  }

  /* "openapiScpRemoteTransferStart() */
  printf("\nTesting openapiScpRemoteTransferStart(): \n");

  sanityResult = OPEN_E_NONE;

  /* client handle is NULL. The result should be failed */ 
  result = openapiScpRemoteTransferStart(NULL, getpid(), &fileParams);
  printf("NULL Client Handle. (result = %d)\n", result);
  if (OPEN_E_NONE == result)
  {
    sanityResult = OPEN_E_FAIL;
  }

  /* file name argument is NULL. The result should be failed */ 
  result = openapiScpRemoteTransferStart(clientHandle, getpid(), NULL);
  printf("Invalid SCP transfer file name. (result = %d)\n", result);
  if (OPEN_E_NONE == result)
  {
    sanityResult = OPEN_E_FAIL;
  }

  if (OPEN_E_NONE == sanityResult)
  {
    printf("openapiScpRemoteTransferStart() sanity successful.\n");
  }
  else
  {
    printf("openapiScpRemoteTransferStart() sanity failed.\n");
  }

  /* openapiScpRemoteTransferComplete() */
  printf("\nTesting openapiScpRemoteTransferComplete(): \n");

  sanityResult = OPEN_E_NONE;

  /* client handle is NULL. The result should be failed */ 
  result = openapiScpRemoteTransferComplete(NULL);
  printf("NULL Client Handle. (result = %d)\n", result);
  if (OPEN_E_NONE == result) 
  {
    sanityResult = OPEN_E_FAIL;
  }

  if (OPEN_E_NONE == sanityResult)
  {
    printf("openapiScpRemoteTransferComplete() sanity successful.\n");
  }
  else 
  {
    printf("openapiScpRemoteTransferComplete() sanity failed.\n");
  }

  /* openapiSSHRemotePasswordGet() */
  printf("\nTesting openapiSSHRemotePasswordGet(): \n");

  result = openapiSSHRemotePasswordGet(NULL, clientSockID, &password);
  printf("NULL client handle:(result = %d)\n", result);
  
  result = openapiSSHRemotePasswordGet(clientHandle, clientSockID, NULL);
  printf("NULL argumet 3:(result = %d)\n", result);

  /* openapiScpRemoteTransferUploadStart() */
  printf("\nTesting openapiScpRemoteTransferUploadStart(): \n");
    
  sanityResult = OPEN_E_NONE;
  
  /* client handle is NULL. The result should be failed */
  result = openapiScpRemoteTransferUploadStart(NULL, getpid(), &fileParams);
  printf("NULL Client Handle. (result = %d)\n", result);
  if (OPEN_E_NONE == result)
  {
    sanityResult = OPEN_E_FAIL;
  }
  
  /* file name argument is NULL. The result should be failed */
  result = openapiScpRemoteTransferUploadStart(clientHandle, getpid(), NULL);
  printf("Invalid SCP transfer file name. (result = %d)\n", result);
  if (OPEN_E_NONE == result)
  {
    sanityResult = OPEN_E_FAIL;
  } 
  
  if (OPEN_E_NONE == sanityResult) 
  {
    printf("openapiScpRemoteTransferUploadStart() sanity successful.\n");
  }
  else
  { 
    printf("openapiScpRemoteTransferUploadStart() sanity failed.\n");
  }

  /* openapiScpTransferRemoteComplete() */
  printf("\nTesting openapiScpTransferRemoteComplete(): \n");

  sanityResult = OPEN_E_NONE;

  /* client handle is NULL. The result should be failed */
  result = openapiScpTransferRemoteComplete(NULL, OPEN_TRANSFER_SUCCESS, OPEN_SCP_PUSH);
  printf("NULL Client Handle. (result = %d)\n", result);
  if (OPEN_E_NONE == result)
  {
    sanityResult = OPEN_E_FAIL;
  }

  if (OPEN_E_NONE == sanityResult)
  {
    printf("openapiScpTransferRemoteComplete() sanity successful.\n");
  }
  else
  {
    printf("openapiScpTransferRemoteComplete() sanity failed.\n");
  }

  /* transfer status is 0. The result should be failed */
  result = openapiScpTransferRemoteComplete(clientHandle, 0, OPEN_SCP_PULL);
  printf("NULL transfer status. (result = %d)\n", result);
  if (OPEN_E_NONE == result)
  {
    sanityResult = OPEN_E_FAIL;
  }

  if (OPEN_E_NONE == sanityResult)
  {
    printf("openapiScpTransferRemoteComplete() sanity successful.\n");
  }
  else
  {
    printf("openapiScpTransferRemoteComplete() sanity failed.\n");
  }

  /* transfer type is 0. The result should be failed */
  result = openapiScpTransferRemoteComplete(clientHandle, OPEN_TRANSFER_SUCCESS, 0);
  printf("NULL transfer status. (result = %d)\n", result);
  if (OPEN_E_NONE == result)
  {
    sanityResult = OPEN_E_FAIL;
  }

  if (OPEN_E_NONE == sanityResult)
  {
    printf("openapiScpTransferRemoteComplete() sanity successful.\n");
  }
  else
  {
    printf("openapiScpTransferRemoteComplete() sanity failed.\n");
  }
 
  printf("Testing openapiSshdPortNumGet():\n");
  result = openapiSshdPortNumGet(NULL, &port);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiSshdPortNumGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiSshdProtoLevelFlagsGet():\n");
  result = openapiSshdProtoLevelFlagsGet(NULL, &sshV1Flag, &sshV2Flag);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiSshdProtoLevelFlagsGet(clientHandle, NULL, &sshV2Flag);
  printf("NULL argument 2:(result = %d)\n", result);
  result = openapiSshdProtoLevelFlagsGet(clientHandle, &sshV1Flag, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiSshdCfgKeyLenGet():\n");
  result = openapiSshdCfgKeyLenGet(NULL, keyType, &length);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiSshdCfgKeyLenGet(clientHandle, keyType, NULL);
  printf("NULL argument3 length:(result = %d)\n", result);

  printf("Testing openapiSshdKeyLenGet():\n");
  result = openapiSshdKeyLenGet(NULL, keyType, &length);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiSshdKeyLenGet(clientHandle, keyType, NULL);
  printf("NULL argument3 length:(result = %d)\n", result);

  printf("Testing openapiSshdKeyDataGet():\n");
  result = openapiSshdKeyDataGet(NULL, keyType, &keyData);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiSshdKeyDataGet(clientHandle, keyType, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiSshdKeyDataGet(clientHandle, keyType, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiSshdKeyDataGet(clientHandle, keyType, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result); 
  return;  
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  uint32_t arg1, arg2;
  open_buffdesc fileParams;
  open_buffdesc password;
  char fileNameStr[OPENAPI_FILE_NAME_LENGTH_MAX];
  uint32_t portNum;
  OPEN_SSH_KEY_TYPE_t type;
  uint32_t length;

  if (argc < 2)
  {
    printSSHAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("ssh_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting SSH API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 3)
      {
        printSSHAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      sshKeyGenerate(&clientHandle, arg1);
      break;
    case 2:
      if (argc != 3)
      {
        printSSHAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      sshKeyStatusGet(&clientHandle, arg1);
      break;
    case 3:
      if (argc != 3)
      {
        printSSHAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      sshKeyDelete(&clientHandle, arg1);
      break;
    case 4:
      if (argc != 3)
      {
        printSSHAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      sshAdminModeSet(&clientHandle, arg1);
      break;
    case 5:
      if (argc != 2)
      {
        printSSHAppMenu();
        exit(1);
      }
      sshAdminModeGet(&clientHandle);
      break;
    case 6:
      if (argc != 3)
      {
        printSSHAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      sshProtocolVersionSet(&clientHandle, arg1);
      break;
    case 7:
      if (argc != 2)
      {
        printSSHAppMenu();
        exit(1);
      }
      sshProtocolVersionGet(&clientHandle);
      break;
    case 8:
      if (argc != 3)
      {
        printSSHAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      sshSessionTimeoutSet(&clientHandle, arg1);
      break;
    case 9:
      if (argc != 2)
      {
        printSSHAppMenu();
        exit(1);
      }
      sshSessionTimeoutGet(&clientHandle);
      break;
    case 10:
      if (argc != 3)
      {
        printSSHAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      sshMaxSessionsSet(&clientHandle, arg1);
      break;
    case 11:
      if (argc != 2)
      {
        printSSHAppMenu();
        exit(1);
      }
      sshMaxSessionsDefaultSet(&clientHandle);
      break;
    case 12:
      if (argc != 2)
      {
        printSSHAppMenu();
        exit(1);
      }
      sshMaxSessionsGet(&clientHandle);
      break;
    case 13:
      if (argc != 2)
      {
        printSSHAppMenu();
        exit(1);
      }
      sshActiveSessionsGet(&clientHandle);
      break;
    case 14:
      if (argc != 2)
      {
        printSSHAppMenu();
        exit(1);
      }
      sshOpENAPISanity(&clientHandle);
      break;
    case 15:
      if (argc != 3)
      {
        printSSHAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      scpServerAdminModeSet(&clientHandle, arg1);
      break;
    case 16:
      if (argc != 2)
      {
        printSSHAppMenu();
        exit(1);
      }
      scpServerAdminModeGet(&clientHandle);
      break;
    case 17:
      if (argc != 3)
      {
        printSSHAppMenu();
        exit(1);
      }
      if (strlen(argv[2]) != 0)
      { 
        strncpy(fileNameStr, argv[2], strlen(argv[2]));
        fileParams.size = sizeof(fileNameStr);
        fileParams.pstart = &fileNameStr;
        scpServerRemoteTransferStart(&clientHandle, &fileParams);
      } 
      break;
    case 18:
      if (argc != 2)
      {
        printSSHAppMenu();
        exit(1);
      }
      ScpServerRemoteTransferComplete(&clientHandle);
      break;

    case 19:
      if (argc != 3)
      {
        printSSHAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      sshRemotePasswordGet(&clientHandle, arg1, &password);
      break;

    case 20:
      if (argc != 3)
      {
        printSSHAppMenu();
        exit(1);
      }
      if (strlen(argv[2]) != 0)
      {
        strncpy(fileNameStr, argv[2], strlen(argv[2]));
        fileParams.size = sizeof(fileNameStr);
        fileParams.pstart = &fileNameStr;
        scpServerRemoteTransferUploadStart(&clientHandle, &fileParams);
      }
      break;
    case 21:
      if (argc != 4)
      {
        printSSHAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      scpServerTransferRemoteComplete(&clientHandle, arg1, arg2);
      break;
    case 22:
      if (argc != 3)
      {
        printSSHAppMenu();
        exit(1);
      }
      portNum = atoi(argv[2]);
      sshdPortNumSet(&clientHandle, portNum);
      break;
    case 23:
      if (2 != argc)
      {
        printSSHAppMenu();
        exit(1);
      }
      sshdPortNumGet(&clientHandle);
      break;
    case 24:
      if (2 != argc)
      {
        printSSHAppMenu();
        exit(1);
      }
      sshdProtoLevelFlagsGet(&clientHandle);
      break;
    case 25:
      if (4 != argc)
      {
        printSSHAppMenu();
        exit(1);
      }
      type = atoi(argv[2]);
      length = atoi(argv[3]);
      sshdKeyLenSet(&clientHandle, type, length);
      break;
    case 26:
      if (3 != argc)
      {
        printSSHAppMenu();
        exit(1);
      }
      type = atoi(argv[2]);
      sshdCfgKeyLenGet(&clientHandle, type);
      break;
    case 27:
      if (3 != argc)
      {
        printSSHAppMenu();
        exit(1);
      }
      type = atoi(argv[2]);
      sshdKeyLenGet(&clientHandle, type);
      break;
    case 28:
      if (3 != argc)
      {
        printSSHAppMenu();
        exit(1);
      }
      type = atoi(argv[2]);
      sshdKeyDataGet(&clientHandle, type);
      break;
    default:
      printSSHAppMenu();
      break;
  }

  
  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping SSH API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
