/*********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  sslt_example.c
*
* @purpose   OpEN sslt example.
*
* @component OpEN
*
* @create    02/01/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_sslt.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the sslt Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Sets admin Mode of the SSL Tunnel server: %s 1 <mode>\n", name);
  printf("Test 2: Gets admin mode of SSL Tunnel server: %s 2\n", name);
  printf("Test 3: Sets secure port number to listen on for SSLT connections: %s 3 <port>\n", name);
  printf("Test 4: Gets secure port number that the SSLT connection is using: %s 4\n", name);
  printf("Test 5: Sets SSL Protocol Level to be used by SSL Tunnel codes: %s 5 <protocolId><mode>\n", name);
  printf("Test 6: Gets SSL Protocol mode status in use by the SSL Tunnel code: %s 6 <protocolId>\n", name);
  printf("Test 7: Sets Soft Timeout (in minutes) for SSLT sessions: %s 7 <timeout>\n", name);
  printf("Test 8: Gets SSLT session soft timeout information.(in minutes): %s 8\n", name);
  printf("Test 9: Sets SSLT session hard timeout (in hours): %s 9 <timeout>\n", name);
  printf("Test 10: Gets SSLT session hard timeout information (in hours): %s 10\n", name);
  printf("Test 11: Sets maximum allowable SSLT sessions: %s 11 <maxSession>\n", name);
  printf("Test 12: Gets  maximum number of SSLT sessions supported: %s 12\n", name);
  printf("Test 13: Gets operation mode of SSL Tunnel server: %s 13\n", name);
  printf("Test 14: Determine if self-signed server certificate exists: %s 14 <certNum>\n", name);
  printf("Test 15: Determine if self-signed server and rootcert certificates exist: %s 15 <certNum>\n", name);
  printf("Test 16: Generate a self-signed server certificate for SSL tunnel: %s 16 <certNum><keyLength><commonName>\n", name);
  printf("                                                                         <orgName><orgUnit><location>\n");
  printf("                                                                         <state><country><email><days>\n"); 
  printf("Test 17: Remove SSLT certificate: %s 17 <certNum>\n", name);
  printf("Test 18: Gets the active certificate for the SSL tunnel: %s 18\n", name);
  printf("Test 19: Gets operational active SSL certificate: %s 19 \n", name);
  printf("Test 20: Gets status of SSL certificate expiry: %s 20 <certNum>\n", name);
  printf("Test 21: Set the active certificate for the SSL tunnel: %s 21 <certNum>\n", name);
  printf("Test 22: Check if HTTPS mode is enabled and the certificate is active: %s 22 <certNum>\n", name);
  printf("Test 23: Run API sanity checks: %s 23 \n", name);
  printf("Test 24: Gets the minimum and maximum supported certificate keylength: %s 24 <minKeyLength><maxKeyLength>\n", name);
  printf("Test 25: Gets the minimum and maximum supported sizes of certificate Subject domain name: %s 25 <minSubjectName><maxSubjectName>\n", name);
  printf("Test 26: Gets the minimum and maximum supported duration of certificate validity: %s 26 <minValidDays><maxValidDays>\n", name);
  printf("Test 27: Get certificate country name size: %s 27 <countryNameSize>\n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t mode;
  uint32_t port;
  uint32_t protocolId = 10;
  uint32_t timeout;
  uint32_t session;
  uint32_t certNum;
  OPEN_BOOL_t isHttpModeEnabled;
  OPEN_BOOL_t isExpired;

  printf("Testing openapiSsltAdminModeGet():\n");

  result = openapiSsltAdminModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiSsltAdminModeGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiSsltSecurePortGet():\n");
  result = openapiSsltSecurePortGet(NULL, &port);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiSsltSecurePortGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiSsltProtocolLevelGet():\n");
  result = openapiSsltProtocolLevelGet(NULL, protocolId, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiSsltProtocolLevelGet(clientHandle, protocolId, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiSsltSessionSoftTimeOutGet():\n");
  result = openapiSsltSessionSoftTimeOutGet(NULL, &timeout);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiSsltSessionSoftTimeOutGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiSsltSessionHardTimeOutGet():\n");
  result = openapiSsltSessionHardTimeOutGet(NULL, &timeout);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiSsltSessionHardTimeOutGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiSsltNumSessionsGet():\n");
  result = openapiSsltNumSessionsGet(NULL, &session);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiSsltNumSessionsGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiSsltOperModeGet():\n");
  result = openapiSsltOperModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiSsltOperModeGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiSsltCertificateActiveGet():\n");
  result = openapiSsltCertificateActiveGet(NULL, &certNum);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiSsltCertificateActiveGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiSsltCertificateOperActiveGet():\n");
  result = openapiSsltCertificateOperActiveGet(NULL, &certNum);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiSsltCertificateOperActiveGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiSsltCertificateExpiryStatusGet():\n");
  result = openapiSsltCertificateExpiryStatusGet(NULL, certNum, &isExpired);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiSsltCertificateExpiryStatusGet(clientHandle, certNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiSsltCheckHTTPSEnabledAndOperational():\n");
  result = openapiSsltCheckHTTPSEnabledAndOperational(NULL, certNum, &isHttpModeEnabled);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiSsltCheckHTTPSEnabledAndOperational(clientHandle, certNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  return;
}


/*********************************************************************
* @purpose  Sets admin Mode of the SSL Tunnel server.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void ssltAdminModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSsltAdminModeSet(clientHandle, mode)))
  {
    printf("Bad return code trying to sets admin Mode of the SSL Tunnel server. (result = %d)\n", result);
  }
  else
  {
    printf("Admin mode of SSL Tunnel server is successfully %s\n", (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets admin mode of SSL Tunnel server.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void ssltAdminModeGet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSsltAdminModeGet(clientHandle, pMode)))
  {
    printf("Bad return code trying to gets admin mode of SSL Tunnel server. (result = %d)\n", result);
  }
  else
  {
    printf("Admin mode of SSL Tunnel server is in %s state\n", (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets secure port number to listen on for SSLT connections.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    port       @b{(input)}  port-number

*
* @returns  none
*
* @end
*********************************************************************/
void ssltSecurePortSet(openapiClientHandle_t *clientHandle, uint32_t port)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSsltSecurePortSet(clientHandle, port)))
  {
    printf("Bad return code trying to sets secure port number to listen on for SSLT connections. (result = %d)\n", result);
  }
  else
  {
    printf("SSLT connection secure port number is succesfully set\n");
  }
  return;
}


/*********************************************************************
* @purpose  Gets secure port number that the SSLT connection is using.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    pPort       @b{(output)}  port number

*
* @returns  none
*
* @end
*********************************************************************/
void ssltSecurePortGet(openapiClientHandle_t *clientHandle, uint32_t *pPort)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSsltSecurePortGet(clientHandle, pPort)))
  {
    printf("Bad return code trying to gets secure port number that the SSLT connection is using. (result = %d)\n", result);
  }
  else
  {
    printf("SSLT connection secure port number is %d\n", *pPort);
  }
  return;
}


/*********************************************************************
* @purpose  Sets SSL Protocol Level to be used by SSL Tunnel codes.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    protocolId       @b{(input)}  ID of SSLT supported protocol
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void ssltProtocolLevelSet(openapiClientHandle_t *clientHandle, uint32_t protocolId, uint32_t mode)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSsltProtocolLevelSet(clientHandle, protocolId, mode)))
  {
    printf("Bad return code trying to sets SSL Protocol Level to be used by SSL Tunnel codes. (result = %d)\n", result);
  }
  else
  {
    printf("SSL tunnel protocol level admin mode is successfully %s\n", (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets SSL Protocol mode status in use by the SSL Tunnel code.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    protocolId       @b{(input)}  ID of SSLT supported protocol
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void ssltProtocolLevelGet(openapiClientHandle_t *clientHandle, uint32_t protocolId, uint32_t *pMode)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSsltProtocolLevelGet(clientHandle, protocolId, pMode)))
  {
    printf("Bad return code trying to gets SSL Protocol mode status in use by the SSL Tunnel code. (result = %d)\n", result);
  }
  else
  {
    printf("SSL tunnel protocol level admin mode is in %s state\n", (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets Soft Timeout (in minutes) for SSLT sessions.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    timeout       @b{(input)}  SSLT session soft timeout

*
* @returns  none
*
* @end
*********************************************************************/
void ssltSessionSoftTimeOutSet(openapiClientHandle_t *clientHandle, uint32_t timeout)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSsltSessionSoftTimeOutSet(clientHandle, timeout)))
  {
    printf("Bad return code trying to sets Soft Timeout (in minutes) for SSLT sessions. (result = %d)\n", result);
  }
  else
  {
    printf("SSLT session soft timeout is succesfully set to %d minutes\n", timeout);
  }
  return;
}


/*********************************************************************
* @purpose  Gets SSLT session soft timeout information.(in minutes)
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    pTimeout       @b{(output)}  SSLT session soft timeout

*
* @returns  none
*
* @end
*********************************************************************/
void ssltSessionSoftTimeOutGet(openapiClientHandle_t *clientHandle, uint32_t *pTimeout)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSsltSessionSoftTimeOutGet(clientHandle, pTimeout)))
  {
    printf("Bad return code trying to gets SSLT session soft timeout information.(in minutes) (result = %d)\n", result);
  }
  else
  {
    printf("SSLT session soft timeout is %d minutes\n", *pTimeout);
  }
  return;
}


/*********************************************************************
* @purpose  Sets SSLT session hard timeout (in hours).
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    timeout       @b{(input)}  SSLT session hard timeout

*
* @returns  none
*
* @end
*********************************************************************/
void ssltSessionHardTimeOutSet(openapiClientHandle_t *clientHandle, uint32_t timeout)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSsltSessionHardTimeOutSet(clientHandle, timeout)))
  {
    printf("Bad return code trying to sets SSLT session hard timeout (in hours). (result = %d)\n", result);
  }
  else
  {
    printf("SSLT session hard timeout is succesfully set to %d hours\n", timeout);
  }
  return;
}


/*********************************************************************
* @purpose  Gets SSLT session hard timeout information (in hours).
*
* @param    clientHandle  @b{(input)}  client handle from registration API
* @param    pTimeout       @b{(output)}  HTTP session hard timeout

*
* @returns  none
*
* @end
*********************************************************************/
void ssltSessionHardTimeOutGet(openapiClientHandle_t *clientHandle, uint32_t *pTimeout)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSsltSessionHardTimeOutGet(clientHandle, pTimeout)))
  {
    printf("Bad return code trying to gets SSLT session hard timeout information (in hours). (result = %d)\n", result);
  }
  else
  {
    printf("SSLT session hard timeout is %d hours\n", *pTimeout);
  }
  return;
}


/*********************************************************************
* @purpose  Sets maximum allowable SSLT sessions.
*
* @param    clientHandle    @b{(input)}  client handle from registration API
* @param    maxSession       @b{(input)}  maximum allowable number of SSLT sessions

*
* @returns  none
*
* @end
*********************************************************************/
void ssltNumSessionsSet(openapiClientHandle_t *clientHandle, uint32_t maxSession)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSsltNumSessionsSet(clientHandle, maxSession)))
  {
    printf("Bad return code trying to sets maximum allowable SSLT sessions. (result = %d)\n", result);
  }
  else
  {
    printf("maximum allowable SSLT sessions is succesfully set to %d\n", maxSession);
  }
  return;
}


/*********************************************************************
* @purpose  Gets  maximum number of SSLT sessions supported.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    pSession       @b{(output)}  maximum allowable number of web sessions configured

*
* @returns  none
*
* @end
*********************************************************************/
void ssltNumSessionsGet(openapiClientHandle_t *clientHandle, uint32_t *pSession)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSsltNumSessionsGet(clientHandle, pSession)))
  {
    printf("Bad return code trying to gets  maximum number of SSLT sessions supported. (result = %d)\n", result);
  }
  else
  {
    printf("maximum number of SSLT sessions is %d\n", *pSession);
  }
  return;
}


/*********************************************************************
* @purpose  Gets operation mode of SSL Tunnel server.
*
* @param    clientHandle  @b{(input)}  client handle from registration API
* @param    pMode          @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void ssltOperModeGet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSsltOperModeGet(clientHandle, pMode)))
  {
    printf("Bad return code trying to gets operation mode of SSL Tunnel server. (result = %d)\n", result);
  }
  else
  {
    printf("operation mode of SSL Tunnel server is in %s state\n",
        (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Determine if self-signed server certificate exists.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    certNum       @b{(input)}  certificate number (1 or 2)

*
* @returns  none
*
* @end
*********************************************************************/
void ssltCertificateExists(openapiClientHandle_t *clientHandle, uint32_t certNum)
{
  open_error_t result;

  result = openapiSsltCertificateExists(clientHandle, certNum);

  if ((OPEN_E_NONE != result) && (OPEN_E_NOT_FOUND != result))
  {
    printf("Bad return code trying to determine if self-signed server certificate exists. (result = %d)\n", result);
  }
  else
  {
    printf("self-signed server certificate %s\n", 
          (OPEN_E_NONE == result) ? "exist" : "doesn't exist");
  }
  return;
}


/*********************************************************************
* @purpose  Determine if self-signed server and rootcert certificates exist.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    certNum       @b{(input)}  certificate number (1 or 2)

*
* @returns  none
*
* @end
*********************************************************************/
void ssltCertificateAndServerKeyExists(openapiClientHandle_t *clientHandle, uint32_t certNum)
{
  open_error_t result;

  result = openapiSsltCertificateAndServerKeyExists(clientHandle, certNum);
  if ((OPEN_E_NONE != result) && (OPEN_E_NOT_FOUND != result))
  {
    printf("Bad return code trying to determine if self-signed server and rootcert certificates exist. (result = %d)\n", result);
  }
  else
  {
    printf("self-signed server and rootcert certificate %s\n", 
          (OPEN_E_NONE == result) ? "exist" : "doesn't exist");
  }
  return;
}

/*********************************************************************
* @purpose  Generate a self-signed server certificate for SSL tunnel.
*
* @param    clientHandle  @b{(input)}  client handle from registration API
* @param    certNum       @b{(input)}  certificate number
* @param    keyLength     @b{(input)}  RSA key length, 0 will use existing key.
* @param    pCommonName   @b{(input)}  subject DN common name field.
* @param    pOrgName      @b{(input)}  subject DN organization name field.
* @param    pOrgUnit      @b{(input)}  subject DN organization unit field.
* @param    pLocation     @b{(input)}  subject DN location field.
* @param    pState        @b{(input)}  subject DN state field.
* @param    pCountry      @b{(input)}  subject DN country field.
* @param    pEmail        @b{(input)}  subject DN email field.
* @param    days          @b{(input)}  days certificate will be valid.
*
* @returns  none
*
* @end
*********************************************************************/
void ssltCertificateGenerate(openapiClientHandle_t *clientHandle,
                             uint32_t certNum, uint32_t keyLength,
                             char *pCommonName, char *pOrgName,
                             char *pOrgUnit, char *pLocation,
                             char *pState, char *pCountry,
                             char *pEmail, uint32_t days)
{
  open_error_t result;
  char cnStr[256] = {0};
  open_buffdesc cnBufDesc = {.pstart = cnStr, .size = sizeof(cnStr)};
  char onStr[256] = {0};
  open_buffdesc onBufDesc = {.pstart = onStr, .size = sizeof(onStr)};
  char ouStr[256] = {0};
  open_buffdesc  ouBufDesc = {.pstart = ouStr, .size = sizeof(ouStr)};
  char locStr[256] = {0};
  open_buffdesc locBufDesc = {.pstart = locStr, .size = sizeof(locStr)};
  char stateStr[256] = {0};
  open_buffdesc stateBufDesc = {.pstart = stateStr, .size = sizeof(stateStr)};
  char cntryStr[256] = {0};
  open_buffdesc cntryBufDesc = {.pstart = cntryStr, .size = sizeof(cntryStr)};
  char emailStr[256] = {0};
  open_buffdesc  emailBufDesc = {.pstart = emailStr, .size = sizeof(emailStr)};

  strncpy(cnStr, pCommonName, sizeof(cnStr)-1);
  cnBufDesc.size = strlen(cnStr);
  strncpy(onStr, pOrgName, sizeof(onStr)-1);
  onBufDesc.size = strlen(onStr);
  strncpy(ouStr, pOrgUnit, sizeof(ouStr)-1);
  ouBufDesc.size = strlen(ouStr);
  strncpy(locStr, pLocation, sizeof(locStr)-1);
  locBufDesc.size = strlen(locStr);
  strncpy(stateStr, pState, sizeof(stateStr)-1);
  stateBufDesc.size = strlen(stateStr);
  strncpy(cntryStr, pCountry, sizeof(cntryStr)-1);
  cntryBufDesc.size = strlen(cntryStr);
  strncpy(emailStr, pEmail, sizeof(emailStr)-1);
  emailBufDesc.size = strlen(emailStr);

  if (OPEN_E_NONE != (result = openapiSsltCertificateGenerate(clientHandle,
                                               certNum, keyLength,
                                               &cnBufDesc, &onBufDesc,
                                               &ouBufDesc, &locBufDesc,
                                               &stateBufDesc, &cntryBufDesc,
                                               &emailBufDesc, days)))
  {
    printf("Bad return code trying to generate a self-signed server certificate for SSL tunnel. (result = %d)\n", result);
  }
  else
  {
    printf("self-signed server certificate for SSL tunnel is succesfully generated.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Remove SSLT certificate.
*
* @param    clientHandle @b{(input)}  client handle from registration API
* @param    certNum       @b{(input)}  certificate number (1 or 2)

*
* @returns  none
*
* @end
*********************************************************************/
void ssltCertRemove(openapiClientHandle_t *clientHandle, uint32_t certNum)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSsltCertRemove(clientHandle, certNum)))
  {
    printf("Bad return code trying to remove SSLT certificate. (result = %d)\n", result);
  }
  else
  {
    printf("SSLT certificate %d is removed.\n", certNum);
  }
  return;
}


/*********************************************************************
* @purpose  Gets the active certificate for the SSL tunnel.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    pCertNum       @b{(output)}  certificate number (1 or 2)

*
* @returns  none
*
* @end
*********************************************************************/
void ssltCertificateActiveGet(openapiClientHandle_t *clientHandle, uint32_t *pCertNum)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSsltCertificateActiveGet(clientHandle, pCertNum)))
  {
    printf("Bad return code trying to gets the active certificate for the SSL tunnel. (result = %d)\n", result);
  }
  else
  {
    printf("Active SSL tunnel certificate is %d.\n", *pCertNum);
  }
  return;
}


/*********************************************************************
* @purpose  Gets operational active SSL certificate.
*
* @param    clientHandle  @b{(input)}  client handle from registration API
* @param    pCertNum       @b{(output)}  certificate number (1 or 2)

*
* @returns  none
*
* @end
*********************************************************************/
void ssltCertificateOperActiveGet(openapiClientHandle_t *clientHandle, uint32_t *pCertNum)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSsltCertificateOperActiveGet(clientHandle, pCertNum)))
  {
    printf("Bad return code trying to gets operational active SSL certificate. (result = %d)\n", result);
  }
  else
  {
    printf("Operational active SSL tunnel certificate is %d.\n", *pCertNum);
  }
  return;
}


/*********************************************************************
* @purpose  Gets status of SSL certificate expiry.
*
* @param    clientHandle @b{(input)}  client handle from registration API
* @param    certNum       @b{(input)}  certificate number (1 or 2)
* @param    pIsExpired    @b{(output)}  TRUE - Certificate is not valid, False - Certificate is valid

*
* @returns  none
*
* @end
*********************************************************************/
void ssltCertificateExpiryStatusGet(openapiClientHandle_t *clientHandle, uint32_t certNum, OPEN_BOOL_t *pIsExpired)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSsltCertificateExpiryStatusGet(clientHandle, certNum, pIsExpired)))
  {
    printf("Bad return code trying to gets status of SSL certificate expiry. (result = %d)\n", result);
  }
  else
  {
    printf("SSL certificate is %s.\n", (OPEN_TRUE == *pIsExpired) ? "expired" : "active");
  }
  return;
}


/*********************************************************************
* @purpose  Set the active certificate for the SSL tunnel.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    certNum       @b{(input)}  certificate number (1 or 2)

*
* @returns  none
*
* @end
*********************************************************************/
void ssltCertificateActiveSet(openapiClientHandle_t *clientHandle, uint32_t certNum)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSsltCertificateActiveSet(clientHandle, certNum)))
  {
    printf("Bad return code trying to set the active certificate for the SSL tunnel. (result = %d)\n", result);
  }
  else
  {
    printf("SSLT certificate %d is set to active.\n", certNum);
  }
  return;
}


/*********************************************************************
* @purpose  Check if HTTPS mode is enabled and the certificate is active.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    certNum       @b{(input)}  certificate number (1 or 2)
* @param    pIsHttpModeEnabled       @b{(output)}  certificate number (1 or 2)

*
* @returns  none
*
* @end
*********************************************************************/
void ssltCheckHTTPSEnabledAndOperational(openapiClientHandle_t *clientHandle, uint32_t certNum, OPEN_BOOL_t *pIsHttpModeEnabled)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSsltCheckHTTPSEnabledAndOperational(clientHandle, certNum, pIsHttpModeEnabled)))
  {
    printf("Bad return code trying to check if HTTPS mode is enabled and the certificate is active. (result = %d)\n", result);
  }
  else
  {
    printf("HTTPS mode and certicate is %s\n", 
          (OPEN_TRUE == *pIsHttpModeEnabled) ? "active" : "in-active");
  }
  return;
}



/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for sslt
*
* @returns   0: Success
* @returns  -1: Failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  OPEN_CONTROL_t mode;
  uint32_t port;
  uint32_t protocolId;
  uint32_t timeout;
  uint32_t session;
  uint32_t certNum;
  OPEN_BOOL_t isHttpModeEnabled;
  OPEN_BOOL_t isExpired;
  uint32_t recvMode;
  uint32_t keyLength;
  uint32_t days;
  uint32_t minVal;
  uint32_t maxVal;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if (OPEN_E_NONE != (result = openapiClientRegister("sslt example", &clientHandle)))
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (OPEN_E_NONE != openapiConnectivityCheck(&clientHandle))
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting sslt API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (OPEN_E_NONE == openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision))
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = (1 == recvMode) ? OPEN_ENABLE : OPEN_DISABLE;
          ssltAdminModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 2)
      {
        ssltAdminModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 3)
      {
        port = atoi(argv[2]);
        ssltSecurePortSet(&clientHandle, port);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 2)
      {
        ssltSecurePortGet(&clientHandle, &port);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 4)
      {
        protocolId = atoi(argv[2]);
        recvMode = atoi(argv[3]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = (1 == recvMode) ? OPEN_ENABLE : OPEN_DISABLE;
          ssltProtocolLevelSet(&clientHandle, protocolId, mode);
        }
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 3)
      {
        protocolId = atoi(argv[2]);
        ssltProtocolLevelGet(&clientHandle, protocolId, &mode);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 3)
      {
        timeout = atoi(argv[2]);
        ssltSessionSoftTimeOutSet(&clientHandle, timeout);
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 2)
      {
        ssltSessionSoftTimeOutGet(&clientHandle, &timeout);
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 3)
      {
        timeout = atoi(argv[2]);
        ssltSessionHardTimeOutSet(&clientHandle, timeout);
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 2)
      {
        ssltSessionHardTimeOutGet(&clientHandle, &timeout);
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 3)
      {
        session = atoi(argv[2]);
        ssltNumSessionsSet(&clientHandle, session);
        show_help = 0;
      }
      break;
    case 12:
      if (argc == 2)
      {
        ssltNumSessionsGet(&clientHandle, &session);
        show_help = 0;
      }
      break;
    case 13:
      if (argc == 2)
      {
        ssltOperModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 14:
      if (argc == 3)
      {
        certNum = atoi(argv[2]);
        ssltCertificateExists(&clientHandle, certNum);
        show_help = 0;
      }
      break;
    case 15:
      if (argc == 3)
      {
        certNum = atoi(argv[2]);
        ssltCertificateAndServerKeyExists(&clientHandle, certNum);
        show_help = 0;
      }
      break;
    case 16:
      if (argc == 12)
      {
        certNum = atoi(argv[2]);
        keyLength = atoi(argv[3]);
        days = atoi(argv[11]);
        ssltCertificateGenerate(&clientHandle, certNum, keyLength,
                                argv[4], argv[5], argv[6], argv[7],
                                argv[8], argv[9], argv[10], days);
        show_help = 0;
      }
      break;
    case 17:
      if (argc == 3)
      {
        certNum = atoi(argv[2]);
        ssltCertRemove(&clientHandle, certNum);
        show_help = 0;
      }
      break;
    case 18:
      if (argc == 2)
      {
        ssltCertificateActiveGet(&clientHandle, &certNum);
        show_help = 0;
      }
      break;
    case 19:
      if (argc == 2)
      {
        ssltCertificateOperActiveGet(&clientHandle, &certNum);
        show_help = 0;
      }
      break;
    case 20:
      if (argc == 3)
      {
        certNum = atoi(argv[2]);
        ssltCertificateExpiryStatusGet(&clientHandle, certNum, &isExpired);
        show_help = 0;
      }
      break;
    case 21:
      if (argc == 3)
      {
        certNum = atoi(argv[2]);
        ssltCertificateActiveSet(&clientHandle, certNum);
        show_help = 0;
      }
      break;
    case 22:
      if (argc == 3)
      {
        certNum = atoi(argv[2]);
        ssltCheckHTTPSEnabledAndOperational(&clientHandle, certNum, &isHttpModeEnabled);
        show_help = 0;
      }
      break;
    case 23:
      if (argc == 2)
      {
        runSanity(&clientHandle);
        show_help = 0;
      }
      break;
    case 24:
      if (argc == 2)
      {
        if (OPEN_E_NONE != (result = openapiSsltCertificateKeyLengthGet(&clientHandle, &minVal, &maxVal)))
        {
          printf("Bad return code trying to get supported values of certificate keylength. (result = %d)\n", result);
        }
        else
        {
          printf("Certificate keylength minimum size is %d and maximum size is %d.\n", minVal, maxVal);
        }
        show_help = 0;
      }
      break;
    case 25:
      if (argc == 2)
      {
        if (OPEN_E_NONE != (result = openapiSsltCertificateSubjectNameSizeGet(&clientHandle, &minVal, &maxVal)))
        {
          printf("Bad return code trying to get supported sizes of certificate subject domain name. (result = %d)\n", result);
        }
        else
        {
          printf("Certificate subject domain name minimum size is %d and maximum size is %d.\n", minVal, maxVal);
        }
        show_help = 0;
      }
      break;
    case 26:
      if (argc == 2)
      {
        if (OPEN_E_NONE != (result = openapiSsltCertificateValidDaysGet(&clientHandle, &minVal, &maxVal)))
        {
          printf("Bad return code trying to get supported values of certificate validity. (result = %d)\n", result);
        }
        else
        {
          printf("Certificate validity minimum days is %d and maximum days is %d.\n", minVal, maxVal);
        }
        show_help = 0;
      }
      break;
    case 27:
      if (argc == 2)
      {
        if (OPEN_E_NONE != (result = openapiSsltCertificateCountryNameSizeGet(&clientHandle, &maxVal)))
        {
          printf("Bad return code trying to get certificate country name size. (result = %d)\n", result);
        }
        else
        {
          printf("Certificate country name size is %d.\n", maxVal);
        }
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping sslt API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
