/*********************************************************************
*
*  Copyright 2022-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  stm_example.c
*
* @purpose   OpEN STM Template example.
*
* @component OpEN
*
* @create    11/28/2022
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_stm.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the STM Template Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Gets current active Template Id: %s 1 <templateId>\n", name);
  printf("Test 2: Sets active template id: %s 2 <templateId>\n", name);
  printf("Test 3: Get the first valid template ID: %s 3 \n", name);
  printf("Test 4: Get the next valid template ID: %s 4 <currentTemplateId>\n", name);
  printf("Test 5: Gets name and description of specific template id: %s 5 <templateId>\n", name);
  printf("Test 6: Sets(Revert to) Default template: %s 6 \n", name);
  printf("Test 7: Print all STMs: %s 7 \n", name);
  printf("Test 8: Run API sanity checks: %s 8 \n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  uint32_t firstTemplateId;
  uint32_t nextTemplateId=0;
  uint32_t currentTemplateId=0;
  uint32_t templateId;
  char buffer[128];
  open_buffdesc templateName, templateDesc;
  open_buffdesc zeroLenBuf;
  open_buffdesc badBufdescPointer;

  badBufdescPointer.pstart = (void *) NULL;
  badBufdescPointer.size = sizeof(buffer);
  badBufdescPointer.size = 0;
  zeroLenBuf.pstart = buffer;
  zeroLenBuf.size = 0;

  printf("Testing STM Template OpEN APIs sanity:\n");

  printf("Testing openapiStmActiveTemplateIdGet():\n");
  result = openapiStmActiveTemplateIdGet(NULL, &templateId);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiStmActiveTemplateIdGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiStmFirstTemplateIdGet():\n");
  result = openapiStmFirstTemplateIdGet(NULL, &firstTemplateId);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiStmFirstTemplateIdGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiStmNextTemplateIdGet():\n");
  result = openapiStmNextTemplateIdGet(NULL, currentTemplateId, &nextTemplateId);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiStmNextTemplateIdGet(clientHandle, currentTemplateId, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiStmTemplateNameDescriptionGet():\n");
  result = openapiStmTemplateNameDescriptionGet(NULL, templateId, &templateName, &templateDesc);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiStmTemplateNameDescriptionGet(clientHandle, templateId, NULL, &templateDesc);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiStmTemplateNameDescriptionGet(clientHandle, templateId, &badBufdescPointer, &templateDesc);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiStmTemplateNameDescriptionGet(clientHandle, templateId, &zeroLenBuf, &templateDesc);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);
  result = openapiStmTemplateNameDescriptionGet(clientHandle, templateId, &templateName, NULL);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiStmTemplateNameDescriptionGet(clientHandle, templateId, &templateName, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiStmTemplateNameDescriptionGet(clientHandle, templateId, &templateName, &zeroLenBuf);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);

  printf("Testing openapiStmTemplateRevertToDefault():\n");
  result = openapiStmTemplateRevertToDefault(NULL);
  printf("NULL client handle:(result = %d)\n", result);

  return;
}


/*********************************************************************
* @purpose  Gets current active Template Id.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    templateId       @b{(output)}  STM template ID

*
* @returns  none
*
* @end
*********************************************************************/
void stmActiveTemplateIdGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  uint32_t     templateId;

  if ((result = openapiStmActiveTemplateIdGet(client_handle, &templateId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets current active Template Id. (result = %d)\n", result);
  }
  else
  {
    printf("Current active template ID is %d\n", templateId);
  }
  return;
}

/*********************************************************************
* @purpose  Sets active template id.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    templateId       @b{(input)}  STM template ID

*
* @returns  none
*
* @end
*********************************************************************/
void stmActiveTemplateIdSet(openapiClientHandle_t *client_handle, uint32_t templateId)
{
  open_error_t result;

  if ((result = openapiStmActiveTemplateIdSet(client_handle, templateId)) != OPEN_E_NONE)
  {
    if (OPEN_E_EXISTS == result)
    {
      printf("Template ID:%d is arleady configured\n", templateId);
    }
    else if (OPEN_E_UNAVAIL == result)
    {
      printf("Template ID:%d is not supported\n", templateId);
    }
    else
    {
      printf("Bad return code trying to set active template ID:%d. (result = %d)\n",
             templateId, result);
    }
  }
  else
  {
    printf("Active template id is set to %d\n", templateId);
  }
  return;
}


/*********************************************************************
* @purpose  Get the first valid template ID.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    firstTemplateId       @b{(output)}  First valid STM template ID

*
* @returns  none
*
* @end
*********************************************************************/
void stmFirstTemplateIdGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  uint32_t firstTemplateId;

  if ((result = openapiStmFirstTemplateIdGet(client_handle, &firstTemplateId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the first valid template ID. (result = %d)\n", result);
  }
  else
  {
    printf("First template ID:%d\n", firstTemplateId);
  }
  return;
}


/*********************************************************************
* @purpose  Get the next valid template ID.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    currentTemplateId       @b{(input)}  Current STM template ID
* @param    nextTemplateId       @b{(output)}  Next valid STM template ID

*
* @returns  none
*
* @end
*********************************************************************/
void stmNextTemplateIdGet(openapiClientHandle_t *client_handle, uint32_t currentTemplateId)
{
  open_error_t result;
  uint32_t nextTemplateId;

  if ((result = openapiStmNextTemplateIdGet(client_handle, currentTemplateId, &nextTemplateId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the next valid template ID. (result = %d)\n", result);
  }
  else
  {
    printf("Current template ID:%d  Next template ID:%d\n",
            currentTemplateId, nextTemplateId);
  }
  return;
}


/*********************************************************************
* @purpose  Gets name and description of specific template id
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    templateId       @b{(input)}  Template ID
* @param    templateName       @b{(output)}  Template name
* @param    templateDesc       @b{(output)}  Template Description

*
* @returns  none
*
* @end
*********************************************************************/
void stmTemplateNameDescriptionGet(openapiClientHandle_t *client_handle, uint32_t templateId)
{
  open_buffdesc templateName, templateDesc;
  open_error_t result;

  char   tempName[100], tempDesc[100];

  templateName.pstart = tempName;
  templateName.size   = OPEN_STM_TEMPLATE_NAME_SIZE;

  templateDesc.pstart = tempDesc;
  templateDesc.size   = OPEN_STM_TEMPLATE_DESCRIPTION_SIZE ;

  if ((result = openapiStmTemplateNameDescriptionGet(client_handle, templateId, &templateName, &templateDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get name and description of specific "
           "template id (result = %d)\n", result);
  }
  else
  {
    printf("Template ID:%d Template Name:%s Template Description:%s\n",
           templateId, tempName, tempDesc);
  }
  return;
}


/*********************************************************************
* @purpose  Sets(Revert to) Default template.
*
* @param    client_handle       @b{(input)}  client handle from registration API

*
* @returns  none
*
* @end
*********************************************************************/
void stmTemplateRevertToDefault(openapiClientHandle_t *client_handle)
{
  open_error_t result;

  if ((result = openapiStmTemplateRevertToDefault(client_handle)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set(Revert to) Default template. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("Unable to revert to default template.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Prints all STM templates.
*
* @param    client_handle       @b{(input)}  client handle from registration API

*
* @returns  none
*
* @end
*********************************************************************/
void printStms(openapiClientHandle_t *client_handle)
{
  uint32_t     first = 0, next = 0;
  char         tempName[OPEN_STM_TEMPLATE_NAME_SIZE], tempDesc[OPEN_STM_TEMPLATE_DESCRIPTION_SIZE];
  open_buffdesc  templName, templDesc;

  templName.pstart = tempName;
  templName.size   = OPEN_STM_TEMPLATE_NAME_SIZE;

  templDesc.pstart = tempDesc;
  templDesc.size   = OPEN_STM_TEMPLATE_DESCRIPTION_SIZE;
  
  if (OPEN_E_NONE != openapiStmFirstTemplateIdGet(client_handle, &first))
  {
    printf("\n Unable to get first template details, return \n");
    return;
  }

  next = first;
  do
  {  
    first = next;
    templName.size   = OPEN_STM_TEMPLATE_NAME_SIZE;
    templDesc.size   = OPEN_STM_TEMPLATE_DESCRIPTION_SIZE;
    if (OPEN_E_NONE == openapiStmTemplateNameDescriptionGet(client_handle, first,
                                                            &templName, &templDesc))
    {      
      printf("\n Template ID:%d Template Name:%s Template Description:%s\n",
             first, (char *)templName.pstart, (char *)templDesc.pstart);
    } 
    else
    {
      printf("\n Unable to get template details for template ID:%d\n", first);
    }
  }while(OPEN_E_NONE == openapiStmNextTemplateIdGet(client_handle, first, &next));
}


/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for STM Template
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum, templateId;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("STM Template example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting STM Template API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 2)
      {
        stmActiveTemplateIdGet(&clientHandle);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 3)
      {
	templateId = atoi(argv[2]);
        stmActiveTemplateIdSet(&clientHandle, templateId);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 2)
      {
        stmFirstTemplateIdGet(&clientHandle);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 3)
      {
	templateId = atoi(argv[2]);
        stmNextTemplateIdGet(&clientHandle, templateId);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 3)
      {
	templateId = atoi(argv[2]);
        stmTemplateNameDescriptionGet(&clientHandle, templateId);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 2)
      {
        stmTemplateRevertToDefault(&clientHandle);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 2)
      {
        printStms(&clientHandle);
        show_help = 0;
      }
      break;

    case 8:
      if (argc == 2)
      {
        runSanity(&clientHandle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping STM Template API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}

