/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  rfc1213_example.c
*
* @purpose   Example application to retrieve RFC 1213 information using OpEN APIs
*
* @component OpEN
*
* @comments
*
* @create    01/11/2015
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_1213.h"

/*********************************************************************
* @purpose  Sanity test for RFC 1213 System parameters  
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void rfc1213SystemSanityTest (openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc buffdesc;
  char *str = NULL;
  char *testSysName="DummyName";
  char *testSysLocation="DummyLocation";
  char *testSysContact="DummyContact";

  if ((str = (char *)malloc(OPEN_RFC1213_PARAM_SIZE)) == NULL)
  {
    printf("Could not allocate memory.\n"); 
    return;
  }

  buffdesc.pstart = str;
  buffdesc.size = OPEN_RFC1213_PARAM_SIZE;

  if ((result = openapi1213SystemSysDescrGet (clientHandle, &buffdesc) != OPEN_E_NONE))
  {
    printf("Bad return code trying to get system description. (result = %d)\n", result);
    printf("System Description Get test -- FAIL\n\n");
  }
  else
  {
    printf("System description retrieved successfully. system_descr=\"%s\" size=%d (result = %d)\n",
           (char *) buffdesc.pstart, buffdesc.size, result);
    printf("System Description Get test -- PASS\n\n");
  }

  buffdesc.size = OPEN_RFC1213_PARAM_SIZE;
  if ((result = openapi1213SystemSysNameGet (clientHandle, &buffdesc) != OPEN_E_NONE))
  {
    printf("Bad return code trying to get system name. (result = %d)\n", result);
  }
  else
  {
    printf("System name retrieved successfully. Configured system name=\"%s\" size=%d (result = %d)\n",
           (char *) buffdesc.pstart, buffdesc.size, result);
  }

  printf("Trying to set the system name to %s\n", testSysName);

  memset(buffdesc.pstart, 0, OPEN_RFC1213_PARAM_SIZE);
  strncpy ((char *) buffdesc.pstart, testSysName, OPEN_RFC1213_PARAM_SIZE-1);
  buffdesc.size = strlen(testSysName)+1;

  if ((result = openapi1213SystemSysNameSet (clientHandle, &buffdesc) != OPEN_E_NONE))
  {
    printf("Bad return code trying to set system name. (result = %d)\n", result);
  }
  else
  {
    printf("System name set is successful. (result = %d)\n", result);
  }

  memset(buffdesc.pstart, 0, OPEN_RFC1213_PARAM_SIZE);
  buffdesc.size = OPEN_RFC1213_PARAM_SIZE;
  if ((result = openapi1213SystemSysNameGet (clientHandle, &buffdesc) != OPEN_E_NONE))
  {
    printf("Bad return code trying to get system name. (result = %d)\n", result);
  }
  else
  {
    printf("System name retrieved successfully. system name=%s size=%d (result = %d)\n",
           (char *) buffdesc.pstart, buffdesc.size, result);
  }

  if (0 == memcmp (buffdesc.pstart, testSysName, strlen(testSysName)))
  {
    printf("System Name Set test -- PASS\n\n");
  }
  else
  {
    printf("System Name Set test -- FAIL (New: %s Org: %s)\n\n", (char *)buffdesc.pstart, testSysName);
  }

  buffdesc.size = OPEN_RFC1213_PARAM_SIZE;
  if ((result = openapi1213SystemSysLocationGet (clientHandle, &buffdesc) != OPEN_E_NONE))
  {
    printf("Bad return code trying to get system location. (result = %d)\n", result);
  }
  else
  {
    printf("System location retrieved successfully. Configured system location=\"%s\" size=%d (result = %d)\n",
           (char *) buffdesc.pstart, buffdesc.size, result);
  }

  printf("Trying to set the system location to %s\n", testSysLocation);

  memset(buffdesc.pstart, 0, OPEN_RFC1213_PARAM_SIZE);
  strncpy ((char *) buffdesc.pstart, testSysLocation, OPEN_RFC1213_PARAM_SIZE-1);
  buffdesc.size = strlen(testSysLocation)+1;
  if ((result = openapi1213SystemSysLocationSet (clientHandle, &buffdesc) != OPEN_E_NONE))
  {
    printf("Bad return code trying to set system location. (result = %d)\n", result);
  }
  else
  {
    printf("System location set is successful. (result = %d)\n", result);
  }

  memset(buffdesc.pstart, 0, OPEN_RFC1213_PARAM_SIZE);
  buffdesc.size = OPEN_RFC1213_PARAM_SIZE;
  if ((result = openapi1213SystemSysLocationGet (clientHandle, &buffdesc) != OPEN_E_NONE))
  {
    printf("Bad return code trying to get system location. (result = %d)\n", result);
  }
  else
  {
    printf("System location retrieved successfully. system location=\"%s\" size=%d (result = %d)\n",
           (char *) buffdesc.pstart, buffdesc.size, result);
  }

  if (0 == memcmp (buffdesc.pstart, testSysLocation, strlen(testSysLocation)))
  {
    printf("System Location Set test -- PASS\n\n");
  }
  else
  {
    printf("System Location Set test -- FAIL (New: %s Org: %s)\n\n", (char *)buffdesc.pstart, testSysLocation);
  }

  buffdesc.size = OPEN_RFC1213_PARAM_SIZE;
  if ((result = openapi1213SystemSysContactGet (clientHandle, &buffdesc) != OPEN_E_NONE))
  {
    printf("Bad return code trying to get system contact. (result = %d)\n", result);
  }
  else
  {
    printf("System contact retrieved successfully. Configured system contact=\"%s\" size=%d (result = %d)\n",
           (char *) buffdesc.pstart, buffdesc.size, result);
  }

  printf("Trying to set the system contact to %s\n", testSysContact);

  memset(buffdesc.pstart, 0, OPEN_RFC1213_PARAM_SIZE);
  strncpy ((char *) buffdesc.pstart, testSysContact, OPEN_RFC1213_PARAM_SIZE-1);
  buffdesc.size = strlen(testSysContact)+1;
  if ((result = openapi1213SystemSysContactSet (clientHandle, &buffdesc) != OPEN_E_NONE))
  {
    printf("Bad return code trying to set system contact. (result = %d)\n", result);
  }
  else
  {
    printf("System contact set is successful. (result = %d)\n", result);
  }

  memset(buffdesc.pstart, 0, OPEN_RFC1213_PARAM_SIZE);
  buffdesc.size = OPEN_RFC1213_PARAM_SIZE;
  if ((result = openapi1213SystemSysContactGet (clientHandle, &buffdesc) != OPEN_E_NONE))
  {
    printf("Bad return code trying to get system contact. (result = %d)\n", result);
  }
  else
  {
    printf("System contact retrieved successfully. system contact=\"%s\" size=%d (result = %d)\n",
           (char *) buffdesc.pstart, buffdesc.size, result);
  }

  if (0 == memcmp (buffdesc.pstart, testSysContact, strlen(testSysContact)))
  {
    printf("System Contact Set test -- PASS\n");
  }
  else
  {
    printf("System Contact Set test -- FAIL (New: %s Org: %s)\n", (char *)buffdesc.pstart, testSysContact);
  }

  buffdesc.size = OPEN_RFC1213_PARAM_SIZE;

  if ((result = openapi1213SysObjectIDGet(clientHandle, &buffdesc) != OPEN_E_NONE))
  {
    printf("\nBad return code trying to get system object id. (result = %d)\n", result);
    printf("System object id get test -- FAIL\n\n");
  }
  else
  {
    printf("\nSystem object id retrieved successfully. system_Oid=\"%s\" size=%d (result = %d)\n",
           (char *) buffdesc.pstart, buffdesc.size, result);
    printf("System object id get test -- PASS\n\n");
  }
  free(str);
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("system_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting System API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  rfc1213SystemSanityTest (&clientHandle);

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping System API example application");
        
  (void) openapiClientTearDown(&clientHandle);        
  return 0;
}
