/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  tacacs_example.c
*
* @purpose   TACACS+ Server OpEN APIs Example 
*
* @component OpEN
*
* @comments
*
* @create    04/01/2013
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_tacacs.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the TACACS+ Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printTacacsAppMenu()
{
  printf("Usage:  tacacs_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: Add TACACS+ server: tacacs_example 1 <tacacs-server>\n");
  printf("Test 2: Set TACACS+ server key: tacacs_example 2 <tacacs-server> <key>\n");
  printf("Test 3: Set TACACS+ server priority: tacacs_example 3 <tacacs-server> <priority>\n");
  printf("Test 4: Set TACACS+ server port: tacacs_example 4 <tacacs-server> <port>\n");
  printf("Test 5: Set TACACS+ server timeout: tacacs_example 5 <tacacs-server> <timeout>\n");
  printf("Test 6: Set TACACS+ global key: tacacs_example 6 <tacacs-server global key>\n");
  printf("Test 7: Set TACACS+ server global timeout: tacacs_example 7 <global timeout>\n");
  printf("Test 8: Set TACACS+ server port to default: tacacs_example 8 <tacacs-server>\n");
  printf("Test 9: Set TACACS+ server priority to default: tacacs_example 9 <tacacs-server>\n");
  printf("Test 10: Set TACACS+ server key to default: tacacs_example 10 <tacacs-server>\n");
  printf("Test 11: Delete TACACS+ server: tacacs_example 11 <tacacs-server>\n");
  printf("Test 12: Delete TACACS+ server global key: taccas_example 12\n");
  printf("Test 13: Get TACACS+ server info: tacacs_example 13\n");
  printf("Test 14: tacacs_example OpEN APIs sanity: tacacs_example 14\n");

  return;
}

/*********************************************************************
* @purpose  Add a TACACS+ server.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    tacacsServerAddr @b{(input)}   DNS Name/IP address of the TACACS+ server
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void tacacsServerAdd(openapiClientHandle_t *clientHandle, char *tacacsServerAddr)
{
  open_error_t result;
  open_buffdesc tacacsServer;
  char str[256];
  
  memset(str, 0, sizeof(str));
  strncpy(str, tacacsServerAddr, (sizeof(str) - 1));
  tacacsServer.pstart = str;
  tacacsServer.size = strlen(str)+1;
  
  if ((result = openapiTacacsServerAdd(clientHandle, &tacacsServer)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to add TACACS+ server. (result = %d)\n", result);
  } 
  else
  {   
    printf("TACACS+ server is added successfully. (result = %d)\n", result);
  } 
    
  return;
}

/*********************************************************************
* @purpose  Set secret key to the TACACS+ server.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    tacacsServerAddr @b{(input)}   DNS Name/IP address of the TACACS+ server
* @param    key              @b{(input)}   TACACS+ server key 
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void tacacsServerKeySet(openapiClientHandle_t *clientHandle, char *tacacsServerAddr, char *key)
{
  open_error_t result;
  open_buffdesc tacacsServer;
  open_buffdesc tacacsKey;
  char str[256];
  char str1[256];

  memset(str, 0, sizeof(str));
  strncpy(str, tacacsServerAddr, (sizeof(str) - 1));
  tacacsServer.pstart = str;
  tacacsServer.size = strlen(str)+1;

  memset(str1, 0, sizeof(str1));
  strncpy(str1, key, (sizeof(str1) - 1));
  tacacsKey.pstart = str1;
  tacacsKey.size = strlen(str1)+1;

  if ((result = openapiTacacsServerKeySet(clientHandle, &tacacsServer, &tacacsKey)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set TACACS+ server key. (result = %d)\n", result);
  }
  else
  {
    printf("TACACS+ server key is set successfully. (result = %d)\n", result);
  }

  return;  
}

/*********************************************************************
* @purpose  Set the priority of TACACS+ server.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    tacacsServerAddr @b{(input)}   DNS Name/IP address of the TACACS+ server
* @param    priority         @b{(input)}   TACACS+ server priority
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void tacacsServerPrioritySet(openapiClientHandle_t *clientHandle, char *tacacsServerAddr, uint32_t priority)
{
  open_error_t result;
  open_buffdesc tacacsServer;
  char str[256];

  memset(str, 0, sizeof(str));
  strncpy(str, tacacsServerAddr, (sizeof(str) - 1));
  tacacsServer.pstart = str;
  tacacsServer.size = strlen(str)+1;

  if ((result = openapiTacacsServerPrioritySet(clientHandle, &tacacsServer, priority)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set TACACS+ server priority. (result = %d)\n", result);
  }
  else
  {
    printf("TACACS+ server priority is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Set the TACACS+ server port.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    tacacsServerAddr @b{(input)}   DNS Name/IP address of the TACACS+ server
* @param    port             @b{(input)}   TACACS+ server port
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void tacacsServerPortSet(openapiClientHandle_t *clientHandle, char *tacacsServerAddr, uint32_t port)
{
  open_error_t result;
  open_buffdesc tacacsServer;
  char str[256];

  memset(str, 0, sizeof(str));
  strncpy(str, tacacsServerAddr, (sizeof(str) - 1));
  tacacsServer.pstart = str;
  tacacsServer.size = strlen(str)+1;

  if ((result = openapiTacacsServerPortSet(clientHandle, &tacacsServer, port)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set TACACS+ server port. (result = %d)\n", result);
  }
  else
  {
    printf("TACACS+ server port is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Set the TACACS+ server timeout.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    tacacsServerAddr @b{(input)}   DNS Name/IP address of the TACACS+ server
* @param    timeout          @b{(input)}   TACACS+ server timeout value
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void tacacsServerTimeoutSet(openapiClientHandle_t *clientHandle, char *tacacsServerAddr, uint32_t timeout)
{
  open_error_t result;
  open_buffdesc tacacsServer;
  char str[256];

  memset(str, 0, sizeof(str));
  strncpy(str, tacacsServerAddr, (sizeof(str) - 1));
  tacacsServer.pstart = str;
  tacacsServer.size = strlen(str)+1;

  if ((result = openapiTacacsServerTimeoutSet(clientHandle, &tacacsServer, timeout)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set TACACS+ server timeout. (result = %d)\n", result);
  }
  else
  {
    printf("TACACS+ server timeout is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Set the TACACS+ server global key.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    tacacsGlobalKey  @b{(input)}   TACACS+ global key
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void tacacsServerGlobalKeySet(openapiClientHandle_t *clientHandle, char *tacacsGlobalKey)
{
  open_error_t result;
  open_buffdesc tacacsGblKeyBuffDesc;
  char str[256];

  memset(str, 0, sizeof(str));
  strncpy(str, tacacsGlobalKey, (sizeof(str) - 1));
  tacacsGblKeyBuffDesc.pstart = str;
  tacacsGblKeyBuffDesc.size = strlen(str)+1;
  
  if ((result = openapiTacacsServerGlobalKeySet(clientHandle, &tacacsGblKeyBuffDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set TACACS+ server global key. (result = %d)\n", result);
  }
  else
  { 
    printf("TACACS+ server global key is set successfully. (result = %d)\n", result);
  }
  
  return;
}

/*********************************************************************
* @purpose  Delete TACACS+ server.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    tacacsServerAddr @b{(input)}   DNS Name/IP address of the TACACS+ server
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void tacacsServerDelete(openapiClientHandle_t *clientHandle, char *tacacsServerAddr)
{
  open_error_t result;
  open_buffdesc tacacsServer;
  char str[256];
  
  memset(str, 0, sizeof(str));
  strncpy(str, tacacsServerAddr, (sizeof(str) - 1));
  tacacsServer.pstart = str;
  tacacsServer.size = strlen(str)+1;
  
  if ((result = openapiTacacsServerDelete(clientHandle, &tacacsServer)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete TACACS+ server. (result = %d)\n", result);
  }
  else
  {
    printf("TACACS+ server deleted is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Set TACACS+ server global timeout.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    timeout          @b{(input)}   TACACS+ server global timeout
* 
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void tacacsServerGlobalTimeoutSet(openapiClientHandle_t *clientHandle, uint32_t timeout)
{
  open_error_t result;

  if ((result = openapiTacacsServerGlobalTimeoutSet(clientHandle, timeout)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set TACACS+ server global timeout. (result = %d)\n", result);
  }
  else
  {
    printf("TACACS+ server global timeout is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Set TACACS+ server port to default.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    tacacsServerAddr @b{(input)}   DNS Name/IP address of the TACACS+ server
* 
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void tacacsServerDefaultPortSet(openapiClientHandle_t *clientHandle, char *tacacsServerAddr)
{
  open_error_t result;
  open_buffdesc tacacsSrvBuffDesc;
  char str[256];

  memset(str, 0, sizeof(str));
  strncpy(str, tacacsServerAddr, (sizeof(str) - 1));
  tacacsSrvBuffDesc.pstart = str;
  tacacsSrvBuffDesc.size = strlen(str)+1;

  if ((result = openapiTacacsServerDefaultPortSet(clientHandle, &tacacsSrvBuffDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set TACACS+ server port to default. (result = %d)\n", result);
  }
  else
  {
    printf("TACACS+ server port is set to default successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Set TACACS+ server priority to default.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    tacacsServerAddr @b{(input)}   DNS Name/IP address of the TACACS+ server
* 
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void tacacsServerDefaultPrioritySet(openapiClientHandle_t *clientHandle, char *tacacsServerAddr)
{
  open_error_t result;
  open_buffdesc tacacsSrvBuffDesc;
  char str[256];

  memset(str, 0, sizeof(str));
  strncpy(str, tacacsServerAddr, (sizeof(str) - 1));
  tacacsSrvBuffDesc.pstart = str;
  tacacsSrvBuffDesc.size = strlen(str)+1;

  if ((result = openapiTacacsServerDefaultPrioritySet(clientHandle, &tacacsSrvBuffDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set TACACS+ server priority to default. (result = %d)\n", result);
  }
  else
  {
    printf("TACACS+ server priority is set to default successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Set TACACS+ server key to default.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    tacacsServerAddr @b{(input)}   DNS Name/IP address of the TACACS+ server
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* @notes  This API will set the key of a TACACS+ server to global key
* 
* @end
*********************************************************************/
void tacacsServerDefaultKeySet(openapiClientHandle_t *clientHandle, char *tacacsServerAddr)
{
  open_error_t result;
  open_buffdesc tacacsSrvBuffDesc;
  char str[256];

  memset(str, 0, sizeof(str));
  strncpy(str, tacacsServerAddr, (sizeof(str) - 1));
  tacacsSrvBuffDesc.pstart = str;
  tacacsSrvBuffDesc.size = strlen(str)+1;

  if ((result = openapiTacacsServerDefaultKeySet(clientHandle, &tacacsSrvBuffDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set TACACS+ server key to default. (result = %d)\n", result);
  }
  else
  {
    printf("TACACS+ server key is set to default successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Delete the TACACS+ server global key.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void tacacsServerGlobalKeyDelete(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  if ((result = openapiTacacsServerGlobalKeyDelete(clientHandle)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete TACACS+ server global key. (result = %d)\n", result);
  }
  else
  {
    printf("TACACS+ server global key is deleted successfully. (result = %d)\n", result);
  }

  return;
}


/*********************************************************************
* @purpose  Display TACACS+ servers information.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void tacacsServerInfoGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t tacacsSrvMaxLen = 0;
  uint32_t tacacsServerKeyMaxLen = 0;
  uint32_t maxTacacsServersSupp = 0;
  open_buffdesc tacacsSrvBuffDesc;
  char *tacacsSrv;
  uint16_t port = 0, priority = 0;
  uint32_t timeout = 0, globalTimeout = 0;
  

  if ((result = openapiTacacsServerMaxLengthGet(clientHandle, &tacacsSrvMaxLen)) != OPEN_E_NONE)
  {
    printf(" Bad return code trying to get TACACS+ server maximum length. (result = %d)\n", result);
    return;
  }

  if ((result = openapiTacacsKeyMaxLengthGet(clientHandle, &tacacsServerKeyMaxLen)) != OPEN_E_NONE)
  {
    printf(" Bad return code trying to get TACACS+ server key maximum length. (result = %d)\n", result);
    return;
  }

  if ((tacacsSrv = (char *)malloc(tacacsSrvMaxLen)) == NULL)
  {
    printf("Could not allocate memory. \n");
    return;
  }

  memset(tacacsSrv, 0, tacacsSrvMaxLen);
  tacacsSrvBuffDesc.pstart = tacacsSrv;
  tacacsSrvBuffDesc.size = tacacsSrvMaxLen;

  printf("\n");

  if ((result = openapiTacacsServersMaxSupportedGet(clientHandle, &maxTacacsServersSupp)) != OPEN_E_NONE)
  {
    printf(" Bad return code trying to get maximum TACACS+ servers supported. (result = %d)\n", result);
    free(tacacsSrv);
    return;
  }
  else
  {
    printf("Maximum TACACS+ Servers Supported: %d\n", maxTacacsServersSupp);
  }

  if ((result = openapiTacacsServerGlobalTimeoutGet(clientHandle, &globalTimeout)) != OPEN_E_NONE)
  {
    printf(" Bad return code trying to get TACACS+ server global timeout. (result = %d)\n", result);
  }
  else
  {
    printf("TACACS+ server global timeout: %d\n", globalTimeout);
  }

  printf("\n");
  printf("TACACS+ server details: \n");
  printf("-----------------------\n");

  if ((result = openapiTacacsServerFirstGet(clientHandle, &tacacsSrvBuffDesc)) != OPEN_E_NONE)
  {
    printf(" Bad return code trying to get first TACACS+ server. (result = %d)\n", result);
    return;
  }

  do
  {
    printf("TACACS+ server Address: %s\n", (char *) tacacsSrvBuffDesc.pstart);

    printf("Port: ");
    if ((result = openapiTacacsServerPortGet(clientHandle, &tacacsSrvBuffDesc, &port))  != OPEN_E_NONE)
    {
      printf("Error getting TACACS+ server port. (result = %d) \n", result);
    }
    else
    {
      printf("%d\n", port);
    }

    printf("Priority: ");
    if ((result = openapiTacacsServerPriorityGet(clientHandle, &tacacsSrvBuffDesc, &priority))  != OPEN_E_NONE)
    {
      printf("Error getting TACACS+ server priority. (result = %d) \n", result);
    }
    else
    {
      printf("%d\n", priority);
    }

    printf("Timeout: ");
    if ((result = openapiTacacsServerTimeoutGet(clientHandle, &tacacsSrvBuffDesc, &timeout))  != OPEN_E_NONE)
    {
      printf("Error getting TACACS+ server timeout. (result = %d) \n", result);
    }
    else
    {
      printf("%d\n", timeout);
    }
    tacacsSrvBuffDesc.size = tacacsSrvMaxLen;
    printf("\n");
  }while(openapiTacacsServerNextGet(clientHandle, &tacacsSrvBuffDesc, &tacacsSrvBuffDesc) == OPEN_E_NONE);
  free(tacacsSrv);
  return;
}

/*********************************************************************
* @purpose  TACACS+ OpEN API Sanity.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void tacacsOpENAPIsSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t tacacsSrvMaxLen = 0;
  uint32_t tacacsServerKeyMaxLen = 0;
  uint32_t maxTacacsServersSupp = 0;
  open_buffdesc tacacsSrvBuffDesc;
  open_buffdesc tacacsSrvKeyBuffDesc;
  char *tacacsSrv, *tacacsSrvKey;
  uint16_t port = 0, priority = 0;
  uint32_t timeout = 0, globalTimeout = 0;


  if ((result = openapiTacacsServerMaxLengthGet(clientHandle, &tacacsSrvMaxLen)) != OPEN_E_NONE)
  {
    printf(" Bad return code trying to get TACACS+ server maximum length. (result = %d)\n", result);
    return;
  }

  if ((result = openapiTacacsKeyMaxLengthGet(clientHandle, &tacacsServerKeyMaxLen)) != OPEN_E_NONE)
  {
    printf(" Bad return code trying to get TACACS+ server key maximum length. (result = %d)\n", result);
    return;
  }

  if ((tacacsSrv = (char *)malloc(tacacsSrvMaxLen)) == NULL)
  {
    printf("Could not allocate memory. \n");
    return;
  }

  if ((tacacsSrvKey = (char *)malloc(tacacsServerKeyMaxLen)) == NULL)
  {
    printf("Could not allocate memory. \n");
    free(tacacsSrv);
    return;
  }

  memset(tacacsSrv, 0, tacacsSrvMaxLen);
  tacacsSrvBuffDesc.pstart = tacacsSrv;
  tacacsSrvBuffDesc.size = tacacsSrvMaxLen;

  memset(tacacsSrvKey, 0, tacacsServerKeyMaxLen);
  tacacsSrvKeyBuffDesc.pstart = tacacsSrvKey;
  tacacsSrvKeyBuffDesc.size = tacacsServerKeyMaxLen;

  /* openapiTacacsServerMaxLengthGet() */
  printf("\nTesting openapiTacacsServerMaxLengthGet():\n");

  result = openapiTacacsServerMaxLengthGet(NULL, &tacacsSrvMaxLen);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiTacacsServerMaxLengthGet(clientHandle, NULL);
  printf("NULL parameter to TACACS+ server maximum length. (result = %d)\n", result);

  printf("openapiTacacsServerMaxLengthGet() sanity successful. \n");

  /* openapiTacacsKeyMaxLengthGet() */
  printf("\nTesting openapiTacacsKeyMaxLengthGet():\n");

  result = openapiTacacsKeyMaxLengthGet(NULL, &tacacsServerKeyMaxLen);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiTacacsKeyMaxLengthGet(clientHandle, NULL);
  printf("NULL parameter to TACACS+ server key maximum length. (result = %d)\n", result);

  printf("openapiTacacsKeyMaxLengthGet() sanity successful. \n");

  /* openapiTacacsServersMaxSupportedGet() */ 
  printf("\nTesting openapiTacacsServersMaxSupportedGet():\n");

  result = openapiTacacsServersMaxSupportedGet(NULL, &maxTacacsServersSupp);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiTacacsServersMaxSupportedGet(clientHandle, NULL);
  printf("NULL parameter to maximum TACACS+ servers supported. (result = %d)\n", result);

  printf("openapiTacacsServersMaxSupportedGet() sanity successful. \n");

  /* openapiTacacsServerAdd() */ 
  printf("\nTesting openapiTacacsServerAdd():\n");

  result = openapiTacacsServerAdd(NULL, &tacacsSrvBuffDesc); 
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiTacacsServerAdd(clientHandle, NULL); 
  printf("NULL buff descriptor to TACACS+ server. (result = %d)\n", result);

  printf("openapiTacacsServerAdd() sanity successful. \n");

 /* openapiTacacsServerFirstGet() */
  printf("\nTesting openapiTacacsServerFirstGet():\n");

  openapiTacacsServerFirstGet(NULL, &tacacsSrvBuffDesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  openapiTacacsServerFirstGet(clientHandle, NULL);
  printf("NULL buff descriptor to TACACS+ server. (result = %d)\n", result);

  printf("openapiTacacsServerFirstGet() sanity successful. \n");

  /* openapiTacacsServerNextGet() */
  printf("\nTesting openapiTacacsServerNextGet():\n");

  openapiTacacsServerNextGet(NULL, &tacacsSrvBuffDesc, &tacacsSrvBuffDesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  openapiTacacsServerNextGet(clientHandle, NULL, &tacacsSrvBuffDesc);
  printf("NULL buff descriptor to TACACS+ server. (result = %d)\n", result);

  openapiTacacsServerNextGet(clientHandle, &tacacsSrvBuffDesc, NULL);
  printf("NULL buff descriptor to next TACACS+ server. (result = %d)\n", result);

  printf("openapiTacacsServerNextGet() sanity successful. \n");

  /* openapiTacacsServerDelete() */
  printf("\nTesting openapiTacacsServerDelete():\n");

  result = openapiTacacsServerDelete(NULL, &tacacsSrvBuffDesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiTacacsServerDelete(clientHandle, NULL);
  printf("NULL buff descriptor to TACACS+ server. (result = %d)\n", result);

  printf("openapiTacacsServerDelete() sanity successful. \n");

  /* openapiTacacsServerKeySet() */
  printf("\nTesting openapiTacacsServerKeySet():\n");

  result = openapiTacacsServerKeySet(NULL, &tacacsSrvBuffDesc, &tacacsSrvKeyBuffDesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiTacacsServerKeySet(clientHandle, NULL, &tacacsSrvKeyBuffDesc);
  printf("NULL buff descriptor to TACACS+ server. (result = %d)\n", result);

  result = openapiTacacsServerKeySet(clientHandle, &tacacsSrvBuffDesc, NULL);
  printf("NULL buff descriptor to TACACS+ server key. (result = %d)\n", result);

  printf("openapiTacacsServerKeySet() sanity successful. \n");

  /* openapiTacacsServerPortSet() */
  printf("\nTesting openapiTacacsServerPortSet():\n");

  result = openapiTacacsServerPortSet(NULL, &tacacsSrvBuffDesc, port);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiTacacsServerPortSet(clientHandle, NULL, port);
  printf("NULL buff descriptor to TACACS+ server. (result = %d)\n", result);

  printf("openapiTacacsServerPortSet() sanity successful. \n");

  /* openapiTacacsServerPortGet() */
  printf("\nTesting openapiTacacsServerPortGet():\n");

  result = openapiTacacsServerPortGet(NULL, &tacacsSrvBuffDesc, &port); 
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiTacacsServerPortGet(clientHandle, NULL, &port); 
  printf("NULL buff descriptor to TACACS+ server. (result = %d)\n", result);
  
  result = openapiTacacsServerPortGet(clientHandle, &tacacsSrvBuffDesc, NULL); 
  printf("NULL parameter to TACACS+ server port. (result = %d)\n", result);

  printf("openapiTacacsServerPortGet() sanity successful. \n");

  /* openapiTacacsServerPrioritySet() */
  printf("\nTesting openapiTacacsServerPrioritySet():\n");

  result = openapiTacacsServerPrioritySet(NULL, &tacacsSrvBuffDesc, priority);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiTacacsServerPrioritySet(clientHandle, NULL, priority);
  printf("NULL buff descriptor to TACACS+ server. (result = %d)\n", result);

  printf("openapiTacacsServerPrioritySet() sanity successful. \n");

  /* openapiTacacsServerPriorityGet() */
  printf("\nTesting openapiTacacsServerPriorityGet():\n");

  result = openapiTacacsServerPriorityGet(NULL, &tacacsSrvBuffDesc, &priority);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiTacacsServerPriorityGet(clientHandle, NULL, &priority);
  printf("NULL buff descriptor to TACACS+ server. (result = %d)\n", result);

  result = openapiTacacsServerPriorityGet(clientHandle, &tacacsSrvBuffDesc, NULL);
  printf("NULL parameter to TACACS+ server priority. (result = %d)\n", result);

  printf("openapiTacacsServerPriorityGet() sanity successful. \n");

  /* openapiTacacsServerTimeoutSet() */
  printf("\nTesting openapiTacacsServerTimeoutSet():\n");

  result = openapiTacacsServerTimeoutSet(NULL, &tacacsSrvBuffDesc, timeout);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiTacacsServerTimeoutSet(clientHandle, NULL, timeout); 
  printf("NULL buff descriptor to TACACS+ server. (result = %d)\n", result);

  result = openapiTacacsServerTimeoutSet(clientHandle, &tacacsSrvBuffDesc, -1); 
  printf("Invalid TACACS+ server timeout value. (result = %d)\n", result);

  result = openapiTacacsServerTimeoutSet(clientHandle, &tacacsSrvBuffDesc, 32); 
  printf("Invalid TACACS+ server timeout value. (result = %d)\n", result);

  printf("openapiTacacsServerTimeoutSet() sanity successful. \n"); 

  /* openapiTacacsServerTimeoutGet() */
  printf("\nTesting openapiTacacsServerTimeoutGet():\n");

  result = openapiTacacsServerTimeoutGet(NULL, &tacacsSrvBuffDesc, &timeout);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiTacacsServerTimeoutGet(clientHandle, NULL, &timeout);
  printf("NULL buff descriptor to TACACS+ server. (result = %d)\n", result);

  result = openapiTacacsServerTimeoutGet(clientHandle, &tacacsSrvBuffDesc, NULL);
  printf("NULL parameter to TACACS+ server timeout. (result = %d)\n", result);

  printf("openapiTacacsServerTimeoutGet() sanity successful. \n");

  /* openapiTacacsServerGlobalKeySet() */
  printf("\nTesting openapiTacacsServerGlobalKeySet():\n");
  
  result = openapiTacacsServerGlobalKeySet(NULL, &tacacsSrvKeyBuffDesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiTacacsServerGlobalKeySet(clientHandle, NULL);
  printf("NULL buff descriptor to TACACS+ server global key. (result = %d)\n", result);

  printf("openapiTacacsServerGlobalKeySet() sanity successful. \n");

  /* openapiTacacsServerGlobalTimeoutGet() */
  printf("\nTesting openapiTacacsServerGlobalTimeoutGet():\n");

  result = openapiTacacsServerGlobalTimeoutGet(NULL, &globalTimeout); 
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiTacacsServerGlobalTimeoutGet(clientHandle, NULL); 
  printf("NULL parameter to TACACS+ server global timeout. (result = %d)\n", result);

  printf("openapiTacacsServerGlobalTimeoutGet() sanity successful. \n");

  /* openapiTacacsServerDefaultKeySet() */
  printf("\nTesting openapiTacacsServerDefaultKeySet():\n");

  result = openapiTacacsServerDefaultKeySet(NULL, &tacacsSrvBuffDesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiTacacsServerDefaultKeySet(clientHandle, NULL);
  printf("NULL buff descriptor to TACACS+ server. (result = %d)\n", result);

  printf("openapiTacacsServerDefaultKeySet() sanity successful. \n");

  /* openapiTacacsServerDefaultPortSet() */
  printf("\nTesting openapiTacacsServerDefaultPortSet():\n");
  result = openapiTacacsServerDefaultPortSet(NULL, &tacacsSrvBuffDesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiTacacsServerDefaultPortSet(clientHandle, NULL);
  printf("NULL buff descriptor to TACACS+ server. (result = %d)\n", result);

  printf("openapiTacacsServerDefaultPortSet() sanity successful. \n");

  /* openapiTacacsServerDefaultPrioritySet() */
  printf("\nTesting openapiTacacsServerDefaultPrioritySet():\n");

  result = openapiTacacsServerDefaultPrioritySet(NULL, &tacacsSrvBuffDesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiTacacsServerDefaultPrioritySet(clientHandle, NULL);
  printf("NULL buff descriptor to TACACS+ server. (result = %d)\n", result);

  printf("openapiTacacsServerDefaultPrioritySet() sanity successful. \n");

  /* openapiTacacsServerGlobalKeyDelete() */
  printf("\nTesting openapiTacacsServerGlobalKeyDelete():\n");

  result = openapiTacacsServerGlobalKeyDelete(NULL);  
  printf("NULL Client Handle. (result = %d)\n", result);

  printf("openapiTacacsServerGlobalKeyDelete() sanity successful. \n");

  /* openapiTacacsServerGlobalTimeoutSet() */
  printf("\nTesting openapiTacacsServerGlobalTimeoutSet():\n");

  result = openapiTacacsServerGlobalTimeoutSet(NULL, timeout);  
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiTacacsServerGlobalTimeoutSet(clientHandle, -1);  
  printf("Invalid global timeout value. (result = %d)\n", result);

  result = openapiTacacsServerGlobalTimeoutSet(clientHandle, 40);  
  printf("Invalid global timeout value. (result = %d)\n", result);

  printf("openapiTacacsServerGlobalTimeoutSet() sanity successful. \n");

  
  
  free(tacacsSrv);
  free(tacacsSrvKey);

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum, arg1;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  
  if (argc < 2)
  {
    printTacacsAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("tacacs_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting TACACS+ API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 3)
      {
        printTacacsAppMenu();
        exit(1);
      }
      tacacsServerAdd(&clientHandle, argv[2]);
      break; 
    case 2:
      if (argc != 4)
      {
        printTacacsAppMenu();
        exit(1);
      }
      tacacsServerKeySet(&clientHandle, argv[2], argv[3]);
      break;
    case 3:
      if (argc != 4)
      {
        printTacacsAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      tacacsServerPrioritySet(&clientHandle, argv[2], arg1);
      break;
    case 4:
      if (argc != 4)
      {
        printTacacsAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      tacacsServerPortSet(&clientHandle, argv[2], arg1);
      break;
    case 5:
      if (argc != 4)
      {
        printTacacsAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      tacacsServerTimeoutSet(&clientHandle, argv[2], arg1);
      break;     
    case 6:
      if (argc != 3)
      {
        printTacacsAppMenu();
        exit(1);
      }
      tacacsServerGlobalKeySet(&clientHandle, argv[2]);
      break;
    case 7:
      if (argc != 3)
      {
        printTacacsAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      tacacsServerGlobalTimeoutSet(&clientHandle, arg1);
      break;
    case 8:
      if (argc != 3)
      {
        printTacacsAppMenu();
        exit(1);
      }
      tacacsServerDefaultPortSet(&clientHandle, argv[2]);
      break;
    case 9:
      if (argc != 3)
      {
        printTacacsAppMenu();
        exit(1);
      }
      tacacsServerDefaultPrioritySet(&clientHandle, argv[2]);
      break;
    case 10:
      if (argc != 3)
      {
        printTacacsAppMenu();
        exit(1);
      }
      tacacsServerDefaultKeySet(&clientHandle, argv[2]);
      break;
    case 11:
      if (argc != 3)
      {
        printTacacsAppMenu();
        exit(1);
      }
      tacacsServerDelete(&clientHandle, argv[2]);
      break;
    case 12:
      if (argc != 2)
      {
        printTacacsAppMenu();
        exit(1);
      }
      tacacsServerGlobalKeyDelete(&clientHandle);
      break;
    case 13:
      if (argc != 2)
      {
        printTacacsAppMenu();
        exit(1);
      }
      tacacsServerInfoGet(&clientHandle);
      break;
    case 14:
      if (argc != 2)
      {
        printTacacsAppMenu();
        exit(1);
      }
      tacacsOpENAPIsSanity(&clientHandle);
      break;
    default:
      printTacacsAppMenu();
      break;
  }
      
  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping TACACS+ API example application");
      
  (void) openapiClientTearDown(&clientHandle);
  return 0;
} 
