/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename tcam_policy.c
*
* @purpose TCAM Policy Hardware Usage API example
*
* @detail  Example application that uses the OpEN API to report
*          the current TCAM utilization for ACLs.
*
* @component TCAM API Example
*
* @comments none
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_tcam.h"

/* For stackable platforms this value represents the maximum number
** of switches in the stack. On some NOS platforms there is only one switch.
*/
#define MAX_STACK_UNITS 16

/* Some switches use multiple ASICs. This program supports up to 8 
** ASICs per switch.
*/
#define MAX_ASIC_PER_UNIT 8

typedef struct {
   uint32_t hardware_group;
   uint32_t in_use_rules;
   uint32_t total_rules;
} acl_policy_usage_t;
   

/* ACL policy capacity table. Note that each unit and ASIC can have different number of 
** rules dedicated to the ACLs. 
*/
typedef struct {
   /* ACL Policy name reported by the switch.
   */
   char policy_name[OPEN_TCAM_POLICY_NAME_SIZE];

   /* ACL Usage information.
   */
   acl_policy_usage_t usage[MAX_STACK_UNITS][MAX_ASIC_PER_UNIT];

   /* Number of ACL policies of this type on the switch.
   */
   uint32_t num_policies;
} acl_policy_t;


static acl_policy_t acl_policy [] = {
   {"ACL/Control-Plane/IPv4/Egress"},
   {"ACL/Control-Plane/IPv6/Egress"},
   {"ACL/Port-Based/IPv4/Ingress"},
   {"ACL/Port-Based/IPv6/Ingress"},
   {"ACL/VLAN-Based/IPv4/Ingress"},
   {"ACL/VLAN-Based/IPv6/Ingress"},
   {"ACL/Port-Based/IPv4/Egress"},
   {"ACL/Port-Based/IPv6/Egress"},
   {"ACL/VLAN-Based/IPv4/Egress"},
   {"ACL/VLAN-Based/IPv6/Egress"}
};

uint32_t acl_list_size = sizeof(acl_policy) / sizeof (acl_policy_t);

/* Number of policies detected on this unit 
*/
uint32_t  num_policies_on_unit[MAX_STACK_UNITS]; 

/* Number of policies detected on this ASIC. 
*/
uint32_t  num_policies_on_asic[MAX_STACK_UNITS][MAX_ASIC_PER_UNIT];
 
/*******************************************************************
*
* @brief  Display TCAM ACL utilization.
*
*
*********************************************************************/
void openTcamACLShow(void)
{
  int unit, asic, acl;
 
  for (unit = 0; unit < MAX_STACK_UNITS; unit++)
  {
    if (num_policies_on_unit[unit] == 0)
    {
      continue;
    }
    printf ("Unit:%d\n", unit);
    for (asic = 0; asic < MAX_ASIC_PER_UNIT; asic++)
    {
      if (num_policies_on_asic[unit][asic] == 0)
      {
        continue;
      }
      printf ("  ASIC:%d\n", asic);
      for (acl = 0; acl < acl_list_size; acl++)
      {
        printf ("    %s", acl_policy[acl].policy_name); 
        if (acl_policy[acl].num_policies == 0)
        {
          printf (" - Not Installed");
        } else
        {
          printf (" - Group:%d", acl_policy[acl].usage[unit][asic].hardware_group);
          printf (" Rules:%d", acl_policy[acl].usage[unit][asic].in_use_rules);
          printf (" Capacity:%d", acl_policy[acl].usage[unit][asic].total_rules);
          printf (" Utilization-Rate:%d%%", 
            (acl_policy[acl].usage[unit][asic].in_use_rules*100)/
                                    acl_policy[acl].usage[unit][asic].total_rules);
        }
        printf ("\n");
      }
    }
  }
   
}

/*******************************************************************
*
* @brief  Display TCAM utilization.
*
* @returns  0: Success
* @returns  1: Failure
*
*********************************************************************/
int openTcamPolicyShow(openapiClientHandle_t *clientHandle)
{

  open_tcamHardwarePolicyUsage_t policy_usage;
  uint32_t unit = 0;
  uint32_t asic_id = 0;
  uint32_t policy_id = 0;
  open_error_t rc;
  uint32_t i;

  do 
  {
    rc = openapiTcamHardwarePolicyUsageGetNext(clientHandle, &unit, &asic_id, 
                                               &policy_id, &policy_usage);
    if (rc != OPEN_E_NONE)
    {
      break;
    }
    if (unit >= MAX_STACK_UNITS)
    {
      printf ("Unit %d is out of range.\n", unit);
      continue;
    }
    if (asic_id >= MAX_ASIC_PER_UNIT)
    {
      printf ("ASIC %d on unit %d is out of range.\n", asic_id, unit);
      continue;
    }
    num_policies_on_unit[unit]++;
    num_policies_on_asic[unit][asic_id]++;

    /* Check whether this policy matches any of the ACL types.
    ** Note that multiple policies may match the same ACL type, but all these 
    ** policies use the same hardware group, so we just record the last
    ** matching policy.
    */
    for (i = 0; i < acl_list_size; i++)
    {
      if (0 == strncmp ((char *) policy_usage.policy_name, acl_policy[i].policy_name, OPEN_TCAM_POLICY_NAME_SIZE))
      {
        acl_policy[i].num_policies++;
        acl_policy[i].usage[unit][asic_id].in_use_rules = policy_usage.in_use_rules;
        acl_policy[i].usage[unit][asic_id].total_rules = policy_usage.total_rules;
        acl_policy[i].usage[unit][asic_id].hardware_group = policy_usage.hardware_group;
      }
    }
/* Uncomment this code to display all policies on the switch.
*/
#if 0
    {
     static  uint32_t last_unit = ~0, last_asic_id = ~0;

      if (last_unit != unit)
      {
        last_unit = unit;
        printf ("Unit:%d\n", last_unit);
      }
      if (last_asic_id != asic_id)
      {
        last_asic_id = asic_id;
        printf ("  ASIC:%d\n", last_asic_id);
      }
      printf("       Policy:%d Name:%s Stage:%d Rules:%d Group:%d HW-Rules:%d HW-Capacity:%d\n",
              policy_id, 
              policy_usage.policy_name,
              policy_usage.hardware_stage,
              policy_usage.num_policy_rules,
              policy_usage.hardware_group,
              policy_usage.in_use_rules,
              policy_usage.total_rules
              );
    }
#endif

  }
  while (1); 
  printf("\n\n");

  return 0;
}

/*******************************************************************
*
* @brief  Example TCAM API: This is the main function that will demonstrate 
*         TCAM OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure
*
*********************************************************************/
int
main (int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  
  l7proc_crashlog_register ();

  /* Register with OpEN */
  if ((result =
       openapiClientRegister ("tcam_policy", &clientHandle)) != OPEN_E_NONE)
  {
    printf ("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit (2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck (&clientHandle) != OPEN_E_NONE)
  {
    sleep (1);
  }

  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0, "Starting TCAM Policy application");

  printf ("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof (switch_os_revision_string);
  if (openapiNetworkOSVersionGet (&clientHandle, &switch_os_revision) == OPEN_E_NONE)
  {
    printf ("Network OS version = %s\n", switch_os_revision_string);
  }
  else
  {
    printf ("Network OS version retrieve error\n");
  }


  printf ("\n");

  (void) openTcamPolicyShow(&clientHandle);

  openTcamACLShow ();

  return 0;
}
