/*********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  timezone_example.c
*
* @purpose   OpEN  example.
*
* @component OpEN
*
* @create    01/03/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_timezone.h"

#define OPEN_API_BUFFER_SIZE 128
/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the  Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Convert hh:mm string to int values: %s 1 <bufd>\n", name);
  printf("Test 2: Convert minute offset into hour and minutes: %s 2 <minutes>\n", name);
  printf("Test 3: Convert month name to number value: %s 3 <name>\n", name);
  printf("Test 4: Set the time zone acronym parameter: %s 4 <acronym>\n", name);
  printf("Test 5: Get the time zone acronym parameter: %s 5 \n", name);
  printf("Test 6: Get non-recurring summertime parameters: %s 6 <parameter>\n", name);
  printf("Test 7: Get recurring summertime parameters: %s 7 <parameter>\n", name);
  printf("Test 8: Set recurring summertime parameters: %s 8 <parameter><bufd>\n", name);
  printf("Test 9: Set non-recurring summertime parameters: %s 9 <parameter><bufd>\n", name);
  printf("Test 10: Get the summer-time mode: %s 10 \n", name);
  printf("Test 11: Set the summer-time mode: %s 11 <mode>\n", name);
  printf("Test 12: Gets the timezone acronym string: %s 12 <zoneName>\n", name);
  printf("Test 13: Get the current synchronized UTC time since the Unix Epoch: %s 13 <>\n", name);
  printf("Test 14: Get the current UTC time since the Unix Epoch: %s 14 <unit>\n", name);
  printf("Test 15: know whether the summer-time is in effect or not: %s 15 <>\n", name);
  printf("Test 16: Adjusts current time for timezone and summer time: %s 16 <year><month><day><hour><minutes><seconds>\n", name);
  printf("Test 17: Converts the given Raw time to UTC and then adds the timezone and summer time offsets if configured: %s 17 <rawTime>\n", name);
  printf("Test 18: Run API sanity checks: %s 18 \n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t hour;
  uint32_t seconds;
  uint32_t minutes;
  uint32_t year;
  uint32_t number;
  uint32_t pMin;
  OPEN_SUMMER_TIME_MODE_t mode;
  uint32_t pHour;
  uint32_t month;
  OPEN_ST_RECURRING_PARAM_t recurringParameter;
  OPEN_ST_NON_RECURRING_PARAM_t nonRecurringParameter;
  uint32_t day;
  char buffer[128];
  open_buffdesc bufd;
  open_buffdesc zeroLenBuf;
  open_buffdesc badBufdescPointer;

  badBufdescPointer.pstart = (void *) NULL;
  badBufdescPointer.size = sizeof(buffer);
  bufd.pstart = buffer;
  bufd.size = sizeof(buffer);
  zeroLenBuf.pstart = buffer;
  zeroLenBuf.size = 0;
  recurringParameter = OPEN_ST_RECURRING_PARAM_START_DAY;
  nonRecurringParameter = OPEN_ST_NON_RECURRING_PARAM_START_DAY;

  printf("Testing  OpEN APIs sanity:\n");

  printf("Testing openapiTimeConvert():\n");
  result = openapiTimeConvert(NULL, &bufd, &pHour, &pMin);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiTimeConvert(clientHandle, NULL, &pHour, &pMin);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiTimeConvert(clientHandle, &badBufdescPointer, &pHour, &pMin);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiTimeConvert(clientHandle, &zeroLenBuf, &pHour, &pMin);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);
  result = openapiTimeConvert(clientHandle, &bufd, NULL, &pMin);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiTimeConvert(clientHandle, &bufd, &pHour, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiMinutesOffsetConvert():\n");
  result = openapiMinutesOffsetConvert(NULL, minutes, &pHour, &pMin);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMinutesOffsetConvert(clientHandle, minutes, NULL, &pMin);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiMinutesOffsetConvert(clientHandle, minutes, &pHour, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiWeekNumberGet():\n");
  result = openapiWeekNumberGet(NULL, &bufd, &number);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiWeekNumberGet(clientHandle, NULL, &number);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiWeekNumberGet(clientHandle, &badBufdescPointer, &number);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiWeekNumberGet(clientHandle, &zeroLenBuf, &number);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiWeekNumberGet(clientHandle, &bufd, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapixTimeZoneAcronymSet():\n");
  result = openapiTimeZoneAcronymSet(NULL, &bufd);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiTimeZoneAcronymSet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);
  result = openapiTimeZoneAcronymSet(clientHandle, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 2:(result = %d)\n", result);
  result = openapiTimeZoneAcronymSet(clientHandle, &zeroLenBuf);
  printf("buffdesc 0 length size argument 2:(result = %d)\n", result);

  printf("Testing openapiTimeZoneAcronymGet():\n");
  result = openapiTimeZoneAcronymGet(NULL, &bufd);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiTimeZoneAcronymGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);
  result = openapiTimeZoneAcronymGet(clientHandle, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 2:(result = %d)\n", result);
  result = openapiTimeZoneAcronymGet(clientHandle, &zeroLenBuf);
  printf("buffdesc 0 length size argument 2:(result = %d)\n", result);

  printf("Testing openapiStNonRecurringGet():\n");
  result = openapiStNonRecurringGet(NULL, nonRecurringParameter, &bufd);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiStNonRecurringGet(clientHandle, nonRecurringParameter, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiStNonRecurringGet(clientHandle, nonRecurringParameter, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiStNonRecurringGet(clientHandle, nonRecurringParameter, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);

  printf("Testing openapiStRecurringGet():\n");
  result = openapiStRecurringGet(NULL, recurringParameter, &bufd);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiStRecurringGet(clientHandle, recurringParameter, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiStRecurringGet(clientHandle, recurringParameter, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiStRecurringGet(clientHandle, recurringParameter, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);

  printf("Testing openapiStRecurringSet():\n");
  result = openapiStRecurringSet(NULL, recurringParameter, &bufd);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiStRecurringSet(clientHandle, recurringParameter, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiStRecurringSet(clientHandle, recurringParameter, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiStRecurringSet(clientHandle, recurringParameter, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);

  printf("Testing openapiStNonRecurringSet():\n");
  result = openapiStNonRecurringSet(NULL, nonRecurringParameter, &bufd);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiStNonRecurringSet(clientHandle, nonRecurringParameter, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiStNonRecurringSet(clientHandle, nonRecurringParameter, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiStNonRecurringSet(clientHandle, nonRecurringParameter, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);

  printf("Testing openapiStModeGet():\n");
  result = openapiStModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiStModeGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiSystemTimeZoneAcronymGet():\n");
  result = openapiSystemTimeZoneAcronymGet(NULL, &bufd);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiSystemTimeZoneAcronymGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);
  result = openapiSystemTimeZoneAcronymGet(clientHandle, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 2:(result = %d)\n", result);
  result = openapiSystemTimeZoneAcronymGet(clientHandle, &zeroLenBuf);
  printf("buffdesc 0 length size argument 2:(result = %d)\n", result);

  printf("Testing openapiUTCTimeGet():\n");
  result = openapiUTCTimeGet(clientHandle , NULL);
  printf("NULL argument 1:(result = %d)\n", result);

  printf("Testing openapiAdjustedTimeGet():\n");
  result = openapiAdjustedTimeGet(NULL, &year, &month, &day, &hour, &minutes, &seconds);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAdjustedTimeGet(clientHandle, NULL, &month, &day, &hour, &minutes, &seconds);
  printf("NULL argument 7:(result = %d)\n", result);
  result = openapiAdjustedTimeGet(clientHandle, &year, NULL, &day, &hour, &minutes, &seconds);
  printf("NULL argument 7:(result = %d)\n", result);
  result = openapiAdjustedTimeGet(clientHandle, &year, &month, NULL, &hour, &minutes, &seconds);
  printf("NULL argument 7:(result = %d)\n", result);
  result = openapiAdjustedTimeGet(clientHandle, &year, &month, &day, NULL, &minutes, &seconds);
  printf("NULL argument 7:(result = %d)\n", result);
  result = openapiAdjustedTimeGet(clientHandle, &year, &month, &day, &hour, NULL, &seconds);
  printf("NULL argument 7:(result = %d)\n", result);
  result = openapiAdjustedTimeGet(clientHandle, &year, &month, &day, &hour, &minutes, NULL);
  printf("NULL argument 7:(result = %d)\n", result);

  return;
}

/*********************************************************************
* @purpose  Convert hh:mm string to int values.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    bufd       @b{(input)}  ptr to buffer discription to hold time string
* @param    pHour       @b{(output)}  pointer to Hour
* @param    pMin       @b{(output)}  pointer to minutes

*
* @returns  none
*
* @end
*********************************************************************/
void timeConvert(openapiClientHandle_t *client_handle, open_buffdesc *bufd, uint32_t *pHour, uint32_t *pMin)
{
  open_error_t result;

  if ((result = openapiTimeConvert(client_handle, bufd, pHour, pMin)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert hh:mm string to int values. (result = %d)\n", result);
  }
  else
  {
    printf("Converted Time into Hour:%d Minutes :%d\n",*pHour,*pMin);
  }
  return;
}


/*********************************************************************
* @purpose  Convert minute offset into hour and minutes
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    minutes       @b{(input)}  Holds minutes
* @param    pHour       @b{(output)}  pointer to Hour
* @param    pMin       @b{(output)}  pointer to minutes

*
* @returns  none
*
* @end
*********************************************************************/
void minutesOffsetConvert(openapiClientHandle_t *client_handle, int32_t minutes, uint32_t *pHour, uint32_t *pMin)
{
  open_error_t result;

  if ((result = openapiMinutesOffsetConvert(client_handle, minutes, pHour, pMin)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert minute offset into hour and minutes (result = %d)\n", result);
  }
  else
  {
    printf("Converted minutes offset to  Hour:%d Minutes :%d\n",*pHour,*pMin);
  }
  return;
}


/*********************************************************************
* @purpose  Convert month name to number value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    name       @b{(input)}  character buffer with week name string
* @param    number       @b{(output)}  pointer to week sequence number

*
* @returns  none
*
* @end
*********************************************************************/
void weekNumberGet(openapiClientHandle_t *client_handle, open_buffdesc *name, uint32_t *number)
{
  open_error_t result;
  if ((result = openapiWeekNumberGet(client_handle, name, number)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert month name to number value (result = %d)\n", result);
  }
  else
  {
    printf("Converted month name to calender weeknumber CW :%d\n",*number);
  }
  return;
}


/*********************************************************************
* @purpose  Set the time zone acronym parameter
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    acronym       @b{(input)}  minutes in timezone structure.

*
* @returns  none
*
* @end
*********************************************************************/
void TimeZoneAcronymSet(openapiClientHandle_t *client_handle, open_buffdesc *acronym)
{
  open_error_t result;

  if ((result = openapiTimeZoneAcronymSet(client_handle, acronym)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the time zone acronym parameter(result = %d)\n", result);
  }
  else
  {
    printf("TIme Zone acronym is set %s \n",(char*)acronym->pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Get the time zone acronym parameter
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    acronym       @b{(output)}  minutes in timezone structure.

*
* @returns  none
*
* @end
*********************************************************************/
void TimeZoneAcronymGet(openapiClientHandle_t *client_handle, open_buffdesc *acronym)
{
  open_error_t result;

  if ((result = openapiTimeZoneAcronymGet(client_handle, acronym)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the time zone acronym parameter(result = %d)\n", result);
  }
  else
  {
    printf("Get the TImeZone Acronym :%s \n",(char*)acronym->pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Get non-recurring summertime parameters.
*
* @param    client_handle   @b{(input)} client handle from registration API
* @param    parameter       @b{(input)} summer-time parameter
* @param    bufd            @b{(output)} summer-time parameter value to be returned

*
* @returns  none
*
* @end
*********************************************************************/
void StNonRecurringGet(openapiClientHandle_t *client_handle, OPEN_ST_NON_RECURRING_PARAM_t parameter, open_buffdesc *bufd)
{
  open_error_t result;

  if ((result = openapiStNonRecurringGet(client_handle, parameter, bufd)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get non-recurring summertime parameters. (result = %d)\n", result);
  }
  else
  {
     printf("Get the non-recurring summertime :%s\n",(char*)bufd->pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Get non-recurring summertime parameters.
*
* @param    client_handle     @b{(input)} client handle from registration API
* @param    parameter         @b{(input)} summer-time parameter 
* @param    bufd              @b{(output)} summer-time parameter value to be returned

*
* @returns  none
*
* @end
*********************************************************************/
void StRecurringGet(openapiClientHandle_t *client_handle, OPEN_ST_RECURRING_PARAM_t parameter, open_buffdesc *bufd)
{
  open_error_t result;

  if ((result = openapiStRecurringGet(client_handle, parameter, bufd)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get non-recurring summertime parameters. (result = %d)\n", result);
  }
  else
  {
    printf("Get the recurring summertime :%s\n",(char*)bufd->pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Set recurring summertime parameters.
*
* @param    client_handle   @b{(input)} client handle from registration API
* @param    parameter       @b{(input)} summer-time parameter
* @param    bufd            @b{(input)} summer-time parameter value to be set 

*
* @returns  none
*
* @end
*********************************************************************/
void StRecurringSet(openapiClientHandle_t *client_handle, OPEN_ST_RECURRING_PARAM_t parameter, open_buffdesc *bufd)
{
  open_error_t result;

  if ((result = openapiStRecurringSet(client_handle, parameter, bufd)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set recurring summertime parameters. (result = %d)\n", result);
  }
  else
  {
    printf("Set the recurring summertime :%s\n",(char*)bufd->pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Set non-recurring summertime parameters.
*
* @param    client_handle   @b{(input)} client handle from registration API
* @param    parameter       @b{(input)} summer-time parameter 
* @param    bufd            @b{(input)} summer-time parameter value to be set 

*
* @returns  none
*
* @end
*********************************************************************/
void StNonRecurringSet(openapiClientHandle_t *client_handle, OPEN_ST_NON_RECURRING_PARAM_t parameter, open_buffdesc *bufd)
{
  open_error_t result;

  if ((result = openapiStNonRecurringSet(client_handle, parameter, bufd)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set non-recurring summertime parameters. (result = %d)\n", result);
  }
  else
  {
    printf("Set the non-recurring summertime :%s\n",(char*)bufd->pstart);
  }
  return;
}


/*********************************************************************
* @purpose  Get the summer-time mode
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(output)}  Summer-time mode to be returned

*
* @returns  none
*
* @end
*********************************************************************/
void stModeGet(openapiClientHandle_t *client_handle, OPEN_SUMMER_TIME_MODE_t *mode)
{
  open_error_t result;

  if ((result = openapiStModeGet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the summer-time mode (result = %d)\n", result);
  }
  else
  {
    printf("Get Summer Time Mode :%d\n", *mode);
  }
  return;
}


/*********************************************************************
* @purpose  Set the summer-time mode
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  summer-time mode

*
* @returns  none
*
* @end
*********************************************************************/
void stModeSet(openapiClientHandle_t *client_handle, OPEN_SUMMER_TIME_MODE_t mode)
{
  open_error_t result;

  if ((result = openapiStModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the summer-time mode (result = %d)\n", result);
  }
  else
  {
    printf("Set Summer Time Mode :%d\n",mode);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the timezone acronym string
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    zoneName       @b{(input)}  timezone name

*
* @returns  none
*
* @end
*********************************************************************/
void SystemTimeZoneAcronymGet(openapiClientHandle_t *client_handle, open_buffdesc *zoneName)
{
  open_error_t result;

  if ((result = openapiSystemTimeZoneAcronymGet(client_handle, zoneName)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the timezone acronym string (result = %d)\n", result);
  }
  else
  {
    printf("Gets the timezone acronym string %s\n",(char*)zoneName->pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Get the current synchronized UTC time since the Unix Epoch.
*
* @param           @b{()}  

*
* @returns  none
*
* @end
*********************************************************************/
void uTCTimeNow(openapiClientHandle_t *client_handle )
{
  open_error_t result;

  result = openapiUTCTimeNow(client_handle);

  printf("Get the current synchronized UTC time since the Unix Epoch %d \n",result);

  return;
}


/*********************************************************************
* @purpose  Get the current UTC time since the Unix Epoch.
*
* @param    unit       @b{(output)}  UTC time

*
* @returns  none
*
* @end
*********************************************************************/
void uTCTimeGet(openapiClientHandle_t *client_handle, open_clocktime * unit)
{
  open_error_t result;

  if ((result = openapiUTCTimeGet(client_handle,unit)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the current UTC time since the Unix Epoch. (result = %d)\n", result);
  }
  else
  {
    printf("Get the current UTC time since the Unix Epoch Seconds:%d nanoseconds :%d\n",(int)unit->seconds, (int)unit->nanoseconds);
  }
  return;
}


/*********************************************************************
* @purpose  know whether the summer-time is in effect or not.
*
* @param           @b{()}  

*
* @returns  none
*
* @end
*********************************************************************/
void summerTimeIsInEffect(openapiClientHandle_t *client_handle)
{
  OPEN_BOOL_t result;

  if ((result = openapiSummerTimeIsInEffect(client_handle)) != OPEN_TRUE)
  {
    printf("Summer time is not in effect");
  }
  else
  {
    printf("Summer-time is in effect\n");
  }
  return;
}


/*********************************************************************
* @purpose  Adjusts current time for timezone and summer time
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    year       @b{(output)}  year
* @param    month       @b{(output)}  month of the year
* @param    day       @b{(output)}  day of the month
* @param    hour       @b{(output)}  hour of the day
* @param    minutes       @b{(output)}  minutes of the hour
* @param    seconds       @b{(output)}  seconds of the hour

*
* @returns  none
*
* @end
*********************************************************************/
void AdjustedTimeGet(openapiClientHandle_t *client_handle, uint32_t *year, uint32_t *month, uint32_t * day, uint32_t * hour, uint32_t * minutes, uint32_t * seconds)
{
  open_error_t result;

  if ((result = openapiAdjustedTimeGet(client_handle, year, month, day, hour, minutes, seconds)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to adjusts current time for timezone and summer time. (result = %d)\n", result);
  }
  else
  {
    printf("Adjusts current time for timezone and summer time year:%d ,month :%d ,day :%d , hour:%d , minutes:%d ,Seconds:%d\n",*year,*month,*day,*hour,*minutes,*seconds);
  }
  return;
}


/*********************************************************************
* @purpose  Converts the given Raw time to UTC and then adds the timezone and summer time offsets if configured.
*
* @param    rawTime       @b{(input)}  Raw time

*
* @returns  none
*
* @end
*********************************************************************/
void rawAdjustedTimeGet(openapiClientHandle_t *client_handle, uint32_t rawTime)
{
  uint32_t result;

  result = openapiRawAdjustedTimeGet(client_handle,rawTime);

  printf("Converted the given Raw time to UTC and then adds the timezone and summer time offsets if configured :%d\n",result);

  return;

}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for 
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  char buffer[128] = {0};
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  OPEN_SUMMER_TIME_MODE_t mode;
  OPEN_ST_RECURRING_PARAM_t recurringParam;
  OPEN_ST_NON_RECURRING_PARAM_t nonRecurringParam;   

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister(" example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting  API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 3)
      {
        uint32_t pMin;
        uint32_t pHour;
        open_buffdesc bufd;
        bufd.pstart = buffer;
        strncpy(bufd.pstart,argv[2],strlen(argv[2])+1);
        bufd.size = strlen(bufd.pstart)+1;
        timeConvert(&clientHandle, &bufd, &pHour, &pMin);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 3)
      {
        uint32_t pMin;
        uint32_t pHour;
        int32_t minutes = atoi(argv[2]);
        minutesOffsetConvert(&clientHandle, minutes, &pHour, &pMin);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 3)
      {
        open_buffdesc name;
        uint32_t number;
        name.pstart = buffer;
        strncpy(name.pstart,argv[2],strlen(argv[2])+1);
        name.size = strlen(name.pstart)+1;
        weekNumberGet(&clientHandle, &name, &number);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 3)
      {
        open_buffdesc acronym;
        acronym.pstart = buffer;
        strncpy(acronym.pstart,argv[2],strlen(argv[2]));
        acronym.size = strlen(acronym.pstart)+1;
        TimeZoneAcronymSet(&clientHandle, &acronym);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 2)
      {
        open_buffdesc acronym;
        acronym.pstart = buffer;
        acronym.size = sizeof(buffer);
        TimeZoneAcronymGet(&clientHandle, &acronym);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 3)
      {
        nonRecurringParam = atoi(argv[2]);
        open_buffdesc bufd;
        bufd.pstart = buffer;
        bufd.size = sizeof(buffer);
        StNonRecurringGet(&clientHandle, nonRecurringParam, &bufd);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 3)
      {
        recurringParam = atoi(argv[2]);
        open_buffdesc bufd;
        bufd.pstart = buffer;
        bufd.size = sizeof(buffer);
        StRecurringGet(&clientHandle, recurringParam, &bufd);
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 4)
      {
        recurringParam = atoi(argv[2]);
        open_buffdesc ptr;
        ptr.pstart = buffer;
        strncpy(ptr.pstart,argv[3],strlen(argv[3]));
        ptr.size = strlen(ptr.pstart);
        StRecurringSet(&clientHandle, recurringParam, &ptr);
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 4)
      {
        nonRecurringParam = atoi(argv[2]);
        open_buffdesc bufd;
        bufd.pstart = buffer;
        strncpy(bufd.pstart,argv[3],strlen(argv[3]));
        bufd.size = strlen(bufd.pstart);
        StNonRecurringSet(&clientHandle, nonRecurringParam, &bufd);
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 2)
      {
        stModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 3)
      {
        mode = atoi(argv[2]);
        stModeSet(&clientHandle, mode);
        show_help = 0;
      }
      break;
    case 12:
      if (argc == 2)
      {
        open_buffdesc zoneName;
        zoneName.pstart = buffer;
        zoneName.size = sizeof(buffer);
        SystemTimeZoneAcronymGet(&clientHandle, &zoneName);
        show_help = 0;
      }
      break;
    case 13:
      if (argc == 2)
      {
        uTCTimeNow(&clientHandle);
        show_help = 0;
      }
      break;
    case 14:
      if (argc == 2)
      {
        open_clocktime unit;
        uTCTimeGet(&clientHandle, &unit);
        show_help = 0;
      }
      break;
    case 15:
      if (argc == 2)
      {
        summerTimeIsInEffect(&clientHandle);
        show_help = 0;
      }
      break;
    case 16:
      if (argc == 2)
      {
        uint32_t hour;
        uint32_t seconds;
        uint32_t minutes;
        uint32_t year;
        uint32_t day;
        uint32_t month;
        AdjustedTimeGet(&clientHandle, &year, &month, &day, &hour, &minutes, &seconds);
        show_help = 0;
      }
      break;
    case 17:
      if (argc == 3)
      {
        uint32_t rawTime =  atoi(argv[2]);
        rawAdjustedTimeGet(&clientHandle, rawTime);
        show_help = 0;
      }
      break;
    case 18:
      if (argc == 2)
      {
        runSanity(&clientHandle);
        show_help = 0;
      }
      break;
    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping  API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}

