
/*********************************************************************
*
*  Copyright 2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  traceroute_example.c
*
* @purpose   Traceroute OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    03/29/2018
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_traceroute.h"

/*******************************************************************
*
* @brief  This function prints the Traceroute Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printTracerouteAppMenu()
{
  printf("Usage: traceroute_example <test#> <arg1> <arg2> ... \n");
  printf("Usage: traceroute_example <test#> <arg1> <arg2> ... OPEN_TRACEROUTE_SOURCE_INTF_SERVICE_PORT\n");
  printf("Usage:  In below, <srcIfType> should be set to 0 for OPEN_TRACEROUTE_SOURCE_INTF_SERVICE_PORT (service port), and 1 for OPEN_TRACEROUTE_SOURCE_INTF_NETWORK_PORT (network port)\n");
  printf("Usage:  In below, <srcType> should be set to 0 for service port, 1 for network port, 2 for ip port, 3 for address and 4 for none \n");
  printf("Test 1: Traceroute host or IPv4/IPv6 Address: traceroute_example 1 <ip-address/hostname/ipv6-address> \n");
  printf("Test 2: Traceroute host or IPv4/IPv6 Address with count: traceroute_example 2 <ip-address/hostname/ipv6-address> count <count> \n");
  printf("Test 3: Traceroute host or IPv4/IPv6 Address with interval: traceroute_example 3 <ip-address/hostname/ipv6-address> interval <interval> \n");
  printf("Test 4: Traceroute host or IPv4/IPv6 Address with Ttl: traceroute_example 4 <ip-address/hostname/ipv6-address> initTtl <initTtl> maxTtl <maxTtl>\n");
  printf("Test 5: Traceroute host or IPv4/IPv6 Address with maximum Failures: traceroute_example 5 <ip-address/hostname/ipv6-address> maxFail <maxFail> \n");
  printf("Test 6: Traceroute host or IPv4/IPv6 Address with size: traceroute_example 6 <ip-address/hostname/ipv6-address> size <size> \n");
  printf("Test 7: Traceroute host or IPv4/IPv6 Address with UDP Dest port: traceroute_example 7 <ip-address/hostname/ipv6-address> udpPort <port> \n");
  printf("Test 8: Traceroute host or IPv4/IPv6 Address with source interface: traceroute_example 8 <ip-address/hostname/ipv6-address> source <srcIfType> \n");
  printf("Test 9: Traceroute host or IPv4/IPv6 Address with count, size, interval and source interface: traceroute_example 9 <ip-address/hostname/ipv6-address> count <count> interval <interval> initTtl <initTtl>  maxTtl <maxTtl> maxFail <maxFail> size <size> udpPort <port> source <srcIfType> \n");
  printf("Test 10: Traceroute host or IPv4/IPv6 Address with source: traceroute_example 10 <vrf-name> <family> <address/hostname> source <srcType> <srcIntIfNum> <srcAddress> \n");
  printf("Test 11: Traceroute host or IPv4/IPv6 Address with count, size, interval and source: traceroute_example 11 <vrf-name> <family> <address/hostname> count <count> interval <interval> initTtl <initTtl>  maxTtl <maxTtl> maxFail <maxFail> size <size> udpPort <port> source <srcType> <srcIntIfNum> <srcAddress> \n");
  printf("Test 12: traceroute_example OpEN APIs sanity: traceroute_example 12 \n");

  return;
}

/*********************************************************************
* @purpose  Traceroute  IPv4/IPv6 address or hostname
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    tracerouteAddr   @b{(input)}   traceroute IPv4/IPv6 address or hostname
*
* @returns  none
*
*
* @end
*********************************************************************/
void tracerouteAddress(openapiClientHandle_t *clientHandle, 
                       char *tracerouteAddr)
{
  open_error_t result;
  open_buffdesc tracerouteIpAddr;
  open_buffdesc tracerouteOutput;
  char str[OPEN_TRACEROUTE_ADDRESS_MAX_LEN];
  char tmpBuf[OPEN_TRACEROUTE_OUTPUT_STRING_SIZE];

  memset(str, 0, sizeof(str));
  strncpy(str, tracerouteAddr, sizeof(str));
  tracerouteIpAddr.pstart = str;
  tracerouteIpAddr.size = strlen(str)+1;

  memset(tmpBuf, 0, sizeof(tmpBuf));
  tracerouteOutput.pstart = tmpBuf;
  tracerouteOutput.size = sizeof(tmpBuf);

  printf("\nTraceroute to %s ,%d max hops %d byte packets:\n", tracerouteAddr, 
                                                               OPEN_TRACEROUTE_DEFAULT_MAX_TTL, 
                                                               OPEN_TRACEROUTE_DEFAULT_PROBE_SIZE);

  result = openapiTracerouteAddress(clientHandle, &tracerouteIpAddr, 
                                    &tracerouteOutput);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to traceroute address %s. (result = %d)\n", tracerouteAddr, result);
    printf("Traceroute to address %s Failed\n", tracerouteAddr);
  }
  else
  {
    printf("\n %s\n", (char *)tracerouteOutput.pstart);
  }

  return;
}

/*********************************************************************
* @purpose  Traceroute IPv4/IPv6 address or hostname with count
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    tracerouteAddr   @b{(input)}   traceroute IPv4/IPv6 address or hostname
* @param    tracerouteCount  @b{(input)}   number of traceroute packets to send
*
* @returns  none
*
*
* @end
*********************************************************************/
void tracerouteAddressWithCount(openapiClientHandle_t *clientHandle, 
                                char *tracerouteAddr,
                                uint32_t tracerouteCount)
{
  open_error_t result;
  open_buffdesc tracerouteIpAddr;
  open_buffdesc tracerouteOutput;
  char str[OPEN_TRACEROUTE_ADDRESS_MAX_LEN];
  char tmpBuf[OPEN_TRACEROUTE_OUTPUT_STRING_SIZE];

  memset(str, 0, sizeof(str));
  strncpy(str, tracerouteAddr, sizeof(str));
  tracerouteIpAddr.pstart = str;
  tracerouteIpAddr.size = strlen(str)+1;

  memset(tmpBuf, 0, sizeof(tmpBuf));
  tracerouteOutput.pstart = tmpBuf;
  tracerouteOutput.size = sizeof(tmpBuf);

  printf("\nTraceroute to %s for %d time(s), %d max hops, %d byte packets:\n", 
            tracerouteAddr, tracerouteCount, OPEN_TRACEROUTE_DEFAULT_MAX_TTL, 
            OPEN_TRACEROUTE_DEFAULT_PROBE_SIZE);

  result = openapiTracerouteAddressWithCount(clientHandle, &tracerouteIpAddr, 
                                             tracerouteCount, &tracerouteOutput);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to traceroute address %s. (result = %d)\n", tracerouteAddr, result);
    printf("Traceroute to address %s Failed\n", tracerouteAddr);
  }
  else
  {
    printf("\n %s\n", (char *)tracerouteOutput.pstart);
  }

  return;
}

/*********************************************************************
* @purpose  Traceroute IPv4/IPv6 address or hostname with interval
*
* @param    clientHandle        @b{(input)}   client handle from registration API
* @param    tracerouteAddr      @b{(input)}   traceroute IPv4/IPv6 address or hostname
* @param    tracerouteInterval  @b{(input)}   interval between traceroute packets in 
*                                             seconds
*
* @returns  none
*
*
* @end
*********************************************************************/
void tracerouteAddressWithInterval(openapiClientHandle_t *clientHandle, 
                                   char *tracerouteAddr,
                                   uint32_t tracerouteInterval)
{
  open_error_t result;
  open_buffdesc tracerouteIpAddr;
  open_buffdesc tracerouteOutput;
  char str[OPEN_TRACEROUTE_ADDRESS_MAX_LEN];
  char tmpBuf[OPEN_TRACEROUTE_OUTPUT_STRING_SIZE];

  memset(str, 0, sizeof(str));
  strncpy(str, tracerouteAddr, sizeof(str));
  tracerouteIpAddr.pstart = str;
  tracerouteIpAddr.size = strlen(str)+1;

  memset(tmpBuf, 0, sizeof(tmpBuf));
  tracerouteOutput.pstart = tmpBuf;
  tracerouteOutput.size = sizeof(tmpBuf);

  printf("\nTraceroute to %s, %d max hops, %d byte packets, %d interval:\n", 
            tracerouteAddr, OPEN_TRACEROUTE_DEFAULT_MAX_TTL, OPEN_TRACEROUTE_DEFAULT_PROBE_SIZE,
            tracerouteInterval);

  result = openapiTracerouteAddressWithInterval(clientHandle, &tracerouteIpAddr, 
                                                tracerouteInterval, &tracerouteOutput);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to traceroute address %s. (result = %d)\n", tracerouteAddr, result);
    printf("Traceroute to address %s Failed\n", tracerouteAddr);
  }
  else
  {
    printf("\n %s\n", (char *)tracerouteOutput.pstart);
  }

  return;
}

/*********************************************************************
* @purpose  Traceroute IPv4/IPv6 address or hostname with size
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    tracerouteAddr   @b{(input)}   traceroute IPv4/IPv6 address or hostname
* @param    tracerouteSize   @b{(input)}   PDU size of traceroute packet 
*
* @returns  none
*
*
* @end
*********************************************************************/
void tracerouteAddressWithSize(openapiClientHandle_t *clientHandle, 
                               char *tracerouteAddr,
                               uint32_t tracerouteSize)
{
  open_error_t result;
  open_buffdesc tracerouteIpAddr;
  open_buffdesc tracerouteOutput;
  char str[OPEN_TRACEROUTE_ADDRESS_MAX_LEN];
  char tmpBuf[OPEN_TRACEROUTE_OUTPUT_STRING_SIZE];

  memset(str, 0, sizeof(str));
  strncpy(str, tracerouteAddr, sizeof(str));
  tracerouteIpAddr.pstart = str;
  tracerouteIpAddr.size = strlen(str)+1;

  memset(tmpBuf, 0, sizeof(tmpBuf));
  tracerouteOutput.pstart = tmpBuf;
  tracerouteOutput.size = sizeof(tmpBuf);

  printf("\nTraceroute to %s, %d max hops, %d byte packets:\n", 
            tracerouteAddr, OPEN_TRACEROUTE_DEFAULT_MAX_TTL, tracerouteSize);

  result = openapiTracerouteAddressWithPDUSize(clientHandle, &tracerouteIpAddr,
                                               tracerouteSize, &tracerouteOutput);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to traceroute address %s. (result = %d)\n", tracerouteAddr, result);
    printf("Traceroute to address %s Failed\n", tracerouteAddr);
  }
  else
  {
    printf("\n %s\n", (char *)tracerouteOutput.pstart);
  }

  return;
}

/*********************************************************************
* @purpose  Traceroute IPv4/IPv6 address or hostname with initTtl and max Ttl
*
* @param    clientHandle      @b{(input)}   client handle from registration API
* @param    tracerouteAddr    @b{(input)}   traceroute IPv4/IPv6 address or hostname
* @param    tracerouteInitTtl @b{(input)}   initial TTL for session
* @param    tracerouteMaxTtl @b{(input)}   maximum TTL for the destination
*
* @returns  none
*
*
* @end
*********************************************************************/
void tracerouteAddressWithTtl(openapiClientHandle_t *clientHandle, 
                              char *tracerouteAddr,
                              uint32_t tracerouteInitTtl,
                              uint32_t tracerouteMaxTtl)
{
  open_error_t result;
  open_buffdesc tracerouteIpAddr;
  open_buffdesc tracerouteOutput;
  char str[OPEN_TRACEROUTE_ADDRESS_MAX_LEN];
  char tmpBuf[OPEN_TRACEROUTE_OUTPUT_STRING_SIZE];

  memset(str, 0, sizeof(str));
  strncpy(str, tracerouteAddr, sizeof(str));
  tracerouteIpAddr.pstart = str;
  tracerouteIpAddr.size = strlen(str)+1;

  memset(tmpBuf, 0, sizeof(tmpBuf));
  tracerouteOutput.pstart = tmpBuf;
  tracerouteOutput.size = sizeof(tmpBuf);

  printf("\nTraceroute to %s, %d min hops, %d max hops, %d byte packets:\n", 
            tracerouteAddr, tracerouteInitTtl, tracerouteMaxTtl, 
            OPEN_TRACEROUTE_DEFAULT_PROBE_SIZE);

  result = openapiTracerouteAddressWithTtl(clientHandle, &tracerouteIpAddr, 
                                               tracerouteInitTtl, tracerouteMaxTtl, 
                                               &tracerouteOutput);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to traceroute address %s. (result = %d)\n", tracerouteAddr, result);
    printf("Traceroute to address %s Failed\n", tracerouteAddr);
  }
  else
  {
    printf("\n %s\n", (char *)tracerouteOutput.pstart);
  }

  return;
}
/*********************************************************************
* @purpose  Traceroute IPv4/IPv6 address or hostname with maxFail
*
* @param    clientHandle      @b{(input)}  client handle from registration API
* @param    tracerouteAddr    @b{(input)}  traceroute IPv4/IPv6 address or hostname
* @param    tracerouteMaxFail @b{(input)}  maximum failures allowed in session 
*
* @returns  none
*
*
* @end 
*********************************************************************/
void tracerouteAddressWithMaxFail(openapiClientHandle_t *clientHandle, 
                                  char *tracerouteAddr,                              
                                  uint32_t tracerouteMaxFail)                        
{
  open_error_t result;
  open_buffdesc tracerouteIpAddr;
  open_buffdesc tracerouteOutput;
  char str[OPEN_TRACEROUTE_ADDRESS_MAX_LEN];
  char tmpBuf[OPEN_TRACEROUTE_OUTPUT_STRING_SIZE];

  memset(str, 0, sizeof(str));
  strncpy(str, tracerouteAddr, sizeof(str));
  tracerouteIpAddr.pstart = str;
  tracerouteIpAddr.size = strlen(str)+1;

  memset(tmpBuf, 0, sizeof(tmpBuf));
  tracerouteOutput.pstart = tmpBuf;
  tracerouteOutput.size = sizeof(tmpBuf);

  printf("\nTraceroute to %s, %d max hops, %d byte packets, %d max failures:\n",
            tracerouteAddr, OPEN_TRACEROUTE_DEFAULT_MAX_TTL, OPEN_TRACEROUTE_DEFAULT_PROBE_SIZE,
            tracerouteMaxFail);

  result = openapiTracerouteAddressWithMaxFail(clientHandle, &tracerouteIpAddr,
                                               tracerouteMaxFail, &tracerouteOutput);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to traceroute address %s. (result = %d)\n", tracerouteAddr, result);
    printf("Traceroute to address %s Failed\n", tracerouteAddr);
  }
  else
  {
    printf("\n %s\n", (char *)tracerouteOutput.pstart);
  }

  return;
}

/*********************************************************************
* @purpose  Traceroute IPv4/IPv6 address or hostname with UDP port
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    tracerouteAddr   @b{(input)}   traceroute IPv4/IPv6 address or hostname
* @param    traceroutePort   @b{(input)}   UDP dest port in probe packets
*
* @returns  none
*
*
* @end
*********************************************************************/
void tracerouteAddressWithPort(openapiClientHandle_t *clientHandle, 
                               char *tracerouteAddr,
                               uint32_t traceroutePort)
{
  open_error_t result;
  open_buffdesc tracerouteIpAddr;
  open_buffdesc tracerouteOutput;
  char str[OPEN_TRACEROUTE_ADDRESS_MAX_LEN];
  char tmpBuf[OPEN_TRACEROUTE_OUTPUT_STRING_SIZE];

  memset(str, 0, sizeof(str));
  strncpy(str, tracerouteAddr, sizeof(str));
  tracerouteIpAddr.pstart = str;
  tracerouteIpAddr.size = strlen(str)+1;

  memset(tmpBuf, 0, sizeof(tmpBuf));
  tracerouteOutput.pstart = tmpBuf;
  tracerouteOutput.size = sizeof(tmpBuf);

  printf("\nTraceroute to %s, %d max hops, %d byte packets, %d UDP port:\n", 
            tracerouteAddr, OPEN_TRACEROUTE_DEFAULT_MAX_TTL, OPEN_TRACEROUTE_DEFAULT_PROBE_SIZE,
            traceroutePort);

  result = openapiTracerouteAddressWithUdpPort(clientHandle, &tracerouteIpAddr,
                                               traceroutePort, &tracerouteOutput);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to traceroute address %s. (result = %d)\n", tracerouteAddr, result);
    printf("Traceroute to address %s Failed\n", tracerouteAddr);
  }
  else
  {
    printf("\n %s\n", (char *)tracerouteOutput.pstart);
  }

  return;
}
/*********************************************************************
* @purpose  Traceroute IPv4/IPv6 address or hostname with source interface
*
* @param    clientHandle      @b{(input)}   client handle from registration API
* @param    tracerouteAddr    @b{(input)}   traceroute IPv4/IPv6 address or hostname
* @param    tracerouteSrcIntf @b{(input)}   source interface
*
* @returns  none
*
*
* @end
*********************************************************************/
void tracerouteAddressWithSourceInterface(openapiClientHandle_t *clientHandle, 
                            char *tracerouteAddr,
                            OPEN_TRACEROUTE_SOURCE_INTF_TYPE_t tracerouteSrcIntf)
{
  open_error_t result;
  open_buffdesc tracerouteIpAddr;
  open_buffdesc tracerouteOutput;
  char str[OPEN_TRACEROUTE_ADDRESS_MAX_LEN];
  char tmpBuf[OPEN_TRACEROUTE_OUTPUT_STRING_SIZE];
  char sourceInterfaceStr[OPEN_TRACEROUTE_STRING_SIZE];

  memset(str, 0, sizeof(str));
  strncpy(str, tracerouteAddr, sizeof(str));
  tracerouteIpAddr.pstart = str;
  tracerouteIpAddr.size = strlen(str)+1;

  memset(tmpBuf, 0, sizeof(tmpBuf));
  tracerouteOutput.pstart = tmpBuf;
  tracerouteOutput.size = sizeof(tmpBuf);

  memset(sourceInterfaceStr, 0, sizeof(sourceInterfaceStr));
  if(OPEN_TRACEROUTE_SOURCE_INTF_SERVICE_PORT == tracerouteSrcIntf)
  {
    strncpy(sourceInterfaceStr, "service port", sizeof(sourceInterfaceStr));
  }
  else if(OPEN_TRACEROUTE_SOURCE_INTF_NETWORK_PORT == tracerouteSrcIntf)
  {
    strncpy(sourceInterfaceStr, "network port", sizeof(sourceInterfaceStr));
  }

  printf("\nTraceroute to %s, %d max hops, %d byte packets via %s:\n", 
            tracerouteAddr, OPEN_TRACEROUTE_DEFAULT_MAX_TTL, OPEN_TRACEROUTE_DEFAULT_PROBE_SIZE,
            sourceInterfaceStr);

  result = openapiTracerouteAddressWithSourceInterface(clientHandle, &tracerouteIpAddr,
                                                       tracerouteSrcIntf, &tracerouteOutput);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to traceroute address %s. (result = %d)\n", tracerouteAddr, result);
    printf("Traceroute to address %s Failed\n", tracerouteAddr);
  }
  else
  {
    printf("\n %s\n", (char *)tracerouteOutput.pstart);
  }

  return;
}
 
/*********************************************************************
* @purpose  Traceroute an IP address or hostname with explicitly 
*           specifying various parameters
*
* @param    clientHandle        @b{(input)}   client handle from registration API
* @param    tracerouteAddr      @b{(input)}   traceroute IPv4/IPv6 address or hostname
* @param    tracerouteCount     @b{(input)}   number of traceroute packets to send
* @param    tracerouteInterval  @b{(input)}   interval between traceroute packets in 
*                                             seconds
* @param    tracerouteInitTtl   @b{(input)}   initial TTL for session
* @param    tracerouteMaxTtl    @b{(input)}   maximum TTL for the destination
* @param    tracerouteMaxFail   @b{(input)}   maximum failures allowed in session 
* @param    tracerouteSize      @b{(input)}   PDU size of traceroute packet 
* @param    traceroutePort      @b{(input)}   UDP dest port in probe packets
* @param    tracerouteSrcIntf   @b{(input)}   source interface
*
* @returns  none
*
*
* @end
*********************************************************************/
void tracerouteAddressExplicit(openapiClientHandle_t *clientHandle, 
                              char *tracerouteAddr,
                              uint32_t tracerouteCount,
                              uint32_t tracerouteInterval,
                              uint32_t tracerouteInitTtl,
                              uint32_t tracerouteMaxTtl,
                              uint32_t tracerouteMaxFail,
                              uint32_t tracerouteSize,
                              uint32_t traceroutePort, 
                              OPEN_TRACEROUTE_SOURCE_INTF_TYPE_t tracerouteSrcIntf)
{
  open_error_t result;
  open_buffdesc tracerouteIpAddr;
  open_buffdesc tracerouteOutput;
  char str[OPEN_TRACEROUTE_ADDRESS_MAX_LEN];
  char tmpBuf[OPEN_TRACEROUTE_OUTPUT_STRING_SIZE];
  char sourceInterfaceStr[OPEN_TRACEROUTE_STRING_SIZE];

  memset(str, 0, sizeof(str));
  strncpy(str, tracerouteAddr, sizeof(str));
  tracerouteIpAddr.pstart = str;
  tracerouteIpAddr.size = strlen(str)+1;

  memset(tmpBuf, 0, sizeof(tmpBuf));
  tracerouteOutput.pstart = tmpBuf;
  tracerouteOutput.size = sizeof(tmpBuf);

  memset(sourceInterfaceStr, 0, sizeof(sourceInterfaceStr));
  if(OPEN_TRACEROUTE_SOURCE_INTF_SERVICE_PORT == tracerouteSrcIntf)
  {
    strncpy(sourceInterfaceStr, "service port", sizeof(sourceInterfaceStr));
  }
  else if(OPEN_TRACEROUTE_SOURCE_INTF_NETWORK_PORT == tracerouteSrcIntf)
  {
    strncpy(sourceInterfaceStr, "network port", sizeof(sourceInterfaceStr));
  }

  printf("\nTraceroute to %s for %d time(s), %d interval, %d min hops, %d max hops, %d byte packets, %d max failures,%d UDP port via %s :\n", 
            tracerouteAddr, tracerouteCount, tracerouteInterval, tracerouteInitTtl, 
            tracerouteMaxTtl, tracerouteSize, tracerouteMaxFail, traceroutePort, 
            sourceInterfaceStr);

  result = openapiTracerouteAddressExplicit(clientHandle, &tracerouteIpAddr, 
                                            tracerouteCount, tracerouteInterval, 
                                            tracerouteInitTtl, tracerouteMaxTtl,
                                            tracerouteMaxFail, tracerouteSize, 
                                            traceroutePort, tracerouteSrcIntf, 
                                            &tracerouteOutput);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to traceroute address %s. (result = %d)\n", tracerouteAddr, result);
    printf("Traceroute to address %s Failed\n", tracerouteAddr);
  }
  else
  {
    printf("\n %s\n", (char *)tracerouteOutput.pstart);
  }

  return;
}
/*********************************************************************
* @purpose  Traceroute IPv4/IPv6 address or hostname with source
*
* @param    clientHandle      @b{(input)}   client handle from registration API
* @param    vrfName           @b{(input)}   vrf name
* @param    family            @b{(input)}   destination address family
* @param    tracerouteAddr    @b{(input)}   traceroute address or hostname
* @param    srcType           @b{(input)}   source type
* @param    intIfNum          @b{(input)}   source internal interface
* @param    srcAddr           @b{(input)}   source address
*
* @returns  none
*
*
* @end
*********************************************************************/
void tracerouteWithSource (openapiClientHandle_t *clientHandle,
                           char *vrfName,
                           OPEN_AF_t family,
                           char *tracerouteAddr,
                           OPEN_TRACEROUTE_SOURCE_TYPE_t srcType,
                           uint32_t intIfNum,
                           char *srcAddr)
{
  open_error_t result;
  open_buffdesc tracerouteIpAddr;
  open_buffdesc tracerouteOutput;
  open_buffdesc vrfNameBuf;
  open_buffdesc srcAddrBuf;
  char vrfStr[OPEN_VRF_MAX_NAME_LEN + 1];
  char str[OPEN_TRACEROUTE_ADDRESS_MAX_LEN];
  char tmpBuf[OPEN_TRACEROUTE_STRING_SIZE];
  char srcStr[OPEN_TRACEROUTE_STRING_SIZE];
  char srcAddrStr[OPEN_TRACEROUTE_ADDRESS_MAX_LEN];

  memset (vrfStr, 0, sizeof (vrfStr));
  strncpy (vrfStr, vrfName, sizeof (vrfStr));
  vrfNameBuf.pstart = vrfStr;
  vrfNameBuf.size = strlen(vrfStr) + 1;

  memset(str, 0, sizeof(str));
  strncpy(str, tracerouteAddr, sizeof(str));
  tracerouteIpAddr.pstart = str;
  tracerouteIpAddr.size = strlen(str)+1;

  memset(tmpBuf, 0, sizeof(tmpBuf));
  tracerouteOutput.pstart = tmpBuf;
  tracerouteOutput.size = sizeof(tmpBuf);

  memset (srcAddrStr, 0, sizeof (srcAddrStr));
  strncpy (srcAddrStr, srcAddr, sizeof (srcAddrStr));
  srcAddrBuf.pstart = srcAddrStr;
  srcAddrBuf.size = strlen(srcAddrStr) + 1;

  memset(srcStr, 0, sizeof(srcStr));
  if(OPEN_TRACEROUTE_SOURCE_SERVICE_PORT == srcType)
  {
    strncpy(srcStr, "service port", sizeof(srcStr));
  }
  else if(OPEN_TRACEROUTE_SOURCE_NETWORK_PORT == srcType)
  {
    strncpy(srcStr, "network port", sizeof(srcStr));
  }
  else if (OPEN_TRACEROUTE_SOURCE_IP_PORT == srcType)
  {
    snprintf (srcStr, sizeof (srcStr),
              "ip port %u", intIfNum);
  }
  else if (OPEN_TRACEROUTE_SOURCE_ADDRESS == srcType)
  {
    snprintf (srcStr, sizeof (srcStr),
              "addr %s", srcAddrStr);
  }
  else if (OPEN_TRACEROUTE_SOURCE_NONE  == srcType)
  {
    snprintf (srcStr, sizeof (srcStr), "none");
  }
  else
  {
    printf ("\nNot a valid source type\n");
  }

  printf("\nTraceroute to %s, %d max hops, %d byte packets via %s:\n", 
            tracerouteAddr, OPEN_TRACEROUTE_DEFAULT_MAX_TTL,
            OPEN_TRACEROUTE_DEFAULT_PROBE_SIZE, srcStr);

  result = openapiTracerouteWithSource (clientHandle, &vrfNameBuf, family,
                                        &tracerouteIpAddr, srcType,
                                        intIfNum, &srcAddrBuf, &tracerouteOutput);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to traceroute address %s. (result = %d)\n", tracerouteAddr, result);
    printf("Traceroute to address %s Failed\n", tracerouteAddr);
  }
  else
  {
    printf("\n %s\n", (char *)tracerouteOutput.pstart);
  }

  return;
}
 
/*********************************************************************
* @purpose  Traceroute an IP address or hostname with explicitly 
*           specifying various parameters
*
* @param    clientHandle        @b{(input)}   client handle from registration API
* @param    vrfName             @b{(input)}   vrf name
* @param    family              @b{(input)}   destination address family
* @param    tracerouteAddr      @b{(input)}   traceroute IPv4/IPv6 address or hostname
* @param    tracerouteCount     @b{(input)}   number of traceroute packets to send
* @param    tracerouteInterval  @b{(input)}   interval between traceroute packets in 
*                                             seconds
* @param    tracerouteInitTtl   @b{(input)}   initial TTL for session
* @param    tracerouteMaxTtl    @b{(input)}   maximum TTL for the destination
* @param    tracerouteMaxFail   @b{(input)}   maximum failures allowed in session 
* @param    tracerouteSize      @b{(input)}   PDU size of traceroute packet 
* @param    traceroutePort      @b{(input)}   UDP dest port in probe packets
* @param    srcType             @b{(input)}   source type
* @param    intIfNum            @b{(input)}   source internal interface
* @param    srcAddr             @b{(input)}   source address
*
* @returns  none
*
*
* @end
*********************************************************************/
void traceroute (openapiClientHandle_t *clientHandle, 
                 char *vrfName,
                 OPEN_AF_t family,
                 char *tracerouteAddr,
                 uint32_t tracerouteCount,
                 uint32_t tracerouteInterval,
                 uint32_t tracerouteInitTtl,
                 uint32_t tracerouteMaxTtl,
                 uint32_t tracerouteMaxFail,
                 uint32_t tracerouteSize,
                 uint32_t traceroutePort, 
                 OPEN_TRACEROUTE_SOURCE_TYPE_t srcType,
                 uint32_t intIfNum,
                 char *srcAddr)
{
  open_error_t result;
  open_buffdesc tracerouteIpAddr;
  open_buffdesc tracerouteOutput;
  open_buffdesc vrfNameBuf;
  open_buffdesc srcAddrBuf;
  char vrfStr[OPEN_VRF_MAX_NAME_LEN + 1];
  char str[OPEN_TRACEROUTE_ADDRESS_MAX_LEN];
  char tmpBuf[OPEN_TRACEROUTE_STRING_SIZE];
  char srcStr[OPEN_TRACEROUTE_STRING_SIZE];
  char srcAddrStr[OPEN_TRACEROUTE_ADDRESS_MAX_LEN];

  memset (vrfStr, 0, sizeof (vrfStr));
  strncpy (vrfStr, vrfName, sizeof (vrfStr));
  vrfNameBuf.pstart = vrfStr;
  vrfNameBuf.size = strlen(vrfStr) + 1;

  memset(str, 0, sizeof(str));
  strncpy(str, tracerouteAddr, sizeof(str));
  tracerouteIpAddr.pstart = str;
  tracerouteIpAddr.size = strlen(str)+1;

  memset(tmpBuf, 0, sizeof(tmpBuf));
  tracerouteOutput.pstart = tmpBuf;
  tracerouteOutput.size = sizeof(tmpBuf);

  memset (srcAddrStr, 0, sizeof (srcAddrStr));
  strncpy (srcAddrStr, srcAddr, sizeof (srcAddrStr));
  srcAddrBuf.pstart = srcAddrStr;
  srcAddrBuf.size = strlen(srcAddrStr) + 1;

  memset(srcStr, 0, sizeof(srcStr));
  if(OPEN_TRACEROUTE_SOURCE_SERVICE_PORT == srcType)
  {
    strncpy(srcStr, "service port", sizeof(srcStr));
  }
  else if(OPEN_TRACEROUTE_SOURCE_NETWORK_PORT == srcType)
  {
    strncpy(srcStr, "network port", sizeof(srcStr));
  }
  else if (OPEN_TRACEROUTE_SOURCE_IP_PORT == srcType)
  {
    snprintf (srcStr, sizeof (srcStr),
              "ip port %u", intIfNum);
  }
  else if (OPEN_TRACEROUTE_SOURCE_ADDRESS == srcType)
  {
    snprintf (srcStr, sizeof (srcStr),
              "addr %s", srcAddrStr);
  }
  else if (OPEN_TRACEROUTE_SOURCE_NONE  == srcType)
  {
    snprintf (srcStr, sizeof (srcStr), "none");
  }
  else
  {
    printf ("\nNot a valid source type\n");
  }

  printf ("\nTraceroute to %s for %d time(s), %d interval,"
          " %d min hops, %d max hops, %d byte packets,"
          " %d max failures,%d UDP port via %s :\n",
          tracerouteAddr, tracerouteCount, tracerouteInterval,
          tracerouteInitTtl, tracerouteMaxTtl, tracerouteSize,
          tracerouteMaxFail, traceroutePort, srcStr);

  result = openapiTraceroute (clientHandle, &vrfNameBuf,
                              family, &tracerouteIpAddr, 
                              tracerouteCount, tracerouteInterval, 
                              tracerouteInitTtl, tracerouteMaxTtl,
                              tracerouteMaxFail, tracerouteSize, 
                              traceroutePort, srcType,
                              intIfNum, &srcAddrBuf, 
                              &tracerouteOutput);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to traceroute address %s. (result = %d)\n", tracerouteAddr, result);
    printf("Traceroute to address %s Failed\n", tracerouteAddr);
  }
  else
  {
    printf("\n %s\n", (char *)tracerouteOutput.pstart);
  }

  return;
}
 
/*********************************************************************
* @purpose  Sanity test Traceroute information
*
* @param    clientHandle   @b{(input)} client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void tracerouteSanityTest(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc tracerouteIpAddr;
  open_buffdesc tracerouteOutput;
  char buf[OPEN_TRACEROUTE_ADDRESS_MAX_LEN];
  char tmpBuf[OPEN_TRACEROUTE_OUTPUT_STRING_SIZE];
  uint32_t count = 3;
  uint32_t size = 0;
  uint32_t initTtl = 1;
  uint32_t maxTtl = 30;
  uint32_t maxFail = 5;
  uint32_t port = 33434;
  uint32_t interval = 3;
  OPEN_TRACEROUTE_SOURCE_INTF_TYPE_t srcIntf = OPEN_TRACEROUTE_SOURCE_INTF_NETWORK_PORT;

  memset(buf, 0, sizeof(buf));
  tracerouteIpAddr.pstart = buf;
  tracerouteIpAddr.size = sizeof(buf);

  memset(tmpBuf, 0, sizeof(tmpBuf));
  tracerouteOutput.pstart = tmpBuf;
  tracerouteOutput.size = sizeof(tmpBuf);

  printf("Testing Traceroute OpEN APIs sanity:\n\n");

  /* openapiTracerouteAddress () */
  printf("\nTesting openapiTracerouteAddress(): \n");

  result = openapiTracerouteAddress(NULL, &tracerouteIpAddr, &tracerouteOutput);
  printf("NULL Client Handle. (result = %d)\n", result);
  
  result = openapiTracerouteAddress(clientHandle, NULL, &tracerouteOutput);
  printf("NULL buff descriptor to traceroute address. (result = %d)\n", result);

  result = openapiTracerouteAddress(clientHandle, &tracerouteIpAddr, NULL); 
  printf("Invalid traceroute output. (result = %d)\n", result);

  printf("openapiTracerouteAddress() sanity successful. \n");

  /* openapiTracerouteAddressWithCount () */
  printf("\nTesting openapiTracerouteAddressWithCount(): \n");

  result = openapiTracerouteAddressWithCount(NULL, &tracerouteIpAddr, count, &tracerouteOutput);
  printf("NULL Client Handle. (result = %d)\n", result);
  
  result = openapiTracerouteAddressWithCount(clientHandle, NULL, count, &tracerouteOutput);
  printf("NULL buff descriptor to traceroute address. (result = %d)\n", result);

  result = openapiTracerouteAddressWithCount(clientHandle, &tracerouteIpAddr, 0, &tracerouteOutput);
  printf("Invalid traceroute count. (result = %d)\n", result);

  result = openapiTracerouteAddressWithCount(clientHandle, &tracerouteIpAddr, count, NULL); 
  printf("Invalid traceroute output. (result = %d)\n", result);

  printf("openapiTracerouteAddressWithCount() sanity successful. \n");

  /* openapiTracerouteAddressWithInterval() */
  printf("\nTesting openapiTracerouteAddressWithInterval(): \n");

  result = openapiTracerouteAddressWithInterval(NULL, &tracerouteIpAddr, interval, &tracerouteOutput);
  printf("NULL Client Handle. (result = %d)\n", result);
  
  result = openapiTracerouteAddressWithInterval(clientHandle, NULL, interval, &tracerouteOutput);
  printf("NULL buff descriptor to traceroute address. (result = %d)\n", result);

  result = openapiTracerouteAddressWithInterval(clientHandle, &tracerouteIpAddr, 0, &tracerouteOutput);
  printf("Invalid interval between traceroute packets in seconds. (result = %d)\n", result);

  result = openapiTracerouteAddressWithInterval(clientHandle, &tracerouteIpAddr, interval, NULL); 
  printf("Invalid traceroute output. (result = %d)\n", result);

  printf("openapiTracerouteAddressWithInterval() sanity successful. \n");

  /* openapiTracerouteAddressWithTtl() */
  printf("\nTesting openapiTracerouteAddressWithTtl(): \n");

  result = openapiTracerouteAddressWithTtl(NULL, &tracerouteIpAddr, initTtl, maxTtl, &tracerouteOutput);
  printf("NULL Client Handle. (result = %d)\n", result);
  
  result = openapiTracerouteAddressWithTtl(clientHandle, NULL, initTtl, maxTtl, &tracerouteOutput);
  printf("NULL buff descriptor to traceroute address. (result = %d)\n", result);

  result = openapiTracerouteAddressWithTtl(clientHandle, &tracerouteIpAddr, 0, maxTtl, &tracerouteOutput);
  printf("Invalid traceroute initTtl. (result = %d)\n", result);

  result = openapiTracerouteAddressWithTtl(clientHandle, &tracerouteIpAddr, initTtl, 0, &tracerouteOutput);
  printf("Invalid traceroute maxTtl. (result = %d)\n", result);

  result = openapiTracerouteAddressWithTtl(clientHandle, &tracerouteIpAddr, initTtl, maxTtl, NULL); 
  printf("Invalid traceroute output. (result = %d)\n", result);

  printf("openapiTracerouteAddressWithTtl() sanity successful. \n");

  /* openapiTracerouteAddressWithMaxFail() */
  printf("\nTesting openapiTracerouteAddressWithMaxFail(): \n");

  result = openapiTracerouteAddressWithMaxFail(NULL, &tracerouteIpAddr, maxFail, &tracerouteOutput);
  printf("NULL Client Handle. (result = %d)\n", result);
  
  result = openapiTracerouteAddressWithMaxFail(clientHandle, NULL, maxFail, &tracerouteOutput);
  printf("NULL buff descriptor to traceroute address. (result = %d)\n", result);

  result = openapiTracerouteAddressWithMaxFail(clientHandle, &tracerouteIpAddr, 0, &tracerouteOutput);
  printf("Invalid traceroute maxFail. (result = %d)\n", result);

  result = openapiTracerouteAddressWithMaxFail(clientHandle, &tracerouteIpAddr, maxFail, NULL); 
  printf("Invalid traceroute output. (result = %d)\n", result);

  printf("openapiTracerouteAddressWithMaxFail() sanity successful. \n");

  /* openapiTracerouteAddressWithPDUSize() */
  printf("\nTesting openapiTracerouteAddressWithPDUSize(): \n");

  result = openapiTracerouteAddressWithPDUSize(NULL, &tracerouteIpAddr, size, &tracerouteOutput);
  printf("NULL Client Handle. (result = %d)\n", result);
  
  result = openapiTracerouteAddressWithPDUSize(clientHandle, NULL, size, &tracerouteOutput);
  printf("NULL buff descriptor to traceroute address. (result = %d)\n", result);

  result = openapiTracerouteAddressWithPDUSize(clientHandle, &tracerouteIpAddr, 39938, &tracerouteOutput);
  printf("Invalid traceroute size. (result = %d)\n", result);

  result = openapiTracerouteAddressWithPDUSize(clientHandle, &tracerouteIpAddr, size, NULL); 
  printf("Invalid traceroute output. (result = %d)\n", result);

  printf("openapiTracerouteAddressWithPDUSize() sanity successful. \n");


  /* openapiTracerouteAddressWithUdpPort() */
  printf("\nTesting openapiTracerouteAddressWithUdpPort(): \n");

  result = openapiTracerouteAddressWithUdpPort(NULL, &tracerouteIpAddr, port, &tracerouteOutput);
  printf("NULL Client Handle. (result = %d)\n", result);
  
  result = openapiTracerouteAddressWithUdpPort(clientHandle, NULL, port, &tracerouteOutput);
  printf("NULL buff descriptor to traceroute address. (result = %d)\n", result);

  result = openapiTracerouteAddressWithUdpPort(clientHandle, &tracerouteIpAddr, 0, &tracerouteOutput);
  printf("Invalid traceroute UDP port. (result = %d)\n", result);

  result = openapiTracerouteAddressWithUdpPort(clientHandle, &tracerouteIpAddr, port, NULL); 
  printf("Invalid traceroute output. (result = %d)\n", result);

  printf("openapiTracerouteAddressWithUdpPort() sanity successful. \n");

  /* openapiTracerouteAddressWithSourceInterface() */
  printf("\nTesting openapiTracerouteAddressWithSourceInterface(): \n");

  result = openapiTracerouteAddressWithSourceInterface(NULL, &tracerouteIpAddr, srcIntf, &tracerouteOutput);
  printf("NULL Client Handle. (result = %d)\n", result);
  
  result = openapiTracerouteAddressWithSourceInterface(clientHandle, NULL, srcIntf, &tracerouteOutput);
  printf("NULL buff descriptor to traceroute address. (result = %d)\n", result);

  result = openapiTracerouteAddressWithSourceInterface(clientHandle, &tracerouteIpAddr, 10, &tracerouteOutput);
  printf("Invalid traceroute source interface. (result = %d)\n", result);

  result = openapiTracerouteAddressWithSourceInterface(clientHandle, &tracerouteIpAddr, srcIntf, NULL); 
  printf("Invalid traceroute output. (result = %d)\n", result);

  printf("openapiTracerouteAddressWithSourceInterface() sanity successful. \n");

  /* openapiTracerouteAddressExplicit() */
  printf("\nTesting openapiTracerouteAddressExplicit(): \n");
  result = openapiTracerouteAddressExplicit(NULL, &tracerouteIpAddr, count, 
                                            interval, initTtl, maxTtl, maxFail, 
                                            size, port, srcIntf, &tracerouteOutput);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiTracerouteAddressExplicit(clientHandle, NULL, count,
                                            interval, initTtl, maxTtl, maxFail, 
                                            size, port, srcIntf, &tracerouteOutput);
  printf("NULL buff descriptor to traceroute address. (result = %d)\n", result);

  result = openapiTracerouteAddressExplicit(clientHandle, &tracerouteIpAddr, 0,
                                            interval, initTtl, maxTtl, maxFail, 
                                            size, port, srcIntf, &tracerouteOutput);
  printf("Invalid traceroute count. (result = %d)\n", result);

  result = openapiTracerouteAddressExplicit(clientHandle, &tracerouteIpAddr, count, 
                                            0, initTtl, maxTtl, maxFail, 
                                            size, port, srcIntf, &tracerouteOutput);
  printf("Invalid interval between traceroute packets in seconds. (result = %d)\n", result);

  result = openapiTracerouteAddressExplicit(clientHandle, &tracerouteIpAddr, count, 
                                            interval, 0, maxTtl, maxFail, 
                                            size, port, srcIntf, &tracerouteOutput);
  printf("Invalid traceroute initTtl. (result = %d)\n", result);


  result = openapiTracerouteAddressExplicit(clientHandle, &tracerouteIpAddr, count, 
                                            interval, initTtl, 0, maxFail, 
                                            size, port, srcIntf, &tracerouteOutput);
  printf("Invalid traceroute maxTtl. (result = %d)\n", result);

  result = openapiTracerouteAddressExplicit(clientHandle, &tracerouteIpAddr, count, 
                                            interval, initTtl, maxTtl, 0, 
                                            size, port, srcIntf, &tracerouteOutput);
  printf("Invalid traceroute maxFail. (result = %d)\n", result);

  result = openapiTracerouteAddressExplicit(clientHandle, &tracerouteIpAddr, count,
                                            interval, initTtl, maxTtl, maxFail, 
                                            39938, port, srcIntf, &tracerouteOutput);
  printf("Invalid traceroute size. (result = %d)\n", result);

  result = openapiTracerouteAddressExplicit(clientHandle, &tracerouteIpAddr, count, 
                                            interval, initTtl, maxTtl, maxFail, 
                                            size, 0, srcIntf, &tracerouteOutput);
  printf("Invalid traceroute UDP port. (result = %d)\n", result);

  result = openapiTracerouteAddressExplicit(clientHandle, &tracerouteIpAddr, count, 
                                            interval, initTtl, maxTtl, maxFail, 
                                            size, port, 10, &tracerouteOutput);
  printf("Invalid traceroute source interface. (result = %d)\n", result);

  result = openapiTracerouteAddressExplicit(clientHandle, &tracerouteIpAddr, count, 
                                            interval, initTtl, maxTtl, maxFail, 
                                            size, port, srcIntf, NULL);
  printf("Invalid traceroute output. (result = %d)\n", result);

  printf("openapiTracerouteAddressExplicit() sanity successful. \n");

  return;
}
/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  uint32_t arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  
  if (argc < 2)
  {
    printTracerouteAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("traceroute_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Traceroute API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS Version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 3)
      {
        printTracerouteAppMenu();
        exit(1);
      }
      tracerouteAddress(&clientHandle, argv[2]);
      break;

    case 2:
      if (argc != 5)
      {
        printTracerouteAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[4]);
      tracerouteAddressWithCount(&clientHandle, argv[2], arg1);
      break;

    case 3:
      if (argc != 5)
      {
        printTracerouteAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[4]);
      tracerouteAddressWithInterval(&clientHandle, argv[2], arg1);
      break;

    case 4:
      if (argc != 7)
      {
        printTracerouteAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[4]);
      arg2 = atoi(argv[6]);
      tracerouteAddressWithTtl(&clientHandle, argv[2], arg1, arg2);
      break;

    case 5:
      if (argc != 5)
      {
        printTracerouteAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[4]);
      tracerouteAddressWithMaxFail(&clientHandle, argv[2], arg1);
      break;

    case 6:
      if (argc != 5)
      {
        printTracerouteAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[4]);
      tracerouteAddressWithSize(&clientHandle, argv[2], arg1);
      break;

    case 7:
      if (argc != 5)
      {
        printTracerouteAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[4]);
      tracerouteAddressWithPort(&clientHandle, argv[2], arg1);
      break;

    case 8:
      if (argc != 5)
      {
        printTracerouteAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[4]);
      tracerouteAddressWithSourceInterface(&clientHandle, argv[2], arg1);
      break;

    case 9:
      if (argc != 19)
      {
        printTracerouteAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[4]);
      arg2 = atoi(argv[6]);
      arg3 = atoi(argv[8]);
      arg4 = atoi(argv[10]);
      arg5 = atoi(argv[12]);
      arg6 = atoi(argv[14]);
      arg7 = atoi(argv[16]);
      arg8 = atoi(argv[18]);
      tracerouteAddressExplicit(&clientHandle, argv[2], arg1,arg2, arg3, 
                                arg4, arg5, arg6, arg7, arg8);
      break;

    case 10:
      if (argc != 9)
      {
        printTracerouteAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      arg2 = atoi(argv[6]);
      if (OPEN_TRACEROUTE_SOURCE_IP_PORT == arg2)
        arg3 = atoi(argv[7]);
      else
        arg3 = 0;

      if (OPEN_TRACEROUTE_SOURCE_ADDRESS == arg2)
        tracerouteWithSource (&clientHandle, argv[2], arg1, argv[4], arg2, arg3, argv[8]);
      else
        tracerouteWithSource (&clientHandle, argv[2], arg1, argv[4], arg2, arg3, "");

      break;

    case 11:
      if (argc != 23)
      {
        printTracerouteAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      arg2 = atoi(argv[6]);
      arg3 = atoi(argv[8]);
      arg4 = atoi(argv[10]);
      arg5 = atoi(argv[12]);
      arg6 = atoi(argv[14]);
      arg7 = atoi(argv[16]);
      arg8 = atoi(argv[18]);
      arg9 = atoi(argv[20]);

      if (OPEN_TRACEROUTE_SOURCE_IP_PORT == arg9)
        arg10 = atoi(argv[21]);
      else
        arg10 = 0;

      if (OPEN_TRACEROUTE_SOURCE_ADDRESS == arg9)
        traceroute (&clientHandle, argv[2], arg1, argv[4], arg2, arg3, 
                    arg4, arg5, arg6, arg7, arg8, arg9, arg10, argv[22]);
      else
        traceroute (&clientHandle, argv[2], arg1, argv[4], arg2, arg3, 
                    arg4, arg5, arg6, arg7, arg8, arg9, arg10, "");

      break;

    case 12:
      if (argc != 2)
      {
        printTracerouteAppMenu();
        exit(1);
      } 
      tracerouteSanityTest(&clientHandle); 
      break;

    default:
      printTracerouteAppMenu();
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stoptraceroute Traceroute API example application");

  return 0;
}
