/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  user_accounts_example.c
*
* @purpose   User Accounts OpEN APIs Example.
*
* @component OpEN
*
* @comments
*
* @create    13/03/2013
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_user_manager.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the User Accounts Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printUserAccountsAppMenu()
{
  printf("Usage: user_accounts <test#> <arg1> <arg2> ... \n");
  printf("Test 1: Set user name and password: user_accounts_example 1 <username> <password> \n");
  printf("Test 2: Set access level to the user: user_accounts_example 2 <username> <level> \n");
  printf("Test 3: Get the information of users configured:  user_accounts_example 3 \n");
  printf("Test 4: Delete the user account: user_accounts_example 4 <username> \n");
  printf("Test 5: Set the enable password: user_accounts_example 5 <password> \n");
  printf("Test 6: Remove enable password: user_accounts_example 6 \n");
  printf("Test 7: Authenticate login user: user_accounts_example 7 <username> <password> \n");
  printf("Test 8: user_accounts_example OpEN APIs sanity: user_accounts_example 8 \n");
  printf("Test 9: Search the next valid ID of SSH session: user_accounts_example 9 <session type> <start session ID> \n");
  printf("Test 10: Closes the session by ID: user_accounts_example 10 <session ID> \n");
  printf("Test 11: Set user name, password, encryption type and encrypted: user_accounts_example 11 <username> <password> <encryption type> <encrypted>\n");
  printf("Test 12: Get encryption type of user password:  user_accounts_example 12 <username>\n");
  printf("Test 13: Set enable password, encryption type and encrypted: user_accounts_example 13 <password> <encryption type> <encrypted> \n");
  printf("Test 14: Get encryption type of enable password:  user_accounts_example 14 \n");
  printf("Test 15: Validates user provided password: user_accounts_example 15 <username> <password>\n");
  printf("Test 16: Set the lockout count for password: user_accounts_example 16 <count>\n");
  printf("Test 17: Get the lockout count for password: user_accounts_example 17\n");
  printf("Test 18: Set the time for the user account to unlock: user_accounts_example 18 <time>\n");
  printf("Test 19: Get the time for the user account to unlock: user_accounts_example 19\n");
  printf("Test 20: Set the timer mode for the user account to unlock: user_accounts_example 20 <mode>\n");
  printf("Test 21: Get the timer mode for the user account to unlock: user_accounts_example 21\n");
  printf("Test 22: Authenticate login user: user_accounts_example 22 <username> <password> <line> <remoteIp>\n");
  printf("Test 23: Get user lockout status:  user_accounts_example 23 <username>\n");
  return;
}

/*********************************************************************
* @purpose  Configure Username and password
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    userName        @b{(input)}   User name to configure.
* @param    password        @b{(input)}   Password for the user.
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void userNamePasswordSet(openapiClientHandle_t *clientHandle, char *userName, char *password)
{
  open_error_t result;
  open_buffdesc userBuffDesc;
  open_buffdesc passwordBuffDesc;
  uint32_t userNameMax;
  uint32_t passwordMax = OPEN_PASSWORD_SIZE;
  char *userNameStr;
  char *passwordStr;

  if ((result = openapiUserNameSizeGet(clientHandle, &userNameMax)) != OPEN_E_NONE)
  {
    printf("Bad return code while getting the username max. (result = %d)\n", result);
    return;
  }

  if ((userNameStr = (char*)malloc(userNameMax)) == NULL)
  {
    printf("Could not allocate memory for user name.\n");
    return;
  }

  if ((passwordStr = (char*)malloc(passwordMax)) == NULL)
  {
    printf("Could not allocate memory for password.\n");
    free(userNameStr);
    return;
  }

  memset(userNameStr, 0, userNameMax);
  strncpy(userNameStr, userName, userNameMax-1);
  userBuffDesc.pstart = userNameStr;
  userBuffDesc.size = userNameMax-1;

  memset(passwordStr, 0, passwordMax);
  strncpy(passwordStr, password, passwordMax-1);
  passwordBuffDesc.pstart = passwordStr;
  passwordBuffDesc.size = passwordMax-1;

  if ((result = openapiUserSet(clientHandle, &userBuffDesc, &passwordBuffDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set username and password. (result = %d)\n", result);
  }
  else
  {
    printf("Username and password are configured successfully\n");
  }

  free(userNameStr);
  free(passwordStr);

  return;
}

/*********************************************************************
* @purpose  Set privilege level to a user.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    userName        @b{(input)}   User name to configure.
* @param    level           @b{(input)}   User privilege level
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void userLevelSet(openapiClientHandle_t *clientHandle, char *userName, OPEN_LOGIN_ACCESS_LEVELS_t level)
{
  open_error_t result;
  open_buffdesc userBuffDesc;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, userName, (sizeof(str) - 1));
  userBuffDesc.pstart = str;
  userBuffDesc.size = strlen(str)+1;

  if ((result = openapiUserLevelSet(clientHandle, &userBuffDesc, level)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set access level to user. (result = %d)\n", result);
  }
  else
  {
    printf("Access level is set successfully to the user\n");
  }

  return;
}

/*********************************************************************
* @purpose  Get the user information
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void userInfoGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc userBuffDesc;
  uint32_t userNameSize;
  uint32_t maxUsers;
  uint32_t level;
  char *str;


  printf("Getting maximum no. of users supported\n");
  if ((result = openapiMaxUsersSupportedGet(clientHandle, &maxUsers)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get maximum number of users supported. (result = %d)\n", result);
  }
  else
  {
    printf("  Maximum no. of users: %d\n", maxUsers);
  }

  if ((result = openapiUserNameSizeGet(clientHandle, &userNameSize)) != OPEN_E_NONE)
  {
    printf("Bad return code while getting the username size. (result = %d)\n", result);
    return;
  }

  if ((str = (char*)malloc(userNameSize)) == NULL)
  {
    printf("Could not allocate memory.\n");
    return;
  }

  memset(str, 0, userNameSize);
  userBuffDesc.pstart = str;
  userBuffDesc.size = userNameSize;

  printf("Printing the user name and level:\n");
  if ((result = openapiUserFirstGet(clientHandle, &userBuffDesc)) != OPEN_E_NONE)
  {
    if (result == OPEN_E_NOT_FOUND)
    {
      printf("  No users found. (result = %d)\n", result);
    }
    else
    {
      printf("  Bad return code while getting first username. (result = %d)\n", result);
    }
    free(str);
    return;
  }
  do
  {
    printf("  Username: %s", str);
    if (openapiUserLevelGet(clientHandle, &userBuffDesc, &level) == OPEN_E_NONE)
    {
      printf("  Level: %d\n", level); 
    }
    userBuffDesc.size = userNameSize;
  }while((result = openapiUserNextGet(clientHandle, &userBuffDesc, &userBuffDesc)) == OPEN_E_NONE);

  free(str);
  return;
}

/*********************************************************************
* @purpose  Delete user.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    userName        @b{(input)}   User name to configure.
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void userDelete(openapiClientHandle_t *clientHandle, char *userName)
{
  open_error_t result;
  open_buffdesc userBuffDesc;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, userName, (sizeof(str) - 1));
  userBuffDesc.pstart = str;
  userBuffDesc.size = strlen(str)+1;

  if ((result = openapiUserDelete(clientHandle, &userBuffDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete username. (result = %d)\n", result);
  }
  else
  {
    printf("User deleted successfully\n");
  }

  return;
}

/*********************************************************************
* @purpose  Set enable password.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    password        @b{(input)}   Enable password
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void enablePasswordSet(openapiClientHandle_t *clientHandle, char *password)
{
  open_error_t result;
  open_buffdesc pswdBufDesc;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, password, (sizeof(str) - 1));
  pswdBufDesc.pstart = str;
  pswdBufDesc.size = strlen(str)+1;

  if ((result = openapiEnablePasswordSet(clientHandle, &pswdBufDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set enable password. (result = %d)\n", result);
  }
  else
  {
    printf("Enable password set successfully. \n");
  }

  return;
}

/*********************************************************************
* @purpose  Remove enable password.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void enablePasswordRemove(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  if ((result = openapiEnablePasswordRemove(clientHandle)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to remove enable password. (result = %d)\n", result);
  }
  else
  {
    printf("Enable password removed successfully. \n");
  }

  return;
}

/*********************************************************************
* @purpose  Authenticate username and password
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    userName        @b{(input)}   Username to validate
* @param    password        @b{(input)}   Password to Validate
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void authenticateUser(openapiClientHandle_t *clientHandle, char *userName, char *password)
{
  open_error_t result;
  OPEN_LOGIN_ACCESS_LEVELS_t accessLevel;
  open_buffdesc userBuffDesc;
  open_buffdesc pswdBuffDesc;
  char str1[100];
  char str2[100];

  memset(str1, 0, sizeof(str1));
  strncpy(str1, userName, (sizeof(str1) - 1));
  userBuffDesc.pstart = str1;
  userBuffDesc.size = strlen(str1)+1;

  memset(str2, 0, sizeof(str2));
  strncpy(str2, password, (sizeof(str2) - 1));
  pswdBuffDesc.pstart = str2;
  pswdBuffDesc.size = strlen(str2)+1;

  if ((result = openapiAuthenticateUser(clientHandle, &userBuffDesc, &pswdBuffDesc,
                                        &accessLevel)) != OPEN_E_NONE)  
  {
    printf("Bad return code trying to authenticate user. (result = %d)\n", result);
  }
  else
  {
    printf("User authenticated successfully. (result = %d)\n", result); 
    printf("Privilege level of the user: %d \n", accessLevel);
  }

  return;
}

/*********************************************************************
* @purpose  User accounts OpEN API sanity.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void userAccountsOpENAPIsTestSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t userNameSize = 0, passwordMinSize = 0, passwordMaxSize = 0;
  uint32_t maxUsers = 0;
  uint32_t val = 5;
  uint32_t level = 0, encryptType = 0;
  uint32_t minPswdLen = 0, maxPswdLen = 0;
  char *username, *password;
  open_buffdesc userBuffDesc;
  open_buffdesc pswdBuffDesc; 
  OPEN_BOOL_t encrypted = OPEN_FALSE;

  printf("Testing User Accounts OpEN APIs sanity:\n");

  if ((result = openapiUserNameSizeGet(clientHandle, &userNameSize)) != OPEN_E_NONE)
  {
    printf("Bad return code while getting the username size. (result = %d)\n", result);
    return;
  }

  if ((result = openapiPasswordSizeGet(clientHandle, &passwordMinSize, &passwordMaxSize)) != OPEN_E_NONE)
  {
    printf("Bad return code while getting the password size. (result = %d)\n", result);
    return;
  }

  if ((username = (char *)malloc(userNameSize)) == NULL)
  {
    printf("Could not allocate memory\n");
    return;
  }
  memset(username, 0, sizeof(userNameSize));
  userBuffDesc.pstart = username;
  userBuffDesc.size = userNameSize; 

  if ((password = (char *)malloc(passwordMaxSize)) == NULL)
  {
    printf("Could not allocate memory\n");
    free(username);
    return;
  }
  memset(password, 0, passwordMaxSize);
  pswdBuffDesc.pstart = password;
  pswdBuffDesc.size = passwordMaxSize;
  

  /* openapiUserNameSizeGet() */
  printf("\nTesting openapiUserNameSizeGet(): \n");
  result = openapiUserNameSizeGet(NULL, &userNameSize);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiUserNameSizeGet(clientHandle, NULL);
  printf("NULL username size. (result = %d)\n", result);

  printf("openapiUserNameSizeGet() sanity successful.\n");

  /* openapiPasswordSizeGet() */
  printf("\nTesting openapiPasswordSizeGet(): \n");

  result = openapiPasswordSizeGet(NULL, &passwordMinSize, &passwordMaxSize);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPasswordSizeGet(clientHandle, NULL, &passwordMaxSize);
  printf("NULL minimum password length. (result = %d)\n", result);

  result = openapiPasswordSizeGet(clientHandle, &passwordMinSize, NULL);
  printf("NULL maximum password length. (result = %d)\n", result);

  printf("openapiPasswordSizeGet() sanity successful.\n");
   
  /* openapiUserSet() */
  printf("\nTesting openapiUserSet(): \n");

  result = openapiUserSet(NULL, &userBuffDesc, &pswdBuffDesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiUserSet(clientHandle, NULL, &pswdBuffDesc);
  printf("NULL buff descriptor to user name. (result = %d)\n", result);

  result = openapiUserSet(clientHandle, &userBuffDesc, NULL);\
  printf("NULL buff descriptor to password. (result = %d)\n", result);

  printf("openapiUserSet() sanity successful.\n");

  /* openapiUserFirstGet() */
  printf("\nTesting openapiUserFirstGet() \n");
  result = openapiUserFirstGet(NULL, &userBuffDesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiUserFirstGet(clientHandle, NULL);
  printf("NULL buff descriptor to user name. (result = %d)\n", result);

  printf("openapiUserFirstGet() sanity successful.\n");


  /* openapiUserNextGet() */
  printf("\nTesting openapiUserNextGet(): \n");

  result = openapiUserNextGet(NULL, &userBuffDesc, &userBuffDesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiUserNextGet(clientHandle, NULL, &userBuffDesc);
  printf("NULL buff descriptor to previous user name. (result = %d)\n", result);

  result = openapiUserNextGet(clientHandle, &userBuffDesc, NULL);
  printf("NULL buff descriptor to next user name. (result = %d)\n", result);

  printf("openapiUserNextGet() sanity successful. \n");
  

  /* openapiMaxUsersSupportedGet() */
  printf("\nTesting openapiMaxUsersSupportedGet(): \n");
  result = openapiMaxUsersSupportedGet(NULL, &maxUsers);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiMaxUsersSupportedGet(clientHandle, NULL);
  printf("NULL parameter to maximum users. (result = %d)\n", result);

  printf("openapiMaxUsersSupportedGet() sanity successful.\n");

  /* openapiUserLevelSet() */
  printf("\nTesting openapiUserLevelSet(): \n");
  result = openapiUserLevelSet(NULL, &userBuffDesc, level);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiUserLevelSet(clientHandle, NULL, level);
  printf("NULL buff descriptor to user name. (result = %d)\n", result);

  result = openapiUserLevelSet(clientHandle, &userBuffDesc, 20);
  printf("Invalid user level. (result = %d)\n", result);

  printf("openapiUserLevelSet() sanity successful. \n");

  /* openapiUserLevelGet() */ 
  printf("\nTesting openapiUserLevelGet(): \n");

  result = openapiUserLevelGet(NULL, &userBuffDesc, &level);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiUserLevelGet(clientHandle, NULL, &level);
  printf("NULL buff descriptor to user name. (result = %d)\n", result);

  result = openapiUserLevelGet(clientHandle, &userBuffDesc, NULL);
  printf("NULL user level. (result = %d)\n", result);

  printf("openapiUserLevelGet() sanity successful. \n");

  /* openapiUserNameValidate() */
  printf("\nTesting openapiUserNameValidate(): \n");

  result = openapiUserNameValidate(NULL, &userBuffDesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiUserNameValidate(clientHandle, NULL);
  printf("NULL buff descriptor to user name. (result = %d)\n", result);

  printf("openapiUserNameValidate() sanity successful. \n");

  /* openapiPasswordSizeGet() */
  printf("\nTesting openapiPasswordSizeGet(): \n");

  result = openapiPasswordSizeGet(NULL, &minPswdLen, &maxPswdLen);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPasswordSizeGet(clientHandle, NULL, &maxPswdLen);
  printf("NULL parameter to passowrd minimum length. (result = %d)\n", result);

  result = openapiPasswordSizeGet(clientHandle, &minPswdLen, NULL);
  printf("NULL parameter to passowrd maximum length. (result = %d)\n", result);

  printf("openapiPasswordSizeGet() sanity successful. \n");

  /* openapiUserDelete() */
  printf("\nTesting openapiUserDelete(): \n");

  result = openapiUserDelete(NULL, &userBuffDesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiUserDelete(clientHandle, NULL);
  printf("NULL buff descriptor to user name. (result = %d)\n", result);

  printf("openapiUserDelete() sanity successful. \n");

  /* openapiAuthenticateUser() */
  printf("\nTesting openapiAuthenticateUser(): \n");
  result = openapiAuthenticateUser(NULL, &userBuffDesc, &pswdBuffDesc, &level);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAuthenticateUser(clientHandle, NULL, &pswdBuffDesc, &level);
  printf("NULL buff descriptor to user name. (result = %d)\n", result);

  result = openapiAuthenticateUser(clientHandle, &userBuffDesc, NULL, &level);
  printf("NULL buff descriptor to password. (result = %d)\n", result);

  result = openapiAuthenticateUser(clientHandle, &userBuffDesc, &pswdBuffDesc, NULL);
  printf("NULL parameter to access level. (result = %d)\n", result);

  printf("openapiAuthenticateUser() sanity successful. \n");
 

  /* openapiUserPasswordEncryptTypeSet() */
  printf("\nTesting openapiUserPasswordEncryptTypeSet(): \n");
  result = openapiUserPasswordEncryptTypeSet(NULL, &userBuffDesc, &pswdBuffDesc, encryptType, encrypted);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiUserPasswordEncryptTypeSet(clientHandle, NULL, &pswdBuffDesc, encryptType, encrypted);
  printf("NULL buff descriptor to user name. (result = %d)\n", result);

  result = openapiUserPasswordEncryptTypeSet(clientHandle, &userBuffDesc, NULL, encryptType, encrypted);
  printf("NULL buff descriptor to password. (result = %d)\n", result);

  result = openapiUserPasswordEncryptTypeSet(clientHandle, &userBuffDesc, &pswdBuffDesc, 10, encrypted);
  printf("Invalid user password encryption type. (result = %d)\n", result);

  result = openapiUserPasswordEncryptTypeSet(clientHandle, &userBuffDesc, &pswdBuffDesc, encryptType, 3);
  printf("Invalid encrypted flag value. (result = %d)\n", result);

  printf("openapiUserPasswordEncryptTypeSet() sanity successful. \n");

  /* openapiUserPasswordEncryptTypeGet() */
  printf("\nTesting openapiUserPasswordEncryptTypeGet(): \n");

  result = openapiUserPasswordEncryptTypeGet(NULL, &userBuffDesc, &encryptType);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiUserPasswordEncryptTypeGet(clientHandle, NULL, &encryptType);
  printf("NULL buff descriptor to user name. (result = %d)\n", result);

  result = openapiUserPasswordEncryptTypeGet(clientHandle, &userBuffDesc, NULL);
  printf("NULL user password encryption type. (result = %d)\n", result);

  printf("openapiUserPasswordEncryptTypeGet() sanity successful. \n");

  /* openapiEnablePasswordEncryptTypeSet() */
  printf("\nTesting openapiEnablePasswordEncryptTypeSet(): \n");
  result = openapiEnablePasswordEncryptTypeSet(NULL, &pswdBuffDesc, encryptType, encrypted);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiEnablePasswordEncryptTypeSet(clientHandle, NULL, encryptType, encrypted);
  printf("Invalid enable password. (result = %d)\n", result);

  result = openapiEnablePasswordEncryptTypeSet(clientHandle, &pswdBuffDesc, 10, encrypted);
  printf("Invalid enable password encryption type. (result = %d)\n", result);

  result = openapiEnablePasswordEncryptTypeSet(clientHandle, &pswdBuffDesc, encryptType, 3);
  printf("Invalid encrypted flag. (result = %d)\n", result);

  printf("openapiEnablePasswordEncryptTypeSet() sanity successful. \n");

  /* openapiEnablePasswordEncryptTypeGet() */
  printf("\nTesting openapiEnablePasswordEncryptTypeGet(): \n");

  result = openapiEnablePasswordEncryptTypeGet(NULL, &encryptType);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiEnablePasswordEncryptTypeGet(clientHandle, NULL);
  printf("NULL enable password encryption type. (result = %d)\n", result);

  printf("openapiEnablePasswordEncryptTypeGet() sanity successful. \n");

  free(username);
  free(password);
  
  /* openapiUserLockoutAttemptsSet() */
  printf("\nTesting openapiUserLockoutAttemptsSet(): \n");
  result = openapiUserLockoutAttemptsSet(NULL, val);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiUserLockoutAttemptsSet(clientHandle, 256);
  printf("Invalid lockout attempts. (result = %d)\n", result);
  printf("openapiUserLockoutAttemptsSet() sanity successful. \n");

  /* openapiUserLockoutAttemptsGet() */
  printf("\nTesting openapiUserLockoutAttemptsGet(): \n");
  result = openapiUserLockoutAttemptsGet(NULL, &val);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiUserLockoutAttemptsGet(clientHandle, NULL);
  printf("NULL lockout attempts. (result = %d)\n", result);
  printf("openapiUserLockoutAttemptsGet() sanity successful. \n");

  /* openapiUserUnlockTimeSet() */
  printf("\nTesting openapiUserUnlockTimeSet(): \n");
  result = openapiUserUnlockTimeSet(NULL, val);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiUserUnlockTimeSet(clientHandle, 256);
  printf("Invalid unlock timer value. (result = %d)\n", result);
  printf("openapiUserUnlockTimeSet() sanity successful. \n");
  
  /* openapiUserUnlockTimeGet() */
  printf("\nTesting openapiUserUnlockTimeGet(): \n");
  result = openapiUserUnlockTimeGet(NULL, &val);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiUserUnlockTimeGet(clientHandle, NULL);
  printf("NULL unlock timer value. (result = %d)\n", result);
  printf("openapiUserUnlockTimeGet() sanity successful. \n");
  
  /* openapiUserUnlockTimerModeSet() */
  printf("\nTesting openapiUserUnlockTimerModeSet(): \n");
  result = openapiUserUnlockTimerModeSet(NULL, val);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiUserUnlockTimerModeSet(clientHandle, 3);
  printf("Invalid unlock timer mode. (result = %d)\n", result);
  printf("openapiUserUnlockTimerModeSet() sanity successful. \n");

  /* openapiUserUnlockTimerModeGet() */
  printf("\nTesting openapiUserUnlockTimerModeGet(): \n");
  result = openapiUserUnlockTimerModeGet(NULL, &val);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiUserUnlockTimerModeGet(clientHandle, NULL);
  printf("NULL unlock timer mode. (result = %d)\n", result);
  printf("openapiUserUnlockTimerModeGet() sanity successful. \n");

  return;
}

/*********************************************************************
* @purpose  Search the next valid ID of session
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void sessionIDNextValidGetTest(openapiClientHandle_t *clientHandle, uint32_t sessionType, uint32_t sessionID)
{
  open_error_t result;
  OPEN_LOGIN_TYPE_t sessionLoginType = (OPEN_LOGIN_TYPE_t)sessionType;
  uint32_t SID = sessionID;
  char buf[256];

  switch (sessionLoginType)
  {
    case OPEN_LOGIN_TYPE_SERIAL:
      sprintf(buf, "%s", "SERIAL");
      break;
    case OPEN_LOGIN_TYPE_TELNET:
      sprintf(buf, "%s", "TELNET");
      break;
    case OPEN_LOGIN_TYPE_SSH:
      sprintf(buf, "%s", "SSH");
      break;
    case OPEN_LOGIN_TYPE_HTTP:
      sprintf(buf, "%s", "HTTP");
      break;
    case OPEN_LOGIN_TYPE_HTTPS:
      sprintf(buf, "%s", "HTTPS");
      break;
    default:
      sprintf(buf, "%s", "UNKNOWN");
      break;
  }

  do
  {
    result = openapiSessionIDNextValidGet(clientHandle, sessionLoginType, &SID);
    switch (result)
    {
      case OPEN_E_NONE:
        printf("The next ID of %s session is returned successfully. (ID = %u, result = %d)\n", buf, SID, result);
        break;
      case OPEN_E_NOT_FOUND:
        printf("The valid next ID of %s session wasn't found. (result = %d)\n", buf, result);
        break;
      case OPEN_E_PARAM:
        printf("Invalid parameter is passed. (result = %d)\n", result);
        break;
      default:
        break;
    }
  } while (result == OPEN_E_NONE);
}

/*********************************************************************
* @purpose  Closes the session by ID
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void sessionLogoutTest(openapiClientHandle_t *clientHandle, uint32_t sessionID)
{
  open_error_t result;

  result = openapiSessionLogout(clientHandle, sessionID);
  switch (result)
  {
    case OPEN_E_NONE:
      printf("The logout of session ID:%u was successful. (result = %d)\n", sessionID, result);
      break;
    case OPEN_E_FAIL:
      printf("The logout of session ID:%u failed. (result = %d)\n", sessionID, result);
      break;
    case OPEN_E_PARAM:
      printf("Invalid parameter is passed. (result = %d)\n", result);
      break;
    default:
      break;
  }
}

/*********************************************************************
* @purpose  Configure Username, password and encryption type
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    userName        @b{(input)}   User name to configure
* @param    password        @b{(input)}   Password for the user
* @param    encryptType     @b{(input)}   Encryption type 
* @param    encrypted       @b{(input)}   flag indicating whether the password param is encrypted
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void userPasswordEncryptTypeSet(openapiClientHandle_t *clientHandle, char *userName,
                               char *password, OPEN_ENCRYPT_TYPE_t encryptType, OPEN_BOOL_t encrypted)
{
  open_error_t result;
  open_buffdesc userBuffDesc;
  open_buffdesc passwordBuffDesc;
  uint32_t userNameMax;
  uint32_t passwordMin;
  uint32_t passwordMax;
  char *userNameStr;
  char *passwordStr;

  if ((result = openapiUserNameSizeGet(clientHandle, &userNameMax)) != OPEN_E_NONE)
  {
    printf("Bad return code while getting the username max. (result = %d)\n", result);
    return;
  }

  if ((result = openapiPasswordSizeGet(clientHandle, &passwordMin, &passwordMax)) != OPEN_E_NONE)
  {
    printf("Bad return code while getting the password max. (result = %d)\n", result);
    return;
  }

  if ((userNameStr = (char*)malloc(userNameMax)) == NULL)
  {
    printf("Could not allocate memory for user name.\n");
    return;
  }

  if ((passwordStr = (char*)malloc(passwordMax)) == NULL)
  {
    printf("Could not allocate memory for password.\n");
    free(userNameStr);
    return;
  }

  memset(userNameStr, 0, userNameMax);
  strncpy(userNameStr, userName, userNameMax-1);
  userBuffDesc.pstart = userNameStr;
  userBuffDesc.size = userNameMax-1;

  memset(passwordStr, 0, passwordMax);
  strncpy(passwordStr, password, passwordMax-1);
  passwordBuffDesc.pstart = passwordStr;
  passwordBuffDesc.size = passwordMax-1;

  if ((result = openapiUserPasswordEncryptTypeSet(clientHandle, &userBuffDesc, &passwordBuffDesc, encryptType, encrypted)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set username, password and encryption type. (result = %d)\n", result);
  }
  else
  {
    printf("Username, password and encryption type are configured successfully\n");
  }

  free(userNameStr);
  free(passwordStr);

  return;
}

/*********************************************************************
* @purpose  Get password encryption type for a given user.
* 
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    userName        @b{(input)}   user name
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void userPasswordEncryptTypeGet(openapiClientHandle_t *clientHandle, char *userName)
{
  open_error_t result;
  OPEN_ENCRYPT_TYPE_t encryptType;
  open_buffdesc userBuffDesc;
  char str[OPEN_LOGIN_SIZE];
  
  memset(str, 0, sizeof(str));
  strncpy(str, userName, (sizeof(str) - 1));
  userBuffDesc.pstart = str;
  userBuffDesc.size = strlen(str)+1;

  if ((result = openapiUserPasswordEncryptTypeGet(clientHandle, &userBuffDesc, &encryptType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get password encryption type. (result = %d)\n", result);
  }
  else
  {
    printf("Password encryption type: %d (result = %d)\n", encryptType, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set enable password and encryption type.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    password        @b{(input)}   Enable password
* @param    encryptType     @b{(input)}   Encryption type
* @param    encrypted       @b{(input)}   flag indicating whether the password param is encrypted
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void enablePasswordEncryptTypeSet(openapiClientHandle_t *clientHandle, char *password,
                                  OPEN_ENCRYPT_TYPE_t encryptType, OPEN_BOOL_t encrypted)
{
  open_error_t result;
  open_buffdesc pswdBufDesc;
  char str[256];

  memset(str, 0, sizeof(str));
  strncpy(str, password, (sizeof(str) - 1));
  pswdBufDesc.pstart = str;
  pswdBufDesc.size = strlen(str)+1;

  if ((result = openapiEnablePasswordEncryptTypeSet(clientHandle, &pswdBufDesc, encryptType, encrypted)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set enable password and encryption type. (result = %d)\n", result);
  }
  else
  {
    printf("Enable password and encryption type are set successfully. \n");
  }

  return;
}

/*********************************************************************
* @purpose  Get enable password encryption type.
* 
* @param    clientHandle    @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void enablePasswordEncryptTypeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_ENCRYPT_TYPE_t encryptType;

  if ((result = openapiEnablePasswordEncryptTypeGet(clientHandle, &encryptType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get enable password encryption type. (result = %d)\n", result);
  }
  else
  {
    printf("Enable password encryption type: %d (result = %d)\n", encryptType, result);
  }

  return;
}

/*********************************************************************
* @purpose  Validates user provided password.
*
* @param    client_handle     @b{(input)}  client handle from registration API
* @param    pUser             @b{(input)}  user name
* @param    pEnteredPwd       @b{(input)}  user provided password

*
* @returns  none
*
* @end
*********************************************************************/
void validateUserPassword(openapiClientHandle_t *client_handle,
                          open_buffdesc *pUser,
                          open_buffdesc *pEnteredPwd)
{
  open_error_t result;

  if ((result = openapiValidateUserPassword(client_handle, pUser, pEnteredPwd)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to validates user provided password. (result = %d)\n", result);
  }
  else
  {
    printf("Successfully validated password for user :%s.\n", (char*)pUser->pstart);
  }
  return;
}

/*********************************************************************
 * @purpose  Set lockout count for password.
 * 
 * @param    clientHandle    @b{(input)}   client handle from registration
 *                                         API
 * @param    lockoutCount    @b{(input)}   lockout count
 * 
 * @returns  none
 *   
 * @notes
 * 
 * @end
 *********************************************************************/
void userLockoutAttemptsSet(openapiClientHandle_t *clientHandle,
                            uint32_t lockoutCount)
{
  open_error_t result;

  if ((result = openapiUserLockoutAttemptsSet(clientHandle, lockoutCount)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set lockout count:%d. (result = %d)\n",
           lockoutCount, result);
  }
  else
  {
    printf("Lockout Count:%d is configured successfully.\n", lockoutCount);
  }

  return;
}

/*********************************************************************
 * @purpose  Get lockout count for password.
 * 
 * @param    clientHandle    @b{(input)}   client handle from registration
 *                                         API
 * 
 * @returns  none
 *   
 * @notes
 * 
 * @end
 *********************************************************************/
void userLockoutAttemptsGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t lockoutCount = 0;

  if ((result = openapiUserLockoutAttemptsGet(clientHandle, &lockoutCount)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get lockout count. (result = %d)\n",
           result);
  }
  else
  {
    printf("Lockout Count:%d\n", lockoutCount);
  }

  return;
}

/*********************************************************************
 * @purpose  Set unlock time for the user to unlock.
 * 
 * @param    clientHandle    @b{(input)}   client handle from registration
 *                                         API
 * @param    unlockTime      @b{(input)}   unlock time
 * 
 * @returns  none
 *   
 * @notes
 * 
 * @end
 *********************************************************************/
void userUnlockTimeSet(openapiClientHandle_t *clientHandle,
                       uint32_t unlockTime)
{
  open_error_t result;

  if ((result = openapiUserUnlockTimeSet(clientHandle, unlockTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set unlock time:%d. (result = %d)\n",
           unlockTime, result);
  }
  else
  {
    printf("Unlock time:%d minutes is configured successfully.\n", unlockTime);
  }

  return;
}

/*********************************************************************
 * @purpose  Get unlock time for the user to unlock.
 * 
 * @param    clientHandle    @b{(input)}   client handle from registration
 *                                         API
 * 
 * @returns  none
 *   
 * @notes
 * 
 * @end
 *********************************************************************/
void userUnlockTimeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t unlockTime = 0;

  if ((result = openapiUserUnlockTimeGet(clientHandle, &unlockTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get unlock time. (result = %d)\n",
           result);
  }
  else
  {
    printf("Unlock Time (mins):%d\n", unlockTime);
  }

  return;
}

/*********************************************************************
 * @purpose  Set unlock timer mode for the user to unlock.
 * 
 * @param    clientHandle    @b{(input)}   client handle from registration
 *                                         API
 * @param    unlockTimerMode @b{(input)}   unlock timer mode
 * 
 * @returns  none
 *   
 * @notes
 * 
 * @end
 *********************************************************************/
void userUnlockTimerModeSet(openapiClientHandle_t *clientHandle,
                            uint32_t unlockTimerMode)
{
  open_error_t result;

  if ((result = openapiUserUnlockTimerModeSet(clientHandle, unlockTimerMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set unlock timer:%s. (result = %d)\n",
           (unlockTimerMode == OPEN_ENABLE) ? "Enable" : "Disable", result);
  }
  else
  {
    printf("Unlock timer mode:%s is configured successfully.\n", (unlockTimerMode == OPEN_ENABLE) ? "Enable" : "Disable");
  }

  return;
}

/*********************************************************************
 * @purpose  Get unlock timer mode for the user to unlock.
 * 
 * @param    clientHandle    @b{(input)}   client handle from registration
 *                                         API
 * 
 * @returns  none
 *   
 * @notes
 * 
 * @end
 *********************************************************************/
void userUnlockTimerModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t unlockTimerMode = 0;

  if ((result = openapiUserUnlockTimerModeGet(clientHandle, &unlockTimerMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get unlock timer mode. (result = %d)\n",
           result);
  }
  else
  {
    printf("Unlock Timer Mode:%s\n", (unlockTimerMode == OPEN_ENABLE) ? "Enable" : "Disable");
  }

  return;
}

static char *getAccessLevelStr(OPEN_LOGIN_ACCESS_LEVELS_t level)
{
  switch (level)
  {
    case OPEN_LOGIN_ACCESS_NONE:
      return "None";
    case OPEN_LOGIN_ACCESS_READ_ONLY:
      return "Read-Only";
    case OPEN_LOGIN_ACCESS_READ_WRITE:
      return "ReadWrite";
    default:
      return "Unknown";
  }
  return "";
}

static char *getAuthMethodStr(OPEN_USER_MGR_AUTH_METHOD_t authMethod)
{
  switch (authMethod)
  {
    case OPEN_AUTH_METHOD_LOCAL:
      return "Local";
    case OPEN_AUTH_METHOD_LINE:
      return "Line";
    case OPEN_AUTH_METHOD_ENABLE:
      return "Enable";
    case OPEN_AUTH_METHOD_NONE:
      return "None";
    case OPEN_AUTH_METHOD_RADIUS:
      return "Radius";
    case OPEN_AUTH_METHOD_TACACS:
      return "Tacacs";
    default:
      return "Unknown";
  }
  return "";
}

/*********************************************************************
* @purpose  Authenticate username and password.
*
* @param    clientHandle    @b{(input)}   client handle from registration API.
* @param    userName        @b{(input)}   Username to validate.
* @param    password        @b{(input)}   Password to Validate.
* @param    accessLine      @b{(input)}   Access line.
* @param    remoteIp        @b{(input)}   Remote IP address.
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void userAuthentication(openapiClientHandle_t *clientHandle,
                        char *userName,
                        char *password,
                        uint32_t accessLine, 
                        char *remoteIp)
{
  open_error_t result;
  open_buffdesc userBuffDesc;
  open_buffdesc pswdBuffDesc;
  open_buffdesc remoteIpBuffDesc;
  char str1[OPEN_LOGIN_SIZE];
  char str2[OPEN_PASSWORD_SIZE];
  char str3[256];
  OPEN_LOGIN_ACCESS_LEVELS_t accessLevel = OPEN_LOGIN_ACCESS_NONE;
  OPEN_USER_MGR_AUTH_METHOD_t authMethod = OPEN_AUTH_METHOD_UNDEFINED;
  OPEN_BOOL_t isDefPassword = OPEN_FALSE;

  memset(str1, 0, sizeof(str1));
  memset(str2, 0, sizeof(str2));
  memset(str3, 0, sizeof(str3));

  snprintf(str1, sizeof(str1), "%s", userName);
  userBuffDesc.pstart = str1;
  userBuffDesc.size =   strlen(str1) + 1;

  snprintf(str2, sizeof(str2), "%s", password);
  pswdBuffDesc.pstart = str2;
  pswdBuffDesc.size =   strlen(str2) + 1;

  snprintf(str3, sizeof(str3), "%s", remoteIp);
  remoteIpBuffDesc.pstart = str3;
  remoteIpBuffDesc.size   = strlen(str3) + 1;

  if ((result = openapiUserSessionAuthentication(clientHandle, &userBuffDesc, &pswdBuffDesc,
                                                 accessLine, &remoteIpBuffDesc, &accessLevel,
                                                 &authMethod, &isDefPassword)) != OPEN_E_NONE)  
  {
    printf("Bad return code trying to authenticate user:%s. (result = %d)\n",
           userName, result);
  }
  else
  {
    printf("\r\nUser: %s\r\n", userName);
    printf("Privilege level: %s(%d)\r\n", getAccessLevelStr(accessLevel), accessLevel);
    printf("Authentication Method: %s(%d)\r\n", getAuthMethodStr(authMethod), authMethod);
    printf("Default Password: %s(%d)\r\n", (isDefPassword == OPEN_TRUE) ? "Yes" : "No", isDefPassword);
  }

  return;
}

/*********************************************************************
* @purpose  Get lockout status for a given user.
* 
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    userName        @b{(input)}   user name
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void userLockoutStatusGet(openapiClientHandle_t *clientHandle, char *userName)
{
  open_error_t result;
  OPEN_BOOL_t lockoutStatus = OPEN_FALSE;
  open_buffdesc userBuffDesc;
  char str[OPEN_LOGIN_SIZE];
  
  memset(str, 0, sizeof(str));
  strncpy(str, userName, (sizeof(str) - 1));
  userBuffDesc.pstart = str;
  userBuffDesc.size = strlen(str)+1;

  if ((result = openapiUserLockoutStatusGet(clientHandle, &userBuffDesc,
                                            &lockoutStatus)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get lockout status for user:%s. (result = %d)\n",
           userName, result);
  }
  else
  {
    printf("\r\nUser: %s\r\n", userName);
    printf("Lockout Status: %s\r\n", (OPEN_TRUE == lockoutStatus) ? "Yes" : "No");
  }

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum, arg1, arg2;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  char usrNameBuf[128] = {0};
  char pwdBuf[128] = {0};
  open_buffdesc usrBufDesc = {.pstart = usrNameBuf, .size = sizeof(usrNameBuf)};
  open_buffdesc pwdBufDesc = {.pstart = pwdBuf, .size = sizeof(pwdBuf)};

  if (argc < 2)
  {
    printUserAccountsAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);
  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("user_accounts_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting User Accounts API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
     if (argc != 4)
     {
       printUserAccountsAppMenu();
       exit(1);
     }
     userNamePasswordSet(&clientHandle, argv[2], argv[3]); 
     break;

    case 2:
      if (argc != 4)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      userLevelSet(&clientHandle, argv[2], arg1);
      break;

    case 3:
      if (argc != 2)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      userInfoGet(&clientHandle);
      break;

    case 4:
      if (argc != 3)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      userDelete(&clientHandle, argv[2]);
      break;

    case 5:
      if (argc != 3)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      enablePasswordSet(&clientHandle, argv[2]);  
      break;

    case 6:
      if (argc != 2)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      enablePasswordRemove(&clientHandle);
      break;

    case 7:
      if (argc != 4)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      authenticateUser(&clientHandle, argv[2], argv[3]);
      break;

    case 8:
      if (argc != 2)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      userAccountsOpENAPIsTestSanity(&clientHandle);
      break;

    case 9:
      if (argc != 4)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      sessionIDNextValidGetTest(&clientHandle, arg1, arg2);
      break;

    case 10:
      if (argc != 3)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      sessionLogoutTest(&clientHandle, arg1);
      break;

    case 11:
      if (argc != 6)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[4]);
      arg2 = atoi(argv[5]);
      userPasswordEncryptTypeSet(&clientHandle, argv[2], argv[3], arg1, arg2);
      break;

     case 12:
      if (argc != 3)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      userPasswordEncryptTypeGet(&clientHandle, argv[2]);
      break;

    case 13:
      if (argc != 5)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      arg2 = atoi(argv[4]);
      enablePasswordEncryptTypeSet(&clientHandle, argv[2], arg1, arg2);
      break;

     case 14:
      if (argc != 2)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      enablePasswordEncryptTypeGet(&clientHandle);
      break;

     case 15:
      if (argc != 4)
      {
        printUserAccountsAppMenu();
        exit(1);
      }

      memset(usrNameBuf, 0, sizeof(usrNameBuf));
      strncpy(usrNameBuf, argv[2], sizeof(usrNameBuf)-1);
      usrBufDesc.size = strlen(usrBufDesc.pstart) + 1;

      memset(pwdBuf, 0, sizeof(pwdBuf));
      strncpy(pwdBuf, argv[3], sizeof(pwdBuf)-1);
      pwdBufDesc.size = strlen(pwdBufDesc.pstart) + 1;

      validateUserPassword(&clientHandle, &usrBufDesc, &pwdBufDesc);
      break;

     case 16:
      if (argc != 3)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      userLockoutAttemptsSet(&clientHandle, atoi(argv[2]));
      break;

     case 17:
      if (argc != 2)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      userLockoutAttemptsGet(&clientHandle);
      break;

     case 18:
      if (argc != 3)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      userUnlockTimeSet(&clientHandle, atoi(argv[2]));
      break;

     case 19:
      if (argc != 2)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      userUnlockTimeGet(&clientHandle);
      break;

     case 20:
      if (argc != 3)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      userUnlockTimerModeSet(&clientHandle, atoi(argv[2]));
      break;

     case 21:
      if (argc != 2)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      userUnlockTimerModeGet(&clientHandle);
      break;

    case 22:
      if (argc != 6)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      userAuthentication(&clientHandle, argv[2], argv[3], atoi(argv[4]),argv[5]);
      break;

     case 23:
      if (argc != 3)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      userLockoutStatusGet(&clientHandle, argv[2]);
      break;

    default:
       printUserAccountsAppMenu();
       break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping User Accounts API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}

