/**********************************************************************
*
*  Copyright 2016-2024 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_boxs.h
*
* @purpose   Box Services Configuration and Status
*
* @component OPEN
*
* @create    11/09/2012
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_BOXS_H_INCLUDED
#define OPENAPI_BOXS_H_INCLUDED

#include "openapi_common.h"

/** Item state definition - for power supplies and fans */
typedef enum
{
  OPEN_ITEMSTATE_NONE           = 0, /* N/A */
  OPEN_ITEMSTATE_NOT_PRESENT    = 1,
  OPEN_ITEMSTATE_OPERATIONAL    = 2,
  OPEN_ITEMSTATE_FAILED         = 3,
  OPEN_ITEMSTATE_POWERING       = 4,
  OPEN_ITEMSTATE_NOPOWER        = 5,
  OPEN_ITEMSTATE_NOT_POWERING   = 6,
  OPEN_ITEMSTATE_INCOMPATIBLE   = 7,  /* This state is possible on boards capable of pluggable Power supplies */
  OPEN_ITEMSTATE_WARNING        = 8,
  OPEN_ITEMSTATE_PRESENT        = 9   /* Added for detecting Xcvr inserted state */ 
} OPEN_BOXS_ITEM_STATE_t;

/** Item state definition - for thermal sensors */
typedef enum
{
  OPEN_TEMPSENSOR_STATE_NONE            = 0, /* N/A */
  OPEN_TEMPSENSOR_STATE_LOW             = 1, 
  OPEN_TEMPSENSOR_STATE_NORMAL          = 2,
  OPEN_TEMPSENSOR_STATE_WARNING         = 3,
  OPEN_TEMPSENSOR_STATE_CRITICAL        = 4,
  OPEN_TEMPSENSOR_STATE_SHUTDOWN        = 5,
  OPEN_TEMPSENSOR_STATE_NOTPRESENT      = 6,
  OPEN_TEMPSENSOR_STATE_NOTOPERATIONAL  = 7
} OPEN_BOXS_TEMPSENSOR_STATE_t;

/** Item type definitions */
typedef enum
{
  OPEN_TYPE_NONE        = 0, /* N/A */
  OPEN_TYPE_FIXED       = 1,
  OPEN_TYPE_REMOVABLE   = 2
} OPEN_BOXS_ITEM_TYPE_t;

typedef enum
{
  OPEN_MODULE_NONE      = 0, /**< None */
  OPEN_MODULE_SFP       = 1, /**< SFP */ 
  OPEN_MODULE_SFPPLUS   = 2, /**< SFP+ */
  OPEN_MODULE_QSFP      = 3, /**< QSFP */
  OPEN_MODULE_DAC       = 4, /**< Direct Attach Cable */
  OPEN_MODULE_AX741     = 5, /**< XFP, AX741, 10G plugin module */
  OPEN_MODULE_AX742     = 6, /**< Stacking module, AX742 */
  OPEN_MODULE_AX743     = 7, /**< SFP+ plugin module, AX743 */
  OPEN_MODULE_AX744     = 8, /**< CX4 plugin module, AX744 */
  OPEN_MODULE_AX745     = 9, /**< Copper 10G plugin module, AX745 */
  OPEN_MODULE_HDMI      = 10  /**< HDMI */
}OPEN_SFP_MODULE_TYPE_t;

typedef enum
{
  OPEN_LED_CMD_OFF = 0,
  OPEN_LED_CMD_ON = 1
}OPEN_LED_CMD_t;

typedef enum
{
  OPEN_LED_STATUS_OFF = 0,
  OPEN_LED_STATUS_ON = 1
}OPEN_LED_STATUS_t;      

#define OPEN_BXS_STR_BUFFER_SIZE 128    /**< 127 chars + terminating nul */

/**********************************************************************************
* @purpose  Get the fault status of fiber port optical transceiver.
*
* @param    client_handle        @b{(input)}   client handle from registration API
* @param    ifNum                @b{(input)}   Interface number
* @param    -                    Minimum value 1
* @param    -                    Maximum value openapiMaxInterfaceCountGet()
* @param    localFaultStatus     @b{(output)}  local fault status (non-zero value 
*                                              indicates fault.)
* @param    remoteFaultStatus    @b{(output)}  remote fault status (non-zero value
*                                              indicates fault.) 
*
* @returns  OPEN_E_NONE        Status obtained successfully.
* @returns  OPEN_E_ERROR       Failed to fetch the fiber port fault status
*                              or the interface does not exist
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Not supported for this platform.
*
* @notes Only valid for fiber ports.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
***************************************************************************************/
open_error_t openapiFiberPortsOpticsFaultStatusGet(openapiClientHandle_t *client_handle, uint32_t ifNum,
                                                   uint32_t *localFaultStatus, uint32_t *remoteFaultStatus);

/**********************************************************************************
* @purpose  Get the diagnostic data for fiber ports optical transceiver.
*
* @param    client_handle        @b{(input)}   client handle from registration API
* @param    ifNum                @b{(input)}   Interface number
* @param    -                    Minimum value 1
* @param    -                    Maximum value openapiMaxInterfaceCountGet()
* @param    temperature          @b{(output)}  Temperature (degrees C)
* @param    voltage              @b{(output)}  Voltage (milliVolts)
* @param    current              @b{(output)}  Current (milliAmps)
* @param    powerIn              @b{(output)}  Power Input (microWatts)
* @param    powerOut             @b{(output)}  Power output (microWatts)
* @param    txFault              @b{(output)}  Transmit Fault
* @param    los                  @b{(output)}  Loss of signal
*
* @returns  OPEN_E_NONE        Fiber ports diagnostic data obtained successfully.
* @returns  OPEN_E_NOT_FOUND   Fiber ports diagnostic data not found.
* @returns  OPEN_E_ERROR       Failed to fetch the fiber port diagnostic data
*                              or the interface does not exist
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Not supported for this platform.
*
* @notes Only valid for fiber ports.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
**************************************************************************************/
open_error_t openapiFiberPortsOpticsGet(openapiClientHandle_t *client_handle, uint32_t ifNum,
                                        uint32_t *temperature, uint32_t *voltage, uint32_t *current,
                                        uint64_t *powerIn, uint64_t *powerOut,
                                        OPEN_BOOL_t *txFault, OPEN_BOOL_t *los);

/**********************************************************************************
* @purpose  Test the fiber cable and return information on a per lane basis.
*
* @param    client_handle        @b{(input)}   client handle from registration API
* @param    ifNum                @b{(input)}   Interface number
* @param    -                    Minimum value 1
* @param    -                    Maximum value openapiMaxInterfaceCountGet()
* @param    lane                 @b{(input)}   Lane number
*                                Can be determined using openapiIfExpPortOperModeLaneCountGet()
* @param    temperature          @b{(output)}  Temperature (degrees C)
* @param    voltage              @b{(output)}  Voltage (milliVolts)
* @param    current              @b{(output)}  Current (milliAmps)
* @param    powerIn              @b{(output)}  Power Input (microWatts)
* @param    powerOut             @b{(output)}  Power output (microWatts)
* @param    txFault              @b{(output)}  Transmit Fault
* @param    los                  @b{(output)}  Loss of signal
*
* @returns  OPEN_E_NONE        Fiber ports diagnostic data obtained successfully.
* @returns  OPEN_E_INTERNAL    Failed to accommadate output string.
* @returns  OPEN_E_NOT_FOUND   Fiber ports diagnostic data not found.
* @returns  OPEN_E_ERROR       Failed to fetch the fiber port diagnostic data
*                              or the interface does not exist
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Not supported for this platform.
*
* @notes Only valid for fiber ports.
*        'powerIn' and 'powerOut' are 64-bit signed floating numbers
*        so for simplicity they are provided in string format.
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiFiberPortsLaneOpticsGet(openapiClientHandle_t *client_handle, uint32_t ifNum,  uint32_t lane,
                                            uint32_t *temperature, uint32_t *voltage, uint32_t *current,
                                            open_buffdesc *powerIn, open_buffdesc *powerOut,
                                            OPEN_BOOL_t *txFault, OPEN_BOOL_t *los);

/*********************************************************************
* @purpose Get the transceiver presence status for fiber ports.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number
* @param    -              Minimum value 1
* @param    -              Maximum value openapiMaxInterfaceCountGet()
* @param    present        @b{(output)}  Transceiver presence status
*
* @returns  OPEN_E_NONE        Presence status obtained successfully.
* @returns  OPEN_E_ERROR       Could not determine status. Possibly due to an
*                              invalid interface number or not a fiber port.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Not supported for this platform.
*
* @notes Only valid for fiber ports.
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiFiberPortsTransceiverPresenceGet(openapiClientHandle_t *client_handle,
                                                     uint32_t ifNum, OPEN_BOOL_t *present);

/*********************************************************************
* @purpose  Gets the general fan speed.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    genFanSpeed     @b{(output)}  general speed of the fans.
*
* @returns  OPEN_E_NONE     Returned general fan speed.
* @returns  OPEN_E_FAIL     Failed to get the general fan speed.
* @returns  OPEN_E_PARAM    Error in parameter passed.
* @returns  OPEN_E_UNAVAIL  Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiBoxsiGeneralFanSpeedGet (openapiClientHandle_t * client_handle,
                                             uint32_t * genFanSpeed);

/*********************************************************************
* @purpose  Gets the general fan duty-level.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    genFanDutyLevel @b{(output)}  General fan duty-level.
*
* @returns  OPEN_E_NONE     Returned general duty-level of the fan.
* @returns  OPEN_E_FAIL     Failed to get general fan duty-cycle.
* @returns  OPEN_E_PARAM    Error in parameter passed.
* @returns  OPEN_E_UNAVAIL  Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiBoxsiGeneralFanDutyLevelGet (openapiClientHandle_t * client_handle,
                                                 uint32_t * genFanDutyLevel);

/**********************************************************************************
* @purpose  Gets the maximum number of FANs on the unit.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    numFans        @b{(output)}  maximum number of fans. 
*
* @returns  OPEN_E_NONE    maximum number of fans supported returned.
* @returns  OPEN_E_FAIL    Failed to get the maximum number of fans .
* @returns  OPEN_E_PARAM   Error in parameter passed.
* @returns  OPEN_E_UNAVAIL Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
***************************************************************************************/
open_error_t openapiBoxsNumOfFansGet (openapiClientHandle_t * client_handle,
                                     uint32_t * numFans);

/*********************************************************************
* @purpose  Gets the speed of a given fan number.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    fanNum         @b{(input)}   fan number, ZERO to maximum.
* @param    fanSpeed       @b{(output)}  speed of the fan.
*
* @returns  OPEN_E_NONE        speed of the fan get successfully.
* @returns  OPEN_E_FAIL        Failed to get the speed of the fan.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Not supported for this platform.
*
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsFanSpeedGet (openapiClientHandle_t * client_handle,
				     uint32_t fanNum, uint32_t * fanSpeed);


/*********************************************************************
* @purpose  Gets the state of a given fan number.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    fanNum         @b{(input)}   fan number, ZERO to maximum.
* @param    fanState       @b{(output)}  state of the fan.
*
* @returns  OPEN_E_NONE        state of the fan returned.
* @returns  OPEN_E_FAIL        Failed to get the state of the fan.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsFanStateGet (openapiClientHandle_t * client_handle,
				    uint32_t fanNum,
				    OPEN_BOXS_ITEM_STATE_t * fanState);

/*********************************************************************
* @purpose  Gets the type of a given fan number.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    fanNum         @b{(input)}   fan number, ZERO to maximum.
* @param    fanType        @b{(output)}  type of the fan.
*
* @returns  OPEN_E_NONE        type of the fan returned.
* @returns  OPEN_E_FAIL        Failed to get the type of the fan.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsFanTypeGet (openapiClientHandle_t * client_handle,
				    uint32_t fanNum,
				    OPEN_BOXS_ITEM_TYPE_t * fanType);

/*********************************************************************
* @purpose  Gets the duty-cycle of a given fan number.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    fanNum         @b{(input)}   fan number, ZERO to maximum.
* @param    fanDuty        @b{(output)}  duty-cycle of the fan.
*
* @returns  OPEN_E_NONE   duty-cycle of the fan returned.
* @returns  OPEN_E_FAIL   Failed to get the duty-cycle of the fan.
* @returns  OPEN_E_PARAM  Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsFanDutyCycleGet (openapiClientHandle_t * client_handle,
					uint32_t fanNum, uint32_t * fanDuty);

/*********************************************************************************************
 * @purpose  Gets the maximum size for the fan description.
 *
 * @param    client_handle  @b{(input)}   client handle from registration API
 * @param    descripSize    @b{(output)}  size for the fan description.
 *
 * @returns  OPEN_E_NONE        Returned fan description size.
 * @returns  OPEN_E_PARAM       Error in parameter passed.
 * @returns  OPEN_E_UNAVAIL     Not supported for this platform.
 *
 * @supported-in-version OpEN API Version: 1.15
 *
 * @end
 **********************************************************************************************/
open_error_t openapiBoxsFanDescripSizeGet (openapiClientHandle_t *
                                           client_handle,
                                           uint32_t * descripSize);

/*********************************************************************************************
 * @purpose  Gets the description of a given fan.
 *
 * @param    client_handle  @b{(input)}   Client handle from registration API
 * @param    fanNum         @b{(input)}   Fan number, ZERO to maximum.
 * @param    descrip        @b{(output)}  Description of the temperature sensor.
 *                                        Use "openapiBoxsFanDescripSizeGet()" to get the size
 *                                        of the description.
 *
 * @returns  OPEN_E_NONE        Returned size of fan description.
 * @returns  OPEN_E_FAIL        Failed to get fan description size.
 * @returns  OPEN_E_PARAM       Error in parameter passed.
 * @returns  OPEN_E_UNAVAIL     Not supported for this platform.
 *
 * @supported-in-version OpEN API Version: 1.15
 *
 * @end
 **********************************************************************************************/
open_error_t openapiBoxsFanDescripGet (openapiClientHandle_t *
                                       client_handle,
                                       uint32_t fanNum,
                                       open_buffdesc * descrip);

/*********************************************************************
* @purpose  Gets the maximum  number of power supplies in the unit.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    numPowSuplies  @b{(output)}   maximum number of power supplies.
*
* @returns  OPEN_E_NONE   maximum number of power supplies get successfully.
* @returns  OPEN_E_FAIL   Failed to get the maximum number of power supplies.
* @returns  OPEN_E_PARAM  Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Not supported for this platform.
*
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsNumOfPowSuppliesGet (openapiClientHandle_t *
					    client_handle,
					    uint32_t * numPowSuplies);

/*********************************************************************
* @purpose  Gets the status of a given power supply.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    psNo           @b{(input)}   power supply number, ZERO to maximum.
* @param    psState        @b{(output)}  state of the power supply.
*
* @returns  OPEN_E_NONE        state of the power supply returned.
* @returns  OPEN_E_FAIL        Failed to get the state of the power supply.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsPowSupplyStateGet (openapiClientHandle_t *
					  client_handle, uint32_t psNo,
					  OPEN_BOXS_ITEM_STATE_t * psState);

/*********************************************************************
* @purpose  Gets the type of a given power supply.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    psNo           @b{(input)}   power supply number, ZERO to maximum.
* @param    psType         @b{(output)}  type of the power supply.
*
* @returns  OPEN_E_NONE        type of the power supply returned.
* @returns  OPEN_E_FAIL        Failed to get the type of the power supply.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsPowSupplyTypeGet (openapiClientHandle_t *
					  client_handle, uint32_t psNo,
					  OPEN_BOXS_ITEM_TYPE_t * psType);

/*********************************************************************************************
 * @purpose  Gets the maximum size for the power supply description.
 *
 * @param    client_handle  @b{(input)}   client handle from registration API
 * @param    descripSize    @b{(output)}  size for the power supply description.
 *
 * @returns  OPEN_E_NONE        Returned power supply description size.
 * @returns  OPEN_E_PARAM       Error in parameter passed.
 * @returns  OPEN_E_UNAVAIL     Not supported for this platform.
 *
 * @supported-in-version OpEN API Version: 1.15
 *
 * @end
 **********************************************************************************************/
open_error_t openapiBoxsPowSupplyDescripSizeGet (openapiClientHandle_t *
                                                 client_handle,
                                                 uint32_t * descripSize);

/*********************************************************************************************
 * @purpose  Gets the description of a given power supply.
 *
 * @param    client_handle  @b{(input)}   Client handle from registration API
 * @param    psNum          @b{(input)}   Power supply number, ZERO to maximum.
 * @param    descrip        @b{(output)}  Description of the power supply.
 *                                        Use "openapiBoxsPowSupplyDescripSizeGet()" to
 *                                        get size of the description.
 *
 * @returns  OPEN_E_NONE        Returned size of power supply description.
 * @returns  OPEN_E_FAIL        Failed to get power supply description size.
 * @returns  OPEN_E_PARAM       Error in parameter passed.
 * @returns  OPEN_E_UNAVAIL     Not supported for this platform.
 *
 * @supported-in-version OpEN API Version: 1.15
 *
 * @end
 **********************************************************************************************/
open_error_t openapiBoxsPowSupplyDescripGet (openapiClientHandle_t *
                                             client_handle,
                                             uint32_t psNum,
                                             open_buffdesc * descrip);

/*********************************************************************
* @purpose Gets allowed temperature range for normal operation.
*
* @param    client_handle     @b{(input)}   client handle from registration API
* @param    minTemp           @b{(output)}  Minimum value in range
* @param    maxTemp           @b{(output)}  Maximum value in range
*
* @returns  OPEN_E_NONE       Returned allowed temperature range.
* @returns  OPEN_E_FAIL       Failed to get temperature range.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* @returns  OPEN_E_UNAVAIL    Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiBoxsTempRangeGet (openapiClientHandle_t *
                                      client_handle,
                                      uint32_t * minTemp,
                                      uint32_t * maxTemp);

/****************************************************************************************
* @purpose  Gets the general temperature of the system
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    genTemp        @b{(output)}  general temperature read from system.
*
* @returns  OPEN_E_NONE        Returned general temperature value.
* @returns  OPEN_E_FAIL        Failed to get general temperature.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Not supported for this platform.
*
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*****************************************************************************************/
open_error_t openapiBoxsGeneralTempGet (openapiClientHandle_t *
                                        client_handle,
                                        uint32_t * genTemp);

/*********************************************************************
* @purpose  Gets the maximum number of temperature sensors in the unit.
*
* @param    client_handle     @b{(input)}   client handle from registration API
* @param    numOfTempSemsors  @b{(output)}  maximum number of temperature sensors.
*
* @returns  OPEN_E_NONE        maximum number of temperature sensors get successfully.
* @returns  OPEN_E_FAIL        Failed to get the maximum number of temperature sensors.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsNumOfTempSensorsGet (openapiClientHandle_t *
					    client_handle,
					    uint32_t * numOfTempSemsors);

/*********************************************************************************************
* @purpose  Gets the maximum size for the temperature sensor description.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    descripSize    @b{(output)}  size for the temperature sensor description.
*
* @returns  OPEN_E_NONE        size for the temperature sensor description returned.
* @returns  OPEN_E_FAIL        Failed to get the size for the temperature sensor description.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
**********************************************************************************************/
open_error_t openapiBoxsTempSensorDescripSizeGet (openapiClientHandle_t *
						 client_handle,
						 uint32_t * descripSize);

/*********************************************************************************************
* @purpose  Gets the description of  a given temperature sensor.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    sensorNum      @b{(input)}   temperature sensor number, ZERO to maximum.
* @param    descrip        @b{(output)}  description of the temperature sensor.
*                                        use "openapiBoxsTempSensorDescripSizeGet()" to get the size 
*                                        of the description.
*
* @returns  OPEN_E_NONE        The description of the temperature sensor returned.
* @returns  OPEN_E_FAIL        Failed to get the description of the temperature sensor.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
**********************************************************************************************/
open_error_t openapiBoxsTempSensorDescripGet (openapiClientHandle_t *
					     client_handle,
					     uint32_t sensorNum,
					     open_buffdesc * descrip);

/****************************************************************************************
* @purpose  Gets the temperature value read by the given temperature sensor.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    sensorNum      @b{(input)}   temperature sensor number, ZERO to maximum.
* @param    temperature    @b{(output)}  temperature value read by the sensor.
*
* @returns  OPEN_E_NONE        Returned sensor temperature successfully.
* @returns  OPEN_E_UNAVAIL     Not supported for this platform.
*
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*****************************************************************************************/
open_error_t openapiBoxsTempGet (openapiClientHandle_t * client_handle,
				uint32_t sensorNum, uint32_t * temperature);

/*********************************************************************
* @purpose  Gets the state of a given temperature sensor.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    sensorNum      @b{(input)}   temperature sensor number, ZERO to maximum.
* @param    state          @b{(output)}  state of the temperature sensorNum.
*
* @returns  OPEN_E_NONE        State of the temperature sensor returned.
* @returns  OPEN_E_FAIL        Failed to get the state of the temperature sensor.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsTempSensorStatusGet (openapiClientHandle_t *
					    client_handle, uint32_t sensorNum,
					    OPEN_BOXS_TEMPSENSOR_STATE_t *
					    state);

/***********************************************************************************
* @purpose  Gets the maximum temperature of a given temperature sensor.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    sensorNum        @b{(input)}   temperature sensor number, ZERO to maximum.
* @param    maxTemperature   @b{(output)}  maximum temperature of the given sensor.
*
* @returns  OPEN_E_NONE        Returned Maximum temperature of the sensor.
* @returns  OPEN_E_FAIL        Failed to get maximum temperature of the sensor.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiBoxsTempSensorMaxTempGet (openapiClientHandle_t *
                                              client_handle,
                                              uint32_t sensorNum,
                                              uint32_t * maxTemperature);

/*********************************************************************
* @purpose  Gets the SFP module type.
*
* @param    client_handle  @b{(input)}   client handle from registration API.
* @param    ifNum          @b{(input)}   interface ID to get the type.
* @param    moduleType     @b{(output)}  type of module.
*
* @returns  OPEN_E_NONE        Interface type returned successfully. 
* @returns  OPEN_E_FAIL        Failed to get the interface type.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes This API is to get the SFP port type only whether it is SFP/SFP+.
*        It does not display the type of actual module plugged in.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsSfpTypeGet(openapiClientHandle_t *client_handle,
                                   uint32_t ifNum, OPEN_SFP_MODULE_TYPE_t *moduleType);

/*********************************************************************
* @purpose  Gets the SFP transceiver vendor name.
*
* @param    client_handle     @b{(input)}   client handle from registration API.
* @param    ifNum             @b{(input)}   interface ID to retrieve SFP vendor name.
* @param    vendorNameDesc    @b{(output)}  buff descriptor to get vendor name
*                                           of the SFP transceiver.
*
* @returns  OPEN_E_NONE        Vendor name returned successfully. 
* @returns  OPEN_E_FAIL        Failed to get the transceiver data.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Initialize vendorNameDesc pstart and size with the char buffer 
*         of size returned by openapiBoxsSfpVendorNameSizeGet(). 
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsSfpVendorNameGet(openapiClientHandle_t *client_handle,
                                         uint32_t ifNum, open_buffdesc *vendorNameDesc);

/*********************************************************************
* @purpose  Gets the SFP transceiver serial number.
*
* @param    client_handle     @b{(input)}   client handle from registration API.
* @param    ifNum             @b{(input)}   interface ID to retrieve SFP serial number.
* @param    serialNumberDesc  @b{(output)}  buff descriptor to get serial number 
*                                           of the SFP transceiver.
*
* @returns  OPEN_E_NONE        Serial number returned successfully. 
* @returns  OPEN_E_FAIL        Failed to get the transceiver data.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes    Initialize serialNumberDesc pstart and size with the char buffer 
*           of size returned by openapiBoxsSfpSerialNumberSizeGet(). 
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsSfpSerialNumberGet(openapiClientHandle_t *client_handle,
                                           uint32_t ifNum, open_buffdesc *serialNumberDesc);

/*********************************************************************
* @purpose  Gets the SFP transceiver part number.
*
* @param    client_handle     @b{(input)}   client handle from registration API.
* @param    ifNum             @b{(input)}   interface ID to retrieve SFP part number.
* @param    partNumberDesc    @b{(output)}  buff descriptor to get part number 
*                                           of the SFP transceiver.
*
* @returns  OPEN_E_NONE        Serial number returned successfully. 
* @returns  OPEN_E_FAIL        Failed to get the transceiver data.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes    Initialize partNumberDesc pstart and size with the char buffer 
*           of size returned by openapiBoxsSfpPartNumberSizeGet(). 
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsSfpPartNumberGet(openapiClientHandle_t *client_handle,
                                          uint32_t ifNum, open_buffdesc *partNumberDesc);


/*********************************************************************
* @purpose  Gets the SFP transceiver vendor revision number.
*
* @param    client_handle     @b{(input)}   client handle from registration API.
* @param    ifNum             @b{(input)}   interface ID to retrieve SFP vendor revision.
* @param    vendorRevDesc     @b{(output)}  buff descriptor to get vendor revision.
*
* @returns  OPEN_E_NONE        Serial number returned successfully. 
* @returns  OPEN_E_FAIL        Failed to get the transceiver data.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes    Initialize vendorRevDesc pstart and size with the char buffer 
*           of size returned by openapiBoxsSfpVendorRevSizeGet(). 
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsSfpVendorRevisionGet(openapiClientHandle_t *client_handle,
                                             uint32_t ifNum, open_buffdesc *vendorRevDesc);

/*********************************************************************
* @purpose  Gets the Link Length(50um) of SFP transceiver.
*
* @param    client_handle     @b{(input)}   client handle from registration API.
* @param    ifNum             @b{(input)}   interface ID to retrieve the link length
*                                           (50um) of SFP transceiver.
* @param    linkLength50um    @b{(output)}  Link length(50um) of the SFP transceiver.
*
* @returns  OPEN_E_NONE        Link Length(50um) returned successfully. 
* @returns  OPEN_E_FAIL        Failed to get the transceiver data.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsSfpLinkLength50umGet(openapiClientHandle_t *client_handle,
                                             uint32_t ifNum, uint32_t *linkLength50um);

/*********************************************************************
* @purpose  Gets the Link Length(62_5um) of SFP transceiver.
*
* @param    client_handle     @b{(input)}   client handle from registration API.
* @param    ifNum             @b{(input)}   interface ID to retrieve the link length
*                                           (62_5um) of SFP transceiver.
* @param    linkLength62_5um  @b{(output)}  Link length(62.5um) of the SFP transceiver.
*
* @returns  OPEN_E_NONE        Link Length(62.5um) returned successfully. 
* @returns  OPEN_E_FAIL        Failed to get the transceiver data.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsSfpLinkLength62_5umGet(openapiClientHandle_t *client_handle,
                                               uint32_t ifNum, uint32_t *linkLength62_5um);

/*********************************************************************
* @purpose  Gets the Nominal Signal Rate of SFP transceiver.
*
* @param    client_handle     @b{(input)}   client handle from registration API.
* @param    ifNum             @b{(input)}   interface ID to retrieve the Nominal
*                                           Signal Rate of SFP transceiver.
* @param    nominalSigRate    @b{(output)}  buff descriptor to get Nominal 
*                                           Signal Rate of the SFP transceiver.
*
* @returns  OPEN_E_NONE        Nominal Signal Rate returned successfully. 
* @returns  OPEN_E_FAIL        Failed to get the transceiver data.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsSfpNominalSignalRate(openapiClientHandle_t *client_handle,
                                             uint32_t ifNum, uint32_t *nominalSigRate);

/*********************************************************************
* @purpose  Gets the compliance of SFP transceiver.
*
* @param    client_handle     @b{(input)}   client handle from registration API.
* @param    ifNum             @b{(input)}   interface ID to retrieve the SFP compliance.
* @param    complianceDesc    @b{(output)}  buff descriptor to get compliance 
*                                           of the SFP transceiver.
*
* @returns  OPEN_E_NONE        Compliance returned successfully. 
* @returns  OPEN_E_FAIL        Failed to get the transceiver data.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes    Initialize complianceDesc pstart and size with the char buffer 
*           of size returned by openapiBoxsSfpComplianceSizeGet(). 
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsSfpComplianceGet(openapiClientHandle_t *client_handle,
                                         uint32_t ifNum, open_buffdesc *complianceDesc);

/*********************************************************************
* @purpose  Gets the size if vendor name of SFP transceiver.
*
* @param    client_handle     @b{(input)}   client handle from registration API.
* @param    vendorNameSize    @b{(output)}  size of SFP vendor name.
*
* @returns  OPEN_E_NONE        size of SFP vendor name returned successfully.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsSfpVendorNameSizeGet(openapiClientHandle_t *client_handle,
                                             uint32_t *vendorNameSize);

/*********************************************************************
* @purpose  Gets the size of serial number of SFP transceiver.
*
* @param    client_handle     @b{(input)}   client handle from registration API.
* @param    serialNumberSize  @b{(output)}  size of SFP serial number.
*
* @returns  OPEN_E_NONE        size of SFP serial number returned successfully.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsSfpSerialNumberSizeGet(openapiClientHandle_t *client_handle,
                                               uint32_t *serialNumberSize);

/*********************************************************************
* @purpose  Gets the size of part number of SFP transceiver.
*
* @param    client_handle     @b{(input)}   client handle from registration API.
* @param    partNumberSize    @b{(output)}  size of SFP part number.
*
* @returns  OPEN_E_NONE        size of SFP part number returned successfully.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsSfpPartNumberSizeGet(openapiClientHandle_t *client_handle,
                                             uint32_t *partNumberSize);

/*********************************************************************
* @purpose  Gets the size of vendor revision of SFP transceiver.
*
* @param    client_handle     @b{(input)}   client handle from registration API.
* @param    vendorRevSize     @b{(output)}  size of SFP vendor revision.
*
* @returns  OPEN_E_NONE        size of SFP vendor revision returned successfully.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsSfpVendorRevSizeGet(openapiClientHandle_t *client_handle,
                                            uint32_t *vendorRevSize);

/*********************************************************************
* @purpose  Gets the size of compliance of SFP transceiver.
*
* @param    client_handle     @b{(input)}   client handle from registration API.
* @param    complianceSize    @b{(output)}  size of SFP compliance.
*
* @returns  OPEN_E_NONE        size of SFP compliance returned successfully.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiBoxsSfpComplianceSizeGet(openapiClientHandle_t *client_handle,
                                             uint32_t *complianceSize);

/*********************************************************************
* @purpose  Disables port locator globally
*
* @param    client_handle     @b{(input)}   client handle from registration API.
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_UNAVAIL     Not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiPortLocatorGlobalDisable(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  enable or disable port locator on an interface
*
* @param    client_handle     @b{(input)}   client handle from registration API.
* @param    interface         @b{(input)}   interface to enable or disable.
* @param    enable            @b{(input)}   port operation mode.
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_UNAVAIL     Not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
*
* @end
*********************************************************************/
open_error_t openapiPortLocatorInterfaceEnable(openapiClientHandle_t *client_handle, 
                                               uint32_t interface, 
                                               OPEN_CONTROL_t enable);

/*********************************************************************
* @purpose  Get the CPLD version of the system
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    cpldVersion      @b{(output)} Pointer to CPLD version string storage
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_VER_BUFF_MAX_SIZE
*
* @returns  OPEN_E_NONE      if operation is successful
* @returns  OPEN_E_FAIL      if operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_UNAVAIL   if no device exist or not supported
*
* @notes
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiCpldVersionGet(openapiClientHandle_t *client_handle,
                                   open_buffdesc         *cpldVersion);

/**********************************************************************************
* @purpose  Gets the board revision number on the unit.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    version        @b{(output)}  Board version on the unit. 
*
* @returns  OPEN_E_NONE    Get successful.   
* @returns  OPEN_E_FAIL    Failed to get board revision.    
* @returns  OPEN_E_PARAM   Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
***************************************************************************************/
open_error_t openapiBoxsHwVersionGet(openapiClientHandle_t *client_handle,
                                     uint32_t *version);

/*********************************************************************
* @purpose  Turns on/off port and PoE LEDs on the given interface(s).
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    val             @b{(input)}  Turn on/off the LED
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_EXISTS      Already configured.
* @returns  OPEN_E_UNAVAIL     If feature is not supported.
* @returns  OPEN_E_INTERNAL    Internal error.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiBoxsPortLedsModeSet(openapiClientHandle_t *client_handle,
                                        OPEN_LED_CMD_t val);

/*********************************************************************
* @purpose  Gets port and PoE LED status on the given interface(s).
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    val             @b{(output)} LED status
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     If feature is not supported.
* @returns  OPEN_E_INTERNAL    Internal error.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiBoxsPortLedsModeGet(openapiClientHandle_t *client_handle,
                                        OPEN_LED_STATUS_t *val);

/*********************************************************************
* @purpose  Turns on/off system LEDs.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    val             @b{(input)}  Turn on/off the LED
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_EXISTS      Already configured.
* @returns  OPEN_E_UNAVAIL     If feature is not supported.
* @returns  OPEN_E_INTERNAL    Internal error.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiBoxsSystemLedModeSet(openapiClientHandle_t *client_handle,
                                         OPEN_LED_CMD_t val);

/*********************************************************************
* @purpose  Gets system LED status.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    val             @b{(output)} LED status
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     If feature is not supported.
* @returns  OPEN_E_INTERNAL    Internal error.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiBoxsSystemLedModeGet(openapiClientHandle_t *client_handle,
                                         OPEN_LED_STATUS_t *val);

/*********************************************************************************************
 * @purpose Gets the maximum size for the power supply hardware data.
 *
 * @param   client_handle  @b{(input)}   Client handle from registration API.
 * @param   size           @b{(output)}  Size for the hardware data.
 *
 * @returns OPEN_E_NONE    Returned power supply hardware data size.
 * @returns OPEN_E_PARAM   Error in parameter passed.
 * @returns OPEN_E_UNAVAIL Not supported for this platform.
 *
 * @supported-in-version OpEN API Version: 1.28
 *
 * @end
 **********************************************************************************************/
open_error_t openapiBoxsPowSupplyHardwareDataSizeGet(openapiClientHandle_t *client_handle,
                                                     uint32_t *size);

/*********************************************************************************************
 * @purpose Get the power supply hardware inventory data.
 *
 * @param   client_handle   @b{(input)}   Client handle from registration API
 * @param   unit            @b{(input)}   The unit for this operation.
 * @param   psNum           @b{(input)}   Power supply number, ZERO to maximum.
 * @param   partNumRev      @b{(output)}  Output buffer to store to the
 *                                        Part number and revision. 
 *                                        Use @ref openapiBoxsPowSupplyHardwareDataSizeGet
 *                                        to get size.
 * @param   mfgPartNum      @b{(output)}  Output buffer to store to the
 *                                        manufacturer part numbrer. 
 *                                        Use @ref openapiBoxsPowSupplyHardwareDataSizeGet
 *                                        to get size.
 * @param   serialNum       @b{(output)}  Output buffer to store to the
 *                                        serial number.
 *                                        Use @ref openapiBoxsPowSupplyHardwareDataSizeGet
 *                                        to get size.
 * @param   dateCode        @b{(output)}  Output buffer to store to the
 *                                        Data code.
 *                                        Use @ref openapiBoxsPowSupplyHardwareDataSizeGet
 *                                        to get size.
 *
 * @returns OPEN_E_NONE     If the power supply hardware inventory data is retrieved
 *                          successfully.
 * @returns OPEN_E_FAIL     Failed to get the hardware inventory data.
 * @returns OPEN_E_PARAM    Error in parameter passed.
 * @returns OPEN_E_UNAVAIL  Not supported for this platform.
 * @returns OPEN_E_INTERNAL Internal error.
 *
 * @supported-in-version OpEN API Version: 1.28
 *
 * @end
 **********************************************************************************************/
open_error_t openapiBoxsPowSupplyHardwareDataGet(openapiClientHandle_t *client_handle,
                                                 uint32_t unit,
                                                 uint32_t psNum,
                                                 open_buffdesc *partNumRev,
                                                 open_buffdesc *mfgPartNum,
                                                 open_buffdesc *serialNum,
                                                 open_buffdesc *dateCode);
#endif /* OPENAPI_BOXS_H_INCLUDED */
