/*********************************************************************
*
*  Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_common.h
*
* @purpose   Base Services. 
*
* @component OpEN
*
* @create    07/13/2012
*
* @end
*
**********************************************************************/
#ifndef _OPENAPI_COMMON_H_
#define _OPENAPI_COMMON_H_

#include <stdint.h>
#include <arpa/inet.h>
#include <netinet/in.h>

/* Some structures need to be packed on platforms that use
** 64 bit OpEN API libraries and 32 bit network operating system.
*/
#if defined (__i386__) || defined (__x86_64__)
  #define OPEN_PACKED_STRUCT typedef struct __attribute__((__packed__))
#else
  #define OPEN_PACKED_STRUCT typedef struct
#endif


/* 
 * OPEN API error codes.
 * 
 * Note: Values used in this enum for non-zero entries must 
 *       be different than those used in L7_RPCCLT_RC_t.
 *       OPEN API calls via RPC may return error code values
 *       from L7_RPCCLT_RC_t in certain situations.
 */
/** OPEN uses these enumerators to indicate the error codes. */
typedef enum
{
  OPEN_E_NONE                 = 0,   /**< Success. */
  OPEN_E_RPC                  = -20, /**< Error in RPC. */
  OPEN_E_INTERNAL             = -21, /**< Internal error. */
  OPEN_E_PARAM                = -22, /**< Invalid parameter. */
  OPEN_E_FULL                 = -23, /**< Maximum count is already reached or table full. */
  OPEN_E_EXISTS               = -24, /**< Already exists. */
  OPEN_E_TIMEOUT              = -25, /**< Operation Timeout. */
  OPEN_E_FAIL                 = -26, /**< Operation Fail. */
  OPEN_E_DISABLED             = -27, /**< Disabled. */
  OPEN_E_UNAVAIL              = -28, /**< Parameter/feature is not supported. */
  OPEN_E_NOT_FOUND            = -29, /**< Parameter not found. */
  OPEN_E_EMPTY                = -30, /**< Nothing to report or table is empty. */
  OPEN_E_ERROR                = -31, /**< Error conditions */
  OPEN_E_DEPENDENCY_NOT_MET   = -32, /**< Operation dependency not met */
  OPEN_E_BUSY                 = -33  /**< System is busy (doing some important/time consuming operations/updates) */ 
} open_error_t;

/** Open API Events for which clients can register.  */
typedef enum
{
  OPEN_EVENT_INTERFACE          = 0, /* This first event must be 0 */
  OPEN_EVENT_VLAN               = 1,
  OPEN_EVENT_ROUTE              = 2,
  OPEN_EVENT_ROUTING_INTERFACE  = 3,
  OPEN_EVENT_ISSU_START         = 4,
  OPEN_EVENT_READY_STATE        = 5, /* Configurator transition to/from execute state */
  OPEN_EVENT_CONFIG             = 6,
  OPEN_EVENT_LAST                    /* Must be last. Can be incremented as needed */
} OPEN_EVENT_e;

typedef struct
{
  unsigned char open_event_mask [((OPEN_EVENT_LAST) / 8) + 1];
} openapiEventList_t;

/* OpEN registration service identifiers supported for
 * external OpEN API clients. 
 *  
 * Internal clients have their own set of identifiers 
 * (unpublished) that are not included here, hence the 
 * use of #define instead of an enumeration. 
 *  
 * NOTE: Defining values that are not likely to get
 *       confused with the actual service ID values.
 *       All values listed here get converted to their
 *       real server ID values by the API implementation.
 */
#define OPEN_REGISTRATION_SERVICE_USERAPPS      100001      /* all user-provided applications */

/* First instance number for a given RPC client (by convention).
 * 
 * Note: Mirrors L7_RPCCLT_INSTANCE_FIRST definition.
 */
#define OPEN_CLIENT_INSTANCE_FIRST              1

/** Open API enhanced registration flags. */
typedef enum
{
  OPEN_REGISTER_FLAGS_NONE              = 0x00000000,       /**< No special flags needed. */
  OPEN_REGISTER_FLAGS_THREADED          = 0x00000001        /**< Multi-threaded application. */
} OPEN_REGISTER_FLAGS_t;

/* OpEN client handle struct. Only use provided accessor functions to
 * retrieve any contents.
 */
struct openapiClientHandle_s
{
  char   content[256];
};

typedef struct openapiClientHandle_s openapiClientHandle_t;

typedef struct
{
  uint32_t                size;          /* total buffer size (IN) / content length (OUT) */
  void                   *pstart;        /* ptr to buffer starting location */
} open_buffdesc;

typedef struct
{
  uint8_t release;
  uint8_t version;
  uint8_t maint_level;
  uint8_t build_num;
} open_revision_data_t;

/* Network interface */
typedef struct
{
  uint8_t       unit;
  uint8_t       slot;
  uint16_t      port;

} open_USP_t;

/** OPEN uses these enumerators to indicate enable or disable 
  * for a given config or status parameter. */
typedef enum
{
  OPEN_DISABLE = 0, /**< disable */
  OPEN_ENABLE  = 1,  /**< enable */
} OPEN_CONTROL_t;

/** OPEN uses these enumerators to indicate port admin mode enable, disable or diag_disable */
typedef enum
{
  OPEN_PORT_DISABLE = 0, /**< Port is disabled */
  OPEN_PORT_ENABLE = 1, /**< Port is enabled */
  OPEN_PORT_DIAG_DISABLE = 2 /**< Port is disabled by protocol */
} OPEN_PORT_STATE_t;

/** OPEN uses these enumerators to indicate true or false 
  * for a given config or status parameter. */
typedef enum
{
  OPEN_FALSE = 0, /**< false */
  OPEN_TRUE  = 1  /**< true */
} OPEN_BOOL_t;

/** OPEN uses these enumerators to indicate the link status. */
typedef enum
{
  OPEN_LINK_UP   = 0, /**< Link Up. */
  OPEN_LINK_DOWN = 1  /**< Link Down. */

} OPEN_LINK_STATE_t;

/** OPEN uses these enumerators to indicate the interface types. */
typedef enum
{
  OPEN_ACTION_ADD    = 0,
  OPEN_ACTION_DELETE = 1

} OPEN_ACTION_t;

typedef enum
{
  OPEN_INTF_TYPE_PHY            = 0, /**< Physical interface. */
  OPEN_INTF_TYPE_CPU            = 1, /**< CPU interface. */
  OPEN_INTF_TYPE_LAG            = 2, /**< LAG interface. */
  OPEN_INTF_TYPE_VLAN           = 3, /**< VLAN interface. */
  OPEN_INTF_TYPE_LOOPBACK       = 4, /**< Loopback interface. */
  OPEN_INTF_TYPE_TUNNEL         = 5, /**< Tunnel interface. */
  OPEN_INTF_TYPE_SERVICE_PORT   = 6, /**< Service port interface. */
  /** Any other interface type. */
  OPEN_INTF_TYPE_OTHER          = 7, /**< Not one of the interfaces defined above */
  OPEN_INTF_TYPE_ANY            = 8  /**< Useful on openapiIfNextGet() to get all interfaces */
} OPEN_INTF_TYPE_t;

/** OPEN uses these enumerators to indicate the VLAN participation of a given interface. */
typedef enum
{
  OPEN_VLAN_PARTICIPATION_MODE_INCLUDE  = 0, /**< Included in the VLAN. */
  OPEN_VLAN_PARTICIPATION_MODE_AUTO     = 1, /**< AUTO detect mode . */
  OPEN_VLAN_PARTICIPATION_MODE_EXCLUDE  = 2  /**< Excluded from the VLAN. */
} OPEN_VLAN_PARTICIPATION_MODE_t;

/** OPEN uses these enumerators to indicate the types of frames
 *  that may be received on this port.*/
typedef enum
{
  OPEN_DOT1Q_ADMIT_ALL                = 1, /**< Untagged or priority tagged frames */
  OPEN_DOT1Q_ADMIN_ONLY_VLAN_TAGGED   = 2, /**< VLAN tagged frames */
  OPEN_DOT1Q_ADMIN_ONLY_VLAN_UNTAGGED = 3  /**< VLAN untagged frames */
} OPEN_DOT1Q_ACCEPT_FRAME_TYPE_t;

#define OPEN_MAC_ADDR_LEN 6
#define OPEN_MAC_ADDR_STR_LEN (OPEN_MAC_ADDR_LEN * 3)

#define OPEN_MIN_U64_STR_BUFFER_SIZE 21    /**< 20 chars + terminating nul */

#define OPEN_DATE_TIME_OUTPUT_FORMAT    "%4.4d-%2.2d-%2.2d %2.2d:%2.2d:%2.2d"
#define OPEN_DATE_TIME_BUF_LEN          20 /* includes trailing '\0' */

/**< Management port types */
typedef enum
{
  OPEN_MGMT_SERVICEPORT = 0,   /**< Servive port */
  OPEN_MGMT_NETWORKPORT,       /**< Network port */
  OPEN_MGMT_IPPORT             /**< IP port */
} OPEN_MGMT_PORT_TYPE_t;

/** OPEN uses these enumerators to indicate address family,
 *  IPV4 or IPV6. */
typedef enum
{
  OPEN_AF_NONE  = 0, /**< Neither IPV4 nor IPV6. */
  OPEN_AF_INET  = 1, /**< IPV4 address family. */
  OPEN_AF_INET6 = 2  /**< IPV6 address family. */
} OPEN_AF_t;

/** OPEN uses these enumerators to indicate BGP address family
 *  index of routes or policy for IPV4 or IPV6. */
typedef enum
{
  OPEN_AFX_IP4 = 0, /**< IPV4 address family. */
  OPEN_AFX_IP6 = 1  /**< IPV6 address family. */
} OPEN_AFX_t;

/** Route Preference Types */
typedef enum
{
  OPEN_PREF_RESERVED                = 0,  /* Not a valid protocol */
  OPEN_PREF_LOCAL                   = 1,
  OPEN_PREF_STATIC                  = 2,
  OPEN_PREF_MPLS                    = 3,
  OPEN_PREF_OSPF_INTRA_AREA         = 4,
  OPEN_PREF_OSPF_INTER_AREA         = 5,
  OPEN_PREF_OSPF_EXTERNAL           = 6,
  OPEN_PREF_RIP                     = 7,
  OPEN_PREF_IBGP                    = 8,
  OPEN_PREF_EBGP                    = 9,
  OPEN_PREF_6TO4                    = 10,
  OPEN_PREF_GLOBAL_DEFAULT_GATEWAY  = 11,
  OPEN_PREF_DEFAULT_DHCP            = 12,
  OPEN_PREF_LOCAL_BGP               = 13,
  OPEN_PREF_LAST                          /* Always keep this last */
} OPEN_PREF_INDICES_t;

/** Redistribution route types*/
typedef enum
{
  OPEN_REDIST_RT_FIRST  = 0,
  OPEN_REDIST_RT_LOCAL  = 1,
  OPEN_REDIST_RT_STATIC = 2,
  OPEN_REDIST_RT_RIP    = 3,
  OPEN_REDIST_RT_OSPF   = 4,
  OPEN_REDIST_RT_BGP    = 5,
  OPEN_REDIST_RT_OSPFV3 = 6,
  OPEN_REDIST_RT_LAST
} OPEN_REDIST_RT_INDICES_t;

/** Hash Mode names */

typedef struct open_in6_addr_s 
{
  union
  {
    uint32_t  addr32[4];
    uint16_t  addr16[8];
    uint8_t   addr8[16];
  }u;
} open_in6_addr_t;

typedef struct open_inet_addr_s
{
  OPEN_AF_t     family;  
  union
  {
    /* IPv4 address always in host byte order */
    uint32_t    ipv4;
    open_in6_addr_t   ipv6;
  } addr;
} open_inet_addr_t;

typedef struct open_inet_pfx_s
{
  open_inet_addr_t ipAddr;
  uint32_t pfxLen;

} open_inet_pfx_t;


typedef enum
{
  OPEN_ACCESS_LINE_UNKNOWN = 0, /**< Undefined */
  OPEN_ACCESS_LINE_CONSOLE,     /**< Console */
  OPEN_ACCESS_LINE_TELNET,     /**< Telnet */
  OPEN_ACCESS_LINE_SSH,        /**< SSH */
  OPEN_ACCESS_LINE_HTTPS,      /**< HTTPS */
  OPEN_ACCESS_LINE_HTTP,       /**< HTTP */
  OPEN_ACCESS_LINE_DOT1X       /**< DOT1X */
} OPEN_ACCESS_LINE_t;


#define OPEN_IS_ADDR_EQUAL(xaddr, yaddr) \
  ( (OPEN_AF_INET6 == ((open_inet_addr_t *)(xaddr))->family) ? \
    ( (((open_inet_addr_t *)(xaddr))->addr.ipv6.u.addr32[0] == \
       ((open_inet_addr_t *)(yaddr))->addr.ipv6.u.addr32[0]) && \
      (((open_inet_addr_t *)(xaddr))->addr.ipv6.u.addr32[1] == \
       ((open_inet_addr_t *)(yaddr))->addr.ipv6.u.addr32[1]) && \
      (((open_inet_addr_t *)(xaddr))->addr.ipv6.u.addr32[2] == \
       ((open_inet_addr_t *)(yaddr))->addr.ipv6.u.addr32[2]) && \
      (((open_inet_addr_t *)(xaddr))->addr.ipv6.u.addr32[3] == \
       ((open_inet_addr_t *)(yaddr))->addr.ipv6.u.addr32[3]) ) : \
    ( ((open_inet_addr_t *)(xaddr))->addr.ipv4 == \
      ((open_inet_addr_t *)(yaddr))->addr.ipv4) )

#define OPEN_INTF_NAME_MAX_LENGTH 60
#define OPEN_IP_ADDR_DISP_LENGTH  46
#define OPENAPI_VER_BUFF_MAX_SIZE 128
#define OPEN_IP_ADDR_LENGTH 16
#define OPENAPI_VER_BUFF_MAX_SIZE 128

/** OpEN uses this enumeration to define encryption types */
typedef enum
{
  OPEN_PASSWORD_ENCRYPT_NONE    = 0,
  OPEN_PASSWORD_ENCRYPT_AES     = 1,
  OPEN_PASSWORD_ENCRYPT_MD5     = 2
} OPEN_PASSWORD_ENCRYPT_ALG_t;

/** OpEN uses this enumeration to define IP address types */
typedef enum
{
  OPEN_IP_ADDRESS_UNKNOWN = 0, /**< Address type unknown */
  OPEN_IP_ADDRESS_IPV4    = 1, /**< Address type IPv4 */
  OPEN_IP_ADDRESS_IPV6    = 2, /**< Address type IPv6 */
  OPEN_IP_ADDRESS_DNS     = 16  /**< Address type DNS */
} OPEN_IP_ADDRESS_TYPE_t;

/** OpEN uses this enumeration to define rate units */
typedef enum
{
  OPEN_RATE_UNIT_NONE            = 0, /**< Rate unit none */
  OPEN_RATE_UNIT_PERCENT         = 1, /**< Rate in percentage of link speed */
  OPEN_RATE_UNIT_PPS             = 2, /**< Rate in packets/sec */
  OPEN_RATE_UNIT_KBPS            = 3  /**< Rate in kilobytes/sec */
} OPEN_RATE_UNIT_t;

typedef unsigned char maskValue_t;

#define OPEN_MASK_UNIT            (sizeof(maskValue_t) * 8)
#define OPEN_MASK_LEN(_size)      ((((_size) - 1) / OPEN_MASK_UNIT) + 1)

/**< SETMASKBIT turns on bit index # k in mask j. Note: k is one-based. */
#define OPEN_SETMASKBIT(j, k)                                          \
  ((j)[(k - 1) / OPEN_MASK_UNIT] |= 1 << ((k - 1) % OPEN_MASK_UNIT))

/**< CLRMASKBIT turns off bit index # k in mask j. Note: k is one-based. */
#define OPEN_CLRMASKBIT(j, k)                                          \
  ((j)[(k - 1) / OPEN_MASK_UNIT] &= ~(1 << ((k - 1) % OPEN_MASK_UNIT)))


/**< NONZEROMASK returns 1 if any bit in word mask of NUM length is turned-on. The result, 1 or 0 is stored in result.
 */
#define OPEN_NONZEROMASK(_mask, _result, _size)             \
  do                                                      \
  {                                                       \
    uint32_t _i_;                                        \
    maskValue_t *_p_ = (maskValue_t *)&(_mask); \
                                                          \
    (_result) = 0;                                        \
    for (_i_ = 0; _i_ < OPEN_MASK_LEN(_size); _i_++)        \
    {                                                     \
      if (_p_[_i_] != 0)                                  \
      {                                                   \
        (_result) = 1;                                    \
        break;                                            \
      }                                                   \
    }                                                     \
  } while (0)


/**< ISMASKBITSET returns 0 if the interface k is not set in mask j */
#define OPEN_ISMASKBITSET(j, k)                                        \
  ((j)[((k - 1) / OPEN_MASK_UNIT)] & ( 1 << ((k - 1) % OPEN_MASK_UNIT)))

/** OpEN uses these defines to define hex codes of
 *  the protocols */
#define OPENAPI_PROTOCOL_HEX_IP   0x0800  /**< HEX code for the IP protocol. */
#define OPENAPI_PROTOCOL_HEX_ARP  0x0806  /**< HEX code for the ARP protocol. */
#define OPENAPI_PROTOCOL_HEX_IPX  0x8137  /**< HEX code for the IPX protocol. */
#define OPENAPI_PROTOCOL_HEX_MIN  0x0600  /**< Minumum HEX  code of the protocol. */
#define OPENAPI_PROTOCOL_HEX_MAX  0xFFFF  /**< Maximim HEX  code of the protocol. */

#define OPEN_MATCH_EXACT     1
#define OPEN_MATCH_GETNEXT   2
#define OPEN_MATCH_FREE      3

#define OPEN_VRF_MAX_NAME_LEN 15

#define OPEN_RPC_CLT_RECV_DEF_TIMEOUT  0  /**< Default RPC client receive timeout. */
#define OPEN_RPC_CLT_RECV_MIN_TIMEOUT  1  /** Minimum RPC client receive timeout */
#define OPEN_RPC_CLT_RECV_MAX_TIMEOUT  65535 /** Maximum RPC client receive timeout */

/* RPC debug counters */
typedef struct 
{
  uint32_t msendok;                    /**< Sent msg success count */
  uint32_t mrecvok;                    /**< Received msg success count */
  uint32_t mnowait;                    /**< Nowait option usage count */
  uint32_t msendfail;                  /**< Failed msg send attempts */
  uint32_t mrecvfail;                  /**< Failed msg recv attempts */
  uint32_t mrecvseqerr;                /**< Msg recv sequence number mismatch (not processed) */
  uint32_t mrecvtimeout;               /**< Msg recv timed out (not processed) */
  uint32_t mrecvtrunc;                 /**< Truncated recv msgs (not processed) */
  uint32_t mconnfail;                  /**< Failed connection attempts */
} openapiClientRpcStats_t;


#define OPEN_MAX_MPLS_IMPOSE_LABELS  3 /**< Maximum MPLS impose lables */

/** MPLS label stack */
typedef struct
{
  uint32_t label[OPEN_MAX_MPLS_IMPOSE_LABELS]; /**< MPLS label */
} OPEN_MPLS_LABELS_t;


/*********************************************************************
* @purpose  Returns the revision of the OPEN API.
*
* @param    client_handle       @b{(input)}  Client handle from registration API.
* @param    version             @b{(output)} Pointer to structure that is used to
*                                              store revision data.
*
* @returns  OPEN_E_NONE         Revision data successfully retrieved.
* @returns  OPEN_E_FAIL         Revision data could not be retrieved.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.1
* 
* @end
*********************************************************************/
open_error_t openapiApiVersionGet(openapiClientHandle_t *client_handle,
                                  open_revision_data_t *version);

/*********************************************************************
* @purpose  Returns the CPU control plane interface number.
*
* @param    client_handle       @b{(input)}  Client handle from registration API.
* @param    intf                @b{(output)} Internal interface number.
*
* @returns  OPEN_E_NONE         Revision data successfully retrieved.
* @returns  OPEN_E_FAIL         Revision data could not be retrieved.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
* 
* @end
*********************************************************************/
open_error_t openapiCpuIntfGet(openapiClientHandle_t *client_handle,
                               uint32_t *intf);

/*********************************************************************
* @purpose  Get the number of equal cost routes supported by this
*           platform.
*
* @param    client_handle       @b{(input)}  Client handle from registration API.
* @param    maxPaths            @b{(output)} Maximum number of equal cost routes.
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_PARAM        Error in parameter passed. 
*
* @notes    The max number of equal cost routes are platform specific
*           and may also vary depending upon ECMP support.
*
* @supported-in-version OpEN API Version: 1.2
* 
* @end
*********************************************************************/
open_error_t openapiOspfEqualCostRoutesMax(openapiClientHandle_t *client_handle,
                                           uint32_t *maxPaths);

/*********************************************************************
* @purpose  Clear all the bit mask of events
* 
* @param    event_mask        @b{(input)}  Bit mask of events which are to be cleared. 
*  
* @returns  void.
* @supported-in-version OpEN API Version: 1.1 
*  
* @end
*********************************************************************/
void openapiEventListClear(openapiEventList_t *event_mask);

/*********************************************************************
* @purpose  Set a particular event type to the event bit mask.
*
* @param    event_mask          @b{(input)}  Bit mask of events in which an event has to set.
* @param    event_handle        @b{(input)}  Storage to contain event information.
* 
* @returns  void.
* @supported-in-version OpEN API Version: 1.1
* 
* @end
*********************************************************************/
void openapiEventSet(openapiEventList_t *event_mask, OPEN_EVENT_e event_handle);

/*********************************************************************
* @purpose  Check if a particular event is set in a bit mask of events.
*
* @param    event_mask          @b{(input)}  Bit mask of events in which an event has to be checked to be set.
* @param    event_handle        @b{(input)}  Storage to contain event information.
* 
* @returns  1                   The event type is set in the bit mask of events.
* @returns  0                   The event type is not set in the bit mask of events.
*
* @supported-in-version OpEN API Version: 1.1
* 
* @end
*********************************************************************/
int openapiEventIsSet(openapiEventList_t *event_mask, OPEN_EVENT_e event_handle);

/*********************************************************************
* @purpose  Register as a client with the OpEN API and wait until
*           OpEN API services are available.
*
* @param    client_name         @b{(input)}  String containing the client name.
* @param    client_handle       @b{(output)} Storage to contain handle information.
* 
* @returns  OPEN_E_NONE         OpEN is ready to accept RPC calls.
* @returns  OPEN_E_INTERNAL     OpEN is not available in this product.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    This API blocks the caller until the Open
*           API services are ready.
* 
*           The same process may register multiple times.
* 
*           The client name is a zero-terminated string of
*           any length and does not have to be unique.
* 
* @notes    Applications designed to run using multiple client threads
*           must call the @ref openapiClientRegisterThreaded API
*           instead of this one.
*
* @supported-in-version OpEN API Version: 1.1
* 
* @end
*********************************************************************/
open_error_t openapiClientRegister(char *client_name,
                                   openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Register as a client with the OpEN API using enhanced options
*           and wait until OpEN API services are available.
*
* @param    client_name         @b{(input)}  String containing the client name.
* @param    reg_flags           @b{(input)}  Client registration flags (bit mask).
*                                 The following flags are supported (all other 
*                                 values are silently ignored) -
*                            \li \b OPEN_REGISTER_FLAGS_NONE: If used without any
*                                  other flags, this is equivalent to calling
*                                  @ref openapiClientRegister.
*	                           \li \b OPEN_REGISTER_FLAGS_THREADED: Register as a multi-threaded 
*                                  client application. Normally a client application runs as
*                                  a single instance within its own process. Applications that
*                                  instantiate multiple threads within the same process
*                                  must register with this flag set.
* @param    client_handle       @b{(output)} Storage to contain handle information.
* 
* @returns  OPEN_E_NONE         OpEN is ready to accept RPC calls.
* @returns  OPEN_E_INTERNAL     OpEN is not available in this product.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* 
* @notes    This API blocks the caller until the Open
*           API services are ready.
* 
*           The same process may register multiple times.
* 
*           The client name is a zero-terminated string of
*           any length and does not have to be unique.
* 
* @notes    This registration API must be used by applications
*           that are designed to be multi-threaded, with each thread
*           registering independently with the OpEN API service. Such
*           applications must call this API with OPEN_REGISTER_FLAGS_THREADED
*           set in the reg_flags input parameter instead of calling
*           @ref openapiClientRegister.
*
* @supported-in-version OpEN API Version: 1.11
* 
* @end
*********************************************************************/
open_error_t openapiClientRegisterEnhanced(char *client_name,
                                           OPEN_REGISTER_FLAGS_t reg_flags,
                                           openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Register as a client of a specified OpEN API service 
*           and wait until the service is available.
*   
* @param    client_name         @b{(input)}  String containing the client name.
* @param    client_inst         @b{(input)}  Client instance number.
* @param    server_id           @b{(input)}  Server identifier.
* @param    client_handle       @b{(output)} Storage to contain handle information.
*  
* @returns  OPEN_E_NONE         OpEN is ready to accept RPC calls.
* @returns  OPEN_E_INTERNAL     OpEN is not available in this product.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*  
* @notes    This API blocks the caller until the Open
*           API service is ready.
* 
*           The same process may register multiple times.
* 
*           The client name is a zero-terminated string of
*           any length and does not have to be unique.
* 
* @notes    This API is primarily intended for internal OpEN client applications.
*           Callers must specify a valid server_id (not published externally).
* 
* @notes    All user-supplied (i.e. external) OpEN clients should continue
*           using openapiClientRegister(). Optionally, external clients may
*           call this API provided that the server_id parameter value is specified
*           as follows:
*               @ref OPEN_REGISTRATION_SERVICE_USERAPPS
* 
* @notes    The client_inst value is used to differentiate multiple client
*           instances having the same name. While it is typically set to a
*           small integer value, it is recommended that @ref OPEN_CLIENT_INSTANCE_FIRST
*           be used as the first (or only) client instance of a given type,
*           with any additional clients using incremental values thereof.
*            
* @supported-in-version OpEN API Version: 1.9
* 
* @end
*********************************************************************/
open_error_t openapiServiceClientRegister(char *client_name,
                                          int32_t client_inst,
                                          int32_t server_id,
                                          openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Register as an internal client with the OpEN API and wait until
*           OpEN API services are available.
*
* @param    client_name         @b{(input)}  String containing the client name.
* @param    client_handle       @b{(output)} Storage to contain handle information.
*
* @returns  OPEN_E_NONE         OpEN is ready to accept RPC calls.
* @returns  OPEN_E_INTERNAL     OpEN is not available in this product.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    This API blocks the caller until the Open
*           API services are ready.
*
*           The same process may register multiple times.
*
*           The client name is a zero-terminated string of
*           any length and does not have to be unique.
*
* @notes    This API is deprecated. Please use openapiServiceClientRegister()
*           instead.
* 
* @open-status-deprecated
* @supported-in-version OpEN API Version: 1.6
* @deprecated-in-version OpEN API Version: 1.9
*
* @end
*********************************************************************/
open_error_t openapiInternalClientRegister(char *client_name,
                                           openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Teardown and cleanup the connection resources.
*   
* @param    client_handle       @b{(input)}  Client handle from registration API.
*  
* @returns  OPEN_E_NONE         Connection closed.
* @returns  OPEN_E_FAIL         Failed to teardown and cleanup.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*  
* @notes    It's good housekeeping to close and free the connection
*           and relinquish the client handle. 
*            
* @supported-in-version OpEN API Version: 1.3
* 
* @end
*********************************************************************/
open_error_t openapiClientTearDown(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Deregister an internal OpEN client.
*
* @param    client_handle       @b{(input)}  Client handle from registration API.
*
* @returns  OPEN_E_NONE         OpEN is ready to accept RPC calls.
* @returns  OPEN_E_INTERNAL     OpEN is not available in this product.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    This API is deprecated. Please use openapiClientTearDown()
*           instead.
*
* @open-status-deprecated
* @supported-in-version OpEN API Version: 1.6
* @deprecated-in-version OpEN API Version: 1.9
*
* @end
*********************************************************************/
open_error_t openapiInternalClientTearDown(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Register the client to receive event notifications
*           for one or more event types.
*
* @param    client_handle       @b{(input)}  Client handle from registration API.
* @param    event_mask          @b{(input)}  Bit mask of events for which to register.
* 
* @returns  OPEN_E_NONE         Registered for all events.
* @returns  OPEN_E_FAIL         Failed to register for all events. 
*
* @notes    There is no API to un-register for events. The clients
*           are un-registered when the client process exits.
*           Use the OPEN_EVENT macros to set up the event mask.
* 
* @supported-in-version OpEN API Version: 1.1
* 
* @end
*********************************************************************/
open_error_t openapiEventRegister(openapiClientHandle_t *client_handle,
                                  openapiEventList_t *event_mask);

/*********************************************************************
* @purpose  Check what events are pending for this client.
*
* @param    client_handle       @b{(input)}  Client handle from registration API.
* @param    changed_mask        @b{(output)} Table-Change event is pending
*                                              for the tables in the bit mask.
* @param    purged_mask         @b{(output)} Table-Purge event is pending
*                                              for the tables in the bit mask.
* 
* @returns  OPEN_E_NONE         Got the event mask.
* @returns  OPEN_E_FAIL         Unexpected failure.
*
* @notes    The pending flag is cleared automatically after it is read.
*           Use the openapiEventIsSet api to examine the event mask.
* 
* @supported-in-version OpEN API Version: 1.1
* 
* @end
*********************************************************************/
open_error_t openapiPendingEventsGet(openapiClientHandle_t *client_handle,
                                     openapiEventList_t *changed_mask,
                                     openapiEventList_t *purged_mask);

/*********************************************************************
* @purpose  Wait for Open API events for this client.
*
* @param    client_handle       @b{(input)}  Client handle from registration API.
* @param    timeout_secs        @b{(input)}  Number of seconds to wait.
*                                                0 - Do not wait. Return immediately.
*                                              < 0 - Wait forever.
*                                              > 0 - Wait specified number of seconds.
* 
* @returns  OPEN_E_NONE         Event is pending for this client.
* @returns  OPEN_E_TIMEOUT      Timed out while waiting for event.
*
* @notes    If the timeout_secs is 0 and there are no pending 
*           events then the function returns OPEN_E_TIMEOUT. 
* 
* @supported-in-version OpEN API Version: 1.1
* 
* @end
*********************************************************************/
open_error_t openapiEventWait(openapiClientHandle_t *client_handle,
                              int32_t timeout_secs);


/*********************************************************************
* @purpose  Get the maximum number of interfaces on this switch.
*
* @param    client_handle       @b{(input)}  Client handle from registration API.
* @param    max_interface_count @b{(output)} Number of interfaces.
*
* @returns  OPEN_E_NONE         Got the number of interfaces.
* @returns  OPEN_E_INTERNAL     Unexpected error.
*
* @notes    
* 
* @supported-in-version OpEN API Version: 1.1
* 
* @end
*********************************************************************/
open_error_t openapiMaxInterfaceCountGet(openapiClientHandle_t *client_handle,
                                         uint32_t *max_interface_count);

/*********************************************************************
* @purpose  Determine whether the system is currently operational
*           or not operational. The function reflects whether the 
*           system is in the "Configurator Execute" state.
*
* @param    client_handle       @b{(input)}  Client handle from registration API.
* @param    system_is_ready     @b{(output)} 1 - System is ready, 0 - Not Ready
*
* @returns  OPEN_E_NONE         Got the system state.
*           OPEN_E_INTERNAL     Unexpected error.
*
* @notes    
* 
* @supported-in-version OpEN API Version: 1.1
* 
* @end
*********************************************************************/
open_error_t openapiSystemReadyStateGet(openapiClientHandle_t *client_handle,
                                         uint32_t *system_is_ready);

/*********************************************************************
* @purpose  Get the next configurator state transition event.
*
* @param    client_handle       @b{(input)}  Client handle from registration API.
* @param    system_is_ready     @b{(output)} 1 - System is ready, 0 - Not Ready
*
* @returns  OPEN_E_NONE         Got the system state.
*           OPEN_E_INTERNAL     Unexpected error.
*
* @notes    This function uses a static variable and is not re-entrant.
*           Make sure that in a multi-threaded application only one
*           thread calls this function.
* 
* @supported-in-version OpEN API Version: 1.1
* 
* @end
*********************************************************************/
open_error_t openapiSystemReadyStateTransitionNextGet(
                                         openapiClientHandle_t *client_handle,
                                         uint32_t *system_is_ready);

/*********************************************************************
* @purpose  Accessor function to retrieve the socket FD for event
*           notifications from OpEN.
*
* @param    client_handle       @b{(input)}  Client handle from registration API.
* 
* @returns  int                 FD for the notify socket.
*
* @supported-in-version OpEN API Version: 1.1
* 
* @end
*********************************************************************/
int openapiClientNotifySocketFDGet(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Return code indicates whether the OpEN remote procedure
*           call infrastructure is ready.
*
* @param    client_handle       @b{(input)}  Client handle from registration API.
* 
* @returns  OPEN_E_NONE         OpEN is ready to accept RPC calls.
* @returns  OPEN_E_RPC          OpEN is not ready to accept RPC calls.
*
* @notes    This API is used to synchronize initialization of processes
*           using RPC calls to the network operating system.
*
* @supported-in-version OpEN API Version: 1.1
* 
* @end
*********************************************************************/
open_error_t openapiConnectivityCheck(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Returns the revision of the network operating system running on the switch.
*
* @param    client_handle       @b{(input)}  Client handle from registration API.
* @param    version             @b{(output)} Pointer to structure that is used to
*                                              store revision data.
*
* @returns  OPEN_E_NONE         Revision data successfully retrieved.
* @returns  OPEN_E_FAIL         Revision data could not be retrieved.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
* 
* @end
*********************************************************************/
open_error_t openapiNetworkOSVersionGet(openapiClientHandle_t *client_handle,
                                        open_buffdesc *version);

/*********************************************************************
* @purpose  Get the maximum size of inet address length.
* 
* @param    client_handle      @b{(input)}  Client handle from 
*                                           registration API
* @param    inetAddrMaxSize    @b{(output)} Inet address maximum length
*
* @returns  OPEN_E_NONE        if maximum size of inet address is returned.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL     if not supported for this platform.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiInetAddrMaxSizeGet(openapiClientHandle_t *client_handle,
                                       uint32_t *inetAddrMaxSize);

/*********************************************************************
* @purpose  Returns hardware version of the switch.
*
* @param    client_handle       @b{(input)}  Client handle from registration API.
* @param    version             @b{(output)} Pointer to structure that is used to
*                                              store revision data.
*           -                                Maximum buffer allocated size 
*           -                                @ref OPENAPI_VER_BUFF_MAX_SIZE
*
* @returns  OPEN_E_NONE         Revision data successfully retrieved.
* @returns  OPEN_E_FAIL         Revision data could not be retrieved.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      Parameter/feature is not supported.
* @returns  OPEN_E_INTERNAL     Internal error.
*
* @supported-in-version OpEN API Version: 1.12
* 
* @end
*********************************************************************/
open_error_t openapiHardwareVersionGet(openapiClientHandle_t *client_handle,
                                        open_buffdesc *version);

/*********************************************************************
* @purpose  Check if the two IP address strings refer to the same address.
*
* @param    client_handle      @b{(input)}  Client handle from
*                                           registration API
* @param    ip1Str             @b{(input)}  IP address string one
* @param    ip2Str             @b{(input)}  IP address string two
* @param    equal              @b{(output)} OPEN_TRUE if equal, else OPEN_FALSE
*
* @returns  OPEN_E_NONE        Success
* @returns  OPEN_E_FAIL        Failure occurred when parsing one of the supplied IP address strings
* @returns  OPEN_E_PARAM       An invalid parameter was passed.
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiInetAddrCompare(openapiClientHandle_t *client_handle,
                                    open_buffdesc *ip1Str,
                                    open_buffdesc *ip2Str,
                                    OPEN_BOOL_t *equal);

/*********************************************************************
* @purpose  Get the IP address in network byte order from string format.
* 
* @param    client_handle       @b{(input)}  Client handle from registration API.
* @param    ipStr               @b{(input)}  IP address string.
* @param    ipAddr              @b{(output)} IP address in inet address format.
*
* @returns  OPEN_E_NONE         Valid IP address is passed and is 
*                                 converted to inet format.
* @returns  OPEN_E_FAIL         Invalid ip address string is passed.
* @returns  OPEN_E_PARAM        Invalid parameter is passed.
*
* @notes This API checks if the address passed is IPv4 or IPv6 address 
*        and updates the parameter of type open_inet_addr_t.
* 
* @supported-in-version OpEN API Version: 1.1
* 
* @end
*********************************************************************/
open_error_t openapiInetAddrGet(openapiClientHandle_t *client_handle,
                                open_buffdesc *ipStr,
                                open_inet_addr_t *ipAddr);

/*********************************************************************
* @purpose  Converts network address string containing either an IPv4/IPv6
*           address or a host name into an inetAddr in network byte order.
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    ipOrHostStr     @b{(input)}  Buffer containing string representation
*                                        of address to convert.
* @param    inetAddr        @b{(output)} IPv4 or IPv6 address in network byte
*                                        order corresponding to address.
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters is not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.10
* 
* @notes    Buffer containing the host string should have no more than
*           @ref OPENAPI_DNS_HOST_NAME_LEN_MAX in length.
*
* @end
*********************************************************************/
open_error_t openapiInetAddrFromIPAddrHostNameStr(openapiClientHandle_t *client_handle, 
                                                  open_buffdesc *ipOrHostStr,
                                                  open_inet_addr_t *inetAddr);

/*********************************************************************
* @purpose  Send the ping response to the client(CLI/Telnet/SSH) session.
*
* @param    client_handle       @b{(input)}  Client handle from registration API.
* @param    seqNo               @b{(input)}  Seq. Nb of ping response.
* @param    rtt                 @b{(input)}  Response time.
* @param    type                @b{(input)}  Response ICMP packet type.
* @param    srcIp               @b{(input)}  Source IP of the responding node.
*           
* @returns  OPEN_E_NONE         Successfully passed output to the client.
* @returns  OPEN_E_FAIL         Failed to pass output to the client.
*
* @notes This API passes the ping output back to the client.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiPingCallback(openapiClientHandle_t *client_handle,
                                uint16_t seqNo, uint32_t rtt, uint32_t type, open_buffdesc *srcIp );

/*********************************************************************
* @purpose  Send the traceroute response to the client(CLI/Telnet/SSH) session.
*
* @param    client_handle       @b{(input)}  Client handle from registration API.
* @param    numProbes           @b{(input)}  Number of probes issued per hop.
* @param    ttl                 @b{(input)}  Starting Time to live value.
* @param    ipDa                @b{(input)}  Destination IP address.
* @param    hopCount            @b{(input)}  The hop for current traceroute probes.
* @param    probeCount          @b{(input)}  The current probe number.
* @param    rtt                 @b{(input)}  Response time.
* @param    errSym              @b{(input)}  The error symbol.
*
* @returns  OPEN_E_NONE         Successfully passed output to the client.
* @returns  OPEN_E_FAIL         Failed to pass output to the client. 
*
* @notes This API passes the traceroute output back to the client.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/

open_error_t openapiTraceRouteCallbackFn(openapiClientHandle_t *client_handle, uint32_t numProbes, uint32_t ttl, uint32_t ipDa,
                                uint16_t hopCount, uint16_t probeCount, uint32_t rtt, uint8_t errSym);

/*********************************************************************
* @purpose  Get the current active SDM template in the system. 
* 
* @param    client_handle       @b{(input)}  Client handle from registration API.
* @param    sdm_template        @b{(input)}  Current active SDM template.
* 
* @returns  OPEN_E_NONE         Successfully passed output to the client.
* @returns  OPEN_E_FAIL         Failed to pass output to the client.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiCurrentSDMGet(openapiClientHandle_t *client_handle, uint32_t *sdm_template);

/*********************************************************************
* @purpose  Check if the given IP address is zero or not
* 
* @param    client_handle       @b{(input)}  Client handle from registration API.
* @param    ipAddr              @b{(input)}  IP address in inet address format.
* @param    isZero              @b{(output)} Returns @ref OPEN_TRUE if the given IP address 
*                                              is '0', @ref OPEN_FALSE if not.
*
* @returns  OPEN_E_NONE         Valid IP address is passed. 
* @returns  OPEN_E_PARAM        Invalid parameter is passed.
*
* @notes This API checks if the address (IPv4 or IPv6) passed is '0 or not,
*        and returns @ref OPEN_TRUE if the address is '0' and returns @ref OPEN_FALSE
*        if the address is not a '0'.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiIsInetAddrZero (openapiClientHandle_t * client_handle,
                                    open_inet_addr_t ipAddr,
                                    OPEN_BOOL_t * isZero);

/********************************************************************* 
* @purpose  Get maximum length of interface name
* 
* @param    client_handle       @b{(input)}  Client handle from registration API.
* @param    maxLen              @b{(output)} Maximum length of interface name:
*                                              @ref OPEN_INTF_NAME_MAX_LENGTH
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes    
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiIntfNameMaxLengthGet (openapiClientHandle_t * client_handle,
                                          uint32_t * maxLen);

/*********************************************************************
* @purpose  Get interface name for a given internal interface number
*
* @param    client_handle       @b{(input)}  Client handle from registration API.
* @param    intfNum             @b{(input)}  Internal interface name.
* @param    bufdName            @b{(output)} Name of the interface.
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiIntfNameGet (openapiClientHandle_t * client_handle,
                                 uint32_t intfNum, 
                                 open_buffdesc * bufdName);

/*********************************************************************
* @purpose  Get maximum string length of IP address
*
* @param    client_handle       @b{(input)}   Client handle from registration API.
* @param    maxLen              @b{(output)}  Maximum length of IP address string length:
*                                               @ref OPEN_IP_ADDR_DISP_LENGTH
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiIpAddressMaxStringLength (openapiClientHandle_t * client_handle,
                                              uint32_t * maxLen);

/********************************************************************* 
* @purpose  Get human readable string format of given IP address
* 
* @param    client_handle       @b{(input)}  Client handle from registration API.
* @param    openAddr            @b{(input)}  IP address to convert to string.
* @param    bufdName            @b{(output)} IP address in string format.
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* 
* @notes    None
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiOpenIPtoStringGet (openapiClientHandle_t * client_handle,
                                       open_inet_addr_t openAddr,
                                       open_buffdesc * bufdName);

/********************************************************************* 
* @purpose  Get the maximum length of a file name (including its path) in the file system. 
*
* @param    client_handle       @b{(input)}  Client handle from registration API.
* @param    fileNameLen         @b{(output)} Maximum length of a file name.
*
* @returns  OPEN_E_NONE         Operation succeeded.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* 
* @notes    The maximum length of a file path and name varies by platform.
* 
* @notes    For optimal usage, assume that the output length value does
*           not include room for the NULL string terminator character
*           when getting a file path/name from the system, meaning
*           add 1 to this value when creating an output buffer large enough
*           to receive it. Conversely, when setting a file path/name string
*           in the system, assume this length value includes the NULL
*           terminator (i.e. actual file path/name string allowed is 
*           one character less).
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiSystemFileNameMaxLengthGet(openapiClientHandle_t *client_handle,
                                               uint32_t *fileNameLen);

/********************************************************************* 
* @purpose  Get the maximum length of an image file name (excluding its path)
*           in the file system.
*
* @param    client_handle       @b{(input)}  Client handle from registration API.
* @param    fileNameLen         @b{(output)} Maximum length of a file name.
*
* @returns  OPEN_E_NONE         Operation succeeded.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    The maximum length of a file name varies by platform.
*
* @notes    For optimal usage, assume that the output length value does
*           not include room for the NULL string terminator character
*           when getting a file name from the system, meaning add 1 to this
*           value when creating an output buffer large enough to receive it.
*           Conversely, when setting a file name string in the system, assume
*           this length value includes the NULL terminator (i.e. actual file
*           name string allowed is one character less).
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
*********************************************************************/
open_error_t openapiImageFileNameMaxLengthGet(openapiClientHandle_t *client_handle,
                                              uint32_t *fileNameLen);

/*********************************************************************
* @purpose  Get the display name of given interface
* 
* @param    client_handle       @b{(input)}  Client handle from registration API.
* @param    unit                @b{(input)}  Unit number.
* @param    slot                @b{(input)}  Slot number.
* @param    port                @b{(input)}  Port number.
* @param    bufdName            @b{(output)} String representation of interface.
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* 
* @notes    None
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiIntfDisplayNameGet (openapiClientHandle_t * client_handle, 
                                        uint32_t unit, 
                                        uint32_t slot,
                                        uint32_t port, 
                                        open_buffdesc * bufdName);

/*********************************************************************
* @purpose  Get the count of all interfaces.
* 
* @param    client_handle       @b{(input)}  Client handle from 
*                                            registration API.
* @param    intfNum             @b{(output)} All interfaces count.
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* 
* @notes    None
* 
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiAllIntfNumGet(openapiClientHandle_t *client_handle,
                                  uint32_t *intfNum);

/*********************************************************************
* @purpose  Get the MAC address in 6-byte hexadecimal format from
*           string format.
* 
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API.
* @param    inputMac         @b{(input)}  MAC address in string format.
*                                         xx:xx:xx:xx:xx:xx
*                                         Use OPEN_MAC_ADDR_STR_LEN to
*                                         get the maximum length.
* @param    outputMac        @b{(output)} MAC address in 6-byte hexadecimal
*                                         format. Use OPEN_MAC_ADDR_LEN to
*                                         get the maximum length.
*
* @returns  OPEN_E_NONE      MAC address is converted to hexadecimal
*                            format successfully.
* @returns  OPEN_E_FAIL      Failure occurred when parsing the MAC address.
* @returns  OPEN_E_PARAM     An invalid parameter was passed.
* 
* @notes    None
* 
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiMacAddrStringToHexGet(openapiClientHandle_t * client_handle,
                                          open_buffdesc *inputMac,
                                          open_buffdesc *outputMac);

/*********************************************************************
 * @purpose  Convert time (seconds) to calender time format.
 * 
 * @param    client_handle    @b{(input)}  Client handle from
 *                                         registration API.
 * @param    inputTime        @b{(input)}  Time in seconds.
 * @param    outputTime       @b{(output)} Calender time.
 *                                         yyyy-mm-dd hh:mm:ss.
 *
 * @returns  OPEN_E_NONE      Converted time in seconds to calender
 *                            time successfully.
 * @returns  OPEN_E_FAIL      Failure occurred when converting time in
 *                            seconds to calender time.
 * @returns  OPEN_E_PARAM     An invalid parameter was passed.
 * 
 * @notes    None
 * 
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *********************************************************************/
open_error_t openapiTimeToCalenderTimeGet(openapiClientHandle_t * client_handle,
                                          uint32_t inputTime,
                                          open_buffdesc *outputTime);

/*********************************************************************
* @purpose  Set the RPC client receive socket time out 
* 
* @param    client_handle  @b{(input)}  Client handle from
*                                       registration API.
* @param    timeout        @b{(input)}  Timeout value in seconds
*
* @returns  OPEN_E_NONE      Receive socket time out set successfully.
* @returns  OPEN_E_FAIL      Failure occurred while setting the timeout.
* @returns  OPEN_E_PARAM     An invalid parameter was passed.
* 
* @notes If the timeout value is 0 socket will never timeout and will wait
*        for the message (blocking call). Refer below constants
*        OPEN_RPC_CLT_RECV_DEF_TIMEOUT, OPEN_RPC_CLT_RECV_MIN_TIMEOUT
*        OPEN_RPC_CLT_RECV_MAX_TIMEOUT
*
* @notes Setting the client socket to non blocking with timeout is not 
*        recommended when same client handle is used across multiple threads.
*        In slow platforms responses from server (RPC and syncdb) may be slow
*        and unintended responses may received by the open api client. It is 
*        recommended to use different client handles if single process with 
*        multiple threads are using openAPI interface.
*
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiClientReceiveTimeoutSet(openapiClientHandle_t *client_handle,
                                            uint32_t timeout);

/*********************************************************************
* @purpose  Get the RPC client receive socket time out 
* 
* @param    client_handle  @b{(input)}  Client handle from
*                                       registration API.
* @param    timeout        @b{(output)} Timeout value in seconds
*
* @returns  OPEN_E_NONE      Receive socket time out set successfully.
* @returns  OPEN_E_PARAM     An invalid parameter was passed.
* 
* @notes  None 
* 
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiClientReceiveTimeoutGet(openapiClientHandle_t *client_handle,
                                            uint32_t *timeout);

/*********************************************************************
* @purpose  Get the RPC client statistics 
* 
* @param    stats        @b{(output)} Statistics of RPC client
*
* @returns  OPEN_E_NONE      Receive socket statistics get successfully 
* @returns  OPEN_E_PARAM     An invalid parameter was passed.
* 
* @notes none
* 
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiClientRpcStatsGet(openapiClientRpcStats_t *stats);

/*********************************************************************
* @purpose  Print the RPC client statistics 
* 
* @returns  OPEN_E_NONE      Receive socket statistics printed successfully 
*                            on the stdout
* 
* @notes none
* 
* @supported-in-version OpEN API Version: 1.27
*
* @end
*********************************************************************/
open_error_t openapiClientStatsPrint(void);

#endif  /* _OPENAPI_COMMON_H_ */
