/*********************************************************************
*
*  Copyright 2017-2024 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_cos.h
*
* @purpose   Class of Service Configuration
*
* @component OpEN
*
* @create    01/11/2017
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_COS_H_INCLUDED
#define OPENAPI_COS_H_INCLUDED

#include "openapi_common.h"

#define OPEN_COS_ALL_INTERFACES 0
#define OPEN_COS_QUEUE_ID_MIN   0

#define OPEN_MIN_CFG_DROP_PREC_LEVELS 1
#define OPEN_MAX_CFG_DROP_PREC_LEVELS 4

/** QOS COS mapping trust modes */
typedef enum
{
  OPEN_QOS_COS_MAP_TRUST_MODE_INVALID = 0,  /**<  trust mode invalid */
  OPEN_QOS_COS_MAP_TRUST_MODE_UNTRUSTED,    /**<  do not trust any pkt markings */
  OPEN_QOS_COS_MAP_TRUST_MODE_DOT1P,        /**<  trust pkt Dot1p (802.1p) value */
  OPEN_QOS_COS_MAP_TRUST_MODE_IPPREC,       /**<  trust pkt IP Precedence value */
  OPEN_QOS_COS_MAP_TRUST_MODE_IPDSCP        /**<  trust pkt IP DSCP value */
} OPEN_QOS_COS_MAP_TRUST_MODE_t;

/** QOS COS queue scheduler types */
typedef enum
{
  OPEN_QOS_COS_QUEUE_SCHED_TYPE_STRICT = 1,   /**< strict priority */
  OPEN_QOS_COS_QUEUE_SCHED_TYPE_WEIGHTED      /**< WRR/WFQ */
} OPEN_QOS_COS_QUEUE_SCHED_TYPE_t;

/** QOS COS queue management types */
typedef enum
{
  OPEN_QOS_COS_QUEUE_MGMT_TYPE_TAILDROP = 1   /**<  tail drop */
} OPEN_QOS_COS_QUEUE_MGMT_TYPE_t;

/*********************************************************************
* @purpose  Get the units used to express rates.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    rateUnit         @b{(output)} rate unit identifier
*
* @returns  OPEN_E_NONE      value returned successfully
* @returns  OPEN_E_PARAM     invalid parameter passed
*
* @notes Various platforms use different ways of expressing traffic rates.
*        Examples of the rate unit are kilobytes per second (KBPS) and
*        percentage of link speed. This function returns the unit used
*        on the switch. The caller should use this information to correctly
*        set parameter values for rates when using other APIs in this file.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueIntfShapingRateUnitsGet(openapiClientHandle_t *client_handle, OPEN_RATE_UNIT_t *rateUnit);

/*********************************************************************
* @purpose  Get the minimum increment allowed when percentage of link
*           speed is used to express rates.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    increment        @b{(output)} minimum percentage increment
*
* @returns  OPEN_E_NONE      value returned successfully
* @returns  OPEN_E_ERROR     platform does not use percentage to express rates
* @returns  OPEN_E_PARAM     invalid parameter passed
*
* @notes For platforms using rate expressed as percentages, the rates are
*        expressed in whole percentage values. In some cases, there are
*        restrictions on the values for the percentage parameter. For example,
*        if this function returns 5 for the increment, only percentage values
*        of 0, 5, 10, 15, ... , 95, 100 are accepted. In many cases, the
*        value of increment is 1.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueIntfShapingPercentageIncrementGet(openapiClientHandle_t *client_handle, uint32_t *increment);

/*********************************************************************
* @purpose  Get the COS egress shaping rate for an interface.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    ifNum            @b{(input)}  interface number
* @param    rate             @b{(output)} shaping rate value
*
* @returns  OPEN_E_NONE      value returned successfully
* @returns  OPEN_E_UNAVAIL   platform does not support COS egress shaping
* @returns  OPEN_E_FAIL      failed to get shaping rate
* @returns  OPEN_E_PARAM     invalid parameter passed
*
* @notes The value for rate is interpreted based on the data returned from
*        @ref openapiCosQueueIntfSapingRateUnitsGet().
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueIntfShapingRateGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *rate);

/*********************************************************************
* @purpose  Set the COS egress shaping rate for an interface.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    ifNum            @b{(input)}  interface number
* @param    rate             @b{(input)}  shaping rate value
*
* @returns  OPEN_E_NONE      value set successfully
* @returns  OPEN_E_UNAVAIL   platform does not support COS egress shaping
* @returns  OPEN_E_FAIL      failed to set shaping rate
* @returns  OPEN_E_PARAM     invalid parameter passed
*
*
* @notes The value for rate is interpreted based on the data returned from
*        @ref openapiCosQueueIntfSapingRateUnitsGet().
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueIntfShapingRateSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t rate);

/*********************************************************************
* @purpose  Set  WRED and ECN drop configuration on a given queue
*           for a given or all interfaces.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    intfNum             @b{(input)}  interface number
* @param    queueNum            @b{(input)}  queue number
* @param    dropPrecedence      @b{(input)}  drop precedence
* @param    ecnEnabled          @b{(input)}  ECN enabled
* @param    wredMinThreshold    @b{(input)}  WRED minimum threshold
* @param    wredMaxThreshold    @b{(input)}  WRED maximum threshold
* @param    wredDropProbability @b{(input)}  WRED drop probability
*
* @returns  OPEN_E_NONE         value set successfully
* @returns  OPEN_E_UNAVAIL      platform does not support WRED or ECN feature
* @returns  OPEN_E_FAIL         failed to set or get drop params config
* @returns  OPEN_E_PARAM        invalid parameter passed
*
* @notes    If intfNum is zero then WRED drop configuration will be applied
*           on all interfaces.
* @notes    Refer openapiCosQueueMaxIdGet() for max cos queue ID value.
* @notes    Refer OPEN_COS_QUEUE_ID_MIN for min cos queue ID value.
* @notes    Refer OPEN_MIN_CFG_DROP_PREC_LEVELS for minimum drop precedence levels
* @notes    Refer OPEN_MAX_CFG_DROP_PREC_LEVELS for maximum drop precedence levels
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueWredDropParamsSet(openapiClientHandle_t *client_handle, uint32_t intfNum, uint8_t queueNum,
                                              uint8_t dropPrecedence, OPEN_BOOL_t ecnEnabled, int16_t wredMinThreshold,
                                              int16_t wredMaxThreshold, int16_t wredDropProbability);

/*********************************************************************
* @purpose  Reset WRED and ECN drop configuration on a given queue
*           for a given or all interfaces.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    intfNum             @b{(input)}  interface number
* @param    queueNum            @b{(input)}  queue number
*
* @returns  OPEN_E_NONE         value set successfully
* @returns  OPEN_E_UNAVAIL      platform does not support WRED feature
* @returns  OPEN_E_FAIL         failed to reset drop params config
*
* @notes    If intfNum is zero then WRED drop configuration will be reset
*           on all interfaces.
* @notes    Refer openapiCosQueueMaxIdGet() for max cos queue ID value.
* @notes    Refer OPEN_COS_QUEUE_ID_MIN for min cos queue ID value.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueWredDropParamsReset(openapiClientHandle_t *client_handle, uint32_t intfNum, uint8_t queueNum);

/*********************************************************************
* @purpose  Get WRED drop and ECN marked counters for a given or all interfaces.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    intfNum             @b{(input)}  interface number
* @param    dropCount           @b{(output)} WRED drop counters
* @param    ecnMarked           @b{(output)} ECN marked counters
*
* @returns  OPEN_E_NONE         counters retrieved successfully
* @returns  OPEN_E_FAIL         unable to retrieve counters
* @returns  OPEN_E_UNAVAIL      feature is not supported on the given interface
*
* @notes    If intfNum is zero then WRED stats wil be retrieved
*           on all interfaces. Physical interfaces are the only valid
*           interfaces.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueWredStatsGet(openapiClientHandle_t *client_handle, uint32_t intfNum, 
                                         uint64_t *dropCount, uint64_t *ecnMarked);

/*********************************************************************
* @purpose  Clear WRED and ECN counters for a given or all interfaces.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    intfNum             @b{(input)}  interface number
*
* @returns  OPEN_E_NONE         counters reset successfully
* @returns  OPEN_E_FAIL         unable to reset counters
*
* @notes    If intfNum is zero then WRED stats will be cleared
*           on all interfaces.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueWredStatsClear(openapiClientHandle_t *client_handle, uint32_t intfNum);

/*********************************************************************
* @purpose  Enable or disable WRED for a given queue and for a given or all interfaces.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    intfNum             @b{(input)}  interface number
* @param    queueNum            @b{(input)}  queue number
* @param    enableWred          @b{(input)}  OPEN_TRUE to enable WRED
*                                            OPEN_FALSE to enable tail drop queue management
*
* @returns  OPEN_E_NONE         enabled WRED successfully
* @returns  OPEN_E_FAIL         unable to enable WRED
*
* @notes    If intfNum is zero then WRED drop configuration will be enabled
*           or disabled on all interfaces.
*
* @notes    Refer openapiCosQueueMaxIdGet() for max cos queue ID value.
* @notes    Refer OPEN_COS_QUEUE_ID_MIN for min cos queue ID value.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueWredSet(openapiClientHandle_t *client_handle,
                                    uint32_t intfNum, uint8_t queueNum,
                                    OPEN_BOOL_t enableWred);

/*********************************************************************
* @purpose  Get the max cos queue ID value
*
* @param    client_handle       @b{(input)}   Client handle from registration API
* @param    maxQueueId          @b{(output)}  Queue number
*
* @returns  OPEN_E_NONE         Max cos queue ID returned successfully
* @returns  OPEN_E_PARAM        Invalid parameter passed
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiCosQueueMaxIdGet(openapiClientHandle_t *client_handle,
                                     uint32_t *maxQueueId);

/*****************************************************************************
 * @purpose   Sets the CoS global trust mode.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     trustMode          @b{(input)}  Trust mode
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets CoS global trust mode.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t 
openapiQosCosMapGlobalTrustModeSet(openapiClientHandle_t *client_handle,
                                   OPEN_QOS_COS_MAP_TRUST_MODE_t trustMode);

/*****************************************************************************
 * @purpose   Gets CoS global trust mode.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     pTrustMode         @b{(output)} Trust mode
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches global trust mode.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosMapGlobalTrustModeGet(openapiClientHandle_t *client_handle,
                                   OPEN_QOS_COS_MAP_TRUST_MODE_t *pTrustMode);

/*****************************************************************************
 * @purpose   Sets the CoS interface trust mode.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 * @param     trustMode          @b{(input)}  Trust mode
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets CoS interface trust mode.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosMapInterfaceTrustModeSet(openapiClientHandle_t *client_handle,
                                      uint32_t intIfNum,
                                      OPEN_QOS_COS_MAP_TRUST_MODE_t trustMode);

/*****************************************************************************
 * @purpose   Gets CoS internal trust mode.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 * @param     pTrustMode         @b{(output)} Trust mode
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches interface trust mode.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosMapInterfaceTrustModeGet(openapiClientHandle_t *client_handle,
                                      uint32_t intIfNum,
                                      OPEN_QOS_COS_MAP_TRUST_MODE_t *pTrustMode);

/*****************************************************************************
 * @purpose   Gets specified trust mode interface index status.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches existence status of specified trust mode interface.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosMapTrustModeIntfIndexGet(openapiClientHandle_t *client_handle,
                                      uint32_t intIfNum);

/*****************************************************************************
 * @purpose   Gets next sequential specified trust mode interface.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 * @param     pNxtIntIfNum       @b{(output)} Next internal interface number
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches next sequential specified trust mode interface.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosMapTrustModeIntfIndexGetNext(openapiClientHandle_t *client_handle,
                                          uint32_t intIfNum,
                                          uint32_t *pNxtIntIfNum);

/*****************************************************************************
 * @purpose   Gets CoS untrusted port default traffic class for given internal interface.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 * @param     pTcVal             @b{(output)} untrusted traffic class value
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches CoS untrusted port default traffic class for given internal interface.
 * @notes     For global config operation intIfNum fetch using
 *            openapiQosCosMapGetIntfIndexForGlobalConfig() should be used.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosMapUntrustedPortDefaultTrafficClassGet(openapiClientHandle_t *client_handle,
                                                    uint32_t intIfNum, uint32_t *pTcVal);

/*****************************************************************************
 * @purpose   Gets specified queue config interface index status.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches existence status of specified queue config interface.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosQueueIntfIndexGet(openapiClientHandle_t *client_handle,
                               uint32_t intIfNum);

/*****************************************************************************
 * @purpose   Gets next sequential specified queue config interface index.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 * @param     pNxtIntIfNum       @b{(output)} Next internal interface number
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches next sequential specified queue config interface.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosQueueIntfIndexGetNext(openapiClientHandle_t *client_handle,
                                   uint32_t intIfNum, uint32_t *pNxtIntIfNum);

/*****************************************************************************
 * @purpose   Gets if the specified interface is valid for COS queue config.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 * @param     pIsQueueIntfValid  @b{(output)} If interface is valid for queue 
 *                                            config then OPEN_TRUE, else OPEN_FALSE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches state of specified interface is valid for COS queue config.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosQueueIntfIsValid(openapiClientHandle_t *client_handle,
                              uint32_t intIfNum, OPEN_BOOL_t *pIsQueueIntfValid);

/*****************************************************************************
 * @purpose   Gets specified queueId  index status.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     queueId            @b{(input)}  queueId
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches existence status of specified queueId.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosQueueIdIndexGet(openapiClientHandle_t *client_handle,
                             uint32_t queueId);

/*****************************************************************************
 * @purpose   Gets next sequential specified queueId index.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     queueId            @b{(input)}  Queue id
 * @param     pNxtqueueId        @b{(output)} Next Queue id
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches next sequential specified queueId.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosQueueIdIndexGetNext(openapiClientHandle_t *client_handle,
                                 uint32_t queueId, uint32_t *pNxtqueueId);

/*****************************************************************************
 * @purpose   Sets minimum bandwidth for a specific queue on a given interface.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 * @param     queueId            @b{(input)}  Queue id
 * @param     minBwVal           @b{(input)}  Min bandwidth value
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets minimum bandwidth for a specific queue of a given interface.
 * @notes     For global config operation intIfNum fetch using
 *            openapiQosCosMapGetIntfIndexForGlobalConfig() should be used.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosQueueMinBandwidthSet(openapiClientHandle_t *client_handle, 
                                  uint32_t intIfNum, uint32_t queueId,
                                  uint32_t minBwVal);

/*****************************************************************************
 * @purpose   Gets minimum bandwidth list for all queues on a given interface.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 * @param     pMinBwList         @b{(output)} Minimum bandwidth list
 *                                            list is in the format of queueId:minBwVal,
 *                                            and would hold information for all queues.
 *                                            (i.e) queueId0:minBwVal0, .... queueId7:minBwVal7
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches list of minimum bandwidth for all queues on a given interface.
 * @notes     For global config operation intIfNum fetch using
 *            openapiQosCosMapGetIntfIndexForGlobalConfig() should be used.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosQueueMinBandwidthListGet(openapiClientHandle_t *client_handle,
                                      uint32_t intIfNum, open_buffdesc *pMinBwList);

/*****************************************************************************
 * @purpose   Sets scheduler type for a specific queue on a given interface.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 * @param     queueId            @b{(input)}  Queue id
 * @param     schTypeVal         @b{(input)}  Scheduler type value
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets scheduler type for a specific queue of a given interface.
 * @notes     For global config operation intIfNum fetch using
 *            openapiQosCosMapGetIntfIndexForGlobalConfig() should be used.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosQueueSchedulerTypeSet(openapiClientHandle_t *client_handle,
                                   uint32_t intIfNum, uint32_t queueId,
                                   OPEN_QOS_COS_QUEUE_SCHED_TYPE_t schTypeVal);

/*****************************************************************************
 * @purpose   Gets scheduler type list for all queues on a given interface.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 * @param     pSchTypeList       @b{(output)} Scheduler type list
 *                                            list is in the format of queueId:schType,
 *                                            and would hold information for all queues.
 *                                            (i.e) queueId0:schType0, .... queueId7:schType7
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches list of scheduler type for all queues on a given interface.
 * @notes     For global config operation intIfNum fetch using
 *            openapiQosCosMapGetIntfIndexForGlobalConfig() should be used.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosQueueSchedulerTypeListGet(openapiClientHandle_t *client_handle,
                                       uint32_t intIfNum, open_buffdesc *pSchTypeList);

/*****************************************************************************
 * @purpose   Sets queue management type for a specific queue on a given interface.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 * @param     queueId            @b{(input)}  Queue id
 * @param     qMgmtType          @b{(input)}  Queue mgmt type value
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets queue management type for a specific queue of a given interface.
 * @notes     For global config operation intIfNum fetch using
 *            openapiQosCosMapGetIntfIndexForGlobalConfig() should be used.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosQueueMgmtTypeSet(openapiClientHandle_t *client_handle,
                              uint32_t intIfNum, uint32_t queueId,
                              OPEN_QOS_COS_QUEUE_MGMT_TYPE_t qMgmtType);

/*****************************************************************************
 * @purpose   Gets queue management type list for all queues on a given interface.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 * @param     pMgmtTypeList      @b{(output)} Queue mgmt type list
 *                                            list is in the format of queueId:mgmtType,
 *                                            and would hold information for all queues.
 *                                            (i.e) queueId0:mgmtType0, .... queueId7:mgmtType7
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches list of queue management type for all queues on a given interface.
 * @notes     For global config operation intIfNum fetch using
 *            openapiQosCosMapGetIntfIndexForGlobalConfig() should be used.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosQueueMgmtTypeListGet(openapiClientHandle_t *client_handle,
                                  uint32_t intIfNum, open_buffdesc *pMgmtTypeList);

/*****************************************************************************
 * @purpose   Sets dot1d traffic class.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 * @param     priority           @b{(input)}  priority
 * @param     tc                 @b{(input)}  traffic class
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets dot1d traffic class.
 * @notes     For global config operation intIfNum fetch using
 *            openapiQosCosMapGetIntfIndexForGlobalConfig() should be used.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosDot1dTrafficClassSet(openapiClientHandle_t *client_handle,
                                  uint32_t intIfNum, uint32_t priority,
                                  uint32_t tc);

/*****************************************************************************
 * @purpose   Gets dot1d traffic class information.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 * @param     priority           @b{(input)}  priority
 * @param     pTc                @b{(output)} traffic class
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches dot1d traffic class information for a given priority on a given interface.
 * @notes     For global config operation intIfNum fetch using
 *            openapiQosCosMapGetIntfIndexForGlobalConfig() should be used.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosDot1dTrafficClassGet(openapiClientHandle_t *client_handle,
                                  uint32_t intIfNum, uint32_t priority,
                                  uint32_t *pTc);

/*****************************************************************************
 * @purpose   Gets status of user priority to traffic class mapping for a given interface.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 * @param     priority           @b{(input)}  priority
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches status of user priority to traffic class mapping for a given interface.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosDot1dTrafficClassEntryGet(openapiClientHandle_t *client_handle,
                                       uint32_t intIfNum, uint32_t priority);

/*****************************************************************************
 * @purpose   Gets next sequential user priority to traffic class mapping.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     pIntIfNum          @b{(inout)}  Internal interface number
 * @param     pPriority          @b{(inout)}  priority
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches next sequential user priority to traffic class mapping.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosDot1dTrafficClassEntryNextGet(openapiClientHandle_t *client_handle,
                                           uint32_t *pIntIfNum, uint32_t *pPriority);

/*****************************************************************************
 * @purpose   Gets factory default dot1d traffic class information.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 * @param     priority           @b{(input)}  priority
 * @param     pTc                @b{(output)} traffic class
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches factory default dot1d traffic class information for a given priority on a given interface.
 * @notes     For global config operation intIfNum fetch using
 *            openapiQosCosMapGetIntfIndexForGlobalConfig() should be used.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosDot1dDefaultTrafficClassGet(openapiClientHandle_t *client_handle,
                                         uint32_t intIfNum, uint32_t priority,
                                         uint32_t *pTc);
/*****************************************************************************
 * @purpose   Gets status of specified IP DSCP mapping table index.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     dscp               @b{(input)}  Ip DSCP
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches status of specified IP DSCP mapping table index.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosMapIpDscpIndexGet(openapiClientHandle_t *client_handle,
                               uint32_t dscp);

/*****************************************************************************
 * @purpose   Gets next sequential ip DSCP mapping table index.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     dscp               @b{(input)}  Ip DSCP
 * @param     pNxtDscp           @b{(output)} Next Ip DSCP
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches next sequential ip DSCP mapping table index.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosMapIpDscpIndexGetNext(openapiClientHandle_t *client_handle,
                                   uint32_t dscp, uint32_t *pNxtDscp);

/*****************************************************************************
 * @purpose   Sets the assigned traffic class (queue) for given Ip DSCP.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 * @param     dscp               @b{(input)}  Ip DSCP
 * @param     tc                 @b{(input)}  traffic class
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets the assigned traffic class (queue) for given Ip DSCP.
 * @notes     For global config operation intIfNum fetch using
 *            openapiQosCosMapGetIntfIndexForGlobalConfig() should be used.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosMapIpDscpTrafficClassSet(openapiClientHandle_t *client_handle,
                                      uint32_t intIfNum, uint32_t dscp,
                                      uint32_t tc);

/*****************************************************************************
 * @purpose   Gets the assigned traffic class (queue) for given ip DSCP.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 * @param     dscp               @b{(input)}  dscp
 * @param     pTc                @b{(output)} traffic class
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the assigned traffic class (queue) for given ip DSCP.
 * @notes     For global config operation intIfNum fetch using
 *            openapiQosCosMapGetIntfIndexForGlobalConfig() should be used.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosMapIpDscpTrafficClassGet(openapiClientHandle_t *client_handle,
                                      uint32_t intIfNum, uint32_t dscp,
                                      uint32_t *pTc);

/*****************************************************************************
 * @purpose   Gets the default traffic class mapping for specified ip DSCP value.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 * @param     dscp               @b{(input)}  dscp
 * @param     pTc                @b{(output)} traffic class
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the default traffic class mapping for specified ip DSCP.
 * @notes     For global config operation intIfNum fetch using
 *            openapiQosCosMapGetIntfIndexForGlobalConfig() should be used.
 *
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosMapIpDscpDefaultTrafficClassGet(openapiClientHandle_t *client_handle,
                                             uint32_t intIfNum, uint32_t dscp,
                                             uint32_t *pTc);

/*****************************************************************************
 * @purpose   Restore default ip DSCP mappings for given interface.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API restore default ip DSCP mappings for given interface.
 * @notes     For global config operation intIfNum fetch using
 *            openapiQosCosMapGetIntfIndexForGlobalConfig() should be used.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosMapIpDscpDefaultsRestore(openapiClientHandle_t *client_handle,
                                      uint32_t intIfNum);

/*****************************************************************************
 * @purpose   Gets status of specified ip DSCP mapping table interface index.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches status of specified IP DSCP mapping table interface index.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosMapIpDscpIntfIndexGet(openapiClientHandle_t *client_handle,
                                   uint32_t intIfNum);

/*****************************************************************************
 * @purpose   Gets next sequential ip DSCP mapping table interface index.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  Internal interface number
 * @param     pNxtIntIfNum       @b{(output)} Next Internal interface number
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches next sequential ip DSCP mapping table index.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosMapIpDscpIntfIndexGetNext(openapiClientHandle_t *client_handle,
                                       uint32_t intIfNum, uint32_t *pNxtIntIfNum);

/*****************************************************************************
 * @purpose   Gets intIfNum to denotes global config operation.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     pIntIfNum         @b{(output)} Internal interface number
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API returns intIfNum to denote global config operation.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiQosCosMapGetIntfIndexForGlobalConfig(openapiClientHandle_t *client_handle,
                                            uint32_t *pIntIfNum);

#endif /* OPENAPI_COS_H_INCLUDED */
