/*********************************************************************
*
* Copyright 2018-2024 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_dhcp_snooping.h
*
* @purpose   DHCP Snooping API
*
* @component OpEN
*
* @comments  This code must not include any ICOS header files.
*
* @create    7/9/2018
*
* @end
*                                    
**********************************************************************/
#ifndef OPENAPI_DHCP_SNOOPING_H_INCLUDED
#define OPENAPI_DHCP_SNOOPING_H_INCLUDED

#include "openapi_common.h"

/** DHCP L2 relay option 82 actions. */
typedef enum
{
  OPEN_DHCP_L2RELAY_NO_OPTION_82_ACTION_DROP   = 0,  /**< Drop action */
  OPEN_DHCP_L2RELAY_NO_OPTION_82_ACTION_KEEP   = 1,  /**< This action is not supported */
  OPEN_DHCP_L2RELAY_NO_OPTION_82_ACTION_UPDATE = 2  /**< Update action */
} OPEN_DHCP_L2RELAY_NO_OPTION_82_ACTION_t;

/** DHCP snooping binding type. */
#define OPEN_DS_BINDING_TENTATIVE   0      /**< Tentative binding */
#define OPEN_DS_BINDING_STATIC      1      /**< Static binding */
#define OPEN_DS_BINDING_DYNAMIC     2      /**< Dynamic binding */
#define OPEN_DS_BINDING_VOLATILE    3      /**< Volatile binding */

#define OPEN_MAC_ADDR_LEN      6
typedef struct OPEN_DHCP_SNOOP_BINDING_s
{
  unsigned char macAddr[OPEN_MAC_ADDR_LEN]; /**< A MAC address uniquely identifies a node in the bindings data base */
  unsigned short vlanId;               /**< VLAN ID */
  uint32_t ipAddr;                     /**< IP address assigned to the station */
  uint32_t intIfNum;                   /**< Physical port where client is attached */
  uint32_t remLease;                   /**< Time left on lease (minutes) */
  uint32_t allottedLeaseTime;          /**< Lease time allotted */
  uint32_t bindingType;                /**< dynamic, static, tentative, volatile */
} OPEN_DHCP_SNOOP_BINDING_t;

typedef struct OPEN_DHCP_V6_SNOOP_BINDING_s
{
  unsigned char macAddr[OPEN_MAC_ADDR_LEN]; /**< A MAC address uniquely identifies a node in the bindings data base */
  unsigned short vlanId;               /**< VLAN ID */
  open_in6_addr_t ipAddr;              /**< IPv6 address assigned to the station */
  uint32_t intIfNum;                   /**< Physical port where client is attached */
  uint32_t remLease;                   /**< Time left on lease (minutes) */
  uint32_t allottedLeaseTime;          /**< Lease time allotted */
  uint32_t bindingType;                /**< dynamic, static, tentative */
} OPEN_DHCP_V6_SNOOP_BINDING_t;

/** DHCP SNOOPING COUNTERS ID  */
#define OPEN_DS_CID_MAC_VERIFY                1
#define OPEN_DS_CID_INTF_MISMATCH             2
#define OPEN_DS_CID_UNTRUSTED_SRVR_MSG        3

#define OPEN_DS_DB_MIN_STORE_INTERVAL 15 
#define OPEN_DS_DB_MAX_STORE_INTERVAL 86400 

#define OPEN_DS_RATE_LIMIT_MIN                0
#define OPEN_DS_RATE_LIMIT_MAX              300
#define OPEN_DS_RATE_LIMIT_DEFAULT           -1 /* rate-limit disabled */
#define OPEN_DS_BURST_INTERVAL_MIN            1
#define OPEN_DS_BURST_INTERVAL_MAX           15
#define OPEN_DS_BURST_INTERVAL_DEFAULT       -1 /* No burst interval by default */
/*********************************************************************
* @purpose  Gets the DHCP Snooping Admin mode
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    family            @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    adminMode         @b{(output)} DHCP snooping admin mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    IPv6 requested but support is unavailable 
*
* @comments Use this function to determine if DHCP snooping is enabled.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingAdminModeGet(openapiClientHandle_t *client_handle,
                                             OPEN_AF_t family,
                                             OPEN_CONTROL_t *adminMode);

/*********************************************************************
* @purpose  Gets whether DHCP snooping verifies source MAC addresses.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    family            @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    sourceMacVerify   @b{(output)} If OPEN_ENABLE, source MAC addresses
*                                          verified, otherwise OPEN_DISABLE
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    IPv6 requested but support is unavailable 
*
* @comments When MAC verify is enabled (and port security is enabled
*           on the interface), DHCP snooping tells the hardware to
*           drop any incoming frames on the interface if the source MAC
*           is not the MAC address for a DHCP client who has been leased
*           an IP address.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingVerifySourceMacGet(openapiClientHandle_t *client_handle,
                                                   OPEN_AF_t family,
                                                   OPEN_CONTROL_t *sourceMacVerify);

/*********************************************************************
* @purpose  Given a VLAN ID, return the endpoints of the next range of
*           VLANs enabled for DHCP snooping.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    family            @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    vlanStart         @b{(output)} Low end of VLAN range
* @param    vlanEnd           @b{(inout)}  High end of VLAN range
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_NOT_FOUND  No further VLANs beyond vlanEnd
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    IPv6 requested but support is unavailable 
*
* @comments Search for next range starts at vlanEnd + 1. To begin a
*           search, set vlanEnd to 0. To search the next following
*           range, vlanEnd should be set to the vlanEnd returned by
*           the previous call. See example code for more details.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingEnabledVlanRangeNextGet(openapiClientHandle_t *client_handle,
                                                        OPEN_AF_t family,
                                                        uint32_t *vlanStart,
                                                        uint32_t *vlanEnd);

/*********************************************************************
* @purpose  Determines whether DHCP snooping is enabled on a port.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    family         @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    ifNum          @b{(input)}  Internal interface number
* @param    enabled        @b{(output)} OPEN_ENABLE if DHCP snooping is 
*                                       enabled on port, else OPEN_DISABLE
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    IPv6 requested but support is unavailable 
*
* @comments    DHCP snooping is considered enabled on a port if DHCP snooping
*              is enabled on one or more VLANs the port participates in.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingPortEnabledGet(openapiClientHandle_t *client_handle,
                                               OPEN_AF_t family,
                                               uint32_t ifNum, 
                                               OPEN_CONTROL_t *enabled);

/*********************************************************************
* @purpose  Determines whether DHCP snooping considers an interface as trusted.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    family         @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    ifNum          @b{(input)}  Internal interface number
* @param    trusted        @b{(output)} OPEN_TRUE if trusted, else OPEN_FALSE
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    IPv6 requested but support is unavailable 
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingIntfTrustedGet(openapiClientHandle_t *client_handle,
                                               OPEN_AF_t family,
                                               uint32_t ifNum, 
                                               OPEN_BOOL_t *trusted);

/*********************************************************************
* @purpose  Determines whether DHCP snooping logs invalid packets on an interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    family         @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    ifNum          @b{(input)}  Internal interface number
* @param    logged         @b{(output)} OPEN_TRUE if invalid packets logged for 
*                                       interface, else OPEN_FALSE
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    IPv6 requested but support is unavailable 
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingIntfLoggedGet(openapiClientHandle_t *client_handle,
                                              OPEN_AF_t family,
                                              uint32_t ifNum, 
                                              OPEN_BOOL_t *logged);

/*********************************************************************
* @purpose  Get the remote DHCP snooping database file name max length.
*
* @param    client_handle  @b{(input)}   Client handle from registration API
* @param    len            @b{(output)}  Maximum length of filename including NULL.
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingDbFileNameMaxLengthGet(openapiClientHandle_t *client_handle, uint32_t *len);

/*********************************************************************
* @purpose  Get the remote DHCP snooping database file name.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    family         @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    fileName       @b((output)) Remote DHCP snooping DB file name
* @param    -                 Minimum allowed value of buffer length is iobtainable by calling openapiDhcpSnoopingFileNameMaxLengthGet()
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingDbFileNameGet(openapiClientHandle_t *client_handle,
                                              OPEN_AF_t family,
                                              open_buffdesc *fileName);

/*********************************************************************
* @purpose  Get the remote DHCP snooping database IPv4 Address.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    family         @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    addr           @b((output)) Remote DHCP snooping DB IP Address
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingDbIpAddressGet(openapiClientHandle_t *client_handle,
                                               OPEN_AF_t family,
                                               open_inet_addr_t *addr);

/*********************************************************************
* @purpose  Get the DHCP snooping database write delay.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    family         @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    delay          @b((output)) Write delay in seconds.
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingDbWriteDelayGet(openapiClientHandle_t *client_handle,
                                                OPEN_AF_t family,
                                                uint32_t *delay);

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay Admin mode
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  adminMode     @b{(output)} DHCP L2 relay admin mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
* @returns  OPEN_E_INTERNAL   Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayAdminModeGet(openapiClientHandle_t *client_handle,
                                          OPEN_CONTROL_t *adminMode);

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay Admin mode
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  adminMode     @b{(input)}  DHCP L2 relay admin mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayAdminModeSet(openapiClientHandle_t *client_handle,
                                          OPEN_CONTROL_t adminMode);

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay interface mode
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  ifNum         @b{(input)}  Internal interface number
* @param  intfMode      @b{(output)} DHCP L2 Relay interface mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
* @returns  OPEN_E_INTERNAL   Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayIntfModeGet(openapiClientHandle_t *client_handle,
                                         uint32_t ifNum, 
                                         OPEN_CONTROL_t *intfMode);

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay interface mode
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  ifNum         @b{(input)}  Internal interface number
* @param  intfMode      @b{(input)}  DHCP L2 Relay interface mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayIntfModeSet (openapiClientHandle_t *client_handle,
                                          uint32_t ifNum, 
                                          OPEN_CONTROL_t intfMode);

/*********************************************************************
* @purpose  Get API for whether DHCP L2 relay considers a port trusted.
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  ifNum         @b{(input)}  Internal interface number
* @param  trustMode     @b{(output)} OPEN_TRUE if port is trusted
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
* @returns  OPEN_E_INTERNAL   Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayIntfTrustGet(openapiClientHandle_t *client_handle,
                                          uint32_t ifNum, 
                                          OPEN_BOOL_t *trustMode);

/*********************************************************************
* @purpose  Set API to configure a port in desired trust mode for DHCP l2 relay.
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  ifNum         @b{(input)}  Internal interface number
* @param  trustMode     @b{(input)}  OPEN_TRUE if port is trusted
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayIntfTrustSet (openapiClientHandle_t *client_handle,
                                           uint32_t ifNum, 
                                           OPEN_BOOL_t trustMode);

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay mode for a VLAN.
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  vlanNum       @b{(input)}  VLAN number
* @param  l2RelayMode   @b{(output)} DHCP L2 Relay  mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
* @returns  OPEN_E_INTERNAL   Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayVlanModeGet(openapiClientHandle_t *client_handle,
                                         uint32_t vlanNum, 
                                         OPEN_CONTROL_t *l2RelayMode);

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay mode for a vlan 
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  vlanNum       @b{(input)}  VLAN number
* @param  l2RelayMode   @b{(input)}  DHCP L2 Relay  mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayVlanModeSet (openapiClientHandle_t *client_handle,
                                          uint32_t vlanNum, 
                                          OPEN_CONTROL_t l2RelayMode);

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay mode for a vlan range
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  vlanStart     @b{(input)}  Starting Vlan number
* @param  vlanEnd       @b{(input)}  Ending Vlan number
* @param  l2RelayMode   @b{(input)}  DHCP L2 Relay  mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayVlanRangeModeSet (openapiClientHandle_t *client_handle,
                                               uint32_t vlanStart, uint32_t vlanEnd,
                                               OPEN_CONTROL_t l2RelayMode);

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay circuit-id mode for a vlan
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  vlanNum       @b{(input)}  VLAN number
* @param  circuitIdMode @b{(output)} DHCP L2 Relay Circuit-Id mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
* @returns  OPEN_E_INTERNAL   Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayCircuitIdGet(openapiClientHandle_t *client_handle,
                                          uint32_t vlanNum, 
                                          OPEN_CONTROL_t *circuitIdMode);

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay  circuit-Id mode for a vlan 
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  vlanNum       @b{(input)}  VLAN number
* @param  circuitIdMode @b{(input)}  DHCP L2 Relay Circuit-Id mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayCircuitIdSet (openapiClientHandle_t *client_handle,
                                           uint32_t  vlanNum, 
                                           OPEN_CONTROL_t circuitIdMode);

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay  circuit-Id mode for a vlan range
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  vlanStart     @b{(input)}  Starting Vlan number
* @param  vlanEnd       @b{(input)}  Ending Vlan number
* @param  circuitIdMode @b{(input)}  DHCP L2 Relay Circuit-Id mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayVlanRangeCircuitIdSet (openapiClientHandle_t *client_handle,
                                                    uint32_t vlanStart, 
                                                    uint32_t vlanEnd,
                                                    OPEN_CONTROL_t circuitIdMode);

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay remote-id string for a vlan
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  vlanNum       @b{(input)}  VLAN number
* @param  remoteId      @b{(output)} DHCP L2 Relay remote Id string
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayRemoteIdGet(openapiClientHandle_t *client_handle,
                                         uint32_t vlanNum, 
                                         open_buffdesc *remoteId);

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay remote-Id string for a vlan
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  vlanNum       @b{(input)}  VLAN number
* @param  remoteId      @b{(input)} DHCP L2 Relay remote Id string
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayRemoteIdSet(openapiClientHandle_t *client_handle,
                                         uint32_t vlanNum, 
                                         open_buffdesc *remoteId);

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay remote-Id string for a vlan range
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  vlanStart     @b{(input)}  Starting Vlan number
* @param  vlanEnd       @b{(input)}  Ending Vlan number
* @param  remoteId      @b{(input)}  DHCP L2 Relay remote Id string
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayVlanRangeRemoteIdSet(openapiClientHandle_t *client_handle,
                                                  uint32_t vlanStart, 
                                                  uint32_t vlanEnd, 
                                                  open_buffdesc *remoteId);

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay mode for a given subscription on an interface.
*
* @param  client_handle    @b{(input)}  client handle from registration API
* @param  ifNum            @b{(input)}  internal interface number
* @param  subscriptionName @b{(input)}  subscription name
* @param  l2RelaySubMode   @b{(output)} DHCP L2 Relay  mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
* @returns  OPEN_E_INTERNAL   Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelaySubscriptionModeGet (openapiClientHandle_t *client_handle,
                                                  uint32_t ifNum,
                                                  open_buffdesc *subscriptionName,
                                                  OPEN_CONTROL_t *l2RelaySubMode);

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay mode for a given subscription on an interface.
*
* @param  client_handle    @b{(input)}  client handle from registration API
* @param  ifNum            @b{(input)}  internal interface number
* @param  subscriptionName @b{(input)}  subscription name
* @param  l2RelaySubMode   @b{(input)}  DHCP L2 Relay  mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelaySubscriptionModeSet (openapiClientHandle_t *client_handle,
                                                  uint32_t ifNum,
                                                  open_buffdesc *subscriptionName,
                                                  OPEN_CONTROL_t l2RelaySubMode);

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay Circuit-id mode for a
*           given subscription on an interface.
*
* @param  client_handle    @b{(input)}  client handle from registration API
* @param  ifNum            @b{(input)}  internal interface number
* @param  subscriptionName @b{(input)}  subscription name
* @param  l2RelayCIdMode   @b{(output)} DHCP L2 Relay circuit-id mode 
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
* @returns  OPEN_E_INTERNAL   Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelaySubscriptionCircuitIdModeGet (openapiClientHandle_t *client_handle,
                                                           uint32_t ifNum,
                                                           open_buffdesc *subscriptionName,
                                                           OPEN_CONTROL_t *l2RelayCIdMode);

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay Circuit-id mode for a
*           given subscription on an interface.
*
* @param  client_handle    @b{(input)}  client handle from registration API
* @param  ifNum            @b{(input)}  internal interface number
* @param  subscriptionName @b{(input)}  subscription name
* @param  l2RelayCIdMode   @b{(input)}  DHCP L2 Relay circuit-id mode 
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelaySubscriptionCircuitIdModeSet (openapiClientHandle_t *client_handle,
                                                           uint32_t ifNum,
                                                           open_buffdesc *subscriptionName,
                                                           OPEN_CONTROL_t l2RelayCIdMode);

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay Remote-id string for a
*           given subscription on an interface.
*
* @param  client_handle    @b{(input)}  client handle from registration API
* @param  ifNum            @b{(input)}  internal interface number
* @param  subscriptionName @b{(input)}  subscription name
* @param  remoteId         @b{(output)} DHCP L2 Relay remote-id string 
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelaySubscriptionRemoteIdGet (openapiClientHandle_t *client_handle,
                                                      uint32_t ifNum,
                                                      open_buffdesc *subscriptionName,
                                                      open_buffdesc *remoteId);

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay Remote-id string for a
*           given subscription on an interface.
*
* @param  client_handle    @b{(input)}  client handle from registration API
* @param  ifNum            @b{(input)}  internal interface number
* @param  subscriptionName @b{(input)}  subscription name
* @param  remoteId         @b{(input)}  DHCP L2 Relay remote-id string 
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelaySubscriptionRemoteIdSet (openapiClientHandle_t *client_handle,
                                                      uint32_t ifNum,
                                                      open_buffdesc *subscriptionName,
                                                      open_buffdesc *remoteId);

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay interface specific statistics. 
*
* @param  client_handle              @b{(input)}   client handle from registration API
* @param  ifNum                      @b{(input)}   internal interface number
* @param  untrustedSrvMsgsWithOpt82  @b{(output)}  L2 Relay un trusted server statistics.
* @param  untrustedCliMsgsWithOpt82  @b{(output)}  L2 Relay un trusted client statistics.
* @param  trustedSrvMsgsWithoutOpt82 @b{(output)}  L2 Relay trusted server statistics.
* @param  trustedCliMsgsWithoutOpt82 @b{(output)}  L2 Relay trusted client statistics.
* @param  trustedSrvMsgsWithOpt82    @b{(output)}  L2 Relay trusted server opt82 statistics.
* @param  trustedCliMsgsWithOpt82    @b{(output)}  L2 Relay trusted client opt82 statistics.
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.28
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayIntfStatsGet (openapiClientHandle_t *client_handle,
                                           uint32_t ifNum,
                                           uint32_t *untrustedSrvMsgsWithOpt82,
                                           uint32_t *untrustedCliMsgsWithOpt82, 
                                           uint32_t *trustedSrvMsgsWithoutOpt82,
                                           uint32_t *trustedCliMsgsWithoutOpt82,
                                           uint32_t *trustedSrvMsgsWithOpt82,
                                           uint32_t *trustedCliMsgsWithOpt82);

/*********************************************************************
* @purpose  Clears the DHCP L2 Relay interface specific statistics.
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  ifNum         @b{(input)}  internal interface number
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayIntfStatsClear (openapiClientHandle_t *client_handle,
                                             uint32_t ifNum);

/*********************************************************************
* @purpose  Fetch configured no-option-82 action on L2 relay trusted
*           interfaces.
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  ifNum         @b{(input)}  internal interface number
* @param  action        @b{(output)} drop/keep/update
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayIntfTrustNoOption82ActionGet (openapiClientHandle_t *client_handle,
                                                           uint32_t ifNum,
                                                           OPEN_DHCP_L2RELAY_NO_OPTION_82_ACTION_t *action);

/*********************************************************************
* @purpose  Configure action for packets received without option-82
*           on L2 relay trusted interfaces.
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  ifNum         @b{(input)}  internal interface number
* @param  action        @b{(input)}  drop/keep/update
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayIntfTrustNoOption82ActionSet (openapiClientHandle_t *client_handle,
                                                           uint32_t ifNum,
                                                           OPEN_DHCP_L2RELAY_NO_OPTION_82_ACTION_t action);

/*********************************************************************
* @purpose  Gets the DHCP L2 relay circuit-id string on an interface
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  intIfNum      @b{(input)}  internal interface number
* @param  circuitId     @b{(output)} DHCP L2 relay circuit id string
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.28
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayIntfCircuitIdStringGet (openapiClientHandle_t *client_handle,
                                                     uint32_t intIfNum,
                                                     open_buffdesc *circuitId);

/*********************************************************************
* @purpose  Sets the DHCP L2 relay circuit-id string on an interface
*
* @param  client_handle @b{(input)} client handle from registration API
* @param  intIfNum      @b{(input)} internal interface number
* @param  circuitId     @b{(input)} DHCP L2 relay circuit id string
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.28
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayIntfCircuitIdStringSet (openapiClientHandle_t *client_handle,
                                                     uint32_t intIfNum,
                                                     open_buffdesc *circuitId);

/*****************************************************************************
 * @purpose   Get the rate limit value of incoming DHCP packets on untrusted 
 *            interfaces.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     family           @b{(input)}  Address family. IPV4 or IPV6
 * @param     ifNum            @b{(input)}  interface
 * @param     rate             @b{(output)} Upper limit for number of incoming 
 *                                          packets processed per second
 *
 * @returns   OPEN_E_UNAVAIL          Feature not available
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsIntfRateLimitGet(openapiClientHandle_t *client_handle, 
                                       OPEN_AF_t family, uint32_t ifNum, 
                                       int32_t *rate);

/*****************************************************************************
 * @purpose   Set the rate limit value of incoming DHCP packets on untrusted 
 *            interfaces.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     family           @b{(input)}  Address family. IPV4 or IPV6
 * @param     ifNum            @b{(input)}  interface
 * @param     rate             @b{(input)}  Upper limit for number of incoming 
 *                                          packets processed per second
 *
 * @returns   OPEN_E_UNAVAIL          Feature not available
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsIntfRateLimitSet(openapiClientHandle_t *client_handle, 
                                       OPEN_AF_t family, uint32_t ifNum, 
                                       int32_t rate);

/*****************************************************************************
 * @purpose   Get the burst interval value of incoming DHCP packets on untrusted 
 *            interfaces.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     family           @b{(input)}  Address family. IPV4 or IPV6
 * @param     ifNum            @b{(input)}  interface
 * @param     interval         @b{(output)} Consecutive interval in seconds, 
 *                                          over which the interface is monitored
 *                                          for a high rate of DHCP packets.
 *
 * @returns   OPEN_E_UNAVAIL          Feature not available
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsIntfBurstIntervalGet(openapiClientHandle_t *client_handle, 
                                           OPEN_AF_t family, uint32_t ifNum,
                                           int32_t *interval);

/*****************************************************************************
 * @purpose   Set the burst interval value of incoming DHCP packets on untrusted
 *            interfaces.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     family           @b{(input)}  Address family. IPV4 or IPV6
 * @param     ifNum            @b{(input)}  interface
 * @param     interval         @b{(input)}  Consecutive interval in seconds, 
 *                                          over which the interface is monitored
 *                                          for a high rate of DHCP packets.
 *
 * @returns   OPEN_E_UNAVAIL          Feature not available
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsIntfBurstIntervalSet(openapiClientHandle_t *client_handle, 
                                           OPEN_AF_t family, uint32_t ifNum, 
                                           int32_t interval);

/*****************************************************************************
 * @purpose   Get the number of DHCP snooping static bindings.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     family           @b{(input)}  Address family. IPV4 or IPV6
 * @param     count            @b{(output)} Number of dhcp snooping static bindings.
 *
 * @returns   OPEN_E_UNAVAIL          Feature not available
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsStaticBindingCountGet(openapiClientHandle_t *client_handle, 
                                            OPEN_AF_t family, uint32_t *count);

/*****************************************************************************
 * @purpose   Get the DHCP snooping binding entry.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     dsBinding        @b{(inout)}  DHCP snooping binding table entry.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     User of the API needs to pass the MAC address to get the details
 *            of the snooping entry corresponding to MAC address.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsBindingGet(openapiClientHandle_t *client_handle, 
                                 OPEN_DHCP_SNOOP_BINDING_t *dsBinding);

/*****************************************************************************
 * @purpose   Get the next DHCP snooping binding entry.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     dsBinding        @b{(input)}  DHCP snooping binding table entry. 
 *                                          MAC address is mandatorily needs to be
 *                                          filled.
 * @param     dsBindingNext    @b{(output)} Next DHCP snooping binding table entry.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsBindingGetNext(openapiClientHandle_t *client_handle, 
                                     OPEN_DHCP_SNOOP_BINDING_t *dsBinding, 
                                     OPEN_DHCP_SNOOP_BINDING_t *dsBindingNext);

/*****************************************************************************
 * @purpose   Get remote IP Address to store and restore the DHCP SNOOPING database.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     family           @b{(input)}  Address family. IPV4 or IPV6
 * @param     ipAddr           @b{(output)} Remote IP address
 *
 * @returns   OPEN_E_UNAVAIL          Feature not available
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsDbIpAddrGet(openapiClientHandle_t *client_handle, 
                                  OPEN_AF_t family, uint32_t *ipAddr);

/*****************************************************************************
 * @purpose   Set remote IP Address to store and restore the DHCP SNOOPING database.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     family           @b{(input)}  Address family. IPV4 or IPV6
 * @param     ipAddr           @b{(input)}  Remote IP address
 *
 * @returns   OPEN_E_UNAVAIL          Feature not available
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsDbIpAddrSet(openapiClientHandle_t *client_handle, 
                                  OPEN_AF_t family, uint32_t ipAddr);

/*****************************************************************************
 * @purpose   Get remote file name to store and restore the DHCP SNOOPING database.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     family           @b{(input)}  Address family. IPV4 or IPV6
 * @param     fileName         @b{(output)} File Name
 *
 * @returns   OPEN_E_UNAVAIL          Feature not available
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsDbFileNameGet(openapiClientHandle_t *client_handle, 
                                    OPEN_AF_t family, open_buffdesc *fileName);

/*****************************************************************************
 * @purpose   Set remote file name to store and restore the DHCP SNOOPING database.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     family           @b{(input)}  Address family. IPV4 or IPV6
 * @param     fileName         @b{(input)}  File Name
 *
 * @returns   OPEN_E_UNAVAIL          Feature not available
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsDbFileNameSet(openapiClientHandle_t *client_handle, 
                                    OPEN_AF_t family, open_buffdesc *fileName);

/*****************************************************************************
 * @purpose   Set write delay interval to store the DHCP SNOOPING database.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     family           @b{(input)}  Address family. IPV4 or IPV6
 * @param     interval         @b{(input)}  Periodic store interval
 *
 * @returns   OPEN_E_UNAVAIL          Feature not available
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsDbStoreIntervalSet(openapiClientHandle_t *client_handle, 
                                         OPEN_AF_t family, uint32_t interval);

/*****************************************************************************
 * @purpose   Get DHCP snooping statistics for passed counter Id.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     family           @b{(input)}  Address family. IPV4 or IPV6
 * @param     ifNum            @b{(input)}  interface
 * @param     counterId        @b{(input)}  Counter ID
 * @param     value            @b{(output)} Statistics value
 *
 * @returns   OPEN_E_UNAVAIL          Feature not available
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 * @notes     Refer OPEN_DS_CID_MAC_VERIFY, OPEN_DS_CID_INTF_MISMATCH, 
 *            OPEN_DS_CID_UNTRUSTED_SRVR_MSG for counter ID 
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsIntfStatsGet(openapiClientHandle_t *client_handle, 
                                   OPEN_AF_t family, uint32_t ifNum, 
                                   uint32_t counterId, uint32_t *value);

/*****************************************************************************
 * @purpose   Clear DHCP snooping bindings.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     family           @b{(input)}  Address family. IPV4 or IPV6
 * @param     ifNum            @b{(input)}  interface
 * @param     clearStaticFlag  @b{(input)}  If set, static entries also will be cleared
 *
 * @returns   OPEN_E_UNAVAIL          Feature not available
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsBindingClear(openapiClientHandle_t *client_handle, 
                                   OPEN_AF_t family, uint32_t ifNum, 
                                   OPEN_BOOL_t clearStaticFlag);

/*****************************************************************************
 * @purpose   Sets the DHCP Snooping admin mode.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     family           @b{(input)}  Address family. IPV4 or IPV6
 * @param     mode             @b{(input)}  DHCP snooping admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsAdminModeSet(openapiClientHandle_t *client_handle, 
                                   OPEN_AF_t family, uint32_t mode);

/*****************************************************************************
 * @purpose   Set API for whether DHCP snooping verifies source MAC addresses.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     family           @b{(input)}  Address family. IPV4 or IPV6
 * @param     status           @b{(input)}  State 1 or 0
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsVerifyMacSet(openapiClientHandle_t *client_handle, 
                                   OPEN_AF_t family, OPEN_BOOL_t status);

/*****************************************************************************
 * @purpose   Get API for whether DHCP snooping is enabled or disabled on a VLAN.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     family           @b{(input)}  Address family. IPV4 or IPV6
 * @param     vlanId           @b{(input)}  Vlan Id
 * @param     mode             @b{(output)} DHCP snooping state on the VLAN
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsVlanConfigGet(openapiClientHandle_t *client_handle, 
                                    OPEN_AF_t family, uint32_t vlanId, 
                                    uint32_t *mode);

/*****************************************************************************
 * @purpose   Set API to enable or disable DHCP snooping on a VLAN.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     family           @b{(input)}  Address family. IPV4 or IPV6
 * @param     vlanStart        @b{(input)}  Starting Vlan number
 * @param     vlanEnd          @b{(input)}  Ending Vlan number
 * @param     mode             @b{(input)}  DHCP snooping state on the VLAN
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsVlanConfigSet(openapiClientHandle_t *client_handle, 
                                    OPEN_AF_t family, uint32_t vlanStart,
                                    uint32_t vlanEnd, uint32_t mode);

/*****************************************************************************
 * @purpose   Set API for whether DHCP snooping considers a port trusted.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     family           @b{(input)}  Address family. IPV4 or IPV6
 * @param     ifNum            @b{(input)}  interface
 * @param     trustMode        @b{(input)}  Trust mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsIntfTrustSet(openapiClientHandle_t *client_handle, 
                                   OPEN_AF_t family, uint32_t ifNum, 
                                   OPEN_BOOL_t trustMode);

/*****************************************************************************
 * @purpose   Set API for whether DHCP snooping logs invalid packets.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     family           @b{(input)}  Address family. IPV4 or IPV6
 * @param     ifNum            @b{(input)}  interface
 * @param     logMode          @b{(input)}  Log mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsIntfLogInvalidSet(openapiClientHandle_t *client_handle, 
                                        OPEN_AF_t family, uint32_t ifNum, 
                                        OPEN_BOOL_t logMode);

/*****************************************************************************
 * @purpose   Get the DHCP IPv6 snooping binding entry.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     dsBinding        @b{(output)} DHCP IPv6 snooping binding table entry.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsIpv6BindingGet(openapiClientHandle_t *client_handle, 
                                     OPEN_DHCP_V6_SNOOP_BINDING_t *dsBinding);

/*****************************************************************************
 * @purpose   Get the next DHCP IPv6 snooping binding entry.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     dsBinding        @b{(input)}  DHCP IPV6 snooping binding table entry. MAC address is mandatoryneeds to be filled.
 * @param     dsBindingNext    @b{(output)} Next DHCP IPV6 snooping binding table entry.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsIpv6BindingGetNext(openapiClientHandle_t *client_handle, 
                                         OPEN_DHCP_V6_SNOOP_BINDING_t *dsBinding, 
                                         OPEN_DHCP_V6_SNOOP_BINDING_t *dsBindingNext);

/*****************************************************************************
 * @purpose   Add a static entry to the DHCP snooping bindings database.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     family           @b{(input)}  Address family. IPV4 or IPV6
 * @param     macAddr          @b{(input)}  MAC address
 * @param     ipAddr           @b{(input)}  IP address
 * @param     vlanId           @b{(input)}  Vlan Id
 * @param     ifNum            @b{(input)}  interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsIpStaticBindingAdd(openapiClientHandle_t *client_handle, 
                                         OPEN_AF_t family, open_buffdesc *macAddr,
                                         open_buffdesc *ipAddr, uint32_t vlanId, 
                                         uint32_t ifNum);

/*****************************************************************************
 * @purpose   Remove a static entry from the DHCP snooping bindings database.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     family           @b{(input)}  Address family. IPV4 or IPV6
 * @param     macAddr          @b{(input)}  MAC address
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsIpStaticBindingRemove(openapiClientHandle_t *client_handle, 
                                            OPEN_AF_t family, 
                                            open_buffdesc *macAddr);

/*****************************************************************************
 * @purpose   Add a volatile entry to the DHCP snooping bindings database.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     family           @b{(input)}  Address family. IPV4 or IPV6
 * @param     macAddr          @b{(input)}  MAC address
 * @param     ipAddr           @b{(input)}  IP address
 * @param     vlanId           @b{(input)}  Vlan Id
 * @param     ifNum            @b{(input)}  interface
 * @param     leaseTime        @b{(input)}  lease time for the volatile entry
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will not change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.28
 *
 * @end
 *****************************************************************************/
open_error_t openapiDsIpVolatileBindingAdd(openapiClientHandle_t *client_handle, 
                                         OPEN_AF_t family, open_buffdesc *macAddr,
                                         open_buffdesc *ipAddr, uint32_t vlanId, 
                                         uint32_t ifNum, uint32_t leaseTime);

#endif /* OPENAPI_DHCP_SNOOPING_H_INCLUDED */
