/*********************************************************************
*
*  Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_diffserv_policy.h
*
* @purpose   Diff serv Policy Configuration and Status
*
* @component OpEN
*
* @create    13/05/2023
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_DIFFSERV_POLICY_H_INCLUDED
#define OPENAPI_DIFFSERV_POLICY_H_INCLUDED

#include "openapi_common.h"

#define OPEN_DIFFSERV_POLICY_NAME_MAX   31
#define OPEN_DIFFSERV_POLICY_NAME_MIN   1


/* Enumerated values for RowStatus object */
typedef enum
{
  OPEN_DIFFSERV_ROW_STATUS_NONE = 0,    /**< not a valid value */

  OPEN_DIFFSERV_ROW_STATUS_ACTIVE,      /**< Active */
  OPEN_DIFFSERV_ROW_STATUS_NOT_IN_SERVICE, /**< Not in Service */
  OPEN_DIFFSERV_ROW_STATUS_NOT_READY,      /**< Not Ready */
  OPEN_DIFFSERV_ROW_STATUS_CREATE_AND_GO,  /**< Create and Go */
  OPEN_DIFFSERV_ROW_STATUS_CREATE_AND_WAIT, /**< Create and Wait */
  OPEN_DIFFSERV_ROW_STATUS_DESTROY        /**< Destroy */
} OPEN_DIFFSERV_ROW_STATUS_t;

typedef enum
{
  OPEN_DIFFSERV_POLICY_TYPE_NONE = 0,   /**< None */
  OPEN_DIFFSERV_POLICY_TYPE_IN,         /**< In */
  OPEN_DIFFSERV_POLICY_TYPE_OUT         /**< Out */
} OPEN_DIFFSERV_POLICY_TYPE_t;

typedef enum
{
  OPEN_DIFFSERV_STORAGE_TYPE_NONE = 0,  /**< not a valid value */

  OPEN_DIFFSERV_STORAGE_TYPE_OTHER,    /**< Other */
  OPEN_DIFFSERV_STORAGE_TYPE_VOLATILE, /**< Volatile */
  OPEN_DIFFSERV_STORAGE_TYPE_NONVOLATILE, /**< Non-Volatile */
  OPEN_DIFFSERV_STORAGE_TYPE_READONLY,  /**< Readonly */
  OPEN_DIFFSERV_STORAGE_TYPE_PERMANENT /**< Permanent */
} OPEN_DIFFSERV_STORAGE_TYPE_t;

/* Values for DiffServ Policy Attribute statement type */
typedef enum
{
  OPEN_DIFFSERV_POLICY_ATTR_TYPE_NONE = 0, /**< None */

  OPEN_DIFFSERV_POLICY_ATTR_TYPE_ASSIGN_QUEUE, /**< Assign-Queue [0-6]*/
  OPEN_DIFFSERV_POLICY_ATTR_TYPE_DROP,  /**< Drop */
  OPEN_DIFFSERV_POLICY_ATTR_TYPE_MARK_COSVAL, /**< CosVal [1-7]*/
  OPEN_DIFFSERV_POLICY_ATTR_TYPE_MARK_COS2VAL, /**< Cos2Val [1-7]*/
  OPEN_DIFFSERV_POLICY_ATTR_TYPE_MARK_COS_AS_COS2, /**< CosAsCos2 [1-7]*/
  OPEN_DIFFSERV_POLICY_ATTR_TYPE_MARK_IPDSCPVAL,  /**< Valid IPDSCP [0-63]*/
  OPEN_DIFFSERV_POLICY_ATTR_TYPE_MARK_IPPRECEDENCEVAL, /**< IP-Precedence [0-7]*/
  OPEN_DIFFSERV_POLICY_ATTR_TYPE_MIRROR,  /**< Mirror */
  OPEN_DIFFSERV_POLICY_ATTR_TYPE_POLICE_SIMPLE, /**< Police-Simple */
  OPEN_DIFFSERV_POLICY_ATTR_TYPE_POLICE_SINGLERATE, /**< Police-Singlerate */
  OPEN_DIFFSERV_POLICY_ATTR_TYPE_POLICE_TWORATE,  /**< Police-Tworate */
  OPEN_DIFFSERV_POLICY_ATTR_TYPE_REDIRECT,   /**< Redirect */
} OPEN_DIFFSERV_POLICY_ATTR_TYPE_t;


typedef enum 
{
  OPEN_BURST_POLICE_TWORATE_NONE = 0, /**< Tworate- none */
  OPEN_BURST_POLICE_TWORATE_CBURST,  /**< Tworate-Commited burst [1-128]*/
  OPEN_BURST_POLICE_TWORATE_PBURST, /**< Tworate-Peak burst [1-128]*/
  OPEN_BURST_POLICE_SINGLERATE_CBURST, /**< Singlerate-Commited burst [1-128]*/
  OPEN_BURST_POLICE_SINGLERATE_EBURST, /**< Singlerate-Egress burst [1-128] */
  OPEN_BURST_POLICE_SIMPLE_CBURST /**< Simple-Commited burst [1-128] */
}OPEN_BURST_POLICE_TYPE_t;

typedef enum
{
  OPEN_DIFFSERV_POLICE_ACT_NONE = 0, /**< None */

  OPEN_DIFFSERV_POLICE_ACT_DROP, /**< Drop */
  OPEN_DIFFSERV_POLICE_ACT_MARKCOS, /**< Mark-Cos */
  OPEN_DIFFSERV_POLICE_ACT_MARKCOS2, /**< Mark-Cos2 */
  OPEN_DIFFSERV_POLICE_ACT_MARKCOS_AS_COS2, /**< Mark-CosAsCos2 */
  OPEN_DIFFSERV_POLICE_ACT_MARKDSCP, /**< Mark-DSCP */
  OPEN_DIFFSERV_POLICE_ACT_MARKPREC, /**< Mark-Precedence */
  OPEN_DIFFSERV_POLICE_ACT_SEND /**< Send */
} OPEN_DIFFSERV_POLICE_ACT_t;



/*********************************************************************
* @purpose  Determine next sequential row in the Policy Table
*
* @param    client_handle    @b{(input)} client handle 
* @param    prevPolicyIndex  @b{(input)} Policy index to begin search
* @param    pPolicyIndex     @b{(output)} Pointer to next sequential policy index
*                                      value
*
* @returns  OPEN_E_NONE     next entry successfully retrieved
* @returns  OPEN_E_FAIL     failed to get next entry
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_ERROR    internal error has occurred
*
* @notes    A prevPolicyIndex value of 0 is used to find the first entry in
*           the table.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyGetNext(openapiClientHandle_t *client_handle,
                                          uint32_t prevPolicyIndex, 
                                          uint32_t *pPolicyIndex);

/*********************************************************************
* @purpose  Get the policy name 
*
* @param    client_handle     @b{(input)} client handle 
* @param    policyIndex i     @b{(input)} Policy index
* @param    bufd              @b{(output)} buffer descriptor for policy name
*
* @returns  OPEN_E_NONE     name successfully retrieved
* @returns  OPEN_E_FAIL     failed to get name
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL Object value never been established or buffer length is 
*                       too short
*
* @notes    `bufd` points to a char buffer of minimum length equals to
*            OPEN_DIFFSERV_POLICY_NAME_MAX
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyNameGet(openapiClientHandle_t *client_handle,
                                    uint32_t policyIndex,
                                    open_buffdesc *bufd);

/*********************************************************************
* @purpose  Set the policy name 
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    *bufd           @b{(input)} buffer descriptor for policy name
*
* @returns  OPEN_E_NONE     name is set successfully
* @returns  OPEN_E_FAIL     failed to set name
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error occured
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyNameSet(openapiClientHandle_t *client_handle,
                                    uint32_t policyIndex,
                                    open_buffdesc *bufd);

/*********************************************************************
* @purpose  Translate a policy name into its associated Policy Table index
*
* @param    client_handle   @b{(input)} client handle 
* @param    stringPtr       @b{(input)} Policy name string pointer
* @param    pPolicyIndex    @b{(output)} Pointer to policy index value
*
* @returns  OPEN_E_NONE     index successfully retrieved
* @returns  OPEN_E_FAIL     failed to get policy index
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error occured 
*
* @notes    This is a supplemental function provided to assist the user
*           interface code with name-to-index translation.  Use the
*           name 'get' function to translate in the reverse direction.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyNameToIndex(openapiClientHandle_t *client_handle, 
                                       open_buffdesc *stringPtr,
                                       uint32_t *pPolicyIndex);

/*********************************************************************
* @purpose  Check if class definition is suitable for inclusion by specified
*           policy
*
* @param    client_handle   @b{(input)} client handle 
* @param    classIndex      @b{(input)}  Class index
* @param    policyIndex     @b{(input)}  Policy index
*
* @returns  OPEN_E_NONE     inclusion is valid
* @returns  OPEN_E_FAIL     inclusion is invalid
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error occured 
*
* @notes    This is a supplemental function provided to assist the user
*           interface code with determining if a class can be assigned
*           to a given policy.  Some platforms impose restrictions on
*           the classifiers that can be used in a certain policy direction
*           (in/out).
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServClassValidForPolicyCheck(openapiClientHandle_t *client_handle,
                                              uint32_t classIndex,
                                              uint32_t policyIndex);

/*********************************************************************
* @purpose  Get the current status of the specified Policy Table row
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    pValue          @b{(output)} Pointer to row status value
*
* @returns  OPEN_E_NONE     row status successfully retrieved
* @returns  OPEN_E_FAIL     failed to get policy row status
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error occured 
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyRowStatusGet(openapiClientHandle_t *client_handle,
                                        uint32_t policyIndex,
                              OPEN_DIFFSERV_ROW_STATUS_t *pValue);

/*********************************************************************
* @purpose  Create a new row in the Policy Table for the specified index
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    activateRow     @b{(input)} Activate row when ready flag
*
* @returns  OPEN_E_NONE     row status successfully retrieved
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error occured 
* @returns  OPEN_E_ERROR    Index value in use, row not complete
* @returns  OPEN_E_EXISTS   Index value in use, row complete
* @returns  OPEN_E_FAIL     Failed to set the row.
*
* @notes    An OPEN_E_ERROR response means the policy index value is currently 
*           in use, but the row is not complete.  This is typically
*           caused when multiple requestors obtain the same "index next"
*           value and race to create the new row.  The caller should
*           get a new index value and re-attempt to create the row.
*
* @notes    An OPEN_E_EXISTS response means the policy index value
*           is currently in use for a completed row.  The caller should
*           get a new index value and re-attempt to create the row.
*
* @notes    An OPEN_E_INTERNAL response means another application   
*           has denied approval of this configuration change.
*
* @notes    The row status is initially set to 'not ready' and remains that 
*           way until every object in the row contains a valid value.  If the
*           activateRow parameter is set to OPEN_TRUE, the row status will then
*           be automatically updated to 'active'.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyCreate(openapiClientHandle_t *client_handle,
                                         uint32_t policyIndex,
                                         OPEN_BOOL_t activateRow);
/*************************************************************************
* @purpose  Delete a row from the Policy Table for the specified index
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
*
* @returns  OPEN_E_NONE     deletion is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_ERROR    Row does not exist for the specified index, or is
*                           referenced by one or more services (interfaces)
* @returns  OPEN_E_INTERNAL Config change not approved
* @returns  OPEN_E_FAIL     Deletion failed
*
* @notes    A policy cannot be deleted if it is currently assigned to an    
*           interface.
*
* @notes    An OPEN_E_INTERNAL response means another application   
*           has denied approval of this configuration change.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyDelete(openapiClientHandle_t *client_handle, uint32_t policyIndex);

/*********************************************************************
* @purpose  Get the type of policy definition
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    pPolicyType     @b{(output)} Pointer to policy type value
*
* @returns  OPEN_E_NONE   Get is successful 
* @returns  OPEN_E_FAIL   Get operation failed 
* @returns  OPEN_E_PARAM  Invalida parameter passed
*
* @notes    The policy type dictates whether inbound or outbound policy
*           attribute statements are used in the policy definition.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyTypeGet(openapiClientHandle_t *client_handle, 
                              uint32_t policyIndex,
                              OPEN_DIFFSERV_POLICY_TYPE_t *pPolicyType);

/*********************************************************************
* @purpose  Set the type of policy definition
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyType      @b{(input)} Pointer to policy type value
*
* @returns  OPEN_E_NONE   Set is successful 
* @returns  OPEN_E_FAIL   Set operation failed 
* @returns  OPEN_E_PARAM  Invalida parameter passed
*
* @notes    The policy type dictates whether inbound or outbound policy
*           attribute statements are used in the policy definition.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyTypeSet(openapiClientHandle_t *client_handle, 
                              uint32_t policyIndex,
                              OPEN_DIFFSERV_POLICY_TYPE_t policyType);

/*************************************************************************
* @purpose  Obtain the next available index value to be used when creating  
*           a new row in the Policy-Class Instance Table
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex @b{(input)} Policy index
* @param    pPolicyInstIndex @b{(output)} Pointer to instance index value
*
* @returns  OPEN_E_NONE     Operation is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     Failed to get next index
*
* @notes    An OPEN_E_NONE return with an output *pPolicyInstIndex value
*           of 0 means the table is full.
*
* @notes    The index value remains free until used in a subsequent create
*           request.  When two or more requestors get the same index value,
*           only the first one to successfully create a new row in this table
*           gets to keep it.  All other requestors' create requests will
*           fail due to the index being in use, and they must call this
*           function again to get a new value.
*
* @notes    Although an index value is considered in use once a
*           successful row create occurs, the next index value is not
*           revealed until the row is either completed or deleted.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyInstIndexNext(openapiClientHandle_t *client_handle, 
                                         uint32_t policyIndex,
                                         uint32_t *pPolicyInstIndex);
/*********************************************************************
* @purpose  Determine next sequential row in the Policy-Class Instance Table
*
* @param    client_handle        @b{(input)} client handle 
* @param    prevPolicyIndex      @b{(input)} Policy index to begin search
* @param    prevPolicyInstIndex  @b{(input)} Policy instance index to begin 
*                                              search
* @param    pPolicyIndex         @b{(output)} Pointer to next sequential policy index
*                                      value
* @param    pPolicyInstIndex     @b{(output)} Pointer to next sequential policy
*                                            instance index value
*
* @returns  OPEN_E_NONE     Operation is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     Failed to get next sequential index 
*
* @notes    A prevPolicyIndex value of 0 is used to find the first policy
*           entry in the table.
*
* @notes    A prevPolicyInstIndex value of 0 is used to find the first class
*           instance relative to the specified prevPolicyIndex.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyInstGetNext(openapiClientHandle_t *client_handle, 
                                       uint32_t prevPolicyIndex, 
                                       uint32_t prevPolicyInstIndex, 
                                       uint32_t *pPolicyIndex,
                                       uint32_t *pPolicyInstIndex);

/*********************************************************************
* @purpose  Get the class index referenced as the policy class instance
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    pClassIndex     @b{(output)} Pointer to class index value
*
* @returns  OPEN_E_NONE     Class index get is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     Class index get failed
*
* @notes    The class index identifies the classifier definition used by 
*           the policy to establish an instance of that class (to which
*           policy attributes can be attached).  This is what connects a 
*           class (instance) to a policy.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyInstClassIndexGet(openapiClientHandle_t *client_handle, 
                                             uint32_t policyIndex,
                                             uint32_t policyInstIndex,
                                             uint32_t *pClassIndex);

/*********************************************************************
* @purpose  Set the class index referenced as the policy class instance
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    classIndex      @b{(input)} class index value
*
* @returns  OPEN_E_NONE     Class index is successfuly set 
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     Class index set failed 
*
* @notes    The class index identifies the classifier definition used by 
*           the policy to establish an instance of that class (to which
*           policy attributes can be attached).  This is what connects a 
*           class (instance) to a policy.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyInstClassIndexSet(openapiClientHandle_t *client_handle, 
                                             uint32_t policyIndex,
                                             uint32_t policyInstIndex,
                                             uint32_t classIndex);

/*********************************************************************
* @purpose  Find the policy instance index from the policy and class indexes
*
* @param    client_handle    @b{(input)} client handle 
* @param    policyIndex      @b{(input)} Policy index
* @param    classIndex       @b{(input)} Class index
* @param    pPolicyInstIndex @b{(output)} Pointer to policy instance index value
*
* @returns  OPEN_E_NONE     Policy instance index find is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     Failed to find the Policy instance index
*
* @notes    This is a supplemental function provided to assist the user
*           interface code by searching through the Policy-Class Instance
*           Table for the policy instance index whose row matches the 
*           specified class index for the policy.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyInstIndexFromClass(openapiClientHandle_t *client_handle, 
                                              uint32_t policyIndex, 
                                              uint32_t classIndex,
                                              uint32_t *pPolicyInstIndex);

/*************************************************************************
* @purpose  Create a new row in the Policy-Class Instance Table for the 
*           specified index
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    activateRow     @b{(input)} Activate row when ready flag
*
* @returns  OPEN_E_NONE     New row creation is successful
* @returns  OPEN_E_PARAM    Invalid parameter  passed
* @returns  OPEN_E_ERROR    Index value in use, row not complete
* @returns  OPEN_E_EXISTS   Index value in use, row complete
* @returns  OPEN_E_INTERNAL Internal error occured
* @returns  OPEN_E_FAIL     New row creation failed
*
* @notes    An OPEN_E_ERROR response means the policy instance index value is
*           currently in use, but the row is not complete.  This is typically
*           caused when multiple requestors obtain the same "index next"
*           value and race to create the new row.  The caller should
*           get a new index value and re-attempt to create the row.
*
* @notes    An OPEN_E_EXISTS response means the policy instance index
*           value is currently in use for a completed row.  The caller should
*           get a new index value and re-attempt to create the row.
*
* @notes    An OPEN_E_INTERNAL response means another application   
*           has denied approval of this configuration change.
*
* @notes    The row status is initially set to 'not ready' and remains that 
*           way until every object in the row contains a valid value.  If the
*           activateRow parameter is set to L7_TRUE, the row status will then
*           be automatically updated to 'active'.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyInstCreate(openapiClientHandle_t *client_handle, 
                                      uint32_t policyIndex,
                                      uint32_t policyInstIndex,
                                      OPEN_BOOL_t activateRow);

/*************************************************************************
* @purpose  Delete a row from the Policy-Class Instance Table for the 
*           specified index
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
*
* @returns  OPEN_E_NONE     deletion is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_ERROR    row does not exist for the specified index
* @returns  OPEN_E_INTERNAL config change not approved
* @returns  OPEN_E_FAIL     deletion is not successful
*
* @notes    An OPEN_E_INTERNAL response means another application   
*           has denied approval of this configuration change.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyInstDelete(openapiClientHandle_t *client_handle, 
                                      uint32_t policyIndex,
                                      uint32_t policyInstIndex);

/*********************************************************************
* @purpose  Get the current status of the specified Policy-Class
*           Instance Table row
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    pValue          @b{(output)} Pointer to row status value
*
* @returns  OPEN_E_NONE     row status successfully retrieved
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to get policy instance row status
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error occured 
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyInstRowStatusGet(openapiClientHandle_t *client_handle,
                                        uint32_t policyIndex,
                                        uint32_t policyInstIndex,
                              OPEN_DIFFSERV_ROW_STATUS_t *pValue);

/*************************************************************************
* @purpose  Obtain the next available index value to be used when creating  
*           a new row in the Policy Table
*
* @param    client_handle   @b{(input)} client handle 
* @param    pPolicyIndex    @b{(inout)} Pointer to policy index value
*
* @returns  OPEN_E_NONE     next index is received successfully
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to get next available index
*
* @notes    An OPEN_E_NONE return with an output *pPolicyIndex value of 0
*           means the table is full.
*
* @notes    The index value remains free until used in a subsequent create
*           request.  When two or more requestors get the same index value,
*           only the first one to successfully create a new row in this table
*           gets to keep it.  All other requestors' create requests will
*           fail due to the index being in use, and they must call this
*           function again to get a new value.
*
* @notes    Although an index value is considered in use once a
*           successful row create occurs, the next index value is not
*           revealed until the row is either completed or deleted.
*
* @supported-in-version OpEN API Version: 1.26
*
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyIndexNext(openapiClientHandle_t *client_handle, 
                                     uint32_t *pPolicyIndex);

/*********************************************************************
* @purpose  Get the storage type for the specified Policy Attribute 
*           Table row
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    pValue          @b{(output)} Pointer to storage type value
*
* @returns  OPEN_E_NONE     storage  type get is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to get the storage type
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStorageTypeGet(openapiClientHandle_t *client_handle,
                                              uint32_t policyIndex,
                                              uint32_t policyInstIndex,
                                              uint32_t policyAttrIndex,
                              OPEN_DIFFSERV_STORAGE_TYPE_t *pValue);

/*********************************************************************
* @purpose  Get the current status of the specified Policy Attribute
*           Table row
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    pValue          @b{(output)} Pointer to row status value
*
* @returns  OPEN_E_NONE     current status is successfully retrieved
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to get the current status
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrRowStatusGet(openapiClientHandle_t *client_handle,
                                            uint32_t policyIndex,
                                            uint32_t policyInstIndex,
                                            uint32_t policyAttrIndex,
                              OPEN_DIFFSERV_ROW_STATUS_t *pValue);
/*************************************************************************
* @purpose  Obtain the next available index value to be used when creating  
*           a new row in the Policy Attribute Table
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    pPolicyAttrIndex @b{(output)} Pointer to policy attribute index 
*
* @returns  OPEN_E_NONE     next index is successfully retrieved
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to get the next index
*
* @notes    An OPEN_E_NONE return with an output *pPolicyAttrIndex value
*           of 0 means the table is full.
*
* @notes    The index value remains free until used in a subsequent create
*           request.  When two or more requestors get the same index value,
*           only the first one to successfully create a new row in this table
*           gets to keep it.  All other requestors' create requests will
*           fail due to the index being in use, and they must call this
*           function again to get a new value.
*
* @notes    Although an index value is considered in use once a
*           successful row create occurs, the next index value is not
*           revealed until the row is either completed or deleted.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrIndexNext(openapiClientHandle_t *client_handle, 
                                         uint32_t policyIndex,
                                         uint32_t policyInstIndex,
                                         uint32_t *pPolicyAttrIndex);

/*************************************************************************
* @purpose  Retrieve the maximum policy attribute index value allowed for the 
*           Policy Attribute Table
*
* @param    client_handle   @b{(input)} client handle 
* @param    pIndexMax       @b{(output)} Pointer to index max output location
*
* @returns  OPEN_E_NONE    maximum policy attribute index is successfully retrieved
* @returns  OPEN_E_PARAM   invalid parameter  passed
* @returns  OPEN_E_FAIL    failed to get maximum policy attribute index
*
* @notes    This is purely informational.  The corresponding table 'IndexNext'
*           function must be used to obtain the current index value.
*
* @notes    There is a difference between this maximum index and the 
*           maximum table size.  The latter encompasses independent
*           sets of index maximums.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrIndexMaxGet(openapiClientHandle_t *client_handle, 
                                           uint32_t *pIndexMax);

/*********************************************************************
* @purpose  Get the Policy Attribute statement entry type
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    pEntryType      @b{(output)} Pointer to entry type value  
*
* @returns  OPEN_E_NONE     policy attribute statement is successfully retrieved
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to get the policy attribute statement
*
* @notes    The entry type is essential to determine which one of the
*           individual statements is defined for this policy attribute.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtEntryTypeGet(openapiClientHandle_t *client_handle, 
                                                uint32_t policyIndex, 
                                                uint32_t policyInstIndex, 
                                                uint32_t policyAttrIndex, 
                        OPEN_DIFFSERV_POLICY_ATTR_TYPE_t *pEntryType);

/*********************************************************************
* @purpose  Get the Policy Attribute statement entry associated data
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex @b{(input)} Policy index
* @param    policyInstIndex  @b{(input)} Policy instance index
* @param    policyAttrIndex  @b{(input)} Policy attribute index
* @param    entryType   @b{(input)} Entry type value  
* @param    data   @b{(output)} associated data  
*
* @returns  OPEN_E_NONE     associated data get is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to get the associated data
*
* @notes    The entry type MUST be set before any of the individual        
*           statements can be set for this policy attribute.  The entry type 
*           value determines which statement get/set commands are valid for 
*           this attribute.
*
* @notes    For a given policy-class instance, one or more attribute
*           statements may be defined.  However, not all statement entry
*           types are allowed to co-exist within the same policy-class
*           instance.  The following lists for each statement entry types 
*           any other types that conflict:
*              _ASSIGN_QUEUE      :  _DROP
*              _DROP              :  _ASSIGN_QUEUE, _MARK_(any), _POLICE_(any), 
*                                        _REDIRECT
*              _MARK_COSVAL       :  _DROP, _MARK_COS2VAL, _MARK_IPDSCPVAL,
*                                        _MARK_IPPPRECEDENCEVAL, _POLICE_(any)
*              _MARK_COS2VAL      :  _DROP, _MARK_COSVAL, _MARK_IPDSCPVAL,
*                                        _MARK_IPPPRECEDENCEVAL, _POLICE_(any)
*              _MARK_IPDSCPVAL    :  _DROP, _MARK_COSVAL, _MARK_COS2VAL,
*                                        _MARK_IPPPRECEDENCEVAL, _POLICE_(any)
*           _MARK_IPPRECEDENCEVAL :  _DROP, _MARK_COSVAL, _MARK_COS2VAL,
*                                        _MARK_IPPDSCPVAL, _POLICE_(any)
*              _MIRROR            :  _DROP, _REDIRECT
*              _REDIRECT          :  _DROP, _MIRROR
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/

open_error_t openapiDiffServPolicyAttrStmtEntryDataGet(openapiClientHandle_t *client_handle, 
                                                uint32_t policyIndex, 
                                                uint32_t policyInstIndex, 
                                                uint32_t policyAttrIndex, 
                              OPEN_DIFFSERV_POLICY_ATTR_TYPE_t entryType,
                                                uint32_t *data);

/*************************************************************************
* @purpose  Create a new row in the Policy Attribute Table for the 
*           specified index
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    activateRow     @b{(input)} Activate row when ready flag
*
* @returns  OPEN_E_NONE     row creation is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     row creation failed
* @returns  OPEN_E_ERROR    index value in use, row not complete
* @returns  OPEN_E_EXISTS   index value in use, row complete
*
* @notes    An OPEN_E_ERROR response means the policy attribute index value is
*           currently in use, but the row is not complete.  This is typically
*           caused when multiple requestors obtain the same "index next"
*           value and race to create the new row.  The caller should
*           get a new index value and re-attempt to create the row.
*
* @notes    An OPEN_E_EXISTS response means the policy attribute index
*           value is currently in use for a completed row.  The caller should
*           get a new index value and re-attempt to create the row.
*
* @notes    The row status is initially set to 'not ready' and remains that 
*           way until every object in the row contains a valid value.  If the
*           activateRow parameter is set to OPEN_TRUE, the row status will then
*           be automatically updated to 'active'.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrCreate(openapiClientHandle_t *client_handle, 
                                      uint32_t policyIndex,
                                      uint32_t policyInstIndex,
                                      uint32_t policyAttrIndex,
                                      OPEN_BOOL_t activateRow);

/*************************************************************************
* @purpose  Delete a row from the Policy Attribute Table for the 
*           specified index
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
*
* @returns  OPEN_E_NONE     row deletion is successful
* @returns  OPEN_E_PARAM    invalid parameter passed
* @returns  OPEN_E_FAIL     row deletion failed
* @returns  OPEN_E_ERROR    row does not exist for the specified index 
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrDelete(openapiClientHandle_t *client_handle, 
                                      uint32_t policyIndex,
                                      uint32_t policyInstIndex, 
                                      uint32_t policyAttrIndex);

/*********************************************************************
* @purpose  Set the Policy Attribute statement entry type and associated data
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    entryType       @b{(input)} Entry type value  
* @param    data            @b{(input)} Entry value  
*
* @returns  OPEN_E_NONE    succesfully set the policy attribute statement and associated data
* @returns  OPEN_E_PARAM   invalid parameter  passed
* @returns  OPEN_E_FAIL    faled to set the policy attribute statement and associated data
*
* @notes    The entry type MUST be set before any of the individual        
*           statements can be set for this policy attribute.  The entry type 
*           value determines which statement get/set commands are valid for 
*           this attribute.
*
* @notes    For a given policy-class instance, one or more attribute
*           statements may be defined.  However, not all statement entry
*           types are allowed to co-exist within the same policy-class
*           instance.  The following lists for each statement entry types 
*           any other types that conflict:
*           _ASSIGN_QUEUE         :   _DROP
*           _DROP                 :   _ASSIGN_QUEUE, _MARK_(any), _POLICE_(any), 
*                                     _REDIRECT
*           _MARK_COSVAL          :   _DROP, _MARK_COS2VAL, _MARK_IPDSCPVAL,
*                                     _MARK_IPPPRECEDENCEVAL, _POLICE_(any)
*           _MARK_COS2VAL         :   _DROP, _MARK_COSVAL, _MARK_IPDSCPVAL,
*                                     _MARK_IPPPRECEDENCEVAL, _POLICE_(any)
*           _MARK_IPDSCPVAL       :   _DROP, _MARK_COSVAL, _MARK_COS2VAL,
*                                     _MARK_IPPPRECEDENCEVAL, _POLICE_(any)
*           _MARK_IPPRECEDENCEVAL :   _DROP, _MARK_COSVAL, _MARK_COS2VAL,
*                                     _MARK_IPPDSCPVAL, _POLICE_(any)
*           _MIRROR               :   _DROP, _REDIRECT
*           _REDIRECT             :   _DROP, _MIRROR
*           _POLICE_SIMPLE        :   _DROP, _MARK_(any), _POLICE_SINGLERATE,
*                                     _POLICE_TWORATE
*           _POLICE_SINGLERATE    :   _DROP, _MARK_(any), _POLICE_SIMPLE,
*                                     _POLICE_TWORATE
*           _POLICE_TWORATE       :  _DROP, _MARK_(any), _POLICE_SIMPLE,
*                                     _POLICE_SINGLERATE
*
* @notes When entryType's are passed with _POLICE_SIMPLE or _POLICE_SINGLERATE,
*        or _POLICE_TWORATE types default actions
*        [conform Action:send , Exceed and nonConform Action:drop] are set
* 
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtEntryTypeDataSet(openapiClientHandle_t *client_handle, 
                                                uint32_t policyIndex, 
                                                uint32_t policyInstIndex, 
                                                uint32_t policyAttrIndex, 
                              OPEN_DIFFSERV_POLICY_ATTR_TYPE_t entryType,
                                               uint32_t data);


/*********************************************************************
* @purpose  Find the Policy Attribute index for the specified attribute 
*
* @param    client_handle    @b{(input)} client handle 
* @param    policyIndex      @b{(input)} Policy index
* @param    policyInstIndex  @b{(input)} Policy instance index
* @param    inEntry          @b{(input)} Policy attribute
* @param    pPolicyAttrIndex @b{(output)} Pointer to attribute index  
*
* @returns  OPEN_E_NONE     policy attribute index find is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to find policy attribute index
*
* @notes   
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/

open_error_t openapiDiffServPolicyAttrIndexFind(openapiClientHandle_t *client_handle, 
                                                uint32_t policyIndex, 
                                                uint32_t policyInstIndex, 
                                                OPEN_DIFFSERV_POLICY_ATTR_TYPE_t inEntry,
                                                uint32_t *pPolicyAttrIndex) ;

/*********************************************************************
* @purpose  Set the simple policing committed rate attribute statement
*           value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    entryType       @b{(input)} Entry type value
* @param    rate_unit       @b{(input)} Rate unit (in kbps
*                                        or percentage)
* @param    rate            @b{(input)} Rate value
*
* @returns  OPEN_E_NONE     simple policing committed rate attribute statement value set is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to set simple policing committed rate attribute statement value
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
*
* @notes    Usage of this function is only allowed for an attribute
*           statement entry types of:  _POLICE_SIMPLE,
*                                      _POLICE_SINGLERATE
*                                      _POLICE_TWORATE.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceDataRateSet(openapiClientHandle_t *client_handle,
                                                     uint32_t policyIndex,
                                                     uint32_t policyInstIndex,
                                                     uint32_t policyAttrIndex,
                                                     OPEN_DIFFSERV_POLICY_ATTR_TYPE_t entryType,
                                                     OPEN_RATE_UNIT_t rate_unit,
                                                     uint32_t rate);

/*********************************************************************
* @purpose  Get the policing committed rate unit attribute statement
*           value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    entryType       @b{(input)} Entry type value
* @param    pRate_unit      @b{(output)} Value unit
*
* @returns  OPEN_E_NONE     policing committed rate unit attribute statement value get is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to get policing committed rate unit attribute statement value
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
*
* @notes    Usage of this function is only allowed for an attribute
*           statement entry type of:  _POLICE_SIMPLE.
*                                     _POLICE_SINGLERATE.
*                                     _POLICE_TWORATE.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceDataRateUnitGet(openapiClientHandle_t *client_handle,
                                                         uint32_t policyIndex,
                                                         uint32_t policyInstIndex,
                                                         uint32_t policyAttrIndex,
                                                         OPEN_DIFFSERV_POLICY_ATTR_TYPE_t entryType,
                                                         OPEN_RATE_UNIT_t *pRate_unit);

/*********************************************************************
* @purpose  Set the policing committed burst size attribute
*           statement value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    entryType       @b{(input)} Entry type value
* @param    burstType       @b{(input)} burst type , i.e 
*                               OPEN_BURST_POLICE_TWORATE_CBURST or
*                               OPEN_BURST_POLICE_TWORATE_PBURST or
*                               OPEN_BURST_POLICE_SINGLERATE_CBURST or 
*                               OPEN_BURST_POLICE_SINGLERATE_EBURST or
*                               OPEN_BURST_POLICE_SIMPLE_CBURST
* @param    burst           @b{(input)} Burst size value
*
* @returns  OPEN_E_NONE     policing committed burst size attribute statement value is set successfully
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to set policing committed burst size attribute statement value
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
*
* @notes    The api should use the correct entry type and associated burst type 
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceBurstSet(openapiClientHandle_t *client_handle, 
                                                          uint32_t policyIndex, 
                                                          uint32_t policyInstIndex, 
                                                          uint32_t policyAttrIndex,
                                                          OPEN_DIFFSERV_POLICY_ATTR_TYPE_t entryType,
                                                          OPEN_BURST_POLICE_TYPE_t burstType, 
                                                          uint32_t burst);

/*********************************************************************
* @purpose  Get the policing committed burst size attribute
*           statement value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    entryType       @b{(input)} Entry type value
* @param    burstType       @b{(input)} burst type , i.e 
*                               OPEN_BURST_POLICE_TWORATE_CBURST or
*                               OPEN_BURST_POLICE_TWORATE_PBURST or
*                               OPEN_BURST_POLICE_SINGLERATE_CBURST or 
*                               OPEN_BURST_POLICE_SINGLERATE_EBURST or
*                               OPEN_BURST_POLICE_SIMPLE_CBURST
*
* @param    burst           @b{(output)} Burst size value
*
* @returns  OPEN_E_NONE    policing committed burst size attribute statement value is set successfully
* @returns  OPEN_E_PARAM   invalid parameter  passed
* @returns  OPEN_E_FAIL    failed to get policing committed burst size attribute statement value
* @returns  OPEN_E_ERROR   Not allowed for current match entry type
*
* @notes    The api should use the correct entry type and associated burst type 
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceBurstGet(openapiClientHandle_t *client_handle, 
                                                          uint32_t policyIndex, 
                                                          uint32_t policyInstIndex, 
                                                          uint32_t policyAttrIndex,
                                                          OPEN_DIFFSERV_POLICY_ATTR_TYPE_t entryType,
                                                          OPEN_BURST_POLICE_TYPE_t burstType, 
                                                          uint32_t *burst);
/*********************************************************************
* @purpose  Get the policing committed rate attribute statement
*           value
*
* @param    client_handle   @b{(input)}  client handle 
* @param    policyIndex     @b{(input)}  Policy index
* @param    policyInstIndex @b{(input)}  Policy instance index
* @param    policyAttrIndex @b{(input)}  Policy attribute index
* @param    entryType       @b{(input)}  Entry type value
* @param    pRate_percent   @b{(output)} Value percentage
*
* @returns  OPEN_E_NONE     policing committed rate attribute statement value is successfully retrieved.
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to get policing committed rate attribute statement value
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
*
* @notes    Usage of this function is only allowed for an attribute
*           statement entry type of:  _POLICE_SIMPLE.
*                                     _POLICE_SINGLERATE.
*                                     _POLICE_TWORATE.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceDataRatePercentGet(openapiClientHandle_t *client_handle,
                                                           uint32_t policyIndex,
                                                           uint32_t policyInstIndex,
                                                           uint32_t policyAttrIndex,
                                                           OPEN_DIFFSERV_POLICY_ATTR_TYPE_t entryType,
                                                           uint32_t *pRate_percent);

/*********************************************************************
* @purpose  Set the policing exceed value attribute statement value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    value           @b{(input)} Police exceed value
*
* @returns  OPEN_E_NONE     policing exceed value attribute statement value is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to set policing exceed value attribute statement value 
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
*
* @notes    Usage of this function is only allowed for an attribute 
*           statement entry type of:  _POLICE_SINGLERATE or _POLICE_TWORATE.
*
* @notes    This value is required when a police exceed action of _MARKxxxx
*           is used.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceExceedValSet(openapiClientHandle_t *client_handle, 
                                                      uint32_t policyIndex, 
                                                      uint32_t policyInstIndex, 
                                                      uint32_t policyAttrIndex, 
                                                      uint32_t value);

/*********************************************************************
* @purpose  Get the policing exceed value attribute statement value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    pValue          @b{(output)} Pointer to police exceed value
*              
* @returns  OPEN_E_NONE     policing exceed value attribute statement value get is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to get policing exceed value attribute statement value
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
*
* @notes    Usage of this function is only allowed for an attribute 
*           statement entry type of:  _POLICE_SINGLERATE or _POLICE_TWORATE.
*
* @notes    This value is required when a police exceed action of _MARKxxxx
*           is used.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceExceedValGet(openapiClientHandle_t *client_handle, 
                                                    uint32_t policyIndex, 
                                                    uint32_t policyInstIndex, 
                                                    uint32_t policyAttrIndex, 
                                                    uint32_t *pValue);

/*********************************************************************
* @purpose  Set the policing exceed action attribute statement value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    action          @b{(input)} Police action value
*
* @returns  OPEN_E_NONE     policing exceed action attribute statement value is set successfully
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to get policing exceed action attribute statement value
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
* @returns  OPEN_E_UNAVAIL  Feature not available.
*
* @notes    Usage of this function is only allowed for an attribute 
*           statement entry type of:  _POLICE_SINGLERATE or _POLICE_TWORATE.
*
* @notes    A policing action is one of _DROP, _MARKxxxx, or _SEND.
*           When _MARKxxxx is used, a police exceed value must 
*           also be specified.  
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceExceedActSet(openapiClientHandle_t *client_handle, 
                                                    uint32_t policyIndex, 
                                                    uint32_t policyInstIndex, 
                                                    uint32_t policyAttrIndex, 
                                        OPEN_DIFFSERV_POLICE_ACT_t action);

/*********************************************************************
* @purpose  Get the policing exceed action attribute statement value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    action          @b{(output)} Police action value
*
* @returns  OPEN_E_NONE     policing exceed action attribute statement value get is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to get policing exceed action attribute statement value
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
*
* @notes    Usage of this function is only allowed for an attribute 
*           statement entry type of:  _POLICE_SINGLERATE or _POLICE_TWORATE.
*
* @notes    A policing action is one of _DROP, _MARKxxxx, or _SEND.
*           When _MARKxxxx is used, a police exceed value must 
*           also be specified.  
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceExceedActGet(openapiClientHandle_t *client_handle, 
                                                    uint32_t policyIndex, 
                                                    uint32_t policyInstIndex, 
                                                    uint32_t policyAttrIndex, 
                                        OPEN_DIFFSERV_POLICE_ACT_t *action);

/*********************************************************************
* @purpose  Determine if a DiffServ class is valid for use as a policing
*           color aware designation
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)}  Policy index
* @param    policyInstIndex @b{(input)}  Policy instance index
* @param    classIndex      @b{(input)}  Class index value to be set
*
* @returns  OPEN_E_NONE    Class is valid for color-aware use
* @returns  OPEN_E_PARAM   invalid parameter  passed
* @returns  OPEN_E_ERROR   invalid class definition for color-aware use
* @returns  OPEN_E_FAIL    if any error or malfunction
*
* @notes    The class definition used for policing color awareness is only
*           allowed to contain a single, non-excluded class match condition
*           identifying one of the supported comparison fields: COS,
*           IP DSCP, IP Precedence, or Secondary COS.  Also, the field
*           used in this class must not conflict with the classifier
*           match conditions for this policy-class instance.  Anything else
*           results in an OPEN_E_ERROR return code.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceColorAwareClassValidate(
                                                       openapiClientHandle_t *client_handle, 
                                                       uint32_t policyIndex, 
                                                       uint32_t policyInstIndex, 
                                                       uint32_t classIndex);

/*********************************************************************
* @purpose  Get the policing color conform value attribute statement value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    pVal            @b{(output)} Pointer to police color value
*
* @returns  OPEN_E_NONE     policing color conform value attribute statement value get is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to get policing color conform value attribute statement value
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
*
* @notes    Usage of this function is only allowed for an attribute 
*           statement entry type of:  _POLICE_SIMPLE, _POLICE_SINGLERATE,
*           or _POLICE_TWORATE.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceColorConformValGet(openapiClientHandle_t *client_handle, 
                                                     uint32_t policyIndex, 
                                                     uint32_t policyInstIndex, 
                                                     uint32_t policyAttrIndex, 
                                                     uint32_t *pVal);

/*********************************************************************
* @purpose  Set the policing color conform attribute class index value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    classIndex      @b{(input)} Class index value to be set
*
* @returns  OPEN_E_NONE     policing color conform attribute class index value set is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_ERROR    invalid class definition for color-aware use
* @returns  OPEN_E_INTERNAL exceed color-aware parms same as Conform.
* @returns  OPEN_E_FAIL     if any other malfunction
*
* @notes    A classIndex of 0 sets the conform mode to 'color blind'.
*
* @notes    In color-aware mode, the color conform mode and value must
*           not be the same as the color exceed mode and value.  The
*           OPEN_E_INTERNAL return value is specifically used to 
*           indicate this error condition.
*
* @notes    Usage of this function is only meaningful for an attribute 
*           statement entry type of:  _POLICE_SIMPLE, _POLICE_SINGLERATE,
*           or _POLICE_TWORATE.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceColorConformIndexSet(
                                                       openapiClientHandle_t *client_handle, 
                                                       uint32_t policyIndex, 
                                                       uint32_t policyInstIndex, 
                                                       uint32_t policyAttrIndex, 
                                                       uint32_t classIndex);

/*********************************************************************
* @purpose  Get the policing color conform attribute class index value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    pClassIndex     @b{(output)} Pointer to class index output value  
*
* @returns  OPEN_E_INTERNAL   exceed color-aware parms same as Conform.
* @returns  OPEN_E_FAIL       failed to get policing color conform attribute class index value 
* @returns  OPEN_E_NONE       policing color conform attribute class index value get is successful
* @returns  OPEN_E_PARAM      invalid parameter  passed
* @returns  OPEN_E_NOT_FOUND  value is not configured
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceColorConformIndexGet(
                                                     openapiClientHandle_t *client_handle, 
                                                     uint32_t policyIndex, 
                                                     uint32_t policyInstIndex, 
                                                     uint32_t policyAttrIndex, 
                                                     uint32_t *pClassIndex);

/*********************************************************************
* @purpose  Set the policing nonconform action attribute statement value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    action          @b{(input)} Police action value
*
* @returns  OPEN_E_NONE     policing nonconform action attribute statement value set is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to set policing nonconform action attribute statement value
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
* @returns  OPEN_E_UNAVAIL  Feature not available.
*
* @notes    Usage of this function is only allowed for an attribute 
*           statement entry type of:  _POLICE_SIMPLE, _POLICE_SINGLERATE,
*           or _POLICE_TWORATE.
*
* @notes    A policing action is one of _DROP, _MARKxxxx, or _SEND.
*           When _MARKxxxx is used, a police nonconform value must 
*           also be specified.  
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceNonconformActSet(
  openapiClientHandle_t *client_handle,  uint32_t policyIndex, uint32_t policyInstIndex, 
  uint32_t policyAttrIndex, OPEN_DIFFSERV_POLICE_ACT_t action);

/*********************************************************************
* @purpose  Get the policing nonconform action attribute statement value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    action          @b{(output)} Police action value
*
* @returns  OPEN_E_NONE     policing nonconform action attribute statement value get is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to get policing nonconform action attribute statement value
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
*
* @notes    Usage of this function is only allowed for an attribute 
*           statement entry type of:  _POLICE_SIMPLE, _POLICE_SINGLERATE,
*           or _POLICE_TWORATE.
*
* @notes    A policing action is one of _DROP, _MARKxxxx, or _SEND.
*           When _MARKxxxx is used, a police nonconform value must 
*           also be specified.  
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceNonconformActGet(
  openapiClientHandle_t *client_handle,  uint32_t policyIndex, uint32_t policyInstIndex, 
  uint32_t policyAttrIndex, OPEN_DIFFSERV_POLICE_ACT_t *action);

/*********************************************************************
* @purpose  Set the policing nonconform value attribute statement value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    value           @b{(input)} Police noncofrm value
*
* @returns  OPEN_E_NONE     policing nonconform value attribute statement value set is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to set policing nonconform value attribute statement value
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
*
* @notes    Usage of this function is only allowed for an attribute 
*           statement entry type of:  _POLICE_SIMPLE, _POLICE_SINGLERATE,
*           or _POLICE_TWORATE.
*
* @notes    This value is required when a police nonconform action of _MARKxxxx
*           is used.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceNonconformValSet(openapiClientHandle_t *client_handle, 
                                                          uint32_t policyIndex, 
                                                          uint32_t policyInstIndex, 
                                                          uint32_t policyAttrIndex, 
                                                          uint32_t value);

/*********************************************************************
* @purpose  Get the policing nonconform value attribute statement value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    value           @b{(output)} Police noncofrm value
*
* @returns  OPEN_E_NONE     policing nonconform value attribute statement value get is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to get policing nonconform value attribute statement value
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
*
* @notes    Usage of this function is only allowed for an attribute 
*           statement entry type of:  _POLICE_SIMPLE, _POLICE_SINGLERATE,
*           or _POLICE_TWORATE.
*
* @notes    This value is required when a police nonconform action of _MARKxxxx
*           is used.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceNonconformValGet(openapiClientHandle_t *client_handle, 
                                                          uint32_t policyIndex, 
                                                          uint32_t policyInstIndex, 
                                                          uint32_t policyAttrIndex, 
                                                          uint32_t *value);

/*********************************************************************
* @purpose  Set the policing conform value attribute statement value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    value           @b{(input)} Police conform value

*
* @returns  OPEN_E_NONE     policing conform value attribute statement value set is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to set policing conform value attribute statement value
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
*
* @notes    Usage of this function is only allowed for an attribute 
*           statement entry type of:  _POLICE_SIMPLE, _POLICE_SINGLERATE,
*           or _POLICE_TWORATE.
*
* @notes    This value is required when a police conform action of _MARKxxxx
*           is used.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceConformValSet(openapiClientHandle_t *client_handle, 
                                                     uint32_t policyIndex, 
                                                     uint32_t policyInstIndex, 
                                                     uint32_t policyAttrIndex, 
                                                     uint32_t value);
/*********************************************************************
* @purpose  Get the policing conform value attribute statement value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    value           @b{(output)} Police conform value

*
* @returns  OPEN_E_NONE     policing conform value attribute statement value get is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to get policing conform value attribute statement value
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
*
* @notes    Usage of this function is only allowed for an attribute 
*           statement entry type of:  _POLICE_SIMPLE, _POLICE_SINGLERATE,
*           or _POLICE_TWORATE.
*
* @notes    This value is required when a police conform action of _MARKxxxx
*           is used.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceConformValGet(openapiClientHandle_t *client_handle, 
                                                     uint32_t policyIndex, 
                                                     uint32_t policyInstIndex, 
                                                     uint32_t policyAttrIndex, 
                                                     uint32_t *value);

/*********************************************************************
* @purpose  Determine next sequential row in the Policy Attribute Table
*
* @param    client_handle        @b{(input)} client handle 
* @param    prevPolicyIndex      @b{(input)} Policy index to begin search
* @param    prevPolicyInstIndex  @b{(input)} Policy instance index to begin 
*                                              search
* @param    prevPolicyAttrIndex  @b{(input)} Policy attribute index to begin 
*                                              search
* @param    pPolicyIndex         @b{(output)} Pointer to next sequential policy index
*                                      value
* @param    pPolicyInstIndex     @b{(output)} Pointer to next sequential policy
*                                            instance index value
* @param    pPolicyAttrIndex     @b{(output)} Pointer to next sequential policy
*                                            attribute index value
*
* @returns  OPEN_E_NONE     next index get is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to get next index
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
*
* @notes    A prevPolicyIndex value of 0 is used to find the first policy
*           entry in the table.
*
* @notes    A prevPolicyInstIndex value of 0 is used to find the first class
*           instance relative to the specified prevPolicyIndex.
*
* @notes    A prevPolicyAttrIndex value of 0 is used to find the first policy
*           attribute relative to the specified prevPolicyIndex, 
*           prevPolicyInstIndex.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrGetNext(openapiClientHandle_t *client_handle, 
                                       uint32_t prevPolicyIndex, 
                                       uint32_t prevPolicyInstIndex, 
                                       uint32_t prevPolicyAttrIndex, 
                                       uint32_t *pPolicyIndex,
                                       uint32_t *pPolicyInstIndex,
                                       uint32_t *pPolicyAttrIndex);

/*********************************************************************
* @purpose  Set the policing conform action attribute statement value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    action          @b{(input)} Police action value
*
* @returns  OPEN_E_NONE     policing conform action attribute statement value set is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to set policing conform action attribute statement value
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
* @returns  OPEN_E_UNAVAIL  Feature not available.
*
* @notes    Usage of this function is only allowed for an attribute 
*           statement entry type of:  _POLICE_SIMPLE, _POLICE_SINGLERATE,
*           or _POLICE_TWORATE.
*
* @notes    A policing action is one of _DROP, _MARKxxxx, or _SEND.
*           When _MARKxxxx is used, a police conform value must 
*           also be specified.  
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceConformActSet(openapiClientHandle_t *client_handle, 
                                                       uint32_t policyIndex, 
                                                       uint32_t policyInstIndex, 
                                                       uint32_t policyAttrIndex, 
                                              OPEN_DIFFSERV_POLICE_ACT_t action);

/*********************************************************************
* @purpose  Get the policing conform action attribute statement value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    action          @b{(output)} Police action value
*
* @returns  OPEN_E_NONE     policing conform action attribute statement value get is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to get policing conform action attribute statement value
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
*
* @notes    Usage of this function is only allowed for an attribute 
*           statement entry type of:  _POLICE_SIMPLE, _POLICE_SINGLERATE,
*           or _POLICE_TWORATE.
*
* @notes    A policing action is one of _DROP, _MARKxxxx, or _SEND.
*           When _MARKxxxx is used, a police conform value must 
*           also be specified.  
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceConformActGet(openapiClientHandle_t *client_handle, 
                                                       uint32_t policyIndex, 
                                                       uint32_t policyInstIndex, 
                                                       uint32_t policyAttrIndex, 
                                              OPEN_DIFFSERV_POLICE_ACT_t *action);

/*********************************************************************
* @purpose  Set the two-rate policing peak rate attribute statement
*           value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    rate            @b{(input)} Rate value
*
* @returns  OPEN_E_NONE     two-rate policing peak rate attribute statement value set is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to set two-rate policing peak rate attribute statement value
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
*
* @notes    Usage of this function is only allowed for an attribute 
*           statement entry type of:  _POLICE_TWORATE.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceTworatePrateSet(openapiClientHandle_t *client_handle, 
                                                         uint32_t policyIndex, 
                                                         uint32_t policyInstIndex, 
                                                         uint32_t policyAttrIndex, 
                                                         uint32_t rate);
/*********************************************************************
* @purpose  Set the simple policing committed rate attribute statement
*           value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    rate            @b{(input)} Rate value
*
* @returns  OPEN_E_NONE     simple policing committed rate attribute statement value set is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to set simple policing committed rate attribute statement value
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
*
* @notes    Usage of this function is only allowed for an attribute 
*           statement entry type of:  _POLICE_SIMPLE.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceSimpleCrateSet(openapiClientHandle_t *client_handle, 
                                                         uint32_t policyIndex, 
                                                         uint32_t policyInstIndex, 
                                                         uint32_t policyAttrIndex, 
                                                         uint32_t rate);

/*********************************************************************
* @purpose  Get the two-rate policing peak rate attribute statement
*           value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    rate            @b{(output)} Rate value
*
* @returns  OPEN_E_NONE     two-rate policing peak rate attribute statement value get is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to two-rate policing peak rate attribute statement value
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
*
* @notes    Usage of this function is only allowed for an attribute 
*           statement entry type of:  _POLICE_TWORATE.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceTworatePrateGet(openapiClientHandle_t *client_handle, 
                                                         uint32_t policyIndex, 
                                                         uint32_t policyInstIndex, 
                                                         uint32_t policyAttrIndex, 
                                                         uint32_t *rate);
/*********************************************************************
* @purpose  Get the simple policing committed rate attribute statement
*           value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    rate            @b{(output)} Rate value
*
* @returns  OPEN_E_NONE     simple policing committed rate attribute statement value get is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to get simple policing committed rate attribute statement value
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
*
* @notes    Usage of this function is only allowed for an attribute 
*           statement entry type of:  _POLICE_SIMPLE.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceSimpleCrateGet(openapiClientHandle_t *client_handle, 
                                                         uint32_t policyIndex, 
                                                         uint32_t policyInstIndex, 
                                                         uint32_t policyAttrIndex, 
                                                         uint32_t *rate);


/*********************************************************************
* @purpose  Set the two-rate policing peak rate attribute statement
*           value as per unit
*
* @param    client_handle    @b{(input)} client handle 
* @param    policyIndex      @b{(input)} Policy index
* @param    policyInstIndex  @b{(input)} Policy instance index
* @param    policyAttrIndex  @b{(input)} Policy attribute index
* @param    rate_unit        @b{(input)} Rate unit (in kbps
*                                        or percentage)
* @param    rate             @b{(input)} Rate value in percentage
*
* @returns  OPEN_E_NONE     two-rate policing peak rate attribute statement value as per unit set is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
* @returns  OPEN_E_FAIL     failed to set two-rate policing peak rate attribute statement value as per unit set
*
* @notes    Usage of this function is only allowed for an attribute
*           statement entry type of:  _POLICE_TWORATE.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceTworatePdataRateSet(openapiClientHandle_t *client_handle,
                                                             uint32_t policyIndex,
                                                             uint32_t policyInstIndex,
                                                             uint32_t policyAttrIndex,
                                                             OPEN_RATE_UNIT_t rate_unit,
                                                             uint32_t rate);
/*********************************************************************
* @purpose  Get the policing peak rate attribute statement
*           value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex @b{(input)} Policy index
* @param    policyInstIndex  @b{(input)} Policy instance index
* @param    policyAttrIndex  @b{(input)} Policy attribute index
* @param    entryType       @b{(input)}  Entry type value
* @param    pRate_percent    @b{(output)} Value percentage
*
* @returns  OPEN_E_NONE  
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL  
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
*
* @notes    Usage of this function is only allowed for an attribute
*           statement entry type of:  _POLICE_TWORATE.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPolicePdataRatePercentGet(openapiClientHandle_t *client_handle,
                                                           uint32_t policyIndex,
                                                           uint32_t policyInstIndex,
                                                           uint32_t policyAttrIndex,
                                                           OPEN_DIFFSERV_POLICY_ATTR_TYPE_t entryType,
                                                           uint32_t *pRate_percent);

/*********************************************************************
* @purpose  Get the two-rate policing committed rate attribute
*           statement value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    pRate           @b{(output)} Pointer to rate value
*
* @returns  OPEN_E_NONE     two-rate policing committed rate attribute statement value get is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to get two-rate policing committed rate attribute statement value
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
*
* @notes    Usage of this function is only allowed for an attribute 
*           statement entry type of:  _POLICE_TWORATE.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceTworateCrateGet(openapiClientHandle_t *client_handle, 
                                                         uint32_t policyIndex, 
                                                         uint32_t policyInstIndex, 
                                                         uint32_t policyAttrIndex, 
                                                         uint32_t *pRate);

/*********************************************************************
* @purpose  Set the two-rate policing committed rate attribute
*           statement value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    rate            @b{(input)} rate value
*
* @returns  OPEN_E_NONE     two-rate policing committed rate attribute statement value set is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_FAIL     failed to set two-rate policing committed rate attribute statement value
* @returns  OPEN_E_ERROR    Not allowed for current match entry type
*
* @notes    Usage of this function is only allowed for an attribute 
*           statement entry type of:  _POLICE_TWORATE.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceTworateCrateSet(openapiClientHandle_t *client_handle, 
                                                         uint32_t policyIndex, 
                                                         uint32_t policyInstIndex, 
                                                         uint32_t policyAttrIndex, 
                                                         uint32_t rate);


/*********************************************************************
* @purpose  Set the policing color exceed attribute class index value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    classIndex      @b{(input)} Class index value to be set
*
* @returns  OPEN_E_NONE     policing color exceed attribute class index value set is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_ERROR    invalid class definition for color-aware use
* @returns  OPEN_E_INTERNAL exceed color-aware parms same as Conform.
* @returns  OPEN_E_FAIL     failed to set policing color exceed attribute class index value
*
* @notes    A classIndex of 0 sets the exceed mode to 'color blind'.
*
* @notes    In color-aware mode, the color exceed mode and value must
*           not be the same as the color conform mode and value.  The
*           OPEN_E_INTERNAL return value is specifically used to 
*           indicate this error condition.
*
* @notes    Usage of this function is only meaningful for an attribute 
*           statement entry type of:  _POLICE_SINGLERATE or _POLICE_TWORATE.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceColorExceedIndexSet(
                                                       openapiClientHandle_t *client_handle, 
                                                       uint32_t policyIndex, 
                                                       uint32_t policyInstIndex, 
                                                       uint32_t policyAttrIndex, 
                                                       uint32_t classIndex);


/*********************************************************************
* @purpose  Get the policing color exceed attribute class index value
*
* @param    client_handle   @b{(input)} client handle 
* @param    policyIndex     @b{(input)} Policy index
* @param    policyInstIndex @b{(input)} Policy instance index
* @param    policyAttrIndex @b{(input)} Policy attribute index
* @param    pClassIndex     @b{(output)} Class index value to be set
*
* @returns  OPEN_E_NONE     policing color exceed attribute class index get is successful
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_ERROR    invalid class definition for color-aware use
* @returns  OPEN_E_INTERNAL exceed color-aware parms same as Conform.
* @returns  OPEN_E_FAIL     failed to get policing color exceed attribute class index
*
*
* @notes    In color-aware mode, the color exceed mode and value must
*           not be the same as the color conform mode and value.  The
*           OPEN_E_INTERNAL return value is specifically used to 
*           indicate this error condition.
*
* @notes    Usage of this function is only meaningful for an attribute 
*           statement entry type of:  _POLICE_SINGLERATE or _POLICE_TWORATE.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiDiffServPolicyAttrStmtPoliceColorExceedIndexGet(
                                                       openapiClientHandle_t *client_handle, 
                                                       uint32_t policyIndex, 
                                                       uint32_t policyInstIndex, 
                                                       uint32_t policyAttrIndex, 
                                                       uint32_t *pClassIndex);

#endif /* OPENAPI_DIFFSERV_POLICY_H_INCLUDED */
