/**********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_diffserv_serv.h
*
* @purpose   This code implements the OpEN Diff services  API
*
* @component OPEN
*
* @create    05/19/2023
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_DIFFSERV_SERVICE_H_INCLUDED
#define OPENAPI_DIFFSERV_SERVICE_H_INCLUDED

#include "openapi_common.h"

/** DiffServ Interface Direction type */
typedef enum
{
  OPEN_MIB_DIFFSERV_IF_DIRECTION_NONE = 0,
  OPEN_MIB_DIFFSERV_IF_DIRECTION_IN,        /**< Interface direction type IN */
  OPEN_MIB_DIFFSERV_IF_DIRECTION_OUT,       /**< Interface direction type OUT */
} OPEN_MIB_DIFFSERV_IF_DIRECTION_t;

/** DiffServ Interface Operational Status type */
typedef enum
{
  OPEN_MIB_DIFFSERV_IF_OPER_STATUS_NONE = 0, /**< not a valid value */
  OPEN_MIB_DIFFSERV_IF_OPER_STATUS_UP,       /**< Interface operational status UP */
  OPEN_MIB_DIFFSERV_IF_OPER_STATUS_DOWN,     /**< Interface operational status DOWN */
} OPEN_MIB_DIFFSERV_IF_OPER_STATUS_t;

/** Enumerated values for RowStatus object */
typedef enum
{
  OPEN_MIB_DIFFSERV_ROW_STATUS_NONE = 0,        /**< Not a valid value */
  OPEN_MIB_DIFFSERV_ROW_STATUS_ACTIVE,          /**< Active */
  OPEN_MIB_DIFFSERV_ROW_STATUS_NOT_IN_SERVICE,  /**< Not in service */
  OPEN_MIB_DIFFSERV_ROW_STATUS_NOT_READY,       /**< Not ready */
  OPEN_MIB_DIFFSERV_ROW_STATUS_CREATE_AND_GO,   /**< Create and go */
  OPEN_MIB_DIFFSERV_ROW_STATUS_CREATE_AND_WAIT, /**< Create and wait */
  OPEN_MIB_DIFFSERV_ROW_STATUS_DESTROY,         /**< Destroy */
} OPEN_MIB_DIFFSERV_ROW_STATUS_t;

/*****************************************************************************
 * @purpose   Create a new row in the Service Table for the specified
 *            interface and direction.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Internal interface number
 * @param     ifDir            @b{(input)}  Interface direction
 * @param     activateRow      @b{(input)}  Activate row when value is TRUE
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_EXISTS           Already configured.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     An OPEN_E_FAIL response means the service index values are
 *            currently in use, but the row is not complete.
 *            This is typically caused when multiple requestors use the same
 *            index values and race to create the new row.
 *            The caller should use new index values and re-attempt 
 *            to create the row.
 *            An OPEN_E_EXISTS response means the service index values
 *            are currently in use for a completed row.
 *            The caller should use new index values and re-attempt to
 *            create the row.
 *            The row status is initially set to 'not ready' and remains that
 *            way until every object in the row contains a valid value.  If the
 *            activateRow parameter is set to OPEN_TRUE, the row status will 
 *            then be automatically updated to 'active'.
 *            Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiDiffServServiceCreate(openapiClientHandle_t *client_handle,
                             uint32_t intIfNum,
                             OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir,
                             OPEN_BOOL_t activateRow);

/*****************************************************************************
 * @purpose   Delete a row from the Service Table for the specified interface
 *            and direction.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Internal interface number
 * @param     ifDir            @b{(input)}  Interface direction
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API Deletea a row from the Service Table for the specified
 *            interface and direction.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiDiffServServiceDelete(openapiClientHandle_t *client_handle,
                             uint32_t intIfNum,
                             OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir);

/*****************************************************************************
 * @purpose   Gets a row from the Service Table for the specified interface
 *            and direction.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Internal interface number
 * @param     ifDir            @b{(input)}  Interface direction
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Verify that a Service Table row exists for the specified 
 *            interface and direction.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiDiffServServiceGet(openapiClientHandle_t *client_handle,
                          uint32_t intIfNum,
                          OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir);

/*****************************************************************************
 * @purpose   Determine next sequential row in the Service Table.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Internal interface number
 * @param     ifDir            @b{(input)}  Interface direction
 * @param     pNxtIntIfNum     @b{(output)} Next sequential internal interface
 *                                          number
 * @param     pNxtIfDir        @b{(output)} Next sequential interface direction
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     intIfNum value of 0 is used to find the first interface 
 *            entry the table.
 * @notes     ifDir value of 0 is used to find the first interface direction
 *            entry relative to the specified intIfNum.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiDiffServServiceGetNext(openapiClientHandle_t *client_handle,
                              uint32_t intIfNum,
                              OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir,
                              uint32_t *pNxtIntIfNum,
                              OPEN_MIB_DIFFSERV_IF_DIRECTION_t *pNxtIfDir);

/*****************************************************************************
 * @purpose   Set the policy index attached to the specified interface
 *            and direction.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Internal interface number
 * @param     ifDir            @b{(input)}  Interface direction
 * @param     policyIndex      @b{(input)}  Policy index
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets policy index attached to the specified interface
 *            and direction.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiDiffServServicePolicyIndexSet(openapiClientHandle_t *client_handle,
                                     uint32_t intIfNum,
                                     OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir,
                                     uint32_t policyIndex);

/*****************************************************************************
 * @purpose   Get the policy index attached to the specified interface
 *            and direction.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Internal interface number
 * @param     ifDir            @b{(input)}  Interface direction
 * @param     pPolicyIndex     @b{(output)} Policy index
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches policy index attached to the specified
 *            interface and direction.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiDiffServServicePolicyIndexGet(openapiClientHandle_t *client_handle,
                                     uint32_t intIfNum,
                                     OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir,
                                     uint32_t *pPolicyIndex);

/*****************************************************************************
 * @purpose   Get the service interface operational status for the
 *            specified Service Table row.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Internal interface number
 * @param     ifDir            @b{(input)}  Interface direction
 * @param     pIfOperStatus    @b{(output)} Interface operational status
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches service interface operational status
 *            for the specified Service Table row.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiDiffServServiceIfOperStatusGet(openapiClientHandle_t *client_handle,
                                      uint32_t intIfNum,
                                      OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir,
                                      OPEN_MIB_DIFFSERV_IF_OPER_STATUS_t *pIfOperStatus);

/*****************************************************************************
 * @purpose   Check if the specified internal interface number is in use by
 *            the ACL component.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Internal interface number
 * @param     ifDir            @b{(input)}  Interface direction
 * @param     pIsAclIntfInUse  @b{(output)} status of interface number in use
 *                                          by the ACL component
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API checks if the specified internal interface number is 
 *            in use by the ACL component.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiDiffServIsAclIntfInUse(openapiClientHandle_t *client_handle,
                              uint32_t intIfNum,
                              OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir,
                              OPEN_BOOL_t *pIsAclIntfInUse);

/*****************************************************************************
 * @purpose   Set the given status for the specified Service Table row.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Internal interface number
 * @param     ifDir            @b{(input)}  Interface direction
 * @param     status           @b{(input)}  Row status value
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets the given status for the specified Service Table row.
 * @notes     Calling this API will change the running configuration of the switch.
 * @notes     Supported row status are OPEN_MIB_DIFFSERV_ROW_STATUS_ACTIVE,
 * @notes     OPEN_MIB_DIFFSERV_ROW_STATUS_NOT_IN_SERVICE and OPEN_MIB_DIFFSERV_ROW_STATUS_DESTROY.
 * @notes     Non supported row status are used for internal purpose.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiDiffServServiceRowStatusSet(openapiClientHandle_t *client_handle,
                                   uint32_t intIfNum,
                                   OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir,
                                   OPEN_MIB_DIFFSERV_ROW_STATUS_t status);

/*****************************************************************************
 * @purpose   Get the current status for the specified Service Table row.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Internal interface number
 * @param     ifDir            @b{(input)}  Interface direction
 * @param     pStatus          @b{(output)} Row status value
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the current status for the specified Service Table row.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiDiffServServiceRowStatusGet(openapiClientHandle_t *client_handle,
                                   uint32_t intIfNum,
                                   OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir,
                                   OPEN_MIB_DIFFSERV_ROW_STATUS_t *pStatus);

/*****************************************************************************
 * @purpose   Verifies the current status for the specified Service Table row.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Internal interface number
 * @param     ifDir            @b{(input)}  Interface direction
 * @param     status           @b{(input)}  Row status value
 *
 * @returns   OPEN_E_UNAVAIL            Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API verifies the current status for the specified Service Table row.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiDiffServServiceRowStatusVerify(openapiClientHandle_t *client_handle,
                                      uint32_t intIfNum,
                                      OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir,
                                      OPEN_MIB_DIFFSERV_ROW_STATUS_t status);

/*****************************************************************************
 * @purpose   Get the inbound offered packets count for the specified
 *            policy class instance.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     policyIndex      @b{(input)}  Policy index
 * @param     policyInstIndex  @b{(input)}  Policy instance index
 * @param     intIfNum         @b{(input)}  Internal interface number
 * @param     pPacktCount      @b{(output)} Packet count
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches inbound packets count for the
 *            specified policy class instance.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiDiffServPolicyPerfInOfferedPacketsGet(openapiClientHandle_t *client_handle,
                                             uint32_t policyIndex,
                                             uint32_t policyInstIndex,
                                             uint32_t intIfNum,
                                             uint64_t *pPacktCount);

/*****************************************************************************
 * @purpose   Get the outbound offered packets count for the
 *            specified policy class instance.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     policyIndex      @b{(input)}  Policy index
 * @param     policyInstIndex  @b{(input)}  Policy instance index
 * @param     intIfNum         @b{(input)}  Internal interface number
 * @param     pPacktCount      @b{(output)} Packet count
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches outbound packets count for the
 *            specified policy class instance.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiDiffServPolicyPerfOutOfferedPacketsGet(openapiClientHandle_t *client_handle,
                                              uint32_t policyIndex,
                                              uint32_t policyInstIndex,
                                              uint32_t intIfNum,
                                              uint64_t *pPacktCount);

/*****************************************************************************
 * @purpose   Get the inbound discarded packet count for the specified
 *            policy class instance.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     policyIndex      @b{(input)}  Policy index
 * @param     policyInstIndex  @b{(input)}  Policy instance index
 * @param     intIfNum         @b{(input)}  Internal interface number
 * @param     pPacktCount      @b{(output)} Packet count
 *
 * @returns   OPEN_E_UNAVAIL          Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches inbound discarded packet count for the
 *            specified policy class instance.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiDiffServPolicyPerfInDiscardedPacketsGet(openapiClientHandle_t *client_handle,
                                               uint32_t policyIndex,
                                               uint32_t policyInstIndex,
                                               uint32_t intIfNum,
                                               uint64_t *pPacktCount);

/*****************************************************************************
 * @purpose   Get the outbound discarded packet count for the specified
 *            policy class instance.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     policyIndex      @b{(input)}  Policy index
 * @param     policyInstIndex  @b{(input)}  Policy instance index
 * @param     intIfNum         @b{(input)}  Internal interface number
 * @param     pPacktCount      @b{(output)} Packet count
 *
 * @returns   OPEN_E_UNAVAIL            Operation unsupported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches outbound discarded packet count for the
 *            specified policy class instance.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t
openapiDiffServPolicyPerfOutDiscardedPacketsGet(openapiClientHandle_t *client_handle,
                                                uint32_t policyIndex,
                                                uint32_t policyInstIndex,
                                                uint32_t intIfNum,
                                                uint64_t *pPacktCount);
#endif /* OPENAPI_DIFFSERV_SERVICE_H_INCLUDED */
