/*********************************************************************
*
* Copyright 2017-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_ets.h
*
* @purpose   Enhanced Transmission Session APIs.
*
* @component OPEN
*
* @create    01/23/2017
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_ETS_H_INCLUDED
#define OPENAPI_ETS_H_INCLUDED

#include "openapi_common.h"

#define OPEN_DOT1P_MIN_PRIORITY   0
#define OPEN_DOT1P_MAX_PRIORITY   7

#define OPEN_ETS_TCG_NUM_MIN      0
#define OPEN_ETS_TCG_NUM_MAX      7

#define OPEN_ETS_BW_PERCENTAGE_MIN 1
#define OPEN_ETS_BW_PERCENTAGE_MAX 100

/*********************************************************************
* @purpose  Gets the count of incompatible configurations received from peer.
*
* @param    client_handle         @b{(input)}   client handle from registration API.
* @param    ifNum                 @b((input))   Interface number.
* @param    incompatibleCfgCount  @b((output))  Count of incompatible configurations.
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_FAIL       Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiEtsPeerIncompatibleCfgCountGet(openapiClientHandle_t *client_handle, 
                                                   uint32_t ifNum, 
                                                   uint32_t *incompatibleCfgCount);

/*********************************************************************
* @purpose  Gets the peer number of traffic class groups for
*           the interface
*
* @param    client_handle   @b{(input)}   client handle from registration API.
* @param    ifNum           @b((input))   Interface number.
* @param    priority        @b{(input)}   The priority.
* @param    numTc           @b((output))  Number of traffic classes.
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_NOT_FOUND  Priority level is not supported.
* @returns  OPEN_E_FAIL       Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiEtsPeerNumTcGet(openapiClientHandle_t *client_handle,
                                    uint32_t ifNum, 
                                    uint32_t priority,
                                    uint8_t  *numTc);

/*********************************************************************
* @purpose  Get the willingness of the peer.
*
* @param    client_handle   @b{(input)}   client handle from registration API.
* @param    ifNum           @b((input))   Interface number.
* @param    willing         @b((output))  Willingness of the peer.
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_NOT_FOUND    Peer not detected.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiEtsPeerWillingGet(openapiClientHandle_t *client_handle,
                                      uint32_t ifNum, 
                                      OPEN_BOOL_t *willing);

/*********************************************************************
* @purpose  Gets the cbs status received from peer.
*
* @param    client_handle   @b{(input)}   client handle from registration API.
* @param    ifNum           @b((input))   Interface number.
* @param    cbsStatus       @b((output))  cbs status of the peer.
*
* @returns  OPEN_E_NONE   On success.
* @returns  OPEN_E_PARAM  Bad parameter.
* @returns  OPEN_E_FAIL   Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiEtsPeerCbsStatusGet(openapiClientHandle_t *client_handle,
                                        uint32_t ifNum, 
                                        OPEN_BOOL_t *cbsStatus);

/*********************************************************************
* @purpose  Gets the peer minimum bandwidth.
*
* @param    client_handle   @b{(input)}   client handle from registration API.
* @param    ifNum           @b((input))   Interface number.
* @param    tcg             @b((output))  TCG index.
* @param    bw              @b((output))  Minimum bandwidth.
*
* @returns  OPEN_E_NONE   On success.
* @returns  OPEN_E_PARAM  Bad parameter.
* @returns  OPEN_E_FAIL   Some failure occurred.
*
* @notes tcg must be in the range [OPEN_ETS_TCG_NUM_MIN, OPEN_ETS_TCG_NUM_MAX]
*        bandwidth values are percentage in range [OPEN_ETS_BW_PERCENTAGE_MIN, OPEN_ETS_BW_PERCENTAGE_MAX]
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiEtsPeerMinBandwidthGet(openapiClientHandle_t *client_handle,
                                           uint32_t ifNum,
                                           uint32_t tcg,
                                           uint8_t *bw);

/*********************************************************************
* @purpose  Gets the peer schedule info.
*
* @param    client_handle   @b{(input)}   client handle from registration API.
* @param    ifNum           @b((input))   Interface number.
* @param    tcg             @b((output))  TCG index.
* @param    sched           @b((output))  Schedule info.
*
* @returns  OPEN_E_NONE   On success.
* @returns  OPEN_E_PARAM  Bad parameter.
* @returns  OPEN_E_FAIL   Some failure occurred.
*
* @notes tcg must be in the range [OPEN_ETS_TCG_NUM_MIN, OPEN_ETS_TCG_NUM_MAX]
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiEtsPeerTsaGet(openapiClientHandle_t *client_handle,
                                  uint32_t ifNum,
                                  uint32_t tcg,
                                  uint8_t *sched);

#endif /* OPENAPI_ETS_H_INCLUDED */
