/**********************************************************************
*
*  Copyright 2024 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_event_notify.h
*
* @purpose   This code implements the APIs required for OpEN Event Notifications
*
* @component OPEN
*
* @create    04/10/2024
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_EVENT_NOTIFY_H_INCLUDED
#define OPENAPI_EVENT_NOTIFY_H_INCLUDED

#include "openapi_common.h" 
#include "openapi_fdb.h"
#include "openapi_if.h"
#include "openapi_poe.h"
#include "openapi_flow_control.h"
#include "openapi_routing_arptable.h"
#include "openapi_dhcp_client.h"

/** OpEN events for which clients can register */
typedef enum
{
  OPEN_NOTIFY_EVENT_IP_INFO                = 1,  /**< Change in IP address/Gateway/DNS server/Hostname */
  OPEN_NOTIFY_EVENT_ARP                    = 2,  /**< Add/Remove of ARP entries */
  OPEN_NOTIFY_EVENT_IPV4_ROUTE             = 3,  /**< Add/Remove/Change of ipv4 route entry */
  OPEN_NOTIFY_EVENT_PORT_INFO              = 4,  /**< Change in admin/oper status or speed/AN/duplex/mdix of port */
  OPEN_NOTIFY_EVENT_LLDP_NEIGHBORS         = 5,  /**< Add/Remove LLDP neighbor */
  OPEN_NOTIFY_EVENT_POE_STATUS             = 6,  /**< Delivering power/disabled/fault status of PoE port */
  OPEN_NOTIFY_EVENT_FDB_ENTRY              = 7,  /**< Add/Remove/Age-out of FDB entries */
  OPEN_NOTIFY_EVENT_LAST                   = 8   /**< For bounds checking  */
} OPEN_NOTIFY_EVENT_t;


#define OPEN_EVENT_CREATE     1 /**< Event change type : Create */
#define OPEN_EVENT_DELETE     2 /**< Event change type : Delete */
#define OPEN_EVENT_UPDATE     3 /**< Event change type : Update */

#define  OPEN_PORT_MDIX_AUTO         0  /**<  Mdix auto mode */
#define  OPEN_PORT_MDIX_FORCE_AUTO   1  /**<  Mdix force auto mode */
#define  OPEN_PORT_MDIX_NORMAL       2  /**<  Mdix normal mode */
#define  OPEN_PORT_MDIX_XOVER        3  /**<  Mdix cross over mode */
#define  OPEN_PORT_MDIX_UNKNOWN      4  /**<  Unknown mdix mode */


/** Lower/Upper limit for number of events to wait
     before sending notification per event type */

#define OPEN_IP_INFO_MIN_EVENTS                  1    /**< Minimum no of ip info events */ 
#define OPEN_IP_INFO_MAX_EVENTS                  32   /**< Maximum no of ip info events */

#define OPEN_ARP_MIN_EVENTS                      16   /**< Minimum no of arp events */
#define OPEN_ARP_MAX_EVENTS                      256  /**< Maximum no of arp events */

#define OPEN_IPV4_ROUTE_MIN_EVENTS               16   /**< Minimum no of ipv4 route events */
#define OPEN_IPV4_ROUTE_MAX_EVENTS               256  /**< Maximum no of ipv4 route events */

#define OPEN_PORT_INFO_MIN_EVENTS                1    /**< Minimum no of port info events */
#define OPEN_PORT_INFO_MAX_EVENTS                256  /**< Maximum no of port info events */

#define OPEN_LLDP_NEIGHBORS_MIN_EVENTS           1    /**< Minimum no of lldp neighbor events */ 
#define OPEN_LLDP_NEIGHBORS_MAX_EVENTS           8    /**< Maximum no of lldp neighbor events */
 
#define OPEN_POE_STATUS_MIN_EVENTS               1    /**< Minimum no of poe status events */
#define OPEN_POE_STATUS_MAX_EVENTS               256  /**< Maximum no of poe status events */

#define OPEN_FDB_ENTRY_MIN_EVENTS                16   /**< Minimum no of fdb events */
#define OPEN_FDB_ENTRY_MAX_EVENTS                512  /**< Maximum no of fdb events */


/** Lower/Upper limit of timeout before
     sending notification per event type */

#define OPEN_IP_INFO_MIN_TIMEOUT                10    /**< Minimum time out for ip info events */
#define OPEN_IP_INFO_MAX_TIMEOUT                600   /**< Maximum time out for ip info events */

#define OPEN_ARP_MIN_TIMEOUT                    10   /**< Minimum time out for arp events */
#define OPEN_ARP_MAX_TIMEOUT                    600  /**< Maximum time out for arp  events */

#define OPEN_IPV4_ROUTE_MIN_TIMEOUT             10   /**< Minimum time out for ipv4 route events */
#define OPEN_IPV4_ROUTE_MAX_TIMEOUT             600  /**< Maximum time out for ipv4 route events */

#define OPEN_PORT_INFO_MIN_TIMEOUT              10   /**< Minimum time out for port info events */
#define OPEN_PORT_INFO_MAX_TIMEOUT              600  /**< Maximum time out for port info events */

#define OPEN_LLDP_NEIGHBORS_MIN_TIMEOUT         10   /**< Minimum time out for lldp neighbor events */
#define OPEN_LLDP_NEIGHBORS_MAX_TIMEOUT         600  /**< Maximum time out for lldp neighbor events */

#define OPEN_POE_STATUS_MIN_TIMEOUT             10   /**< Minimum time out for poe status events */
#define OPEN_POE_STATUS_MAX_TIMEOUT             600  /**< Maximum time out for poe status events */

#define OPEN_FDB_ENTRY_MIN_TIMEOUT              20   /**< Minimum time out for fdb events */
#define OPEN_FDB_ENTRY_MAX_TIMEOUT              600  /**< Maximum time out for fdb events */


#define OPEN_EVENT_NOTIFY_MAX_STR_LEN       256     /**< Maximum string length */
#define OPEN_EVENT_NOTIFY_DATA_MAX_SIZE     16384   /**< Maximum size of the data for socket */
#define OPEN_EVENT_NOTIFY_SOCK_NAME  "/tmp/event_notify/sockets/client-notify-sock-%d"  /**< Socket name */


/** Information received in notification for event OPEN_NOTIFY_EVENT_PORT_INFO */
typedef struct
{
  uint32_t                      ifNum;         /**< index to identify port uniquely */
  OPEN_PORT_STATE_t             portState;     /**< Admin state disabled/enabled/diag-disabled */
  OPEN_LINK_STATE_t             linkState;     /**< Link state up/down */
  OPEN_FLOWCONTROL_OPER_t       flowControl;   /**< Active or inactive */
  OPEN_DISABLE_REASON_t         disableReason; /**< disable reason applicable when portState is diag-disabled */
  OPEN_PORT_SPEEDS_t            ifSpeed;       /**< port speed */
  OPEN_PORT_AUTO_NEG_t          autoneg;       /**< auto-negotiation  enabled/disabled */
  OPEN_PORT_DUPLEX_t            duplex;        /**< duplex mode */
  uint8_t                       mdixMode;      /**< mdix mode */
  OPEN_PORT_CABLE_MEDIA_TYPE_t  medium;        /**< active medium for this port copper/fiber */
} OPEN_PORT_INFO_EVENT_t;

/** Information received in notification for event OPEN_NOTIFY_EVENT_LLDP_NEIGHBORS */
typedef struct 
{
  uint32_t                         eventChangeType;  /**< indicates add/remove */
  uint32_t                         ifNum;            /**< index to identify local port */

  /* The unique values to identify an interface remote entry
   * are ifNum, remote index and timestamp. Any details about
   * the entry can be fetched using OpENAPI once we have the indices.
   *
   * For add and update info can be fetched.
   * For delete use srcMac as additional info to clean up
   * Future: if there is usecase we should consider providing
   * chassis id, port id ( lldpRemDataKey_s ).
   */
  uint32_t                   remIndex;     /**< part index to identify remote port data */
  uint32_t                   timestamp;    /**< timestamp also a part index to identify remote port data */
  unsigned char              srcMac[OPEN_MAC_ADDR_LEN]; /**< source MAC address of remote port */
} OPEN_LLDP_NEIGHBOR_EVENT_t;

/** enum values to indicate the change on POE port */
typedef enum
{
  OPEN_POE_PORT_ON              = 1,  /**<  port is turned on */
  OPEN_POE_PORT_OFF             = 2,  /**<  port is turned off */
  OPEN_POE_PORT_ADMIN_DISABLE   = 3,  /**<  port admin state is disable */
  OPEN_POE_PORT_ADMIN_ENABLE    = 4   /**<  port admin state is enable */
} OPEN_POE_TRAP_PORT_CHANGE_t;

/** Information received in notification for event OPEN_NOTIFY_EVENT_POE_STATUS */
typedef struct 
{
  uint32_t                      ifNum;             /**< Index to identify port uniquely */
  OPEN_POE_TRAP_PORT_CHANGE_t   state;             /**< Current port state */
  OPEN_POE_DETECTION_t          detectionStatus;   /**< Detection status of power device connected on the port */
  OPEN_POE_PORT_FAULT_STATUS_t  status;            /**< Fault code on the POE port */
  uint32_t                      powerLimit;        /**< Allowed power on the port */
  uint32_t                      maxPower;          /**< Maximum power possible to deliver on the port */
  OPEN_POE_PORT_POWER_CLASS_t   pdClass;           /**< Class of the connected power device */
  OPEN_POE_TYPE_t               pdType;            /**< Device type (poe/pos+/upoe/BT) */
} OPEN_POE_PORT_STATUS_EVENT_t;

/** Information received in notification for event OPEN_NOTIFY_EVENT_FDB_ENTRY */
typedef struct 
{
  uint8_t                    eventChangeType;         /**< Create or delete */
  unsigned char              mac[OPEN_MAC_ADDR_LEN];  /**< MAC address of learned/aged out station */
  uint32_t                   vlanId;                  /**< VLAN ID on which the station is learned */
  OPEN_FDB_ADDR_FLAG_t       entryType;               /**< Flag indicate the learnt mode */
  uint32_t                   ifNum;                   /**< Port number on which the station is learned */
} OPEN_FDB_EVENT_t;

/** Information received in notification for event OPEN_NOTIFY_EVENT_ARP */
typedef struct 
{
  OPEN_ARP_ENTRY_t    arpEntry;         /**< Arp entry information */
  uint8_t             eventChangeType;  /**< Create, delete or update */
} OPEN_ARP_EVENT_t;

/** Information received in notification for event OPEN_NOTIFY_EVENT_IP_INFO */
typedef struct 
{
  OPEN_MGMT_PORT_TYPE_t mgmtPortType;                     /**< Management port type */
  uint32_t intIfNum;                                      /**< Internal interface number if port type is not service or network */
  uint32_t ipAddr;                                        /**< Host IP address */
  uint32_t netMask;                                       /**< Host network mask */
  uint32_t gateway;                                       /**< Gateway */
  uint32_t dnsServerIpAddr[OPEN_DNS_NAME_SERVER_ENTRIES]; /**< IP addres of DNS servers */
  unsigned char hostName[OPEN_HOSTNAME_MAX_LEN + 1];      /**< Hostname */
  uint8_t  eventChangeType;                               /**< Create, delete or update */
} OPEN_IP_INFO_EVENT_t;

/** Information received in notification for event OPEN_NOTIFY_EVENT_IPV4_ROUTE */
typedef struct 
{
  uint8_t        eventChangeType;  /**< Create, delete or update */
  uint32_t       routeDest;        /**< IP address of the route destination */
  uint32_t       routeMask;        /**< Route  mask */
  uint16_t       vrfId;            /**< VRF ID */
  uint32_t       routeProto;       /**< Value of OPEN_RTO_PROTOCOL_INDICES_t (refer to IP route table). */
  uint32_t       routePref;        /**< Route preference (administrative distance). [0 to 255]. */
} OPEN_IPV4_ROUTE_EVENT_t;

/** The notification event encapsulating all event types */
typedef struct 
{
  uint32_t            eventSeqNum;      /**< Sequence number for the event identified by eventType */
  uint32_t            globalSeqNum;     /**< Sequence number of all the events delievered so far */
  OPEN_NOTIFY_EVENT_t eventType;        /**< Event type to interpret the eventData */
  uint32_t            eventCount;       /**< Number of events in this notification packed in eventData */
  OPEN_BOOL_t         eventMissFlag;    /**< Flag indicating one or more events of this eventType are missed */
  /* buffdesc should be last field in this structure */
  open_buffdesc       eventData;        /**< Buffer with event information for eventCount number of events of eventType */
} OPEN_DEVICE_EVENT_t;

/** Configuration for each event to control the frequency of notifications */
typedef struct
{
  OPEN_NOTIFY_EVENT_t eventType;   /**< Type of the event */
  uint32_t            timeout;     /**< Time interval between two notifications */
  uint32_t            eventCount;  /**< Number of events before triggering Notification */
} openEventNotifyConfig_t;


/*****************************************************************************
 * @purpose   Register a client for receiving OpEN Event Notifications.
 *
 * @param     client_handle      @b{(input)}  client handle from OpEN client registration API
 * @param     clientDescription  @b{(input)}  Description of the registering client
 *
 * @returns   OPEN_E_NONE        On success.
 * @returns   OPEN_E_FAIL        On failure.
 * @returns   OPEN_E_PARAM       Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.28
 *
 * @end
 *****************************************************************************/
open_error_t openapiEventNotifyClientRegister (openapiClientHandle_t *client_handle,
                                               open_buffdesc *clientDescription);

/*****************************************************************************
 * @purpose   Deregister a client from receiving OpEN Event Notifications.
 *
 * @param     client_handle  @b{(input)}  client handle from OpEN client registration API
 *
 * @returns   OPEN_E_NONE    On success.
 * @returns   OPEN_E_FAIL    On failure.
 * @returns   OPEN_E_PARAM   Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.28
 *
 * @end
 *****************************************************************************/
open_error_t openapiEventNotifyClientDeregister (openapiClientHandle_t *client_handle);

/*****************************************************************************
 * @purpose   Register for receiving notifications of specific events.
 *
 * @param     client_handle      @b{(input)}  client handle from OpEN client registration API
 * @param     numOfNotifyEvents  @b{(input)}  Number of event types to be registered.
 * @param     eventNotifyCfg     @b{(input)}  Configuration specific to events (@ref openEventNotifyConfig_t).
 *
 * @returns   OPEN_E_NONE        On success.
 * @returns   OPEN_E_FAIL        On failure.
 * @returns   OPEN_E_PARAM       Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.28
 *
 * @end
 *****************************************************************************/
open_error_t openapiEventNotifyEventRegister (openapiClientHandle_t *client_handle,
                                              uint32_t numOfNotifyEvents,
                                              open_buffdesc *eventNotifyCfg);

/*****************************************************************************
 * @purpose   Deregister for receiving OpEN Event Notifications for all events.
 *
 * @param     client_handle  @b{(input)}  client handle from OpEN client registration API
 *
 * @returns   OPEN_E_NONE    On success.
 * @returns   OPEN_E_FAIL    On failure.
 * @returns   OPEN_E_PARAM   Invalid argument.
 *
 * @notes     Client handle still remains valid.
 *            Client can re-register for a new set of events.
 *
 * @supported-in-version OpEN API Version: 1.28
 *
 * @end
 *****************************************************************************/
open_error_t openapiEventNotifyEventDeregister (openapiClientHandle_t *client_handle);

/*****************************************************************************
 * @purpose   Initialize the socket to receive notification data,
 *             register a callback function and
 *             wait on socket to receive notification data.
 *
 * @param     client_handle  @b{(input)}  client handle from OpEN client registration API
 * @param     callBack       @b{(input)}  Callback function to be invoked on receiving the notification data.
 *
 * @returns   OPEN_E_NONE    On success.
 * @returns   OPEN_E_FAIL    On failure.
 * @returns   OPEN_E_PARAM   Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.28
 *
 * @end
 *****************************************************************************/
open_error_t openapiEventNotifyDataReceive (openapiClientHandle_t *client_handle,
                                            open_error_t (*callBack) (OPEN_DEVICE_EVENT_t *eventInfo));

#endif /* OPENAPI_EVENT_NOTIFY_H_INCLUDED */
