/**********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_fdb.h
*
* @purpose   FDB Configuration
*
* @component OPEN
*
* @create    06/02/2015
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_FDB_H_INCLUDED
#define OPENAPI_FDB_H_INCLUDED

#include <stdbool.h>
#include "openapi_common.h"

/** Supported VLAN Learning types */
typedef enum
{
  OPEN_IVL_TYPE = 1, /**< Independent VLAN Learning */
  OPEN_SVL_TYPE = 2, /**< Shared VLAN Learning */
} OPEN_VL_TYPE_t;

typedef enum
{
  OPEN_FDB_ADDR_FLAG_STATIC                = 0,  /** Static **/
  OPEN_FDB_ADDR_FLAG_LEARNED               = 1,  /** Learned **/
  OPEN_FDB_ADDR_FLAG_MANAGEMENT            = 2,  /** Management**/
  OPEN_FDB_ADDR_FLAG_GMRP_LEARNED          = 3,  /** GMRP Learned **/
  OPEN_FDB_ADDR_FLAG_SELF                  = 4,  /** Self **/
  OPEN_FDB_ADDR_FLAG_DOT1X_STATIC          = 5,  /** Dot1x Static **/
  OPEN_FDB_ADDR_FLAG_DOT1AG_STATIC         = 6,  /** Dot1ag Static **/
  OPEN_FDB_ADDR_FLAG_L3_MANAGEMENT         = 7,  /** Routing Intf address **/
  OPEN_FDB_ADDR_FLAG_LEARNED_UNCONFIRMED   = 8,  /** Address is learned, but not 
                                                     guaranteed to be in HW 
                                                     (relevant for SW learning). **/
  OPEN_FDB_ADDR_FLAG_FIP_SNOOP_LEARNED     = 9,  /** FIP Snooping Learned **/
  OPEN_FDB_ADDR_FLAG_CAPTIVE_PORTAL_STATIC = 10, /** CP client MAC Addr **/
  OPEN_FDB_ADDR_FLAG_ETH_CFM_STATIC        = 11, /** ethcfm Static **/
  OPEN_FDB_ADDR_FLAG_Y1731_STATIC          = 12  /** Y.1731 Static **/
} OPEN_FDB_ADDR_FLAG_t;

/*********************************************************************
* @purpose  Get the next entry from the forwarding database.
*
* @param    client_handle       @b{(input)} client handle from registration API
* @param    mac                 @b{(inout)} MAC address of the entry
* @param    vlanId              @b{(input)} VLAN ID of the entry 
* @param    fdbIfNum            @b{(inout)} internal interface number of the
*                                           entry
* @param    entryType           @b{(inout)} entry type
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_NOT_FOUND    Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiFdbEntryNextGet(openapiClientHandle_t *client_handle,
                                    open_buffdesc *mac, uint32_t *vlanId,
                                    uint32_t *fdbIfNum,
                                    OPEN_FDB_ADDR_FLAG_t *entryType);

/*********************************************************************
* @purpose  Verify that a forwarding database entry with the specified ID exists.
*
* @param    client_handle       @b{(input)} client handle from registration API
* @param    fdbId               @b{(input)} FDB ID to be checked
* 
* @returns  OPEN_E_NONE         FDB exists.
* @returns  OPEN_E_NOT_FOUND    FDB not exists.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiFdbIdGet(openapiClientHandle_t *client_handle,
                             uint32_t fdbId);

/*********************************************************************
* @purpose  Get the aging timeout value for a forwarding database.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    fdbId               @b{(input)}  FDB ID
* @param    timeOut             @b{(output)} aging timeout value
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiFdbAddrAgingTimeoutGet(openapiClientHandle_t *client_handle,
                                           uint32_t fdbId, uint32_t *timeOut);

/*********************************************************************
* @purpose  Set the aging timeout value for a forwarding database.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    fdbId               @b{(input)}  FDB ID
* @param    timeOut             @b{(input)}  aging timeout value in seconds.
* @param    -                   Minimum value @ref OPENAPI_FDB_MIN_TIMEOUT_ID
* @param    -                   Maximum value @ref OPENAPI_FDB_MAX_TIMEOUT_ID
* 
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiFdbAddrAgingTimeoutSet(openapiClientHandle_t *client_handle,
                                           uint32_t fdbId, uint32_t timeOut);

/*********************************************************************
* @purpose  Get the maximum number of entries that the forwarding database 
*           table can hold.
*
* @param    client_handle       @b{(input)} client handle from registration API
* @param    maxEntries          @b((output)) number of total entries
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiFdbMaxTableEntriesGet(openapiClientHandle_t *client_handle,
                                          uint32_t *maxEntries);

/*********************************************************************
* @purpose  Get the current number of entries in forwarding database.
*
* @param    client_handle       @b{(input)} client handle from registration API
* @param    entriesNum          @b((output)) current number of entries
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiActiveAddrEntriesGet(openapiClientHandle_t *client_handle,
                                         uint32_t *entriesNum);

/*********************************************************************
* @purpose  Get the forwarding database entries counter greatest value.
*
* @param    client_handle       @b{(input)} client handle from registration API
* @param    entriesNum          @b((output)) greatest number of entries
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiFdbMaxCountGet(openapiClientHandle_t *client_handle,
                                   uint32_t *entriesNum);

/*********************************************************************
* @purpose  Reset the greatest number of entries in the forwarding database.
* 
* @param    client_handle       @b{(input)} client handle from registration API
* 
* @returns  OPEN_E_NONE         Reset successful.
* @returns  OPEN_E_FAIL         Reset failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiMaxCountReset(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Get count of dynamic entries in the L2 forwarding database.
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    entries             @b{{output}} number of dynamic entries
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiFdbDynamicEntriesGet(openapiClientHandle_t *client_handle,
                                         uint32_t *entries);

/*********************************************************************
* @purpose  Get count of internally added entries in the L2 forwarding database.
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    entries             @b{{output}} number of internally added entries
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiFdbInternalEntriesGet(openapiClientHandle_t *client_handle,
                                          uint32_t *entries);

/*********************************************************************
* @purpose  Determine if the specified MAC address is a member
*           of the restricted set.
* 
* @param    client_handle       @b{(input)} client handle from registration API
* @param    macaddr             @b((input)) MAC Address of the filter
* 
* @returns  OPEN_E_NONE         MAC is not restricted.
* @returns  OPEN_E_DISABLED     MAC is restricted.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiIsRestrFilterMacGet(openapiClientHandle_t *client_handle,
                                        open_buffdesc *macaddr);

/*********************************************************************
* @purpose  Delete a entry from the forwarding database.
* 
* @param    client_handle       @b{(input)} client handle from registration API
* @param    mac                 @b{(input)} MAC address of the entry
* @param    vlanId              @b{(input)} VLAN ID of the entry 
* @param    intIfNum            @b{(input)} internal interface number of the
*                                           entry
* 
* @returns  OPEN_E_NONE         FDB entry is deleted.
* @returns  OPEN_E_FAIL         FDB entry is not deleted.
* @returns  OPEN_E_INTERNAL     Other internal error.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiFdbEntryDelete(openapiClientHandle_t *client_handle,
                                   open_buffdesc *mac, uint32_t vlanId,
                                   uint32_t intIfNum);

/*********************************************************************
* @purpose  Get the type of supported VLAN learning.
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    type                @b{(output)} pointer to the type
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiFdbTypeOfVLGet(openapiClientHandle_t *client_handle,
                                   OPEN_VL_TYPE_t *type);

/*********************************************************************
* @purpose  Get the next forwarding database ID starting from a certain index
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    startFrom           @b{(input)}  index to start search from
* @param    fdbId               @b{(output)} pointer to the return value
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiFdbIdNextGet(openapiClientHandle_t *client_handle,
                                 uint32_t startFrom, uint32_t *fdbId);

/*********************************************************************
* @purpose  Get the number of failed forwarding database insert attempts due to
*           failure to allocate space.
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    badAdds             @b{(output)} pointer to the number of failed FDB
*                                            inserts to be returned
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiFdbBadAddsGet(openapiClientHandle_t *client_handle,
                                  uint32_t *badAdds);

/*********************************************************************
* @purpose  Get the max limit of MAC entries determined by hardware
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    maxMacHwLimit       @b{(output)} Max MAC limit set in hardware
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiFdbMacMaxHwLimitGet(openapiClientHandle_t *client_handle,
                                        uint32_t *maxMacHwLimit);

/*********************************************************************
* @purpose  Get the count of dynamic entries in the L2 forwarding database for 
*           a specific interface
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum            @b((input))  internal interface number
* @param    entries             @b{{output}} number of entries
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiFdbIntfDynamicEntriesGet(openapiClientHandle_t *client_handle,
                                             uint32_t intIfNum, uint32_t *entries);

/*********************************************************************
* @purpose  Get the count of dynamic entries in the L2 forwarding database for
*           a specific VLAN
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    vlanId              @b((input))  VLAN identifier
* @param    entries             @b{{output}} number of entries
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiFdbVlanDynamicEntriesGet(openapiClientHandle_t *client_handle,
                                             uint16_t vlanId, uint32_t *entries);

/*********************************************************************
* @purpose  Get the count of internal entries in the L2 forwarding database for
*           a specific interface
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum            @b((input))  internal interface number
* @param    entries             @b{{output}} number of entries
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiFdbIntfInternalEntriesGet(openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum, uint32_t *entries);

/*********************************************************************
* @purpose  Get the count of internal entries in the L2 forwarding database for 
*           a specific VLAN
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    vlanId              @b((input))  VLAN identifier
* @param    entries             @b{{output}} number of entries
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiFdbVlanInternalEntriesGet(openapiClientHandle_t *client_handle,
                                              uint16_t vlanId, uint32_t *entries);

/*********************************************************************
* @purpose  Flush all entries in forwarding database learned on this interface.
* 
* @param    client_handle       @b{(input)} client handle from registration API
* @param    intIfNum            @b{(input)} internal interface number of the  entry
* 
* @returns  OPEN_E_NONE         All FDB entries are flushed.
* @returns  OPEN_E_FAIL         Failed to flush FDB entries.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiFdbFlushByPort(openapiClientHandle_t *client_handle,
                                   uint32_t intIfNum);

/*********************************************************************
* @purpose  Flush all entries in forwarding database learned on the specified VLAN.
* 
* @param    client_handle       @b{(input)} client handle from registration API
* @param    vlanId              @b{(input)} VLAN number
* 
* @returns  OPEN_E_NONE         All FDB entries are flushed.
* @returns  OPEN_E_FAIL         Failed to flush FDB entries.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiFdbFlushByVlan(openapiClientHandle_t *client_handle,
                                   uint32_t vlanId);

/*********************************************************************
* @purpose  Flush all entries in forwarding database learned on the specified MAC.
* 
* @param    client_handle       @b{(input)} client handle from registration API
* @param    mac                 @b{(input)} MAC address
* 
* @returns  OPEN_E_NONE         All FDB entries are flushed.
* @returns  OPEN_E_FAIL         Failed to flush FDB entries.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiFdbFlushByMac(openapiClientHandle_t *client_handle,
                                  open_buffdesc *mac);

/*********************************************************************
* @purpose  Flush all learned entries from the L2 forwarding database.
* 
* @param    client_handle       @b{(input)} client handle from registration API
* 
* @returns  OPEN_E_NONE         All FDB entries are flushed.
* @returns  OPEN_E_FAIL         Failed to flush FDB entries.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiFdbFlush(openapiClientHandle_t *client_handle);


#endif /* OPENAPI_FDB_H_INCLUDED */
