/**********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_flow_control.h
*
* @purpose   This code implements the OpEN Flow Control API
*
* @component OPEN
*
* @create    06/08/2023
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_FLOWCONTROL_H_INCLUDED
#define OPENAPI_FLOWCONTROL_H_INCLUDED

#include "openapi_common.h" 

typedef enum
{
  OPEN_FLOWCTRL_OFF = 0,  /**< Disable    */
  OPEN_FLOWCTRL_SYMM,     /**< Symmetric  */
  OPEN_FLOWCTRL_ASYMM,    /**< Asymmetric */
} OPEN_FLOWCONTROL_t;

typedef enum
{
  OPEN_FLOWCTRL_INACTIVE = 0, /**< Inactive  */
  OPEN_FLOWCTRL_ACTIVE,       /**< Active    */
} OPEN_FLOWCONTROL_OPER_t;

/*****************************************************************************
 * @purpose   Get the Flow Control mode of the system.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     mode           @b{(output)} Flow control mode
 *
 * @returns   OPEN_E_UNAVAIL          Parameter/feature is not supported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiFlowControlModeGet(openapiClientHandle_t *client_handle, 
                                       OPEN_FLOWCONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the Flow Control mode of the system.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     mode           @b{(input)}  Flow control mode
 *
 * @returns   OPEN_E_UNAVAIL          Parameter/feature is not supported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiFlowControlModeSet(openapiClientHandle_t *client_handle, 
                                       OPEN_FLOWCONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the Flow Control mode of the interafce.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     mode           @b{(output)} Flow control mode
 *
 * @returns   OPEN_E_UNAVAIL          Parameter/feature is not supported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiFlowControlIfModeGet(openapiClientHandle_t *client_handle, 
                                         uint32_t ifNum, OPEN_FLOWCONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the Flow Control mode of the interafce.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     mode           @b{(input)}  Flow control mode
 *
 * @returns   OPEN_E_UNAVAIL          Parameter/feature is not supported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiFlowControlIfModeSet(openapiClientHandle_t *client_handle, 
                                         uint32_t ifNum, OPEN_FLOWCONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the Flow Control operational mode of the interafce.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     mode           @b{(output)} Flow control operational mode
 *
 * @returns   OPEN_E_UNAVAIL          Parameter/feature is not supported.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiFlowControlIfOperModeGet(openapiClientHandle_t *client_handle, 
                                             uint32_t ifNum, 
                                             OPEN_FLOWCONTROL_OPER_t *mode);

/*****************************************************************************
 * @purpose   Get the pause frames receive count of the interafce.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     rxCount        @b{(output)} Pause frames receive count
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiFlowControlIfPauseFramesRxGet(openapiClientHandle_t *client_handle, 
                                                  uint32_t ifNum, uint32_t *rxCount);

/*****************************************************************************
 * @purpose   Get the pause frames transmit count of the interafce.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     txCount        @b{(output)} Pause frames transmit count
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiFlowControlIfPauseFramesTxGet(openapiClientHandle_t *client_handle, 
                                                  uint32_t ifNum, uint32_t *txCount);


#endif /* OPENAPI_FLOWCONTROL_H_INCLUDED */


