/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_green_ethernet.h
*
* @purpose   Green Ethernet API
*
* @component OpEN
*
* @create    05/31/2023
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_GREEN_ETHERNET_H
#define OPENAPI_GREEN_ETHERNET_H

typedef enum
{
  OPEN_GREENETHR_ADMIN_DOWN,      /**< Admin down */
  OPEN_GREENETHR_FIBER_PORT,      /**< Fiber port */
  OPEN_GREENETHR_LINK_UP,         /**< Link up */
  OPEN_GREENETHR_LINK_DOWN,       /**< Link down */
  OPEN_GREENETHR_LONG_CABLE,      /**< Long cable */
  OPEN_GREENETHR_SHORT_CABLE,     /**< Short cable */
  OPEN_GREENETHR_UNKWN_CABLE_LEN, /**< Unknown cable length */
  OPEN_GREENETHR_NO_ENERGY,       /**< No energy */
  OPEN_GREENETHR_FORCED,          /**< Forced */
  OPEN_GREENETHR_NOT1GIG,         /**< Not 1 gig */
  OPEN_GREENETHR_NOT_ATTACHED,    /**< Not attached */
  OPEN_GREENETHR_INVALID_VAL,     /**< Invalid val */
} OPEN_GREEN_ETH_OPER_STATUS_t;


/*********************************************************************
* @purpose  Get the first interface that is valid for green ethernet
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(output)} internal interface number
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrValidIntfFirstGet(openapiClientHandle_t *client_handle,
                                               uint32_t *intIfNum);

/*********************************************************************
* @purpose  Get the next interface that is valid for green ethernet
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    prevIfNum      @b{(input)}  previous interface number
* @param    intIfNum       @b{(output)} internal interface number
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrValidIntfNextGet(openapiClientHandle_t *client_handle,
                                              uint32_t prevIfNum,
                                              uint32_t *intIfNum);

/*********************************************************************
* @purpose  Determine if the interface is valid for green ethernet operation
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    valid          @b{(output)} valid or not
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrIsValidIntfType(openapiClientHandle_t *client_handle,
                                             uint32_t intIfNum,
                                             OPEN_BOOL_t *valid);

/*********************************************************************
* @purpose   Determine if the interface is EEE capable
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    valid          @b{(output)} EEE capable or not
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrIsIntfEEECapable(openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum,
                                              OPEN_BOOL_t *valid);

/*********************************************************************
* @purpose  Set the energy-detect mode
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    mode           @b{(input)} energy detect mode
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEnergyDetectModeSet(openapiClientHandle_t *client_handle,
                                                 uint32_t intIfNum,
                                                 OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose  Set the short-reach auto mode
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    mode           @b{(input)} short reach auto mode
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrShrtRchAutoModeSet(openapiClientHandle_t *client_handle,
                                                uint32_t intIfNum,
                                                OPEN_BOOL_t mode);

/*********************************************************************
* @purpose  Set the short-reach into force mode
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    mode           @b{(input)} force mode
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrShrtRchForceModeSet(openapiClientHandle_t *client_handle,
                                                 uint32_t intIfNum,
                                                 OPEN_BOOL_t mode);

/*********************************************************************
* @purpose  Get the energy detect admin mode
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    mode           @b{(output)} admin mode
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEnergyDetectAdminModeGet(openapiClientHandle_t *client_handle,
                                                      uint32_t intIfNum,
                                                      OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose  Get the short-reach auto admin mode
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    mode           @b{(output)} admin mode
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrShrtRchAutoAdminModeGet(openapiClientHandle_t *client_handle,
                                                     uint32_t intIfNum,
                                                     OPEN_BOOL_t *mode);

/*********************************************************************
* @purpose  Get the short-reach force admin mode
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    mode           @b{(output)} admin mode
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrShrtRchForceAdminModeGet(openapiClientHandle_t *client_handle,
                                                      uint32_t intIfNum,
                                                      OPEN_BOOL_t *mode);

/*********************************************************************
* @purpose  Get the energy-detect operational status and reason
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    oprStatus      @b{(output)} operational status
* @param    reason         @b{(output)} reason
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEnergyDetectOprStatusGet(openapiClientHandle_t *client_handle,
                                                      uint32_t intIfNum,
                                                      OPEN_BOOL_t *oprStatus,
                                                      OPEN_GREEN_ETH_OPER_STATUS_t *reason);

/*********************************************************************
* @purpose  Get the short-reach operational status and reason
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    oprStatus      @b{(output)} operational status
* @param    reason         @b{(output)} reason
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrShrtRchOprStatusGet(openapiClientHandle_t *client_handle,
                                                 uint32_t intIfNum,
                                                 OPEN_BOOL_t *oprStatus,
                                                 OPEN_GREEN_ETH_OPER_STATUS_t *reason);

/*********************************************************************
* @purpose  Get the energy-detect operational status and reason
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    oprStatus      @b{(output)} operational status
* @param    reasonStr      @b{(output)} reason string
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEnergyDetectOprStatusStrGet(openapiClientHandle_t *client_handle,
                                                         uint32_t intIfNum,
                                                         OPEN_BOOL_t *oprStatus,
                                                         open_buffdesc *reasonStr);

/*********************************************************************
* @purpose  Get the short-reach operational status and reason
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    oprStatus      @b{(output)} operational status
* @param    bufd           @b{(output)} reason string
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrShrtRchOprStatusStrGet(openapiClientHandle_t *client_handle,
                                                    uint32_t intIfNum,
                                                    OPEN_BOOL_t *oprStatus,
                                                    open_buffdesc *bufd);

/*********************************************************************
* @purpose  Get the global energy detect administrative mode
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    adminMode      @b{(output)} global energy detect admin mode
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrGlobalEnergyDetectModeGet(openapiClientHandle_t *client_handle,
                                                       OPEN_CONTROL_t *adminMode);

/*********************************************************************
* @purpose  Set the global energy detect administrative mode
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    adminMode      @b{(input)} global energy detect admin mode
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrGlobalEnergyDetectModeSet(openapiClientHandle_t *client_handle,
                                                       OPEN_CONTROL_t adminMode);

/*********************************************************************
* @purpose  Get the global auto short reach administrative mode
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    adminMode      @b{(output)} global auto short reach admin mode
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrGlobalShortReachModeGet(openapiClientHandle_t *client_handle,
                                                     OPEN_CONTROL_t *adminMode);

/*********************************************************************
* @purpose  Set the global auto short reach administrative mode
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    adminMode      @b{(input)} global auto short reach admin mode
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrGlobalShortReachModeSet(openapiClientHandle_t *client_handle,
                                                     OPEN_CONTROL_t adminMode);
#endif /* OPENAPI_GREEN_ETHERNET_H */
