
/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_green_ethernet_eee_api.h
*
* @purpose   Green Mode EEE API
*
* @component OpEN
*
* @create    05/31/2023
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_GREEN_ETHERNET_EEE_API_H
#define OPENAPI_GREEN_ETHERNET_EEE_API_H

/*********************************************************************
* @purpose  Determine if the interface is valid
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    valid          @b{(output)} valid interface
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEeeIsValidIntf(openapiClientHandle_t *client_handle,
                                            uint32_t intIfNum, OPEN_BOOL_t *valid);

/*********************************************************************
* @purpose  Get the first interface that is valid for green Ethernet EEE
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(output)} internal interface number
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEeeValidIntfFirstGet(openapiClientHandle_t *client_handle,
                                                  uint32_t *intIfNum);

/*********************************************************************
* @purpose  Get the next interface that is valid for Green Ethernet EEE
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    prevIfNum      @b{(input)}  previous interface number
* @param    intIfNum       @b{(output)} internal interface number
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEeeValidIntfNextGet(openapiClientHandle_t *client_handle,
                                                 uint32_t prevIfNum, uint32_t *intIfNum);

/**************************************************************************
*
* @purpose  Get EEE admin mode
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    adminMode      @b{(output)} admin mode
*
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*************************************************************************/
open_error_t openapiGreenEthrEeeAdminModeGet(openapiClientHandle_t *client_handle,
                                             uint32_t intIfNum, OPEN_CONTROL_t *adminMode);

/**************************************************************************
*
* @purpose  Get EEE transmit Idle Time
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    eeeTxIdleTime  @b{(output)} EEE Transmit Idle Time
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*************************************************************************/
open_error_t openapiGreenEthrEeeTxIdleTimeGet(openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum, uint32_t *eeeTxIdleTime);

/**************************************************************************
*
* @purpose  Get EEE transmit Wake Time
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    *eeeTxWakeTime @b{(output)} EEE Transmit Wake Time
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*************************************************************************/
open_error_t openapiGreenEthrEeeTxWakeTimeGet(openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum, uint32_t *eeeTxWakeTime);

/**************************************************************************
*
* @purpose  Set EEE administrative mode
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    intIfNum       @b{(input)} internal interface number
* @param    adminMode      @b{(input)} admin mode
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*************************************************************************/
open_error_t openapiGreenEthrEeeAdminModeSet(openapiClientHandle_t *client_handle,
                                             uint32_t intIfNum, OPEN_CONTROL_t adminMode);

/**************************************************************************
*
* @purpose  Set EEE transmit idle time
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    intIfNum       @b{(input)} internal interface number
* @param    eeeTxIdleTime  @b{(input)} EEE Transmit Idle Time
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*************************************************************************/
open_error_t openapiGreenEthrEeeTxIdleTimeSet(openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum, uint32_t eeeTxIdleTime);

/**************************************************************************
*
* @purpose  Set EEE transmit wake time
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    intIfNum       @b{(input)} internal interface number
* @param    eeeTxWakeTime  @b{(input)} EEE Transmit Wake Time
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*************************************************************************/
open_error_t openapiGreenEthrEeeTxWakeTimeSet(openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum, uint32_t eeeTxWakeTime);

/**************************************************************************
*
* @purpose  Clear EEE statistics counters
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    intIfNum       @b{(input)} internal interface number
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*************************************************************************/
open_error_t openapiGreenEthrEeeStatisticsClear(openapiClientHandle_t *client_handle,
                                                uint32_t intIfNum);

/**************************************************************************
*
* @purpose  Get EEE transmit LPI event count
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    *count         @b{(output)} EEE Transmit LPI event count
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*************************************************************************/
open_error_t openapiGreenEthrEeeTxEventsGet(openapiClientHandle_t *client_handle,
                                            uint32_t intIfNum, uint32_t *count);

/**************************************************************************
*
* @purpose  Get EEE transmit LPI duration (counts in 10usec increments)
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    high           @b{(output)} high word of duration in uSec
* @param    low            @b{(output)} low word of duration in uSec
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*************************************************************************/
open_error_t openapiGreenEthrEeeTxDurationGet(openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum, uint32_t *high,
                                              uint32_t *low);

/**************************************************************************
*
* @purpose  Get EEE receive LPI event count
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    *count         @b{(output)} EEE Recieve event count
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*************************************************************************/
open_error_t openapiGreenEthrEeeRxEventsGet(openapiClientHandle_t *client_handle,
                                            uint32_t intIfNum, uint32_t *count);

/**************************************************************************
*
* @purpose  Get EEE receive LPI duration (counts in 10usec increments)
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    high           @b{(output)} high word of duration in uSec
* @param    low            @b{(output)} low word of duration in uSec
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*************************************************************************/
open_error_t openapiGreenEthrEeeRxDurationGet(openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum, uint32_t *high,
                                              uint32_t *low);

/**************************************************************************
*
* @purpose  Get EEE transmit Tw_sys (uSec)
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    txTwsys        @b{(output)} transmit twsys
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*************************************************************************/
open_error_t openapiGreenEthrEeeTxTwsysGet(openapiClientHandle_t *client_handle,
                                           uint32_t intIfNum, uint32_t *txTwsys);

/**************************************************************************
*
* @purpose  Get EEE receive Tw_sys (uSec)
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    rxTwsys        @b{(output)} receive twsys
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*************************************************************************/
open_error_t openapiGreenEthrEeeRxTwsysGet(openapiClientHandle_t *client_handle,
                                           uint32_t intIfNum, uint32_t *rxTwsys);

/*********************************************************************
*
* @purpose  Get the port's EEE fallback receive Twsys
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    fbTwsys        @b{(output)} fallback receive Twsys
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEeeFbTwsysGet(openapiClientHandle_t *client_handle,
                                           uint32_t intIfNum, uint32_t *fbTwsys);

/*********************************************************************
*
* @purpose  Get the port's EEE echo transmit Twsys
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    echoTxTwsys    @b{(output)} echo transmit Twsys
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEeeEchoTxTwsysGet(openapiClientHandle_t *client_handle,
                                               uint32_t intIfNum, uint32_t *echoTxTwsys);

/*********************************************************************
*
* @purpose  Get the port's EEE echo receive Twsys
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param   echoRxTwsys     @b{(output)} echo receive Twsys
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEeeEchoRxTwsysGet(openapiClientHandle_t *client_handle,
                                               uint32_t intIfNum, uint32_t *echoRxTwsys);

/*********************************************************************
*
* @purpose  Get the remote port's EEE transmit Twsys
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    intIfNum      @b{(input)}  internal interface number
* @param    remIndex      @b{(input)}  index associated with this remote entry
* @param    timestamp     @b{(input)}  time stamp associated with this entry
* @param    remTxTwsys    @b{(output)} remote transmit Twsys
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEeeRemTxTwsysGet(openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum, uint32_t remIndex,
                                              uint32_t timestamp, uint32_t *remTxTwsys);

/*********************************************************************
*
* @purpose  Get the remote port's EEE transmit Twsys echo
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    remIndex       @b{(input)}  Index associated with this remote entry
* @param    timestamp      @b{(input)}  time stamp associated with this entry
* @param    remTxTwsysEcho @b{(output)} remote transmit Twsys echo
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEeeRemTxTwsysEchoGet(openapiClientHandle_t *client_handle,
                                                  uint32_t intIfNum, uint32_t remIndex,
                                                  uint32_t timestamp, uint32_t *remTxTwsysEcho);

/*********************************************************************
*
* @purpose  Get the remote port's EEE receive Twsys
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal Interface Number
* @param    remIndex       @b{(input)}  index associated with this remote entry
* @param    timestamp      @b{(input)}  time stamp associated with this entry
* @param    remRxTwsys     @b{(output)} remote receive Twsys
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEeeRemRxTwsysGet(openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum, uint32_t remIndex,
                                              uint32_t timestamp, uint32_t *remRxTwsys);

/*********************************************************************
*
* @purpose  Get the remote port's EEE receive Twsys echo
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    remIndex       @b{(input)}  index associated with this remote entry
* @param    timestamp      @b{(input)}  time stamp associated with this entry
* @param    remRxTwsysEcho @b{(output)} remote receive Twsys echo
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEeeRemRxTwsysEchoGet(openapiClientHandle_t *client_handle,
                                                  uint32_t intIfNum, uint32_t remIndex,
                                                  uint32_t timestamp, uint32_t *remRxTwsysEcho);

/*********************************************************************
*
* @purpose  Get the remote port's EEE fallback Twsys
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    remIndex       @b{(input)}  index associated with this remote entry
* @param    timestamp      @b{(input)}  time stamp associated with this entry
* @param    remFbTwsys     @b{(output)} remote fallback Twsys
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEeeRemFbTwsysGet(openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum, uint32_t remIndex,
                                              uint32_t timestamp, uint32_t *remFbTwsys);

/*********************************************************************
*
* @purpose  Get the transmit DLL enable status
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    enable         @b{(output)} OPEN_TRUE if DLL is enabled
*                                       OPEN_FALSE otherwise
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEeeTxDllEnableGet(openapiClientHandle_t *client_handle,
                                               uint32_t intIfNum, OPEN_BOOL_t *enable);

/*********************************************************************
*
* @purpose  Get the transmit DLL ready status
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    ready          @b{(output)} OPEN_TRUE if DLL is enabled
*                                       OPEN_FALSE otherwise
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEeeTxDllReadyGet(openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum, OPEN_BOOL_t *ready);

/*********************************************************************
*
* @purpose  Get the receive DLL enable status
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    enable         @b{(output)} OPEN_TRUE if DLL is enabled
*                                       OPEN_FALSE otherwise
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEeeRxDllEnableGet(openapiClientHandle_t *client_handle,
                                               uint32_t intIfNum, OPEN_BOOL_t *enable);

/*********************************************************************
*
* @purpose  Get the receive DLL ready status
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    intIfNum       @b{(input)}   internal interface number
* @param    ready          @b{(output)}  OPEN_TRUE if DLL is enabled
*                                        OPEN_FALSE otherwise
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEeeRxDllReadyGet(openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum, OPEN_BOOL_t *ready);

/*********************************************************************
* @purpose  Determine time of last stats reset
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    value          @b{(output)} time in seconds
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEeeTimeSinceLastStatsResetGet(openapiClientHandle_t *client_handle,
                                                           uint32_t intIfNum, uint64_t *value);

/*********************************************************************
* @purpose  Determine time of last stats reset
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    days           @b{(output)} Pointer to days
* @param    hours          @b{(output)} Pointer to hours
* @param    minutes        @b{(output)} Pointer to minutes
* @param    seconds        @b{(output)} Pointer to seconds
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapixGreenEthrEeeTimeSinceLastStatsResetGet(openapiClientHandle_t *client_handle,
                                                            uint32_t intIfNum, uint32_t *days,
                                                            uint32_t *hours, uint32_t *minutes,
                                                            uint32_t *seconds);

/**************************************************************************
*
* @purpose  Get percentage of total LPI time spent for all interfaces in stack
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    lpiTotalTime   @b{(output)} LPI total time
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*************************************************************************/
open_error_t openapiGreenEthrEeeTotalLpiTimeGet(openapiClientHandle_t *client_handle,
                                                uint32_t *lpiTotalTime);

/**************************************************************************
*
* @purpose  Get some common green ethernet eee counters for the given local
*           interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    intIfNum        @b{(input)}  internal interface number
* @param    twSysTx         @b{(output)} transmit Twsys
* @param    twSysTxEcho     @b{(output)} echo transmit Twsys
* @param    twSysRx         @b{(output)} receive Twsys
* @param    twSysRxEcho     @b{(output)} echo receive Twsys
* @param    twSysTxFallback @b{(output)} fallback Twsys
* @param    txDllEnabled    @b{(output)} transmit dll enable
* @param    txDllReady      @b{(output)} transmit dll ready
* @param    rxDllEnabled    @b{(output)} receive dll enable
* @param    rxDllReady      @b{(output)} receive dll ready
* 
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*************************************************************************/
open_error_t openapiGreenEthrEeeLocalSummaryGet(openapiClientHandle_t *client_handle,
                                                uint32_t intIfNum,
                                                uint32_t *twSysTx,
                                                uint32_t *twSysTxEcho,
                                                uint32_t *twSysRx,
                                                uint32_t *twSysRxEcho,
                                                uint32_t *twSysTxFallback,
                                                uint32_t *txDllEnabled,
                                                uint32_t *txDllReady,
                                                uint32_t *rxDllEnabled,
                                                uint32_t *rxDllReady);

/**************************************************************************
*
* @purpose  Get some common green ethernet eee counters for the given remote
*           interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    intIfNum        @b{(input)}  internal interface number
* @param    twSysTx         @b{(output)} transmit Twsys
* @param    twSysTxEcho     @b{(output)} echo transmit Twsys
* @param    twSysRx         @b{(output)} receive Twsys
* @param    twSysRxEcho     @b{(output)} echo receive Twsys
* @param    twSysTxFallback @b{(output)} fallback Twsys
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*************************************************************************/
open_error_t openapiGreenEthrEeeRemoteSummaryGet(openapiClientHandle_t *client_handle,
                                                 uint32_t intIfNum,
                                                 uint32_t *twSysTx,
                                                 uint32_t *twSysTxEcho,
                                                 uint32_t *twSysRx,
                                                 uint32_t *twSysRxEcho,
                                                 uint32_t *twSysTxFallback);

/*********************************************************************
*
* @purpose  Get the port's green feature list supported
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    unit           @b{(input)}  unit number
* @param    featureString  @b{(output)} feature string
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrFeaturesSupportedGet(openapiClientHandle_t *client_handle,
                                                  uint32_t unit, open_buffdesc *featureString);

/**************************************************************************
*
* @purpose  Get the global EEE administrative mode
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    adminMode      @b{(output)} global EEE admin mode
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*************************************************************************/
open_error_t openapiGreenEthrGlobalEeeAdminModeGet(openapiClientHandle_t *client_handle,
                                                   OPEN_CONTROL_t *adminMode);
 
/**************************************************************************
*
* @purpose  Set the global EEE administrative mode
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    adminMode      @b{(input)} global EEE admin mode
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*************************************************************************/
open_error_t openapiGreenEthrGlobalEeeAdminModeSet(openapiClientHandle_t *client_handle,
                                                   OPEN_CONTROL_t adminMode);

#endif /* OPENAPI_GREEN_ETHERNET_EEE_API_H */

