/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_green_pwr_sav_history.h
*
* @purpose   Green mode power savings history API
*
* @component OpEN
*
* @create    05/31/2023
*
* @end
*
**********************************************************************/
#ifndef _OPENAPI_GREEN_PWR_SAV_HISTORY_H_
#define _OPENAPI_GREEN_PWR_SAV_HISTORY_H_

#include "openapi_common.h"

typedef struct
{
  uint16_t sampleNo;            /**< Sample Index */
  uint32_t sampleTime;          /**< Time stamp at which the sample is collected */
  uint16_t percentLpiTime;      /**< percentage of time spent in LPI mode since last sample is taken */
  uint16_t percentLpiTimeTotal; /**< percentage of time spent in LPI mode since last reset */
  void     *nextEntry;          /**< Pointer to next entry in the linked list */

}openGreenPwrHistoryIntfEntry_t;

/*********************************************************************
* @purpose  Set green mode power saving history sampling interval
*
* @param    client_handle   @b{(input)} client handle from registration API
* @param    sampleInterval  @b{(input)} sampling interval
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEeeLpiHistoryIntfSampleIntervalSet(openapiClientHandle_t *client_handle,
                                                                uint32_t sampleInterval);

/*********************************************************************
* @purpose  Get green mode power saving history sampling interval
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    sampleInterval @b{(output)} sampling interval
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEeeLpiHistoryIntfSampleIntervalGet(openapiClientHandle_t *client_handle,
                                                                uint32_t *sampleInterval);

/*********************************************************************
* @purpose  Set green mode power saving history max samples to keep
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    maxSamples     @b{(input)} max samples
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEeeLpiHistoryIntfMaxSamplesSet(openapiClientHandle_t *client_handle,
                                                            uint32_t maxSamples);

/*********************************************************************
* @purpose  Get green mode power saving history max samples to keep
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    maxSamples     @b{(output)} max samples
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @comments none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiGreenEthrEeeLpiHistoryIntfMaxSamplesGet(openapiClientHandle_t *client_handle,
                                                            uint32_t *maxSamples);

/*********************************************************************
* @purpose  Get first valid entry of green mode power savings history
*           on given interface - latest sample first
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    entry          @b{(output)} first valid history entry on the interface
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @notes    To get the time since the sample was recorded, subtract the entry->sampleTime from
*           the system up time
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapix2GreenEthrPwrSavHistoryIntfFirstEntryGet(openapiClientHandle_t *client_handle,
                                                              uint32_t intIfNum,
                                                              openGreenPwrHistoryIntfEntry_t *entry);

/*********************************************************************
* @purpose  Get next valid entry of green mode power savings history
*           on given interface
*
* @param    client_handle  @b{(input)}        client handle from registration API
* @param    entry          @b{(input/output)} current entry on the interface
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @notes    To get the time since the sample was recorded, subtract the entry->sampleTime from
*           the system up time
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapix2GreenEthrPwrSavHistoryIntfNextEntryGet(openapiClientHandle_t *client_handle,
                                                             openGreenPwrHistoryIntfEntry_t *entry);

/*********************************************************************
* @purpose  Get next entry of green mode power savings history
*           on given interface with current sample number as input - get entries in reverse order
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    curSample      @b{(input)}  current Sample Number on the interface
* @param    entry          @b{(output)} next entry
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
* @returns  OPEN_E_UNAVAIL          Not available.
*
* @notes none
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapix2GreenEthrPwrSavHistoryIntfNextSampleGet(openapiClientHandle_t *client_handle,
                                                              uint32_t intIfNum,
                                                              uint16_t curSample,
                                                              openGreenPwrHistoryIntfEntry_t *entry);

/**************************************************************************
*
* @purpose  Clear green mode power saving history statistics for the given interface
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    intIfNum       @b{(input)} internal interface number
*
* @returns  OPEN_E_NONE             On success.
* @returns  OPEN_E_FAIL             On failure.
* @returns  OPEN_E_PARAM            Invalid argument.
* @returns  OPEN_E_INTERNAL         Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.26
*
* @end
*************************************************************************/
open_error_t openapiGreenEthrPwrSavHistoryIntfStatisticsClear(openapiClientHandle_t *client_handle,
                                                              uint32_t intIfNum);

#endif /* OPENAPI_GREEN_PWR_SAV_HISTORY_H */
