/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename openapi_instru_flow_tracker.h
*
* @purpose FLOW TRACKER API: Provide an interface for configuring 
*                            flow groups in hardware.
*
* @component FLOW TRACKER (FLOW TRACKER API)
*
* @create   07/20/2017
*
* @author   Naveen Kumar Aketi
* @end
*
**********************************************************************/
#ifndef OPENAPI_FT_H_INCLUDED
#define OPENAPI_FT_H_INCLUDED
 
#include <stdbool.h>
#include "openapi_common.h"
#include "openapi_instru_nos.h"

/** Enum to identify packet type */
typedef enum _open_ft_pkt_type_
{
  OPEN_FT_PKT_TYPE_FLW_TEMPLATE       = 1,
  OPEN_FT_PKT_TYPE_FLW_GRP_TEMPLATE,
  OPEN_FT_PKT_TYPE_FLW_GRP_REC
} OPEN_FT_PKT_TYPE_t;

/** IPFIX configuration of flow and flow group record exporting processes */
typedef struct _open_ft_ipfix_config_
{
  unsigned int   periodicity;     /* Template periodicity*/
  unsigned int   rec_periodicity; /* Record periodicity */
  unsigned short src_port;        /* Source port */ 
  unsigned int   obs_dmn;         /* Observation domain */
} OPEN_FT_IPFIX_CONFIG_t;

/* FT config feature params mask */
typedef enum _open_ft_config_feat_param_mask_ 
{
  OPEN_FT_CONFIG_FEAT_ENABLE_MASK               = (1 << 0), /* Mask of enable field in config feature */
  OPEN_FT_CONFIG_FEAT_PARAM_FLW_AGE_TIME_MASK   = (1 << 1), /* Mask of flow aging time param in config feat */ 
} OPEN_FT_CONFIG_FEAT_PARAM_MASK_t;


/** Flow Tracker feature configuration */  
typedef struct _open_ft_configure_feature_
{
  bool enable;
  unsigned int flow_aging_time;            /* Aging time of a Flow */
  OPEN_FT_IPFIX_CONFIG_t flw_exp_ipfix;     /* IPFIX configuration of flow exporting process */
  OPEN_FT_IPFIX_CONFIG_t flw_grp_exp_ipfix; /* IPFIX configuration of flow group exporting process */
} OPEN_FT_CONFIG_t;

/* Flow Tracker feature configuration params */
typedef struct _open_ft_config_feat_param_ 
{
  OPEN_FT_CONFIG_FEAT_PARAM_MASK_t param_mask;
  OPEN_FT_CONFIG_t feature;
} OPEN_FT_CONFIG_FEAT_PARAM_t;

/** Transport type used to communicate with collector */
typedef enum _open_ft_cl_trans_type_
{
  OPEN_FT_CL_TRANS_TYPE_IPv4_UDP = (1<<0),    /* IPv4, UDP */ 
  OPEN_FT_CL_TRANS_TYPE_IPv6_UDP = (1<<1),    /* IPv6, UDP */
  OPEN_FT_CL_TRANS_TYPE_RAW      = (1<<2)     /* RAW       */
} OPEN_FT_CL_TRANS_TYPE_t;

typedef enum _open_ft_cl_config_param_mask_
{
  OPEN_FT_CL_CONFIG_PORT_MASK = (1 << 0),  /* Mask of port param */
} OPEN_FT_CL_CONFIG_PARAM_MASK_t;

/** Flow Tracker Collector Protocol */
typedef enum _open_ft_cl_proto_
{
  OPEN_FT_CL_PROTO_IPFIXv10 = 10
} OPEN_FT_CL_PROTO_t;

#define OPEN_FT_MAX_NAME_LENGTH               128

/** Flow tracker collector config */
typedef struct _open_ft_cl_config
{
  char               name[OPEN_FT_MAX_NAME_LENGTH];
  OPEN_FT_CL_PROTO_t proto;
  open_inet_addr_t   addr;
  unsigned short     port;
  OPEN_FT_CL_CONFIG_PARAM_MASK_t param_mask;
} OPEN_FT_CL_CONFIG_t;

/** Flow tracker collector configuration */
typedef struct _open_ft_cl_info_
{
  bool                    configured;   
  OPEN_FT_CL_TRANS_TYPE_t trans_type;
  OPEN_FT_CL_CONFIG_t     config;
  unsigned int            max_pkt_len;
  uint32_t                  seq_num;
} OPEN_FT_CL_INFO_t;

/** 5-tuple parameters */
typedef struct _open_ft_5_tuple_key_
{
  OPEN_INSTRU_IP_PREFIX_t src_ip;
  OPEN_INSTRU_IP_PREFIX_t dst_ip;
  unsigned int            protocol;
  unsigned short          src_port;
  unsigned short          dst_port;
  unsigned int            tuple_mask;  /* Valid params mask */
} OPEN_FT_5_TUPLE_KEY_t;

#define     OPEN_FT_MASK_BASE_UNIT     unsigned int
#define     OPEN_FT_MASKWID            (8*sizeof(unsigned int)) 

/* (internal) Number of OPEN_FT_MASK_BASE_UNITs needed to contain _max bits */
#define     OPEN_FT_MASK_SIZE(_max)    (((_max) + OPEN_FT_MASKWID - 1) / OPEN_FT_MASKWID)

#define OPEN_FT_PORT_MAX_RANGE_STR_LEN 256
/** Maximum number of ports **/
/** OPEN_ASIC_ICOS_FT_MAX_PORTS may be deprecated in a future release, use OPEN_ASIC_FT_MAX_PORTS */
#define OPEN_ASIC_ICOS_FT_MAX_PORTS   300
#define OPEN_ASIC_FT_MAX_PORTS        OPEN_ASIC_ICOS_FT_MAX_PORTS

/* Interface storage */
typedef struct
{
  OPEN_FT_MASK_BASE_UNIT value[OPEN_FT_MASK_SIZE(OPEN_ASIC_FT_MAX_PORTS)];
  uint8_t                port_list_str[OPEN_FT_PORT_MAX_RANGE_STR_LEN];
} OPEN_FT_PORT_MASK_t;

/** Union of flow group monitoring params */
typedef union _open_ft_flw_grp_mon_params_
{
  OPEN_FT_5_TUPLE_KEY_t five_tuple;
   /* List of egress ports*/
  OPEN_FT_PORT_MASK_t     egress_port_mask;
} OPEN_FT_FLW_GRP_MON_PARAMS_t;

/* Flow group actions mask */ 
typedef enum _open_ft_flw_grp_action_mask_
{
  OPEN_FT_FLW_GRP_REP_FLWS     = (1 << 0),  /* Report flows */  
  OPEN_FT_FLW_GRP_REP_FLW_GRPS = (1 << 1),  /* Report flow groups */  
} OPEN_FT_FLW_GRP_ACTION_MASK_t;

/** Max collector for each flow group */
#define OPEN_FT_MAX_COLLECTOR_LIST_PER_FG      1

/** Flow Tracker  flow group monitor type */
typedef enum _open_ft_flw_grp_mon_type_
{
  OPEN_FT_FLW_GRP_MON_TYPE_FIVE_TUPLE  = (1 << 0),  /* Monitoring based on five tuple   */
  OPEN_FT_FLW_GRP_MON_TYPE_EGR_PORTS   = (1 << 1),  /* Monitoring based on egress ports */
  OPEN_FT_FLW_GRP_MON_TYPE_ING_PORTS   = (1 << 2),  /* Monitoring based on ingress ports*/
  OPEN_FT_FLW_GRP_MON_TYPE_CONGESTION  = (1 << 3),  /* Monitoring based on congestion */
} OPEN_FT_FLW_GRP_MON_TYPE_t;

/** Flow group configuration */
typedef struct _open_ft_flw_grp_config_
{
  char                         name[OPEN_FT_MAX_NAME_LENGTH];                                            /* Name of flow group */
  unsigned int                 cl_num;                                                                   /* Number of Collectors */
  char                         cl_name_list[OPEN_FT_MAX_COLLECTOR_LIST_PER_FG][OPEN_FT_MAX_NAME_LENGTH]; /* Name of Collectors */
  unsigned int                id;                                                                       /* Group id */
  OPEN_FT_FLW_GRP_MON_TYPE_t   type;                                                                     /* Monitoring type of a Flow group */  
  OPEN_FT_FLW_GRP_MON_PARAMS_t mon_params;                                                               /* Monitoring params */
  bool         rep_flws;                         /* Report flows */
  bool         rep_flw_grps;                     /* Report flow groups */
  OPEN_FT_FLW_GRP_ACTION_MASK_t action_mask;
} OPEN_FT_FLW_GRP_CONFIG_t;

/** Max supported sets in a template packet by FT */
#define OPEN_FT_TEMPLATE_MAX_SETS             1

/** Max elements in a template record */
#define OPEN_IP_FIX_INF_MAX_ELEMENTS          16

/** Max templates to used by flow exporting process */
#define OPEN_FT_MAX_TEMPLATES_IN_FLW          OPEN_FT_TEMPLATE_MAX_SETS

/** Min payload length */
#define OPEN_FT_MIN_PAYLOAD_LEN               64 

/** Max payload length */
#define OPEN_FT_MAX_PAYLOAD_LEN               1500 

/** Max number of ASICs supported*/
#define OPEN_FT_ASIC_MAX_COUNT                1

/* Broadview uses byte aligned structure(packed) for the same. */
#pragma pack(push, 1)
typedef struct _open_ip_fix_fld_spec_with_en_
{
  unsigned short ent_ele_id;
  unsigned short len;
  bool           ent_num_present;
  unsigned long  ent_num;
} OPEN_IP_FIX_FLD_SPEC_WITH_EN_t;
#pragma pack(pop)

typedef struct _open_ft_flw_rec_template_info_
{
  char                            name[OPEN_FT_MAX_NAME_LENGTH];
  unsigned short                  template_hdr_id;
  unsigned int                    num_ele;                               /* Number of elements in the template */
  OPEN_IP_FIX_FLD_SPEC_WITH_EN_t ele_info[OPEN_IP_FIX_INF_MAX_ELEMENTS]; /* Elements information */
} OPEN_FT_FLW_REC_TEMPLATE_INFO_t;

typedef struct _open_ft_flw_rec_template_
{
  unsigned int                    num_templates;
  OPEN_FT_FLW_REC_TEMPLATE_INFO_t data[OPEN_FT_MAX_TEMPLATES_IN_FLW];
} OPEN_FT_FLW_REC_TEMPLATE_t;

/** Structure to hold Group stats */
typedef struct _open_ft_flw_grp_stats_t_
{
  uint64_t     flow_cnt;
  uint64_t     octet_cnt;
  uint64_t     pkt_cnt;
  uint64_t     tcp_syn_cnt;
  uint64_t     tcp_fin_cnt;
  uint64_t     tcp_rst_cnt;
} OPEN_FT_FLW_GRP_STATS_t;

/** Structure to hold capbilities */
typedef struct _open_ft_capabilities_t_
{
  bool         multiple_collector_support;
  unsigned int max_collectors;
  unsigned int export_protocols;
  unsigned int flw_grp_types;
  unsigned int max_flw_grps;
  unsigned int max_flws;
  bool         flw_sampling_support;
  bool         flw_template_support_in_fw;
} OPEN_FT_CAPABILITIES_t;

/* Structure to clear flow group stats params */
typedef struct _open_ft_clear_flw_grp_stats_
{
  char         name[OPEN_FT_MAX_NAME_LENGTH];   /* Flow group name */
  bool         grp_rec_clear;  /* Clear group records flag */
  bool         flw_rec_clear;  /* Clear flow records flag */
} OPEN_FT_CLEAR_FLW_GRP_STATS_t;

typedef struct _open_ft_template_xmit_config_s
{
  char collectorName[OPEN_FT_MAX_NAME_LENGTH];
  char templateName[OPEN_FT_MAX_NAME_LENGTH];
  int periodicity;
  int initialBurst;
} OPEN_FT_TEMPLATE_XMIT_CONFIG_t; 

/*********************************************************************
* @purpose  Flow tracker feature configuration set function
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    ftCfg         @b{(input)}         Flow tracker config
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiFtConfigSet(openapiClientHandle_t *client_handle,
                                int asic,
                                OPEN_FT_CONFIG_t *ftCfg);

/*********************************************************************
* @purpose  Flow collector configuration create function
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    clInfo        @b{(input)}         Collector configuration
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiFtCollectorConfigCreate(openapiClientHandle_t *client_handle,
                                            int asic,
                                            OPEN_FT_CL_INFO_t *clInfo);

/*********************************************************************
* @purpose  Flow collector configuration remove function
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    name          @b{(input)}         Collector name
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiFtCollectorConfigRemove(openapiClientHandle_t *client_handle,
                                            int asic, open_buffdesc *name);

/*********************************************************************
* @purpose  Flow group configuration create function
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    ftFlwGrpCfg   @b{(input)}         Group config
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiFtFlowGroupConfigCreate(openapiClientHandle_t *client_handle,
                                            int asic,
                                            OPEN_FT_FLW_GRP_CONFIG_t *ftFlwGrpCfg);

/*********************************************************************
* @purpose  Flow group configuration remove function
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    name          @b{(input)}         Group name
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiFtFlowGroupConfigRemove(openapiClientHandle_t *client_handle,
                                            int asic, open_buffdesc *name);

/*********************************************************************
* @purpose  Flow template configuration create function
*
* @param    client_handle     @b{(input)}         OpenAPI connection Handle
* @param    asic             @b{(input)}         Asic/Unit number 
* @param    ftFlwTemplateCfg @b{(input)}         Template config
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiFtFlowTemplateConfigCreate(openapiClientHandle_t *client_handle,
                                               int asic,
                                               OPEN_FT_FLW_REC_TEMPLATE_t *ftFlwTemplateCfg);

/*********************************************************************
* @purpose  Flow template configuration remove function
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    name          @b{(input)}         Template name
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiFtFlowTemplateConfigRemove(openapiClientHandle_t *client_handle,
                                               int asic, open_buffdesc *name);

/*********************************************************************
* @purpose  Flow group template format packet send function
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    name          @b{(input)}         Flow group name
* @param    pkt_type      @b{(input)}         Packet type
* @param    payloadSize   @b{(input)}         Packet payload size
* @param    payload       @b{(input)}         Packet payload
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiFtPktSend(openapiClientHandle_t *client_handle,
                              int asic, open_buffdesc *name,
                              OPEN_FT_PKT_TYPE_t pkt_type,
                              int payloadSize, open_buffdesc *payload);

/*********************************************************************
* @purpose  Flow group statistics clear function
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    name          @b{(input)}         Flow group name
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiFtFlowGroupStatsClear(openapiClientHandle_t *client_handle,
                                          int asic, open_buffdesc *name);

/*********************************************************************
* @purpose  Flow group statistics get function
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    name          @b{(input)}         Flow group name
* @param    pStat         @b{(output)}        Flow group statistics
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiFtFlowGroupStatsGet(openapiClientHandle_t *client_handle,
                                        int asic, open_buffdesc *name,
                                        OPEN_FT_FLW_GRP_STATS_t *pStat);

/*********************************************************************
* @purpose  Flow tracker feature capabilities get function
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    pCapabilities @b{(output)}        Flow tracker capabilities
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiFtCapabilitiesGet(openapiClientHandle_t *client_handle,
                                      int asic,
                                      OPEN_FT_CAPABILITIES_t *pCapabilities);

/*********************************************************************
* @purpose  Flow group statistics clear function
*
* @param    client_handle    @b{(input)}         OpenAPI connection Handle
* @param    asic            @b{(input)}         Asic/Unit number 
* @param    clearStatParams @b{(input)}         Clear statistics configuration.
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.16
*
* @end
*********************************************************************/
open_error_t openapiFtStatsClear(openapiClientHandle_t *client_handle,
                                 int asic,
                                 OPEN_FT_CLEAR_FLW_GRP_STATS_t *clearStatParams);

/*********************************************************************
* @purpose  Flow tracker template transmit configuration.
*
* @param    client_handle         @b{(input)} OpenAPI connection Handle
* @param    asic                 @b{(input)} Asic/Unit number 
* @param    ftTemplateXmitConfig @b{(input)} Template transmit configuration
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.16
*
* @end
*********************************************************************/
open_error_t openapiFtTemplateTransmitConfig(openapiClientHandle_t *client_handle,
                                             int asic,
                                             OPEN_FT_TEMPLATE_XMIT_CONFIG_t *ftTemplateXmitConfig);

#endif /* OPENAPI_FT_H_INCLUDED*/
