/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License")
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename openapi_instru_ifa.h
*
* @purpose INBAND FLOW ANALYZER API: Provide an interface for configuring 
*                                    IFA sessions in hardware.
*
* @component INBAND FLOW ANALYZER (INBAND FLOW ANALYZER API)
*
* @create   09/06/2018
*
* @author   Naveen Kumar Aketi
* @end
*
**********************************************************************/
#ifndef OPENAPI_IFA_H_INCLUDED
#define OPENAPI_IFA_H_INCLUDED
 
#include <stdbool.h>
#include "openapi_common.h"
#include "openapi_instru_nos.h"

/** Maximum length of name param */
#define OPEN_IFA_MAX_NAME_LENGTH        128 

/* IFA max flow per session */
#define OPEN_IFA_MAX_FLOWS_PER_SESSION  8

/** Max number of ASICs supported*/
#define OPEN_IFA_ASIC_MAX_COUNT         1

/* IFA flow key*/
typedef struct _open_ifa_flow_key_
{
  char         name[OPEN_IFA_MAX_NAME_LENGTH];   /* Name of the singature*/
} OPEN_IFA_FLOW_KEY_t;

/* IFA flow type*/
typedef enum _open_ifa_flow_type_
{
  OPEN_IFA_FLOW_FIVE_TUPLE = (1<<0),   /* Five tuple based flow */
  OPEN_IFA_FLOW_EGR_PORTS  = (1<<1),   /* Egress ports based flow */
  OPEN_IFA_FLOW_ING_PORTS  = (1<<2),   /* Ingress ports based flow */
} OPEN_IFA_FLOW_TYPE_t;

/* Five tuple params mask */
typedef enum _open_ifa_tuple_mask_ 
{
  OPEN_IFA_CONFIG_TUPLE_SRC_IP   = (1 << 0),
  OPEN_IFA_CONFIG_TUPLE_DST_IP   = (1 << 1),
  OPEN_IFA_CONFIG_TUPLE_PROTOCOL = (1 << 2),
  OPEN_IFA_CONFIG_TUPLE_SRC_PORT = (1 << 3),
  OPEN_IFA_CONFIG_TUPLE_DST_PORT = (1 << 4)
} OPEN_IFA_TUPLE_MASK_t;

/* 5-tuple parameters */
typedef struct _open_ifa_5_tuple_key_
{
  OPEN_INSTRU_IP_PREFIX_t src_ip;
  OPEN_INSTRU_IP_PREFIX_t dst_ip;
  unsigned int            protocol;
  unsigned short          src_port;
  unsigned short          dst_port;
  unsigned int            tuple_mask;  /* Valid 5 tuple params mask */
} OPEN_IFA_5_TUPLE_KEY_t;

/* enum that represents params */
typedef enum _open_ifa_flow_param_mask_
{
  OPEN_IFA_FLOW_PARAM_ING_PORT      = (1 << 0),
  OPEN_IFA_FLOW_PARAM_SAMPLING_RATE = (1 << 1),
} OPEN_IFA_FLOW_PARAM_t;

/* structure of different flow params */  
typedef struct _open_ifa_flow_params_
{
  unsigned int           ing_port;      /* Ingress port */
  OPEN_IFA_5_TUPLE_KEY_t five_tuple;    /* five tuple info */
  OPEN_PORT_MASK_t       ing_port_mask; /* Ingress port mask */
  OPEN_PORT_MASK_t       egr_port_mask; /* Egress port mask */
  unsigned int           sampling_rate; /* sampling rate */
  unsigned int           mask;          /* Mask of valid params */
} OPEN_IFA_FLOW_PARAMS_t; 

/* IFA singature information */
typedef struct _open_ifa_flow_
{
  OPEN_IFA_FLOW_KEY_t    key;             /* IFA flow key*/
  OPEN_IFA_FLOW_TYPE_t   type;            /* IFA flow type*/
  OPEN_IFA_FLOW_PARAMS_t params;          /* Flow information*/
} OPEN_IFA_FLOW_t;

typedef enum _open_ifa_cl_config_param_mask_
{
  OPEN_IFA_CL_CONFIG_VLAN_MASK     = (1 << 0),  /* Mask of vlan param */
  OPEN_IFA_CL_CONFIG_PRIORITY_MASK = (1 << 1),  /* Mask of priority param */
} OPEN_IFA_CL_CONFIG_PARAM_MASK_t;

/* IFA Collector key*/
typedef struct _open_ifa_cl_key_
{
  char         name[OPEN_IFA_MAX_NAME_LENGTH];   /* Name of the singature*/
} OPEN_IFA_CL_KEY_t;

/* IFA collector config */
typedef struct _open_ifa_cl_config_
{
  OPEN_IFA_CL_KEY_t                key;      /* Collector Key */
  OPEN_INSTRU_IP_ADDR_t            addr;     /* IP address of the collector */
  unsigned short                   port;     /* Port number of the collector */
  unsigned short                   vlan_id;  /* Vlan ID */
  unsigned char                    priority; /* Priority */
  OPEN_IFA_CL_CONFIG_PARAM_MASK_t  mask;     /* Param mask */
} OPEN_IFA_CL_CONFIG_t;

/* IFA session key*/
typedef struct _open_ifa_session_key_
{
  char         name[OPEN_IFA_MAX_NAME_LENGTH];
} OPEN_IFA_SESSION_KEY_t;

/* IFA session information */
typedef struct _open_ifa_session_
{
  OPEN_IFA_SESSION_KEY_t key;                                  /* IFA session key*/
  unsigned int           num_of_flows;                         /* Number of flows */ 
  OPEN_IFA_FLOW_KEY_t    flow[OPEN_IFA_MAX_FLOWS_PER_SESSION]; /* IFA session flows infromation */
  OPEN_IFA_CL_KEY_t      collector_key;                        /* Collector key */ 
  unsigned int           congestion_limit;                     /* Queue congestion limit */
  unsigned int           drop_limit;                           /* Drop limit */
} OPEN_IFA_SESSION_t;

/* IFA device type */
typedef enum _open_ifa_dev_type_
{
  IFA_DEV_TYPE_INGRESS      = (1 << 0), /* Ingress device */
  IFA_DEV_TYPE_INTERMEDIATE = (1 << 1), /* Intermediate device */
  IFA_DEV_TYPE_EGRESS       = (1 << 2)  /* Egress device */
} OPEN_IFA_DEV_TYPE_t;

/* IFA feature config mask */
typedef enum _open_ifa_feat_config_mask_
{
  OPEN_IFA_FEAT_CONFIG_DEV_ID   = (1 << 1),      /* dev_id param mask */
  OPEN_IFA_FEAT_CONFIG_DEV_TYPE = (1 << 2)       /* dev_type params mask */
} OPEN_IFA_FEAT_CONFIG_MASK_t;

/* ifa feature information */
typedef struct _open_ifa_feature_
{
  bool                        enable;   /* IFA feature status */
  unsigned int                dev_id;   /* Switch ID   */ 
  OPEN_IFA_DEV_TYPE_t         dev_type; /* Device Type */
  OPEN_IFA_FEAT_CONFIG_MASK_t mask;     /* Config params mask */
} OPEN_IFA_FEATURE_t;

/* IFA Capabilities */
typedef struct _open_ifa_capabilities_
{
  unsigned int max_flows;               /* Max supported flows */
  unsigned int max_sessions;            /* Max supported sessions */
  unsigned int max_collectors;          /* Max supported collectors */
  unsigned int max_payload;             /* Maximum number of bytes that an IFA payload can take. */ 
  unsigned int flow_types;              /* Indicates the types of flows that can be monitored for IFA
                                           Supported values */

  unsigned int supported_dev_type_mask; /* Supported device types mask */
} OPEN_IFA_CAPABILITIES_t;

/* IFA Session Statistics */
typedef struct _open_ifa_session_stats_
{
  uint64_t count;
} OPEN_IFA_SESSION_STATS_t;

/*********************************************************************
* @purpose  Inband flow analyzer feature configuration set function
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    config        @b{(input)}         Inband flow analyzer config
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiIfaConfigSet(openapiClientHandle_t *client_handle,
                                 int asic,
                                 OPEN_IFA_FEATURE_t *config);

/*********************************************************************
* @purpose  Inband flow analyzer feature configuration get function
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    config        @b{(input/output)}  Inband flow analyzer config
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiIfaConfigGet(openapiClientHandle_t *client_handle,
                                 int asic,
                                 OPEN_IFA_FEATURE_t *config);

/*********************************************************************
* @purpose  Create IFA session. 
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    session       @b{(input)}         IFA session configuration
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiIfaSessionCreate(openapiClientHandle_t *client_handle,
                                     int asic,
                                     OPEN_IFA_SESSION_t *session);

/*********************************************************************
* @purpose  Create IFA flow. 
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    appFlow       @b{(input)}         IFA flow configuration
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiIfaFlowCreate(openapiClientHandle_t *client_handle,
                                  int asic,
                                  OPEN_IFA_FLOW_t *appFlow);

/*********************************************************************
* @purpose  Remove IFA flow. 
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    name          @b{(input)}         IFA flow name
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiIfaFlowRemove(openapiClientHandle_t *client_handle,
                                  int asic,
                                  open_buffdesc *name);

/*********************************************************************
* @purpose  Start IFA session. 
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    name          @b{(input)}         IFA session name
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiIfaSessionStart(openapiClientHandle_t *client_handle,
                                    int asic,
                                    open_buffdesc *name);

/*********************************************************************
* @purpose  Stop IFA session. 
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    name          @b{(input)}         IFA session name
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiIfaSessionStop(openapiClientHandle_t *client_handle,
                                   int asic,
                                   open_buffdesc *name);

/*********************************************************************
* @purpose  Remove IFA session. 
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    name          @b{(input)}         IFA session name
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiIfaSessionRemove(openapiClientHandle_t *client_handle,
                                     int asic,
                                     open_buffdesc *name);

/*********************************************************************
* @purpose  Get IFA capabilities. 
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    cap           @b{(input)}         IFA capabilities
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiIfaCapabilitiesGet(openapiClientHandle_t *client_handle,
                                       int asic,
                                       OPEN_IFA_CAPABILITIES_t *cap);

/*********************************************************************
* @purpose  Clear IFA stats. 
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    name          @b{(input)}         IFA session name
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiIfaClearStats(openapiClientHandle_t *client_handle,
                                  int asic,
                                  open_buffdesc *name);

/*********************************************************************
* @purpose  Get IFA stats. 
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    name          @b{(input)}         IFA session name
* @param    data          @b{(inout)}         IFA session stats
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiIfaSessionStatsGet(openapiClientHandle_t *client_handle,
                                       int asic,
                                       open_buffdesc *name,
                                       OPEN_IFA_SESSION_STATS_t *data);

/*********************************************************************
* @purpose  Create IFA collector 
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    clConfig      @b{(input)}         IFA collector configuration
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiIfaCollectorCreate(openapiClientHandle_t *client_handle,
                                       int asic,
                                       OPEN_IFA_CL_CONFIG_t *clConfig);

/*********************************************************************
* @purpose  Remove IFA collector.
*
* @param    client_handle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    name          @b{(input)}         IFA collector name
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @returns  OPEN_E_PARAM 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiIfaCollectorRemove(openapiClientHandle_t *client_handle,
                                       int asic,
                                       open_buffdesc *name);
#endif /* OPENAPI_IFA_H_INCLUDED*/
