/**********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_ip4dhcpclient.h
*
* @purpose   This code implements the OpEN IPv4 DHCP Client API
*
* @component OPEN
*
* @create    02/08/2023
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_IP4DHCPCLIENT_H_INCLUDED
#define OPENAPI_IP4DHCPCLIENT_H_INCLUDED

#include "openapi_common.h" 


/**< DHCP Client State Machine States */
typedef enum
{
  OPEN_DHCPSTATE_INIT = 1,
  OPEN_DHCPSTATE_SELECTING,
  OPEN_DHCPSTATE_REQUESTING,
  OPEN_DHCPSTATE_REQUEST_RECV,
  OPEN_DHCPSTATE_BOUND,
  OPEN_DHCPSTATE_RENEWING,
  OPEN_DHCPSTATE_RENEW_RECV,
  OPEN_DHCPSTATE_REBINDING,
  OPEN_DHCPSTATE_REBIND_RECV,
  OPEN_DHCPSTATE_BOOTP_FALLBACK,
  OPEN_DHCPSTATE_NOTBOUND,
  OPEN_DHCPSTATE_FAILED,
  OPEN_DHCPSTATE_DO_RELEASE,
  OPEN_DHCPSTATE_INFORM_REQUEST,
  OPEN_DHCPSTATE_INFORM_ACK_WAIT,
  OPEN_DHCPSTATE_INFORM_FAILED,
  OPEN_DHCPSTATE_INFORM_BOUND

} OPEN_DHCP_CLIENT_STATE_t;

#define OPEN_DHCP_CLIENTID_MAX  (128)             /**< Max length of the DHCP Client Identifier */
#define OPEN_DHCP_VENDOR_CLASS_STRING_MAX  (128)  /**< Max length of the vendor class string */

/*****************************************************************************
 * @purpose   Get the DHCP Vendor Option Mode.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     mode              @b{(output)} DHCP vendor option admin mode(OPEN_ENABLE or OPEN_DISABLE)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpVendorClassOptionAdminModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the DHCP Vendor Option Mode.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     mode              @b{(input)}  DHCP vendor option admin mode(OPEN_ENABLE or OPEN_DISABLE)
 * @param     actImmediate      @b{(input)}  Immediate action flag(OPEN_TRUE or OPEN_FALSE)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     If actImmediate is True, the setting will be applied immediately without posting an event to the DHCP Client task. 
 *            Use this option only when there are synchronization issues, especially with SETS.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpVendorClassOptionAdminModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode, OPEN_BOOL_t actImmediate);

/*****************************************************************************
 * @purpose   Get the DHCP Vendor Class Option string.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     bufd              @b{(output)} Buffer descriptor to DHCP Vendor Class Option string
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpVendorClassOptionStringGet(openapiClientHandle_t *client_handle, open_buffdesc *bufd);

/*****************************************************************************
 * @purpose   Set the DHCP Vendor Class Option string.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     bufd              @b{(input)}  Buffer descriptor to DHCP Vendor Class Option string
 * @param     actImmediate      @b{(input)}  Immediate action flag (OPEN_TRUE or OPEN_FALSE)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpVendorClassOptionStringSet(openapiClientHandle_t *client_handle, open_buffdesc *bufd, OPEN_BOOL_t actImmediate);

/*****************************************************************************
 * @purpose   Get the DHCP server IP address on a management interface
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     ifNum             @b{(input)}  Internal Interface Number
 * @param     mgmtPortType      @b{(input)}  Mangement Interface Type
 * @param     serverAddr        @b{(output)} Pointer to the DHCP server IP Address on an interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpDhcpClientDhcpServerIntfIPAddressGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_MGMT_PORT_TYPE_t mgmtPortType, uint32_t *serverAddr);

/*****************************************************************************
 * @purpose   Get the current state of the DHCP client for specified interface or management port
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     ifNum             @b{(input)}  Internal Interface Number
 * @param     mgmtPortType      @b{(input)}  Mangement Interface Type
 * @param     clientState       @b{(output)} Pointer to the DHCP client state
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpDhcpClientIntfStateGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_MGMT_PORT_TYPE_t mgmtPortType, OPEN_DHCP_CLIENT_STATE_t *clientState);

/*****************************************************************************
 * @purpose   Get the transaction ID of the DHCP client for specified interface or management port
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     ifNum             @b{(input)}  Internal Interface Number
 * @param     mgmtPortType      @b{(input)}  Mangement Interface Type
 * @param     transId           @b{(output)} Pointer to the DHCP client transaction ID
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpDhcpClientTransactionIntfIdGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_MGMT_PORT_TYPE_t mgmtPortType, uint32_t *transId);

/*****************************************************************************
 * @purpose   Get the lease time information of the DHCP client for specified interface or management port
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     ifNum             @b{(input)}  Internal Interface Number
 * @param     mgmtPortType      @b{(input)}  Mangement Interface Type
 * @param     leaseTime         @b{(output)} Pointer to the DHCP client lease time
 * @param     renewalTime       @b{(output)} Pointer to the DHCP client renewal time
 * @param     rebindTime        @b{(output)} Pointer to the DHCP client rebinding time
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpDhcpClientLeaseTimeIntfInfoGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_MGMT_PORT_TYPE_t mgmtPortType, uint32_t *leaseTime, uint32_t *renewalTime, uint32_t *rebindTime);

/*****************************************************************************
 * @purpose   Get the retry count of the DHCP client for specified interface or management port
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     ifNum             @b{(input)}  Internal Interface Number
 * @param     mgmtPortType      @b{(input)}  Mangement Interface Type
 * @param     retryCount        @b{(output)} Pointer to the DHCP client's retry count
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpDhcpClientIntfRetryCountGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_MGMT_PORT_TYPE_t mgmtPortType, uint32_t *retryCount);

/*****************************************************************************
 * @purpose   Get the generated client identifier on the interface.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     ifNum             @b{(input)}  Internal Interface Number
 * @param     mgmtPortType      @b{(input)}  Mangement Interface Type
 * @param     clientId          @b{(output)} Buffer descriptor to the clientId string
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpDhcpClientIdGeneratedGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_MGMT_PORT_TYPE_t mgmtPortType, open_buffdesc *clientId);

/*****************************************************************************
 * @purpose   Get the DHCP Client Statistics on an interface.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     ifNum             @b{(input)}  Internal Interface Number
 * @param     numDiscoverSent   @b{(output)} Number of Discovers sent
 * @param     numOfferRecvd     @b{(output)} Number of Offers received
 * @param     numRequestSent    @b{(output)} Number of Requests sent
 * @param     numAckRecvd       @b{(output)} Number of Acks received
 * @param     numNackRecvd      @b{(output)} Number of Nacks received
 * @param     numReleaseSent    @b{(output)} Number of Releases sent
 * @param     numInformSent     @b{(output)} Number of Informs sent
 * @param     numRebindSent     @b{(output)} Number of Rebinds sent
 * @param     numRenewSent      @b{(output)} Number of Renews sent
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpClientStatsGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *numDiscoverSent, uint32_t *numOfferRecvd, uint32_t *numRequestSent, uint32_t *numAckRecvd, uint32_t *numNackRecvd, uint32_t *numReleaseSent, uint32_t *numInformSent, uint32_t *numRebindSent, uint32_t *numRenewSent);

/*****************************************************************************
 * @purpose   Clear the DHCP Client Statistics on an interface.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     ifNum             @b{(input)}  Internal Interface Number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpClientStatsClear(openapiClientHandle_t *client_handle, uint32_t ifNum);

#endif /* OPENAPI_IP4DHCPCLIENT_H_INCLUDED */

