/**********************************************************************
*
*  Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_ipv4_device_tracking.h
*
* @purpose   IPv4 Device Tracking configuration and status.
*
* @component OPEN
*
* @create    11/21/2018
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_IPV4_DEVICE_TRACKING_H_INCLUDED
#define OPENAPI_IPV4_DEVICE_TRACKING_H_INCLUDED

#include <stdio.h>
#include <stddef.h>
#include <sys/un.h>

#include "openapi_common.h"

typedef enum OPEN_IPV4DT_CLEAR_s
{
  OPEN_IPV4DT_CLEAR_ALL,           /* clear all entries                                       */
  OPEN_IPV4DT_CLEAR_HOST_IP,       /* clear only the entries matching a specific IP address   */
  OPEN_IPV4DT_CLEAR_HOST_MAC,      /* clear only the entries matching a specific MAC address  */
  OPEN_IPV4DT_CLEAR_INTERFACE,     /* clear only the entries related to a specific interface  */
} OPEN_IPV4DT_CLEAR_t;

typedef struct OPEN_IPV4DT_CLEAR_REQUEST_s
{
  OPEN_IPV4DT_CLEAR_t   clear;
  uint32_t              hostIpAddr;
  unsigned char         hostMacAddr[OPEN_MAC_ADDR_LEN];
  uint32_t              intIfNum;

} OPEN_IPV4DT_CLEAR_REQUEST_t;

typedef enum
{
  OPEN_IPV4DT_GET_FIRST = 0,
  OPEN_IPV4DT_GET,
  OPEN_IPV4DT_GET_NEXT

} OPEN_IPV4DT_GET_TYPE_t;

typedef enum
{
  OPEN_IPV4DT_GET_ALL = 0,
  OPEN_IPV4DT_GET_ALL_ACTIVE,
  OPEN_IPV4DT_GET_ALL_INACTIVE,
  OPEN_IPV4DT_GET_COUNT

} OPEN_IPV4DT_GET_ALL_TYPE_t;

typedef struct OPEN_IPV4DT_SHOW_REQUEST_s
{
  OPEN_IPV4DT_GET_TYPE_t      getType;
  OPEN_BOOL_t                 noFilter;
  OPEN_IPV4DT_GET_ALL_TYPE_t  getAllType;
  OPEN_BOOL_t                 hostIpFilter;
  uint32_t                    hostIpAddr;
  OPEN_BOOL_t                 hostMacFilter;
  unsigned char               hostMacAddr[OPEN_MAC_ADDR_LEN];
  OPEN_BOOL_t                 intfFilter;
  uint32_t                    intIfNum;

} OPEN_IPV4DT_SHOW_REQUEST_t;

typedef enum
{
  OPEN_IPV4DT_SOURCE_ARP = 0,
  OPEN_IPV4DT_SOURCE_DHCP

} OPEN_IPV4DT_Source_t;

typedef enum
{
  OPEN_IPV4DT_STATE_INACTIVE = 0,
  OPEN_IPV4DT_STATE_ACTIVE

} OPEN_IPV4DT_State_t;

typedef struct OPEN_IPV4DT_ENTRY_s
{
  OPEN_IPV4DT_Source_t    source;
  OPEN_IPV4DT_State_t     state;
  uint32_t                hostIpAddr;
  unsigned char           hostMacAddr[OPEN_MAC_ADDR_LEN];
  uint32_t                intIfNum;
  uint16_t                vlanId;
  uint32_t                timeToInActive;
  uint32_t                timeSinceInActive;

} OPEN_IPV4DT_ENTRY_t;

typedef struct OPEN_IPV4DT_ENTRIES_COUNT_s
{
  uint32_t   arpEntriesCount;
  uint32_t   dhcpEntriesCount;
  uint32_t   activeEntriesCount;
  uint32_t   inactiveEntriesCount;
  uint32_t   totalEntriesCount;

} OPEN_IPV4DT_ENTRIES_COUNT_t;

/************************************************************************
* @purpose  Gets the Ipv4dt Admin mode
*
* @param    client_handle   {(input)}   OpEN client handle
* @param    enable         {(output)}  Mode (Enable or Disable).
*
*
* @returns  OPEN_E_NONE:   Admin mode get is successful
* @returns  OPEN_E_PARAM:  Parameters are not valid
* @returns  OPEN_E_FAIL:   Failed to get the admin mode
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*************************************************************************/
open_error_t openapiIpv4dtAdminModeGet(openapiClientHandle_t *client_handle,
                                       OPEN_CONTROL_t *enable);

/*************************************************************************
* @purpose  Sets the Ipv4dt Admin mode
*
* @param    client_handle   {(input)}   OpEN client handle
* @param    enable         {(input)}  Mode (Enable or Disable).
*
*
* @returns  OPEN_E_NONE:   Admin mode set is successful
* @returns  OPEN_E_PARAM:  Parameters are not valid
* @returns  OPEN_E_FAIL:   Failed to set the admin mode
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*************************************************************************/
open_error_t openapiIpv4dtAdminModeSet(openapiClientHandle_t *client_handle,
                                       OPEN_CONTROL_t enable);

/************************************************************************
* @purpose  Gets the Ipv4dt Probe mode
*
* @param    client_handle   {(input)}   OpEN client handle
* @param    enable         {(output)}  Mode (Enable or Disable).
*
*
* @returns  OPEN_E_NONE:   Probe mode get is successful
* @returns  OPEN_E_PARAM:  Parameters are not valid
* @returns  OPEN_E_FAIL:   Failed to get the probe mode
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*************************************************************************/
open_error_t openapiIpv4dtProbeModeGet(openapiClientHandle_t *client_handle,
                                       OPEN_CONTROL_t *enable);

/**************************************************************************
* @purpose  Sets the Ipv4dt Probe mode
*
* @param    client_handle   {(input)}   OpEN client handle
* @param    enable         {(input)}   Mode (Enable or Disable).
*
*
* @returns  OPEN_E_NONE:   Probe mode set is successful
* @returns  OPEN_E_PARAM:  Parameters are not valid
* @returns  OPEN_E_FAIL:   Failed to set the probe mode
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*************************************************************************/
open_error_t openapiIpv4dtProbeModeSet(openapiClientHandle_t *client_handle,
                                       OPEN_CONTROL_t enable);

/**************************************************************************
* @purpose  Gets the Ipv4dt Probe Interval
*
* @param    client_handle   {(input)}   OpEN client handle
* @param    interval        {(output)}  Probe Interval.
*
*
* @returns  OPEN_E_NONE:   Probe interval get is successful
* @returns  OPEN_E_PARAM:  Parameters are not valid
* @returns  OPEN_E_FAIL:   Failed to get the probe interval
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*****************************************************************************/
open_error_t openapiIpv4dtProbeIntervalGet(openapiClientHandle_t *client_handle,
                                           uint32_t *interval);

/*****************************************************************************
* @purpose  Sets the Ipv4dt Probe Interval
*
* @param    client_handle   {(input)}   OpEN client handle
* @param    interval        {(input)}   Probe Interval.
*
*
* @returns  OPEN_E_NONE:     Probe interval set is successful
* @returns  OPEN_E_PARAM:    Parameters are not valid
* @returns  OPEN_E_INTERNAL: Failed to set the probe interval
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*******************************************************************************/
open_error_t openapiIpv4dtProbeIntervalSet(openapiClientHandle_t *client_handle,
                                           uint32_t interval);

/**************************************************************************
* @purpose  Gets the Ipv4dt Probe Count
*
* @param    client_handle   {(input)}   OpEN client handle
* @param    count          {(output)}  Probe Count.
*
*
* @returns  OPEN_E_NONE:   Probe count get is successful
* @returns  OPEN_E_PARAM:  Parameters are not valid
* @returns  OPEN_E_FAIL:   Failed to get the probe count
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
****************************************************************************/
open_error_t openapiIpv4dtProbeCountGet(openapiClientHandle_t *client_handle,
                                        uint32_t *count);

/**************************************************************************
* @purpose  Sets the Ipv4dt Probe Count
*
* @param    client_handle   {(input)}   OpEN client handle
* @param    count          {(input)}   Probe Count.
*
*
* @returns  OPEN_E_NONE:     Probe count set is successful
* @returns  OPEN_E_PARAM:    Parameters are not valid
* @returns  OPEN_E_INTERNAL: Failed to set the probe count
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
****************************************************************************/
open_error_t openapiIpv4dtProbeCountSet(openapiClientHandle_t *client_handle,
                                        uint32_t count);

/**************************************************************************
* @purpose  Gets the Ipv4dt Probe Delay
*
* @param    client_handle   {(input)}   OpEN client handle
* @param    delay          {(output)}  Probe Delay.
*
*
* @returns  OPEN_E_NONE:  Probe delay get is successful
* @returns  OPEN_E_PARAM: Parameters are not valid
* @returns  OPEN_E_FAIL:  Failed to get the probe delay
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
****************************************************************************/
open_error_t openapiIpv4dtProbeDelayGet(openapiClientHandle_t *client_handle,
                                        uint32_t *delay);

/**************************************************************************
* @purpose  Sets the Ipv4dt Probe Delay
*
* @param    client_handle   {(input)}   OpEN client handle
* @param    delay          {(input)}   Probe Delay.
*
*
* @returns  OPEN_E_NONE:     Probe delay set is successful
* @returns  OPEN_E_PARAM:    Parameters are not valid
* @returns  OPEN_E_INTERNAL: Failed to set the probe delay
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
****************************************************************************/
open_error_t openapiIpv4dtProbeDelaySet(openapiClientHandle_t *client_handle,
                                        uint32_t delay);

/***************************************************************************
* @purpose  Gets the Ipv4dt Probe Auto-Source parameters
*
* @param    client_handle  {(input)}    OpEN client handle
* @param    host          {(output)}   Auto-source Host.
* @param    mask          {(output)}   Auto-source Mask.
*
*
* @returns  OPEN_E_NONE:  Auto source get is successful
* @returns  OPEN_E_FAIL:  Failed to get the auto source parameters
* @returns  OPEN_E_PARAM: Parameters are not valid
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*****************************************************************************/
open_error_t openapiIpv4dtProbeAutoSourceGet(openapiClientHandle_t *client_handle,
                                             uint32_t *host,
                                             uint32_t *mask);

/***************************************************************************
* @purpose  Sets the Ipv4dt Probe Auto-Source parameters
*
* @param    client_handle  {(input)}    OpEN client handle
* @param    host          {(output)}   Auto-source Host.
* @param    mask          {(output)}   Auto-source Mask.
*
*
* @returns  OPEN_E_NONE:  Auto source set is successful
* @returns  OPEN_E_FAIL:  Failed to set the auto source parameters
* @returns  OPEN_E_PARAM: Parameters are not valid
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*****************************************************************************/
open_error_t openapiIpv4dtProbeAutoSourceSet(openapiClientHandle_t *client_handle,
                                             uint32_t *host,
                                             uint32_t *mask);

/***************************************************************************
* @purpose  Gets the Ipv4dt Interface Maximum limit
*
* @param    client_handle  {(input)}   OpEN client handle
* @param    intf          {(input)}   Interface number
* @param    intfMax       {(output)}  Max Limit
*
*
* @returns  OPEN_E_NONE:     Max limit get is successful
* @returns  OPEN_E_PARAM:    Parameters are not valid
* @returns  OPEN_E_UNAVAIL:  Interface is not supported
* @returns  OPEN_E_INTERNAL: Failed to set the max limit on a interface
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*****************************************************************************/
open_error_t openapiIpv4dtIntfMaxGet(openapiClientHandle_t *client_handle,
                                     uint32_t intf,
                                     uint32_t *intfMax);

/***************************************************************************
* @purpose  Sets the Ipv4dt Interface Maximum limit
*
* @param    client_handle  {(input)}   OpEN client handle
* @param    intf          {(input)}   Interface number
* @param    intfMax       {(input)}   Max Limit
*
*
* @returns  OPEN_E_NONE:     Max limit set on a interface is successful
* @returns  OPEN_E_PARAM:    Parameters are not valid
* @returns  OPEN_E_UNAVAIL:  Interface is not supported
* @returns  OPEN_E_INTERNAL: Failed to set the max limit on a interface
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*****************************************************************************/
open_error_t openapiIpv4dtIntfMaxSet(openapiClientHandle_t *client_handle,
                                     uint32_t intf,
                                     uint32_t intfMax);

/***************************************************************************
* @purpose  Clears the Ipv4dt Entries
*
* @param    client_handle    {(input)}   OpEN client handle
* @param    ipv4dtClear     {(input)}   IPv4DT Entries Clear
*
*
* @returns  OPEN_E_NONE:    Entries clear is successful
* @returns  OPEN_E_FAIL:    Failed to clear the entries.
* @returns  OPEN_E_PARAM:   Parameters are not valid
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*****************************************************************************/
open_error_t openapiIpv4dtClearEntries(openapiClientHandle_t *client_handle,
                                       OPEN_IPV4DT_CLEAR_REQUEST_t *ipv4dtClear);

/***************************************************************************
* @purpose  Gets the Ipv4dt Entries Count
*
* @param    client_handle          {(input)}   OpEN client handle
* @param    ipv4dtEntriesCount    {(input)}   Entries Count (ARP, DHCP)
*
*
* @returns  OPEN_E_NONE:    Entries count get is successful
* @returns  OPEN_E_FAIL:    Failed to get the entries count
* @returns  OPEN_E_PARAM:   Parameters are not valid
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*****************************************************************************/
open_error_t openapiIpv4dtEntriesCountGet(openapiClientHandle_t *client_handle,
                                          OPEN_IPV4DT_ENTRIES_COUNT_t *ipv4dtEntriesCount);

/***************************************************************************
* @purpose  Gets the Ipv4dt Entries
*
* @param    client_handle    {(input)}   OpEN client handle
* @param    ipv4dtShow      {(input)}   Show request parameters
* @param    entry           {(output)}  IPv4DT Entry
*
*
* @returns  OPEN_E_NONE:    Next Entry get is successful
* @returns  OPEN_E_FAIL:    Failed to get the next entry
* @returns  OPEN_E_PARAM:   Parameters are not valid
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*****************************************************************************/
open_error_t openapiIpv4dtTableGetNext(openapiClientHandle_t *client_handle,
                                       OPEN_IPV4DT_SHOW_REQUEST_t *ipv4dtShow,
                                       OPEN_IPV4DT_ENTRY_t *entry);

/***************************************************************************
* @purpose  Get the Next Valid Ipv4dt Interface
*
* @param    client_handle  {(input)}   OpEN client handle
* @param    intIfNum      {(input)}   Current Interface number
* @param    nextIntIfNum  {(output)}  Next Interface number
*
*
* @returns  OPEN_E_NONE:     Next Interface get is successful
* @returns  OPEN_E_PARAM:    Parameters are not valid
* @returns  OPEN_E_UNAVAIL:  Interface is not supported
* @returns  OPEN_E_INTERNAL: If failed to get the next interface
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*****************************************************************************/
open_error_t openapiIpv4dtIntfNextGet(openapiClientHandle_t *client_handle,
                                      uint32_t intIfNum, uint32_t *nextIntIfNum);

/***************************************************************************
* @purpose  Check the Ipv4dt Interface is supported or not
*
* @param    client_handle  {(input)}   OpEN client handle
* @param    intIfNum      {(input)}   Interface number
* @param    isSupported   {(output)}  OPEN_TRUE if interface is supported
*                                     OPEN_FALSE if interface is not supported 
*
*
* @returns  OPEN_E_NONE:    Interface is supported
* @returns  OPEN_E_PARAM:   Parameters are not valid
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*****************************************************************************/
open_error_t openapiIpv4dtIntfSupported(openapiClientHandle_t *client_handle,
                                        uint32_t intIfNum, OPEN_BOOL_t *isSupported);

#endif
