/**********************************************************************
*
*  Copyright 2023-2024 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_linuxptp.h
*
* @purpose   This code implements the OpEN LinuxPTP API
*
* @component OPEN
*
* @create    02/13/2023
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_LINUXPTP_H_INCLUDED
#define OPENAPI_LINUXPTP_H_INCLUDED

#include "openapi_common.h" 

/* Types of interface statistics */
typedef enum
{
  OPEN_LINUXPTP_STATS_NONE = 0,
  OPEN_LINUXPTP_ANNOUNCE_TX_COUNT,
  OPEN_LINUXPTP_ANNOUNCE_RX_COUNT,
  OPEN_LINUXPTP_ANNOUNCE_RX_TIMEOUTS,
  OPEN_LINUXPTP_SYNC_TX_COUNT,
  OPEN_LINUXPTP_SYNC_RX_COUNT,
  OPEN_LINUXPTP_SYNC_MISMATCH,
  OPEN_LINUXPTP_SYNC_RX_TIMEOUTS,
  OPEN_LINUXPTP_FOLLOWUP_TX_COUNT,
  OPEN_LINUXPTP_FOLLOWUP_RX_COUNT,
  OPEN_LINUXPTP_FOLLOWUP_MISMATCH,
  OPEN_LINUXPTP_DELAY_REQ_TX_COUNT,
  OPEN_LINUXPTP_DELAY_REQ_RX_COUNT,
  OPEN_LINUXPTP_PDELAY_REQ_TX_COUNT,
  OPEN_LINUXPTP_PDELAY_REQ_RX_COUNT,
  OPEN_LINUXPTP_PDELAY_RESP_TX_COUNT,
  OPEN_LINUXPTP_PDELAY_RESP_RX_COUNT,
  OPEN_LINUXPTP_DELAY_RESP_TX_COUNT,
  OPEN_LINUXPTP_DELAY_RESP_RX_COUNT,
  OPEN_LINUXPTP_PDELAY_RESP_FOLLOWUP_TX_COUNT,
  OPEN_LINUXPTP_PDELAY_RESP_FOLLOWUP_RX_COUNT,
  OPEN_LINUXPTP_SIGNALING_TX_COUNT,
  OPEN_LINUXPTP_SIGNALING_RX_COUNT,
  OPEN_LINUXPTP_MANAGEMENT_TX_COUNT,
  OPEN_LINUXPTP_MANAGEMENT_RX_COUNT,
  OPEN_LINUXPTP_STATS_LAST

} OPEN_LINUXPTP_STATS_TYPE_t;

/* Parent, default and current data sets - local, GM and parent clock information */
typedef enum
{
  OPEN_LINUXPTP_DS_NONE = 0,
  OPEN_LINUXPTP_PARENTDS_PARENT_IDENTITY,
  OPEN_LINUXPTP_PARENTDS_PARENT_STATS,
  OPEN_LINUXPTP_PARENTDS_PARENT_OFFSET,
  OPEN_LINUXPTP_PARENTDS_PARENT_PHASE_CHANGE_RATE,
  OPEN_LINUXPTP_PARENTDS_GM_PRIORITY1,
  OPEN_LINUXPTP_PARENTDS_GM_CLASS,
  OPEN_LINUXPTP_PARENTDS_GM_ACCURACY,
  OPEN_LINUXPTP_PARENTDS_GM_OFFSET,
  OPEN_LINUXPTP_PARENTDS_GM_PRIORITY2,
  OPEN_LINUXPTP_PARENTDS_GM_IDENTITY,

  OPEN_LINUXPTP_CURRENTDS_OFFSET_FROM_MASTER,
  OPEN_LINUXPTP_CURRENTDS_STEPS_REMOVED,
  OPEN_LINUXPTP_CURRENTDS_MEAN_PATH_DELAY,

  OPEN_LINUXPTP_DEFAULTDS_TWO_STEP_FLAG,
  OPEN_LINUXPTP_DEFAULTDS_NUMBER_OF_PORTS,
  OPEN_LINUXPTP_DEFAULTDS_PRIORITY1,
  OPEN_LINUXPTP_DEFAULTDS_CLASS,
  OPEN_LINUXPTP_DEFAULTDS_ACCURACY,
  OPEN_LINUXPTP_DEFAULTDS_OFFSET,
  OPEN_LINUXPTP_DEFAULTDS_PRIORITY2,
  OPEN_LINUXPTP_DEFAULTDS_IDENTITY,
  OPEN_LINUXPTP_DEFAULTDS_DOMAIN_IDENTIFIER,
  OPEN_LINUXPTP_DS_LAST

} OPEN_LINUXPTP_DS_TYPE_t;

/* linuxPTP Priority types */
typedef enum
{
  OPEN_LINUXPTP_CLOCK_PRIORITY1 = 0,
  OPEN_LINUXPTP_CLOCK_PRIORITY2
} OPEN_LINUXPTP_CLOCK_PRIORITY_t;

/* linuxPTP Clock Operation modes */
typedef enum
{
  OPEN_LINUXPTP_CLOCK_OPER_NONE = 0,
  OPEN_LINUXPTP_TWO_STEP = 1,
  OPEN_LINUXPTP_ONE_STEP = 2
} OPEN_LINUXPTP_CLOCK_OPERATION_t;

/* linuxPTP Profiles */
typedef enum
{
  OPEN_LINUXPTP_PROFILE_DEFAULT = 0,
  OPEN_LINUXPTP_PROFILE_AES67 = 1,
  OPEN_LINUXPTP_PROFILE_SMPTE_2059_2 = 2,
  OPEN_LINUXPTP_PROFILE_AES_R16_2016 = 3
} OPEN_LINUXPTP_PROFILES_t;

/* linuxPTP Delay Mechanism*/
typedef enum
{
  OPEN_LINUXPTP_E2E_DELAY_MECHANISM = 1,
  OPEN_LINUXPTP_P2P_DELAY_MECHANISM = 2,
  OPEN_LINUXPTP_COMMON_P2P_DELAY_MECHANISM = 3,
  OPEN_LINUXPTP_SPECIAL_DELAY_MECHANISM = 4,
  OPEN_LINUXPTP_NO_DELAY_MECHANISM = 254,
} OPEN_LINUXPTP_DELAY_MECHANISM_t;

typedef enum
{
  OPEN_LINUXPTP_TRANSPORT_NONE = 0,
  OPEN_LINUXPTP_TRANSPORT_V4,
  OPEN_LINUXPTP_TRANSPORT_V6,
  OPEN_LINUXPTP_TRANSPORT_L2,
} OPEN_LINUXPTP_TRANSPORT_MODE_t;

/* PTP port states */
typedef enum
{
  OPEN_LINUXPTP_PTP_PORT_STATE_NONE = 0,
  OPEN_LINUXPTP_PTP_PORT_STATE_INITIALIZING,
  OPEN_LINUXPTP_PTP_PORT_STATE_FAULTY,
  OPEN_LINUXPTP_PTP_PORT_STATE_DISABLED,
  OPEN_LINUXPTP_PTP_PORT_STATE_LISTENING,
  OPEN_LINUXPTP_PTP_PORT_STATE_PRE_MASTER,
  OPEN_LINUXPTP_PTP_PORT_STATE_MASTER,
  OPEN_LINUXPTP_PTP_PORT_STATE_PASSIVE,
  OPEN_LINUXPTP_PTP_PORT_STATE_UNCALIBRATED,
  OPEN_LINUXPTP_PTP_PORT_STATE_SLAVE,
  OPEN_LINUXPTP_PTP_PORT_STATE_LAST
} OPEN_LINUXPTP_PORTSTATE_TYPE_t;

/* Time properties information */
typedef enum
{
  OPEN_LINUXPTP_TIME_PROPERTIES_NONE = 0,
  OPEN_LINUXPTP_TIME_PROPERTIES_CURRENTUTCOFFSET,
  OPEN_LINUXPTP_TIME_PROPERTIES_LEAP61,
  OPEN_LINUXPTP_TIME_PROPERTIES_LEAP59,
  OPEN_LINUXPTP_TIME_PROPERTIES_CURRENTUTCOFFSETVALID,
  OPEN_LINUXPTP_TIME_PROPERTIES_PTPTIMESCALE,
  OPEN_LINUXPTP_TIME_PROPERTIES_TIMETRACEABLE,
  OPEN_LINUXPTP_TIME_PROPERTIES_FREQUENCYTRACEABLE,
  OPEN_LINUXPTP_TIME_PROPERTIES_TIMESOURCE,
  OPEN_LINUXPTP_TIME_PROPERTIES_LAST

} OPEN_LINUXPTP_TIME_PROPERTIES_TYPE_t;


#define OPEN_LINUXPTP_PRIORITY_1_MINIMUM                        0
#define OPEN_LINUXPTP_PRIORITY_1_DEFAULT                        128
#define OPEN_LINUXPTP_PRIORITY_1_MAXIMUM                        255

#define OPEN_LINUXPTP_PRIORITY_2_MINIMUM                        0
#define OPEN_LINUXPTP_PRIORITY_2_DEFAULT                        128
#define OPEN_LINUXPTP_PRIORITY_2_MAXIMUM                        255

/* 1588 Default Profile */
#define OPEN_LINUXPTP_1588_DEFAULT_PROFILE_DOMAIN_MINIMUM              0
#define OPEN_LINUXPTP_1588_DEFAULT_PROFILE_DOMAIN_DEFAULT              0
#define OPEN_LINUXPTP_1588_DEFAULT_PROFILE_DOMAIN_MAXIMUM              127

#define OPEN_LINUXPTP_1588_DEFAULT_PROFILE_ANNOUNCE_INTERVAL_MINIMUM   0
#define OPEN_LINUXPTP_1588_DEFAULT_PROFILE_ANNOUNCE_INTERVAL_DEFAULT   1
#define OPEN_LINUXPTP_1588_DEFAULT_PROFILE_ANNOUNCE_INTERVAL_MAXIMUM   4

#define OPEN_LINUXPTP_1588_DEFAULT_PROFILE_ANNOUNCE_TIMEOUT_MINIMUM    2
#define OPEN_LINUXPTP_1588_DEFAULT_PROFILE_ANNOUNCE_TIMEOUT_DEFAULT    3
#define OPEN_LINUXPTP_1588_DEFAULT_PROFILE_ANNOUNCE_TIMEOUT_MAXIMUM    10

#define OPEN_LINUXPTP_1588_DEFAULT_PROFILE_SYNC_INTERVAL_MINIMUM      -1
#define OPEN_LINUXPTP_1588_DEFAULT_PROFILE_SYNC_INTERVAL_DEFAULT       0
#define OPEN_LINUXPTP_1588_DEFAULT_PROFILE_SYNC_INTERVAL_MAXIMUM       1

#define OPEN_LINUXPTP_1588_DEFAULT_PROFILE_DELAY_REQ_INTERVAL_MINIMUM  0
#define OPEN_LINUXPTP_1588_DEFAULT_PROFILE_DELAY_REQ_INTERVAL_DEFAULT  0
#define OPEN_LINUXPTP_1588_DEFAULT_PROFILE_DELAY_REQ_INTERVAL_MAXIMUM  5

#define OPEN_LINUXPTP_1588_DEFAULT_PROFILE_PDELAY_REQ_INTERVAL_MINIMUM  0
#define OPEN_LINUXPTP_1588_DEFAULT_PROFILE_PDELAY_REQ_INTERVAL_DEFAULT  0
#define OPEN_LINUXPTP_1588_DEFAULT_PROFILE_PDELAY_REQ_INTERVAL_MAXIMUM  5

/*AES67 Profile */
#define OPEN_LINUXPTP_AES67_PROFILE_DOMAIN_MINIMUM              0
#define OPEN_LINUXPTP_AES67_PROFILE_DOMAIN_DEFAULT              0
#define OPEN_LINUXPTP_AES67_PROFILE_DOMAIN_MAXIMUM              127

#define OPEN_LINUXPTP_AES67_PROFILE_ANNOUNCE_INTERVAL_MINIMUM   0
#define OPEN_LINUXPTP_AES67_PROFILE_ANNOUNCE_INTERVAL_DEFAULT   1
#define OPEN_LINUXPTP_AES67_PROFILE_ANNOUNCE_INTERVAL_MAXIMUM   4

#define OPEN_LINUXPTP_AES67_PROFILE_ANNOUNCE_TIMEOUT_MINIMUM    2
#define OPEN_LINUXPTP_AES67_PROFILE_ANNOUNCE_TIMEOUT_DEFAULT    3
#define OPEN_LINUXPTP_AES67_PROFILE_ANNOUNCE_TIMEOUT_MAXIMUM    10

#define OPEN_LINUXPTP_AES67_PROFILE_SYNC_INTERVAL_MINIMUM      -4
#define OPEN_LINUXPTP_AES67_PROFILE_SYNC_INTERVAL_DEFAULT      -3
#define OPEN_LINUXPTP_AES67_PROFILE_SYNC_INTERVAL_MAXIMUM       1

#define OPEN_LINUXPTP_AES67_PROFILE_DELAY_REQ_INTERVAL_MINIMUM -3
#define OPEN_LINUXPTP_AES67_PROFILE_DELAY_REQ_INTERVAL_DEFAULT  0
#define OPEN_LINUXPTP_AES67_PROFILE_DELAY_REQ_INTERVAL_MAXIMUM  5

#define OPEN_LINUXPTP_AES67_PROFILE_PDELAY_REQ_INTERVAL_MINIMUM  0
#define OPEN_LINUXPTP_AES67_PROFILE_PDELAY_REQ_INTERVAL_DEFAULT  0
#define OPEN_LINUXPTP_AES67_PROFILE_PDELAY_REQ_INTERVAL_MAXIMUM  5

/*SMPTE-2059-2 Profile */
#define OPEN_LINUXPTP_SMPTE_2059_2_PROFILE_DOMAIN_MINIMUM              0
#define OPEN_LINUXPTP_SMPTE_2059_2_PROFILE_DOMAIN_DEFAULT              127
#define OPEN_LINUXPTP_SMPTE_2059_2_PROFILE_DOMAIN_MAXIMUM              127

#define OPEN_LINUXPTP_SMPTE_2059_2_PROFILE_ANNOUNCE_INTERVAL_MINIMUM  -3
#define OPEN_LINUXPTP_SMPTE_2059_2_PROFILE_ANNOUNCE_INTERVAL_DEFAULT   0
#define OPEN_LINUXPTP_SMPTE_2059_2_PROFILE_ANNOUNCE_INTERVAL_MAXIMUM   1

#define OPEN_LINUXPTP_SMPTE_2059_2_PROFILE_ANNOUNCE_TIMEOUT_MINIMUM    2
#define OPEN_LINUXPTP_SMPTE_2059_2_PROFILE_ANNOUNCE_TIMEOUT_DEFAULT    3
#define OPEN_LINUXPTP_SMPTE_2059_2_PROFILE_ANNOUNCE_TIMEOUT_MAXIMUM    10

#define OPEN_LINUXPTP_SMPTE_2059_2_PROFILE_SYNC_INTERVAL_MINIMUM      -7
#define OPEN_LINUXPTP_SMPTE_2059_2_PROFILE_SYNC_INTERVAL_DEFAULT      -3
#define OPEN_LINUXPTP_SMPTE_2059_2_PROFILE_SYNC_INTERVAL_MAXIMUM      -1

#define OPEN_LINUXPTP_SMPTE_2059_2_PROFILE_DELAY_REQ_INTERVAL_MINIMUM -7
#define OPEN_LINUXPTP_SMPTE_2059_2_PROFILE_DELAY_REQ_INTERVAL_DEFAULT -3
#define OPEN_LINUXPTP_SMPTE_2059_2_PROFILE_DELAY_REQ_INTERVAL_MAXIMUM  4

#define OPEN_LINUXPTP_SMPTE_2059_2_PROFILE_PDELAY_REQ_INTERVAL_MINIMUM -7
#define OPEN_LINUXPTP_SMPTE_2059_2_PROFILE_PDELAY_REQ_INTERVAL_DEFAULT -3
#define OPEN_LINUXPTP_SMPTE_2059_2_PROFILE_PDELAY_REQ_INTERVAL_MAXIMUM  4

/*AES-R16-2016 Profile */
#define OPEN_LINUXPTP_AES_R16_2016_PROFILE_DOMAIN_MINIMUM              127
#define OPEN_LINUXPTP_AES_R16_2016_PROFILE_DOMAIN_DEFAULT              127
#define OPEN_LINUXPTP_AES_R16_2016_PROFILE_DOMAIN_MAXIMUM              127

#define OPEN_LINUXPTP_AES_R16_2016_PROFILE_ANNOUNCE_INTERVAL_MINIMUM   0
#define OPEN_LINUXPTP_AES_R16_2016_PROFILE_ANNOUNCE_INTERVAL_DEFAULT   0
#define OPEN_LINUXPTP_AES_R16_2016_PROFILE_ANNOUNCE_INTERVAL_MAXIMUM   1

#define OPEN_LINUXPTP_AES_R16_2016_PROFILE_ANNOUNCE_TIMEOUT_MINIMUM    3
#define OPEN_LINUXPTP_AES_R16_2016_PROFILE_ANNOUNCE_TIMEOUT_DEFAULT    3
#define OPEN_LINUXPTP_AES_R16_2016_PROFILE_ANNOUNCE_TIMEOUT_MAXIMUM    3

#define OPEN_LINUXPTP_AES_R16_2016_PROFILE_SYNC_INTERVAL_MINIMUM      -4
#define OPEN_LINUXPTP_AES_R16_2016_PROFILE_SYNC_INTERVAL_DEFAULT      -3
#define OPEN_LINUXPTP_AES_R16_2016_PROFILE_SYNC_INTERVAL_MAXIMUM       1

#define OPEN_LINUXPTP_AES_R16_2016_PROFILE_DELAY_REQ_INTERVAL_MINIMUM -4
#define OPEN_LINUXPTP_AES_R16_2016_PROFILE_DELAY_REQ_INTERVAL_DEFAULT -3
#define OPEN_LINUXPTP_AES_R16_2016_PROFILE_DELAY_REQ_INTERVAL_MAXIMUM  5

#define OPEN_LINUXPTP_AES_R16_2016_PROFILE_PDELAY_REQ_INTERVAL_MINIMUM -4
#define OPEN_LINUXPTP_AES_R16_2016_PROFILE_PDELAY_REQ_INTERVAL_DEFAULT -3
#define OPEN_LINUXPTP_AES_R16_2016_PROFILE_PDELAY_REQ_INTERVAL_MAXIMUM  5

/*****************************************************************************
 * @purpose   Set the linuxPTP Global Admin Mode.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     mode                @b{(input)}  Global admin mode
 *
 * @returns   OPEN_E_NONE               On success.
 * @returns   OPEN_E_FAIL               On failure.
 * @returns   OPEN_E_PARAM              Invalid argument.
 * @returns   OPEN_E_UNAVAIL            Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpAdminModeSet(openapiClientHandle_t *client_handle, 
                                         OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the linuxPTP Global Admin Mode.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     mode                @b{(output)} Global admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpAdminModeGet(openapiClientHandle_t *client_handle, 
                                         OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the linuxPTP interface Admin Mode.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ifNum               @b{(input)}  Interface number
 * @param     mode                @b{(input)}  Interface admin mode
 *
 * @returns   OPEN_E_NONE               On success.
 * @returns   OPEN_E_FAIL               On failure.
 * @returns   OPEN_E_PARAM              Invalid argument.
 * @returns   OPEN_E_UNAVAIL            Feature not available
 * @returns   OPEN_E_ERROR              If linuxPTP config is done on member
 *                                      unit interface.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpIntfAdminModeSet(openapiClientHandle_t *client_handle, 
                                             uint32_t ifNum, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the linuxPTP interface Admin Mode.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ifNum               @b{(input)}  Interface number
 * @param     mode                @b{(output)} interface admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpIntfAdminModeGet(openapiClientHandle_t *client_handle, 
                                             uint32_t ifNum, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Fetch E2E mean path delay.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     clockMeanPathDelay  @b{(output)} E2E mean path delay
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpClockMeanPathDelayGet(openapiClientHandle_t *client_handle, 
                                                  open_buffdesc *clockMeanPathDelay);

/*****************************************************************************
 * @purpose   Fetch PTP port state.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ifNum               @b{(input)}  Interface number
 * @param     portState           @b{(output)} PTP port state
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpIntfStateGet(openapiClientHandle_t *client_handle, 
                                         uint32_t ifNum, int32_t *portState);

/*****************************************************************************
 * @purpose   Fetch P2P mean path delay.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ifNum               @b{(input)}  Interface number
 * @param     p2pPathDelay        @b{(output)} delay measured using p2p
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpIntfDelayGet(openapiClientHandle_t *client_handle, 
                                         uint32_t ifNum, 
                                         open_buffdesc *p2pPathDelay);

/*****************************************************************************
 * @purpose   Fetch statistics for the given type on an interface.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ifNum               @b{(input)}  Interface number
 * @param     statType            @b{(input)}  statistic type
 * @param     statsCount          @b{(output)} statistics for the given type
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpIntfStatsGet(openapiClientHandle_t *client_handle, 
                                         uint32_t ifNum, 
                                         OPEN_LINUXPTP_STATS_TYPE_t statType, 
                                         int32_t *statsCount);

/*****************************************************************************
 * @purpose   Fetch parent or GM or local clock identity.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     type                @b{(input)}  Clock type
 * @param     identity            @b{(output)} clock identity
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpClockIdGet(openapiClientHandle_t *client_handle, 
                                       OPEN_LINUXPTP_DS_TYPE_t type, 
                                       open_buffdesc *identity);

/*****************************************************************************
 * @purpose   Fetch parent or GM or local clock parameters.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     type                @b{(input)}  Clock type
 * @param     value               @b{(output)} fetched value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpClockParamGet(openapiClientHandle_t *client_handle, 
                                          OPEN_LINUXPTP_DS_TYPE_t type, 
                                          int32_t *value);

/*****************************************************************************
 * @purpose   Configure priority value of the local clock.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     prioType            @b{(input)}  priority 1 or 2
 * @param     prioVal             @b{(input)}  priority value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpBridgePrioritySet(openapiClientHandle_t *client_handle, 
                                              OPEN_LINUXPTP_CLOCK_PRIORITY_t prioType, 
                                              uint32_t prioVal);

/*****************************************************************************
 * @purpose   Get priority value of the local clock.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     prioType            @b{(input)}  priority 1 or 2
 * @param     prioVal             @b{(output)} priority value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpBridgePriorityGet(openapiClientHandle_t *client_handle, 
                                              OPEN_LINUXPTP_CLOCK_PRIORITY_t prioType, 
                                              uint32_t *prioVal);

/*****************************************************************************
 * @purpose   Configure PTP clock operation mode.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     operMode            @b{(input)}  one-step or two-step
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpClockOperModeSet(openapiClientHandle_t *client_handle, 
                                             OPEN_LINUXPTP_CLOCK_OPERATION_t operMode);

/*****************************************************************************
 * @purpose   Get the Clock Operation for the local clock.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     operMode            @b{(output)} one-step or two-step
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpClockOperModeGet(openapiClientHandle_t *client_handle, 
                                             OPEN_LINUXPTP_CLOCK_OPERATION_t *operMode);

/*****************************************************************************
 * @purpose   Configure source IPv4 address to be used for linuxPTP communication.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ipAddr              @b{(input)}  source IPv4 address
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpSourceIPv4AddrSet(openapiClientHandle_t *client_handle, 
                                              uint32_t ipAddr);

/*****************************************************************************
 * @purpose   Get the source IPv4 address used for linuxPTP communication.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ipAddr              @b{(output)} source IPv4 address
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpSourceIPv4AddrGet(openapiClientHandle_t *client_handle, 
                                              uint32_t *ipAddr);

/*****************************************************************************
 * @purpose   Configure source IPv6 address to be used for linuxPTP communication.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ipAddr              @b{(input)}  source IPv6 address
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpSourceIPv6AddrSet(openapiClientHandle_t *client_handle, 
                                              open_in6_addr_t ipAddr);

/*****************************************************************************
 * @purpose   Get the source IPv6 address used for linuxPTP communication.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ipAddr              @b{(output)}  source IPv6 address
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpSourceIPv6AddrGet(openapiClientHandle_t *client_handle, 
                                              open_in6_addr_t *ipAddr);

/*****************************************************************************
 * @purpose   Configure PTP profile.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ptpProfile          @b{(input)}  PTP profile
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpProfileSet(openapiClientHandle_t *client_handle, 
                                       OPEN_LINUXPTP_PROFILES_t ptpProfile);

/*****************************************************************************
 * @purpose   Get the active linuxPTP profile.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ptpProfile          @b{(output)} PTP profile
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpProfileGet(openapiClientHandle_t *client_handle, 
                                       OPEN_LINUXPTP_PROFILES_t *ptpProfile);

/*****************************************************************************
 * @purpose   Configure PTP profile's transport mode.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ptpProfile          @b{(input)}  PTP profile
 * @param     transMode           @b{(input)}  Mode : ethernet or ipv4 or ipv6
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpTransportSet(openapiClientHandle_t *client_handle, 
                                         OPEN_LINUXPTP_PROFILES_t ptpProfile, 
                                         OPEN_LINUXPTP_TRANSPORT_MODE_t transMode);

/*****************************************************************************
 * @purpose   Get the linuxPTP transport for a given profile.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ptpProfile          @b{(input)}  PTP profile
 * @param     transMode           @b{(output)} Mode : ethernet or ipv4 or ipv6
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpTransportGet(openapiClientHandle_t *client_handle, 
                                         OPEN_LINUXPTP_PROFILES_t ptpProfile, 
                                         OPEN_LINUXPTP_TRANSPORT_MODE_t *transMode);

/*****************************************************************************
 * @purpose   Configure announce interval for given PTP profile.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ptpProfile          @b{(input)}  PTP profile
 * @param     announceInterval    @b{(input)}  time interval b/w announce frames
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpAnnounceIntervalSet(openapiClientHandle_t *client_handle, 
                                                OPEN_LINUXPTP_PROFILES_t ptpProfile, 
                                                int32_t announceInterval);

/*****************************************************************************
 * @purpose   Get the configured linuxPTP announce interval for a given profile.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ptpProfile          @b{(input)}  PTP profile
 * @param     announceInterval    @b{(output)} Announce interval
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpAnnounceIntervalGet(openapiClientHandle_t *client_handle,
                                                OPEN_LINUXPTP_PROFILES_t ptpProfile,
                                                int32_t *announceInterval);

/*****************************************************************************
 * @purpose   Configure announce timeout for given PTP profile.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ptpProfile          @b{(input)}  PTP profile
 * @param     announceTimeout     @b{(input)}  Announce timeout
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpAnnounceTimeoutSet(openapiClientHandle_t *client_handle,
                                               OPEN_LINUXPTP_PROFILES_t ptpProfile, 
                                               uint32_t announceTimeout);

/*****************************************************************************
 * @purpose   Get the linuxPTP announce timeout value for a given profile.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ptpProfile          @b{(input)}  PTP profile
 * @param     announceTimeout     @b{(output)} Announce timeout
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpAnnounceTimeoutGet(openapiClientHandle_t *client_handle, 
                                               OPEN_LINUXPTP_PROFILES_t ptpProfile, 
                                               uint32_t *announceTimeout);

/*****************************************************************************
 * @purpose   Configure sync interval for given PTP profile.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ptpProfile          @b{(input)}  PTP profile
 * @param     syncInterval        @b{(input)}  Time interval b/w sync frames
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpSyncIntervalSet(openapiClientHandle_t *client_handle, 
                                            OPEN_LINUXPTP_PROFILES_t ptpProfile, 
                                            int32_t syncInterval);

/*****************************************************************************
 * @purpose   Get the linuxPTP Sync Interval for a given profile.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ptpProfile          @b{(input)}  PTP profile
 * @param     syncInterval        @b{(output)} Time interval b/w sync frames
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpSyncIntervalGet(openapiClientHandle_t *client_handle, 
                                            OPEN_LINUXPTP_PROFILES_t ptpProfile, 
                                            int32_t *syncInterval);

/*****************************************************************************
 * @purpose   Configure delay request interval for given PTP profile.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ptpProfile          @b{(input)}  PTP profile
 * @param     delayReqInterval    @b{(input)}  Time interval b/w delay-request frames
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpDelayRequestIntervalSet(openapiClientHandle_t *client_handle, 
                                                    OPEN_LINUXPTP_PROFILES_t ptpProfile, 
                                                    int32_t delayReqInterval);

/*****************************************************************************
 * @purpose   Get the linuxPTP delay request Interval for a given profile.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ptpProfile          @b{(input)}  PTP profile
 * @param     delayReqInterval    @b{(output)} Time interval b/w delay-request frames
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpDelayRequestIntervalGet(openapiClientHandle_t *client_handle, 
                                                    OPEN_LINUXPTP_PROFILES_t ptpProfile, 
                                                    int32_t *delayReqInterval);

/*****************************************************************************
 * @purpose   Configure peer delay request interval for given PTP profile.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ptpProfile          @b{(input)}  PTP profile
 * @param     pDelayReqInterval   @b{(input)}  Time interval b/w peer delay-request frames
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpPdelayRequestIntervalSet(openapiClientHandle_t *client_handle, 
                                                     OPEN_LINUXPTP_PROFILES_t ptpProfile, 
                                                     int32_t pDelayReqInterval);

/*****************************************************************************
 * @purpose   Get the linuxPTP peer delay request Interval for a given profile.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ptpProfile          @b{(input)}  PTP profile
 * @param     pDelayReqInterval   @b{(output)} Time interval b/w peer delay-request frames
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpPdelayRequestIntervalGet(openapiClientHandle_t *client_handle, 
                                                     OPEN_LINUXPTP_PROFILES_t ptpProfile, 
                                                     int32_t *pDelayReqInterval);

/*****************************************************************************
 * @purpose   Configure domain number for given PTP profile.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ptpProfile          @b{(input)}  PTP profile
 * @param     domain              @b{(input)}  PTP domain identifier
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpDomainSet(openapiClientHandle_t *client_handle, 
                                      OPEN_LINUXPTP_PROFILES_t ptpProfile, 
                                      uint32_t domain);

/*****************************************************************************
 * @purpose   Get the linuxPTP Domain for a given profile.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ptpProfile          @b{(input)}  PTP profile
 * @param     domain              @b{(output)} PTP domain identifier
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpDomainGet(openapiClientHandle_t *client_handle, 
                                      OPEN_LINUXPTP_PROFILES_t ptpProfile, 
                                      uint32_t *domain);

/*****************************************************************************
 * @purpose   Configure PTP delay mechanism.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ptpProfile          @b{(input)}  PTP profile
 * @param     delayMech           @b{(input)}  Either end-to-end or peer-to-peer delay calculation mechanism
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpDelayMechanismSet(openapiClientHandle_t *client_handle, 
                                              OPEN_LINUXPTP_PROFILES_t ptpProfile, 
                                              OPEN_LINUXPTP_DELAY_MECHANISM_t delayMech);

/*****************************************************************************
 * @purpose   Get the linuxPTP delay mechanism for a given profile.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ptpProfile          @b{(input)}  PTP profile
 * @param     delayMech           @b{(output)} Delay Mechanism, E2E or P2P
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpDelayMechanismGet(openapiClientHandle_t *client_handle, 
                                              OPEN_LINUXPTP_PROFILES_t ptpProfile, 
                                              OPEN_LINUXPTP_DELAY_MECHANISM_t *delayMech);

/*****************************************************************************
 * @purpose   Configure PTP logging severity level.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     level               @b{(input)}  severity level
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpLoggingSet(openapiClientHandle_t *client_handle, 
                                       uint32_t level);

/*****************************************************************************
 * @purpose   Fetch PTP logging severity level.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     level               @b{(output)} severity level
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpLoggingGet(openapiClientHandle_t *client_handle, 
                                       uint32_t *level);

/*****************************************************************************
 * @purpose   Clear PTP statistics on an interface.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ifNum               @b{(input)}  Interface number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpIntfStatsClear(openapiClientHandle_t *client_handle, 
                                           uint32_t ifNum);

/*****************************************************************************
 * @purpose   Get PTP interface operational status.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ifNum               @b{(input)}  Interface number
 * @param     status              @b{(output)} Operational status
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpIntfOperStateGet(openapiClientHandle_t *client_handle, 
                                             uint32_t ifNum, OPEN_BOOL_t *status);

/*****************************************************************************
 * @purpose   Check whether the interface is eligible for linuxptp operation.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     ifNum               @b{(input)}  Interface number
 * @param     isValid             @b{(output)} Valid status
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpIsValidIntf(openapiClientHandle_t *client_handle, 
                                        uint32_t ifNum, OPEN_BOOL_t *isValid);

/*****************************************************************************
 * @purpose   Check if the passed domain identifier is configured.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     domain              @b{(input)}  domain identifier
 * @param     isValidDomain       @b{(output)} Valid status
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpIsDomainExists(openapiClientHandle_t *client_handle, 
                                           uint32_t domain, 
                                           OPEN_BOOL_t *isValidDomain);

/*****************************************************************************
 * @purpose   Fetch time properties information.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     type                @b{(input)}  Time properties type
 * @param     value               @b{(output)} Value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpTimePropGet(openapiClientHandle_t *client_handle, 
                                        OPEN_LINUXPTP_TIME_PROPERTIES_TYPE_t type, 
                                        int32_t *value);

/*****************************************************************************
 * @purpose   Fetch number of boundary clocks between local clock and foreign master.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     stepsRemoved        @b{(output)} hops to foreign master
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpClockStepsRemovedGet(openapiClientHandle_t *client_handle, 
                                                 uint32_t *stepsRemoved);

/*****************************************************************************
 * @purpose   Fetch current clock offset in time between slave and master.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     offsetFromMaster    @b{(output)} clock offset from master clock
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpClockOffsetFromMaster(openapiClientHandle_t *client_handle, 
                                                  open_buffdesc *offsetFromMaster);

/*****************************************************************************
 * @purpose   Status of the mode the clock is operating in master or slave.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     clockRole           @b{(output)} Clock role
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpClockRoleGet(openapiClientHandle_t *client_handle, 
                                         uint32_t *clockRole);

/*****************************************************************************
 * @purpose   Configure PTP path trace TLV.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     traceMode           @b{(input)}  Trace mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpPathTraceTlvSet(openapiClientHandle_t *client_handle, 
                                            OPEN_BOOL_t traceMode);

/*****************************************************************************
 * @purpose   Fetch PTP path trace TLV status.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     traceMode           @b{(output)} Trace mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpPathTraceTlvGet(openapiClientHandle_t *client_handle, 
                                            OPEN_BOOL_t *traceMode);

/*****************************************************************************
 * @purpose   Fetch next clock identity in the path.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     type                @b{(input)}  Parent identity type
 * @param     identity            @b{(output)} Clock identity
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpPathTraceClockIdGetNext(openapiClientHandle_t *client_handle,  
                                                    OPEN_LINUXPTP_DS_TYPE_t type, 
                                                    open_buffdesc *identity);

/*****************************************************************************
 * @purpose   Check whether Linux PTP is supported.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API.
 * @param     isLinuxptpSupported @b{(output)} is Linux PTP supported or not.
 *
 * @returns   OPEN_E_NONE         On success.
 * @returns   OPEN_E_FAIL         On failure.
 * @returns   OPEN_E_PARAM        Invalid argument.
 * @returns   OPEN_E_UNAVAIL      Feature not available.
 *
 * @notes     This API needs to be invoked before calling the Linux PTP OpEN APIs.
 *            If Linux PTP is not supported then OpEN APIs should not be called.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpIsSupported(openapiClientHandle_t *client_handle,
                                        OPEN_BOOL_t           *isLinuxptpSupported);

/*****************************************************************************
 * @purpose   Get LinuxPTP process running status.
 *
 * @param     client_handle         @b{(input)}  client handle from registration API
 * @param     processStatus         @b{(output)} Running status of LinuxPTP process
 *
 * @returns   OPEN_E_NONE           On success.
 * @returns   OPEN_E_FAIL           On failure.
 * @returns   OPEN_E_PARAM          Invalid argument.
 * @returns   OPEN_E_UNAVAIL        Feature not available.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpProcessRunningStatusGet(openapiClientHandle_t *client_handle, 
                                                    OPEN_BOOL_t *processStatus);

/*****************************************************************************
 * @purpose   Get LinuxPTP parent clock operation mode.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     operMode       @b{(output)} Parent clock operation mode: one-step
 *                                        or two-step
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Not available.
 * @returns   OPEN_E_EMPTY            Not applicable.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpParentClockOperModeGet(openapiClientHandle_t *client_handle, 
                                                   OPEN_LINUXPTP_CLOCK_OPERATION_t *operMode);

/*****************************************************************************
 * @purpose   Get LinuxPTP IPv6 multicast destination scope.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     mcScope        @b{(output)} scope range is 0x0-0xf
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpIpv6McScopeGet(openapiClientHandle_t *client_handle, 
                                           uint32_t *mcScope);

/*****************************************************************************
 * @purpose   Set LinuxPTP IPv6 multicast desination scope.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     mcScope        @b{(input)}  allowed scope values: 0x2,0x4,0x5
 *                                        0x8 and 0xE

 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.28
 *
 * @end
 *****************************************************************************/
open_error_t openapiLinuxptpIpv6McScopeSet(openapiClientHandle_t *client_handle, 
                                           open_buffdesc *mcScope);

#endif /* OPENAPI_LINUXPTP_H_INCLUDED */

