/**********************************************************************
*
*  Copyright 2022-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_loop_protect.h
*
* @purpose   This code implements the OpEN LoopProtect API
*
* @component OPEN
*
* @create    12/23/2022
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_LOOP_PROTECT_H_INCLUDED
#define OPENAPI_LOOP_PROTECT_H_INCLUDED

#include "openapi_common.h" 

typedef enum
{
  OPEN_LOOP_PROTECT_ACTION_LOGMSG = 0,
  OPEN_LOOP_PROTECT_ACTION_SHUTDOWN,
  OPEN_LOOP_PROTECT_ACTION_LOGMSG_SHUTDOWN
} OPEN_LOOP_PROTECT_PORT_ACTION_t;

/*****************************************************************************
 * @purpose   Sets the LoopProtection global mode.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     adminMode      @b{(input)}  LoopProtection global admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectAdminModeSet(openapiClientHandle_t *client_handle, 
                                            OPEN_CONTROL_t adminMode);

/*****************************************************************************
 * @purpose   Gets the LoopProtection global mode.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     adminMode      @b{(output)} LoopProtection global admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectAdminModeGet(openapiClientHandle_t *client_handle, 
                                            OPEN_CONTROL_t *adminMode);

/*****************************************************************************
 * @purpose   Sets the transmission timer for the LoopProtection PDU's.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     timerVal       @b{(input)}  Timer value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectTxTimerSet(openapiClientHandle_t *client_handle, 
                                          uint32_t timerVal);

/*****************************************************************************
 * @purpose   Gets the transmission timer for the LoopProtection PDU's.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     timerVal       @b{(output)} Timer value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectTxTimerGet(openapiClientHandle_t *client_handle, 
                                          uint32_t *timerVal);

/*****************************************************************************
 * @purpose   Sets the maximum number of PDU's to be received on a port before
 *            action is taken.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     val            @b{(input)}  Maximum number of PDU's
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectMaxPduReceiveSet(openapiClientHandle_t *client_handle,
                                                uint32_t val);

/*****************************************************************************
 * @purpose   Gets the maximum number of PDU's to be received on a port before 
 *            action is taken.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     val            @b{(output)} Maximum number of PDU's
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectMaxPduReceiveGet(openapiClientHandle_t *client_handle,
                                                uint32_t *val);

/*****************************************************************************
 * @purpose   Sets the transmission mode for the LoopProtection PDU's.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Interface number
 * @param     mode           @b{(input)}  Transmit mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectPortTxModeSet(openapiClientHandle_t *client_handle,
                                             uint32_t ifNum, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Gets the transmission timer Mode for the LoopProtection PDU's.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Interface number
 * @param     mode           @b{(output)} Transmit mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectPortTxModeGet(openapiClientHandle_t *client_handle,
                                             uint32_t ifNum, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Sets the disable duration for the port.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     val            @b{(input)}  Disable duration in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectDisableTimerSet(openapiClientHandle_t *client_handle,
                                               uint32_t val);

/*****************************************************************************
 * @purpose   Gets the disable duration for the port.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     val            @b{(output)} Disable duration in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectDisableTimerGet(openapiClientHandle_t *client_handle,
                                               uint32_t *val);

/*****************************************************************************
 * @purpose   Sets the loop protection admin mode for the interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Interface number
 * @param     mode           @b{(input)}  Interface admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectPortAdminModeSet(openapiClientHandle_t *client_handle,
                                                uint32_t ifNum, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Gets the loop protection admin mode for the interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Interface number
 * @param     mode           @b{(output)} Interface admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectPortAdminModeGet(openapiClientHandle_t *client_handle,
                                                uint32_t ifNum, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Sets the port action for the the specified interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Interface number
 * @param     mode           @b{(input)}  Port action to be configured on the port
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectPortActionSet(openapiClientHandle_t *client_handle,
                                             uint32_t ifNum, 
                                             OPEN_LOOP_PROTECT_PORT_ACTION_t mode);

/*****************************************************************************
 * @purpose   Gets the port action for the the specified interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Interface number
 * @param     mode           @b{(output)} Port action configured on the port
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectPortActionGet(openapiClientHandle_t *client_handle,
                                             uint32_t ifNum,
                                             OPEN_LOOP_PROTECT_PORT_ACTION_t *mode);

/*****************************************************************************
 * @purpose   Sets the VLAN ID and the TPID to be used for detecting a loop on
 *            the specified interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Interface number
 * @param     tpid           @b{(input)}  TPID value
 * @param     vlanId         @b{(input)}  VLAN ID
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectPortVlanPropertiesSet(openapiClientHandle_t *client_handle, 
                                                     uint32_t ifNum, uint32_t tpid,
                                                     uint32_t vlanId);

/*****************************************************************************
 * @purpose   Gets the VLAN ID and the TPID used for detecting a loop on the 
 *            specified interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Interface number
 * @param     tpid           @b{(output)} TPID value
 * @param     vlanId         @b{(output)} VLAN ID
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectPortVlanPropertiesGet(openapiClientHandle_t *client_handle, 
                                                     uint32_t ifNum, uint32_t *tpid,
                                                     uint32_t *vlanId);

/*****************************************************************************
 * @purpose   Clears statistics of the specified interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Interface number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectPortStatsClear(openapiClientHandle_t *client_handle, 
                                              uint32_t ifNum);

/*****************************************************************************
 * @purpose   Clears the statistics for all interfaces.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectAllStatsClear(openapiClientHandle_t *client_handle);

/*****************************************************************************
 * @purpose   Gets the loop Count for the specified interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Interface number
 * @param     loopCount      @b{(output)} Loop count value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectPortloopDetectCountGet(openapiClientHandle_t *client_handle, 
                                                      uint32_t ifNum, 
                                                      uint32_t *loopCount);

/*****************************************************************************
 * @purpose   Gets the status of loop on the specified interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Interface number
 * @param     status         @b{(output)} Loop status
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectIsLoopDetected(openapiClientHandle_t *client_handle,
                                              uint32_t ifNum, OPEN_BOOL_t *status);

/*****************************************************************************
 * @purpose   Gets the port status. whether port is in D-Disable state due to 
 *            loop protection.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Interface number
 * @param     status         @b{(output)} Port status
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectIsPortDisabled(openapiClientHandle_t *client_handle,
                                              uint32_t ifNum, OPEN_BOOL_t *status);

/*****************************************************************************
 * @purpose   Gets the time since last loop occured for the specified interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Interface number
 * @param     timeVal        @b{(output)} Time value
 * @param     needUtc        @b{(input)}  Flag L7_TRUE for UTC time and 
 *                                        L7_FALSE for raw time
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Use @ref openapiTimeToCalenderTimeGet to convert timeVal to
 *            calender time - yyyy-mm-dd hh:mm:ss. 
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectPortTimeSinceLastLoopGet(openapiClientHandle_t *client_handle, 
                                                        uint32_t ifNum, 
                                                        uint32_t *timeVal,
                                                        OPEN_BOOL_t needUtc);

/*****************************************************************************
 * @purpose   Gets the transmit loop count for the specified interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Interface number
 * @param     count          @b{(output)} Transmit loop count
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectPortPacketTxCountGet(openapiClientHandle_t *client_handle, 
                                                    uint32_t ifNum, uint32_t *count);

/*****************************************************************************
 * @purpose   Gets the receive loop count for the specified interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Interface number
 * @param     count          @b{(output)} Receive loop count
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectPortPacketRxCountGet(openapiClientHandle_t *client_handle, 
                                                    uint32_t ifNum, uint32_t *count);

/*****************************************************************************
 * @purpose   Gets the status of an interface whether port is valid for
 *            configuration for loop protection.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Interface number
 * @param     status         @b{(output)} Port status
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectIsValidIntf(openapiClientHandle_t *client_handle,
                                           uint32_t ifNum, OPEN_BOOL_t *status);

/*****************************************************************************
 * @purpose   Gets the first interface that is valid for loop protection.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(output)} Interface number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectValidIntfFirstGet(openapiClientHandle_t *client_handle, 
                                                 uint32_t *ifNum);

/*****************************************************************************
 * @purpose   Gets the next interface that is valid for loop protection.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Interface number
 * @param     nextIfNum      @b{(output)} Next valid interface number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiLoopProtectValidIntfNextGet(openapiClientHandle_t *client_handle, 
                                                uint32_t ifNum, uint32_t *nextIfNum);

#endif /* OPENAPI_LOOP_PROTECT_H_INCLUDED */

