/**********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_macal.h
*
* @purpose   This code implements the OpEN Management Access Control and 
*            Administration List (MACAL) API
*
* @component OPEN
*
* @create    01/26/2023
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_MACAL_H_INCLUDED
#define OPENAPI_MACAL_H_INCLUDED

#include "openapi_common.h" 

typedef enum
{
  OPEN_MACAL_ACTION_DENY,
  OPEN_MACAL_ACTION_PERMIT
} OPEN_MACAL_ACTION_t;

typedef enum
{
  OPEN_MACAL_PROTOCOL_NONE,
  OPEN_MACAL_PROTOCOL_TELNET,
  OPEN_MACAL_PROTOCOL_HTTP,
  OPEN_MACAL_PROTOCOL_HTTPS,
  OPEN_MACAL_PROTOCOL_SNMP,
  OPEN_MACAL_PROTOCOL_SSH,
  OPEN_MACAL_PROTOCOL_TFTP,
  OPEN_MACAL_PROTOCOL_SNTP,
  OPEN_MACAL_PROTOCOL_JAVA,
  OPEN_MACAL_PROTOCOL_NTP,
  OPEN_MACAL_CUSTOM_PORT
} OPEN_MACAL_SERVICE_t;

typedef enum
{
  OPEN_MACAL_SERVICE = (1 << 0),
  OPEN_MACAL_SRCIP = (1 << 1),
  OPEN_MACAL_SRCIP_MASK = (1 << 2),
  OPEN_MACAL_VLAN = (1 << 3),
  OPEN_MACAL_IF = (1 << 4),
  OPEN_MACAL_SERVICE_PORT = (1 << 5),
  OPEN_MACAL_TCP_UDP_PORT = (1 << 6)

} OPEN_MACAL_RULE_FIELD_t;

#define OPEN_MACAL_REQUEST_PRIORITY 0xFFFF 
#define OPEN_MACAL_MIN_PRIORITY 1
#define OPEN_MACAL_MAX_PRIORITY 64

/*****************************************************************************
 * @purpose   To create a new management ACAL.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalListCreate(openapiClientHandle_t *client_handle, 
                                    open_buffdesc *macalName);

/*****************************************************************************
 * @purpose   To delete an existing management ACAL.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalListDelete(openapiClientHandle_t *client_handle, 
                                    open_buffdesc *macalName);

/*****************************************************************************
 * @purpose   Sets VRF of a management ACAL.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     vrfName               @b{(input)}  VRF name
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalListVrfSet(openapiClientHandle_t *client_handle, 
                                    open_buffdesc *vrfName);

/*****************************************************************************
 * @purpose   Gets VRF of a management ACAL.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     vrfName               @b{(input)}  VRF name
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalListVrfGet(openapiClientHandle_t *client_handle, 
                                    open_buffdesc *vrfName);

/*****************************************************************************
 * @purpose   Activate a management ACAL.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalActivate(openapiClientHandle_t *client_handle, 
                                  open_buffdesc *macalName);

/*****************************************************************************
 * @purpose   Deactivate a management ACAL.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalDeactivate(openapiClientHandle_t *client_handle, 
                                    open_buffdesc *macalName);

/*****************************************************************************
 * @purpose   To add a management ACAL rule.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 * @param     macalRulePrio         @b{(inout)}  MACAL rule priority. 
 *                                               This is used as the rule index
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 * @notes     If macalRulePrio input value is 0xFFFF then application will
 *            allocate available priority/index and set the value in
 *            macalRulePrio. If input is other than 0xFFFF, application will
 *            create rule with the given priority.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalRuleEntryAdd(openapiClientHandle_t *client_handle, 
                                      open_buffdesc *macalName, 
                                      uint32_t *macalRulePrio);

/*****************************************************************************
 * @purpose   To remove a management rule from an existing ACAL.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 * @param     macalRulePrio         @b{(input)}  MACAL rule priority. 
 *                                               This is used as the rule index
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalRuleRemove(openapiClientHandle_t *client_handle, 
                                    open_buffdesc *macalName, 
                                    uint32_t macalRulePrio);

/*****************************************************************************
 * @purpose   To add an action to management ACAL rule.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 * @param     macalRulePrio         @b{(input)}  MACAL rule priority. 
 *                                               This is used as the rule index
 * @param     action                @b{(input)}  Action (permit/deny)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalRuleActionAdd(openapiClientHandle_t *client_handle, 
                                       open_buffdesc *macalName,
                                       uint32_t macalRulePrio, 
                                       OPEN_MACAL_ACTION_t action);

/*****************************************************************************
 * @purpose   To add the source ip address and net mask to a management ACAL rule.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 * @param     macalRulePrio         @b{(input)}  MACAL rule priority. 
 *                                               This is used as the rule index
 * @param     ipAddr                @b{(input)}  IP address
 * @param     mask                  @b{(input)}  Net Mask
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalRuleSrcIpMaskAdd(openapiClientHandle_t *client_handle, 
                                          open_buffdesc *macalName, 
                                          uint32_t macalRulePrio, 
                                          uint32_t ipAddr, uint32_t mask);

/*****************************************************************************
 * @purpose   To add an interface (ethernet/LAG) to an existing management ACAL rule.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 * @param     macalRulePrio         @b{(input)}  MACAL rule priority. 
 *                                               This is used as the rule index
 * @param     ifNum                 @b{(input)}  Interface number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalRuleInterfaceAdd(openapiClientHandle_t *client_handle, 
                                          open_buffdesc *macalName, 
                                          uint32_t macalRulePrio, uint32_t ifNum);

/*****************************************************************************
 * @purpose   To add a vlan to an existing management ACAL rule.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 * @param     macalRulePrio         @b{(input)}  MACAL rule priority. 
 *                                               This is used as the rule index
 * @param     vlanId                @b{(input)}  Vlan ID to be added
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalRuleVlanAdd(openapiClientHandle_t *client_handle, 
                                     open_buffdesc *macalName, 
                                     uint32_t macalRulePrio, uint32_t vlanId);

/*****************************************************************************
 * @purpose   To add a serviceport to an existing management ACAL rule.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 * @param     macalRulePrio         @b{(input)}  MACAL rule priority. 
 *                                               This is used as the rule index
 * @param     isServicePort         @b{(input)}  TRUE if the rule is for the service 
 *                                               port
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalRuleServicePortSet(openapiClientHandle_t *client_handle, 
                                            open_buffdesc *macalName, 
                                            uint32_t macalRulePrio, 
                                            OPEN_BOOL_t isServicePort);

/*****************************************************************************
 * @purpose   Check if this management ACAL rule is for the service port.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 * @param     macalRulePrio         @b{(input)}  MACAL rule priority. 
 *                                               This is used as the rule index
 * @param     isRuleForServicePort  @b{(output)} TRUE if the rule is for the service 
 *                                               port
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalRuleIsServicePortSet(openapiClientHandle_t *client_handle, 
                                              open_buffdesc *macalName, 
                                              uint32_t macalRulePrio, 
                                              OPEN_BOOL_t *isRuleForServicePort);

/*****************************************************************************
 * @purpose   To add a service to management ACAL rule.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 * @param     macalRulePrio         @b{(input)}  MACAL rule priority. 
 *                                               This is used as the rule index
 * @param     service               @b{(input)}  The service to be added
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalRuleServiceAdd(openapiClientHandle_t *client_handle, 
                                        open_buffdesc *macalName, 
                                        uint32_t macalRulePrio, 
                                        OPEN_MACAL_SERVICE_t service);

/*****************************************************************************
 * @purpose   Get the name of the management ACAL already existing.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(output)} MACAL name
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalNameGet(openapiClientHandle_t *client_handle, 
                                 open_buffdesc *macalName);

/*****************************************************************************
 * @purpose   Get the active management ACAL.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(output)} Active MACAL name
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalActiveListGet(openapiClientHandle_t *client_handle, 
                                       open_buffdesc *macalName);

/*****************************************************************************
 * @purpose   Get the first management ACAL rule.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 * @param     macalRulePrio         @b{(output)} MACAL rule priority.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalRuleFirstGet(openapiClientHandle_t *client_handle, 
                                      open_buffdesc *macalName, 
                                      uint32_t *macalRulePrio);

/*****************************************************************************
 * @purpose   Get the next management ACAL rule.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 * @param     macalRulePrio         @b{(input)}  MACAL rule priority. 
 *                                               This is used as index
 * @param     nextMacalRulePrio     @b{(output)} Next MACAL rule index
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalRuleNextGet(openapiClientHandle_t *client_handle, 
                                     open_buffdesc *macalName, 
                                     uint32_t macalRulePrio, 
                                     uint32_t *nextMacalRulePrio);

/*****************************************************************************
 * @purpose   Check if this management ACAL rule with given priority exists.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 * @param     macalRulePrio         @b{(input)}  MACAL rule priority. 
 *                                               This is used as the rule index
 * @param     isIndexExists         @b{(output)} TRUE if the rule exists
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalRuleExists(openapiClientHandle_t *client_handle, 
                                    open_buffdesc *macalName, 
                                    uint32_t macalRulePrio, 
                                    OPEN_BOOL_t *isIndexExists);

/*****************************************************************************
 * @purpose   Get the action for a management ACAL rule.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 * @param     macalRulePrio         @b{(input)}  MACAL rule priority. 
 *                                               This is used as the rule index
 * @param     action                @b{(output)} Action (permit/deny)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalRuleActionGet(openapiClientHandle_t *client_handle, 
                                       open_buffdesc *macalName, 
                                       uint32_t macalRulePrio, 
                                       OPEN_MACAL_ACTION_t *action);

/*****************************************************************************
 * @purpose   Get the source ip address and net mask of a management ACAL rule.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 * @param     macalRulePrio         @b{(input)}  MACAL rule priority. 
 *                                               This is used as the rule index
 * @param     ipAddr                @b{(output)} IP address
 * @param     mask                  @b{(output)} Net Mask
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalRuleSrcIpMaskGet(openapiClientHandle_t *client_handle, 
                                          open_buffdesc *macalName, 
                                          uint32_t macalRulePrio, 
                                          uint32_t *ipAddr, uint32_t *mask);

/*****************************************************************************
 * @purpose   Get the interface (ethernet/LAG) of an existing management ACAL rule.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 * @param     macalRulePrio         @b{(input)}  MACAL rule priority. 
 *                                               This is used as the rule index
 * @param     ifNum                 @b{(output)} Interface number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalRuleInterfaceGet(openapiClientHandle_t *client_handle, 
                                          open_buffdesc *macalName, 
                                          uint32_t macalRulePrio, 
                                          uint32_t *ifNum);

/*****************************************************************************
 * @purpose   Gets the vlan ID of an existing management ACAL rule.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 * @param     macalRulePrio         @b{(input)}  MACAL rule priority. 
 *                                               This is used as the rule index
 * @param     vlanId                @b{(output)} Vlan ID
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalRuleVlanGet(openapiClientHandle_t *client_handle, 
                                     open_buffdesc *macalName, 
                                     uint32_t macalRulePrio, 
                                     uint32_t *vlanId);

/*****************************************************************************
 * @purpose   Get the service of a management ACAL rule.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 * @param     macalRulePrio         @b{(input)}  MACAL rule priority. 
 *                                               This is used as the rule index
 * @param     service               @b{(output)} Service like telnet, ssh ...etc
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalRuleServiceGet(openapiClientHandle_t *client_handle, 
                                        open_buffdesc *macalName, 
                                        uint32_t macalRulePrio, 
                                        OPEN_MACAL_SERVICE_t *service);

/*****************************************************************************
 * @purpose   Get the service port statistics for a management ACAL rule.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 * @param     macalRulePrio         @b{(input)}  MACAL rule priority. 
 *                                               This is used as the rule index
 * @param     packets               @b{(output)} Number of packets
 * @param     bytes                 @b{(output)} Number of bytes
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalServicePortRuleStatisticsGet(openapiClientHandle_t *client_handle, 
                                                      open_buffdesc *macalName, 
                                                      uint32_t macalRulePrio, 
                                                      uint32_t *packets, 
                                                      uint32_t *bytes);

/*****************************************************************************
 * @purpose   Get the TCP/UDP port number of a management ACAL rule.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 * @param     macalRulePrio         @b{(input)}  MACAL rule priority. 
 *                                               This is used as the rule index
 * @param     portNum               @b{(output)} TCP/UDP port number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalRuleTcpUdpPortNumberGet(openapiClientHandle_t *client_handle, 
                                                 open_buffdesc *macalName, 
                                                 uint32_t macalRulePrio, 
                                                 uint32_t *portNum);

/*****************************************************************************
 * @purpose   Add the TCP/UDP port number to a management ACAL rule.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 * @param     macalRulePrio         @b{(input)}  MACAL rule priority. 
 *                                               This is used as the rule index
 * @param     portNum               @b{(input)}  TCP/UDP port number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalRuleTcpUdpPortNumberAdd(openapiClientHandle_t *client_handle, 
                                                 open_buffdesc *macalName, 
                                                 uint32_t macalRulePrio, 
                                                 uint32_t portNum);

/*****************************************************************************
 * @purpose   Checks to see if a user has configured a field in a management ACAL rule.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 * @param     macalRulePrio         @b{(input)}  MACAL rule priority. 
 *                                               This is used as the rule index
 * @param     ruleField             @b{(input)}  Management ACAL rule field
 * @param     isConfigured          @b{(output)} TRUE if the specified filed is configured
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalIsFieldConfigured(openapiClientHandle_t *client_handle, 
                                           open_buffdesc *macalName, 
                                           uint32_t macalRulePrio, 
                                           OPEN_MACAL_RULE_FIELD_t ruleField, 
                                           OPEN_BOOL_t *isConfigured);

/*****************************************************************************
 * @purpose   Check if a MACAL list is created.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     isCreated             @b{(output)} TRUE if the MACAL is created
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalIsAnyListCreated(openapiClientHandle_t *client_handle, 
                                          OPEN_BOOL_t *isCreated);

/*****************************************************************************
 * @purpose   Check if a MACAL list is activated.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     isActivated           @b{(output)} TRUE if the MACAL is activated
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalIsAnyListActivated(openapiClientHandle_t *client_handle, 
                                            OPEN_BOOL_t *isActivated);

/*****************************************************************************
 * @purpose   Get the number of packets filtered.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     filtPackets           @b{(output)} Number of packets filtered by MACAL
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalStatsPacketsFilteredGet(openapiClientHandle_t *client_handle, 
                                                 uint32_t *filtPackets);

/*****************************************************************************
 * @purpose   Get the macal deny counter interval.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     interval              @b{(output)} Deny counter interval
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalDenyCounterIntervalGet(openapiClientHandle_t *client_handle, 
                                                uint32_t *interval);

/*****************************************************************************
 * @purpose   Set the macal deny counter interval.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     interval              @b{(input)}  Deny counter interval
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalDenyCounterIntervalSet(openapiClientHandle_t *client_handle, 
                                                uint32_t interval);

/*****************************************************************************
 * @purpose   Get the first available priority (index) for the specified management 
 *            ACAL.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     macalName             @b{(input)}  MACAL name
 * @param     macalRulePrio         @b{(output)} MACAL rule priority. This is used as the rule index
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalRulePriorityIndexGet(openapiClientHandle_t *client_handle, 
                                              open_buffdesc *macalName, 
                                              uint32_t *macalRulePrio);

/*****************************************************************************
 * @purpose   Checks the specified VRF name is valid or not.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     vrfName               @b{(input)}  VRF name
 * @param     isValid               @b{(output)} TRUE if the VRF is valid
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalVrfNameValid(openapiClientHandle_t *client_handle, 
                                      open_buffdesc *vrfName, 
                                      OPEN_BOOL_t *isValid);

/*****************************************************************************
 * @purpose   Clears  macal service port statistics.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalClearServicePortStatistics(openapiClientHandle_t *client_handle);

/*****************************************************************************
 * @purpose   Check if the Net Filter table support is present on this platform.
 *
 * @param     client_handle         @b{(input)}  Client handle from registration API
 * @param     isSupported           @b{(output)} TRUE if the platform supports net 
 *                                               filter table
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiMacalIsNetFilterTablesFeatureSupported(openapiClientHandle_t *client_handle, 
                                                           OPEN_BOOL_t *isSupported);


#endif /* OPENAPI_MACAL_H_INCLUDED */

