/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_mpls.h
*
* @purpose   MPLS Configuration and Status 
*
* @component OpEN
*
* @create    01/16/2014
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_MPLS_H_INCLUDED
#define OPENAPI_MPLS_H_INCLUDED

#include <stdio.h>
#include <stddef.h>
#include <sys/un.h>

#include "openapi_common.h"

/** MPLS LFDB protocol type */
typedef enum
{
  OPEN_MPLS_PROTO_STATIC  = 1, /**< Static label.                        */
  OPEN_MPLS_PROTO_DYNAMIC = 2, /**< Dynamic label.                       */
  OPEN_MPLS_PROTO_BGP     = 3, /**< Label created with BGP.              */
  OPEN_MPLS_PROTO_LAST    = 4  /**< This entry must be last in the list. */
} OPEN_MPLS_LFDB_PROTOCOL_t;

/** MPLS LFDB label type */
typedef enum
{
  OPEN_MPLS_TYPE_LAYER_2 = 1, /**< L2 label type                               */
  OPEN_MPLS_TYPE_IPV4    = 2, /**< L3 label type that works with IPv4 address. */
  OPEN_MPLS_TYPE_IPV6    = 3, /**< L3 label type that works with IPv6 address. */
  OPEN_MPLS_TYPE_LAST    = 4  /**< This entry must be last in the list.        */
} OPEN_MPLS_LFDB_ENTRY_TYPE_t;

/** MPLS LFDB label action */
typedef enum
{
  OPEN_MPLS_LABEL_SWAP     = 1, /**< Swap action.     */
  OPEN_MPLS_LABEL_POP      = 2, /**< Pop action.      */
  OPEN_MPLS_LABEL_LAST_POP = 3  /**< Last pop action. */
} OPEN_MPLS_LFDB_LABEL_ACTION_t;

/** MPLS LFDB label status */
typedef enum
{
  OPEN_MPLS_LABEL_IN_HARDWARE     = 0, /**< A label was applied to HW.                             */
  OPEN_MPLS_LABEL_NOT_IN_HARDWARE = 1  /**< A label was not applied into HW. The reason is located
                                        **  in @ref OPEN_MPLS_LFDB_NOT_INSERTED_REASON_t.          */
} OPEN_MPLS_LFDB_IN_HARDWARE_STATUS_t;

/** Reason for which the MPLS LFDB label is not applied to HW */
typedef enum
{
  OPEN_MPLS_NONE           = 0, /**< Label is inserted in hardware.                                   */
  OPEN_MPLS_UNKNOWN_SUBNET = 1, /**< L3 label does not match any subnet.                              */
  OPEN_MPLS_LOCAL_SUBNET   = 3, /**< L3 label matches a locally attached subnet.                      */
  OPEN_MPLS_NO_RESOURCES   = 4, /**< Out of resources error.                                          */
  OPEN_MPLS_NO_VLAN        = 5, /**< L2 label is not installed because reference VLAN is not created. */
  OPEN_MPLS_NO_PORT        = 6  /**< L2 label is not installed because egress port is not attached. 
                                 **  Either a LAG member or a LAG with no members.                    */
} OPEN_MPLS_LFDB_NOT_INSERTED_REASON_t;

/** MPLS LFDB entry information */
OPEN_PACKED_STRUCT
{
  uint32_t                      ingressLabel;     /**< Ingress label ID. It is also used as table index.      */
  OPEN_MPLS_LFDB_PROTOCOL_t     protocol;         /**< Protocol by which the label is created.                */
  OPEN_MPLS_LFDB_ENTRY_TYPE_t   entryType;        /**< Type of label.                                         */
  OPEN_MPLS_LFDB_LABEL_ACTION_t labelAction;      /**< Label action.                                          */

  /* LFDB entry status */
  OPEN_MPLS_LFDB_IN_HARDWARE_STATUS_t  inHardware;        /**< Label status.                                  */
  OPEN_MPLS_LFDB_NOT_INSERTED_REASON_t notInsertedReason; /**< Reason why a MPLS label is not applied in HW.  */

  /* The following fields are applicable only for IPv4 label types */
  uint32_t        ipv4Addr;                      /**< IPv4 address.                                           */
  uint32_t        ipv4PrefixLen;                 /**< IPv4 prefix len.                                        */

  /* The following fields are applicable only for IPv6 label types */
  open_in6_addr_t ipv6Subnet;                    /**< IPv6 address.                                           */
  uint32_t        ipv6PrefixLen;                 /**< IPv6 prefix len.                                        */
  
  /* The following fields are applicable only for Layer-2 label types */
  unsigned char   egressMac[OPEN_MAC_ADDR_LEN];  /**< Egress MAC.                                             */
  uint32_t        egressVlan;                    /**< Egress VLAN.                                            */
  uint32_t        egressLabel;                   /**< Egress label ID. For "pop" action the field is ignored. */
  uint32_t        egressInterface;               /**< Egress interface.                                       */

  /* LFDB entry statistics */
  uint32_t        duplicateInsertions;           /**< The counter shows how many duplicate insertion 
                                                      attempts happened for the LFDB entry.                   */
  uint64_t        inBytes;                       /**< Statistics in bytes.                                    */
  uint64_t        inPackets;                     /**< Statistics in packets.                                  */
} OPEN_MPLS_LFDB_ENTRY_t;

/** MPLS global statistics information */
OPEN_PACKED_STRUCT
{
  unsigned char mplsMac[OPEN_MAC_ADDR_LEN];  /**< MPLS MAC address.                                             */
  uint32_t lfdbSize;                         /**< Maximum number of LFDB entries available in the hardware.     */
  uint32_t minLfdbLabel;                     /**< Minimal number that can be used as a label ID.                */
  uint32_t maxLfdbLabel;                     /**< Maximum number that can be used as a label ID.                */
  uint32_t numLfdbEntries;                   /**< Number of LFDB entries that have been already created.        */
  uint32_t numLfdbInHardware;                /**< Number of LFDB entries that have been applied into HW.        */
  uint32_t numLfdbNotInHardware;             /**< Number of LFDB entries that have not been applied into HW.    */
  uint32_t numLfdbPerProto[OPEN_MPLS_PROTO_LAST]; /**< Number of LFDB entries per protocol.                     */
  uint32_t numLfdbPerType[OPEN_MPLS_TYPE_LAST];   /**< Number of LFDB entries per type.                         */

  /* Statistics.
  ** These parameters are reset to 0 when the stats are cleared. */ 
  uint32_t dynamicLfdbInsertFailures;        /**< LFDB dynamic insert failure count.                            */
  uint32_t lfdbHighWaterMark;                /**< LFDB high water mark.                                         */
  uint64_t lfdbLookupFailurePackets;         /**< LFDB lookup failure packets.                                  */

  /* ECMP Group Statistics. */
  uint32_t ecmpInUse;                        /**< Current number of ECMP groups in use. The count includes 
                                              **  entries used by the routing component.                        */
  uint32_t ecmpHighInUse;                    /**< The highest number of ECMP entries used since the switch 
                                              **  was last powered up.                                          */
  uint32_t ecmpMax;                          /**< The maximum number of ECMP entries available in the hardware. */

  uint32_t maxTunnelInitiators;              /**< Maximum number of MPLS tunnels available in the hardware.     */
  uint32_t numTunnelInitiators;              /**< Number of MPLS tunnels that have been already created.     */
  uint32_t numTunnelsOneLabel;               /**< Number of MPLS tunnels created with one label */
  uint32_t numTunnelsTwoLabel;               /**< Number of MPLS tunnels created with two labels */
  uint32_t numTunnelsThreeLabel;             /**< Number of MPLS tunnels created with three labels */

} OPEN_MPLS_GLOBAL_STATS_t;

/** MPLS tunnel initiator informaton */
typedef struct
{
  open_inet_addr_t nextHop;   /**< The IPv4 or IPv6 address of the next hop with 
                               * which this tunnel is associated */
  uint32_t       label[OPEN_MAX_MPLS_IMPOSE_LABELS];  /**< The MPLS Label(s) in the packet transmitted 
                               * via this tunnel initiator. */
  uint32_t       egressIfNum; /**< The interface number to which this tunnel is pointing */
  uint16_t       egressVlan;  /**< VLAN on which MPLS tunneled packets exit the switch */
  unsigned char  egressMac [OPEN_MAC_ADDR_LEN]; /**< MAC Address used as the destination MAC
                                                 * for packets transmitted on this MPLS tunnel */
  uint32_t       numRoutes; /**< Number of routes pointing to this MPLS tunnel initiator. */ 
  uint32_t       age; /**< Seconds since last update */
} OPEN_MPLS_TUNNEL_INITIATOR_t;

/*****************************************************************************
* @purpose  Set an interface label. 
* 
* @param    client_handle   @b{(input)}  client handle from registration API.
* @param    intIfNum        @b{(input)}  the interface for which to set the label. 
* @param    type            @b{(input)}  type for which need to set the label. 
* @param    label           @b{(input)}  label assigned to the interface.
*  
* @returns  OPEN_E_NONE     Label obtained successfully
* @returns  OPEN_E_PARAM    Error in parameter passed 
* @returns  OPEN_E_EXISTS   The label is already in use and cannot be assigned repeatedly
* @returns  OPEN_E_INTERNAL Other internal failure
*  
 
* @notes  Set the label to 0 in order to disable label distribution
*         for the specified interface. 
* @notes  Certain error conditions cause SYSLOG events to be
*         generated. 
* @notes  Calling this API will change the running configuration of the switch. 
* 
* @supported-in-version OpEN API Version: 1.3
* 
* @end
****************************************************************************/
open_error_t openapiMplsBgpIntfLabelSet(openapiClientHandle_t *client_handle, 
                                        uint32_t intIfNum, OPEN_AFX_t type, uint32_t label);

/*****************************************************************************
* @purpose  Get the label configured for the specified interface.
* 
* @param    client_handle   @b{(input)}  client handle from registration API.
* @param    intIfNum        @b{(input)}  the interface for which to get the label.
* @param    type            @b{(input)}  type for which need to get the label. 
* @param    label           @b{(output)} label assigned to the interface.
*  
* @returns  OPEN_E_NONE     Label obtained successfully
* @returns  OPEN_E_FAIL     Failed to get label ID
* @returns  OPEN_E_PARAM    Error in parameter passed
* @returns  OPEN_E_INTERNAL Other internal failure
*  
* @notes  0 - No label assigned to the interface.
*         label - Label assigned to the interface.
* 
* @supported-in-version OpEN API Version: 1.3
* 
* @end
****************************************************************************/
open_error_t openapiMplsBgpIntfLabelGet(openapiClientHandle_t *client_handle, 
                                        uint32_t intIfNum, OPEN_AFX_t type, uint32_t *label);

/*****************************************************************************
* @purpose  Clears MPLS statistics.
* 
* @param    client_handle   @b{(input)}  client handle from registration API
*
* @returns  OPEN_E_NONE     Statistics were cleared successfully
* @returns  OPEN_E_FAIL     Failed to clear statistics
* @returns  OPEN_E_INTERNAL Other internal failure
* 
* @supported-in-version OpEN API Version: 1.3
* 
* @end
****************************************************************************/
open_error_t openapiMplsCountersClear(openapiClientHandle_t *client_handle);

/*****************************************************************************
* @purpose  Gets the global MPLS status and statistics. 
* 
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    global_status   @b{(output)}  MPLS global status/statistics
*
* @returns  OPEN_E_NONE     Status obtained successfully
* @returns  OPEN_E_FAIL     Failed to get status
* @returns  OPEN_E_PARAM    Error in parameter passed 
* @returns  OPEN_E_INTERNAL Other internal failure
* 
* @supported-in-version OpEN API Version: 1.3
* 
* @end
****************************************************************************/
open_error_t openapiMplsGlobalStatusGet(openapiClientHandle_t *client_handle, 
                                        OPEN_MPLS_GLOBAL_STATS_t *global_status);

/*****************************************************************************
* @purpose  Gets the BGP label distribution mode. 
* 
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    mode            @b{(output)}  BGP label distribution mode
*
* @returns  OPEN_E_NONE     Mode obtained successfully
* @returns  OPEN_E_FAIL     Failed to get the mode
* @returns  OPEN_E_PARAM    Error in parameter passed 
* @returns  OPEN_E_INTERNAL Other internal failure
* 
* @notes  OPEN_ENABLE  - BGP label distribution is enabled.
* @notes  OPEN_DISABLE - BGP label distribution is disabled.
* 
* @supported-in-version OpEN API Version: 1.3
* 
* @end
****************************************************************************/
open_error_t openapiMplsBgpLabelModeGet(openapiClientHandle_t *client_handle, 
                                        OPEN_CONTROL_t *mode);

/*****************************************************************************
* @purpose  Sets the BGP label distribution mode.
* 
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    mode            @b{(input)}  BGP label distribution mode
*
* @returns  OPEN_E_NONE     Mode was set successfully
* @returns  OPEN_E_FAIL     Failed to set mode
* @returns  OPEN_E_PARAM    Error in parameter passed
* @returns  OPEN_E_INTERNAL Other internal failure
* 
* @notes  OPEN_ENABLE  - BGP label distribution is enabled.
* @notes  OPEN_DISABLE - BGP label distribution is disabled.
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.3
* 
* @end
****************************************************************************/
open_error_t openapiMplsBgpLabelModeSet(openapiClientHandle_t *client_handle, 
                                        OPEN_CONTROL_t mode);

/*****************************************************************************
* @purpose  Gets the LFDB entry for the specified label.
* 
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    label            @b((input))   label for which to get the LFDB entry
* @param    lfdb_entry       @b{(output)}  LFDB entry
*
* @returns  OPEN_E_NONE      Operation successful
* @returns  OPEN_E_FAIL      Failed to get the specified label
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_NOT_FOUND Specified LFDB entry doesn't exist
* @returns  OPEN_E_INTERNAL  Other internal failure
* 
* @notes  The lfdb_entry must point to allocated memory, but does not
*         need to be initialized.
* 
* @supported-in-version OpEN API Version: 1.3
* 
* @end
****************************************************************************/
open_error_t openapiMplsLfdbGet(openapiClientHandle_t *client_handle, 
                                uint32_t label, OPEN_MPLS_LFDB_ENTRY_t *lfdb_entry);

/*****************************************************************************
* @purpose  Gets the LFDB entry for the next numerically higher
*           label after the specified label.
* 
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    label            @b((input))   ingress label ID
* @param    lfdb_entry       @b{(output)}  LFDB entry
*
* @returns  OPEN_E_NONE      Operation successful
* @returns  OPEN_E_FAIL      Failed to get specified label
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_NOT_FOUND The next entry was not found
* @returns  OPEN_E_INTERNAL  Other internal failure
* 
* @notes  The lfdb_entry must point to allocated memory, but does not
*         need to be initialized.
* 
* @supported-in-version OpEN API Version: 1.3
* 
* @end
****************************************************************************/
open_error_t openapiMplsLfdbNextGet(openapiClientHandle_t *client_handle, 
                                    uint32_t label, OPEN_MPLS_LFDB_ENTRY_t *lfdb_entry);

/*****************************************************************************
* @purpose  Inserts a LFDB entry into the database.
* 
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    lfdb_entry      @b{(input)}  LFDB entry
*
* @returns  OPEN_E_NONE     Operation successful
* @returns  OPEN_E_FAIL     Failed to insert LFDB entry
* @returns  OPEN_E_PARAM    Error in parameter passed
* @returns  OPEN_E_FULL     Attempting to create an entry, but already
*                           reached maximum number of labels
* @returns  OPEN_E_EXISTS   Attempting to create an entry for a label
*                           which already exists in the database.
* @returns  OPEN_E_ERROR    Attempting to create IPv4 or IPv6 entry with a subnet
*                           that is already assigned to another label.
* @returns  OPEN_E_INTERNAL Other internal failure
* 
* @notes  The caller must populate the lfdb_entry structure correctly.
* @notes  Certain error conditions cause SYSLOG events to be
* @notes  generated.
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.3
* 
* @end
****************************************************************************/
open_error_t openapiMplsLfdbCreate(openapiClientHandle_t *client_handle, 
                                   OPEN_MPLS_LFDB_ENTRY_t *lfdb_entry);

/*****************************************************************************
* @purpose  Deletes an LFDB entry from the database.
* 
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    label            @b{(input)}  label that needs to be deleted
*
* @returns  OPEN_E_NONE      Operation successful
* @returns  OPEN_E_FAIL      Failed to delete LFDB entry
* @returns  OPEN_E_NOT_FOUND LFDB entry doesn't exist
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_INTERNAL  Other internal failure
* 
* @notes  The database is indexed by ingress label ID, so only the label
* @notes  needs to be passed to this function.
* 
* @supported-in-version OpEN API Version: 1.3
* 
* @end
****************************************************************************/
open_error_t openapiMplsLfdbDelete(openapiClientHandle_t *client_handle, 
                                   uint32_t label);

/*****************************************************************************
* @purpose  Get the specified MPLS tunnel initiator information.
*
* @param    client_handle    @b{(input)}         client handle from registration API
* @param    tunnel_entry     @b{(input/output)}  MPLS tunnel initiator entry
*
* @returns  OPEN_E_NONE      Operation successful
* @returns  OPEN_E_FAIL      Failed to get the specified tunnel
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_NOT_FOUND Specified tunnel entry doesn't exist
* @returns  OPEN_E_INTERNAL  Other internal failure
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
****************************************************************************/
open_error_t openapiMplsTunnelInitiatorGet(openapiClientHandle_t *client_handle,
                                           OPEN_MPLS_TUNNEL_INITIATOR_t *tunnel_entry);

/*****************************************************************************
* @purpose Get the next MPLS tunnel initiator information.
*
* @param    client_handle     @b{(input)}   client handle from registration API
* @param    tunnel_entry      @b{(input)}   Current MPLS tunnel initiator entry
* @param    next_tunnel_entry @b{(output)}  Next MPLS tunnel initiator entry
*
* @returns  OPEN_E_NONE      Operation successful
* @returns  OPEN_E_FAIL      Failed to get specified tunnel initiator
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_NOT_FOUND The next MPLS tunnel entry was not found
* @returns  OPEN_E_INTERNAL  Other internal failure
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
****************************************************************************/
open_error_t openapiMplsTunnelInitiatorGetNext(openapiClientHandle_t *client_handle,
                                               OPEN_MPLS_TUNNEL_INITIATOR_t *tunnel_entry,
                                               OPEN_MPLS_TUNNEL_INITIATOR_t *next_tunnel_entry);
#endif /* OPENAPI_MPLS_H_INCLUDED */

