/*********************************************************************
*
*  Copyright 2018-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_pml.h
*
* @purpose   Port MAC Locking (PML) Configuration and Status
*
* @component OPEN
*
* @create    09/10/2018
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_PML_H_INCLUDED
#define OPENAPI_PML_H_INCLUDED

#include "openapi_common.h"

/*********************************************************************
*
* @purpose  Get the Port MAC Locking Global Admin mode
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    adminMode      @b{(output)}  on success, PML global admin mode.
*
* @returns OPEN_E_NONE     On success. 
* @returns OPEN_E_PARAM    Bad parameter was passed.
* @returns OPEN_E_FAIL     On failure.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiPmlGlobalAdminModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *adminMode);

/*********************************************************************
*
* @purpose Get whether an interface is a valid PML interface or not.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   interface number
* @param    isValid        @b{(output)}  if OPEN_TRUE, valid. Otherwise, OPEN_FALSE
*
* @returns OPEN_E_NONE     On success. 
* @returns OPEN_E_PARAM    Bad parameter was passed.
* @returns OPEN_E_FAIL     On failure.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*

*********************************************************************/
open_error_t openapiPmlIntfIsValid(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_BOOL_t *isValid);

/*********************************************************************
*
* @purpose  Get the Port MAC Locking mode for the specified interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   interface number
* @param    mode           @b{(output)}  on success, PML interface admin mode
*
* @returns OPEN_E_NONE     On success.
* @returns OPEN_E_UNAVAIL  Interface is not a valid PML interface.
* @returns OPEN_E_PARAM    Bad parameter was passed.
* @returns OPEN_E_FAIL     On failure.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiPmlIntfModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t *mode);

/*********************************************************************
*
* @purpose  Get the Port MAC Locking dynamic limit for the specified interface
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   interface number
* @param    limit          @b{(output)}  on success, dynamic limit
*
* @returns OPEN_E_NONE     On success. 
* @returns OPEN_E_UNAVAIL  Interface is not a valid PML interface.
* @returns OPEN_E_PARAM    Bad parameter was passed.
* @returns OPEN_E_FAIL     On failure.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiPmlIntfDynamicLimitGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *limit);

/*********************************************************************
*
* @purpose  Get the Port MAC Locking static limit for the specified interface
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   interface number
* @param    limit          @b{(output)}  on success, static limit
*
* @returns OPEN_E_NONE     On success.
* @returns OPEN_E_UNAVAIL  Interface is not a valid PML interface.
* @returns OPEN_E_PARAM    Bad parameter was passed.
* @returns OPEN_E_FAIL     On failure.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiPmlIntfStaticLimitGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *limit);

/*********************************************************************
*
* @purpose  Get the Port MAC Locking violation trap mode for the specified interface
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   interface number
* @param    trapMode       @b{(output)}  on success, PML violation trap mode
*
* @returns OPEN_E_NONE     On success.
* @returns OPEN_E_UNAVAIL  Interface is not a valid PML interface.
* @returns OPEN_E_PARAM    Bad parameter was passed.
* @returns OPEN_E_FAIL     On failure.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiPmlIntfViolationTrapModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t *trapMode);

/*********************************************************************
*
* @purpose  Get the Port MAC Locking violation D-Disable mode for the specified interface
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   interface number
* @param    mode           @b{(output)}  on success, PML violation D-Disable mode
*
* @returns OPEN_E_NONE     On success.
* @returns OPEN_E_UNAVAIL  Interface is not a valid PML interface.
* @returns OPEN_E_PARAM    Bad parameter was passed.
* @returns OPEN_E_FAIL     On failure.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiPmlIntfViolationDDisableModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t *mode);

/*********************************************************************
*
* @purpose  Get the Port MAC Locking sticky mode for the specified interface
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   interface number
* @param    stickyMode     @b{(output)}  on success, PML sticky mode
*
* @returns OPEN_E_NONE     On success.
* @returns OPEN_E_UNAVAIL  Interface is not a valid PML interface.
* @returns OPEN_E_PARAM    Bad parameter was passed.
* @returns OPEN_E_FAIL     On failure.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiPmlIntfStickyModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t *stickyMode);

/*********************************************************************
*
* @purpose Gets the first static entry for an interface.  
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   interface number
* @param    macAddr        @b{(output)}  MAC address
* @param    vlanId         @b{(output)}  VLAN ID
* @param    isSticky       @b{(output)}  on success, whether entry is sticky 
*                                        or not
*
* @returns OPEN_E_NONE     On success.
* @returns OPEN_E_UNAVAIL  Interface is not a valid PML interface.
* @returns OPEN_E_PARAM    Bad parameter was passed.
* @returns OPEN_E_FAIL     On failure.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiPmlIntfStaticEntryGetFirst (openapiClientHandle_t *client_handle, 
                                                uint32_t ifNum,
                                                open_buffdesc *macAddr, 
                                                uint32_t *vlanId, 
                                                OPEN_BOOL_t *isSticky);

/*********************************************************************
*
* @purpose Gets the next static entry for an interface.  
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    ifNum             @b{(input)}   interface number
* @param    macAddr           @b{(input)}   MAC address
* @param    vlanId            @b{(input)}   VLAN ID
* @param    nextMacAddr       @b{(output)}  Next MAC address
* @param    nextVlanId        @b{(output)}  next VLAN ID
* @param    isnextEntrySticky @b{(output)}  on success, whether next entry is 
*                                           sticky or not
*
* @returns OPEN_E_NONE     On success.
* @returns OPEN_E_UNAVAIL  Interface is not a valid PML interface.
* @returns OPEN_E_PARAM    Bad parameter was passed.
* @returns OPEN_E_FAIL     On failure.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiPmlIntfStaticEntryGetNext (openapiClientHandle_t *client_handle, 
                                               uint32_t ifNum,
                                               open_buffdesc *macAddr, 
                                               uint32_t vlanId,
                                               open_buffdesc *nextMacAddr, 
                                               uint32_t *nextVlanId, 
                                               OPEN_BOOL_t *isnextEntrySticky);

/*********************************************************************
*
* @purpose Gets the first dynamic entry for an interface.  
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   interface number
* @param    macAddr        @b{(output)}  MAC address
* @param    vlanId         @b{(output)}  VLAN ID
*
* @returns OPEN_E_NONE     On success.
* @returns OPEN_E_UNAVAIL  Interface is not a valid PML interface.
* @returns OPEN_E_PARAM    Bad parameter was passed.
* @returns OPEN_E_FAIL     On failure.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiPmlIntfDynamicEntryGetFirst (openapiClientHandle_t *client_handle, 
                                                 uint32_t ifNum,
                                                 open_buffdesc *macAddr, 
                                                 uint32_t *vlanId);

/*********************************************************************
*
* @purpose Gets the next dynamic entry for an interface.  
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    ifNum             @b{(input)}   interface number
* @param    macAddr           @b{(input)}   MAC address
* @param    vlanId            @b{(input)}   VLAN ID
* @param    nextMacAddr       @b{(output)}  Next MAC address
* @param    nextVlanId        @b{(output)}  next VLAN ID
*
* @returns OPEN_E_NONE     On success.
* @returns OPEN_E_UNAVAIL  Interface is not a valid PML interface.
* @returns OPEN_E_PARAM    Bad parameter was passed.
* @returns OPEN_E_FAIL     On failure.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiPmlIntfDynamicEntryGetNext (openapiClientHandle_t *client_handle, 
                                                uint32_t ifNum,
                                                open_buffdesc *macAddr, 
                                                uint32_t vlanId,
                                                open_buffdesc *nextMacAddr, 
                                                uint32_t *nextVlanId);

/*****************************************************************************
 * @purpose   Sets the PML global admin mode.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     adminMode      @b{(input)}  PML global admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlGlobalAdminModeSet(openapiClientHandle_t *client_handle, 
                                          OPEN_CONTROL_t adminMode);

/*****************************************************************************
 * @purpose   Set the Port MAC Locking mode for the specified interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     intfMode       @b{(input)}  PML interface admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlIntfModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                                   OPEN_CONTROL_t intfMode);

/*****************************************************************************
 * @purpose   Gets the Port MAC Locking maximum dynamic limit for the specified interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     limit          @b{(output)} Maximum dynamic limit
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlIntfDynamicMaxLimitGet(openapiClientHandle_t *client_handle, 
                                              uint32_t ifNum, uint32_t *limit);

/*****************************************************************************
 * @purpose   Sets the Port MAC Locking dynamic limit for the specified interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     limit          @b{(input)}  Dynamic limit
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlIntfDynamicLimitSet(openapiClientHandle_t *client_handle, 
                                           uint32_t ifNum, uint32_t limit);

/*****************************************************************************
 * @purpose   Gets the Port MAC Locking maximum static limit for the specified interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     limit          @b{(output)} Maximum static limit
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlIntfStaticMaxLimitGet(openapiClientHandle_t *client_handle, 
                                             uint32_t ifNum, uint32_t *limit);

/*****************************************************************************
 * @purpose   Gets the Port MAC Locking static count for the specified interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     limit          @b{(output)} Static count
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlIntfStaticCountGet(openapiClientHandle_t *client_handle, 
                                          uint32_t ifNum, uint32_t *limit);

/*****************************************************************************
 * @purpose   Sets the Port MAC Locking static limit for the specified interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     limit          @b{(input)}  Static limit
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlIntfStaticLimitSet(openapiClientHandle_t *client_handle, 
                                          uint32_t ifNum, uint32_t limit);

/*****************************************************************************
 * @purpose   Gets the frequency interval between successive SNMP traps.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     interval       @b{(output)} Time interval in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlIntfTrapFrequencyGet(openapiClientHandle_t *client_handle, 
                                            uint32_t ifNum, uint32_t *interval);

/*****************************************************************************
 * @purpose   Sets the frequency interval between successive SNMP traps.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     interval       @b{(input)}  Time interval in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlIntfTrapFrequencySet(openapiClientHandle_t *client_handle, 
                                            uint32_t ifNum, uint32_t interval);

/*****************************************************************************
 * @purpose   Adds a statically locked entry to an interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     macAddr        @b{(input)}  MAC address
 * @param     vlanId         @b{(input)}  VLAN ID
 * @param     sticky         @b{(input)}  Sticky
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlIntfStaticEntryAdd(openapiClientHandle_t *client_handle, 
                                          uint32_t ifNum, open_buffdesc *macAddr, 
                                          uint32_t vlanId, OPEN_BOOL_t sticky);

/*****************************************************************************
 * @purpose   Deletes a statically locked entry from an interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     macAddr        @b{(input)}  MAC address
 * @param     vlanId         @b{(input)}  VLAN ID
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlIntfStaticEntryDelete(openapiClientHandle_t *client_handle, 
                                             uint32_t ifNum, open_buffdesc *macAddr, 
                                             uint32_t vlanId);

/*****************************************************************************
 * @purpose   Converts dynamically locked entries to statically locked entries 
 *            on this interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlIntfDynamicToStaticMove(openapiClientHandle_t *client_handle, 
                                               uint32_t ifNum);

/*****************************************************************************
 * @purpose   Configures PML sticky mode on this interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     sticky         @b{(input)}  Sticky mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlIntfStickyModeSet(openapiClientHandle_t *client_handle, 
                                         uint32_t ifNum, OPEN_CONTROL_t sticky);

/*****************************************************************************
 * @purpose   Gets the last violation entry for an interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     macAddr        @b{(output)} MAC address
 * @param     vlanId         @b{(output)} VLAN ID
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlIntfLastViolationAddrGet(openapiClientHandle_t *client_handle, 
                                                uint32_t ifNum, open_buffdesc *macAddr, 
                                                uint32_t *vlanId);

/*****************************************************************************
 * @purpose   Gets the count of dynamcially learned mac Addresses.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     count          @b{(output)} Learned dynamic address count
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlIntfDynamicLearnedAddrCountGet(openapiClientHandle_t *client_handle, 
                                                      uint32_t ifNum, uint32_t *count);

/*****************************************************************************
 * @purpose   Gets the count of statically learned mac Addresses.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     count          @b{(output)} Static address count
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlIntfStaticLearnedAddrCountGet(openapiClientHandle_t *client_handle, 
                                                     uint32_t ifNum, uint32_t *count);

/*****************************************************************************
 * @purpose   Gets the count of statically learned mac Addresses.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vlanId         @b{(input)}  VLAN ID
 * @param     count          @b{(output)} Static address count
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlStaticAddrCountByVlanGet(openapiClientHandle_t *client_handle, 
                                                uint32_t vlanId, uint32_t *count);

/*****************************************************************************
 * @purpose   Gets the global count of statically learned mac Addresses.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     count          @b{(output)} Learned static address count
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlStaticAddrGlobalCountGet(openapiClientHandle_t *client_handle, 
                                                uint32_t *count);

/*****************************************************************************
 * @purpose   Sets the port MAC locking violation D-Disable mode for the specified interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     mode           @b{(input)}  Violation D-Disable Mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlIntfViolationDDisableModeSet(openapiClientHandle_t *client_handle, 
                                                    uint32_t ifNum, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Gets the PML vlan admin mode.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     adminMode      @b{(output)} PML vlan admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlVlanAdminModeGet(openapiClientHandle_t *client_handle, 
                                        OPEN_CONTROL_t *adminMode);

/*****************************************************************************
 * @purpose   Sets the PML vlan admin mode.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     adminMode      @b{(input)}  PML vlan admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlVlanAdminModeSet(openapiClientHandle_t *client_handle, 
                                        OPEN_CONTROL_t adminMode);

/*****************************************************************************
 * @purpose   Gets the MAC limit per Vlan.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vlanId         @b{(input)}  VLAN ID
 * @param     limit          @b{(output)} MAC lock limit per vlan
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlVlanDynamicLimitGet(openapiClientHandle_t *client_handle, 
                                           uint32_t vlanId, uint32_t *limit);

/*****************************************************************************
 * @purpose   Sets the vlan MAC Locking limit.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vlanId         @b{(input)}  VLAN ID
 * @param     limit          @b{(input)}  MAC lock limit per vlan
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlVlanDynamicLimitSet(openapiClientHandle_t *client_handle, 
                                           uint32_t vlanId, uint32_t limit);

/*****************************************************************************
 * @purpose   Gets operational vlan MAC locking limit.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vlanId         @b{(input)}  VLAN ID
 * @param     limit          @b{(output)} MAC lock operational limit per vlan
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlVlanOperationalLimitGet(openapiClientHandle_t *client_handle, 
                                               uint32_t vlanId, uint32_t *limit);

/*****************************************************************************
 * @purpose   Gets the MAC Locking violation D-Disable mode for the specified vlan.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vlanId         @b{(input)}  VLAN ID
 * @param     mode           @b{(output)} Violation D-Disable Mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlVlanViolationDDisableModeGet(openapiClientHandle_t *client_handle, 
                                                    uint32_t vlanId, OPEN_BOOL_t *mode);

/*****************************************************************************
 * @purpose   Sets the MAC Locking violation D-Disable mode for the specified vlan.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vlanId         @b{(input)}  VLAN ID
 * @param     mode           @b{(input)}  Violation D-Disable Mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlVlanViolationDDisableModeSet(openapiClientHandle_t *client_handle, 
                                                    uint32_t vlanId, OPEN_BOOL_t mode);

/*****************************************************************************
 * @purpose   Gets the MAC Locking violation trap mode for the specified vlan.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vlanId         @b{(input)}  VLAN ID
 * @param     mode           @b{(output)} Violation trap Mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlVlanViolationTrapModeGet(openapiClientHandle_t *client_handle, 
                                                uint32_t vlanId, OPEN_BOOL_t *mode);

/*****************************************************************************
 * @purpose   Sets the MAC Locking violation trap mode for the specified vlan.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vlanId         @b{(input)}  VLAN ID
 * @param     mode           @b{(input)}  Violation trap Mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlVlanViolationTrapModeSet(openapiClientHandle_t *client_handle, 
                                                uint32_t vlanId, OPEN_BOOL_t mode);

/*****************************************************************************
 * @purpose   Gets the MAC Locking mode for the specified vlan.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vlanId         @b{(input)}  VLAN ID
 * @param     mode           @b{(output)} VLAN MAC Locking mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlVlanModeGet(openapiClientHandle_t *client_handle, 
                                   uint32_t vlanId, uint32_t *mode);

/*****************************************************************************
 * @purpose   Sets the MAC Locking mode for the specified vlan.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     vlanId         @b{(input)}  VLAN ID
 * @param     mode           @b{(input)}  VLAN MAC Locking mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlVlanModeSet(openapiClientHandle_t *client_handle, 
                                   uint32_t vlanId, uint32_t mode);

/*****************************************************************************
 * @purpose   Gets the Port MAC Locking vlan dynamic limit for the specified interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     vlanId         @b{(input)}  VLAN ID
 * @param     limit          @b{(output)} MAC lock limit per vlan per interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlIntfVlanDynamicLimitGet(openapiClientHandle_t *client_handle, 
                                               uint32_t ifNum, uint32_t vlanId, 
                                               uint32_t *limit);

/*****************************************************************************
 * @purpose   Sets the Port MAC Locking vlan dynamic limit for the specified interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     vlanId         @b{(input)}  VLAN ID
 * @param     limit          @b{(input)}  MAC lock limit per vlan per interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlIntfVlanDynamicLimitSet(openapiClientHandle_t *client_handle, 
                                               uint32_t ifNum, uint32_t vlanId, 
                                               uint32_t limit);

/*****************************************************************************
 * @purpose   Gets the Port MAC Locking vlan dynamic learned address count for 
 *            the specified interface and vlan.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     vlanId         @b{(input)}  VLAN ID
 * @param     count          @b{(output)} MAC lock learned address count per 
 *                                        vlan per interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlIntfVlanDynamicLearnedAddrCountGet(openapiClientHandle_t *client_handle, 
                                                          uint32_t ifNum, uint32_t vlanId, 
                                                          uint32_t *count);

/*****************************************************************************
 * @purpose   Gets secure MAC address aging time for the specified interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     ageTime        @b{(output)} Age time in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlIntfAgeTimeGet(openapiClientHandle_t *client_handle, 
                                      uint32_t ifNum, uint32_t *ageTime);

/*****************************************************************************
 * @purpose   Sets secure MAC address aging time for the specified interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     ageTime        @b{(input)}  Age time in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlIntfAgeTimeSet(openapiClientHandle_t *client_handle, 
                                      uint32_t ifNum, uint32_t ageTime);

/*****************************************************************************
 * @purpose   Gets the next valid PML interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     nextIfNum      @b{(output)} Next valid PML interface number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlNextValidIntfGet(openapiClientHandle_t *client_handle, 
                                        uint32_t ifNum, uint32_t *nextIfNum);

/*****************************************************************************
 * @purpose   Gets the first dynamic entry for a vlan on a given interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     macAddr        @b{(output)} MAC address
 * @param     vlanId         @b{(output)} VLAN ID
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlVlanDynamicEntryGetFirst(openapiClientHandle_t *client_handle, 
                                                uint32_t ifNum, open_buffdesc *macAddr, 
                                                uint32_t *vlanId);

/*****************************************************************************
 * @purpose   Gets the next dynamic entry for a vlan on a given interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     macAddr        @b{(input)}  MAC address
 * @param     nextMacAddr    @b{(output)} MAC address
 * @param     vlanId         @b{(output)} VLAN ID
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlVlanDynamicEntryGetNext(openapiClientHandle_t *client_handle, 
                                               uint32_t ifNum, open_buffdesc *macAddr, 
                                               open_buffdesc *nextMacAddr, 
                                               uint32_t *vlanId);

/*****************************************************************************
 * @purpose   Associate the 802.1Qci stream with static PML entry.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     macAddr        @b{(input)}  MAC address
 * @param     vlanId         @b{(input)}  VLAN ID
 * @param     streamID       @b{(input)}  802.1QCI StreamID
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlDot1qciStreamSet(openapiClientHandle_t *client_handle, 
                                        uint32_t ifNum, open_buffdesc *macAddr, 
                                        uint32_t vlanId, uint32_t streamID);

/*****************************************************************************
 * @purpose   Deassociate the 802.1Qci stream with static PML entry.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     macAddr        @b{(input)}  MAC address
 * @param     vlanId         @b{(input)}  VLAN ID
 * @param     streamID       @b{(input)}  802.1QCI StreamID
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlDot1qciStreamDelete(openapiClientHandle_t *client_handle, 
                                           uint32_t ifNum, open_buffdesc *macAddr, 
                                           uint32_t vlanId, uint32_t streamID);

/*****************************************************************************
 * @purpose   Get the 802.1Qci stream ID associated with static PML entry.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface number
 * @param     macAddr        @b{(input)}  MAC address
 * @param     vlanId         @b{(input)}  VLAN ID
 * @param     streamID       @b{(output)} 802.1QCI StreamID
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature not available.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiPmlDot1qciStreamGet(openapiClientHandle_t *client_handle, 
                                        uint32_t ifNum, 
                                        open_buffdesc *macAddr, uint32_t vlanId, 
                                        uint32_t *streamID);

/*********************************************************************
*
* @purpose  Set the Port MAC Locking violation trap mode for the specified interface
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   interface number
* @param    trapMode       @b{(input)}   trap mode
*
* @returns OPEN_E_NONE     On success.
* @returns OPEN_E_UNAVAIL  Interface is not a valid PML interface or feature is
*                          not available.
* @returns OPEN_E_PARAM    Bad parameter was passed.
* @returns OPEN_E_FAIL     On failure.
*
 * @notes     If the given interface number is 0 then trap mode will be set to
 *            all valid PML interfaces.
* @supported-in-version OpEN API Version: 1.26
*
* @end
*********************************************************************/
open_error_t openapiPmlIntfViolationTrapModeSet(openapiClientHandle_t *client_handle,
                                                uint32_t ifNum, 
                                                OPEN_CONTROL_t trapMode);

#endif /* OPENAPI_PML_H_INCLUDED */
