/**********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_protectedport.h
*
* @purpose   This code implements the OpEN Protected Port API
*
* @component OPEN
*
* @create    05/26/2023
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_PROTECTEDPORT_H_INCLUDED
#define OPENAPI_PROTECTEDPORT_H_INCLUDED

#include "openapi_common.h" 


#define OPEN_PROTECTED_PORT_GROUP_NAME_SIZE  49

/*****************************************************************************
 * @purpose   Get the maximum number of protected port groups.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     maxGroupNum    @b{(output)} Group number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiProtectedPortMaxGroups (openapiClientHandle_t *client_handle,
                                            uint32_t *maxGroupNum); 

/*****************************************************************************
 * @purpose   Get the name of the group for a given group ID.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     groupId        @b{(input)}  Group identifier
 * @param     groupName      @b{(output)} Name of the group
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiProtectedPortGroupNameGet(openapiClientHandle_t *client_handle, 
                                              uint32_t groupId, 
                                              open_buffdesc *groupName);

/*****************************************************************************
 * @purpose   Set the name of the group for a given group ID.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     groupId        @b{(input)}  Group identifier
 * @param     groupName      @b{(input)}  Name of the group
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiProtectedPortGroupNameSet(openapiClientHandle_t *client_handle, 
                                              uint32_t groupId, 
                                              open_buffdesc *groupName);

/*****************************************************************************
 * @purpose   Get the number of protected ports which are members of a given group ID.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     groupId        @b{(input)}  Group identifier
 * @param     intfCount      @b{(output)} Number of interfaces
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes  None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiProtectedPortGroupIntfCountGet(openapiClientHandle_t *client_handle, 
                                                   uint32_t groupId, 
                                                   uint32_t *intfCount);

/*****************************************************************************
 * @purpose   Get all the protected ports which are members of a given group ID.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     groupId        @b{(input)}  Group identifier
 * @param     intfList       @b{(output)} List of interfaces with comma separation
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes  Buffer allocated size (intfList) must be 4 * number of interfaces in
 *         the group. To get the number of interfaces in the group use 
 *         @ openapiProtectedPortGroupCountGet()
ne
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiProtectedPortGroupIntfListGet(openapiClientHandle_t *client_handle, 
                                                  uint32_t groupId, 
                                                  open_buffdesc *intfList);

/*****************************************************************************
 * @purpose   Get the group ID for a given interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     groupId        @b{(output)} Group identifier
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiProtectedPortGroupGet(openapiClientHandle_t *client_handle, 
                                          uint32_t ifNum, uint32_t *groupId);

/*****************************************************************************
 * @purpose   Get the next group and interface for a given group and interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     groupId        @b{(input)}  Group identifier
 * @param     NextIfNum      @b{(output)} Next interface
 * @param     NextGroupId    @b{(output)} Next group identifier
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiProtectedPortNextGroupGet(openapiClientHandle_t *client_handle, 
                                              uint32_t ifNum, uint32_t groupId, 
                                              uint32_t *NextIfNum, 
                                              uint32_t *NextGroupId);

/*****************************************************************************
 * @purpose   Add an interface to a protected port group.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     groupId        @b{(input)}  Group identifier
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiProtectedPortGroupIntfAdd(openapiClientHandle_t *client_handle, 
                                              uint32_t ifNum, uint32_t groupId);

/*****************************************************************************
 * @purpose   Delete an interface from a protected port group.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     groupId        @b{(input)}  Group identifier
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiProtectedPortGroupIntfDelete(openapiClientHandle_t *client_handle, 
                                                 uint32_t ifNum, uint32_t groupId);


#endif /* OPENAPI_PROTECTEDPORT_H_INCLUDED */
