/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_routing_stats.h
*
* @purpose   IPv4 and IPv6 routing statistics.
*
* @component OPEN
*
* @create    12/15/2016
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_ROUTING_STATS_H_INCLUDED
#define OPENAPI_ROUTING_STATS_H_INCLUDED

#include <stdio.h>
#include <stddef.h>
#include <sys/un.h>

#include "openapi_common.h"
#include "openapi_ip_route_table.h"

/* Data structure holds results returned by both openapiIpv4RouteTableStatsGet() and openapiIpv6RouteTableStatsGet() */

typedef struct openRouteStats_s
{
  uint64_t mask;                    /*<< defines which stats are reported in numRoutes */

  uint32_t bestRoutes;              /*<< Number of routes installed in the forwarding table */
  uint32_t bestRoutesHigh;          /*<< Number of routes installed in the forwarding table, high */

  uint32_t altRoutes;               /*<< Number of routes in RTO not selected as a best route */

  uint32_t leakedRoutes;            /*<< Number of routes leaked into RTO from other VR. */
  OPEN_BOOL_t leakedRoutesValid;    /*<< If OPEN_FALSE, leakedRoutes not reportable */

  uint32_t mplsRoutes;              /*<< Number of MPLS routes */
  OPEN_BOOL_t mplsRoutesValid;      /*<< If OPEN_FALSE, mplsRoutes not reportable */

  uint32_t rfc5549Routes;           /*<< RFC5549 routes - IPv4 routes with IPv6 nexthop */
  OPEN_BOOL_t rfc5549RoutesValid;   /*<< If OPEN_FALSE, rfc5549Routes not reportable */

  uint32_t rejectRoutes;            /*<< Number of reject routes */
  OPEN_BOOL_t rejectRoutesValid;    /*<< If OPEN_FALSE, rejectRoutes not reportable */

  uint32_t adds;                    /*<< Number of route adds */
  uint32_t mods;                    /*<< Number of route mods */
  uint32_t dels;                    /*<< Number of route deletes */

  uint32_t invalidAdds;             /*<< route is invalid */
  uint32_t unresAdds;               /*<< No next hops resolve to local subnet */
  uint32_t failedAdds;              /*<< resource limitation */

  uint32_t activeLocals;            /*<< Number of local routes */

  uint32_t resvLocals;              /*<< Number of local routes not currently installed but for which RTO has reserved space */

  uint32_t nextHops;                /*<< Number of unique next hops currently used among all routes. */
  uint32_t nextHopsHigh;            /*<< Next hops high water */

  uint32_t ecmpRoutes;              /*<< Current number of ECMP routes. Does not count truncated routes. */

  uint32_t nextHopGroups;           /*<< Current number of ECMP groups. May exceed what hardware can install. */
  uint32_t nhGroupsHigh;            /*<< Next hop ECMP groups high water */

  uint32_t ecmpGroups;              /*<< Number of ECMP groups. An ECMP group is a next hop group with more than 1 next hop. */
  uint32_t ecmpGroupsHigh;          /*<< ECMP group high water */

  uint32_t ecmpTruncs;              /*<< Number of current best routes that failed to be installed in the forwarding table, most likely because the ECMP group table is full. */

  uint32_t ecmpRetries;             /*<< Number of attempts to reinstall ECMP routes in hardware */

  uint32_t hwFailureRoutes;         /*<< Number of routes failed to add into hardware */

  uint32_t kernelFailedAdds;        /*<< Number of kernel routes failed to add */
} openRouteStats_t;

/* Data structure holds results returned by both openapiIpv4EcmpRouteProtocolCountsGet() and openapiIpv6EcmpRouteProtocolCountsGet() */

typedef struct openEcmpRouteProtocolCount_s
{
  uint32_t protocol;                       /*<< @ref OPEN_RTO_PROTOCOL_INDICES_t or @ref OPEN_RTO6_ROUTE_TYPE_INDICES_t depending on ipv4 flag, below*/
  OPEN_AF_t family;                        /*<< OPEN_AF_INET or OPEN_AF_INET6 */
  uint32_t numEcmpRoutes;                  /*<< Number of ECMP routes */
} openEcmpRouteProtocolCount_t;

/*********************************************************************
* @purpose  Get the number of ECMP Routes.
*
* @param    client_handle   @b{(input)}   OpEN client handle
* @param    count           @b{(output)}  Number of ECMP routes.
*
* @returns  OPEN_E_NONE        Success.
* @returns  OPEN_E_PARAM       Invalid argument.
* @returns  OPEN_E_FAIL        Failure occurred.
*
* @notes Use this function to determine the number of uint32_t sized elements
*        to allocate for the ecmpHist argument to openapiIpv4RouteTableStatsGet 
*        or openapiIpv6RouteTableStatsGet.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiNumEcmpRoutesGet(openapiClientHandle_t *client_handle, uint32_t *count);

/*********************************************************************
* @purpose  Get a table containing ECMP counts for each IPv4 routing protocol.
*
* @param    client_handle   @b{(input)}   OpEN client handle
* @param    vrf             @b{(input)}   VRF name. Use "" for default.
* @param    counts          @b{(output)}  Number of ECMP routes returned as a 
*                                         vector of openEcmpRouteProtocolCount_t. 
*                                         Use @ref openapiIpv4NumRoutingProtocolsGet()
*                                         to determine the number of elements
*                                         to allocate in this buffer.
*                                         On return, the size field of the
*                                         buffdesc will contain the number
*                                         of bytes consumed by the
*                                         elements that were returned.
*
* @returns  OPEN_E_NONE        Success.
* @returns  OPEN_E_PARAM       Invalid argument.
* @returns  OPEN_E_FAIL        Failure occurred.
* @returns  OPEN_E_INTERNAL    Internal error occurred.
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiIpv4EcmpRouteProtocolCountsGet(openapiClientHandle_t *client_handle, open_buffdesc *vrf, open_buffdesc *counts);

/*********************************************************************
* @purpose  Get a table containing ECMP counts for each IPv6 routing protocol.
*
* @param    client_handle   @b{(input)}   OpEN client handle.
* @param    counts          @b{(output)}  Number of ECMP routes returned as a 
*                                         vector of openEcmpRouteProtocolCount_t. 
*                                         Use @ref openapiIpv6NumRoutingProtocolsGet()
*                                         to determine the number of elements
*                                         to allocate in this buffer.
*                                         On return, the size field of the
*                                         buffdesc will contain the size in 
*                                         bytes consumed by the number
*                                         of elements that were returned.
*
* @returns  OPEN_E_NONE        Success.
* @returns  OPEN_E_PARAM       Invalid argument.
* @returns  OPEN_E_FAIL        Failure occurred.
* @returns  OPEN_E_UNAVAIL     Failure occurred.
* @returns  OPEN_E_INTERNAL    Internal error occurred.
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiIpv6EcmpRouteProtocolCountsGet(openapiClientHandle_t *client_handle, open_buffdesc *counts);

/*********************************************************************
* @purpose  Get the number of IPv6 prefixes.
*
* @param    client_handle   @b{(input)}   OpEN client handle
* @param    count           @b{(output)}  Number of IPv6 prefixes.
*
* @returns  OPEN_E_NONE        Success.
* @returns  OPEN_E_PARAM       Invalid argument.
* @returns  OPEN_E_FAIL        Failure occurred.
*
* @notes Use this function to determine the number of uint32_t sized elements
*        to allocate for the prefixesNo argument to 
*        openapiIpv6RouteTableStatsGet.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiNumIpv6PrefixesGet(openapiClientHandle_t *client_handle, uint32_t *count);

/*********************************************************************
* @purpose  Get the number of IPv4 routing protocols.
*
* @param    client_handle   @b{(input)}   OpEN client handle
* @param    count           @b{(output)}  Number of IPv4 routing protocols.
*
* @returns  OPEN_E_NONE        Success.
* @returns  OPEN_E_PARAM       Invalid argument.
* @returns  OPEN_E_FAIL        Failure occurred.
*
* @notes Use this function to determine the number of uint32_t sized elements
*        to allocate for the numroutes argument to 
*        @ref openapiIpv4RouteTableStatsGet and the counts argument to
*        @ref openapiIpv4EcmpRouteProtocolCountsGet.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiIpv4NumRoutingProtocolsGet(openapiClientHandle_t *client_handle, uint32_t *count);

/*********************************************************************
* @purpose  Get the number of IPv6 routing protocols.
*
* @param    client_handle   @b{(input)}   OpEN client handle
* @param    count           @b{(output)}  Number of IPv6 routing protocols.
*
* @returns  OPEN_E_NONE        Success.
* @returns  OPEN_E_PARAM       Invalid argument.
* @returns  OPEN_E_FAIL        Failure occurred.
*
* @notes Use this function to determine the number of uint32_t sized elements
*        to allocate for the numroutes argument to 
*        @ref openapiIpv6RouteTableStatsGet and the counts argument to
*        @ref openapiIpv6EcmpRouteProtocolCountsGet.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiIpv6NumRoutingProtocolsGet(openapiClientHandle_t *client_handle, uint32_t *count);

/*********************************************************************
* @purpose    Report IPv4 routing table statistics
*
* @param      client_handle  @b{(input)}   OpEN client handle.
* @param      vrfName        @b{(input)}   VRF Name
* @param      bestRoutes     @b{(input)}   Whether to report statistics for
*                                          just best routes or for all routes.
*                                          Set to OPEN_TRUE to only get best 
*                                          routes and OPEN_FALSE to get all 
*                                          routes.
* @param      routeStats     @b{(output)}  A place to put the stats.
* @param      numRoutes      @b{(output)}  Contains number of routes for
*                                          each supported routing protocol.
* @param      ecmpHisto      @b{(output)}  Contains ECMP next hop histogram.
*
* @returns  OPEN_E_NONE        Success.
* @returns  OPEN_E_PARAM       Invalid argument.
* @returns  OPEN_E_FAIL        Failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
* ********************************************************************/
open_error_t openapiIpv4RouteTableStatsGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_BOOL_t bestRoutes, openRouteStats_t *routeStats, open_buffdesc *numRoutes, open_buffdesc *ecmpHisto);

/*********************************************************************
* @purpose    Report IPv6 routing table statistics
*
* @param      client_handle  @b{(input)}   OpEN client handle.
* @param      bestRoutes     @b{(input)}   Whether to report statistics for
*                                          just best routes or for all routes
* @param      routeStats     @b{(output)}  A place to put the stats.
* @param      numRoutes      @b{(output)}  Contains number of routes for
*                                          each supported routing protocol.
* @param      ecmpHisto      @b{(output)}  Contains ECMP next hop histogram.
* @param      prefixesNo     @b{(output)}  Contains vector of prefix numbers.
*
* @returns  OPEN_E_NONE        Success.
* @returns  OPEN_E_PARAM       Invalid argument.
* @returns  OPEN_E_FAIL        Failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
* ********************************************************************/
open_error_t openapiIpv6RouteTableStatsGet(openapiClientHandle_t *client_handle, OPEN_BOOL_t bestRoutes, openRouteStats_t *routeStats, open_buffdesc *numRoutes, open_buffdesc *ecmpHisto, open_buffdesc *prefixesNo);

/*********************************************************************
* @purpose  Iterate through the route types with a given protocol ID.
*
* @param      client_handle   @b{(input)}         OpEN client handle.
* @param      protoId         @b{(input}          protocol ID
* @param      routeType       @b{(input/output)}  route type. Set to 0 on 
*                                                 input to get the first value. 
*                                                 Set to previous value on 
*                                                 input to get the next value.
*
* @returns  OPEN_E_NONE       If a route type returned
*           OPEN_E_NOT_FOUND  If there is no protocol with the given protocol ID
*           OPEN_E_FAIL       If no more route types for this protocol
*           OPEN_E_PARAM      Bad parameter 
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiIpMapProtoRouteTypeNextGet(openapiClientHandle_t *client_handle, uint32_t protoId, uint32_t *routeType);

/*********************************************************************
* @purpose  Iterate through the protocols that have been dynamically registered
*           at run time.
*
* @param    client_handle  @b{(input)}         OpEN client handle.
* @param    protoId        @b{(input/output)}  Protocol ID. Set to 0 on input 
*                                              to get the first value. Set to 
*                                              previous value on input to get 
*                                              the next value.
*
* @returns  OPEN_E_NONE If a protocol ID is returned
*           OPEN_E_FAIL If no more dynamic protocols
*           OPEN_E_PARAM  Bad parameter 
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiIpMapDynamicProtoNextGet(openapiClientHandle_t *client_handle, uint32_t *protoId);

/*********************************************************************
* @purpose  Get the name for a given protocol.
*
* @param    client_handle   @b{(input)}  OpEN client handle.
* @param    protoId         @b{(input)}  A protocol ID
* @param    protoName       @b{(output)} Protocol name. Size must be at least 
*                                        the value obtained by calling 
*                                        openapiIpMapProtoNameLenGet() in bytes.
*
* @returns  OPEN_E_NONE If protocol name found
*           OPEN_E_PARAM Invalid parameter.
*           OPEN_E_FAIL Otherwise
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiIpMapProtoNameGet(openapiClientHandle_t *client_handle, uint32_t protoId, open_buffdesc *protoName);

/*********************************************************************
* @purpose  Get the max protocol name length.
*
* @param    client_handle   @b{(input)}  OpEN client handle.
* @param    len             @b{(output)} Max len for a protocol name.
*
* @returns  OPEN_E_NONE On success
*           OPEN_E_PARAM Bad argument
*           OPEN_E_FAIL Otherwise
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiIpMapProtoNameLenGet(openapiClientHandle_t *client_handle, uint32_t *len);

/*********************************************************************
* @purpose  Get the name for a given route type.
*
* @param    client_handle   @b{(input)}  OpEN client handle.
* @param    routeType       @b{(input)}  A route type
* @param    routeTypeName   @b{(output)} Route type name. Size must be at least
*                                        the value obtained by calling
*                                        openapiIpMapRouteTypeNameLenGet() in bytes.
*
* @returns  OPEN_E_NONE If route type name found
*           OPEN_E_PARAM Invalid parameter.
*           OPEN_E_FAIL Otherwise
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiIpMapRouteTypeNameGet(openapiClientHandle_t *client_handle, uint32_t routeType, open_buffdesc *routeTypeName);

/*********************************************************************
* @purpose  Get the max route type name length.
*
* @param    client_handle   @b{(input)}  OpEN client handle.
* @param    len             @b{(output)} Max length for a route type name.
*
* @returns  OPEN_E_NONE On success
*           OPEN_E_PARAM Bad argument
*           OPEN_E_FAIL Otherwise
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiIpMapRouteTypeNameLenGet(openapiClientHandle_t *client_handle, uint32_t *len);

/*********************************************************************
* @purpose  Get the IPv4/IPv6 reportable route stats masks.
*
* @param    client_handle   @b{(input)}     OpEN client handle.
* @param    ipv4Mask        @b{(output)}    IPv4 mask
* @param    ipv6Mask        @b{(output)}    IPv6 mask
*
* @returns  OPEN_E_NONE   On success
*           OPEN_E_FAIL   Otherwise
*           OPEN_E_PARAM  Bad parameter (see notes)
*
* @notes    Each bit in a mask corresponds to a route protocol index defined
*           by the OPEN_RTO6_ROUTE_TYPE_INDICES_t (IPv6) and 
*           OPEN_RTO_PROTOCOL_INDICES_t (IPv4) enums. If stats for a protocol 
*           are supported by the implementation, the corresponding bit 
*           (1 << <proto>) will be set, where <proto> is a value in one of the 
*           above enums. For example, the following code tests if MPLS route 
*           stats are supported for IPv6:
*
*           if (mask & (1 << OPEN_RTO6_MPLS))
*           {
*               printf("MPLS route stats are supported\n");
*           }
*           else
*           {
*               printf("MPLS route stats are not suppoted\n");
*           }
*
*           A field in openRouteStats_t named mask holds the same mask when
*           stats are reported by openapiIpv4RouteTableStatsGet and
*           openapiIpv6RouteTableStatsGet. However, this function can be 
*           used to get the masks without querying for statistics.
*
*           Passing either ipv4Mask or ipv6Mask as NULL will cause that
*           mask to not be retrieved. Passing both as NULL will result in
*           a OPEN_E_PARAM return.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiGetReportableRouteStats(openapiClientHandle_t *client_handle, uint64_t *ipv4Mask, uint64_t *ipv6Mask);

#endif /* OPENAPI_ROUTING_STATS_H_INCLUDED */
