/*********************************************************************
*
*  Copyright 2018-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_switch_cpu.h
*
* @purpose   Switch CPU API
*
* @component OpEN
*
* @create    08/31/2018
*
* @end
*                                    
**********************************************************************/
#ifndef OPENAPI_SWITCH_CPU_H_INCLUDED
#define OPENAPI_SWITCH_CPU_H_INCLUDED

#include "openapi_common.h"

#define OPEN_CPU_UTIL_SCALAR 10000

#define OPEN_CPU_TASK_UTIL_SIZE         65

/* Header and footer prints */
#define OPEN_CPU_TASK_UTIL_HEADER_ROWS  5

/* The OPEN_CPU_TASK_COUNT_AVERAGE is used as average task count and is used to
 * calculate the minimum buffer size required. If the tasks are more, provide
 * the buffer size calculated with L7_CPU_UTIL_MAX_TASKS. If the tasks are more
 * than OPEN_CPU_TASK_COUNT_AVERAGE and provided a buffer calculated with
 * OPEN_CPU_TASK_COUNT_AVERAGE, the provided buffer size will be filled and 
 * skipping the remaining data.
 */
#define OPEN_CPU_TASK_COUNT_AVERAGE     100

/** OPEN cpu threshold param */
typedef enum
{
  OPEN_CPU_UTIL_MONITOR_RISING_THRESHOLD_PARAM = 0,     /**< Rising threshold    */
  OPEN_CPU_UTIL_MONITOR_RISING_PERIOD_PARAM = 1,        /**< Rising period       */
  OPEN_CPU_UTIL_MONITOR_FALLING_THRESHOLD_PARAM = 2,    /**< Falling threshold   */
  OPEN_CPU_UTIL_MONITOR_FALLING_PERIOD_PARAM = 3,       /**< Falling period      */
  OPEN_CPU_UTIL_MONITOR_LAST_PARAM = 4                  /**< Last one */
} OPEN_CPU_UTIL_MONITOR_PARAMS_t;

typedef struct
{
  uint32_t timePeriod;   /**< Time period (secs) over which CPU utilization is being measured. */
  uint32_t utilization;  /**< Utilization of the CPU (% * OPEN_CPU_UTIL_SCALAR) */
} OPEN_CPU_PERIOD_UTIL_INFO_t;

/*********************************************************************
* @purpose  Get  CPU utilization for all tracked periods.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    data             @b{(output)} CPU Utilization data.
*
* @returns  OPEN_E_NONE      operation was successful.
* @returns  OPEN_E_FAIL      a failure occurred.
* @returns  OPEN_E_PARAM     an invalid parameter was passed.
*
* @notes   Minimum number of bytes in data buffer is found by calling
*          openapiCpuUtilMaxPeriodsGet() and multiplying the result by 
*          sizeof(OPEN_CPU_PERIOD_UTIL_INFO_t). 
* @notes   Values in the utilization field of OPEN_CPU_PERIOD_UTIL_INFO_t
*          are floating point scaled by OPEN_CPU_UTIL_SCALAR and represented
*          as unsigned integer values. For example, 15.4567% will be 
*          represented as 154567. 
* 
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiTotalCpuUtilArrayGet(openapiClientHandle_t *client_handle,
                                         open_buffdesc *data);

/*********************************************************************
* @purpose  Get the maximum number of periods tracked for CPU utilization.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    max              @b{(output)} Maximum number of tracked CPU utilization periods.
*
* @returns  OPEN_E_NONE      operation was successful.
* @returns  OPEN_E_PARAM     an invalid parameter was passed.
*
* @notes    
* 
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiCpuUtilMaxPeriodsGet(openapiClientHandle_t *client_handle,
                                         uint32_t *max);

/*****************************************************************************
 * @purpose   Get a CPU utilization monitoring parameter.
 *
 * @param     client_handle       @b{(input)}  Client handle from registration API
 * @param     param_Type          @b{(input)}  Parameter type
 * @param     *param_val          @b{(output)}  Value of the parameter
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiCpuUtilMonitorParamGet(openapiClientHandle_t *client_handle,
                                           OPEN_CPU_UTIL_MONITOR_PARAMS_t param_Type,
                                           uint32_t *param_val);

/*****************************************************************************
 * @purpose   Set a CPU utilization monitoring parameter.
 *
 * @param     client_handle       @b{(input)}  Client handle from registration API
 * @param     param_Type          @b{(input)}  Parameter type
 * @param     param_val           @b{(input)}  Value of the parameter
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiCpuUtilMonitorParamSet(openapiClientHandle_t *client_handle,
                                           OPEN_CPU_UTIL_MONITOR_PARAMS_t param_Type,
                                           uint32_t param_val);

/*********************************************************************
 * @purpose  Get the task CPU utilization for all tracked periods.
 *
 * @param    client_handle    @b{(input)}  Client handle from
 *                                         registration API
 * @param    data             @b{(output)} Task CPU Utilization data.
 *
 * @returns  OPEN_E_NONE      operation was successful.
 * @returns  OPEN_E_FAIL      a failure occurred.
 * @returns  OPEN_E_PARAM     an invalid parameter was passed.
 *
 * @notes   Minimum number of bytes in data buffer is found by multiplying
 *          (OPEN_CPU_TASK_COUNT_AVERAGE + OPEN_CPU_TASK_UTIL_HEADER_ROWS
 *          (header and footer rows)) with single task print size
 *          OPEN_CPU_TASK_UTIL_SIZE.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *********************************************************************/
open_error_t openapiTaskUtilPrint(openapiClientHandle_t *client_handle,
                                  open_buffdesc *data);

/*****************************************************************************
 * @purpose  Get the configured CPU Free memory threshold.
 *
 * @param    client_handle    @b{(input)}  Client handle from registration API.
 * @param    threshold        @b{(output)} Free memory threshold in KB. A value
 *                                         of 0 indicates that no threshold is
 *                                         configured.
 *
 * @returns  OPEN_E_NONE      On success.
 * @returns  OPEN_E_FAIL      On failure.
 * @returns  OPEN_E_PARAM     Invalid argument.
 *
 * @notes    none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiCpuFreeMemoryThresholdGet(openapiClientHandle_t * client_handle,
                                              uint32_t *threshold);

/*****************************************************************************
 * @purpose  Set the CPU Free memory threshold for monitoring.
 *
 * @param    client_handle    @b{(input)}  Client handle from registration API.
 * @param    threshold        @b{(input)}  Free memory threshold in KB. A value 
 *                                         of 0 indicates that threshold monitoring
 *                                         should be stopped.
 *
 * @returns  OPEN_E_NONE      On success.
 * @returns  OPEN_E_FAIL      On failure.
 * @returns  OPEN_E_PARAM     Invalid argument.
 *
 * @notes    none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiCpuFreeMemoryThresholdSet(openapiClientHandle_t * client_handle,
                                              uint32_t threshold);
#endif  /* OPENAPI_SWITCH_CPU_H_INCLUDED */
