/*********************************************************************
*
*  Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_time.h
*
* @purpose   Date & Time Configuration and Status 
*
* @component OpEN
*
* @create    09/29/2016
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_TIME_H_INCLUDED
#define OPENAPI_TIME_H_INCLUDED

#include "openapi_common.h"

typedef enum
{
  OPEN_CLOCK_SRC_NONE  = 0, /**< Clock Source is local */
  OPEN_CLOCK_SRC_SNTP  = 1, /**< Clock Source is SNTP */
  OPEN_CLOCK_SRC_NTP  = 2   /**< Clock Source is NTP */
} OPEN_TIME_CLOCK_SRC_t;

#define OPEN_TIMEZONE_BUF_LEN       15 /* includes trailing '\0' */

/*********************************************************************
* @purpose  Get the date and time from the system clock
* 
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    bufd             @b{(output)} buffer descriptor containing date and time
*                            Buffer length value @ref OPEN_DATE_TIME_BUF_LEN
* @param    clockSrc         @b{(output)} indicates clock source
*                            Possible values @ref OPEN_TIME_CLOCK_SRC_t
* 
* @returns  OPEN_E_NONE      Get successful.
* @returns  OPEN_E_FAIL      Get failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  SNTP internal error.
* 
* @notes    System clock is returned in "yyyy-mm-dd HH:mm:ss" format.
* 
* @supported-in-version OpEN API Version: 1.14
* 
* @end
*********************************************************************/
open_error_t openapiClockGet(openapiClientHandle_t *client_handle,
                             open_buffdesc *bufd,
                             OPEN_TIME_CLOCK_SRC_t *clockSrc);

/*********************************************************************
* @purpose  Set the system clock date and time
* 
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    bufd             @b{(input)}  buffer descriptor containing date and time
*                            Buffer length value @ref OPEN_DATE_TIME_BUF_LEN
* 
* @returns  OPEN_E_NONE      Set successful.
* @returns  OPEN_E_FAIL      Set failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  SNTP is not supported on the platform or 
*                            an SNTP internal error.
* 
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Supported date format is "yyyy-mm-dd".
* @notes    Supported time format is "HH:mm:ss", where HH is 24-hour format.
* @notes    Leading zeros are not required for the 'set' operation, 
*           (e.g. 6:5:4 implies 06:05:04, 2016-8-1 implies 2016-08-01).
* 
* @supported-in-version OpEN API Version: 1.14
* 
* @end
*********************************************************************/
open_error_t openapiClockSet(openapiClientHandle_t *client_handle,
                             open_buffdesc *bufd);

/*********************************************************************
* @purpose  Get the time zone from the system clock
* 
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    bufd             @b{(output)} buffer descriptor containing time zone
*                            Buffer length value @ref OPEN_TIMEZONE_BUF_LEN
* 
* @returns  OPEN_E_NONE      Get successful.
* @returns  OPEN_E_FAIL      Get failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @notes    Time zone with acronym ZZZZ, if set, is returned in
*           "UTC+hh:mm ZZZZ" or "UTC-hh:mm ZZZZ" format.
* @notes    Time zone without acronym is returned in 
*           "UTC+hh:mm" or "UTC-hh:mm" format.
* 
* @supported-in-version OpEN API Version: 1.14
* 
* @end
*********************************************************************/
open_error_t openapiTimeZoneGet(openapiClientHandle_t *client_handle,
                                open_buffdesc *bufd);

/*********************************************************************
* @purpose  Set the system clock time zone
* 
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    bufd             @b{(input)}  buffer descriptor containing time zone
*                            Buffer length value @ref OPEN_TIMEZONE_BUF_LEN
* 
* @returns  OPEN_E_NONE      Set successful.
* @returns  OPEN_E_FAIL      Set failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Supported input string format is "UTC+hh:mm ZZZZ" or "UTC-hh:mm ZZZZ", 
*           where ZZZZ represents a time zone acronym from 1-4 alpha characters [a-zA-Z].
* @notes    Supported input string format without time zone acronym is
*           "UTC+hh:mm" or "UTC-hh:mm" format.
* @notes    Leading zeros are not required for the hh, mm 'set' operation, 
*           (e.g. UTC+1 implies UTC+01:00).
* 
* @supported-in-version OpEN API Version: 1.14
* 
* @end
*********************************************************************/
open_error_t openapiTimeZoneSet(openapiClientHandle_t *client_handle,
                                open_buffdesc *bufd);

#endif /* OPENAPI_TIME_H_INCLUDED */

