/*********************************************************************
*
*  Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_voice_vlan.h
*
* @purpose   Voice VLAN Configuration and Status
*
* @component OpEN
*
* @create    11/02/2016
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_VOICE_VLAN_H_INCLUDED
#define OPENAPI_VOICE_VLAN_H_INCLUDED

#include "openapi_common.h"

typedef enum
{
  OPEN_VOICE_VLAN_CONFIG_DISABLE  = 0,  /**< voice VLAN disabled, no data needed to configure */
  OPEN_VOICE_VLAN_CONFIG_VLAN_ID  = 1,  /**< tag with specified VLAN ID, VLAN ID specified when configured */
  OPEN_VOICE_VLAN_CONFIG_DOT1P    = 2,  /**< use 802.1p priority tagging, priority data specified when configured */
  OPEN_VOICE_VLAN_CONFIG_NONE     = 3,  /**< allow device to use its own configuration, no data needed to configure */
  OPEN_VOICE_VLAN_CONFIG_UNTAGGED = 4,  /**< send untagged packets, no data needed to configure */
} OPEN_VOICE_VLAN_CONFIG_TYPE_t;

#define OPEN_VOICE_VLAN_DSCP_MIN          0  /**< Minimum value for DSCP */
#define OPEN_VOICE_VLAN_DSCP_MAX          64 /**< Maximum value for DSCP */
#define OPEN_VOICE_VLAN_DOT1P_MIN         0  /**< Minimum value for DOT1P */
#define OPEN_VOICE_VLAN_DOT1P_MAX         7  /**< Maximum value for DOT1P */
/*********************************************************************
* @purpose  Sets Voice VLAN global administrative mode.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    mode            @b{(input)}   administrative mode
*
* @returns  OPEN_E_NONE        Administrative mode set successful.
* @returns  OPEN_E_FAIL        Administrative mode set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiVoiceVlanAdminModeSet(openapiClientHandle_t *client_handle,
                                          OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose  Gets Voice VLAN global administrative mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    mode            @b{(output)} administrative mode
*
* @returns  OPEN_E_NONE        Administrative mode get successful.
* @returns  OPEN_E_FAIL        Administrative mode get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiVoiceVlanAdminModeGet(openapiClientHandle_t *client_handle,
                                          OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose  Set the Voice VLAN interface mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mode            @b{(input)}  mode
* @param    dataValue       @b{(input)}  Certain modes are configured with data.
*                                        See comments for @ref OPEN_VOICE_VLAN_CONFIG_TYPE_t
*                                        for information about which modes require data.
*
* @returns  OPEN_E_NONE        Success
* @returns  OPEN_E_FAIL        Failure
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiVoiceVlanPortAdminModeSet(openapiClientHandle_t *client_handle,
                                              uint32_t ifNum, OPEN_VOICE_VLAN_CONFIG_TYPE_t mode,
                                              uint32_t dataValue);

/*********************************************************************
* @purpose  Get the Voice VLAN interface mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mode            @b{(output)} mode
*
* @returns  OPEN_E_NONE        Success
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiVoiceVlanPortAdminModeGet(openapiClientHandle_t *client_handle,
                                              uint32_t ifNum, OPEN_VOICE_VLAN_CONFIG_TYPE_t *mode);

/*********************************************************************
* @purpose  Get the Voice VLAN port VLAN ID.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    vlanId          @b{(output)} vlanId
*
* @returns  OPEN_E_NONE        Success
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiVoiceVlanPortVlanIdGet(openapiClientHandle_t *client_handle,
                                           uint32_t ifNum, uint32_t *vlanId);

/*********************************************************************
* @purpose  Get the Voice VLAN port DOT1P priority.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    priority        @b{(output)} priority
*
* @returns  OPEN_E_NONE        Success
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiVoiceVlanPortDot1pPriorityGet(openapiClientHandle_t *client_handle,
                                                  uint32_t ifNum, uint32_t *priority);

/*****************************************************************************
 * @purpose   Get the value of the cos override mode.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     mode           @b{(output)} Cos override mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiVoiceVlanPortCosOverrideModeGet(openapiClientHandle_t *client_handle, 
                                                    uint32_t ifNum, 
                                                    OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the value of the cos override mode.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     mode           @b{(input)}  Cos override mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiVoiceVlanPortCosOverrideModeSet(openapiClientHandle_t *client_handle, 
                                                    uint32_t ifNum, 
                                                    OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Get voice vlan Auth State.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     state          @b{(output)} Auth state
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiVoiceVlanPortAuthStateGet(openapiClientHandle_t *client_handle, 
                                              uint32_t ifNum, OPEN_CONTROL_t *state);

/*****************************************************************************
 * @purpose   Set voice vlan Auth State.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     state          @b{(input)}  Auth state
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiVoiceVlanPortAuthStateSet(openapiClientHandle_t *client_handle, 
                                              uint32_t ifNum, OPEN_CONTROL_t state);

/*****************************************************************************
 * @purpose   Get the value of the dscp.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     dscp           @b{(output)} Dscp value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiVoiceVlanPortDscpGet(openapiClientHandle_t *client_handle, 
                                         uint32_t ifNum, uint32_t *dscp);

/*****************************************************************************
 * @purpose   Set the value of the dscp.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     dscp           @b{(input)}  Dscp value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiVoiceVlanPortDscpSet(openapiClientHandle_t *client_handle, 
                                         uint32_t ifNum, uint32_t dscp);

/*****************************************************************************
 * @purpose   Set the value of the dot1p.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     dot1pPriority  @b{(input)}  Dot1p priority value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiVoiceVlanPortDot1pPrioritySet(openapiClientHandle_t *client_handle, 
                                                 uint32_t ifNum, 
                                                 uint32_t dot1pPriority);

/*****************************************************************************
 * @purpose   Set the value of the vlan ID.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     vlanId         @b{(input)}  Vlan ID
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiVoiceVlanPortVlanIdSet(openapiClientHandle_t *client_handle, 
                                           uint32_t ifNum, uint32_t vlanId);

/*****************************************************************************
 * @purpose   Get the value voice vlan status.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  interface
 * @param     mode           @b{(output)} Voice vlan status
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiVoiceVlanPortStatusGet(openapiClientHandle_t *client_handle, 
                                           uint32_t ifNum, OPEN_CONTROL_t *mode);


#endif /* OPENAPI_VOICE_VLAN_H_INCLUDED */
