/*********************************************************************
*
*  Copyright 2017-2023 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_voip.h
*
* @purpose   Auto-VoIP Configuration
*
* @component OpEN
*
* @comments  This code must not include any ICOS header files.
*
* @create    08/02/2017
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_VOIP_H_INCLUDED
#define OPENAPI_VOIP_H_INCLUDED

#include "openapi_common.h"

/**< Voip types */
typedef enum
{
  OPEN_QOS_VOIP_TYPE_PROTOCOL=1,   /**< Protocol based VOIP */                 
  OPEN_QOS_VOIP_TYPE_OUI,          /**< OUI based VOIP */ 
  OPEN_QOS_VOIP_TYPE_TOTAL         /**< Last VOIP type */                
} OPEN_QOS_VOIP_TYPE_t;

/**< Voip priority schemes */
typedef enum
{
   OPEN_VOIP_PRI_REMARK = 1,    /**< Re mark */
   OPEN_VOIP_PRI_TRAFFIC_CLASS, /**< Traffic class */
} OPEN_VOIP_PRI_SCHEME_t; 

#define OUI_STR_LEN 9
#define OUI_DESC_STRINGLEN 33
/*********************************************************************
* @purpose  Sets the VLAN ID for voip traffic on switch          
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    vlanId             @b{(input)}  VLAN ID 
*
* @returns  OPEN_E_NONE        Set successful.    
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipGlobalVlanIdSet(openapiClientHandle_t *client_handle,
                                        uint32_t vlanId);

/*********************************************************************
* @purpose  Resets the VLAN ID for voip traffic on switch          
*
* @param    client_handle      @b{(input)}  client handle from registration API
*
* @returns  OPEN_E_NONE        Set successful.    
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipGlobalVlanIdReset(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Gets the VLAN ID for voip traffic on switch          
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    vlanId             @b{(output)} VLAN ID 
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipGlobalVlanIdGet(openapiClientHandle_t *client_handle,
                                        uint32_t *vlanId);

/*********************************************************************
* @purpose  Get the maximum number of Auto-VoIP calls supported          
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    maxCalls           @b{(output)} maximum number of Auto-VoIP calls 
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipMaxCallsGet(openapiClientHandle_t *client_handle,
                                    uint32_t *maxCalls);

/*********************************************************************
* @purpose  Gets VoIP Protocol Mode Active Sessions 
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    sessions           @b{(output)} number of Auto-VoIP sessions 
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipNumActiveProtocolSessionsGet(openapiClientHandle_t *client_handle,
                                                     uint32_t *sessions);

/*********************************************************************
* @purpose  Gets the VOIP profile Cos Queue 
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    pType              @b{(output)} priScheme (remark or traffic-class) 
* @param    pTypeVal           @b{(output)} value
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipProfileCosQueueGet(openapiClientHandle_t *client_handle,
                                           OPEN_VOIP_PRI_SCHEME_t *pType,
                                           uint32_t *pTypeVal);

/*********************************************************************
* @purpose  Sets the VOIP profile cos queue 
*
* @param    client_handle      @b{(input)} client handle from registration API
* @param    pType              @b{(input)} priScheme (remark or traffic-class) 
* @param    pTypeVal           @b{(input)} value
*
* @returns  OPEN_E_NONE        Set successful.    
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipProfileCosQueueSet(openapiClientHandle_t *client_handle,
                                           OPEN_VOIP_PRI_SCHEME_t pType,
                                           uint32_t pTypeVal);

/*********************************************************************
* @purpose  Gets the VOIP profile mode on interface 
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface to be configured
* @param    type           @b{(input)}  Auto VoIP type (protocol/OUI based)
* @param    mode           @b{(output)} Mode value
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipIntfModeGet(openapiClientHandle_t *client_handle,
                                    uint32_t ifNum, 
                                    OPEN_QOS_VOIP_TYPE_t type,
                                    OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose  Sets the VOIP profile mode on interface 
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface to be configured
* @param    type           @b{(input)}  Auto VoIP type (protocol/OUI based)
* @param    mode           @b{(input)}  Mode value
*
* @returns  OPEN_E_NONE        Set successful.    
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipIntfModeSet(openapiClientHandle_t *client_handle,
                                    uint32_t ifNum, 
                                    OPEN_QOS_VOIP_TYPE_t type,
                                    OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose  Gets VoIP Protocol mode operational port status 
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  Interface to be configured
* @param    val            @b{(output)} Operational status (UP or DOWN)
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipIntfProtocolStatusGet(openapiClientHandle_t *client_handle,
                                              uint32_t ifNum, 
                                              OPEN_LINK_STATE_t *val);

/*********************************************************************
* @purpose  Gets VoIP OUI mode operational port status 
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  Interface to be configured
* @param    val            @b{(output)} Operational status (UP or DOWN)
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipIntfOUIStatusGet(openapiClientHandle_t *client_handle,
                                         uint32_t ifNum, 
                                         OPEN_LINK_STATE_t *val);

/*********************************************************************
* @purpose  Gets the priority for voip traffic on switch 
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    priority       @b{(output)} priority value
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipOUIPriorityGet(openapiClientHandle_t *client_handle,
                                       uint32_t *priority); 

/*********************************************************************
* @purpose  Sets the priority for voip traffic on switch 
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    priority       @b{(input)}  priority value
*
* @returns  OPEN_E_NONE        Set successful.    
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipOUIPrioritySet(openapiClientHandle_t *client_handle,
                                       uint32_t priority); 

/*****************************************************************************
 * @purpose   Adds a voip OUI entry.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     voipOui        @b{(input)}  Voip OUI string
 * @param     ouiDesc        @b{(input)}  Voip OUI description
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 * @notes     OUI should contain ascii string as "xx:xx:xx"
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiVoipOuiAdd(openapiClientHandle_t *client_handle, 
                               open_buffdesc *voipOui, open_buffdesc *ouiDesc);

/*****************************************************************************
 * @purpose   Removes a voip OUI entry.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     voipOui        @b{(input)}  Voip OUI string
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 * @notes     OUI should contain ascii string as "xx:xx:xx"
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiVoipOuiRemove(openapiClientHandle_t *client_handle, 
                                  open_buffdesc *voipOui);

/*****************************************************************************
 * @purpose   Gets the count of OUIs present in voip database.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     voipOuiCount   @b{(output)} Number of OUIs present in database
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     None
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiVoipOuiCountGet(openapiClientHandle_t *client_handle, 
                                    uint32_t *voipOuiCount);

/*****************************************************************************
 * @purpose   Gets the voip OUI entry for the index.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ouiIndex       @b{(input)}  OUI index
 * @param     voipOui        @b{(output)} Voip OUI string
 * @param     ouiDesc        @b{(output)} Voip OUI description
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     OUI contains ascii string as "xx:xx:xx"
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiVoipOuiEntryGet(openapiClientHandle_t *client_handle, 
                                    uint32_t ouiIndex, open_buffdesc *voipOui, 
                                    open_buffdesc *ouiDesc);

/*****************************************************************************
 * @purpose   Gets the voip OUI entry for the given index and also returns next 
 *            OUI Index.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ouiIndex       @b{(input)}  OUI index
 * @param     voipOui        @b{(output)} Voip OUI string
 * @param     ouiDesc        @b{(output)} Voip OUI description
 * @param     nextOuiIndex   @b{(output)} Next OUI index
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     OUI contains ascii string as "xx:xx:xx"
 * @notes     This API returns the next OUI index. It may not be valid. User of 
 *            this API needs to enusre the validity by invoking
 *            openapiVoipOuiEntryGet()
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiVoipOuiEntryNextGet(openapiClientHandle_t *client_handle, 
                                        uint32_t ouiIndex, open_buffdesc *voipOui,
                                        open_buffdesc *ouiDesc, 
                                        uint32_t *nextOuiIndex);

/*****************************************************************************
 * @purpose   Gets the first voip OUI entry.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     voipOui        @b{(output)} Voip OUI string
 * @param     ouiDesc        @b{(output)} Voip OUI description
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     OUI contains ascii string as "xx:xx:xx"
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiVoipOuiFirstEntryGet(openapiClientHandle_t *client_handle, 
                                         open_buffdesc *voipOui, 
                                         open_buffdesc *ouiDesc);

/*****************************************************************************
 * @purpose   Gets the voip OUI description for a given OUI.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     voipOui        @b{(input)}  Voip OUI string
 * @param     ouiDesc        @b{(output)} Voip OUI description
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     OUI contains ascii string as "xx:xx:xx"
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiVoipOuiDescGet(openapiClientHandle_t *client_handle, 
                                   open_buffdesc *voipOui, open_buffdesc *ouiDesc);

/*****************************************************************************
 * @purpose   Gets the next voip OUI entry for a given OUI.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     voipOui        @b{(input)}  Voip OUI string
 * @param     nextVoipOui    @b{(output)} Next voip OUI string
 * @param     nextOuiDesc    @b{(output)} Next voip OUI description
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     OUI contains ascii string as "xx:xx:xx"
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiVoipOuiNextEntryGet(openapiClientHandle_t *client_handle, 
                                        open_buffdesc *voipOui, 
                                        open_buffdesc *nextVoipOui, 
                                        open_buffdesc *nextOuiDesc);

/*****************************************************************************
 * @purpose   Adds a voip OUI entry in the given index.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ouiIndex       @b{(input)}  OUI index
 * @param     voipOui        @b{(input)}  Voip OUI string
 * @param     ouiDesc        @b{(input)}  Voip OUI description
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 * @notes     OUI contains ascii string as "xx:xx:xx"
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiVoipIndexOuiAdd(openapiClientHandle_t *client_handle, 
                                    uint32_t ouiIndex, open_buffdesc *voipOui, 
                                    open_buffdesc *ouiDesc);

/*****************************************************************************
 * @purpose   Removes a voip OUI entry in the given index.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ouiIndex       @b{(input)}  OUI index
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Calling this API will change the running configuration of the switch
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiVoipIndexOuiRemove(openapiClientHandle_t *client_handle, 
                                       uint32_t ouiIndex);

#endif /*#ifndef OPENAPI_VOIP_H_INCLUDED*/
