/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  app_status_example.c
*
* @purpose   Application Status OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    11/23/2015
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_app_ctrl.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the App Control Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printAppCtrlMenu()
{
  printf("\nUsage:  app_ctrl_example <test#> <arg1> <arg2> ... \n\n");
  printf("Test 1: Create a new Application Process Table entry for the given file name: app_ctrl_example 1 <app-name> \n");
  printf("Test 2: Delete an Application Process Table entry for the given file name: app_ctrl_example 2 <app-name> \n");
  printf("Test 3: Get start on boot, auto-restart, CPU sharing and memory limit settings for the process entry: app_ctrl_example 3 <app-name> \n");
  printf("Test 4: Set start on boot, auto-restart, CPU sharing and memory limit settings for the process entry: app_ctrl_example 4 <app-name> <start-on-boot> <auto-restart> <cpu-share> <max-memory> \n");
  printf("Test 5: Start the process with Process Manager using the parameters configured in the process entry: app_ctrl_example 5 <app-name> \n");
  printf("Test 6: Stop the process with Process Manager: app_ctrl_example 6 <app-name> \n");
  printf("Test 7: Get application status for the process entry: app_ctrl_example 7 <app-name> \n");
  printf("Test 8: Set application status for the process entry: app_ctrl_example 8 <app-name> <status-info> \n");
  printf("Test 9: Get application version for the process entry: app_ctrl_example 9 <app-name> \n");
  printf("Test 10: Set application version for the process entry: app_ctrl_example 10 <app-name> <version-identifier> \n");
  printf("\n");

  return;
}

/*********************************************************************
* @purpose  Create a new Application Process Table entry for the given file name. 
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    appExecName      @b{(input)}   application name
*
* @returns  none
* 
* @notes     
* @end
*********************************************************************/
void appCtrlEntryCreate(openapiClientHandle_t *clientHandle, char *appExecName)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, appExecName, (sizeof(str) - 1));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  if ((result = openapiAppCtrlEntryCreate(clientHandle, &buffDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to create a new application process table entry. (result = %d)\n", result);
  }
  else
  {
    printf("New application process table entry created successfully. \n");
  }

  return;
}

/*********************************************************************
* @purpose  Delete an Application Process Table entry for the given file name. 
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    appExecName      @b{(input)}   application name
*
* @returns  none
* 
* @notes     
* @end
*********************************************************************/
void appCtrlEntryDelete(openapiClientHandle_t *clientHandle, char *appExecName)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, appExecName, (sizeof(str) - 1));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  if ((result = openapiAppCtrlEntryDelete(clientHandle, &buffDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete an application process table entry. (result = %d)\n", result);
  }
  else
  {
    printf("Application process table entry deleted successfully. \n");
  }

  return;
}

/*********************************************************************
* @purpose  Get start on boot, auto-restart, CPU sharing and memory limit settings for the process entry. 
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    appExecName      @b{(input)}   application name
*
* @returns  none
* 
* @notes     
* @end
*********************************************************************/
void appCtrlAttrGet(openapiClientHandle_t *clientHandle, char *appExecName)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100];
  OPEN_CONTROL_t startOnBoot = 0;
  OPEN_CONTROL_t autoRestart = 0;
  uint32_t cpuShare = 0;
  uint32_t maxMemory = 0;

  memset(str, 0, sizeof(str));
  strncpy(str, appExecName, (sizeof(str) - 1));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  if ((result = openapiAppCtrlAttrGet(clientHandle, &buffDesc, &startOnBoot, &autoRestart, &cpuShare, &maxMemory)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get settings for the process entry. (result = %d)\n", result);
  }
  else
  {
    printf("Start on boot mode: %s is set\n", (startOnBoot == OPEN_ENABLE ? "Enabled" : "Disabled"));
    printf("Auto restart mode: %s is set\n", (autoRestart == OPEN_ENABLE ? "Enabled" : "Disabled"));
    printf("CPU share: (%u)%%\n", cpuShare);
    printf("Maximum memory an application is allowed to consume: (%u)MB\n", maxMemory);
  }

  return;
}

/*********************************************************************
* @purpose  Set start on boot, auto-restart, CPU sharing and memory limit settings for the process entry. 
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    appExecName      @b{(input)}   application name
* @param    startOnBootStr   @b{(input)}   start on boot
* @param    autoRestartStr   @b{(input)}   auto restart 
* @param    cpuShareStr      @b{(input)}   CPU sharing 
* @param    maxMemoryStr     @b{(input)}   memory limit 
*
* @returns  none
* 
* @notes    Calling this API will overwrite the app settings in the switch.
*           If user don't want to set any specific parameter, then pass NULL as value for that specific parameter.
* 
* @end
*********************************************************************/
void appCtrlAttrSet(openapiClientHandle_t *clientHandle, char *appExecName,
                    char *startOnBootStr, char *autoRestartStr,
                    char *cpuShareStr, char *maxMemoryStr)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100];
  OPEN_CONTROL_t *startOnBoot = NULL;
  OPEN_CONTROL_t *autoRestart = NULL;
  uint32_t *cpuShare = NULL;
  uint32_t *maxMemory = NULL;
  int val1, val2, val3, val4;

  memset(str, 0, sizeof(str));
  strncpy(str, appExecName, (sizeof(str) - 1));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  if (strcmp(startOnBootStr, "NULL") != 0)
  {
    val1 = atoi(startOnBootStr);
    startOnBoot = (OPEN_CONTROL_t *)&val1;
  }

  if (strcmp(autoRestartStr, "NULL") != 0)
  {
    val2 = atoi(autoRestartStr);
    autoRestart = (OPEN_CONTROL_t *)&val2;
  }

  if (strcmp(cpuShareStr, "NULL") != 0)
  {
    val3 = atoi(cpuShareStr);
    cpuShare = (uint32_t *)&val3;
  }

  if (strcmp(maxMemoryStr, "NULL") != 0)
  {
    val4 = atoi(maxMemoryStr);
    maxMemory = (uint32_t *)&val4;
  }

  if ((result = openapiAppCtrlAttrSet(clientHandle, &buffDesc, startOnBoot, autoRestart, cpuShare, maxMemory)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set application control attributes for the process entry. (result = %d)\n", result);
  }
  else
  {
    printf("Application control attributes set successfully for process entry. \n");
  }

  return;
}

/*********************************************************************
* @purpose  Start the process with Process Manager using the parameters configured in the process entry. 
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    appExecName      @b{(input)}   application name
*
* @returns  none
* 
* @notes     
* @end
*********************************************************************/
void appCtrlStart(openapiClientHandle_t *clientHandle, char *appExecName)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, appExecName, (sizeof(str) - 1));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  if ((result = openapiAppCtrlStart(clientHandle, &buffDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to start an application. (result = %d)\n", result);
  }
  else
  {
    printf("Application start successfully.. \n");
  }

  return;
}

/*********************************************************************
* @purpose  Stop the process with Process Manager. 
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    appExecName      @b{(input)}   application name
*
* @returns  none
* 
* @notes     
* @end
*********************************************************************/
void appCtrlStop(openapiClientHandle_t *clientHandle, char *appExecName)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, appExecName, (sizeof(str) - 1));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  if ((result = openapiAppCtrlStop(clientHandle, &buffDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to stop an application. (result = %d)\n", result);
  }
  else
  {
    printf("Application stop successfully. \n");
  }

  return;
}

/*********************************************************************
* @purpose  Get status string for a given application.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    appExecName      @b{(input)}   application name
*
* @returns  none
* 
* @notes     
* @end
*********************************************************************/
void appCtrlStatusGet(openapiClientHandle_t *clientHandle, char *appExecName)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100];
  open_buffdesc statusInfo;
  char bufStatusInfo[OPEN_APP_CTRL_STATUS_SIZE] = {0};

  statusInfo.pstart = bufStatusInfo;
  statusInfo.size = sizeof(bufStatusInfo);

  memset(str, 0, sizeof(str));
  strncpy(str, appExecName, (sizeof(str) - 1));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  if ((result = openapiAppCtrlStatusGet(clientHandle, &buffDesc, &statusInfo)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get an application status for process entry. (result = %d)\n", result);
  }
  else
  {
    printf("Application status for process entry: %s\n", (char *)statusInfo.pstart);
  }

  return;
}

/*********************************************************************
* @purpose  Set status string for a given application.
*
* @param    clientHandle      @b{(input)}   client handle from registration API
* @param    appExecName       @b{(input)}   application name
* @param    statusInformation @b{(input)}   application status string
*
* @returns  none
* 
* @notes    Calling this API will overwrite the app status info in the switch.
* 
* @end
*********************************************************************/
void appCtrlStatusSet(openapiClientHandle_t *clientHandle, char *appExecName, char *statusInformation)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100];
  open_buffdesc statusInfo;
  char bufStatusInfo[OPEN_APP_CTRL_STATUS_SIZE];

  memset(str, 0, sizeof(str));
  strncpy(str, appExecName, (sizeof(str) - 1));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(bufStatusInfo, 0, sizeof(bufStatusInfo));
  strncpy(bufStatusInfo, statusInformation, (sizeof(bufStatusInfo) - 1));
  statusInfo.pstart = bufStatusInfo;
  statusInfo.size = strlen(bufStatusInfo) + 1;

  if ((result = openapiAppCtrlStatusSet(clientHandle, &buffDesc, &statusInfo)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set an application status for process entry. (result = %d)\n", result);
  }
  else
  {
    printf("Successfully set an application status for process entry. \n");
  }

  return;
}

/*********************************************************************
* @purpose  Get version string for a given application.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    appExecName      @b{(input)}   application name
*
* @returns   none
* 
* @notes
* @end
*********************************************************************/
void appCtrlVersionGet(openapiClientHandle_t *clientHandle, char *appExecName)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100];
  open_buffdesc versionIdentifier;
  char bufVersionIdentifier[OPEN_APP_CTRL_VERSION_SIZE] = {0};

  versionIdentifier.pstart = bufVersionIdentifier;
  versionIdentifier.size = sizeof(bufVersionIdentifier);

  memset(str, 0, sizeof(str));
  strncpy(str, appExecName, (sizeof(str) - 1));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  if ((result = openapiAppCtrlVersionGet(clientHandle, &buffDesc, &versionIdentifier)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get an application version for process entry. (result = %d)\n", result);
  }
  else
  {
    printf("Application version for process entry: %s\n", (char *)versionIdentifier.pstart);
  }

  return;
}

/*********************************************************************
* @purpose  Set version string for a given application.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    appExecName      @b{(input)}   application name
* @param    versionId        @b{(input)}   application version string
*
* @returns   none
* 
* @notes     Calling this API will overwrite the app version info in the switch.
* 
* @end
*********************************************************************/
void appCtrlVersionSet(openapiClientHandle_t *clientHandle, char *appExecName, char *versionId)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100];
  open_buffdesc versionIdentifier;
  char bufVersionIdentifier[OPEN_APP_CTRL_VERSION_SIZE];

  memset(str, 0, sizeof(str));
  strncpy(str, appExecName, (sizeof(str) - 1));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(bufVersionIdentifier, 0, sizeof(bufVersionIdentifier));
  strncpy(bufVersionIdentifier, versionId, (sizeof(bufVersionIdentifier) - 1));
  versionIdentifier.pstart = bufVersionIdentifier;
  versionIdentifier.size = strlen(bufVersionIdentifier) + 1;

  if ((result = openapiAppCtrlVersionSet(clientHandle, &buffDesc, &versionIdentifier)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set an application version for process entry. (result = %d)\n", result);
  }
  else
  {
    printf("Successfully set an application version for process entry. \n");
  }

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t          result;
  open_buffdesc         bufd;
  char                  tmp[256];
  open_revision_data_t  version;
  uint32_t testNum;
  
  if (argc < 2)
  {
    printAppCtrlMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("app_ctrl_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting App Control API example application");

  /* prepare buffer to receive string */
  memset(tmp, 0, sizeof(tmp));
  bufd.pstart = tmp;
  bufd.size = sizeof(tmp);

  printf("\n");
  if (openapiNetworkOSVersionGet(&clientHandle, &bufd) == OPEN_E_NONE)
  {
    printf("Network OS version = %s\n", tmp);
  }
  else
  {
    printf("Network OS version retrieve error\n");
  }

  if (openapiApiVersionGet(&clientHandle, &version) == OPEN_E_NONE)
  {
    printf("OpEN version = %u.%u.%u.%u\n", version.release, version.version, version.maint_level, version.build_num);
  }
  else
  {
    printf("OpEN version retrieve error.\n");
  }

  printf("\n");
  switch (testNum)
  {
    case 1:
      if (argc != 3)
      {
        printAppCtrlMenu();
        exit(1);
      }
      appCtrlEntryCreate(&clientHandle, argv[2]);
      break;

    case 2:
      if (argc != 3)
      {
        printAppCtrlMenu();
        exit(1);
      }
      appCtrlEntryDelete(&clientHandle, argv[2]);
      break;

    case 3:
      if (argc != 3)
      {
        printAppCtrlMenu();
        exit(1);
      }
      appCtrlAttrGet(&clientHandle, argv[2]);
      break;

    case 4:
      if (argc != 7)
      {
        printAppCtrlMenu();
        exit(1);
      }
      appCtrlAttrSet(&clientHandle, argv[2], argv[3], argv[4], argv[5], argv[6]);
      break;

    case 5:
      if (argc != 3)
      {
        printAppCtrlMenu();
        exit(1);
      }
      appCtrlStart(&clientHandle, argv[2]);
      break;

    case 6:
      if (argc != 3)
      {
        printAppCtrlMenu();
        exit(1);
      }
      appCtrlStop(&clientHandle, argv[2]);
      break;

    case 7:
      if (argc != 3)
      {
        printAppCtrlMenu();
        exit(1);
      }
      appCtrlStatusGet(&clientHandle, argv[2]);
      break;

    case 8:
      if (argc != 4)
      {
        printAppCtrlMenu();
        exit(1);
      }
      appCtrlStatusSet(&clientHandle, argv[2], argv[3]);
      break;

    case 9:
      if (argc != 3)
      {
        printAppCtrlMenu();
        exit(1);
      }
      appCtrlVersionGet(&clientHandle, argv[2]);
      break;

    case 10:
      if (argc != 4)
      {
        printAppCtrlMenu();
        exit(1);
      }
      appCtrlVersionSet(&clientHandle, argv[2], argv[3]);
      break;

    default:
      printAppCtrlMenu();
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping App Control API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
