#!/usr/bin/ruby

require 'optparse'
require "OpEN"
require "OpENUtil"
require "ipaddr"

#
# Copyright 2016 Broadcom.
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#


#
# Ruby 1.8.7.
#

$open = OpENUtil.new()

def performAction(addFamily)
  # Demonstrate OpEN usage for BGP APIs

  ret = $open.connect("bgp_statistics")
  if ret == OpEN::OPEN_E_NONE
    printf("\n")
    $open.getNetworkOSVersion()
    $open.getAPIVersion()
    client = $open.client

    # Fetch IPv4 Routing
    v4rtrAdminMode_p = OpEN.new_OPEN_CONTROL_tp()
    v4rtrAdminMode_rc = OpEN.openapiRtrAdminModeGet(client, OpEN::OPEN_AF_INET, v4rtrAdminMode_p)
    if v4rtrAdminMode_rc == OpEN::OPEN_E_NONE
       v4rtrAdminMode = OpEN.OPEN_CONTROL_tp_value(v4rtrAdminMode_p)
    end
    OpEN.delete_OPEN_CONTROL_tp(v4rtrAdminMode_p)

    # Fetch IPv6 Routing
    v6rtrAdminMode_p = OpEN.new_OPEN_CONTROL_tp()
    v6rtrAdminMode_rc = OpEN.openapiRtrAdminModeGet(client, OpEN::OPEN_AF_INET6, v6rtrAdminMode_p)
    if v6rtrAdminMode_rc == OpEN::OPEN_E_NONE
       v6rtrAdminMode = OpEN.OPEN_CONTROL_tp_value(v6rtrAdminMode_p)
    end
    OpEN.delete_OPEN_CONTROL_tp(v6rtrAdminMode_p)

    # Fetch BGP Admin Mode
    bgpAdminMode_p = OpEN.new_OPEN_CONTROL_tp()
    bgpAdminMode_rc = OpEN.openapiBgpAdminModeGet(client, bgpAdminMode_p)
    if bgpAdminMode_rc == OpEN::OPEN_E_NONE
       bgpAdminMode = OpEN.OPEN_CONTROL_tp_value(bgpAdminMode_p)
    end
    OpEN.delete_OPEN_CONTROL_tp(bgpAdminMode_p)

    # Fetch BGP Router ID
    bgpLocalId_p = OpEN.new_uint32_tp()
    bgpLocalId = 0
    bgpLocalId_rc = OpEN.openapiBgpLocalIdGet(client, bgpLocalId_p)
    if bgpLocalId_rc == OpEN::OPEN_E_NONE
       bgpLocalId = OpEN.uint32_tp_value(bgpLocalId_p)
    end
    OpEN.delete_uint32_tp(bgpLocalId_p)

    bgp_valid_status = (v4rtrAdminMode_rc | v6rtrAdminMode_rc | bgpAdminMode_rc | bgpLocalId_rc)
    if bgp_valid_status != OpEN::OPEN_E_NONE
       printf("\r\nWarning:  Unknown error - BGP data is unavailable at this time.")
    else
      routingEnabled = OpEN::OPEN_FALSE
      if ((v4rtrAdminMode == OpEN::OPEN_ENABLE) || (v6rtrAdminMode == OpEN::OPEN_ENABLE))
        routingEnabled = OpEN::OPEN_TRUE
      end

      valid = true
      if ((bgpAdminMode == OpEN::OPEN_DISABLE) || !routingEnabled || (bgpLocalId == 0))
         valid = false
         printf("\r\nWarning:  BGP is not running. Further data is unavailable.")
      end

      i = 0
      if routingEnabled == OpEN::OPEN_FALSE
         i += 1
         printf("\r\n   %d) IP routing is not enabled.", ++i)
      end
      if bgpAdminMode == OpEN::OPEN_DISABLE
         i += 1
         printf("\r\n   %d) BGP is administratively disabled.", ++i)
      end
      if bgpLocalId == 0
         i += 1
         printf("\r\n   %d) No BGP router ID is configured.", ++i);
      end

      if valid
         numEntries_p = OpEN.new_uint32_tp()
         result = OpEN.openapiBgpMapDecProcHistoryCountGet(client, addFamily, numEntries_p)
         if OpEN::OPEN_E_NONE == result
            numEntries = OpEN.uint32_tp_value(numEntries_p)
            printf("\n%9s%6s%8s%6s%16s%16s%10s%6s%6s%6s\n",
                   "Delta T","Phase","Upd Grp","GenId","Reason","Peer",
                   "Duration","Adds","Mods","Dels")
            $i = 0
            until $i > (numEntries-1) do
              pHist_p = OpEN.new_open_bgpDecProcHist_tp()
              rc = OpEN.openapiBgpMapDecProcHistoryByIndexGet(client, addFamily,
                                                              $i, pHist_p)
              if OpEN::OPEN_E_NONE == rc
                 pHist = OpEN.open_bgpDecProcHist_tp_value(pHist_p)

                 timeStr = ""
                 if pHist.decProcTime != 0
                   let = pHist.decProcTime
                   mm, ss = let.divmod(60)
                   hh, mm = mm.divmod(60)
                   dd, hh = hh.divmod(24)
                   if dd >= 1
                     timeStr = "%02ud:%02uh" % [dd, hh]
                   else
                     timeStr = "%02u:%02u:%02u" % [hh, mm, ss]
                   end
                 end

                 updateGroupStr = ""
                 if (pHist.updateGroupId != 65535)
                   updateGroupStr = "%u" % pHist.updateGroupId
                 end

                 peerAddrStr = ""
                 tmp_p = OpEN.new_OPEN_BOOL_tp()
                 rc = OpEN.openapiIsInetAddrZero(client, pHist.peerAddr, tmp_p)
                 if OpEN::OPEN_E_NONE == rc
                    isZero = OpEN.OPEN_BOOL_tp_value(tmp_p)
                    if isZero == OpEN::OPEN_TRUE
                       max_len_p = OpEN.new_uint32_tp()
                       result = OpEN.openapiIpAddressMaxStringLength(client, max_len_p)
                       if result == OpEN::OPEN_E_NONE
                          max_len = OpEN.uint32_tp_value(max_len_p)
                          buff_string = $open.getCharBuffer(max_len)
                          buff        = OpEN::Open_buffdesc.new
                          buff.pstart = buff_string
                          buff.size   = max_len
                          result = OpEN.openapiOpenIPtoStringGet(client, remoteAddr, buff)
                          if (result == OpEN::OPEN_E_NONE and (buff_string.cast().length>0))
                             peerAddrStr = buff_string.cast()
                          end
                          OpEN.delete_uint32_tp(max_len_p)
                       end
                    end
                 end
                 OpEN.delete_OPEN_BOOL_tp(tmp_p)
                 reasonStr = ""
                 reasonStr = OpEN.getStringFromUcharArray(pHist.reason)

                 printf("%9s%6u%8s%6u%16s%16s%10u%6u%6u%6u\n", timeStr,
                                                               pHist.decProcPhase,
                                                               updateGroupStr,
                                                               pHist.genId,
                                                               reasonStr,
                                                               peerAddrStr,
                                                               pHist.duration,
                                                               pHist.adds,
                                                               pHist.mods,
                                                               pHist.dels)
              end
              $i += 1
              OpEN.delete_open_bgpDecProcHist_tp(pHist_p)
            end
         end
      end
    end
    OpEN.delete_uint32_tp(numEntries_p)
    $open.terminate()
    printf("\n")
  else
    print "Unable to connect"
  end
end

options = {}
optparse = OptionParser.new do |opts|
  opts.on('-4', '--ipv4', 'display statistics about the BGP IPv4 decision process') do
    options[:ipv4] = true
  end

  opts.on('-6', '--ipv6', 'display statistics about the BGP IPv6 decision process') do
    options[:ipv6] = true
  end

  opts.on('-h', '--help', 'display this message') do
    puts opts
    exit
  end
end

begin
  optparse.parse!
  t = options.length
  if (t > 1) || (t == 0)
    puts optparse
    exit
  end
rescue OptionParser::InvalidOption, OptionParser::MissingArgument
  puts $!.to_s
  puts optparse
  exit
end

if options[:ipv4] != nil
  performAction(OpEN::OPEN_AF_INET)
end

if options[:ipv6] != nil
  performAction(OpEN::OPEN_AF_INET6)
end

