#
# Chef Cookbook   : netdev_icos
# File            : provider/vlan.rb
#    
# Copyright 2016 Broadcom.
#
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#

require "OpEN"
require "OpENUtil"

def whyrun_supported?
  true
end

action :create do
  if !@current_resource.exists
    converge_by("create VLAN '#{@new_resource.vlan_id}' and set name to '#{@new_resource.name}'") do
      create_vlan
    end
  else
    edit_vlan
  end
end

action :delete do
  if @current_resource.exists
    converge_by("remove VLAN '#{@new_resource.vlan_id}'") do
      delete_vlan
    end
  else
    msg = "VLAN doesn't exist, nothing to delete"
    log_msg("info", msg)
  end
end

def load_current_resource
  msg = "Loading current resource '#{@new_resource.name}'"
  log_msg("info", msg)
  @current_resource = Chef::Resource::NetdevVlan.new(@new_resource.name)
  @current_resource.vlan_id(@new_resource.vlan_id)
  @current_resource.exists = false
 
  if resource_exists?
    vlan_name = get_vlan_name(new_resource.vlan_id) 
    @current_resource.vlan_id(@new_resource.vlan_id)
    @current_resource.name(vlan_name)
    @current_resource.exists = true
  else
    msg = "VLAN '#{@new_resource.vlan_id}' doesn't exist"
    log_msg("info", msg)
  end
end

def vlan_exists(vid)
  open = OpENUtil.new()
  conn_ret = open.connect("netdev-icos-vlan")
  exists = false
  if conn_ret == OpEN::OPEN_E_NONE
    client = open.client
    ret = OpEN::openapiVlanCreatedCheck(client,vid)
    if ret == OpEN::OPEN_E_NONE
      exists = true
    end
  else
    msg = "Failed to connect to 'netdev-icos-vlan'"
    log_msg("info", msg)
  end
  open.terminate()
  return exists
end

def resource_exists?
  msg = "Looking to see if VLAN '#{@new_resource.vlan_id}' exists"
  log_msg("info", msg)
  return vlan_exists(@new_resource.vlan_id)
end

def set_vlan_name(vlan_id, name)
  open = OpENUtil.new()
  conn_ret = open.connect("netdev-icos-vlan")
  if conn_ret == OpEN::OPEN_E_NONE
    client = open.client
    buffdesc = OpEN::Open_buffdesc.new
    buffdesc.size   = name.length + 1
    buffdesc.pstart = open.getCharBuffer(buffdesc.size,name)
    ret = OpEN::openapiVlanNameSet(client,vlan_id,buffdesc)
    if ret != OpEN::OPEN_E_NONE
      msg = "Failed to set name '#{name}' for VLAN '#{vlan_id}', returned '#{ret}'"
      log_msg("info", msg)
    end
  else
    msg = "Failed to connect to 'netdev-icos-vlan'"
    log_msg("info", msg)
  end
  open.terminate()
end

def get_vlan_name(vlan_id)
  vlan_name = ""
  open = OpENUtil.new()
  conn_ret = open.connect("netdev-icos-vlan")
  if conn_ret == OpEN::OPEN_E_NONE
    client = open.client

    # Get the maximum vlan name length
    name_len_p = OpEN.new_uint32_tp()
    ret = OpEN::openapiVlanNameLenGet(client, name_len_p)
    if ret == OpEN::OPEN_E_NONE
      name_len = OpEN.uint32_tp_value(name_len_p)

      # Get the vlan name
      buffdesc   = OpEN::Open_buffdesc.new
      buffdesc.size   = name_len
      name = open.getCharBuffer(buffdesc.size, nil)
      buffdesc.pstart = name
      ret = OpEN::openapiVlanNameGet(client, vlan_id, buffdesc)
      if ret == OpEN::OPEN_E_NONE
        vlan_name = name.cast()
      else
        msg = "Failed to get VLAN name for VLAN ID: '#{vlan_id}', returned: '#{ret}'"
        log_msg("info", msg)
      end
    else
      msg = "Failed to get VLAN name length, returned: '#{ret}'"
      log_msg("info", msg)
    end

    OpEN.delete_uint32_tp(name_len_p)
  else
    msg = "Failed to connect to 'netdev-icos-vlan'"
    log_msg("info", msg)
  end
  open.terminate()
  return vlan_name
end

def create_vlan
  open = OpENUtil.new()
  conn_ret = open.connect("netdev-icos-vlan")
  if conn_ret == OpEN::OPEN_E_NONE
    client = open.client

    ret = OpEN::openapiVlanCreate(client,new_resource.vlan_id)
    if ret == OpEN::OPEN_E_NONE
      count = 0
      ret = OpEN::openapiVlanCreatedCheck(client,new_resource.vlan_id)
      while ret != OpEN::OPEN_E_NONE && count < 10
        count += 1
        sleep 0.5
        ret = OpEN::openapiVlanCreatedCheck(client,new_resource.vlan_id)
      end
      #assign name to created VLAN
      set_vlan_name(new_resource.vlan_id.to_i, new_resource.name)
    else
      msg = "Failed to create VLAN '#{new_resource.vlan_id}', returned '#{ret}'"
      log_msg("info", msg)
    end
  else
    msg = "Failed to connect to 'netdev-icos-vlan'"
    log_msg("info", msg)
  end
  open.terminate()
end

def delete_vlan
  open = OpENUtil.new()
  conn_ret = open.connect("netdev-icos-vlan")
  if conn_ret == OpEN::OPEN_E_NONE
    client = open.client
    ret = OpEN::openapiVlanDelete(client,new_resource.vlan_id)
    if ret != OpEN::OPEN_E_NONE
      msg = "Failed to delete VLAN '#{new_resource.vlan_id}', returned '#{ret}'"
      log_msg("info", msg)
    end
  else
    msg = "Failed to connect to 'netdev-icos-vlan'"
    log_msg("info", msg)
  end
  open.terminate()
end

def edit_vlan
  #assign name to created VLAN
  if new_resource.name != current_resource.name
    converge_by("edit VLAN name of '#{@new_resource.vlan_id}' from '#{current_resource.name}' to '#{new_resource.name}'") do
      set_vlan_name(new_resource.vlan_id.to_i, new_resource.name)
    end
  end
end

