/*********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  dhcp6s_example.c
*
* @purpose   OpEN DHCP6s example.
*
* @component OpEN
*
* @create    07/26/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_dhcp6s.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the DHCP6s Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Get the administrative mode of the IPv6 DHCP-Server for given VRF: %s 1 <vrfName>\n", name);
  printf("Test 2: Set the administrative mode of the IPv6 DHCP-Server for given VRF: %s 2 <vrfName> <mode>\n", name);
  printf("Test 3: Get the DHCPv6 Server DUID: %s 3\n", name);
  printf("Test 4: Get the number of pools for a DHCPv6 Server: %s 4\n", name);
  printf("Test 5: Check if a DHCPv6 pool corresponding to number is valid or not: %s 5 <poolNumber>\n", name);
  printf("Test 6: Get the pool name corresponding to a valid pool number: %s 6 <poolNumber>\n", name);
  printf("Test 7: Create a DHCPv6 address pool: %s 7 <poolName>\n", name);
  printf("Test 8: Delete a DHCPv6 address pool: %s 8 <poolName>\n", name);
  printf("Test 9: Get the first entry in DHCPv6 address pool: %s 9 \n", name);
  printf("Test 10: Get the next entry in DHCPv6 address pool: %s 10 <poolIndex>\n", name);
  printf("Test 11: Add a host within a DHCPv6 Server pool: %s 11 <poolName> <hostName> <prefixAddr> <prefixLen> <validLifetime> <preferLifetime>\n", name);
  printf("Test 12: Delete the host within a DHCPv6 Server pool: %s 12 <poolName> <prefixAddr> <prefixLen>\n", name);
  printf("Test 13: Print the host entries within a DHCPv6 Server pool: %s 13 <poolName>\n", name);
  printf("Test 14: Get the DHCPv6 Server pool type: %s 14 <poolName>\n", name);
  printf("Test 15: Get the DHCPv6 DNS Domain Names from a pool: %s 15 <poolName>\n", name);
  printf("Test 16: Add a DHCPv6 DNS Domain Name for a pool: %s 16 <poolName> <domainName>\n", name);
  printf("Test 17: Remove a DHCPv6 DNS Domain Name from a pool: %s 17 <poolName> <domainName>\n", name);
  printf("Test 18: Add a DHCPv6 host DNS Server for a automatic pool: %s 18 <poolName> <address>\n", name);
  printf("Test 19: Get DHCPv6 host DNS Servers for a automatic pool: %s 19 <poolName>\n", name);
  printf("Test 20: Delete a DHCPv6 host DNS Servers for a automatic pool: %s 20 <poolName> <address>\n", name);
  printf("Test 21: Delete DHCPv6 Server/Relay parameters on a given interface and VRF: %s 21 <ifNum> <vrfName>\n", name);
  printf("Test 22: Set DHCPv6 Relay parameters for an interface: %s 22 <vrfName> <ifNum> <serverAddr> <serverIfNum> <remoteId>\n", name);
  printf("Test 23: Set DHCPv6 Server parameters for an interface: %s 23 <ifNum> <poolName> <serverPref> <rapidCommit> <allowUnicast>\n", name);
  printf("Test 24: Get the DHCPv6 Interface parameters for a given interface: %s 24 <ifNum>\n", name);
  printf("Test 25: Get the next interface with active DHCPv6 configuration: %s 25 <ifNum>\n", name);
  printf("Test 26: Verify the interface has active DHCPv6 configuration: %s 26 <ifNum>\n", name);
  printf("Test 27: Determine if the interface is valid for participation in DHCPv6 components: %s 27 <ifNum>\n", name);
  printf("Test 28: Get the first valid interface for participation in DHCPv6 components: %s 28\n", name);
  printf("Test 29: Get the next valid interface for participation in DHCPv6 components: %s 29 <prevIfNum>\n", name);
  printf("Test 30: Print the active bindings from the DHCPv6 server for a given VRF: %s 30 <vrfName>\n", name);
  printf("Test 31: Clear the binding specified by the IPv6 address for the given VRF: %s 31 <vrfName> <prefixType> <clientAddr>\n", name);
  printf("Test 32: Clear all the binding entries for a given VRF: %s 32 <vrfName>\n", name);
  printf("Test 33: Get the count of number of bindings in the binding database for a given VRF: %s 33 <vrfName>\n", name);
  printf("Test 34: Get the DHCPv6-Server statistics for a given VRF or interface: %s 34 <ifNum> <vrfName>\n", name);
  printf("Test 35: Clear the DHCPv6-Server statistics for a given interface or VRF: %s 35 <ifNum> <vrfName>\n", name);
  printf("Test 36: Print the DHCPv6 interface relay entries for all interfaces: %s 36\n", name);
  printf("Test 37: Get the exact DHCPv6 interface relay entry: %s 37 <ifNum> <relayAddr>\n", name);
  printf("Test 38: Delete DHCPv6 Relay entry/parameters for an interface: %s 38 <vrfName> <ifNum> <serverAddr> <serverIfNum> <remoteId>\n", name);

  return;
}

/******************************************************************************
* @purpose  Get the administrative mode of the IPv6 DHCP-Server for given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    vrfName        @b{(input)}  VRF name
*
* @returns  none
*
* @end
******************************************************************************/
void dhcp6sAdminModeGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  OPEN_CONTROL_t mode = OPEN_DISABLE;

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiDhcp6sAdminModeGet(client_handle, &vrfNameBufd, &mode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe administrative mode of the IPv6 DHCP-Server for the given VRF instance is %u.\n", mode);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the administrative mode of the IPv6 DHCP-Server for the given VRF instance. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Set the administrative mode of the IPv6 DHCP-Server for given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    vrfName        @b{(input)}  VRF name
* @param    mode           @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sAdminModeSet(openapiClientHandle_t *client_handle, char *vrfName, OPEN_CONTROL_t mode)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t openMode = OPEN_DISABLE;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  if (1 < mode)
  {
    printf("\nERROR: Invalid mode value. Expected 0(for Disable) or 1(for Enable).\n");
    return;
  }

  if (1 == mode)
  {
    openMode = OPEN_ENABLE;
  }

  result = openapiDhcp6sAdminModeSet(client_handle, &vrfNameBufd, openMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the administrative mode of the IPv6 DHCP-Server for given VRF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the administrative mode of the IPv6 DHCP-Server for given VRF instance. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the DHCPv6 Server DUID.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sServerDuidGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc duid; /* String representation of DUID */
  char str[OPEN_DHCP6S_MAX_DUID_STR_LEN+1] = "";

  memset(str, 0, sizeof(str));
  duid.pstart = str;
  duid.size = OPEN_DHCP6S_MAX_DUID_STR_LEN+1;
  
  result = openapiDhcp6sServerDuidGet(client_handle, &duid);
  switch(result)
  {
    case OPEN_E_NONE:
         if (duid.size < 2)
         {
           printf("\nERROR: Invalid DUID string length received %u.\n",duid.size);
         }
         else
         {
           printf("\nThe DUID string is %s.\n", str);
         }
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the DHCPv6 Server DUID. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Get the number of pools for DHCPv6 Server.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sPoolNumGet(openapiClientHandle_t *client_handle)
{
  uint32_t count = 0; /* Count of number of Pools */
  open_error_t result = OPEN_E_FAIL;

  result = openapiDhcp6sPoolNumGet(client_handle, &count);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe the number of DHCPv6 Server pools is %u.\n", count);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the number of DHCPv6 Server pools. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Check if a DHCPv6 pool corresponding to number is valid or not.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    poolNumber       @b{(input)}  Pool number
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sPoolValidate(openapiClientHandle_t *client_handle, uint32_t poolNumber)
{
  open_error_t result = OPEN_E_FAIL;

  if ((result = openapiDhcp6sPoolValidate(client_handle, poolNumber)) != OPEN_E_NONE)
  {
    printf("\nGiven Pool-number is not valid. (result = %d)\n", result);
  }
  else
  {
    printf("\nDHCPv6 pool corresponding to given pool-number is valid.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the pool name corresponding to a valid pool number.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    poolNumber     @b{(input)}  Pool number
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sNumberToPoolNameGet(openapiClientHandle_t *client_handle, uint32_t poolNumber)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc poolName; /* Name of address pool */
  char str[OPEN_DHCP6S_POOL_NAME_MAXLEN+1] = "";

  memset(str, 0, sizeof(str));
  poolName.pstart = str;
  poolName.size = OPEN_DHCP6S_POOL_NAME_MAXLEN+1;
  
  result = openapiDhcp6sNumberToPoolNameGet(client_handle, poolNumber, &poolName);
  switch(result)
  {
    case OPEN_E_NONE:
         if (poolName.size < 2)
         {
           printf("\nERROR: Invalid Pool Name string length received %u.\n",poolName.size);
         }
         else
         {
           printf("\nThe Pool name corresponding to a given pool number is %s.\n", str);
         }
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the pool name corresponding to a valid pool number. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Create a DHCPv6 address pool.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    poolName       @b{(input)}  Name of address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sPoolCreate(openapiClientHandle_t *client_handle, char *poolName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc poolNameBufd; /* Name of address pool in buff descriptor format */
  char poolNameStr[OPEN_DHCP6S_POOL_NAME_MAXLEN+1] = "";

  memset(poolNameStr, 0, sizeof(poolNameStr));
  if (OPEN_DHCP6S_POOL_NAME_MAXLEN < strlen(poolName))
  {
    printf("\nERROR: Invalid pool name string.\n");
    return;
  }
  strncpy(poolNameStr, poolName, sizeof(poolNameStr) - 1);
  poolNameBufd.pstart = poolNameStr;
  poolNameBufd.size = strlen(poolNameStr) + 1;

  if ((result = openapiDhcp6sPoolCreate(client_handle, &poolNameBufd)) != OPEN_E_NONE)
  {
    printf("\nERROR: Bad return code trying to create a DHCPv6 address pool. (result = %d)\n", result);
  }
  else
  {
    printf("\nSuccessfully created pool with given pool name.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Delete a DHCPv6 address pool.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    poolName       @b{(input)}  Name of address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sPoolDelete(openapiClientHandle_t *client_handle, char *poolName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc poolNameBufd; /* Name of address pool in buff descriptor format */
  char poolNameStr[OPEN_DHCP6S_POOL_NAME_MAXLEN+1] = "";

  memset(poolNameStr, 0, sizeof(poolNameStr));
  if (OPEN_DHCP6S_POOL_NAME_MAXLEN < strlen(poolName))
  {
    printf("\nERROR: Invalid pool name string.\n");
    return;
  }
  strncpy(poolNameStr, poolName, sizeof(poolNameStr) - 1);
  poolNameBufd.pstart = poolNameStr;
  poolNameBufd.size = strlen(poolNameStr) + 1;

  if ((result = openapiDhcp6sPoolDelete(client_handle, &poolNameBufd)) != OPEN_E_NONE)
  {
    printf("\nERROR: Bad return code trying to delete DHCPv6 address pool with given pool name. (result = %d)\n", result);
  }
  else
  {
    printf("\nSuccessfully deleted DHCPv6 address pool with given pool name.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the first entry in DHCPv6 address pool.
*
* @param    client_handle  @b{(input)}   Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sPoolEntryFirst(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t poolNumber = 0; /* Pool index to fetch next entry */
  open_buffdesc poolName; /* Name of address pool */
  char str[OPEN_DHCP6S_POOL_NAME_MAXLEN+1] = "";

  memset(str, 0, sizeof(str));
  poolName.pstart = str;
  poolName.size = OPEN_DHCP6S_POOL_NAME_MAXLEN+1;
  
  result = openapiDhcp6sPoolEntryFirst(client_handle, &poolName, &poolNumber);
  switch(result)
  {
    case OPEN_E_NONE:
         if (poolName.size < 2)
         {
           printf("\nERROR: Invalid Pool Name string length received %u.\n",poolName.size);
         }
         else
         {
           printf("\nThe First entry in DHCPv6 address pools has Pool name %s and Pool-number %u.\n", str, poolNumber);
         }
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the first entry in DHCPv6 address pool. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Get the next entry in DHCPv6 address pool.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    poolIndex      @b{(inout)}  pool index to fetch next entry
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sPoolEntryNext(openapiClientHandle_t *client_handle, uint32_t poolIndex)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc poolName; /* Name of address pool */
  char str[OPEN_DHCP6S_POOL_NAME_MAXLEN+1] = "";

  memset(str, 0, sizeof(str));
  poolName.pstart = str;
  poolName.size = OPEN_DHCP6S_POOL_NAME_MAXLEN+1;
  
  result = openapiDhcp6sPoolEntryNext(client_handle, &poolName, &poolIndex);
  switch(result)
  {
    case OPEN_E_NONE:
         if (poolName.size < 2)
         {
           printf("\nERROR: Invalid Pool Name string length received %u.\n",poolName.size);
         }
         else
         {
           printf("\nThe next entry in DHCPv6 address pools has Pool name %s and Pool number %u.\n", str, poolIndex);
         }
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the next entry in DHCPv6 address pool. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Add a host within a DHCPv6 Server pool.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    poolName        @b{(input)}  Name of address pool
* @param    hostName        @b{(input)}  Host name
* @param    prefixAddrStr   @b{(input)}  delegated prefix address string
* @param    prefixLen       @b{(input)}  delegated prefix mask length
* @param    validLifetime   @b{(input)}  delegated prefix valid lifetime
* @param    preferLifetime  @b{(input)}  delegated prefix prefer lifetime
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sPoolHostAdd(openapiClientHandle_t *client_handle, char *poolName, char *hostName,
                       char *prefixAddrStr, uint16_t prefixLen, uint32_t validLifetime, uint32_t preferLifetime)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc poolNameBufd; /* Name of address pool in open buff descriptor format */
  char poolNameStr[OPEN_DHCP6S_POOL_NAME_MAXLEN+1] = "";
  open_buffdesc hostNameBufd; /* Host Name in open buff descriptor format */
  char hostNameStr[OPEN_DHCP6S_HOST_NAME_MAXLEN+1] = "";
  open_inet_addr_t openPrefixAddr;
  open_buffdesc ipBuffdesc;
  char str[40];


  memset(str, 0, sizeof(str));
  strncpy(str, prefixAddrStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(&openPrefixAddr, 0, sizeof(openPrefixAddr));
  if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &openPrefixAddr)) != OPEN_E_NONE)
  {
    printf("\nERROR: Bad return code trying to convert IP address. (result = %d).\n", result);
    return;
  }

  if (OPEN_DHCP6S_POOL_NAME_MAXLEN < strlen(poolName))
  {
    printf("\nERROR: Invalid pool name string.\n");
    return;
  }
  memset(poolNameStr, 0, sizeof(poolNameStr));
  strncpy(poolNameStr, poolName, sizeof(poolNameStr) - 1);
  poolNameBufd.pstart = poolNameStr;
  poolNameBufd.size = strlen(poolNameStr) + 1;  

  if (OPEN_DHCP6S_HOST_NAME_MAXLEN < strlen(hostName))
  {
    printf("\nERROR: Invalid host name string.\n");
    return;
  }
  memset(hostNameStr, 0, sizeof(hostNameStr));
  strncpy(hostNameStr, hostName, sizeof(hostNameStr) - 1);
  hostNameBufd.pstart = hostNameStr;
  hostNameBufd.size = strlen(hostNameStr) + 1;  

  result = openapiDhcp6sPoolHostAdd(client_handle, &poolNameBufd, &hostNameBufd,
                                    &openPrefixAddr, prefixLen, validLifetime, preferLifetime);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully added a host within the given DHCPv6 Server pool.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to add a host within a DHCPv6 Server pool. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Delete the host within a DHCPv6 Server pool.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    poolName       @b{(input)}  Name of address pool
* @param    prefixAddr     @b{(input)}  delegated prefix address string
* @param    prefixLen      @b{(input)}  delegated prefix mask length
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sPoolHostRemove(openapiClientHandle_t *client_handle, char *poolName,
                          char *prefixAddrStr, uint16_t prefixLen)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc poolNameBufd; /* Name of address pool in open buff descriptor format */
  char poolNameStr[OPEN_DHCP6S_POOL_NAME_MAXLEN+1] = "";
  open_inet_addr_t openPrefixAddr;
  open_buffdesc ipBuffdesc;
  char str[40];


  memset(str, 0, sizeof(str));
  strncpy(str, prefixAddrStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(&openPrefixAddr, 0, sizeof(openPrefixAddr));
  if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &openPrefixAddr)) != OPEN_E_NONE)
  {
    printf("\nERROR: Bad return code trying to convert IP address. (result = %d).\n", result);
    return;
  }

  if (OPEN_DHCP6S_POOL_NAME_MAXLEN < strlen(poolName))
  {
    printf("\nERROR: Invalid pool name string.\n");
    return;
  }
  memset(poolNameStr, 0, sizeof(poolNameStr));
  strncpy(poolNameStr, poolName, sizeof(poolNameStr) - 1);
  poolNameBufd.pstart = poolNameStr;
  poolNameBufd.size = strlen(poolNameStr) + 1;  

  result = openapiDhcp6sPoolHostRemove(client_handle, &poolNameBufd, &openPrefixAddr, prefixLen);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully removed the specified host within the given DHCPv6 Server pool.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to delete the host within a DHCPv6 Server pool. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Print the host entries within a DHCPv6 Server pool.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    poolName        @b{(input)}  Name of address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sPoolHostGetNext(openapiClientHandle_t *client_handle, char *poolName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc poolNameBufd; /* Name of address pool in open buff descriptor format */
  char poolNameStr[OPEN_DHCP6S_POOL_NAME_MAXLEN+1] = "";
  open_buffdesc hostNameBufd; /* Host Name in open buff descriptor format */
  char hostNameStr[OPEN_DHCP6S_HOST_NAME_MAXLEN+1] = "";
  open_buffdesc duidBufd; /* DUID string in open buff descriptor format */
  char duidString[OPEN_DHCP6S_MAX_DUID_STR_LEN+1] = "";
  open_inet_addr_t openPrefixAddr;
  char ipStr[80];
  uint16_t prefixLen = 0;
  uint32_t count = 0;
  uint32_t validLifetime = 0; /* delegated prefix valid lifetime */
  uint32_t preferLifetime = 0; /* delegated prefix prefer lifetime */
  uint32_t iaid = 0; /* Identity Association ID */


  openPrefixAddr.family = OPEN_AF_INET6;
  memset(&(openPrefixAddr.addr.ipv6), 0, sizeof(open_in6_addr_t));

  if (OPEN_DHCP6S_POOL_NAME_MAXLEN < strlen(poolName))
  {
    printf("\nERROR: Invalid pool name string.\n");
    return;
  }
  memset(poolNameStr, 0, sizeof(poolNameStr));
  strncpy(poolNameStr, poolName, sizeof(poolNameStr) - 1);
  poolNameBufd.pstart = poolNameStr;
  poolNameBufd.size = strlen(poolNameStr) + 1;  

  memset(duidString, 0, sizeof(duidString));
  duidBufd.pstart = duidString;
  duidBufd.size = OPEN_DHCP6S_MAX_DUID_STR_LEN+1;

  memset(hostNameStr, 0, sizeof(hostNameStr));
  hostNameBufd.pstart = hostNameStr;
  hostNameBufd.size = sizeof(hostNameStr);

  result = openapiDhcp6sPoolHostGetNext(client_handle, &poolNameBufd, &duidBufd, &hostNameBufd, &iaid,
                                        &openPrefixAddr, &prefixLen, &validLifetime, &preferLifetime);

  if (result != OPEN_E_NONE)
  {
    printf("\nERROR: Bad return code trying to get next host within a DHCPv6 Server pool. (result = %d)\n", result);
    return;
  }

  do
  {
    count++;
    memset(ipStr, 0, sizeof(ipStr));
    (void) inet_ntop(AF_INET6, &(openPrefixAddr.addr.ipv6.u.addr8), ipStr, sizeof(ipStr));
    printf("\n\nSuccessfully got the host entry %u within the given DHCPv6 Server pool. Below are the host entry details:\n",count);
    printf("\nName of address pool...................... [%s]\n", poolNameStr);
    printf("\nHost DUID................................. [%s]\n", duidString);
    printf("\nHost name................................. [%s]\n", hostNameStr);
    printf("\nIdentity Association ID................... [%u]\n", iaid);
    printf("\nDelegated prefix address.................. [%s]\n", ipStr);
    printf("\nDelegated prefix mask length.............. [%u]\n", prefixLen);
    printf("\nDelegated prefix valid lifetime........... [%u]\n", validLifetime);
    printf("\nDelegated prefix prefer lifetime.......... [%u]\n", preferLifetime);

    result = openapiDhcp6sPoolHostGetNext(client_handle, &poolNameBufd, &duidBufd, &hostNameBufd, &iaid,
                                          &openPrefixAddr, &prefixLen, &validLifetime, &preferLifetime);
  } while (result == OPEN_E_NONE);

  return;
}

/*********************************************************************
* @purpose  Get the DHCPv6 Server pool type.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    poolName       @b{(input)}  Name of address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sPoolTypeGet(openapiClientHandle_t *client_handle, char *poolName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc poolNameBufd; /* Name of address pool in open buff descriptor format */
  char poolNameStr[OPEN_DHCP6S_POOL_NAME_MAXLEN+1] = "";
  uint32_t poolType = 0;

  if (OPEN_DHCP6S_POOL_NAME_MAXLEN < strlen(poolName))
  {
    printf("\nERROR: Invalid pool name string.\n");
    return;
  }
  memset(poolNameStr, 0, sizeof(poolNameStr));
  strncpy(poolNameStr, poolName, sizeof(poolNameStr) - 1);
  poolNameBufd.pstart = poolNameStr;
  poolNameBufd.size = strlen(poolNameStr) + 1;  

  result = openapiDhcp6sPoolTypeGet(client_handle, &poolNameBufd, &poolType);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe given DHCPv6 Server pool's pool-type is %u.\n",poolType);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the DHCPv6 Server pool type. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Get the DHCPv6 DNS Domain Names from a pool.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    poolName        @b{(input)}  Name of address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sPoolDnsDomainsGet(openapiClientHandle_t *client_handle, char *poolName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc poolNameBufd; /* Name of address pool in open buff descriptor format */
  char poolNameStr[OPEN_DHCP6S_POOL_NAME_MAXLEN+1] = "";
  openDnsDomainTable_t dnsDomainNames; /* Array of DNS domain names */
  uint32_t i = 0;

  if (OPEN_DHCP6S_POOL_NAME_MAXLEN < strlen(poolName))
  {
    printf("\nERROR: Invalid pool name string.\n");
    return;
  }
  memset(poolNameStr, 0, sizeof(poolNameStr));
  strncpy(poolNameStr, poolName, sizeof(poolNameStr) - 1);
  poolNameBufd.pstart = poolNameStr;
  poolNameBufd.size = strlen(poolNameStr) + 1; 

  memset(&dnsDomainNames,0,sizeof(dnsDomainNames));
  result = openapiDhcp6sPoolDnsDomainsGet(client_handle, &poolNameBufd, &dnsDomainNames);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully got the DNS Domain Names from a pool. Below is the list of the Domain Names:\n");
         for (i=0; i < OPEN_DHCP6S_DNS_DOMAIN_MAX; i++)
         {
           if (strlen((const char *)(dnsDomainNames.dns_domains[i])) != 0)
           {
             printf("\n %s",dnsDomainNames.dns_domains[i]);
           }
         }
         printf("\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the DHCPv6 DNS Domain Names from a pool. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Add a DHCPv6 DNS Domain Name for a pool.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    poolName       @b{(input)}  Name of address pool
* @param    domainName     @b{(input)}  DNS domain name
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sPoolDnsDomainAdd(openapiClientHandle_t *client_handle, char *poolName, char *domainName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc poolNameBufd;   /* Name of address pool in open buff descriptor format */
  char poolNameStr[OPEN_DHCP6S_POOL_NAME_MAXLEN+1] = "";
  open_buffdesc domainNameBufd; /* DNS domain name in open buff descriptor format */
  char domainNameStr[OPEN_DHCP6S_DOMAIN_NAME_MAXLEN+1] = "";


  if (OPEN_DHCP6S_POOL_NAME_MAXLEN < strlen(poolName))
  {
    printf("\nERROR: Invalid pool name string.\n");
    return;
  }
  memset(poolNameStr, 0, sizeof(poolNameStr));
  strncpy(poolNameStr, poolName, sizeof(poolNameStr) - 1);
  poolNameBufd.pstart = poolNameStr;
  poolNameBufd.size = strlen(poolNameStr) + 1;  

  if (OPEN_DHCP6S_DOMAIN_NAME_MAXLEN < strlen(domainName))
  {
    printf("\nERROR: Invalid domain name string.\n");
    return;
  }
  memset(domainNameStr, 0, sizeof(domainNameStr));
  strncpy(domainNameStr, domainName, sizeof(domainNameStr) - 1);
  domainNameBufd.pstart = domainNameStr;
  domainNameBufd.size = strlen(domainNameStr) + 1;

  result = openapiDhcp6sPoolDnsDomainAdd(client_handle, &poolNameBufd, &domainNameBufd);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully add the specified DHCPv6 DNS Domain Name to the given DHCPv6 pool.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to add a DHCPv6 DNS Domain Name to a DHCPv6 pool. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Remove a DHCPv6 DNS Domain Name from a pool.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    poolName       @b{(input)}  Name of address pool
* @param    domainName     @b{(input)}  DNS domain name
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sPoolDnsDomainRemove(openapiClientHandle_t *client_handle, char *poolName, char *domainName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc poolNameBufd; /* Name of address pool in open buff descriptor format */
  char poolNameStr[OPEN_DHCP6S_POOL_NAME_MAXLEN+1] = "";
  open_buffdesc domainNameBufd; /* DNS domain name in open buff descriptor format */
  char domainNameStr[OPEN_DHCP6S_DOMAIN_NAME_MAXLEN+1] = "";


  if (OPEN_DHCP6S_POOL_NAME_MAXLEN < strlen(poolName))
  {
    printf("\nERROR: Invalid pool name string.\n");
    return;
  }
  memset(poolNameStr, 0, sizeof(poolNameStr));
  strncpy(poolNameStr, poolName, sizeof(poolNameStr) - 1);
  poolNameBufd.pstart = poolNameStr;
  poolNameBufd.size = strlen(poolNameStr) + 1;  


  if (OPEN_DHCP6S_DOMAIN_NAME_MAXLEN < strlen(domainName))
  {
    printf("\nERROR: Invalid domain name string.\n");
    return;
  }
  memset(domainNameStr, 0, sizeof(domainNameStr));
  strncpy(domainNameStr, domainName, sizeof(domainNameStr) - 1);
  domainNameBufd.pstart = domainNameStr;
  domainNameBufd.size = strlen(domainNameStr) + 1; 

  result = openapiDhcp6sPoolDnsDomainRemove(client_handle, &poolNameBufd, &domainNameBufd);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully removed the specified DHCPv6 DNS Domain Name from the given DHCPv6 pool.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to remove a DHCPv6 DNS Domain Name from a DHCPv6 pool. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Add a DHCPv6 host DNS Server for a automatic pool.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    poolName      @b{(input)}  Name of address pool
* @param    prefixAddrStr @b{(input)}  An IPv6 address of a DNS Server
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sPoolDnsServerAdd(openapiClientHandle_t *client_handle, char *poolName, char *prefixAddrStr)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc poolNameBufd; /* Name of address pool in open buff descriptor format */
  char poolNameStr[OPEN_DHCP6S_POOL_NAME_MAXLEN+1] = "";
  open_inet_addr_t openPrefixAddr;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, prefixAddrStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(&openPrefixAddr, 0, sizeof(openPrefixAddr));
  if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &openPrefixAddr)) != OPEN_E_NONE)
  {
    printf("\nERROR: Bad return code trying to convert IP address. (result = %d).\n", result);
    return;
  }

  if (OPEN_DHCP6S_POOL_NAME_MAXLEN < strlen(poolName))
  {
    printf("\nERROR: Invalid pool name string.\n");
    return;
  }
  memset(poolNameStr, 0, sizeof(poolNameStr));
  strncpy(poolNameStr, poolName, sizeof(poolNameStr) - 1);
  poolNameBufd.pstart = poolNameStr;
  poolNameBufd.size = strlen(poolNameStr) + 1;  

  result = openapiDhcp6sPoolDnsServerAdd(client_handle, &poolNameBufd, &openPrefixAddr);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully added the specified DHCPv6 host DNS server from the given automatic DHCPv6 pool.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to add a DHCPv6 host DNS Server to an automatic pool. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Get DHCPv6 host DNS Servers for a automatic pool.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    poolName       @b{(input)}  Name of address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sPoolDnsServersGet(openapiClientHandle_t *client_handle, char *poolName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc poolNameBufd; /* Name of address pool in open buff descriptor format */
  char poolNameStr[OPEN_DHCP6S_POOL_NAME_MAXLEN+1] = "";
  openDnsServerTable_t dnsServers; /* Array of ipAddress of DNS Servers */
  open_in6_addr_t zeroAddr;
  uint32_t i = 0;
  char ipStr[80];

  if (OPEN_DHCP6S_POOL_NAME_MAXLEN < strlen(poolName))
  {
    printf("\nERROR: Invalid pool name string.\n");
    return;
  }
  memset(poolNameStr, 0, sizeof(poolNameStr));
  strncpy(poolNameStr, poolName, sizeof(poolNameStr) - 1);
  poolNameBufd.pstart = poolNameStr;
  poolNameBufd.size = strlen(poolNameStr) + 1; 

  memset(&dnsServers, 0, sizeof(dnsServers));
  result = openapiDhcp6sPoolDnsServersGet(client_handle, &poolNameBufd, &dnsServers);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully got the DNS Servers from given pool. Below is the list of the DNS Servers:\n");
         memset(&zeroAddr,0,sizeof(zeroAddr));
         for (i=0; i < OPEN_DHCP6S_DNS_SERVER_MAX; i++)
         {
           if (memcmp(&(dnsServers.dns_servers[i]),&zeroAddr,sizeof(zeroAddr)) != 0)
           {
             memset(ipStr, 0, sizeof(ipStr));
             (void) inet_ntop(AF_INET6, &((dnsServers.dns_servers[i]).u.addr8), ipStr, sizeof(ipStr));
             printf("\n %s",ipStr);
           }
         }
         printf("\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the DHCPv6 DNS Servers from a pool. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Delete a DHCPv6 host DNS Servers for a automatic pool.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    poolName       @b{(input)}  Name of address pool
* @param    prefixAddrStr  @b{(input)}  An IPv6 address of a DNS Server
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sPoolDnsServerRemove(openapiClientHandle_t *client_handle, char *poolName, char *prefixAddrStr)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc poolNameBufd; /* Name of address pool in open buff descriptor format */
  char poolNameStr[OPEN_DHCP6S_POOL_NAME_MAXLEN+1] = "";
  open_inet_addr_t openPrefixAddr;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, prefixAddrStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(&openPrefixAddr, 0, sizeof(openPrefixAddr));
  if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &openPrefixAddr)) != OPEN_E_NONE)
  {
    printf("\nERROR: Bad return code trying to convert IP address. (result = %d).\n", result);
    return;
  }

  if (OPEN_DHCP6S_POOL_NAME_MAXLEN < strlen(poolName))
  {
    printf("\nERROR: Invalid pool name string.\n");
    return;
  }
  memset(poolNameStr, 0, sizeof(poolNameStr));
  strncpy(poolNameStr, poolName, sizeof(poolNameStr) - 1);
  poolNameBufd.pstart = poolNameStr;
  poolNameBufd.size = strlen(poolNameStr) + 1;  

  result = openapiDhcp6sPoolDnsServerRemove(client_handle, &poolNameBufd, &openPrefixAddr);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully removed the specified DNS server from the given automatic DHCPv6 pool.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to delete a DHCPv6 host DNS Server from an automatic pool. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Delete DHCPv6 Server/Relay parameters on a given interface and VRF.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number of the interface
* @param    vrfName       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sIntfModeDelete(openapiClientHandle_t *client_handle, uint32_t ifNum, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiDhcp6sIntfModeDelete(client_handle, ifNum, &vrfNameBufd);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully deleted DHCPv6 Server/Relay parameters on a given interface and VRF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to delete DHCPv6 Server/Relay parameters on a given interface and VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set DHCPv6 Relay parameters for an interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vrfName       @b{(input)}  VRF name
* @param    ifNum         @b{(input)}  Internal Interface Number of the interface
* @param    serverAddr    @b{(input)}  Relay Server address
* @param    serverIfNum   @b{(input)}  Internal Interface Number of relay server interface
* @param    remoteId      @b{(input)}  remote-id relay option
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sIntfModeRelaySet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t ifNum,
                            char *serverAddrStr, uint32_t serverIfNum, char *remoteId)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  open_buffdesc remoteIdBufd;
  char remoteIdStr[OPEN_DHCP6S_RELAYOPT_REMOTEID_MAXLEN + 1] = "";
  open_inet_addr_t openPrefixAddr;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, serverAddrStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(&openPrefixAddr, 0, sizeof(openPrefixAddr));
  if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &openPrefixAddr)) != OPEN_E_NONE)
  {
    printf("\nERROR: Bad return code trying to convert IP address. (result = %d).\n", result);
    return;
  }

  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  if (OPEN_DHCP6S_RELAYOPT_REMOTEID_MAXLEN < strlen(remoteId))
  {
    printf("\nERROR: Invalid remoteId string.\n");
    return;
  }
  memset(remoteIdStr, 0, sizeof(remoteIdStr));
  strncpy(remoteIdStr, remoteId, sizeof(remoteIdStr) - 1);
  remoteIdBufd.pstart = remoteIdStr;
  remoteIdBufd.size = strlen(remoteIdStr) + 1;

  result = openapiDhcp6sIntfModeRelaySet(client_handle, &vrfNameBufd, ifNum, &openPrefixAddr, serverIfNum, &remoteIdBufd);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the given DHCPv6 Relay parameters on the given interface and VRF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set DHCPv6 Relay parameters for an interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set DHCPv6 Server parameters for an interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal Interface Number of the interface
* @param    poolName       @b{(input)}  DHCPv6 pool name
* @param    serverPref     @b{(input)}  Server preference
* @param    rapidCommit    @b{(input)}  rapid commit option
* @param    allowUnicast   @b{(input)}  allow unicast option
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sIntfModeServerSet(openapiClientHandle_t *client_handle, uint32_t ifNum, char *poolName,
                             uint32_t serverPref, uint32_t rapidCommit, uint32_t allowUnicast)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc poolNameBufd; /* Name of address pool in open buff descriptor format */
  char poolNameStr[OPEN_DHCP6S_POOL_NAME_MAXLEN+1] = "";
  OPEN_BOOL_t rapidCommitFlag  = OPEN_FALSE;
  OPEN_BOOL_t allowUnicastFlag = OPEN_FALSE;

  if ((rapidCommit > 1) || (allowUnicast > 1))
  {
    printf("\nERROR: Invalid argument passed. Valid values for rapidCommit and allowUnicast are [0,1]. \n");
    return;
  }
  if (rapidCommit)
  {
    rapidCommitFlag = OPEN_TRUE;
  }
  
  if (allowUnicast)
  {
    allowUnicastFlag = OPEN_TRUE;
  }

  memset(poolNameStr, 0, sizeof(poolNameStr));
  strncpy(poolNameStr, poolName, sizeof(poolNameStr) - 1);
  poolNameBufd.pstart = poolNameStr;
  poolNameBufd.size = strlen(poolNameStr) + 1;  

  result = openapiDhcp6sIntfModeServerSet(client_handle, ifNum, &poolNameBufd, serverPref, rapidCommitFlag, allowUnicastFlag);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the given DHCPv6 Server parameters on the given interface.\n");
         break; 

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set DHCPv6 Server parameters for an interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the DHCPv6 Interface parameters for a given interface.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    ifNum             @b{(input)}  Internal Interface Number of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sIntfModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_DHCP6_MODE_t mode = OPEN_DHCP6_MODE_NONE;
  open_buffdesc poolNameBufd;
  char poolNameStr[OPEN_DHCP6S_POOL_NAME_MAXLEN+1] = "";
  uint32_t serverPref = 0;
  open_inet_addr_t relayAddr;
  char ipStr[40] ="";
  uint32_t relayIfNum = 0;
  open_buffdesc remoteIdBufd;
  char remoteIdStr[OPEN_DHCP6S_RELAYOPT_REMOTEID_MAXLEN + 1] = "";
  OPEN_BOOL_t rapidCommit  = OPEN_FALSE;
  OPEN_BOOL_t allowUnicast = OPEN_FALSE;
  open_buffdesc localVrfNameBufd;
  char localVrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  open_buffdesc relayDestVrfNameBufd;
  char destVrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint32_t pdClientIntIfNum = 0;


  memset(poolNameStr, 0, sizeof(poolNameStr));
  poolNameBufd.pstart = poolNameStr;
  poolNameBufd.size = sizeof(poolNameStr);

  memset(remoteIdStr, 0, sizeof(remoteIdStr));
  remoteIdBufd.pstart = remoteIdStr;
  remoteIdBufd.size = sizeof(remoteIdStr);

  memset(localVrfNameStr, 0, sizeof(localVrfNameStr));
  localVrfNameBufd.pstart = localVrfNameStr;
  localVrfNameBufd.size = sizeof(localVrfNameStr);

  memset(destVrfNameStr, 0, sizeof(destVrfNameStr));
  relayDestVrfNameBufd.pstart = destVrfNameStr;
  relayDestVrfNameBufd.size = sizeof(destVrfNameStr);

  result = openapiDhcp6sIntfModeGet(client_handle, ifNum, &mode, &poolNameBufd, &serverPref, &relayAddr, &relayIfNum,
                                    &remoteIdBufd, &rapidCommit, &allowUnicast, &localVrfNameBufd, &relayDestVrfNameBufd, &pdClientIntIfNum);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully got the DHCPv6 Interface parameters for a given interface. Below are the details:\n");
         printf("\nInternal Interface Number of the given interface: %u", ifNum);
         printf("\nDHCP6 mode on interface %u", mode);
         printf("\nDHCPv6 pool name [%s]", poolNameStr);
         printf("\nServer preference %u", serverPref);
         memset(ipStr, 0, sizeof(ipStr));
         (void) inet_ntop(AF_INET6, &(relayAddr.addr.ipv6.u.addr8), ipStr, sizeof(ipStr));
         printf("\nIPv6 address of relay server [%s]", (char *)ipStr);
         printf("\nInternal Interface Number of relay server interface %u", relayIfNum);
         printf("\nremote-id Relay option %s", remoteIdStr);
         printf("\nrapid commit option %u", rapidCommit);
         printf("\nallow unicast option %u", allowUnicast);
         printf("\nVRF name of the local interface [%s]", localVrfNameStr);
         printf("\nVRF name of the relay destination [%s]", destVrfNameStr);
         printf("\nInternal Interface Number of the PD client interface: %u\n", pdClientIntIfNum);
         break; 

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the DHCPv6 Interface parameters for a given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the next interface with active DHCPv6 configuration.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    ifNum           @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sIntfGetNext(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiDhcp6sIntfGetNext(client_handle, &ifNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe next interface with active DHCPv6 configuration for given interface is %u.\n", ifNum);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the next interface with active DHCPv6 configuration. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Verify the interface has active DHCPv6 configuration.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    ifNum           @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sIntfValidate(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiDhcp6sIntfValidate(client_handle, ifNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe given interface has active DHCPv6 configuration.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nThe given interface doesn't have active DHCPv6 configuration. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Determine if the interface is valid for participation in DHCPv6 components.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    ifNum             @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6MapIntfIsValid(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_BOOL_t isValidFlag = OPEN_FALSE;

  result = openapiDhcp6MapIntfIsValid(client_handle, ifNum, &isValidFlag);
  switch(result)
  {
    case OPEN_E_NONE:
         if (OPEN_TRUE == isValidFlag)
         {
           printf("\nThe given interface is valid for participation in DHCPv6 components.\n");
         }
         else
         {
           printf("\nThe given interface is not valid for participation in DHCPv6 components.\n");
         }
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to determine if the interface is valid for participation in DHCPv6 components. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the first valid interface for participation in DHCPv6 components.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6MapIntfValidFirstGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t ifNum = 0;

  result = openapiDhcp6MapIntfValidFirstGet(client_handle, &ifNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe first valid interface for participation in DHCPv6 components is %u.\n", ifNum);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the first valid interface for participation in DHCPv6 components. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the next valid interface for participation in DHCPv6 components.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    prevIfNum       @b{(input)}  Previous Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6MapIntfValidNextGet(openapiClientHandle_t *client_handle, uint32_t prevIfNum)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t ifNum = 0; /* Next Internal Interface Number */

  result = openapiDhcp6MapIntfValidNextGet(client_handle, prevIfNum, &ifNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe next valid interface for participation in DHCPv6 components for the given interface is %u.\n", ifNum);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the next valid interface for participation in DHCPv6 components. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Print the active bindings from the dhcpv6 server for a given VRF.
*
* @param    client_handle   @b{(input)}  client handle from registration api
* @param    vrfName         @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sBindingGetNext(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  open_buffdesc duidBufd; /* String representation of DUID */
  char duidStr[OPEN_DHCP6S_MAX_DUID_STR_LEN+1] = "";
  uint32_t iaid = 0;
  uint32_t count = 0;
  open_inet_addr_t prefixAddr;
  uint16_t prefixLen = 0;
  uint32_t prefixType = 0;
  open_inet_addr_t clientAddr;
  char ipStr[40] = "";
  uint32_t clientIntfNum = 0;
  uint32_t expiration = 0;
  uint32_t validLifetime = 0;
  uint32_t preferLifetime = 0;


  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  memset(duidStr, 0, sizeof(duidStr));
  duidBufd.pstart = duidStr;
  duidBufd.size = OPEN_DHCP6S_MAX_DUID_STR_LEN+1;

  prefixAddr.family = OPEN_AF_INET6;
  memset(&(prefixAddr.addr.ipv6), 0, sizeof(open_in6_addr_t));
  clientAddr.family = OPEN_AF_INET6;
  memset(&(clientAddr.addr.ipv6), 0, sizeof(open_in6_addr_t));

  result = openapiDhcp6sBindingGetNext(client_handle, &vrfNameBufd, &duidBufd, &iaid, &prefixAddr, &prefixLen, &prefixType,
                                       &clientAddr, &clientIntfNum, &expiration, &validLifetime, &preferLifetime);
  if (result != OPEN_E_NONE)
  {
    printf("\nERROR: Bad return code trying to get next active binding from the DHCPv6 server for a given VRF. (result = %d)\n", result);
    return;
  }

  do 
  {
    count++;
    memset(ipStr, 0, sizeof(ipStr));
    (void) inet_ntop(AF_INET6, &(prefixAddr.addr.ipv6.u.addr8), ipStr, sizeof(ipStr));
    printf("\n\nSuccessfully got the active binding from the DHCPv6 server for a given VRF. Below are the details of Entry Count [%u]:\n",count);
    printf("\nVRF name: [%s]", (char *)(vrfNameBufd.pstart));
    printf("\nString representing client DUID: %s", ((char *)duidBufd.pstart));
    printf("\nIdentity Association ID: %u", iaid);
    printf("\nBinding prefix IPv6 address: %s", ipStr);
    printf("\nBinding prefix mask length: %u", prefixLen);
    printf("\nBinding prefix type: %u", prefixType);
    memset(ipStr, 0, sizeof(ipStr));
    (void) inet_ntop(AF_INET6, &(clientAddr.addr.ipv6.u.addr8), ipStr, sizeof(ipStr));
    printf("\nIPv6 address of the Client: %s", ipStr);
    printf("\nInternal Interface Number of Client interface: %u", clientIntfNum);
    printf("\nExpiration (in seconds): %u", expiration);
    printf("\nBinding prefix valid lifetime: %u", validLifetime);
    printf("\nBinding prefix prefer lifetime: %u\n", preferLifetime);

    result = openapiDhcp6sBindingGetNext(client_handle, &vrfNameBufd, &duidBufd, &iaid, &prefixAddr, &prefixLen, &prefixType,
                                         &clientAddr, &clientIntfNum, &expiration, &validLifetime, &preferLifetime);
  } while (result == OPEN_E_NONE);

  return;
}

/*********************************************************************
* @purpose  Clear the binding specified by the IPv6 address for the given VRF.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    vrfName          @b{(input)}  VRF name
* @param    prefixType       @b{(input)}  Binding prefix type
* @param    clientAddrStr    @b{(input)}  Client IPv6 address
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sBindingClear(openapiClientHandle_t *client_handle, char *vrfName, uint32_t prefixType, char *clientAddrStr)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  open_inet_addr_t openClientAddr;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, clientAddrStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(&openClientAddr, 0, sizeof(openClientAddr));
  if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &openClientAddr)) != OPEN_E_NONE)
  {
    printf("\nERROR: Bad return code trying to convert IP address. (result = %d).\n", result);
    return;
  }

  if ((prefixType != OPEN_DHCP6_PREFIX_IAPD) &&  
      (prefixType != OPEN_DHCP6_PREFIX_IANA) &&
      (prefixType != OPEN_DHCP6_PREFIX_IATA))
  {
    printf("\nERROR: Invalid prefix-type value. Valid range (0-2).\n");
    return;
  }

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiDhcp6sBindingClear(client_handle, &vrfNameBufd, prefixType, &openClientAddr);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully cleared the binding entry specified by the IPv6 address for a given VRF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to clear the binding specified by the IPv6 address for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Clear all the binding entries for a given VRF.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vrfName       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sBindingClearAll(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiDhcp6sBindingClearAll(client_handle, &vrfNameBufd);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully cleared all the binding entries for a given VRF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to clear all the binding entries for a given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the count of number of bindings in the binding database for a given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    vrfName        @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sBindingCountGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint32_t count = 0; /* Count of the number of bindings */

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiDhcp6sBindingCountGet(client_handle, &vrfNameBufd, &count);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nCount of number of bindings in the binding database for a given VRF is %u.\n", count);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the count of number of bindings in the binding database for a given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the DHCPv6-Server statistics for a given VRF or interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal Interface Number
* @param    vrfName        @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sStatisticsGet(openapiClientHandle_t *client_handle, uint32_t ifNum, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  openDhcp6sStatData_t dhcp6Stats;

  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  memset(&dhcp6Stats, 0, sizeof(openDhcp6sStatData_t));
  result = openapiDhcp6sStatisticsGet(client_handle, ifNum, &vrfNameBufd, &dhcp6Stats);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully got the DHCPv6-Server statistics for a given interface or VRF. Stats listed below: \n");
         printf("\nSolicit Received is         %u  ", dhcp6Stats.numOfDhcpsSolicitReceived);
         printf("\nRequest Received is         %u  ", dhcp6Stats.numOfDhcpsRequestReceived);
         printf("\nConfirm Received is         %u  ", dhcp6Stats.numOfDhcpsConfirmReceived);
         printf("\nRenew Received is           %u  ", dhcp6Stats.numOfDhcpsRenewReceived);
         printf("\nRebind Received is          %u  ", dhcp6Stats.numOfDhcpsRebindReceived);
         printf("\nRelease Received is         %u  ", dhcp6Stats.numOfDhcpsReleaseReceived);
         printf("\nDecline Received is         %u  ", dhcp6Stats.numOfDhcpsDeclineReceived);
         printf("\nInform Received is          %u  ", dhcp6Stats.numOfDhcpsInformReceived);
         printf("\nRelay Reply Received is     %u  ", dhcp6Stats.numOfDhcpsRelayReplyReceived);
         printf("\nRelay Forwarded Received is %u  ", dhcp6Stats.numOfDhcpsRelayForwReceived);
         printf("\nTotal Packet Received is    %u  ", dhcp6Stats.totalPacketsReceived);
         printf("\nAdvertise Sent is           %u  ", dhcp6Stats.numOfDhcpsAdvertiseSent);
         printf("\nReply Sent is               %u  ", dhcp6Stats.numOfDhcpsReplySent);
         printf("\nReconfigure Sent is         %u  ", dhcp6Stats.numOfDhcpsReconfigureSent);
         printf("\nRelay Reply Sent is         %u  ", dhcp6Stats.numOfDhcpsRelayReplySent);
         printf("\nRelay Forwarded Sent is     %u  ", dhcp6Stats.numOfDhcpsRelayForwSent);
         printf("\nMalformed Received is       %u  ", dhcp6Stats.numOfMalformedMessages);
         printf("\nDiscarded Packets is        %u  ", dhcp6Stats.numOfDhcpPacketsDiscarded);
         printf("\nTotal Packets Sent is       %u\n", dhcp6Stats.totalPacketsSent);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the DHCPv6-Server statistics for a given VRF or interface. (result = %d)\n", result);
         break;
  }
  return;  
}

/*********************************************************************
* @purpose  Clear the DHCPv6-Server statistics for a given interface or VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal Interface Number
* @param    vrfName        @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sStatisticsClear(openapiClientHandle_t *client_handle, uint32_t ifNum, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiDhcp6sStatisticsClear(client_handle, ifNum, &vrfNameBufd);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully cleared the DHCPv6-Server statistics for a given interface or VRF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to clear the DHCPv6-Server statistics for a given interface or VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Print the DHCPv6 interface relay entries for all interfaces.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpv6RelayEntryNextGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t ifNum = 0;
  open_inet_addr_t relayAddr;
  uint32_t relayIfNum = 0;
  char destVrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  open_buffdesc relayDestVrfNameBufd;
  uint32_t count = 0;
  uint32_t nextIfNum = 0;
  open_inet_addr_t nextRelayAddr;
  char ipStr[40] = "";
  uint32_t nextRelayIfNum = 0;
  open_buffdesc nextRelayDestVrfNameBufd;
  char nextDestVrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";


  relayAddr.family = OPEN_AF_INET6;
  memset(&(relayAddr.addr.ipv6), 0, sizeof(open_in6_addr_t));
  
  memset(destVrfNameStr, 0, sizeof(destVrfNameStr));
  relayDestVrfNameBufd.pstart = destVrfNameStr;
  relayDestVrfNameBufd.size = strlen(destVrfNameStr) + 1;

  nextRelayAddr.family = OPEN_AF_INET6;
  memset(&(nextRelayAddr.addr.ipv6), 0, sizeof(open_in6_addr_t));

  memset(nextDestVrfNameStr, 0, sizeof(nextDestVrfNameStr));
  nextRelayDestVrfNameBufd.pstart = nextDestVrfNameStr;
  nextRelayDestVrfNameBufd.size = sizeof(nextDestVrfNameStr);

  result = openapiDhcpv6RelayEntryNextGet(client_handle, ifNum, &relayAddr, &relayIfNum, &relayDestVrfNameBufd,
                                          &nextIfNum, &nextRelayAddr, &nextRelayIfNum, &nextRelayDestVrfNameBufd);

  if (result != OPEN_E_NONE)
  {
    printf("\nERROR: Bad return code trying to get the next DHCPv6 interface relay parameters for an interface. (result = %d)\n", result);
    return;
  }

  do
  {
    count++;
    memset(ipStr, 0, sizeof(ipStr));
    (void) inet_ntop(AF_INET6, &(nextRelayAddr.addr.ipv6.u.addr8), ipStr, sizeof(ipStr));
    printf("\n\nSuccessfully got the DHCPv6 interface relay entry count [%u] . Below are the details:\n",count);
    printf("\nInternal Interface Number of ingress interface: %u", nextIfNum);
    printf("\nIPv6 address of relay server %s", ipStr);
    printf("\nInternal Interface Number of relay server interface %u", nextRelayIfNum);
    printf("\nVRF name of the relay destination is [%s]\n", nextDestVrfNameStr);

    ifNum = nextIfNum;
    memcpy(&relayAddr, &nextRelayAddr, sizeof(relayAddr));
    relayIfNum = nextRelayIfNum;
    memcpy(&relayDestVrfNameBufd, &nextRelayDestVrfNameBufd, sizeof(relayDestVrfNameBufd));
    
    result = openapiDhcpv6RelayEntryNextGet(client_handle, ifNum, &relayAddr, &relayIfNum, &relayDestVrfNameBufd,
                                            &nextIfNum, &nextRelayAddr, &nextRelayIfNum, &nextRelayDestVrfNameBufd);
  } while (result == OPEN_E_NONE);

  return;
}

/*********************************************************************
* @purpose  Get the exact DHCPv6 interface relay entry.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    ifNum           @b{(input)}  Internal Interface Number of ingress interface
* @param    relayAddr       @b{(input)}  IPv6 address of server
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpv6RelayEntryGet(openapiClientHandle_t *client_handle, uint32_t ifNum, char *relayAddrStr)
{
  open_error_t result = OPEN_E_FAIL;
  open_inet_addr_t openRelayAddr;
  open_buffdesc ipBuffdesc;
  char str[40];
  uint32_t relayIfNum = 0;

  memset(str, 0, sizeof(str));
  strncpy(str, relayAddrStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(&openRelayAddr, 0, sizeof(openRelayAddr));
  if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &openRelayAddr)) != OPEN_E_NONE)
  {
    printf("\nERROR: Bad return code trying to convert IP address. (result = %d).\n", result);
    return;
  }

  result = openapiDhcpv6RelayEntryGet(client_handle, ifNum, &(openRelayAddr.addr.ipv6), &relayIfNum); 
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully got the exact DHCPv6 interface relay entry (Ingress interface =[%u], Server address =[%s], Relay interface =[%u]).\n",
                ifNum, str, relayIfNum);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the exact DHCPv6 interface relay entry. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Delete DHCPv6 Relay entry/parameters for an interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vrfName       @b{(input)}  VRF name
* @param    ifNum         @b{(input)}  Internal Interface Number of the interface
* @param    serverAddr    @b{(input)}  Relay Server address
* @param    serverIfNum   @b{(input)}  Internal Interface Number of relay server interface
* @param    remoteId      @b{(input)}  remote-id relay option
*
* @returns  none
*
* @end
*********************************************************************/
void dhcp6sIntfRelayEntryDelete(openapiClientHandle_t *client_handle, char *vrfName, uint32_t ifNum,
                            char *serverAddrStr, uint32_t serverIfNum, char *remoteId)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  open_buffdesc remoteIdBufd;
  char remoteIdStr[OPEN_DHCP6S_RELAYOPT_REMOTEID_MAXLEN + 1] = "";
  open_inet_addr_t openPrefixAddr;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, serverAddrStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(&openPrefixAddr, 0, sizeof(openPrefixAddr));
  if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &openPrefixAddr)) != OPEN_E_NONE)
  {
    printf("\nERROR: Bad return code trying to convert IP address. (result = %d).\n", result);
    return;
  }

  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  if (OPEN_DHCP6S_RELAYOPT_REMOTEID_MAXLEN < strlen(remoteId))
  {
    printf("\nERROR: Invalid remoteId string.\n");
    return;
  }
  memset(remoteIdStr, 0, sizeof(remoteIdStr));
  strncpy(remoteIdStr, remoteId, sizeof(remoteIdStr) - 1);
  remoteIdBufd.pstart = remoteIdStr;
  remoteIdBufd.size = strlen(remoteIdStr) + 1;

  result = openapiDhcp6sIntfRelayEntryDelete(client_handle, &vrfNameBufd, ifNum, &openPrefixAddr, serverIfNum, &remoteIdBufd);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully deleted the given DHCPv6 Relay entry/parameters on the given interface and VRF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to delete the given DHCPv6 Relay entry/parameters for an interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for DHCP6s
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum = 0;
  uint32_t ifNum = 0, prevIfNum = 0, serverIfNum = 0;
  uint32_t mode = 0, rapidCommit = 0, allowUnicast = 0;
  uint32_t prefixType = 0, prefixLen = 0, value = 0;
  uint32_t pref = 0;
  uint32_t validLifetime = 0, preferLifetime = 0;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("DHCP6S example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting DHCP6s API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1: /* Test 1: Get the administrative mode of the IPv6 DHCP-Server for given VRF: %s 1 <vrfName> */
      if (argc == 3)
      {
        dhcp6sAdminModeGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 2: /* Test 2: Set the administrative mode of the IPv6 DHCP-Server for given VRF: %s 2 <vrfName> <mode> */
      if (argc == 4)
      {
        mode = atoi(argv[3]);
        dhcp6sAdminModeSet(&client_handle, argv[2], mode);
        show_help = 0;
      }
      break;
    case 3: /* Test 3: Get the DHCPv6 Server DUID: %s 3 */
      if (argc == 2)
      {
        dhcp6sServerDuidGet(&client_handle);
        show_help = 0;
      }
      break;
    case 4: /* Test 4: Get the number of pools for a DHCPv6 Server: %s 4 */
      if (argc == 2)
      {
        dhcp6sPoolNumGet(&client_handle);
        show_help = 0;
      }
      break;
    case 5: /* Test 5: Check if a DHCPv6 pool corresponding to number is valid or not: %s 5 <poolNumber> */
      if (argc == 3)
      {
        value = atoi(argv[2]);
        dhcp6sPoolValidate(&client_handle, value);
        show_help = 0;
      }
      break;
    case 6: /* Test 6: Get the pool name corresponding to a valid pool number: %s 6 <poolNumber> */
      if (argc == 3)
      {
        value = atoi(argv[2]);
        dhcp6sNumberToPoolNameGet(&client_handle, value);
        show_help = 0;
      }
      break;
    case 7: /* Test 7: Create a DHCPv6 address pool: %s 7 <poolName> */
      if (argc == 3)
      {
        dhcp6sPoolCreate(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 8: /* Test 8: Delete a DHCPv6 address pool: %s 8 <poolName> */
      if (argc == 3)
      {
        dhcp6sPoolDelete(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 9: /* Test 9: Get the first entry in DHCPv6 address pool: %s 9 */
      if (argc == 2)
      {
        dhcp6sPoolEntryFirst(&client_handle);
        show_help = 0;
      }
      break;
    case 10: /* Test 10: Get the next entry in DHCPv6 address pool: %s 10 <poolIndex> */
      if (argc == 3)
      {
        value = atoi(argv[2]);
        dhcp6sPoolEntryNext(&client_handle, value);
        show_help = 0;
      }
      break;
    case 11: /* Test 11: Add a host within a DHCPv6 Server pool: %s 11 <poolName> <hostName> <prefixAddr> <prefixLen> <validLifetime> <preferLifetime> */
      if (argc == 8)
      {
        prefixLen = atoi(argv[5]);
        validLifetime = atoi(argv[6]);
        preferLifetime = atoi(argv[7]);
        dhcp6sPoolHostAdd(&client_handle, argv[2], argv[3], argv[4], prefixLen, validLifetime, preferLifetime);
        show_help = 0;
      }
      break;
    case 12: /* Test 12: Delete the host within a DHCPv6 Server pool: %s 12 <poolName> <prefixAddr> <prefixLen> */
      if (argc == 5)
      {
        prefixLen = atoi(argv[4]);
        dhcp6sPoolHostRemove(&client_handle, argv[2], argv[3], prefixLen);
        show_help = 0;
      }
      break;
    case 13: /* Test 13: Print the host entries within a DHCPv6 Server pool: %s 13 <poolName> */
      if (argc == 3)
      {
        dhcp6sPoolHostGetNext(&client_handle, argv[2]);
        show_help = 0;
      }
    case 14: /* Test 14: Get the DHCPv6 Server pool type: %s 14 <poolName> */
      if (argc == 3)
      {
        dhcp6sPoolTypeGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 15: /* Test 15: Get the DHCPv6 DNS Domain Names from a pool: %s 15 <poolName> */
      if (argc == 3)
      {
        dhcp6sPoolDnsDomainsGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 16: /* Test 16: Add a DHCPv6 DNS Domain Name for a pool: %s 16 <poolName> <domainName> */
      if (argc == 4)
      {
        dhcp6sPoolDnsDomainAdd(&client_handle, argv[2], argv[3]);
        show_help = 0;
      }
      break;
    case 17: /* Test 17: Remove a DHCPv6 DNS Domain Name from a pool: %s 17 <poolName> <domainName> */
      if (argc == 4)
      {
        dhcp6sPoolDnsDomainRemove(&client_handle, argv[2], argv[3]);
        show_help = 0;
      }
      break;
    case 18: /* Test 18: Add a DHCPv6 host DNS Server for a automatic pool: %s 18 <poolName> <address> */
      if (argc == 4)
      {
        dhcp6sPoolDnsServerAdd(&client_handle, argv[2], argv[3]);
        show_help = 0;
      }
      break;
    case 19: /* Test 19: Get DHCPv6 host DNS Servers for a automatic pool: %s 19 <poolName> */
      if (argc == 3)
      {
        dhcp6sPoolDnsServersGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 20: /* Test 20: Delete a DHCPv6 host DNS Servers for a automatic pool: %s 20 <poolName> <Address> */
      if (argc == 4)
      {
        dhcp6sPoolDnsServerRemove(&client_handle, argv[2], argv[3]);
        show_help = 0;
      }
      break;
    case 21: /* Test 21: Delete DHCPv6 Server/Relay parameters on a given interface and VRF: %s 21 <ifNum> <vrfName> */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        dhcp6sIntfModeDelete(&client_handle, ifNum, argv[3]);
        show_help = 0;
      }
      break;
    case 22: /* Test 22: Set DHCPv6 Relay parameters for an interface: %s 22 <vrfName> <ifNum> <serverAddr> <serverIfNum> <remoteId> */
      if (argc == 7)
      {
        ifNum = atoi(argv[3]);
        serverIfNum = atoi(argv[5]);
        dhcp6sIntfModeRelaySet(&client_handle, argv[2], ifNum, argv[4], serverIfNum, argv[6]);
        show_help = 0;
      }
      break;
    case 23: /* Test 23: Set DHCPv6 Server parameters for an interface: %s 23 <ifNum> <poolName> <serverPref> <rapidCommit> <allowUnicast> */
      if (argc == 7)
      {
        ifNum = atoi(argv[2]);
        pref = atoi(argv[4]);
        rapidCommit = atoi(argv[5]);
        allowUnicast = atoi(argv[6]);
        dhcp6sIntfModeServerSet(&client_handle, ifNum, argv[3], pref, rapidCommit, allowUnicast);
        show_help = 0;
      }
      break;
    case 24: /* Test 24: Get the DHCPv6 Interface parameters for a given interface: %s 24 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        dhcp6sIntfModeGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 25: /* Test 25: Get the next interface with active DHCPv6 configuration: %s 25 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        dhcp6sIntfGetNext(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 26: /* Test 26: Verify the interface has active DHCPv6 configuration: %s 26 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        dhcp6sIntfValidate(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 27: /* Test 27: Determine if the interface is valid for participation in DHCPv6 components: %s 27 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        dhcp6MapIntfIsValid(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 28: /* Test 28: Get the first valid interface for participation in DHCPv6 components: %s 28 */
      if (argc == 2)
      {
        dhcp6MapIntfValidFirstGet(&client_handle);
        show_help = 0;
      }
      break;
    case 29: /* Test 29: Get the next valid interface for participation in DHCPv6 components: %s 29 <prevIfNum> */
      if (argc == 3)
      {
        prevIfNum = atoi(argv[2]);
        dhcp6MapIntfValidNextGet(&client_handle, prevIfNum);
        show_help = 0;
      }
      break;
    case 30: /* Test 30: Print the active bindings from the DHCPv6 server for a given VRF: %s 30 <vrfName> */
      if (argc == 3)
      {
        dhcp6sBindingGetNext(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 31: /* Test 31: Clear the binding specified by the IPv6 address for the given VRF: %s 31 <vrfName> <prefixType> <clientAddr> */
      if (argc == 5)
      {
        prefixType = atoi(argv[3]);
        dhcp6sBindingClear(&client_handle, argv[2], prefixType, argv[4]);
        show_help = 0;
      }
      break;
    case 32: /* Test 32: Clear all the binding entries for a given VRF: %s 32 <vrfName> */
      if (argc == 3)
      {
        dhcp6sBindingClearAll(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 33: /* Test 33: Get the count of number of bindings in the binding database for a given VRF: %s 33 <vrfName> */
      if (argc == 3)
      {
        dhcp6sBindingCountGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 34: /* Test 34: Get the DHCPv6-Server statistics for a given VRF or interface: %s 34 <ifNum> <vrfName> */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        dhcp6sStatisticsGet(&client_handle, ifNum, argv[3]);
        show_help = 0;
      }
      break;
    case 35: /* Test 35: Clear the DHCPv6-Server statistics for a given interface or VRF: %s 35 <ifNum> <vrfName> */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        dhcp6sStatisticsClear(&client_handle, ifNum, argv[3]);
        show_help = 0;
      }
      break;
    case 36: /* Test 36: Print the DHCPv6 interface relay entries for an interface or for all interfaces: %s 36 */
      if (argc == 2)
      {
        dhcpv6RelayEntryNextGet(&client_handle);
        show_help = 0;
      }
      break;
    case 37: /* Test 37: Get the exact DHCPv6 interface relay entry: %s 37 <ifNum> <relayAddr> */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        dhcpv6RelayEntryGet(&client_handle, ifNum, argv[3]);
        show_help = 0;
      }
      break;
    case 38: /* Test 38: Delete DHCPv6 Relay entry/parameters for an interface: %s 38 <vrfName> <ifNum> <serverAddr> <serverIfNum> <remoteId> */
      if (argc == 7)
      {
        ifNum = atoi(argv[3]);
        serverIfNum = atoi(argv[5]);
        dhcp6sIntfRelayEntryDelete(&client_handle, argv[2], ifNum, argv[4], serverIfNum, argv[6]);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping DHCP6s API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}
