/*********************************************************************
*
* Copyright 2016-2021 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
*************************************************************************
*
* @filename  dhcp_client_example.c
*
* @purpose   Dhcp Client Example.
*
* @component OPEN
*
* @comments
*
* @create    2/1/2016
*
* @end
*
***************************************************************************/
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_dhcp_client.h"

/*******************************************************************
*
* @brief  This is the main function that will demonstrate 
*         Dhcp Client OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure
*
********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  char string[100];
  l7proc_crashlog_register();
  int intIfNum = 1;
  int i = 0, j = 0, k = 0;
  open_inet_addr_t clientAddr,netmask,serverAddr;
  printf("please enter internal interface number \n");
  scanf( "%d", &intIfNum);

  /* Register with OpEN */
  if ((result = openapiClientRegister("dhcp_client_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting result = %d\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }
  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
  {
    printf("Network OS version = %s\n", switch_os_revision_string);
  }
  else
  {
    printf("Network OS version retrieve error\n");
  }
  printf("\n");

  if( 0 == openapiDhcpClientIPAddrGet(&clientHandle,intIfNum,&clientAddr))
  {
    sprintf(string, "%d.%d.%d.%d",
           (clientAddr.addr.ipv4 & 0xff000000) >> 24,
           (clientAddr.addr.ipv4 & 0x00ff0000) >> 16,
           (clientAddr.addr.ipv4 & 0x0000ff00) >> 8,
           (clientAddr.addr.ipv4 & 0x000000ff) );
    printf("Get the DHCP Client IP Address %s for interface number: %d  SUCCESS \n",string,intIfNum);
  }
  else
  {
    printf("Get the DHCP Client IP Address for interface number: %d  FAILED\n",intIfNum);
  }


  if( 0 == openapiDhcpClientNetMaskGet(&clientHandle,intIfNum,&netmask))
  {
    sprintf(string, "%d.%d.%d.%d",
           (netmask.addr.ipv4 & 0xff000000) >> 24,
           (netmask.addr.ipv4 & 0x00ff0000) >> 16,
           (netmask.addr.ipv4 & 0x0000ff00) >> 8,
           (netmask.addr.ipv4 & 0x000000ff) );
    printf("Get the DHCP Client subnet mask  %s for interface number: %d  SUCCESS\n",string,intIfNum);
  }
  else
  {
    printf("Get the DHCP Client subnet mask for interface number: %d  FAILED\n",intIfNum);
  }

  if( 0 == openapiDhcpClientDhcpServerIPAddrGet(&clientHandle,intIfNum,&serverAddr))
  {
    sprintf(string, "%d.%d.%d.%d",
           (serverAddr.addr.ipv4 & 0xff000000) >> 24,
           (serverAddr.addr.ipv4 & 0x00ff0000) >> 16,
           (serverAddr.addr.ipv4 & 0x0000ff00) >> 8,
           (serverAddr.addr.ipv4 & 0x000000ff) );
    printf("Get the DHCP Server IP Address %s for interface number: %d  SUCCESS\n", string,intIfNum);
  }
  else
  {
    printf("Get the DHCP Server IP Address for interface number: %d  FAILED\n",intIfNum);
  }

  open_bootp_dhcp_t networkParams;

  memset(&networkParams, 0, sizeof(networkParams));
  if ((networkParams.vendSpecificOptions = (open_dhcp_vendor_specific_option_t *) malloc (sizeof(open_dhcp_vendor_specific_option_t))) == NULL)
  {
    printf("Could not allocate memory. \n");
    exit(2);
  }
  memset(networkParams.vendSpecificOptions, 0, sizeof(open_dhcp_vendor_specific_option_t));

  if (OPEN_E_NONE == openapiDhcpClientOptionsGet(&clientHandle, &networkParams))
  {
      printf("Get the DHCP Options returned by the DHCP Server: SUCCESS\n");
      sprintf(string, "%u.%u.%u.%u",
              (networkParams.ip & 0xff000000) >> 24,
              (networkParams.ip & 0x00ff0000) >> 16,
              (networkParams.ip & 0x0000ff00) >> 8,
              (networkParams.ip & 0x000000ff));
      printf("Host ip address: %s\n", string);
      sprintf(string, "%u.%u.%u.%u",
              (networkParams.netMask & 0xff000000) >> 24,
              (networkParams.netMask & 0x00ff0000) >> 16,
              (networkParams.netMask & 0x0000ff00) >> 8,
              (networkParams.netMask & 0x000000ff));
      printf("Host netmask: %s\n", string);
      sprintf(string, "%u.%u.%u.%u",
              (networkParams.gateway & 0xff000000) >> 24,
              (networkParams.gateway & 0x00ff0000) >> 16,
              (networkParams.gateway & 0x0000ff00) >> 8,
              (networkParams.gateway & 0x000000ff));
      printf("Host Gateway: %s\n", string);
      sprintf(string, "%u.%u.%u.%u",
              (networkParams.siaddr & 0xff000000) >> 24,
              (networkParams.siaddr & 0x00ff0000) >> 16,
              (networkParams.siaddr & 0x0000ff00) >> 8,
              (networkParams.siaddr & 0x000000ff));
      printf("Next server address: %s\n", string);
      sprintf(string, "%u.%u.%u.%u",
              (networkParams.optTftpSvrIpAddr & 0xff000000) >> 24,
              (networkParams.optTftpSvrIpAddr & 0x00ff0000) >> 16,
              (networkParams.optTftpSvrIpAddr & 0x0000ff00) >> 8,
              (networkParams.optTftpSvrIpAddr & 0x000000ff));
      printf("IP address of TFTP server, got from option 150 field: %s\n", string);
      printf("IP addresses of DNS server, got from option 6 field: \n");
      for (i = 0; i < OPEN_DNS_NAME_SERVER_ENTRIES; i++)
      {
        sprintf(string, "%u.%u.%u.%u",
                (networkParams.optDnsServerIpAddr[i] & 0xff000000) >> 24,
                (networkParams.optDnsServerIpAddr[i] & 0x00ff0000) >> 16,
                (networkParams.optDnsServerIpAddr[i] & 0x0000ff00) >> 8,
                (networkParams.optDnsServerIpAddr[i] & 0x000000ff));
        printf("\t [%u]=%s\n", i, string);
      }
      printf("Domain name, got from option 15 field: %s\n", networkParams.optDefDomainName);
      printf("hostname name, got from option 12 field: %s\n", networkParams.optHostName);
      printf("IP addres of NTP server, got from option 42 field:\n");
      for (i = 0; i < OPEN_NTP_MAX_SERVER_ENTRIES; i++)
      {
        sprintf(string, "%u.%u.%u.%u",
                (networkParams.optNtpServerIpAddr[i] & 0xff000000) >> 24,
                (networkParams.optNtpServerIpAddr[i] & 0x00ff0000) >> 16,
                (networkParams.optNtpServerIpAddr[i] & 0x0000ff00) >> 8,
                (networkParams.optNtpServerIpAddr[i] & 0x000000ff));
        printf("\t [%u]=%s\n", i, string);
      }
      printf("Bootfile name: %s\n", networkParams.bootFile);
      printf("Bootfile name, got from option 67 field: %s\n", networkParams.optBootFile);
      printf("Server name: %s\n", networkParams.sname);
      printf("TFTP server name, got from option 66 field: %s\n", networkParams.optTftpSvrName);
      if ((0 != networkParams.vendSpecificOptions) &&
          ((networkParams.offeredOptionsMask & OPEN_BOOTP_DHCP_OPTION_VENDOR_SPECIFIC) != 0))
      {
        printf("DHCP Option-43:\n");
        for (i = 0;i < networkParams.vendSpecificOptions->numSubOptions; i++)
        {
          printf("\t[%u]: len=%u, type=%u option=",
                 i,
                 networkParams.vendSpecificOptions->subOptions[i].len,
                 networkParams.vendSpecificOptions->subOptions[i].type);
          for (j = 0; j < OPEN_DHCP_BOOTP_VEND_OPTIONS_LEN; j++)
          {
            printf("%u", networkParams.vendSpecificOptions->subOptions[i].option[j]);
          }
          printf("\n");
        }
      }
      else
      {
        printf("No DHCP Option-43 received\n");
      }
      printf("Offered options mask: %u\n", networkParams.offeredOptionsMask);
      printf("Opt 125. Vendor Identified option: \n");
      for (i = 0; i < OPEN_DHCP_OPT125_VENDOR_NUMBER_MAX; i++)
      {
        printf("\t[%u]: Vendor ID: %u\n", i, networkParams.opt125.vendor[i].vendorID);
        for (j = 0; j < networkParams.opt125.vendor[i].numSubOptions; j++)
        {
          printf("\t[%u, %u]: len=%u, type=%u, option=",
                 i, j,
                 networkParams.opt125.vendor[i].subOptions[j].len,
                 networkParams.opt125.vendor[i].subOptions[j].type);
          for (k = 0; k < OPEN_DHCP_BOOTP_VEND_OPTIONS_LEN; k++)
          {
            printf("%u", networkParams.opt125.vendor[i].subOptions[j].option[k]);
          }
          printf("\n");
        }
      }
      printf("DHCP Address Flush Flag: %u\n", networkParams.dhcpAddressFlushFlag);
  }
  else
  {
    printf("Get the DHCP Options returned by the DHCP Server: FAILED\n");
  }

  /* Log goodbye message */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping DHCP client example  application");
  free (networkParams.vendSpecificOptions);

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}

