/*********************************************************************
*
*  Copyright 2018-2024 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  dhcp_server_example.c
*
* @purpose   DHCP OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    7/13/2018
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_dhcp_server.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the DHCP server example application menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printAppMenu(char *name)
{
  printf("Usage:  %s <test#> <arg1> <arg2> ... \n", name);
  printf("Usage:  For arguments defaultRouters, dnsServers, netBiosNameServers, "
         "ntpServers provide mulitple addresses seperated by \",\"\n");
  printf("Test 1: Get lease data for ipaddr in default VRF: %s 1 <ipaddr>\n", name);
  printf("Test 2: Get lease data for VRF and ipaddr: %s 2 <vrf-name> <ipaddr>\n", name);
  printf("Test 3: Get pool entries: %s 3\n", name);
  printf("Test 4: Given a pool name, get lease entries: %s 4 <pool name>\n", name);
  printf("Test 5: Given a pool name, get pool type: %s 5 <pool name>\n", name);
  printf("Test 6: Display DHCP server statistics: %s 6\n", name);
  printf("Test 7: Test DHCP server OpEN API Sanity: %s 7\n", name);
  printf("Test 8: Add an excluded IP-address range for a VRF: %s 8 <low-ipaddr> <high-ipaddr> <vrf-name>\n", name);
  printf("Test 9: Delete an excluded IP-address range for a VRF: %s 9 <low-ipaddr> <high-ipaddr> <vrf-name>\n", name);
  printf("Test 10: Set the VRF name for a DHCP Server Pool: %s 10 <pool-name> <vrf-name>\n", name);
  printf("Test 11: Reset the VRF name to default VRF's name for a DHCP Server Pool: %s 11 <pool-name>\n", name);
  printf("Test 12: Get excluded addresses range entries: %s 12\n", name);
  printf("Test 13: Clear all lease data of a VRF: %s 13 <vrf-name>\n", name);
  printf("Test 14: Clear lease data for VRF and ipaddr: %s 14 <vrf-name> <ipaddr>\n", name);
  printf("Test 15: Clear all lease data of a Pool: %s 15 <pool-name>\n", name);
  printf("Test 16: Clear lease data for Pool and ipaddr: %s 16 <pool-name> <ipaddr>\n", name);
  printf("Test 17: Clear all lease data across all VRFs: %s 17\n", name);
  printf("Test 18: Set the DHCP server admin mode: %s 18 <mode>\n", name);
  printf("Test 19: Get the DHCP server admin mode: %s 19\n", name);
  printf("Test 20: Set the Bootp client allocation mode: %s 20 <mode>\n", name);
  printf("Test 21: Get the Bootp client allocation mode: %s 21\n", name);
  printf("Test 22: Set the number of pings to be sent out by DHCP server before assigning the address: %s 22 <pingPacketCount>\n", name);
  printf("Test 23: Get the number of pings to be sent out by DHCP server before assigning the address: %s 23 \n", name);
  printf("Test 24: Set the number of pings to be sent out by DHCP server before assigning the address, to default count: %s 24 \n", name);
  printf("Test 25: Create a DHCP server address pool: %s 25 <poolName>\n", name);
  printf("Test 26: Delete a DHCP server address pool: %s 26 <poolName>\n", name);
  printf("Test 27: Set the default routers for DHCP hosts in a given pool: %s 27 <poolName> <defaultRouters>\n", name);
  printf("Test 28: Get the default routers for DHCP hosts in a given pool: %s 28 <poolName>\n", name);
  printf("Test 29: Reset the default routers for DHCP hosts in a given pool: %s 29 <poolName>\n", name);
  printf("Test 30: Set the DHCP Host network for a pool: %s 30 <poolName> <networkAddr> <prefixLen>\n", name);
  printf("Test 31: Get the DHCP Host network for a pool: %s 31 <poolName>\n", name);
  printf("Test 32: Delete the DHCP Host network for a pool: %s 32 <poolName>\n", name);
  printf("Test 33: Set the DNS servers for DHCP hosts in a given pool: %s 33 <poolName> <dnsServers>\n", name);
  printf("Test 34: Get the DNS servers for DHCP hosts in a given pool: %s 34 <poolName>\n", name);
  printf("Test 35: Reset the DNS servers for DHCP hosts in a given pool: %s 35 <poolName>\n", name);
  printf("Test 36: Set the Netbios name server addresses in given pool: %s 36 <poolName> <netBiosNameServers>\n", name);
  printf("Test 37: Get the Netbios name server addresses in given pool: %s 37 <poolName>\n", name);
  printf("Test 38: Delete the Netbios name server addresses in given pool: %s 38 <poolName>\n", name);
  printf("Test 39: Set the type of Netbios node: %s 39 <poolName> <nodeType>\n", name);
  printf("Test 40: Get the type of Netbios node: %s 40 <poolName>\n", name);
  printf("Test 41: Delete the type of Netbios node: %s 41 <poolName>\n", name);
  printf("Test 42: Set the DHCP lease time: %s 42 <poolName> <leaseTime>\n", name);
  printf("Test 43: Get the DHCP lease time: %s 43 <poolName>\n", name);
  printf("Test 44: Reset the DHCP lease time: %s 44 <poolName>\n", name);
  printf("Test 45: Set the domain name of a DHCP client: %s 45 <poolName> <domainName>\n", name);
  printf("Test 46: Get the domain name of a DHCP client: %s 46 <poolName>\n", name);
  printf("Test 47: Delete the domain name of a DHCP client: %s 47 <poolName>\n", name);
  printf("Test 48: Set the next server IP address: %s 48 <poolName> <nextAddress>\n", name);
  printf("Test 49: Get the next server IP address: %s 49 <poolName>\n", name);
  printf("Test 50: Delete the next server IP address: %s 50 <poolName>\n", name);
  printf("Test 51: Set the DHCP host hardware address for a manual pool: %s 51 <poolName> <hwAddr> <hwType>\n", name);
  printf("Test 52: Get the DHCP host hardware address for a manual pool: %s 52 <poolName>\n", name);
  printf("Test 53: Delete the DHCP host hardware address for a manual pool: %s 53 <poolName>\n", name);
  printf("Test 54: Set the DHCP Host network for a manual pool: %s 54 <poolName> <hostAddr> <prefixLen>\n", name);
  printf("Test 55: Get the DHCP Host network for a manual pool: %s 55 <poolName>\n", name);
  printf("Test 56: Delete the DHCP Host address and mask for a pool: %s 56 <poolName>\n", name);
  printf("Test 57: Set the boot file name: %s 57 <poolName>\n", name);
  printf("Test 58: Get the boot file name: %s 58 <poolName> <fileName>\n", name);
  printf("Test 59: Delete the boot file name: %s 59 <poolName>\n", name);
  printf("Test 60: Set the NTP server addresses for a given pool: %s 60 <poolName> <ntpServers>\n", name);
  printf("Test 61: Get the NTP server addresses for a given pool: %s 61 <poolName>\n", name);
  printf("Test 62: Delete the NTP server addresses for a given pool: %s 62 <poolName>\n", name);
  printf("Test 63: Clear the DHCP server statistics: %s 63 \n", name);
  printf("Test 64: Set the client id of a DHCP client: %s 64 <poolName> <clientId>\n", name);
  printf("Test 65: Get the client id of a DHCP client: %s 65 <poolName>\n", name);
  printf("Test 66: Delete the client id of a DHCP client: %s 66 <poolName>\n", name);
  printf("Test 67: Set the client name of a DHCP client: %s 67 <poolName> <clientName>\n", name);
  printf("Test 68: Get the client name of a DHCP client: %s 68 <poolName>\n", name);
  printf("Test 69: Delete the client name of a DHCP client: %s 69 <poolName>\n", name);
  printf("Test 70: Set the auto default router mode for a given DHCP Pool name: %s 70 <poolName> <isAutoDefRtrModeEnabled[0-False,1-True]>\n", name);
  printf("Test 71: Get the auto default router mode for a given DHCP Pool name: %s 71 <poolName>\n", name);
  return;
}

/*********************************************************************
* @purpose  Get lease data for ipaddr
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    ipaddr           @b{(input)}   IPv4 address
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void getDhcpServerLeaseData(openapiClientHandle_t *clientHandle, char *ipaddr)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  uint32_t mask;
  open_buffdesc hwaddr;
  OPEN_DHCP_SERVER_CLIENT_ID_t clientId;
  OPEN_DHCP_SERVER_POOL_NAME_t poolName;
  uint32_t remaining;
  unsigned char buffer[OPEN_MAC_ADDR_LEN];

  hwaddr.pstart = buffer;
  hwaddr.size = sizeof(buffer);

  memset(&ipAddr, 0, sizeof(ipAddr));
  ipAddr.family = OPEN_AF_INET;
  /* ipAddr.addr.ipv4 = inet_addr(ipaddr);  OpEN requires host order */
  inet_pton(AF_INET, ipaddr, (void*)&(ipAddr.addr.ipv4));
  ipAddr.addr.ipv4 = htonl(ipAddr.addr.ipv4);
  if (ipAddr.addr.ipv4 == INADDR_NONE) 
  {
    printf("Invalid IP address string '%s', IPv4 address string required\n",
           ipaddr);
  }
  else
  {
    result = openapiDhcpServerLeaseDataGet(clientHandle,
                                           &ipAddr,
                                           &mask,
                                           &hwaddr,
                                           &clientId,
                                           &poolName,
                                           &remaining);
    if (result != OPEN_E_NONE)
    {
      printf("Bad return code trying to get lease data. (result = %d)\n", result);
    }
    else
    {
      printf("IP Addr          IP Mask         hwaddr            Pool Name                        Remaining Client ID\n");
      printf("-------------------------------------------------------------------------------------------------------\n");
      printf("%-16s %03d.%03d.%03d.%03d %02x:%02x:%02x:%02x:%02x:%02x %-32s%5d      %s\n",
             ipaddr, (mask & 0xff000000) >> 24, (mask & 0x00ff0000) >> 16,
             (mask & 0x0000ff00) >> 8, mask & 0x000000ff, 
             buffer[0], buffer[1], buffer[2], buffer[3], buffer[4], 
             buffer[5], poolName.name, remaining, clientId.clientid);
    }
  }
}

/*********************************************************************
* @purpose  Get lease data for ipaddr and VRF-name 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    vrfName          @b{(input)}   VRF name 
* @param    ipaddr           @b{(input)}   IPv4 address
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void getDhcpServerVrfIpAddrLeaseData( openapiClientHandle_t *clientHandle, char *vrfName, char *ipaddr)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  uint32_t mask;
  open_buffdesc hwaddr;
  OPEN_DHCP_SERVER_CLIENT_ID_t clientId;
  OPEN_DHCP_SERVER_POOL_NAME_t poolName;
  OPEN_DHCP_SERVER_POOL_VRF_NAME_t strVrfName;
  uint32_t remaining;
  unsigned char buffer[OPEN_MAC_ADDR_LEN];


  hwaddr.pstart = buffer;
  hwaddr.size = sizeof(buffer);

  memset(&ipAddr, 0, sizeof(ipAddr));
  ipAddr.family = OPEN_AF_INET;
  /* ipAddr.addr.ipv4 = inet_addr(ipaddr);  OpEN requires host order */
  inet_pton(AF_INET, ipaddr, (void*)&(ipAddr.addr.ipv4));
  ipAddr.addr.ipv4 = htonl(ipAddr.addr.ipv4);
  if (ipAddr.addr.ipv4 == INADDR_NONE) 
  {
    printf("Invalid IP address string '%s', IPv4 address string required\n",
           ipaddr);
  }
  else
  {
    strncpy(strVrfName.name, vrfName, sizeof(strVrfName.name));
    result = openapiDhcpServerVrfIpAddrLeaseDataGet(clientHandle,
                                                    &strVrfName,
                                                    &ipAddr,
                                                    &mask,
                                                    &hwaddr,
                                                    &clientId,
                                                    &poolName,
                                                    &remaining);
    if (result != OPEN_E_NONE)
    {
      printf("Bad return code trying to get lease data. (result = %d)\n", result);
    }
    else
    {
      printf("IP Addr          IP Mask         hwaddr            Pool Name                        Remaining Client ID\n");
      printf("-------------------------------------------------------------------------------------------------------\n");
      printf("%-16s %03d.%03d.%03d.%03d %02x:%02x:%02x:%02x:%02x:%02x %-32s%5d      %s\n",
             ipaddr, (mask & 0xff000000) >> 24, (mask & 0x00ff0000) >> 16,
             (mask & 0x0000ff00) >> 8, mask & 0x000000ff, 
             buffer[0], buffer[1], buffer[2], buffer[3], buffer[4], 
             buffer[5], poolName.name, remaining, clientId.clientid);
    }
  }
}

/*********************************************************************
* @purpose  Clear lease data for ipaddr and VRF-name 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    vrfName          @b{(input)}   VRF name 
* @param    ipaddr           @b{(input)}   IPv4 address
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void clearDhcpServerVrfIpAddrLeaseData( openapiClientHandle_t *clientHandle, char *vrfName, char *ipaddr)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  OPEN_DHCP_SERVER_POOL_VRF_NAME_t strVrfName;


  memset(&ipAddr, 0, sizeof(ipAddr));
  ipAddr.family = OPEN_AF_INET;
  /* ipAddr.addr.ipv4 = inet_addr(ipaddr);  OpEN requires host order */
  inet_pton(AF_INET, ipaddr, (void*)&(ipAddr.addr.ipv4));
  ipAddr.addr.ipv4 = htonl(ipAddr.addr.ipv4);
  if (ipAddr.addr.ipv4 == INADDR_NONE) 
  {
    printf("Invalid IP address string '%s', IPv4 address string required\n", ipaddr);
  }
  else
  {
    strncpy(strVrfName.name, vrfName, sizeof(strVrfName.name));
    result = openapiDhcpServerVrfIpAddrLeaseDataClear(clientHandle,
                                                      &strVrfName,
                                                      &ipAddr);
    if (result != OPEN_E_NONE)
    {
      printf("Bad return code trying to clear lease data. (result = %d)\n", result);
    }
    else
    {
      printf("Cleared successfully.\n");
    }
  }
}

/*********************************************************************
* @purpose  Clear lease data of a given VRF instance 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    vrfName          @b{(input)}   VRF name 
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void clearDhcpServerVrfLeaseData( openapiClientHandle_t *clientHandle, char *vrfName)
{
  open_error_t result;
  OPEN_DHCP_SERVER_POOL_VRF_NAME_t strVrfName;


  strncpy(strVrfName.name, vrfName, sizeof(strVrfName.name));
  result = openapiDhcpServerVrfLeaseDataClear(clientHandle,
                                              &strVrfName);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to clear lease data. (result = %d)\n", result);
  }
  else
  {
    printf("Cleared successfully.\n");
  }
}

/*********************************************************************
* @purpose  Clear lease data for ipaddr and Pool-name 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    poolName         @b{(input)}   Pool name 
* @param    ipaddr           @b{(input)}   IPv4 address
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void clearDhcpServerPoolIpAddrLeaseData( openapiClientHandle_t *clientHandle, char *poolName, char *ipaddr)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  OPEN_DHCP_SERVER_POOL_NAME_t strPoolName;


  memset(&ipAddr, 0, sizeof(ipAddr));
  ipAddr.family = OPEN_AF_INET;
  /* ipAddr.addr.ipv4 = inet_addr(ipaddr);  OpEN requires host order */
  inet_pton(AF_INET, ipaddr, (void*)&(ipAddr.addr.ipv4));
  ipAddr.addr.ipv4 = htonl(ipAddr.addr.ipv4);
  if (ipAddr.addr.ipv4 == INADDR_NONE) 
  {
    printf("Invalid IP address string '%s', IPv4 address string required\n", ipaddr);
  }
  else
  {
    strncpy(strPoolName.name, poolName, sizeof(strPoolName.name));
    result = openapiDhcpServerPoolIpAddrLeaseDataClear(clientHandle,
                                                       &strPoolName,
                                                       &ipAddr);
    if (result != OPEN_E_NONE)
    {
      printf("Bad return code trying to clear lease data. (result = %d)\n", result);
    }
    else
    {
      printf("Cleared successfully.\n");
    }
  }
}

/*********************************************************************
* @purpose  Clear lease data of a given Pool-name 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    poolName         @b{(input)}   Pool name 
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void clearDhcpServerPoolLeaseData( openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  OPEN_DHCP_SERVER_POOL_NAME_t strPoolName;


  strncpy(strPoolName.name, poolName, sizeof(strPoolName.name));
  result = openapiDhcpServerPoolLeaseDataClear(clientHandle,
                                               &strPoolName);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to clear lease data. (result = %d)\n", result);
  }
  else
  {
    printf("Cleared successfully.\n");
  }
}

/*********************************************************************
* @purpose  Clear all lease data across all VRFs
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void clearDhcpServerAllLeaseData( openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  result = openapiDhcpServerAllLeaseDataClear(clientHandle);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to clear lease data. (result = %d)\n", result);
  }
  else
  {
    printf("Cleared successfully.\n");
  }
}

/*********************************************************************
* @purpose  Get pool entries. 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void getDhcpServerPoolEntries(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_DHCP_SERVER_POOL_NAME_t name;
  OPEN_DHCP_SERVER_POOL_VRF_NAME_t vrfName;
  uint32_t index;
  
  result = openapiDhcpServerPoolEntryFirstGet(clientHandle, &index, &name);
  if (result == OPEN_E_NONE)
  {
    result = openapiDhcpServerPoolVrfNameGet(clientHandle, &name, &vrfName); 
    if (result != OPEN_E_NONE)
    {
      printf("Non-success return code from openapiDhcpServerPoolVrfNameGet(result = %d).\n", result);
      return;
    }

    printf("Pool-Index                       Pool-Name VRF-Name \n");
    printf("---------------------------------------------------------------\n");
    while (result == OPEN_E_NONE)
    {
      printf("%10d %31s %s\n", index, name.name, vrfName.name);
      result = openapiDhcpServerPoolEntryNextGet(clientHandle, &index, &name);
      if (result != OPEN_E_NONE && result != OPEN_E_NOT_FOUND) 
      { 
        printf("Non-success return code from openapiDhcpServerPoolEntryNextGet(result = %d).\n", result);
      }

      if (result == OPEN_E_NONE)
      {
        result = openapiDhcpServerPoolVrfNameGet(clientHandle, &name, &vrfName); 
        if (result != OPEN_E_NONE)
        {
          printf("Non-success return code from openapiDhcpServerPoolVrfNameGet(result = %d).\n", result);
        }
      }
    }
  }
  else
  {
    printf("Non-success return code from openapiDhcpServerPoolEntryFirstGet (result = %d).\n", result);
  }
}

/*********************************************************************
* @purpose  Map a lease state to a string. 
* 
* @param    state             @b{(input)}   Instance of OPEN_DHCP_SERVER_LEASE_STATE_t
* 
* @returns  a static string representing the passed-in lease state
*   
* @notes  
* 
* @end
*********************************************************************/

static const char *stateToStr(OPEN_DHCP_SERVER_LEASE_STATE_t state) 
{
  char *ret = "Unknown lease state";

  switch(state) 
  {
  case OPEN_DHCP_SERVER_FREE_LEASE:
    ret = "free";
    break;
  case OPEN_DHCP_SERVER_ACTIVE_LEASE:
    ret = "active";
    break;
  case OPEN_DHCP_SERVER_OFFERED_LEASE:
    ret = "offered";
    break;
  case OPEN_DHCP_SERVER_EXPIRED_LEASE:
    ret = "expired";
    break;
  case OPEN_DHCP_SERVER_ABANDONED_LEASE:
    ret = "abandoned";
    break;
  }
  return ret;
}

/*********************************************************************
* @purpose  Get lease entries.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    name             @b{(input)}   Pool name 
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getDhcpServerLeaseEntries(openapiClientHandle_t *clientHandle, char *name)
{
  open_error_t result;
  OPEN_DHCP_SERVER_POOL_NAME_t poolName;
  uint32_t index;
  open_inet_addr_t addr;
  OPEN_DHCP_SERVER_LEASE_STATE_t state;
  char ipStr[INET_ADDRSTRLEN];
  
  strncpy(poolName.name, name, sizeof(poolName.name));
  result = openapiDhcpServerLeaseEntryFirstGet(clientHandle, &poolName, &index,
                                               &addr, &state);
  if (result == OPEN_E_NONE) 
  {
    printf("Pool name         Lease index  IPv4 Addr        State\n");
    printf("-----------------------------------------------------\n");
    while (result == OPEN_E_NONE)
    {
      addr.addr.ipv4=ntohl(addr.addr.ipv4);
      inet_ntop(AF_INET, &(addr.addr.ipv4), ipStr, sizeof(ipStr));
      printf("%-16s  %05d        %-16s %s\n", poolName.name, index, ipStr, stateToStr(state));
      result = openapiDhcpServerLeaseEntryNextGet(clientHandle, &poolName, &index, &addr, &state);
      if (result != OPEN_E_NONE && result != OPEN_E_NOT_FOUND) 
      { 
        printf("Non-success return code from openapiDhcpServerPoolEntryNextGet(result = %d)\n", result);
      }
    }
  }
  else
  {
    printf("Non-success return code from openapiDhcpServerLeaseEntryFirstGet (result = %d)\n", result);
  }
}

/*********************************************************************
* @purpose  Map a pool type to a string
* 
* @param    type             @b{(input)}   Instance of OPEN_DHCP_SERVER_POOL_TYPE_t
* 
* @returns  a static string representing the passed-in type
*   
* @notes  
* 
* @end
*********************************************************************/

static const char *poolTypeToStr(OPEN_DHCP_SERVER_POOL_TYPE_t type) 
{
  char *ret = "Unknown type";

  switch(type) 
  {
  case OPEN_DHCP_SERVER_POOL_TYPE_INACTIVE:
    ret = "inactive";
    break;
  case OPEN_DHCP_SERVER_POOL_TYPE_DYNAMIC:
    ret = "dynamic";
    break;
  case OPEN_DHCP_SERVER_POOL_TYPE_MANUAL:
    ret = "manual";
    break;
  }
  return ret;
}

/*********************************************************************
* @purpose  Given a pool name, get pool type
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    name             @b{(input)}   Pool name 
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void getDhcpServerPoolTypeForName(openapiClientHandle_t *clientHandle, char *name)
{
  open_error_t result;
  OPEN_DHCP_SERVER_POOL_NAME_t poolName;
  OPEN_DHCP_SERVER_POOL_TYPE_t type;

  strncpy(poolName.name, name, sizeof(poolName.name));

  if ((result = openapiDhcpServerPoolTypeGet(clientHandle, &poolName, &type)) != OPEN_E_NONE)
  {
    printf("Bad return code from openapiDhcpServerPoolTypeGet. (result = %d)\n", result);
  }
  else
  {
    printf("Type for pool %s is %s\n", name, poolTypeToStr(type));
  }
}

/*********************************************************************
* @purpose  Set the name of the VRF instance of a given DHCP Server Pool
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    pool_name        @b{(input)}   Pool name 
* @param    vrf_name         @b{(input)}   VRF name 
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void setDhcpServerPoolVrfName( openapiClientHandle_t *clientHandle, char *pool_name, char *vrf_name)
{
  open_error_t result;
  OPEN_DHCP_SERVER_POOL_NAME_t poolName;
  OPEN_DHCP_SERVER_POOL_VRF_NAME_t vrfName;

  strncpy(poolName.name, pool_name, sizeof(poolName.name));
  strncpy(vrfName.name, vrf_name, sizeof(vrfName.name));

  if ((result = openapiDhcpServerPoolVrfNameSet(clientHandle, &poolName, &vrfName) != OPEN_E_NONE))
  {
    printf("Bad return code from openapiDhcpServerPoolVrfNameSet. (result = %d)\n", result);
  }
  else
  {
    printf("Successfully set. \n");
  }
}

/*********************************************************************
* @purpose Reset the name of the VRF instance of a given DHCP Server Pool to default VRF
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    pool_name        @b{(input)}   Pool name 
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void resetDhcpServerPoolVrfNameToDefaultVrf( openapiClientHandle_t *clientHandle, char *pool_name)
{
  open_error_t result;
  OPEN_DHCP_SERVER_POOL_NAME_t poolName;

  strncpy(poolName.name, pool_name, sizeof(poolName.name));

  if ((result = openapiDhcpServerPoolVrfNameSetToDefaultVrf(clientHandle, &poolName) != OPEN_E_NONE))
  {
    printf("Bad return code from resetDhcpServerPoolVrfNameToDefaultVrf. (result = %d)\n", result);
  }
  else
  {
    printf("Successfully reset. \n");
  }
}

/*********************************************************************
* @purpose  Get excluded addresses range entries. 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void getDhcpServerExcludedAddressesRangeEntries(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_inet_addr_t startIpAddr;
  open_inet_addr_t endIpAddr;
  char startIpStr[INET_ADDRSTRLEN];
  char   endIpStr[INET_ADDRSTRLEN];
  OPEN_DHCP_SERVER_POOL_VRF_NAME_t strVrfName;
  uint32_t index;

  result = openapiDhcpServerExcludedAddressRangeEntryFirstGet(clientHandle,
                                                              &index,
                                                              &strVrfName,
                                                              &startIpAddr,
                                                              &endIpAddr);

  if (result == OPEN_E_NONE)
  {
    printf("Index  Start-IPaddress   End-IPaddress     VRF-Name \n");
    printf("---------------------------------------------------------------\n");
    while (result == OPEN_E_NONE)
    {
      startIpAddr.addr.ipv4=ntohl(startIpAddr.addr.ipv4);
      inet_ntop(AF_INET, &(startIpAddr.addr.ipv4), startIpStr, sizeof(startIpStr));
      endIpAddr.addr.ipv4=ntohl(endIpAddr.addr.ipv4);
      inet_ntop(AF_INET, &(endIpAddr.addr.ipv4), endIpStr, sizeof(endIpStr));

      printf("%05d  %-16s  %-16s  %s\n", index, startIpStr, endIpStr, strVrfName.name);

      result = openapiDhcpServerExcludedAddressRangeEntryNextGet(clientHandle, &index, &strVrfName, &startIpAddr, &endIpAddr);
      if (result != OPEN_E_NONE && result != OPEN_E_NOT_FOUND)
      {
        printf("Non-success return code from openapiDhcpServerExcludedAddressRangeEntryNextGet(result = %d).\n", result);
      }
    }
  }
  else
  {
    printf("Non-success return code from openapiDhcpServerExcludedAddressRangeEntryFirstGet(result = %d).\n", result);
  }
}

/*********************************************************************
* @purpose  Add an excluded-address range for a given VRF instance
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    vrfName          @b{(input)}   VRF name 
* @param    lowIpAddr        @b{(input)}   start IPv4 address of the address range
* @param    highIpAddr       @b{(input)}   end IPv4 address of the address range
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void addDhcpServerExcludedAddressRangeForVrf( openapiClientHandle_t *clientHandle, char *lowIpAddr, char *highIpAddr, char *vrfName)
{
  open_error_t result;
  open_inet_addr_t startIpAddr;
  open_inet_addr_t endIpAddr;
  OPEN_DHCP_SERVER_POOL_VRF_NAME_t strVrfName;


  memset(&startIpAddr, 0, sizeof(startIpAddr));
  startIpAddr.family = OPEN_AF_INET;
  /* startIpAddr.addr.ipv4 = inet_addr(lowIpAddr);  OpEN requires host order */
  inet_pton(AF_INET, lowIpAddr, (void*)&(startIpAddr.addr.ipv4));
  startIpAddr.addr.ipv4 = htonl(startIpAddr.addr.ipv4);
  if (startIpAddr.addr.ipv4 == INADDR_NONE) 
  {
    printf("Invalid low IP address string '%s', IPv4 address string required\n", lowIpAddr);
    return;
  }

  memset(&endIpAddr, 0, sizeof(endIpAddr));
  endIpAddr.family = OPEN_AF_INET;
  /* endIpAddr.addr.ipv4 = inet_addr(highIpAddr);  OpEN requires host order */
  inet_pton(AF_INET, highIpAddr, (void*)&(endIpAddr.addr.ipv4));
  endIpAddr.addr.ipv4 = htonl(endIpAddr.addr.ipv4);
  if (endIpAddr.addr.ipv4 == INADDR_NONE) 
  {
    printf("Invalid high IP address string '%s', IPv4 address string required\n", highIpAddr);
    return;
  }

  strncpy(strVrfName.name, vrfName, sizeof(strVrfName.name));
  result = openapiDhcpServerExcludedAddressRangeAdd(clientHandle,
                                                    &strVrfName,
                                                    &startIpAddr,
                                                    &endIpAddr);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to add an excluded-address range. (result = %d).\n", result);
  }
  else
  {
    printf("Successfully added. \n");
  }
}

/*********************************************************************
* @purpose  Delete an excluded-address range for a given VRF instance
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    vrfName          @b{(input)}   VRF name 
* @param    lowIpAddr        @b{(input)}   start IPv4 address of the address range
* @param    highIpAddr       @b{(input)}   end IPv4 address of the address range
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void deleteDhcpServerExcludedAddressRangeForVrf( openapiClientHandle_t *clientHandle, char *lowIpAddr, char *highIpAddr, char *vrfName)
{
  open_error_t result;
  open_inet_addr_t startIpAddr;
  open_inet_addr_t endIpAddr;
  OPEN_DHCP_SERVER_POOL_VRF_NAME_t strVrfName;


  memset(&startIpAddr, 0, sizeof(startIpAddr));
  startIpAddr.family = OPEN_AF_INET;
  /* startIpAddr.addr.ipv4 = inet_addr(lowIpAddr);  OpEN requires host order */
  inet_pton(AF_INET, lowIpAddr, (void*)&(startIpAddr.addr.ipv4));
  startIpAddr.addr.ipv4 = htonl(startIpAddr.addr.ipv4);
  if (startIpAddr.addr.ipv4 == INADDR_NONE) 
  {
    printf("Invalid low IP address string '%s', IPv4 address string required\n", lowIpAddr);
    return;
  }

  memset(&endIpAddr, 0, sizeof(endIpAddr));
  endIpAddr.family = OPEN_AF_INET;
  /* endIpAddr.addr.ipv4 = inet_addr(highIpAddr);  OpEN requires host order */
  inet_pton(AF_INET, highIpAddr, (void*)&(endIpAddr.addr.ipv4));
  endIpAddr.addr.ipv4 = htonl(endIpAddr.addr.ipv4);
  if (endIpAddr.addr.ipv4 == INADDR_NONE) 
  {
    printf("Invalid high IP address string '%s', IPv4 address string required\n", highIpAddr);
    return;
  }

  strncpy(strVrfName.name, vrfName, sizeof(strVrfName.name));
  result = openapiDhcpServerExcludedAddressRangeDelete(clientHandle,
                                                       &strVrfName,
                                                       &startIpAddr,
                                                       &endIpAddr);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete an excluded-address range. (result = %d).\n", result);
  }
  else
  {
    printf("Successfully deleted. \n");
  }
}

/*********************************************************************
* @purpose  Display the set of DHCP server statistics
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void displayDhcpServerStats(openapiClientHandle_t *clientHandle)
{
  static char *statLabel[OPEN_DHCPS_STAT_ID_LAST] =
  {
    "Reserved",
    "Active Leases",
    "Expired Leases",
    "Malformed Messages",
    "Discover Dropped",
    "Discover Received",
    "Request Received",
    "Decline Received",
    "Release Received",
    "Inform Received",
    "Offer Sent",
    "Ack Sent",
    "Nack Sent"
  };
  open_error_t result;
  uint32_t value;
  OPEN_DHCPS_STAT_ID_t id;

  for (id = OPEN_DHCPS_STAT_ID_ACTIVE_LEASES; id < OPEN_DHCPS_STAT_ID_LAST; id++)
  {
    if ((result = openapiDhcpServerStatGet(clientHandle, id, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code from openapiDhcpServerStatGet for ID %u. (result = %d)\n", id, result);
    }
    else
    {
      printf("%-20s: %u\n", statLabel[id], value);
    }
  }
}

/*********************************************************************
* @purpose  Set the DHCP server admin mode.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    mode           @b{(input)}  DHCP Server mode

*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsAdminModeSet (openapiClientHandle_t *clientHandle, uint32_t mode)
{
  open_error_t result;

  if ((result = openapiDhcpsAdminModeSet (clientHandle, (OPEN_CONTROL_t)mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the DHCP server admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("\nDHCP sever admin mode is now %s\n", ((OPEN_ENABLE == mode) ? "ENABLED" : "DISABLED"));
  }
  return;
}


/*********************************************************************
* @purpose  Get the DHCP server admin mode.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsAdminModeGet (openapiClientHandle_t *clientHandle)
{
  open_error_t   result;
  OPEN_CONTROL_t mode = OPEN_DISABLE;

  if ((result = openapiDhcpsAdminModeGet (clientHandle, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the DHCP server admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("\nDHCP sever admin mode is %s\n", ((OPEN_ENABLE == mode) ? "ENABLED" : "DISABLED"));
  }
  return;
}


/*********************************************************************
* @purpose  Set the Bootp client allocation mode.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    mode           @b{(input)}  Bootp auto allocation mode

*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsBootpAutomaticSet (openapiClientHandle_t *clientHandle, uint32_t mode)
{
  open_error_t result;

  if ((result = openapiDhcpsBootpAutomaticSet (clientHandle, (OPEN_CONTROL_t)mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the Bootp client allocation mode. (result = %d)\n", result);
  }
  else
  {
    printf("\nDHCP Bootp automatic allocation is now %s\n", ((OPEN_ENABLE == mode) ? "ENABLED" : "DISABLED"));
  }
  return;
}


/*********************************************************************
* @purpose  Get the Bootp client allocation mode.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsBootpAutomaticGet (openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t mode = OPEN_DISABLE;

  if ((result = openapiDhcpsBootpAutomaticGet (clientHandle, &mode)) != OPEN_E_NONE)
  {
    printf ("Bad return code trying to get the Bootp client allocation mode. (result = %d)\n", result);
  }
  else
  {
    printf ("\nDHCP Bootp automatic allocation is %s\n", ((OPEN_ENABLE == mode) ? "ENABLED" : "DISABLED"));
  }
  return;
}


/*********************************************************************
* @purpose  Set the number of pings to be sent out by DHCP server before assigning the address.
*
* @param    clientHandle     @b{(input)}  client handle from registration API
* @param    pingPacketCount  @b{(input)}  Number of ping packets to be sent
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsNoOfPingPktSet (openapiClientHandle_t *clientHandle, uint32_t pingPacketCount)
{
  open_error_t result;

  if ((result = openapiDhcpsNoOfPingPktSet (clientHandle, pingPacketCount)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the number of pings to be sent out by DHCP server before assigning the address. (result = %d)\n", result);
  }
  else
  {
    printf("\nThe number of pings sent out by DHCP server is now %u\n", pingPacketCount);
  }
  return;
}


/*********************************************************************
* @purpose  Get the number of pings to be sent out by DHCP server before assigning the address.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsNoOfPingPktGet (openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t pingPacketCount = 0;

  if ((result = openapiDhcpsNoOfPingPktGet(clientHandle, &pingPacketCount)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the number of pings to be sent out by DHCP server before assigning the address. (result = %d)\n", result);
  }
  else
  {
    printf("\nThe number of pings sent out by DHCP server is %u\n", pingPacketCount);
  }
  return;
}


/*********************************************************************
* @purpose  Set the number of pings to be sent out by DHCP server before assigning the address, to default count.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsNoOfPingPktReset (openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  if ((result = openapiDhcpsNoOfPingPktReset(clientHandle)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the number of pings to be sent out by DHCP server before assigning the address, to default count. (result = %d)\n", result);
  }
  else
  {
    printf ("\nThe number of ping packets sent out by DHCP server is now set to defaults\n");
  }
  return;
}


/*********************************************************************
* @purpose  Create a DHCP server address pool.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of DHCP server address pool

*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsPoolCreate (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsPoolCreate (clientHandle, &poolNameBuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to create a DHCP server address pool. (result = %d)\n", result);
  }
  else
  {
    printf("\nDHCP server Pool \"%s\" is created\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Delete a DHCP server address pool.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of DHCP server address pool

*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsPoolDelete (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsPoolDelete (clientHandle, &poolNameBuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete a DHCP server address pool. (result = %d)\n", result);
  }
  else
  {
    printf("\nDHCP server Pool \"%s\" is deleted\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Set the default routers for DHCP hosts in a given pool.
*
* @param    clientHandle    @b{(input)}  client handle from registration API
* @param    poolName        @b{(input)}  Name of the DHCP server address pool
* @param    defaultRouters  @b{(input)}  List of default router address
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsDefaultRoutersSet (openapiClientHandle_t *clientHandle,
                             char *poolName, char *defRtrs)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  openDhcpsDefaultRouters_t defaultRouters;
  char *str_ptr = NULL;
  char *ptr = NULL;
  uint32_t i = 0;
  open_inet_addr_t ipAddr;
  open_buffdesc ipBuffdesc;
  char str[40];
      
  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  memset (&defaultRouters, 0, sizeof (defaultRouters));
  ptr = strtok_r (defRtrs, ",", &(str_ptr));
  while (NULL != ptr)
  {
    /* Extract and validate each IP-Address string */
    memset(str, 0, sizeof(str));
    strncpy(str, ptr, sizeof(str) - 1);
    ipBuffdesc.pstart = str;
    ipBuffdesc.size = strlen(str) + 1;
    memset(&ipAddr, 0, sizeof(ipAddr));
    if ((result = openapiInetAddrGet(clientHandle, &ipBuffdesc, &ipAddr)) != OPEN_E_NONE)
    {     
      printf("Bad return code trying to convert IP address. (result = %d)\n", result);
      return;
    }
    if (ipAddr.family != OPEN_AF_INET)
    {
      printf("Invalid IPv4 address. \n");
      return;
    }
    defaultRouters.routers[i] = ipAddr.addr.ipv4;
    ptr = strtok_r (NULL, ",", &(str_ptr));
    i++;
  }

  if ((result = openapiDhcpsDefaultRoutersSet (clientHandle, &poolNameBuf, &defaultRouters)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the default routers for DHCP hosts in a given pool. (result = %d)\n", result);
  }
  else
  {
    printf("\nConfigured the pool \"%s\" with default routers\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Get the default routers for DHCP hosts in a given pool.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsDefaultRoutersGet (openapiClientHandle_t *clientHandle,
                             char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  openDhcpsDefaultRouters_t defaultRouters;
  uint32_t i = 0;
  char ipStr[16];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  memset (&defaultRouters, 0, sizeof (defaultRouters));

  if ((result = openapiDhcpsDefaultRoutersGet (clientHandle, &poolNameBuf, &defaultRouters)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the default routers for DHCP hosts in a given pool. (result = %d)\n", result);
  }
  else
  {
    printf ("\nDefault routers configured for the pool \"%s\" :", poolNameStr);
    for (i = 0; i < OPEN_DHCPS_DEFAULT_ROUTER_MAX; i++)
    {
      if (0 == defaultRouters.routers[i])
      {
        continue;
      }
      defaultRouters.routers[i] = ntohl (defaultRouters.routers[i]);
      memset (ipStr, 0, sizeof (ipStr));
      inet_ntop (AF_INET, &defaultRouters.routers[i],
                 ipStr, 16);
      printf ("\n     %s", ipStr);
    }
    printf ("\n");
  }
  return;
}


/*********************************************************************
* @purpose  Reset the default routers for DHCP hosts in a given pool.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool

*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsDefaultRoutersIpReset (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsDefaultRoutersReset (clientHandle, &poolNameBuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to reset the default routers for DHCP hosts in a given pool. (result = %d)\n", result);
  }
  else
  {
    printf ("\nCleared the default routers configuration for the pool \"%s\"\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Set the DHCP Host network for a pool.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
* @param    nwAddr         @b{(input)}  Address pool to be assigned to the hosts
* @param    prefixlen      @b{(input)}  prefix length of the network
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsNetworkSet (openapiClientHandle_t *clientHandle,
                      char *poolName, char *nwAddr, uint32_t prefixLen)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  open_inet_pfx_t networkAddr;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  memset (str, 0, sizeof (str));
  strncpy (str, nwAddr, sizeof (str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset (&networkAddr, 0, sizeof (networkAddr));

  if ((result = openapiInetAddrGet (clientHandle, &ipBuffdesc, &(networkAddr.ipAddr))) != OPEN_E_NONE)
  {
    printf ("Bad return code trying to convert IP address. (result = %d)\n", result);
    return;
  }

  networkAddr.pfxLen = prefixLen;
  networkAddr.ipAddr.family = OPEN_AF_INET;

  if ((result = openapiDhcpsNetworkSet(clientHandle, &poolNameBuf, networkAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the DHCP Host network for a pool. (result = %d)\n", result);
  }
  else
  {
    printf ("\nNetwork configured for the pool \"%s\"\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Get the DHCP Host network for a pool.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsNetworkGet (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  open_inet_pfx_t networkAddr;
  char ipStr[16];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsNetworkGet(clientHandle, &poolNameBuf, &networkAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the DHCP Host network for a pool. (result = %d)\n", result);
  }
  else
  {
    networkAddr.ipAddr.addr.ipv4 = ntohl (networkAddr.ipAddr.addr.ipv4);
    memset (ipStr, 0, sizeof (ipStr));
    inet_ntop (AF_INET, &(networkAddr.ipAddr.addr.ipv4), ipStr, 16);
    printf ("\nNetwork configured for the pool \"%s\" : %s/%u\n",
            poolNameStr, ipStr, networkAddr.pfxLen);
  }
  return;
}


/*********************************************************************
* @purpose  Delete the DHCP Host network for a pool.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsNetworkDelete (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsNetworkDelete (clientHandle, &poolNameBuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete the DHCP Host network for a pool. (result = %d)\n", result);
  }
  else
  {
    printf ("\nCleared the network configuration for the pool \"%s\"\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Set the DNS servers for DHCP hosts in a given pool.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
* @param    dnsSrvs        @b{(input)}  List of DNS servers
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsDnsServersSet (openapiClientHandle_t *clientHandle, char *poolName, char *dnsSrvs)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  openDhcpsDnsServers_t dnsServers;
  char *str_ptr = NULL;
  char *ptr = NULL;
  uint32_t i = 0;

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  memset (&dnsServers, 0, sizeof (dnsServers));
  ptr = strtok_r (dnsSrvs, ",", &(str_ptr));
  while (NULL != ptr)
  {
    inet_pton (AF_INET, ptr, &dnsServers.dns_servers[i]);
    dnsServers.dns_servers[i] = htonl (dnsServers.dns_servers[i]);
    ptr = strtok_r (NULL, ",", &(str_ptr));
    i++;
  }

  if ((result = openapiDhcpsDnsServersSet(clientHandle, &poolNameBuf, &dnsServers)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the DNS servers for DHCP hosts in a given pool. (result = %d)\n", result);
  }
  else
  {
    printf("\nConfigured the pool \"%s\" with DNS servers\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Get the DNS servers for DHCP hosts in a given pool.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsDnsServersGet (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  openDhcpsDnsServers_t dnsServers;
  uint32_t i = 0;
  char ipStr[16];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsDnsServersGet(clientHandle, &poolNameBuf, &dnsServers)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the DNS servers for DHCP hosts in a given pool. (result = %d)\n", result);
  }
  else
  {
    printf ("\nDNS servers configured for the pool \"%s\" :", poolNameStr);
    for (i = 0; i < OPEN_DHCPS_DNS_SERVER_MAX; i++)
    {
      if (0 == dnsServers.dns_servers[i])
      {
        continue;
      }
      dnsServers.dns_servers[i] = ntohl (dnsServers.dns_servers[i]);
      memset (ipStr, 0, sizeof (ipStr));
      inet_ntop (AF_INET, &dnsServers.dns_servers[i],
                 ipStr, 16);
      printf ("\n     %s", ipStr);
    }
    printf ("\n");
  }
  return;
}


/*********************************************************************
* @purpose  Reset the DNS servers for DHCP hosts in a given pool.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsDnsServersReset (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsDnsServersReset(clientHandle, &poolNameBuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to reset the DNS servers for DHCP hosts in a given pool. (result = %d)\n", result);
  }
  else
  {
    printf ("\nCleared the DNS server configuration for the pool \"%s\"\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Set the Netbios name server addresses in given pool.
*
* @param    clientHandle     @b{(input)}  client handle from registration API
* @param    poolName         @b{(input)}  Name of the DHCP server address pool
* @param    netbiosNameSrvs  @b{(input)}  List of Netbios name server addresses
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsNetbiosNameServerAddressSet (openapiClientHandle_t *clientHandle,
                                       char *poolName, char *netbiosNameSrvs)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  openDhcpsNetbiosServers_t netBiosNameServers;
  char *str_ptr = NULL;
  char *ptr = NULL;
  uint32_t i = 0;

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  memset (&netBiosNameServers, 0, sizeof (netBiosNameServers));
  ptr = strtok_r (netbiosNameSrvs, ",", &(str_ptr));
  while (NULL != ptr)
  {
    inet_pton (AF_INET, ptr, &netBiosNameServers.servers[i]);
    netBiosNameServers.servers[i] = htonl (netBiosNameServers.servers[i]);
    ptr = strtok_r (NULL, ",", &(str_ptr));
    i++;
  }

  if ((result = openapiDhcpsNetbiosNameServerAddressSet (clientHandle, &poolNameBuf, &netBiosNameServers)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the Netbios name server addresses in given pool. (result = %d)\n", result);
  }
  else
  {
    printf("\nConfigured the pool \"%s\" with netbios name servers\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Get the Netbios name server addresses in given pool.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsNetbiosNameServerAddressGet (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  openDhcpsNetbiosServers_t netBiosNameServers;
  uint32_t i = 0;
  char ipStr[16];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsNetbiosNameServerAddressGet(clientHandle, &poolNameBuf, &netBiosNameServers)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Netbios name server addresses in given pool. (result = %d)\n", result);
  }
  else
  {
    printf ("\nNetbios name servers configured for the pool \"%s\" :", poolNameStr);
    for (i = 0; i < OPEN_DHCPS_NETBIOS_NAME_SERVER_MAX; i++)
    {
      if (0 == netBiosNameServers.servers[i])
      {
        continue;
      }
      netBiosNameServers.servers[i] = ntohl (netBiosNameServers.servers[i]);
      memset (ipStr, 0, sizeof (ipStr));
      inet_ntop (AF_INET, &netBiosNameServers.servers[i],
                 ipStr, 16);
      printf ("\n     %s", ipStr);
    }
    printf ("\n");
  }
  return;
}


/*********************************************************************
* @purpose  Delete the Netbios name server addresses in given pool.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool

*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsNetbiosNameServerAddressDelete(openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsNetbiosNameServerAddressDelete(clientHandle, &poolNameBuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete the Netbios name server addresses in given pool. (result = %d)\n", result);
  }
  else
  {
    printf ("\nCleared the netbios name server configuration for the pool \"%s\"\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Set the type of Netbios node.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
* @param    nodeType       @b{(input)}  Type of the Netbios node
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsNetbiosNodeTypeSet (openapiClientHandle_t *clientHandle,
                             char *poolName, uint32_t nodeType)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsNetbiosNodeTypeSet (clientHandle, &poolNameBuf,
                                                (OPEN_NETBIOS_NODE_TYPES_t)nodeType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the type of Netbios node. (result = %d)\n", result);
  }
  else
  {
    printf("\nConfigured the pool \"%s\" with node type %u\n", poolNameStr, nodeType);
  }
  return;
}


/*********************************************************************
* @purpose  Get the type of Netbios node.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsNetbiosNodeTypeGet (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  OPEN_NETBIOS_NODE_TYPES_t nodeType;

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsNetbiosNodeTypeGet(clientHandle, &poolNameBuf, &nodeType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the type of Netbios node. (result = %d)\n", result);
  }
  else
  {
    printf("\nThe node type of the pool \"%s\" is set to %u\n", poolNameStr, (uint32_t)nodeType);
  }
  return;
}


/*********************************************************************
* @purpose  Delete the type of Netbios node.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool

*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsNetbiosNodeTypeDelete(openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsNetbiosNodeTypeDelete(clientHandle, &poolNameBuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete the type of Netbios node. (result = %d)\n", result);
  }
  else
  {
    printf ("\nCleared the node type configuration for the pool \"%s\"\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Set the auto default router mode for a given DHCP Pool name.
*
* @param    clientHandle             @b{(input)} client handle from registration API
* @param    poolName                 @b{(input)} Name of the DHCP server address pool
* @param    isAutoGatewayModeEnabled @b{(input)} Lease time in minutes
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsPoolAutoDefaultRouterModeSet( openapiClientHandle_t *clientHandle,
                                        char *poolName, uint32_t isAutoGatewayModeEnabled)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  OPEN_BOOL_t isAutoDefRtrModeEnabled = OPEN_FALSE;


  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);
  isAutoDefRtrModeEnabled = (isAutoGatewayModeEnabled == 0)? OPEN_FALSE:OPEN_TRUE;

  if ((result = openapiDhcpsPoolAutoDefaultRouterModeSet(clientHandle, &poolNameBuf, isAutoDefRtrModeEnabled)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the DHCP pool auto-default-router mode. (result = %d)\n", result);
  }
  else
  {
    printf("\nConfigured the pool \"%s\" with the given auto-default-router mode.\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Get the auto default router mode for a given DHCP Pool name.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsPoolAutoDefaultRouterModeGet( openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  OPEN_BOOL_t isAutoDefRtrModeEnabled = OPEN_FALSE;


  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsPoolAutoDefaultRouterModeGet(clientHandle, &poolNameBuf, &isAutoDefRtrModeEnabled)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the DHCP pool auto-default-router mode. (result = %d)\n", result);
  }
  else
  {
    if (isAutoDefRtrModeEnabled == OPEN_FALSE)
    {
      printf("\nThe pool \"%s\" has the auto-default-router mode set to FALSE.\n", poolNameStr);
    }
    else
    {
      printf("\nThe pool \"%s\" has the auto-default-router mode set to TRUE.\n", poolNameStr);
    }
  }
  return;
}


/*********************************************************************
* @purpose  Set the DHCP lease time.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
* @param    leaseTime      @b{(input)}  Lease time in minutes
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsLeaseTimeSet (openapiClientHandle_t *clientHandle,
                        char *poolName, uint32_t leaseTime)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsLeaseTimeSet (clientHandle, &poolNameBuf, leaseTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the DHCP lease time. (result = %d)\n", result);
  }
  else
  {
    printf("\nConfigured the pool \"%s\" with lease time %u\n", poolNameStr, leaseTime);
  }
  return;
}


/*********************************************************************
* @purpose  Get the DHCP lease time.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsLeaseTimeGet (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  uint32_t leaseTime;

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsLeaseTimeGet(clientHandle, &poolNameBuf, &leaseTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the DHCP lease time. (result = %d)\n", result);
  }
  else
  {
    printf("\nLease time for the pool \"%s\" is %u\n", poolNameStr, leaseTime);
  }
  return;
}


/*********************************************************************
* @purpose  Reset the DHCP lease time.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool

*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsLeaseTimeReset(openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsLeaseTimeReset(clientHandle, &poolNameBuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to reset the DHCP lease time. (result = %d)\n", result);
  }
  else
  {
    printf ("\nCleared the lease time configuration for the pool \"%s\"\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Set the domain name of a DHCP client.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
* @param    domainName     @b{(input)}  Domain name of the DHCP client
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsDomainNameSet (openapiClientHandle_t *clientHandle, char *poolName, char *domainName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  open_buffdesc domainNameBuf;
  char domainNameStr[OPEN_DHCPS_DOMAIN_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  memset (domainNameStr, 0, sizeof (domainNameStr));
  memcpy (domainNameStr, domainName, strlen (domainName));
  domainNameBuf.pstart = domainNameStr;
  domainNameBuf.size = sizeof (domainNameStr);

  if ((result = openapiDhcpsDomainNameSet(clientHandle, &poolNameBuf, &domainNameBuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the domain name of a DHCP client. (result = %d)\n", result);
  }
  else
  {
    printf("\nConfigured the pool \"%s\" with domain name %s\n", poolNameStr, domainNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Get the domain name of a DHCP client.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsDomainNameGet (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  open_buffdesc domainNameBuf;
  char domainNameStr[OPEN_DHCPS_DOMAIN_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  memset (domainNameStr, 0, sizeof (domainNameStr));
  domainNameBuf.pstart = domainNameStr;
  domainNameBuf.size = sizeof (domainNameStr);

  if ((result = openapiDhcpsDomainNameGet(clientHandle, &poolNameBuf, &domainNameBuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the domain name of a DHCP client. (result = %d)\n", result);
  }
  else
  {
    printf("\nDomain name for the pool \"%s\" is %s\n", poolNameStr, domainNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Delete the domain name of a DHCP client.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool

*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsDomainNameDelete(openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsDomainNameDelete(clientHandle, &poolNameBuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete the domain name of a DHCP client. (result = %d)\n", result);
  }
  else
  {
    printf ("\nCleared the domain name configuration for the pool \"%s\"\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Set the next server IP address.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
* @param    nextAddress    @b{(input)}  Next server address
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsNextServerSet (openapiClientHandle_t *clientHandle,
                         char *poolName, char *nextAddress)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  open_inet_addr_t nextServerAddr;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  memset (str, 0, sizeof (str));
  strncpy (str, nextAddress, sizeof (str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset (&nextServerAddr, 0, sizeof (nextServerAddr));

  if ((result = openapiInetAddrGet (clientHandle, &ipBuffdesc, &nextServerAddr)) != OPEN_E_NONE)
  {
    printf ("Bad return code trying to convert IP address. (result = %d)\n", result);
    return;
  }

  if ((result = openapiDhcpsNextServerSet (clientHandle, &poolNameBuf, nextServerAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the next server IP address. (result = %d)\n", result);
  }
  else
  {
    printf ("\nNext server address configured for the pool \"%s\"\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Get the next server IP address.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsNextServerGet (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  open_inet_addr_t nextServerAddr;
  char ipStr[16];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsNextServerGet(clientHandle, &poolNameBuf, &nextServerAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the next server IP address. (result = %d)\n", result);
  }
  else
  {
    nextServerAddr.addr.ipv4 = ntohl (nextServerAddr.addr.ipv4);
    memset (ipStr, 0, sizeof (ipStr));
    inet_ntop (AF_INET, &(nextServerAddr.addr.ipv4), ipStr, 16);
    printf ("\nNext server configured for the pool \"%s\" : %s\n",
            poolNameStr, ipStr);
  }
  return;
}


/*********************************************************************
* @purpose  Delete the next server IP address.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsNextServerDelete (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsNextServerDelete(clientHandle, &poolNameBuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete the next server IP address. (result = %d)\n", result);
  }
  else
  {
    printf ("\nCleared the next server configuration for the pool \"%s\"\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Set the DHCP host hardware address for a manual pool.
*
* @param    clientHandle  @b{(input)}  client handle from registration API
* @param    poolName      @b{(input)}  Name of the DHCP server address pool
* @param    hwAddr        @b{(input)}  Hardware address
* @param    hwType        @b{(input)}  Type of hardware address

*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsHwAddressSet (openapiClientHandle_t *clientHandle, char *poolName,
                        char *hwAddr, uint32_t hwType)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  open_buffdesc hwAddrBuf;
  char hwAddrStr[OPEN_MAC_ADDR_STR_LEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  memset (hwAddrStr, 0, sizeof (hwAddrStr));
  memcpy (hwAddrStr, hwAddr, strlen (hwAddr));
  hwAddrBuf.pstart = hwAddrStr;
  hwAddrBuf.size = sizeof (hwAddrStr);

  if ((result = openapiDhcpsHwAddressSet (clientHandle, &poolNameBuf, &hwAddrBuf,
                                          (OPEN_DHCPS_HW_TYPES_t)hwType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the DHCP host hardware address for a manual pool. (result = %d)\n", result);
  }
  else
  {
    printf ("\nClient Hardware address configured for the manual pool \"%s\"\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Get the DHCP host hardware address for a manual pool.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsHwAddressGet (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  open_buffdesc hwAddrBuf;
  char hwAddrStr[OPEN_MAC_ADDR_STR_LEN];
  OPEN_DHCPS_HW_TYPES_t hwType;

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  memset (hwAddrStr, 0, sizeof (hwAddrStr));
  hwAddrBuf.pstart = hwAddrStr;
  hwAddrBuf.size = sizeof (hwAddrStr);


  if ((result = openapiDhcpsHwAddressGet (clientHandle, &poolNameBuf, &hwAddrBuf, &hwType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the DHCP host hardware address for a manual pool. (result = %d)\n", result);
  }
  else
  {
    printf ("\nClient Hardware address configured for the manual pool \"%s\" is %s and type is %u\n",
            poolNameStr, hwAddrStr, hwType);
  }
  return;
}


/*********************************************************************
* @purpose  Delete the DHCP host hardware address for a manual pool.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsHwAddressDelete (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsHwAddressDelete(clientHandle, &poolNameBuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete the DHCP host hardware address for a manual pool. (result = %d)\n", result);
  }
  else
  {
    printf ("\nCleared the hardware address configuration for the pool \"%s\"\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Set the DHCP Host network for a manual pool.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
* @param    ipAddr         @b{(input)}  Address to be assigned to the host
* @param    prefixlen      @b{(input)}  prefix length of the network
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsHostSet (openapiClientHandle_t *clientHandle, char *poolName,
                   char *ipAddr, uint32_t prefixLen)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  open_inet_pfx_t hostAddr;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  memset (str, 0, sizeof (str));
  strncpy (str, ipAddr, sizeof (str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset (&hostAddr, 0, sizeof (hostAddr));

  if ((result = openapiInetAddrGet (clientHandle, &ipBuffdesc, &(hostAddr.ipAddr))) != OPEN_E_NONE)
  {
    printf ("Bad return code trying to convert IP address. (result = %d)\n", result);
    return;
  }

  hostAddr.pfxLen = prefixLen;
  hostAddr.ipAddr.family = OPEN_AF_INET;

  if ((result = openapiDhcpsHostSet(clientHandle, &poolNameBuf, hostAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the DHCP Host network for a manual pool. (result = %d)\n", result);
  }
  else
  {
    printf ("\nHost address configured for the manual pool \"%s\"\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Get the DHCP Host network for a manual pool.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsHostGet (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  open_inet_pfx_t hostAddr;
  char ipStr[16];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsHostGet(clientHandle, &poolNameBuf, &hostAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the DHCP Host network for a manual pool. (result = %d)\n", result);
  }
  else
  {
    hostAddr.ipAddr.addr.ipv4 = ntohl (hostAddr.ipAddr.addr.ipv4);
    memset (ipStr, 0, sizeof (ipStr));
    inet_ntop (AF_INET, &(hostAddr.ipAddr.addr.ipv4), ipStr, 16);
    printf ("\nNetwork configured for the manual pool \"%s\" : %s/%u\n",
            poolNameStr, ipStr, hostAddr.pfxLen);
  }
  return;
}


/*********************************************************************
* @purpose  Delete the DHCP Host address and mask for a pool.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsHostDelete (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsHostDelete(clientHandle, &poolNameBuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete the DHCP Host address and mask for a pool. (result = %d)\n", result);
  }
  else
  {
    printf ("\nCleared the host configuration for the pool \"%s\"\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Set the boot file name.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
* @param    fileName       @b{(input)}  Boot file name for Bootp client

*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsBootfileSet (openapiClientHandle_t *clientHandle, char *poolName, char *fileName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  open_buffdesc fileNameBuf;
  char fileNameStr[OPEN_DHCPS_BOOT_FILE_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  memset (fileNameStr, 0, sizeof (fileNameStr));
  memcpy (fileNameStr, fileName, strlen (fileName));
  fileNameBuf.pstart = fileNameStr;
  fileNameBuf.size = sizeof (fileNameStr);

  if ((result = openapiDhcpsBootfileSet(clientHandle, &poolNameBuf, &fileNameBuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the boot file name. (result = %d)\n", result);
  }
  else
  {
    printf("\nConfigured the pool \"%s\" with bootp file name %s\n", poolNameStr, fileNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Get the boot file name.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsBootfileGet (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  open_buffdesc fileNameBuf;
  char fileNameStr[OPEN_DHCPS_BOOT_FILE_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  memset (fileNameStr, 0, sizeof (fileNameStr));
  fileNameBuf.pstart = fileNameStr;
  fileNameBuf.size = sizeof (fileNameStr);

  if ((result = openapiDhcpsBootfileGet (clientHandle, &poolNameBuf, &fileNameBuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the boot file name. (result = %d)\n", result);
  }
  else
  {
    printf("\nBootp filename configured for the pool \"%s\" is %s\n", poolNameStr, fileNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Delete the boot file name.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsBootfileDelete (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsBootfileDelete(clientHandle, &poolNameBuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete the boot file name. (result = %d)\n", result);
  }
  else
  {
    printf ("\nCleared the bootp file name configuration for the pool \"%s\"\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Set the NTP server addresses for a given pool.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
* @param    ntpServers     @b{(input)}  List of NTP server addresses

*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsNTPServerSet (openapiClientHandle_t *clientHandle, char *poolName, char *ntpSrvs)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  openDhcpsNtpServers_t ntpServers;
  char *str_ptr = NULL;
  char *ptr = NULL;
  uint32_t i = 0;

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  memset (&ntpServers, 0, sizeof (ntpServers));
  ptr = strtok_r (ntpSrvs, ",", &(str_ptr));
  while (NULL != ptr)
  {
    inet_pton (AF_INET, ptr, &ntpServers.ntp_servers[i]);
    ntpServers.ntp_servers[i] = htonl (ntpServers.ntp_servers[i]);
    ptr = strtok_r (NULL, ",", &(str_ptr));
    i++;
  }

  if ((result = openapiDhcpsNTPServerSet (clientHandle, &poolNameBuf, &ntpServers)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the NTP server addresses for a given pool. (result = %d)\n", result);
  }
  else
  {
    printf("\nConfigured the pool \"%s\" with ntp servers\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Get the NTP server addresses for a given pool.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsNTPServerGet (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  openDhcpsNtpServers_t ntpServers;
  uint32_t i = 0;
  char ipStr[16];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsNTPServerGet (clientHandle, &poolNameBuf, &ntpServers)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the NTP server addresses for a given pool. (result = %d)\n", result);
  }
  else
  {
    printf ("\nNTP servers configured for the pool \"%s\" :", poolNameStr);
    for (i = 0; i < OPEN_DHCPS_NTP_SERVER_MAX; i++)
    {
      if (0 == ntpServers.ntp_servers[i])
      {
        continue;
      }
      ntpServers.ntp_servers[i] = ntohl (ntpServers.ntp_servers[i]);
      memset (ipStr, 0, sizeof (ipStr));
      inet_ntop (AF_INET, &ntpServers.ntp_servers[i],
                 ipStr, 16);
      printf ("\n     %s", ipStr);
    }
    printf ("\n");
  }
  return;
}


/*********************************************************************
* @purpose  Delete the NTP server addresses for a given pool.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsNTPServerDelete (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsNTPServerDelete(clientHandle, &poolNameBuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete the NTP server addresses for a given pool. (result = %d)\n", result);
  }
  else
  {
    printf ("\nCleared the ntp server configuration for the pool \"%s\"\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Clear the DHCP Server statistics.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsClearStatistics (openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  if ((result = openapiDhcpsStatisticsClear (clientHandle)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete the NTP server addresses for a given pool. (result = %d)\n", result);
  }
  else
  {
    printf ("\nCleared the DHCP server statistics.\n");
  }
  return;
}


/*********************************************************************
* @purpose  Set the client id of a DHCP client.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
* @param    clientId       @b{(input)}  Client id of the DHCP client
*                                        in xx:xx:xx:xx:xx:xx... format
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsClientIdSet (openapiClientHandle_t *clientHandle, char *poolName, char *clientId)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  open_buffdesc clientIdBuf;
  char clientIdStr[OPEN_DHCP_SERVER_CLIENT_ID_STR_MAXLEN];
  OPEN_DHCP_SERVER_CLIENT_ID_t clientIdArray;
  uint32_t clientIdLen = 0;

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  memset (clientIdStr, 0, sizeof (clientIdStr));
  memcpy (clientIdStr, clientId, strlen (clientId));
  clientIdBuf.pstart = clientIdStr;
  clientIdBuf.size = strnlen (clientIdStr, OPEN_DHCP_SERVER_CLIENT_ID_STR_MAXLEN) + 1;

  if ((result = openapiConvertClientId (clientHandle, &clientIdBuf, &clientIdArray, &clientIdLen)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert the client id. (result = %d)\n", result);
  }

  if ((result = openapiDhcpsClientIdSet (clientHandle, &poolNameBuf,
                                         &clientIdArray, clientIdLen)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the client id of a DHCP client. (result = %d)\n", result);
  }
  else
  {
    printf("\nConfigured the pool \"%s\" with client id %s\n", poolNameStr, clientIdStr);
  }
  return;
}


/*********************************************************************
* @purpose  Get the client Id of a DHCP client.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsClientIdGet (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  OPEN_DHCP_SERVER_CLIENT_ID_t clientIdArray;
  uint32_t clientIdLen = 0;
  uint32_t i = 0;

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsClientIdGet (clientHandle, &poolNameBuf,
                                         &clientIdArray, &clientIdLen)) != OPEN_E_NONE)
  {
    printf ("Bad return code trying to get the client id. (result = %d)\n", result);
  }
  else
  {

    printf ("\nClient id for the pool \"%s\" is : ", poolNameStr);
    for (i = 0; i < clientIdLen; i++)
    {
      if (i == 0)
      {
        printf ("%02x", (uint8_t)clientIdArray.clientid[i]);
      }
      else
      {
        printf (":%02x", (uint8_t)clientIdArray.clientid[i]);
      }
    }
    printf ("\n");
  }
  return;
}


/*********************************************************************
* @purpose  Delete the client name of a DHCP client.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool

*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsClientIdDelete(openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsClientIdDelete(clientHandle, &poolNameBuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete the client name of a DHCP client. (result = %d)\n", result);
  }
  else
  {
    printf ("\nCleared the client id configuration for the pool \"%s\"\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Set the client name of a DHCP client.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
* @param    clientName     @b{(input)}  Client name of the DHCP client
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsClientNameSet (openapiClientHandle_t *clientHandle, char *poolName, char *clientName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  open_buffdesc clientNameBuf;
  char clientNameStr[OPEN_DHCP_SERVER_CLIENT_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  memset (clientNameStr, 0, sizeof (clientNameStr));
  memcpy (clientNameStr, clientName, strlen (clientName));
  clientNameBuf.pstart = clientNameStr;
  clientNameBuf.size = sizeof (clientNameStr);

  if ((result = openapiDhcpsClientNameSet(clientHandle, &poolNameBuf, &clientNameBuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the client name of a DHCP client. (result = %d)\n", result);
  }
  else
  {
    printf("\nConfigured the pool \"%s\" with client name %s\n", poolNameStr, clientNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Get the client name of a DHCP client.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsClientNameGet (openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];
  open_buffdesc clientNameBuf;
  char clientNameStr[OPEN_DHCP_SERVER_CLIENT_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  memset (clientNameStr, 0, sizeof (clientNameStr));
  clientNameBuf.pstart = clientNameStr;
  clientNameBuf.size = sizeof (clientNameStr);

  if ((result = openapiDhcpsClientNameGet(clientHandle, &poolNameBuf, &clientNameBuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the client name of a DHCP client. (result = %d)\n", result);
  }
  else
  {
    printf("\nClient name for the pool \"%s\" is %s\n", poolNameStr, clientNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Delete the client name of a DHCP client.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    poolName       @b{(input)}  Name of the DHCP server address pool

*
* @returns  none
*
* @end
*********************************************************************/
void dhcpsClientNameDelete(openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  open_buffdesc poolNameBuf;
  char poolNameStr[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];

  memset (poolNameStr, 0, sizeof (poolNameStr));
  memcpy (poolNameStr, poolName, strlen (poolName));
  poolNameBuf.pstart = poolNameStr;
  poolNameBuf.size = sizeof (poolNameStr);

  if ((result = openapiDhcpsClientNameDelete(clientHandle, &poolNameBuf)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete the client name of a DHCP client. (result = %d)\n", result);
  }
  else
  {
    printf ("\nCleared the client name configuration for the pool \"%s\"\n", poolNameStr);
  }
  return;
}


/*********************************************************************
* @purpose  Perform Sanity of DHCP Server OpEN APIs.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*
* @notes
*   
* @end
*********************************************************************/
void dhcpServerOpENAPIsSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_inet_addr_t ipaddr;
  uint32_t mask;
  uint32_t poolIndex;
  uint32_t leaseIndex;
  uint32_t val1;
  open_buffdesc hwaddr;
  OPEN_DHCP_SERVER_CLIENT_ID_t clientId;
  OPEN_DHCP_SERVER_POOL_NAME_t poolName;
  uint32_t remaining;
  char buf[OPEN_MAC_ADDR_LEN];
  OPEN_DHCP_SERVER_LEASE_STATE_t state;
  OPEN_DHCP_SERVER_POOL_TYPE_t type;

  hwaddr.pstart = buf;
  hwaddr.size = sizeof(buf);

  /* openapiDhcpServerLeaseDataGet */
  printf("\nTesting openapiDhcpServerLeaseDataGet(): \n");

  result = openapiDhcpServerLeaseDataGet(NULL, &ipaddr, &mask, &hwaddr, &clientId, &poolName, &remaining);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpServerLeaseDataGet(clientHandle, NULL, &mask, &hwaddr, &clientId, &poolName, &remaining);
  printf("NULL ipaddr. (result = %d)\n", result);
  result = openapiDhcpServerLeaseDataGet(clientHandle, &ipaddr, NULL, &hwaddr, &clientId, &poolName, &remaining);
  printf("NULL mask. (result = %d)\n", result);
  result = openapiDhcpServerLeaseDataGet(clientHandle, &ipaddr, &mask, NULL, &clientId, &poolName, &remaining);
  printf("NULL hwaddr. (result = %d)\n", result);
  result = openapiDhcpServerLeaseDataGet(clientHandle, &ipaddr, &mask, &hwaddr, NULL, &poolName, &remaining);
  printf("NULL client ID. (result = %d)\n", result);
  result = openapiDhcpServerLeaseDataGet(clientHandle, &ipaddr, &mask, &hwaddr, &clientId, NULL, &remaining);
  printf("NULL pool name pointer. (result = %d)\n", result);
  result = openapiDhcpServerLeaseDataGet(clientHandle, &ipaddr, &mask, &hwaddr, &clientId, &poolName, NULL);
  printf("NULL remaining pointer. (result = %d)\n", result);
  hwaddr.pstart = buf;
  hwaddr.size = 1;
  result = openapiDhcpServerLeaseDataGet(clientHandle, &ipaddr, &mask, &hwaddr, &clientId, &poolName, &remaining);
  printf("hwaddr size too small. (result = %d)\n", result);
  hwaddr.pstart = NULL;
  hwaddr.size = sizeof(buf);
  result = openapiDhcpServerLeaseDataGet(clientHandle, &ipaddr, &mask, &hwaddr, &clientId, &poolName, &remaining);
  printf("NULL hwaddr pstart. (result = %d)\n", result);

  printf("openapiDhcpServerLeaseDataGet() sanity successful. \n");

  /* openapiDhcpServerPoolEntryFirstGet */
  printf("\nTesting openapiDhcpServerPoolEntryFirstGet(): \n");

  result = openapiDhcpServerPoolEntryFirstGet(NULL, &poolIndex, &poolName);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpServerPoolEntryFirstGet(clientHandle, NULL, &poolName);
  printf("NULL pool index. (result = %d)\n", result);
  result = openapiDhcpServerPoolEntryFirstGet(clientHandle, &poolIndex, NULL);
  printf("NULL pool name. (result = %d)\n", result);

  printf("openapiDhcpServerPoolEntryFirstGet() sanity successful. \n");

  /* openapiDhcpServerPoolEntryNextGet */
  printf("\nTesting openapiDhcpServerPoolEntryNextGet(): \n");

  result = openapiDhcpServerPoolEntryNextGet(NULL, &poolIndex, &poolName);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpServerPoolEntryNextGet(clientHandle, NULL, &poolName);
  printf("NULL pool index. (result = %d)\n", result);
  result = openapiDhcpServerPoolEntryNextGet(clientHandle, &poolIndex, NULL);
  printf("NULL pool name. (result = %d)\n", result);

  printf("openapiDhcpServerPoolEntryNextGet() sanity successful. \n");

  /* openapiDhcpServerPoolTypeGet */
  printf("\nTesting openapiDhcpServerPoolTypeGet(): \n");

  result = openapiDhcpServerPoolTypeGet(NULL, &poolName, &type);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpServerPoolTypeGet(clientHandle, NULL, &type);
  printf("NULL pool name. (result = %d)\n", result);
  result = openapiDhcpServerPoolTypeGet(clientHandle, &poolName, NULL);
  printf("NULL type. (result = %d)\n", result);

  printf("openapiDhcpServerPoolTypeGet() sanity successful. \n");

  /* openapiDhcpServerLeaseEntryFirstGet */
  printf("\nTesting openapiDhcpServerLeaseEntryFirstGet(): \n");

  result = openapiDhcpServerLeaseEntryFirstGet(NULL, &poolName, &leaseIndex, &ipaddr, &state);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpServerLeaseEntryFirstGet(clientHandle, NULL, &leaseIndex, &ipaddr, &state);
  printf("NULL pool name. (result = %d)\n", result);
  result = openapiDhcpServerLeaseEntryFirstGet(clientHandle, &poolName, NULL, &ipaddr, &state);
  printf("NULL lease index. (result = %d)\n", result);
  result = openapiDhcpServerLeaseEntryFirstGet(clientHandle, &poolName, &leaseIndex, NULL, &state);
  printf("NULL IP addr. (result = %d)\n", result);
  result = openapiDhcpServerLeaseEntryFirstGet(clientHandle, &poolName, &leaseIndex, &ipaddr, NULL);
  printf("NULL state. (result = %d)\n", result);

  printf("openapiDhcpServerLeaseEntryFirstGet() sanity successful. \n");

  /* openapiDhcpServerLeaseEntryNextGet */
  printf("\nTesting openapiDhcpServerLeaseEntryNextGet(): \n");

  result = openapiDhcpServerLeaseEntryNextGet(NULL, &poolName, &leaseIndex, &ipaddr, &state);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpServerLeaseEntryNextGet(clientHandle, NULL, &leaseIndex, &ipaddr, &state);
  printf("NULL pool name. (result = %d)\n", result);
  result = openapiDhcpServerLeaseEntryNextGet(clientHandle, &poolName, NULL, &ipaddr, &state);
  printf("NULL lease index. (result = %d)\n", result);
  result = openapiDhcpServerLeaseEntryNextGet(clientHandle, &poolName, &leaseIndex, NULL, &state);
  printf("NULL IP addr. (result = %d)\n", result);
  result = openapiDhcpServerLeaseEntryNextGet(clientHandle, &poolName, &leaseIndex, &ipaddr, NULL);
  printf("NULL state. (result = %d)\n", result);

  printf("openapiDhcpServerLeaseEntryNextGet() sanity successful. \n");

  /* openapiDhcpServerStatGet */
  printf("\nTesting openapiDhcpServerStatGet(): \n");

  result = openapiDhcpServerStatGet(NULL, 1, &val1);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpServerStatGet(clientHandle, 0, &val1);
  printf("Invalid statistic ID (too low). (result = %d)\n", result);
  result = openapiDhcpServerStatGet(clientHandle, 9999, &val1);
  printf("Invalid statistic ID (too high). (result = %d)\n", result);
  result = openapiDhcpServerStatGet(clientHandle, 1, NULL);
  printf("NULL statistic output value pointer. (result = %d)\n", result);

  printf("openapiDhcpServerStatGet() sanity successful. \n");
}


/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for DHCP server
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  
  if (argc < 2)
  {
    printAppMenu(argv[0]);
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister(argv[0], &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting DHCP Server API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS Version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 3) 
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      getDhcpServerLeaseData(&clientHandle, argv[2]);
      break;
    case 2:
      if (argc != 4) 
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      getDhcpServerVrfIpAddrLeaseData(&clientHandle, argv[2], argv[3]);
      break;
    case 3:
      if (argc != 2)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      getDhcpServerPoolEntries(&clientHandle);
      break;
    case 4:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      getDhcpServerLeaseEntries(&clientHandle, argv[2]);
      break;
    case 5:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      getDhcpServerPoolTypeForName(&clientHandle, argv[2]);
      break;
    case 6:
      if (argc != 2)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      displayDhcpServerStats(&clientHandle);
      break;
    case 7:
      if (argc != 2)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpServerOpENAPIsSanity(&clientHandle);
      break;
    case 8:
      if (argc != 5) 
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      addDhcpServerExcludedAddressRangeForVrf(&clientHandle, argv[2], argv[3], argv[4]);
      break;
    case 9:
      if (argc != 5) 
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      deleteDhcpServerExcludedAddressRangeForVrf(&clientHandle, argv[2], argv[3], argv[4]);
      break;
    case 10:
      if (argc != 4) 
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      setDhcpServerPoolVrfName(&clientHandle, argv[2], argv[3]);
      break;
    case 11:
      if (argc != 3) 
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      resetDhcpServerPoolVrfNameToDefaultVrf(&clientHandle, argv[2]);
      break;
    case 12:
      if (argc != 2)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      getDhcpServerExcludedAddressesRangeEntries(&clientHandle);
      break;
     case 13:
      if (argc != 3) 
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      clearDhcpServerVrfLeaseData(&clientHandle, argv[2]);
      break;
    case 14:
      if (argc != 4) 
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      clearDhcpServerVrfIpAddrLeaseData(&clientHandle, argv[2], argv[3]);
      break;
      case 15:
      if (argc != 3) 
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      clearDhcpServerPoolLeaseData(&clientHandle, argv[2]);
      break;
    case 16:
      if (argc != 4) 
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      clearDhcpServerPoolIpAddrLeaseData(&clientHandle, argv[2], argv[3]);
      break;
     case 17:
      if (argc != 2)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      clearDhcpServerAllLeaseData(&clientHandle);
      break;
    case 18:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsAdminModeSet (&clientHandle, atoi(argv[2]));
      break;
    case 19:
      if (argc != 2)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsAdminModeGet (&clientHandle);
      break;
    case 20:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsBootpAutomaticSet (&clientHandle, atoi(argv[2]));
      break;
    case 21:
      if (argc != 2)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsBootpAutomaticGet (&clientHandle);
      break;
    case 22:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsNoOfPingPktSet (&clientHandle, atoi(argv[2]));
      break;
    case 23:
      if (argc != 2)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsNoOfPingPktGet (&clientHandle);
      break;
    case 24:
      if (argc != 2)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsNoOfPingPktReset (&clientHandle);
      break;
    case 25:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsPoolCreate (&clientHandle, argv[2]);
      break;
    case 26:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsPoolDelete (&clientHandle, argv[2]);
      break;
    case 27:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsDefaultRoutersSet (&clientHandle, argv[2], argv[3]);
      break;
    case 28:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsDefaultRoutersGet (&clientHandle, argv[2]);
      break;
    case 29:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsDefaultRoutersIpReset (&clientHandle, argv[2]); 
      break;
    case 30:
      if (argc != 5)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsNetworkSet (&clientHandle, argv[2], argv[3], atoi(argv[4]));
      break;
    case 31:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsNetworkGet (&clientHandle, argv[2]);
      break;
    case 32:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsNetworkDelete (&clientHandle, argv[2]);
      break;
    case 33:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsDnsServersSet (&clientHandle, argv[2], argv[3]);
      break;
    case 34:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsDnsServersGet (&clientHandle, argv[2]);
      break;
    case 35:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsDnsServersReset (&clientHandle, argv[2]);
      break;
    case 36:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsNetbiosNameServerAddressSet (&clientHandle, argv[2], argv[3]);
      break;
    case 37:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsNetbiosNameServerAddressGet (&clientHandle, argv[2]);
      break;
    case 38:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsNetbiosNameServerAddressDelete (&clientHandle, argv[2]);
      break;
    case 39:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsNetbiosNodeTypeSet (&clientHandle, argv[2], atoi (argv[3]));
      break;
    case 40:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsNetbiosNodeTypeGet (&clientHandle, argv[2]);
      break;
    case 41:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsNetbiosNodeTypeDelete (&clientHandle, argv[2]);
      break;
    case 42:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsLeaseTimeSet (&clientHandle, argv[2], atoi (argv[3]));
      break;
    case 43:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsLeaseTimeGet (&clientHandle, argv[2]);
      break;
    case 44:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsLeaseTimeReset (&clientHandle, argv[2]);
      break;
    case 45:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsDomainNameSet (&clientHandle, argv[2], argv[3]);
      break;
    case 46:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsDomainNameGet (&clientHandle, argv[2]);
      break;
    case 47:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsDomainNameDelete (&clientHandle, argv[2]);
      break;
    case 48:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsNextServerSet (&clientHandle, argv[2], argv[3]);
      break;
    case 49:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsNextServerGet (&clientHandle, argv[2]);
      break;
    case 50:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsNextServerDelete (&clientHandle, argv[2]);
      break;
    case 51:
      if (argc != 5)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsHwAddressSet (&clientHandle, argv[2], argv[3], atoi(argv[4]));
      break;
    case 52:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsHwAddressGet (&clientHandle, argv[2]);
      break;
    case 53:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsHwAddressDelete (&clientHandle, argv[2]);
      break;
    case 54:
      if (argc != 5)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsHostSet (&clientHandle, argv[2], argv[3], atoi(argv[4]));
      break;
    case 55:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsHostGet (&clientHandle, argv[2]);
      break;
    case 56:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsHostDelete (&clientHandle, argv[2]);
      break;
    case 57:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsBootfileSet (&clientHandle, argv[2], argv[3]);
      break;
    case 58:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsBootfileGet (&clientHandle, argv[2]);
      break;
    case 59:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsBootfileDelete (&clientHandle, argv[2]);
      break;
    case 60:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsNTPServerSet (&clientHandle, argv[2], argv[3]);
      break;
    case 61:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsNTPServerGet (&clientHandle, argv[2]);
      break;
    case 62:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsNTPServerDelete (&clientHandle, argv[2]);
      break;
    case 63:
      if (argc != 2)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsClearStatistics (&clientHandle);
      break;
    case 64:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsClientIdSet (&clientHandle, argv[2], argv[3]);
      break;
    case 65:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsClientIdGet (&clientHandle, argv[2]);
      break;
    case 66:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsClientIdDelete (&clientHandle, argv[2]);
      break;
    case 67:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsClientNameSet (&clientHandle, argv[2], argv[3]);
      break;
    case 68:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsClientNameGet (&clientHandle, argv[2]);
      break;
    case 69:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsClientNameDelete (&clientHandle, argv[2]);
      break;
    case 70:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsPoolAutoDefaultRouterModeSet (&clientHandle, argv[2], atoi(argv[3]));
      break;
    case 71:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpsPoolAutoDefaultRouterModeGet (&clientHandle, argv[2]);
      break;

    default:
      printAppMenu(argv[0]);
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping DHCP Server API example application");
        
  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
