/*********************************************************************
*
*  Copyright 2020-2024 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  dot1as_example.c
*
* @purpose   OpEN dot1as example.
*
* @component OpEN
*
* @create    04/09/2020
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_dot1as.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the dot1as Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Get the 802.1AS global admin mode: %s 1 \n", name);
  printf("Test 2: Set the 802.1AS global admin mode: %s 2 <adminMode(0-disable,1-enable)>\n", name);
  printf("Test 3: Check if there is a Grand Master elected in the 802.1AS domain: %s 3 \n", name);
  printf("Test 4: Check if this bridge is capable of being a grandmaster: %s 4 \n", name);
  printf("Test 5: Get the Best clock identity as detected by 802.1AS protocol: %s 5 \n", name);
  printf("Test 6: Get the priority1/2 value of Best clock in the 802.1AS domain: %s 6 <clockPriorityType(0-priority 1,1- priority 2)>\n", name);
  printf("Test 7: Get the steps to Best clock as detected by 802.1AS protocol: %s 7 \n", name);
  printf("Test 8: Get the local clock identity of this bridge: %s 8 \n",name);
  printf("Test 9: Get the priority1/2 value of the local clock: %s 9 <clockPriorityType(0-priority 1,1- priority 2)>\n", name);
  printf("Test 10: Set the priority1/2 value of the local clock: %s 10 <clockPriorityType(1- priority 2)> <clockPriority> (priority 1 cannot be set by user)\n", name);
  printf("Test 11: Get the number of times the grandmaster has changed: %s 11\n", name);
  printf("Test 12: Get the timestamp of the given event: %s 12 <gmEventType(0-GM change event,1-GM phase change,2-GM frequency change)>\n", name);
  printf("Test 13: Get the 802.1AS admin mode for the given interface: %s 13 <intfNum>\n", name);
  printf("Test 14: Set the 802.1AS admin mode for the given interface: %s 14 <intfNum> <adminMode(0-disable,1-enable)>\n", name);
  printf("Test 15: Check if the given interface is 802.1AS capable (as determined by the peer delay mechanism: %s 15 <intfNum>\n", name);
  printf("Test 16: Get the peer delay measuring status, which is set to TRUE if the given interface is receiving PDELAY response messages from other end of the link: %s 16 <intfNum>\n", name);
  printf("Test 17: Get the propagation delay on the given interface: %s 17 <intfNum>\n", name);
  printf("Test 18: Get the 802.1AS role of the given interface: %s 18 <intfNum>\n", name);
  printf("Test 19: Get the propagation delay threshold on the given interface: %s 19 <intfNum>\n", name);
  printf("Test 20: Set the propagation delay threshold on the given interface: %s 20 <intfNum> <threshold>\n", name);
  printf("Test 21: Get the number of lost peer delay request messages allowed.: %s 21 <intfNum>\n", name);
  printf("Test 22: Set the number of lost peer delay request messages allowed: %s 22 <intfNum> <numAllowedLostResp>\n", name);
  printf("Test 23: Get the neighbor rate ratio on the given interface (as computed by the pdelay mechanism) (extensible version): %s 23 <intfNum>\n", name);
  printf("Test 24: Get the given interval on the given interface: %s 24 <intfNum> <intervalType>\n", name);
  printf("Test 25: Set the given interval on the given interface: %s 25 <intfNum> <intervalType> <interval>\n", name);
  printf("Test 26: Get the requested 802.1AS counter for the given interface: %s 26 <intfNum> <counterType>\n", name);
  printf("Test 27: Reset 802.1AS counters for the given interface: %s 27 <intfNum>\n", name);
  printf("Test 28: Run API sanity checks: %s 28 \n", name);
  printf("Test 29: Get 802.1AS path trace: %s 29 \n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  OPEN_BOOL_t isCapable;
  uint32_t pDelay;
  OPEN_CONTROL_t adminMode;
  uint32_t numAllowedLostResp;
  OPEN_DOT1AS_CLOCK_PRIORITY_t clockPriorityType;
  OPEN_DOT1AS_CLOCK_IDENTITY_t bridgeClkId;
  uint32_t rateRatio;
  uint32_t counterValue;
  OPEN_BOOL_t isMeasuringPdelay;
  OPEN_BOOL_t grandmasterPresent;
  uint32_t threshold;
  uint32_t timeStamp;
  OPEN_DOT1AS_PORT_ROLE_t intfRole;
  uint32_t gmChangeCount;
  uint32_t intfNum;
  uint16_t bestClkSteps;
  OPEN_DOT1AS_GM_EVENT_TYPE_t gmEventType;
  OPEN_DOT1AS_CLOCK_IDENTITY_t bestClockId;
  OPEN_DOT1AS_INTF_INTERVAL_t intervalType;
  int32_t interval;
  OPEN_BOOL_t grandmasterCapable;
  OPEN_DOT1AS_COUNTER_TYPE_t counterType;
  uint32_t clockPriority;
  open_buffdesc identity;
  uint8_t clockPriority1;

  printf("Testing dot1as OpEN APIs sanity:\n");

  printf("Testing openapiDot1asModeGet():\n");
  result = openapiDot1asModeGet(NULL, &adminMode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1asModeGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiDot1asIsGrandmasterPresentGet():\n");
  result = openapiDot1asIsGrandmasterPresentGet(NULL, &grandmasterPresent);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1asIsGrandmasterPresentGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiDot1asGrandmasterCapableGet():\n");
  result = openapiDot1asGrandmasterCapableGet(NULL, &grandmasterCapable);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1asGrandmasterCapableGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiDot1asBestClkIdGet():\n");
  result = openapiDot1asBestClkIdGet(NULL, &bestClockId);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1asBestClkIdGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiDot1asBestClkPriorityGet():\n");
  clockPriorityType = 0;
  result = openapiDot1asBestClkPriorityGet(NULL, clockPriorityType, &clockPriority1);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1asBestClkPriorityGet(clientHandle, clockPriorityType, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1asBestClkStepsGet():\n");
  result = openapiDot1asBestClkStepsGet(NULL, &bestClkSteps);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1asBestClkStepsGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiDot1asBridgeClkIdGet():\n");
  result = openapiDot1asBridgeClkIdGet(NULL, &bridgeClkId);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1asBridgeClkIdGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiDot1asBridgePriorityGet():\n");
  result = openapiDot1asBridgePriorityGet(NULL, clockPriorityType, &clockPriority);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1asBridgePriorityGet(clientHandle, clockPriorityType, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1asGrandmasterChangeCountGet():\n");
  result = openapiDot1asGrandmasterChangeCountGet(NULL, &gmChangeCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1asGrandmasterChangeCountGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiDot1asEventTimestampGet():\n");
  gmEventType = 0;
  result = openapiDot1asEventTimestampGet(NULL, gmEventType, &timeStamp);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1asEventTimestampGet(clientHandle, gmEventType, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1asIntfModeGet():\n");
  intfNum = 0;
  result = openapiDot1asIntfModeGet(NULL, intfNum, &adminMode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1asIntfModeGet(clientHandle, intfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1asIntfIsCapableGet():\n");
  result = openapiDot1asIntfIsCapableGet(NULL, intfNum, &isCapable);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1asIntfIsCapableGet(clientHandle, intfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1asIntfIsMeasuringPdelayGet():\n");
  result = openapiDot1asIntfIsMeasuringPdelayGet(NULL, intfNum, &isMeasuringPdelay);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1asIntfIsMeasuringPdelayGet(clientHandle, intfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1asIntfPdelayGet():\n");
  result = openapiDot1asIntfPdelayGet(NULL, intfNum, &pDelay);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1asIntfPdelayGet(clientHandle, intfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1asIntfRoleGet():\n");
  result = openapiDot1asIntfRoleGet(NULL, intfNum, &intfRole);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1asIntfRoleGet(clientHandle, intfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1asIntfPdelayThresholdGet():\n");
  result = openapiDot1asIntfPdelayThresholdGet(NULL, intfNum, &threshold);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1asIntfPdelayThresholdGet(clientHandle, intfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1asIntfAllowedLostRespGet():\n");
  result = openapiDot1asIntfAllowedLostRespGet(NULL, intfNum, &numAllowedLostResp);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1asIntfAllowedLostRespGet(clientHandle, intfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1asIntfNeighborRateRatioGet():\n");
  result = openapiDot1asIntfNeighborRateRatioGet(NULL, intfNum, &rateRatio);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1asIntfNeighborRateRatioGet(clientHandle, intfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1asIntfIntervalGet():\n");
  intervalType = 0;
  result = openapiDot1asIntfIntervalGet(NULL, intfNum, intervalType, &interval);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1asIntfIntervalGet(clientHandle, intfNum, intervalType, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiDot1asIntfStatsGet():\n");
  counterType = 1;
  result = openapiDot1asIntfStatsGet(NULL, intfNum, counterType, &counterValue);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1asIntfStatsGet(clientHandle, intfNum, counterType, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiDot1asPathTraceClockIdGetNext():\n");
  result = openapiDot1asPathTraceClockIdGetNext((void *)NULL, &identity);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1asPathTraceClockIdGetNext(clientHandle, (void *)NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  return;
}

/*********************************************************************
* @purpose  Get the 802.1AS global admin mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    adminMode       @b{(output)}  Global 802.1as admin mode.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *adminMode)
{
  open_error_t result;
  char enableDisableStr[16] = "disabled"; 

  if ((result = openapiDot1asModeGet(client_handle, adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the 802.1AS global admin mode. (result = %d)\n", result);
  }
  else
  {
    if (*adminMode == OPEN_ENABLE)
    {
      snprintf(enableDisableStr, sizeof(enableDisableStr), "enabled");
    }
    printf("802.1AS global admin mode is %s\n",enableDisableStr);
  }
  return;
}

/*********************************************************************
* @purpose  Set the 802.1AS global admin mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    adminMode       @b{(input)}  Global 802.1as admin mode.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t  adminMode)
{
  open_error_t result;

  if ((result = openapiDot1asModeSet(client_handle, adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the 802.1AS global admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("802.1AS global admin mode is set. (result = %d)\n", result);
  }
  return;
}

/*********************************************************************
* @purpose  Check if there is a Grand Master elected in the 802.1AS domain.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    grandmasterPresent  @b{(output)}  Grand master present status.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asIsGrandmasterPresentGet(openapiClientHandle_t *client_handle, 
                                   OPEN_BOOL_t *grandmasterPresent)
{
  open_error_t result;

  if ((result = openapiDot1asIsGrandmasterPresentGet(client_handle, grandmasterPresent)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to check if there is a Grand Master elected in the 802.1AS domain. (result = %d)\n", result);
  }
  else
  {
    if (*grandmasterPresent == OPEN_FALSE)
    {
      printf("Grand master is not present \n");
    }
    else
    {
      printf("Grand master is present \n");
    }
  }
  return;
}


/*********************************************************************
* @purpose  Check if this bridge is capable of being a grandmaster.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    grandmasterCapable  @b{(output)}  Grand master capability status.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asGrandmasterCapableGet(openapiClientHandle_t *client_handle, 
                                 OPEN_BOOL_t *grandmasterCapable)
{
  open_error_t result;

  if ((result = openapiDot1asGrandmasterCapableGet(client_handle, grandmasterCapable)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to check if this bridge is capable of being a grandmaster. (result = %d)\n", result);
  }
  else
  {
    if (*grandmasterCapable == OPEN_FALSE)
    {
      printf("Not grand master capable\n");
    }
    else
    {
      printf("Grand master capable\n");
    }
  }
  return;
}

/*********************************************************************
* @purpose  Get the Best clock identity as detected by this 801AS protocol.
*
* @param    client_handle     @b{(input)}  client handle from registration API
* @param    bestClockId       @b{(output)}  Best clock id as detected by 801AS protocol.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asBestClkIdGet(openapiClientHandle_t *client_handle, 
                        OPEN_DOT1AS_CLOCK_IDENTITY_t *bestClockId)
{
  open_error_t result;

  if ((result = openapiDot1asBestClkIdGet(client_handle, bestClockId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Best clock identity as detected by this 801AS protocol. (result = %d)\n", result);
  }
  else
  {
    printf("Best clock id is : %02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X\n",
	                       bestClockId->id[0], bestClockId->id[1], 
			       bestClockId->id[2], bestClockId->id[3],
	                       bestClockId->id[4], bestClockId->id[5], 
			       bestClockId->id[6], bestClockId->id[7]);
  }
  return;
}


/*********************************************************************
* @purpose  Get the priority1/2 value of Best clock in the 802.1AS domain.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    clockPriorityType   &b{(input)}  Type of clock prority requested, 
*                                            Priority 1 or Priority2.
* @param    clockPriority       @b{(output)}  Clock priority of type clockPriorityType.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asBestClkPriorityGet(openapiClientHandle_t *client_handle, 
                              OPEN_DOT1AS_CLOCK_PRIORITY_t  clockPriorityType, 
			      uint8_t *clockPriority)
{
  open_error_t result;

  if ((result = openapiDot1asBestClkPriorityGet(client_handle, clockPriorityType, clockPriority)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the priority1/2 value of Best clock in the 802.1AS domain. (result = %d)\n", result);
  }
  else
  {
    printf("Best clock priority%d is %d\n", 
	    clockPriorityType+1, *clockPriority);
  }
  return;
}


/*********************************************************************
* @purpose  Get the steps to Best clock as detected by this protocol.
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    bestClkSteps       @b{(output)}  Number of steps to the best clock.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asBestClkStepsGet(openapiClientHandle_t *client_handle, 
                           uint16_t *bestClkSteps)
{
  open_error_t result;

  if ((result = openapiDot1asBestClkStepsGet(client_handle, bestClkSteps)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the steps to Best clock as detected by this protocol. (result = %d)\n", result);
  }
  else
  {
    printf("Steps to best clock %d\n", *bestClkSteps);
  }
  return;
}

/*********************************************************************
* @purpose  Get the clock identity of this bridge.
*
* @param    client_handle     @b{(input)}  client handle from registration API
* @param    bridgeClkId       @b{(output)}  Clock identity of this bridge.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asBridgeClkIdGet(openapiClientHandle_t *client_handle, 
                          OPEN_DOT1AS_CLOCK_IDENTITY_t *bridgeClkId)
{
  open_error_t result;

  if ((result = openapiDot1asBridgeClkIdGet(client_handle, bridgeClkId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the clock identity of this bridge. (result = %d)\n", result);
  }
  else
  {
    printf("Local clock id is : %02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X\n",
	                       bridgeClkId->id[0], bridgeClkId->id[1], 
			       bridgeClkId->id[2], bridgeClkId->id[3],
	                       bridgeClkId->id[4], bridgeClkId->id[5], 
			       bridgeClkId->id[6], bridgeClkId->id[7]);
  }
  return;
}


/*********************************************************************
* @purpose  Get the priority1/2 value of the local clock.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    clockPriorityType   @b{(input)}  Type of clock prority requested, 
*                                            Priority 1 or Priority2.
* @param    clockPriority       @b{(output)}  Clock priority of type clockPriorityType.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asBridgePriorityGet(openapiClientHandle_t *client_handle, 
                             OPEN_DOT1AS_CLOCK_PRIORITY_t  clockPriorityType, 
			     uint32_t *clockPriority)
{
  open_error_t result;

  if ((result = openapiDot1asBridgePriorityGet(client_handle, clockPriorityType, clockPriority)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the priority1/2 value of the local clock. (result = %d)\n", result);
  }
  else
  {
    printf("Local clock priority%d is %d\n", clockPriorityType+1, *clockPriority);
  }
  return;
}

/*********************************************************************
* @purpose  Set the priority1/2 value of the local clock.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    clockPriorityType   @b{(input)}  Type of clock prority to be set, 
*                                            Priority 1 or Priority2.
* @param    clockPriority       @b{(input)}  Clock priority of type clockPriorityType.

*
* @returns  none 
*
* @notes    clockPriorityType should not be Priority1, this is not allowed. 
* @end
*********************************************************************/
void dot1asBridgePrioritySet(openapiClientHandle_t *client_handle, 
                             OPEN_DOT1AS_CLOCK_PRIORITY_t  clockPriorityType, 
			     uint32_t  clockPriority)
{
  open_error_t result;

  if ((result = openapiDot1asBridgePrioritySet(client_handle, clockPriorityType, clockPriority)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the priority of the local clock. (result = %d)\n", result);
  }
  else
  {
    printf("Local clock priority is set.\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the number of times the grandmaster has changed.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    gmChangeCount       @b{(output)}  count of number of times grand master has changed.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asGrandmasterChangeCountGet(openapiClientHandle_t *client_handle, 
                                     uint32_t *gmChangeCount)
{
  open_error_t result;

  if ((result = openapiDot1asGrandmasterChangeCountGet(client_handle, gmChangeCount)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the number of times the grandmaster has changed. (result = %d)\n", result);
  }
  else
  {
    printf("Grand master change count is %d\n", *gmChangeCount);
  }
  return;
}

/*********************************************************************
* @purpose  Get the timestamp of the given event.
*
* @param    client_handle     @b{(input)}  client handle from registration API
* @param    gmEventType       @b{(input)}  Event type like GM change, 
*                                          phase change, freq change.
* @param    timeStamp         @b{(output)} System time when the last requested 
*                                          event occured.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asEventTimestampGet(openapiClientHandle_t *client_handle, 
                             OPEN_DOT1AS_GM_EVENT_TYPE_t gmEventType, 
			     uint32_t *timeStamp)
{
  open_error_t result;

  if ((result = openapiDot1asEventTimestampGet(client_handle, gmEventType, timeStamp)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the timestamp of the given event. (result = %d)\n", result);
  }
  else
  {
    printf("802.1AS event timestamp is %d\n", *timeStamp);
  }
  return;
}

/*********************************************************************
* @purpose  Get the 802.1AS admin mode for the given interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    intfNum         @b{(input)}  Interface number.
* @param    adminMode       @b{(output)}  Interface 802.1as admin mode.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asIntfModeGet(openapiClientHandle_t *client_handle, 
                       uint32_t  intfNum, OPEN_CONTROL_t *adminMode)
{
  open_error_t result;
  char enableDisableStr[16] = "disabled"; 
  uint8_t bufIntfName[OPEN_INTF_NAME_MAX_LENGTH] = {0};
  open_buffdesc bufdName = {sizeof(bufIntfName), bufIntfName};

  if ((result = openapiIntfNameGet(client_handle, intfNum, &bufdName)) != OPEN_E_NONE)
  {
    printf("Invalid interface number %d. (result = %d)\n", intfNum, result);
    return;
  }

  if ((result = openapiDot1asIntfModeGet(client_handle, intfNum, adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the 802.1AS admin mode for the given interface. (result = %d)\n", result);
  }
  else
  {
    if (*adminMode == OPEN_ENABLE)
    {
      snprintf(enableDisableStr, sizeof(enableDisableStr), "enabled");
    }
    printf("\n Name is %s\n",bufIntfName); 
    printf("On interface %d (interface name %s) 802.1AS admin mode is %s\n",
	   intfNum, (char *)bufdName.pstart, enableDisableStr);
  }
  return;
}

/*********************************************************************
* @purpose  Set the 802.1AS admin mode for the given interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    intfNum         @b{(input)}  Interface number.
* @param    adminMode       @b{(output)}  Interface 802.1as admin mode.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asIntfModeSet(openapiClientHandle_t *client_handle, 
                       uint32_t  intfNum, OPEN_CONTROL_t  adminMode)
{
  open_error_t result;
  uint8_t bufIntfName[OPEN_INTF_NAME_MAX_LENGTH] = {0};
  open_buffdesc bufdName = {sizeof(bufIntfName), bufIntfName};

  if ((result = openapiIntfNameGet(client_handle, intfNum, &bufdName)) != OPEN_E_NONE)
  {
    printf("Invalid interface number %d. (result = %d)\n", intfNum, result);
    return;
  }

  if ((result = openapiDot1asIntfModeSet(client_handle, intfNum, adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the 802.1AS admin mode for the given interface. (result = %d)\n", result);
  }
  else
  {
    printf("On interface %d (interface name %s) 802.1AS admin mode is set\n",
	   intfNum, (char *)bufdName.pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Check if the given interface 802.1AS capable (as determined by the peer delay mechanism.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    intfNum         @b{(input)}  Interface number.
* @param    isCapable       @b{(output)}  802.1as capability of interface.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asIntfIsCapableGet(openapiClientHandle_t *client_handle, 
                            uint32_t  intfNum, OPEN_BOOL_t *isCapable)
{
  open_error_t result;
  uint8_t bufIntfName[OPEN_INTF_NAME_MAX_LENGTH] = {0};
  open_buffdesc bufdName = {sizeof(bufIntfName), bufIntfName};

  if ((result = openapiIntfNameGet(client_handle, intfNum, &bufdName)) != OPEN_E_NONE)
  {
    printf("Invalid interface number %d. (result = %d)\n", intfNum, result);
    return;
  }

  if ((result = openapiDot1asIntfIsCapableGet(client_handle, intfNum, isCapable)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to check if the given interface 802.1AS capable (as determined by the peer delay mechanism. (result = %d)\n", result);
  }
  else
  {
    if(*isCapable == OPEN_TRUE)
    {
      printf("Interface %d (interface name %s) is 802.1AS capable.\n",
	     intfNum, (char *)bufdName.pstart);
    }
    else
    {
      printf("Interface %d (interface name %s) is not 802.1AS capable.\n",
	     intfNum, (char *)bufdName.pstart);
    }

  }
  return;
}

/*********************************************************************
* @purpose  Get the 'isMeasuringPdelay' property, which is set to TRUE if the given interface is receiving PDELAY response messages from other end of the link.
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    intfNum            @b{(input)}  Interface number.
* @param    isMeasuringPdelay  @b{(output)}  Peer delay measuring status.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asIntfIsMeasuringPdelayGet(openapiClientHandle_t *client_handle, 
                                    uint32_t  intfNum, 
				    OPEN_BOOL_t *isMeasuringPdelay)
{
  open_error_t result;
  uint8_t bufIntfName[OPEN_INTF_NAME_MAX_LENGTH] = {0};
  open_buffdesc bufdName = {sizeof(bufIntfName), bufIntfName};

  if ((result = openapiIntfNameGet(client_handle, intfNum, &bufdName)) != OPEN_E_NONE)
  {
    printf("Invalid interface number %d. (result = %d)\n", intfNum, result);
    return;
  }

  if ((result = openapiDot1asIntfIsMeasuringPdelayGet(client_handle, intfNum, isMeasuringPdelay)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the status of peer delay measuing.(result = %d)\n", result);
  }
  else
  {
    if(*isMeasuringPdelay == OPEN_TRUE)
    {
      printf("Interface %d (interface name %s) is measuring peer delay.\n",
	     intfNum, (char *)bufdName.pstart);
    }
    else
    {
      printf("Interface %d (interface name %s) is not measuring peer delay.\n",
	     intfNum, (char *)bufdName.pstart);
    }
  }
  return;
}


/*********************************************************************
* @purpose  Get the propagation delay on the given interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    intfNum         @b{(input)}  Interface number.
* @param    pDelay          @b{(output)}  Propogation delay on the interface.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asIntfPdelayGet(openapiClientHandle_t *client_handle, 
                         uint32_t  intfNum, uint32_t *pDelay)
{
  open_error_t result;
  uint8_t bufIntfName[OPEN_INTF_NAME_MAX_LENGTH] = {0};
  open_buffdesc bufdName = {sizeof(bufIntfName), bufIntfName};

  if ((result = openapiIntfNameGet(client_handle, intfNum, &bufdName)) != OPEN_E_NONE)
  {
    printf("Invalid interface number %d. (result = %d)\n", intfNum, result);
    return;
  }

  if ((result = openapiDot1asIntfPdelayGet(client_handle, intfNum, pDelay)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the propagation delay on the given interface. (result = %d)\n", result);
  }
  else
  {
    printf("Interface %d (interface name %s) Propogation delay %d.\n",
	   intfNum, (char *)bufdName.pstart, *pDelay);
  }
  return;
}


/*********************************************************************
* @purpose  Get the 802.1AS role of the given interface.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intfNum        @b{(input)}  Interface number.
* @param    intfRole       @b{(output)}  Interface role in the 802.1AS protocol.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asIntfRoleGet(openapiClientHandle_t *client_handle, 
                       uint32_t  intfNum, 
		       OPEN_DOT1AS_PORT_ROLE_t *intfRole)
{
  open_error_t result;
  uint8_t bufIntfName[OPEN_INTF_NAME_MAX_LENGTH] = {0};
  open_buffdesc bufdName = {sizeof(bufIntfName), bufIntfName};

  if ((result = openapiIntfNameGet(client_handle, intfNum, &bufdName)) != OPEN_E_NONE)
  {
    printf("Invalid interface number %d. (result = %d)\n", intfNum, result);
    return;
  }

  if ((result = openapiDot1asIntfRoleGet(client_handle, intfNum, intfRole)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the 802.1AS role of the given interface. (result = %d)\n", result);
  }
  else
  { 
    printf("Interface %d (interface name %s) 802.1AS Role is %d.\n", 
	   intfNum, (char *)bufdName.pstart, *intfRole);
  }
  return;
}

/*********************************************************************
* @purpose  Get the propagation delay threshold on the given interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intfNum       @b{(input)}  Interface number.
* @param    threshold       @b{(output)}  Propogation delay threshold on the interface.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asIntfPdelayThresholdGet(openapiClientHandle_t *client_handle, 
                                  uint32_t  intfNum, uint32_t *threshold)
{
  open_error_t result;
  uint8_t bufIntfName[OPEN_INTF_NAME_MAX_LENGTH] = {0};
  open_buffdesc bufdName = {sizeof(bufIntfName), bufIntfName};

  if ((result = openapiIntfNameGet(client_handle, intfNum, &bufdName)) != OPEN_E_NONE)
  {
    printf("Invalid interface number %d. (result = %d)\n", intfNum, result);
    return;
  }

  if ((result = openapiDot1asIntfPdelayThresholdGet(client_handle, intfNum, threshold)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the propagation delay threshold on the given interface. (result = %d)\n", result);
  }
  else
  {
    printf("Interface %d (interface name %s) Propogation delay threshold %d.\n",
	   intfNum, (char *)bufdName.pstart, *threshold);
  }
  return;
}


/*********************************************************************
* @purpose  Set the propagation delay threshold on the given interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intfNum       @b{(input)}  Interface number.
* @param    threshold       @b{(input)}  Propogation delay threshold on the interface.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asIntfPdelayThresholdSet(openapiClientHandle_t *client_handle, 
                                  uint32_t  intfNum, uint32_t  threshold)
{
  open_error_t result;
  uint8_t bufIntfName[OPEN_INTF_NAME_MAX_LENGTH] = {0};
  open_buffdesc bufdName = {sizeof(bufIntfName), bufIntfName};

  if ((result = openapiIntfNameGet(client_handle, intfNum, &bufdName)) != OPEN_E_NONE)
  {
    printf("Invalid interface number %d. (result = %d)\n", intfNum, result);
    return;
  }

  if ((result = openapiDot1asIntfPdelayThresholdSet(client_handle, intfNum, threshold)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the propagation delay threshold on the given interface. (result = %d)\n", result);
  }
  else
  {
    printf("On interface %d (interface name %s) Propogation delay threshold is set.\n",	intfNum, (char *)bufdName.pstart);
  }
  return;
}


/*********************************************************************
* @purpose  Get the number of Pdelay_Req messages for which a valid response is not received, on which a port is considered to not be exchanging peer delay messages with its neighbor.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intfNum       @b{(input)}  Interface number.
* @param    numAllowedLostResp       @b{(output)}  Number of allowed lost responses.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asIntfAllowedLostRespGet(openapiClientHandle_t *client_handle, 
                                  uint32_t  intfNum, uint32_t *numAllowedLostResp)
{
  open_error_t result;
  uint8_t bufIntfName[OPEN_INTF_NAME_MAX_LENGTH] = {0};
  open_buffdesc bufdName = {sizeof(bufIntfName), bufIntfName};

  if ((result = openapiIntfNameGet(client_handle, intfNum, &bufdName)) != OPEN_E_NONE)
  {
    printf("Invalid interface number %d. (result = %d)\n", intfNum, result);
    return;
  }

  if ((result = openapiDot1asIntfAllowedLostRespGet(client_handle, intfNum, numAllowedLostResp)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the number of PDELAY lost responses allowed. (result = %d)\n", result);
  }
  else
  {
    printf("On interface %d (interface name %s) PDELAY lost responses allowed %u.\n",           intfNum, (char *)bufdName.pstart, *numAllowedLostResp);
  }
  return;
}


/*********************************************************************
* @purpose  Set the number of Pdelay_Req messages for which a valid response is not received, on which a port is considered to not be exchanging peer delay messages with its neighbor.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intfNum             @b{(input)}  Interface number.
* @param    numAllowedLostResp  @b{(input)}  Number of allowed lost responses.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asIntfAllowedLostRespSet(openapiClientHandle_t *client_handle, uint32_t  intfNum, uint32_t  numAllowedLostResp)
{
  open_error_t result;
  uint8_t bufIntfName[OPEN_INTF_NAME_MAX_LENGTH] = {0};
  open_buffdesc bufdName = {sizeof(bufIntfName), bufIntfName};

  if ((result = openapiIntfNameGet(client_handle, intfNum, &bufdName)) != OPEN_E_NONE)
  {
    printf("Invalid interface number %d. (result = %d)\n", intfNum, result);
    return;
  }

  if ((result = openapiDot1asIntfAllowedLostRespSet(client_handle, intfNum, numAllowedLostResp)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the number of PDELAY lost responses allowed. (result = %d)\n", result);
  }
  else
  {
    printf("On interface %d (interface name %s) PDELAY lost responses allowed is set.\n", intfNum, (char *)bufdName.pstart);
  }
  return;
}


/*********************************************************************
* @purpose  Get the neighbor rate ratio on the given interface (as computed by the pdelay mechanism) (extensible version).
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    intfNum         @b{(input)}  Interface number.
* @param    rateRatio       @b{(output)}  The current value of the rate ratio.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asIntfNeighborRateRatioGet(openapiClientHandle_t *client_handle, 
                                    uint32_t  intfNum, uint32_t *rateRatio)
{
  open_error_t result;
  uint8_t bufIntfName[OPEN_INTF_NAME_MAX_LENGTH] = {0};
  open_buffdesc bufdName = {sizeof(bufIntfName), bufIntfName};

  if ((result = openapiIntfNameGet(client_handle, intfNum, &bufdName)) != OPEN_E_NONE)
  {
    printf("Invalid interface number %d. (result = %d)\n", intfNum, result);
    return;
  }

  if ((result = openapiDot1asIntfNeighborRateRatioGet(client_handle, intfNum, rateRatio)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the neighbor rate ratio on the given interface (as computed by the pdelay mechanism) (extensible version). (result = %d)\n", result);
  }
  else
  {
    printf("On interface %d (interface name %s) Neighbour rate ratio %d.\n", intfNum, (char *)bufdName.pstart, *rateRatio);
  }
  return;
}

/*********************************************************************
* @purpose  Get the given interval on the given interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intfNum       @b{(input)}  Interface number.
* @param    intervalType       @b{(input)}  DOT1AS interval type.
* @param    interval       @b{(output)}  The interface interval/timeout value.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asIntfIntervalGet(openapiClientHandle_t *client_handle, 
                           uint32_t  intfNum, 
                           OPEN_DOT1AS_INTF_INTERVAL_t  intervalType, 
                           int32_t *interval)
{
  open_error_t result;
  uint8_t bufIntfName[OPEN_INTF_NAME_MAX_LENGTH] = {0};
  open_buffdesc bufdName = {sizeof(bufIntfName), bufIntfName};

  if ((result = openapiIntfNameGet(client_handle, intfNum, &bufdName)) != OPEN_E_NONE)
  {
    printf("Invalid interface number %d. (result = %d)\n", intfNum, result);
    return;
  }

  if ((result = openapiDot1asIntfIntervalGet(client_handle, intfNum, intervalType, interval)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the given interval on the given interface. (result = %d)\n", result);
  }
  else
  {
    printf("On interface %d (interface name %s) interval of type %d is %d.\n", intfNum,(char *)bufdName.pstart, intervalType, *interval);
  }
  return;
}


/*********************************************************************
* @purpose  Set the given interval on the given interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intfNum       @b{(input)}  Interface number.
* @param    intervalType       @b{(input)}  DOT1AS interval type.
* @param    interval       @b{(input)}  The interface interval/timeout value.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asIntfIntervalSet(openapiClientHandle_t *client_handle, 
                           uint32_t  intfNum, 
                           OPEN_DOT1AS_INTF_INTERVAL_t  intervalType, 
                           int32_t  interval)
{
  open_error_t result;
  uint8_t bufIntfName[OPEN_INTF_NAME_MAX_LENGTH] = {0};
  open_buffdesc bufdName = {sizeof(bufIntfName), bufIntfName};

  if ((result = openapiIntfNameGet(client_handle, intfNum, &bufdName)) != OPEN_E_NONE)
  {
    printf("Invalid interface number %d. (result = %d)\n", intfNum, result);
    return;
  }

  if ((result = openapiDot1asIntfIntervalSet(client_handle, intfNum, intervalType, interval)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the given interval on the given interface. (result = %d)\n", result);
  }
  else
  {
    printf("On interface %d (interface name %s) interval is set.\n", intfNum, (char *)bufdName.pstart);
  }
  return;
}


/*********************************************************************
* @purpose  Get the requested 802.1AS counter for the given interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intfNum       @b{(input)}  Interface number.
* @param    counterType       @b{(input)}  DOT1AS counter type.
* @param    counterValue       @b{(output)}  Counter value.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asIntfStatsGet(openapiClientHandle_t *client_handle, 
                        uint32_t  intfNum, 
			OPEN_DOT1AS_COUNTER_TYPE_t  counterType, 
			uint32_t *counterValue)
{
  open_error_t result;
  uint8_t bufIntfName[OPEN_INTF_NAME_MAX_LENGTH] = {0};
  open_buffdesc bufdName = {sizeof(bufIntfName), bufIntfName};

  if ((result = openapiIntfNameGet(client_handle, intfNum, &bufdName)) != OPEN_E_NONE)
  {
    printf("Invalid interface number %d. (result = %d)\n", intfNum, result);
    return;
  }

  if ((result = openapiDot1asIntfStatsGet(client_handle, intfNum, counterType, counterValue)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the requested 802.1AS counter for the given interface. (result = %d)\n", result);
  }
  else
  {
    printf("On interface %d (interface name %s) stats value %u.\n", 
	    intfNum, (char *)bufdName.pstart, *counterValue);
  }
  return;
}


/*********************************************************************
* @purpose  Reset 802.1AS counters for the given interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intfNum       @b{(input)}  Interface number.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1asIntfStatsReset(openapiClientHandle_t *client_handle, uint32_t  intfNum)
{
  open_error_t result;
  uint8_t bufIntfName[OPEN_INTF_NAME_MAX_LENGTH] = {0};
  open_buffdesc bufdName = {sizeof(bufIntfName), bufIntfName};

  if ((result = openapiIntfNameGet(client_handle, intfNum, &bufdName)) != OPEN_E_NONE)
  {
    printf("Invalid interface number %d. (result = %d)\n", intfNum, result);
    return;
  }

  if ((result = openapiDot1asIntfStatsReset(client_handle, intfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to reset 802.1AS counters for the given interface. (result = %d)\n", result);
  }
  else
  {
    printf("On interface %d (interface name %s) counters are reset.\n", 
	    intfNum, (char *)bufdName.pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Fetch next clock identity in 802.1AS path trace.
*
* @param    client_handle @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1asPathTraceClockIdGetNext(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  open_buffdesc identity;
  char id[256] = {0};
  OPEN_BOOL_t clockFetchSuccess = OPEN_FALSE;
  uint32_t hopCount = 0;

  identity.pstart = id;
  identity.size = 256;

  openapiDot1asHopCountGet(client_handle, &hopCount);

  do 
  {
    if ((result = openapiDot1asPathTraceClockIdGetNext(client_handle, &identity))
        != OPEN_E_NONE)
    {
      break;
    }
    else
    {
      if (clockFetchSuccess == OPEN_FALSE)
      {
        clockFetchSuccess = OPEN_TRUE;
        printf("Hop(s): %d\n", hopCount);
        printf("Clock identity\n");
        printf("------------------------\n");
      }

      printf("%s\n", (char *)identity.pstart);
    }
  } while (result != OPEN_E_FAIL);

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for dot1as
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  OPEN_BOOL_t isCapable;
  uint32_t pDelay;
  OPEN_CONTROL_t adminMode;
  uint32_t numAllowedLostResp;
  OPEN_DOT1AS_CLOCK_PRIORITY_t clockPriorityType;
  OPEN_DOT1AS_CLOCK_IDENTITY_t bridgeClkId;
  uint32_t rateRatio;
  uint32_t counterValue;
  OPEN_BOOL_t isMeasuringPdelay;
  OPEN_BOOL_t grandmasterPresent;
  uint32_t threshold;
  uint32_t timeStamp;
  OPEN_DOT1AS_PORT_ROLE_t intfRole;
  uint32_t gmChangeCount;
  uint32_t intfNum;
  uint16_t bestClkSteps;
  OPEN_DOT1AS_GM_EVENT_TYPE_t gmEventType;
  OPEN_DOT1AS_CLOCK_IDENTITY_t bestClockId;
  OPEN_DOT1AS_INTF_INTERVAL_t intervalType;
  int32_t interval;
  OPEN_BOOL_t grandmasterCapable;
  OPEN_DOT1AS_COUNTER_TYPE_t counterType;
  uint8_t clockPriority;
  uint32_t clockPriority1;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("dot1as example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting dot1as API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 2)
      {
        dot1asModeGet(&client_handle, &adminMode);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 3)
      {
        show_help = 1;
	if (argv[2][0] == '0' || argv[2][0]=='1')
	{
	  adminMode = atoi(argv[2]);
          dot1asModeSet(&client_handle, adminMode);
          show_help = 0;
	}
      }
      break;
    case 3:
      if (argc == 2)
      {
        dot1asIsGrandmasterPresentGet(&client_handle, &grandmasterPresent);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 2)
      {
        dot1asGrandmasterCapableGet(&client_handle, &grandmasterCapable);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 2)
      {
        dot1asBestClkIdGet(&client_handle, &bestClockId);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 3)
      {
        show_help = 1;
	if (argv[2][0] == '0' || argv[2][0]=='1')
	{
	 clockPriorityType  = atoi(argv[2]);
         dot1asBestClkPriorityGet(&client_handle, clockPriorityType, &clockPriority);
         show_help = 0;
	}
      }
      break;
    case 7:
      if (argc == 2)
      {
        dot1asBestClkStepsGet(&client_handle, &bestClkSteps);
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 2)
      {
        dot1asBridgeClkIdGet(&client_handle, &bridgeClkId);
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 3)
      {
        show_help = 1;
	if (argv[2][0] == '0' || argv[2][0]=='1')
	{
	 clockPriorityType  = atoi(argv[2]);
         dot1asBridgePriorityGet(&client_handle, clockPriorityType, &clockPriority1);
         show_help = 0;
	}
      }
      break;
    case 10:
      if (argc == 4)
      {
        show_help = 1;
	if (argv[2][0] == '0' || argv[2][0]=='1')
	{
	 clockPriorityType  = atoi(argv[2]);
         dot1asBridgePrioritySet(&client_handle, clockPriorityType, atoi(argv[3]));
         show_help = 0;
	}
      }
      break;
    case 11:
      if (argc == 2)
      {
        dot1asGrandmasterChangeCountGet(&client_handle, &gmChangeCount);
        show_help = 0;
      }
      break;
    case 12:
      if (argc == 3)
      {
        show_help = 1;
	if (argv[2][0] == '0' || argv[2][0]=='1' || argv[2][0]=='2')
	{
	 gmEventType  = atoi(argv[2]);
         dot1asEventTimestampGet(&client_handle, gmEventType, &timeStamp);
         show_help = 0;
	}
      }
      break;
    case 13:
      if (argc == 3)
      {
	intfNum = atoi(argv[2]);
        dot1asIntfModeGet(&client_handle, intfNum, &adminMode);
        show_help = 0;
      }
      break;
    case 14:
      if (argc == 4)
      {
	intfNum = atoi(argv[2]);
	adminMode = atoi(argv[3]);
        dot1asIntfModeSet(&client_handle, intfNum, adminMode);
        show_help = 0;
      }
      break;
    case 15:
      if (argc == 3)
      {
	intfNum = atoi(argv[2]);
        dot1asIntfIsCapableGet(&client_handle, intfNum, &isCapable);
        show_help = 0;
      }
      break;
    case 16:
      if (argc == 3)
      {
	intfNum = atoi(argv[2]);
        dot1asIntfIsMeasuringPdelayGet(&client_handle, intfNum, &isMeasuringPdelay);
        show_help = 0;
      }
      break;
    case 17:
      if (argc == 3)
      {
	intfNum = atoi(argv[2]);
        dot1asIntfPdelayGet(&client_handle, intfNum, &pDelay);
        show_help = 0;
      }
      break;
    case 18:
      if (argc == 3)
      {
	intfNum = atoi(argv[2]);
        dot1asIntfRoleGet(&client_handle, intfNum, &intfRole);
        show_help = 0;
      }
      break;
    case 19:
      if (argc == 3)
      {
	intfNum = atoi(argv[2]);
        dot1asIntfPdelayThresholdGet(&client_handle, intfNum, &threshold);
        show_help = 0;
      }
      break;
    case 20:
      if (argc == 4)
      {
	intfNum = atoi(argv[2]);
	threshold = atoi(argv[3]);
        dot1asIntfPdelayThresholdSet(&client_handle, intfNum, threshold);
        show_help = 0;
      }
      break;
    case 21:
      if (argc == 3)
      {
	intfNum = atoi(argv[2]);
        dot1asIntfAllowedLostRespGet(&client_handle, intfNum, &numAllowedLostResp);
        show_help = 0;
      }
      break;
    case 22:
      if (argc == 4)
      {
	intfNum = atoi(argv[2]);
	numAllowedLostResp = atoi(argv[3]);
        dot1asIntfAllowedLostRespSet(&client_handle, intfNum, numAllowedLostResp);
        show_help = 0;
      }
      break;
    case 23:
      if (argc == 3)
      {
	intfNum = atoi(argv[2]);
        dot1asIntfNeighborRateRatioGet(&client_handle, intfNum, &rateRatio);
        show_help = 0;
      }
      break;
    case 24:
      if (argc == 4)
      {
        intfNum = atoi(argv[2]);
        intervalType = atoi(argv[3]);
        dot1asIntfIntervalGet(&client_handle, intfNum, intervalType, &interval);
        show_help = 0;
      }
      break;
    case 25:
      if (argc == 5)
      {
        intfNum = atoi(argv[2]);
        intervalType = atoi(argv[3]);
        interval = atoi(argv[4]);
        dot1asIntfIntervalSet(&client_handle, intfNum, intervalType, interval);
        show_help = 0;
      }
      break;
    case 26:
      if (argc == 4)
      {
	intfNum = atoi(argv[2]);
	counterType = atoi(argv[3]);
        dot1asIntfStatsGet(&client_handle, intfNum, counterType, &counterValue);
        show_help = 0;
      }
      break;
    case 27:
      if (argc == 3)
      {
	intfNum = atoi(argv[2]);
        dot1asIntfStatsReset(&client_handle, intfNum);
        show_help = 0;
      }
      break;
    case 28:
      if (argc == 2)
      {
        runSanity(&client_handle);
        show_help = 0;
      }
      break;

    case 29:
      if (argc == 2)
      {
        dot1asPathTraceClockIdGetNext(&client_handle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping dot1as API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}
