/*********************************************************************
*
*  Copyright 2016-2023 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  dot1x_example.c
*
* @purpose   IEEE 802.1x OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    11/16/2016
*
* @end
*
**********************************************************************/
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_dot1x.h"
#include "openapi_authmgr.h"

#define OPENAPI_TITLE "802.1x"
#define EXAMPLE_APP_NAME "dot1x_example"

typedef void exampleFunction_t(openapiClientHandle_t *clientHandle, int argc, char **argv);
typedef struct
{
  exampleFunction_t *function;
  char              *description;
  char              *parmSyntax;
} functionTableEntry_t;

struct
{
  OPEN_USER_MGR_AUTH_METHOD_t val;
  char *text;
} dot1xUserMgrAuthMethodTextList[] =
{
  {OPEN_AUTH_METHOD_UNDEFINED, "Undefined"},
  {OPEN_AUTH_METHOD_ENABLE,    "Enable"},
  {OPEN_AUTH_METHOD_LINE,      "Line"},
  {OPEN_AUTH_METHOD_LOCAL,     "Local"},
  {OPEN_AUTH_METHOD_NONE,      "None"},
  {OPEN_AUTH_METHOD_RADIUS,    "RADIUS"},
  {OPEN_AUTH_METHOD_TACACS,    "TACACS"},
  {OPEN_AUTH_METHOD_REJECT,    "Reject"},
  {OPEN_AUTH_METHOD_IAS,       "IAS"},
};

#define DOT1X_USER_MGR_AUTH_METHOD_TEXT_LIST_SIZE (sizeof(dot1xUserMgrAuthMethodTextList)/sizeof(dot1xUserMgrAuthMethodTextList[0]))

static char *dot1xUserMgrAuthMethodTextGet(OPEN_USER_MGR_AUTH_METHOD_t val)
{
  int i;

  for (i = 0; i < DOT1X_USER_MGR_AUTH_METHOD_TEXT_LIST_SIZE; i++)
  {
    if (dot1xUserMgrAuthMethodTextList[i].val == val)
    {
      return(dot1xUserMgrAuthMethodTextList[i].text);
    }
  }
  return("UNKNOWN");
}

struct
{
  OPEN_DOT1X_PORT_STATUS_t val;
  char *text;
} dot1xPortStatusTextList[] =
{
  {OPEN_DOT1X_PORT_STATUS_AUTHORIZED,   "Authorized"},
  {OPEN_DOT1X_PORT_STATUS_UNAUTHORIZED, "Unauthorized"},
  {OPEN_DOT1X_PORT_STATUS_NA,           "N/A"},
};

struct
{
  OPEN_DOT1X_APM_STATES_t val;
  char *text;
} dot1xPortApmStateList[] =
{
  {OPEN_DOT1X_APM_INITIALIZE,     "Initialize"},
  {OPEN_DOT1X_APM_DISCONNECTED,   "Disconnected"},
  {OPEN_DOT1X_APM_CONNECTING,     "Connecting"},
  {OPEN_DOT1X_APM_AUTHENTICATING, "Authenticating"},
  {OPEN_DOT1X_APM_AUTHENTICATED,  "Authenticated"},
  {OPEN_DOT1X_APM_ABORTING,       "Aborting"},
  {OPEN_DOT1X_APM_HELD,           "Held"},
  {OPEN_DOT1X_APM_FORCE_AUTH,     "Force Authorized"},
  {OPEN_DOT1X_APM_FORCE_UNAUTH,   "Force Unauthorized"},
};

#define DOT1X_PORT_APM_STATE_TEXT_LIST_SIZE (sizeof(dot1xPortApmStateList)/sizeof(dot1xPortApmStateList[0]))

static char *dot1xPortApmStateTextGet(OPEN_DOT1X_APM_STATES_t val)
{
  int i;

  for (i = 0; i < DOT1X_PORT_APM_STATE_TEXT_LIST_SIZE; i++)
  {
    if (dot1xPortApmStateList[i].val == val)
    {
      return(dot1xPortApmStateList[i].text);
    }
  }
  return("UNKNOWN");
}

/* MAB Authentication type */
struct
{
  OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_t val;
  char *text;
} mabPortMABAuhTypeTextList[] =
{
  {OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_INVALID,        "Invalid"},
  {OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_EAP_MD5,        "EAP MD5"},
  {OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_PAP,            "PAP"},
  {OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_CHAP,           "CHAP"},
};

void systemAuthenticationControlModeGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  result = openapiDot1xSystemAuthControlModeGet(clientHandle, &mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s system authentication control mode (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s system authentication control mode get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, (mode == OPEN_ENABLE)?"Enabled":"Disabled", mode, result);
  }
  return;
}

void systemAuthenticationControlModeSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  if (argc < 3)
  {
    printf("Mode parameter required.\n");
    return;
  }

  errno = 0;
  mode = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid mode parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xSystemAuthControlModeSet(clientHandle, mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s system authentication control mode (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s system authentication control mode set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

void defaultAuthenticationMethodGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  OPEN_USER_MGR_AUTH_METHOD_t method;

  result = openapiDot1xDefaultAuthenMethodGet(clientHandle, &method);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s default authentication method (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s default authentication method get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, dot1xUserMgrAuthMethodTextGet(method), method, result);
  }
  return;
}

void defaultAuthenticationMethodSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  OPEN_CONTROL_t method;

  if (argc < 3)
  {
    printf("Method parameter required.\n");
    return;
  }

  errno = 0;
  method = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid method parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xDefaultAuthenMethodSet(clientHandle, method);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s default authentication method (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s default authentication method set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

void authPortPaeStateGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_DOT1X_APM_STATES_t state;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xPortPaeStateGet(clientHandle, ifNum, &state);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s port PAE state: ifNum = %d (result = %d)\n",
           OPENAPI_TITLE, ifNum, result);
  }
  else
  {
    printf("%s port PAE state get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, dot1xPortApmStateTextGet(state), state, result);
  }
  return;
}

void dot1xEapolFloodModeGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  result = openapiDot1xEapolFloodModeGet(clientHandle, &mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s eapol flood mode (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s eapol flood mode get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, (mode == OPEN_ENABLE)?"Enabled":"Disabled", mode, result);
  }
  return;
}

void dot1xEapolFloodModeSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  if (argc < 3)
  {
    printf("Mode parameter required.\n");
    return;
  }

  mode = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid mode parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xEapolFloodModeSet(clientHandle, mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s eapol flood mode (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s eapol flood mode set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

void dot1xPortQuietPeriodGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  uint32_t quietPeriod;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xPortQuietPeriodGet(clientHandle, ifNum, &quietPeriod);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s port quiet period (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s port quiet period = %u. (result = %d)\n",
           OPENAPI_TITLE, quietPeriod, result);
  }
  return;
}

void dot1xPortQuietPeriodSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  uint32_t quietPeriod;

  if (argc < 4)
  {
    printf("Interface and quietPeriod parameters required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  errno = 0;
  quietPeriod = strtoul(argv[3], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid quietPeriod parameter: (%s)\n", argv[3]);
    return;
  }

  result = openapiDot1xPortQuietPeriodSet(clientHandle, ifNum, quietPeriod);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s port quiet period (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s port quiet period set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

void dot1xPortTxPeriodGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  uint32_t txPeriod;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xPortTxPeriodGet(clientHandle, ifNum, &txPeriod);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s port transmit period (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s port transmit period = %u. (result = %d)\n",
           OPENAPI_TITLE, txPeriod, result);
  }
  return;
}

void
dot1xPortMaxReAuthReqIdentitySet(openapiClientHandle_t *clientHandle,
                                 int argc, char **argv)
{
  open_error_t result;
  uint32_t intIfNum;
  uint32_t maxReAuthReqIndentity;

  if (4 != argc)
  {
    printf("Incorrect args passed.\n");
    return;
  }

  errno = 0;
  intIfNum = strtoul(argv[2], NULL, 0);
  if (0 != errno)
  {
    printf("Invalid interface parameter: (%s).\n", argv[2]);
    return;
  }

  errno = 0;
  maxReAuthReqIndentity = strtoul(argv[3], NULL, 0);
  if (0 != errno)
  {
    printf("Invalid maximum request identity count value: (%s).\n", argv[3]);
    return;
  }

  if (OPEN_E_NONE != 
      (result = openapiDot1xPortMaxReAuthReqIdentitySet(clientHandle, intIfNum,
                                                        maxReAuthReqIndentity)))
  {
    printf("Bad return code trying to set the dot1x max reauth request-identity"
        " retries count on the specified port. (result = %d)\n", result);
  }
  else
  {
    printf("Max reauth request-identity retries count for interface:%d "
           "is successfully set to %d.\n",
           intIfNum, maxReAuthReqIndentity);
  }
  return;
}

void dot1xPortMaxReAuthReqIdentityGet(openapiClientHandle_t *clientHandle,
                                      int argc, char **argv)
{
  open_error_t result;
  uint32_t intIfNum;
  uint32_t maxReAuthReqIndentity;

  if (3 != argc)
  {
    printf("Incorrect args passed.\n");
    return;
  }

  errno = 0;
  intIfNum = strtoul(argv[2], NULL, 0);
  if (0 != errno)
  {
    printf("Invalid interface parameter: (%s).\n", argv[2]);
    return;
  }

  if (OPEN_E_NONE != 
      (result = openapiDot1xPortMaxReAuthReqIdentityGet(clientHandle, intIfNum,
                                                        &maxReAuthReqIndentity)))
  {
    printf("Bad return code trying to get the configured dot1x max reauth "
           "request-identity retries count on the specified port. (result = %d)\n",
           result);
  }
  else
  {
    printf("Max reauth request-identity retries count for interface:%d is %d.\n",
           intIfNum, maxReAuthReqIndentity);
  }
  return;
}

void dot1xPortTxPeriodSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  uint32_t txPeriod;

  if (argc < 4)
  {
    printf("Interface and txPeriod parameters required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  errno = 0;
  txPeriod = strtoul(argv[3], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid txPeriod parameter: (%s)\n", argv[3]);
    return;
  }

  result = openapiDot1xPortTxPeriodSet(clientHandle, ifNum, txPeriod);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s port txPeriod (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s port txPeriod set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

functionTableEntry_t functionTable[] =
{
  {systemAuthenticationControlModeGet, "Get "OPENAPI_TITLE" configured global operating control mode.", ""},
  {systemAuthenticationControlModeSet, "Set "OPENAPI_TITLE" configured global operating control mode.", "<mode>"},
  {defaultAuthenticationMethodGet, "Get "OPENAPI_TITLE" default authentication method.", ""},
  {defaultAuthenticationMethodSet, "Set "OPENAPI_TITLE" default authentication method.", "<method>"},
  {authPortPaeStateGet,            "Get "OPENAPI_TITLE" port PAE state.", "<interface>"},
  {dot1xEapolFloodModeSet,         "Set "OPENAPI_TITLE" eapol flood mode.", "<mode>"},
  {dot1xEapolFloodModeGet,         "Get "OPENAPI_TITLE" eapol flood mode.", ""},
  {dot1xPortQuietPeriodSet,        "Set "OPENAPI_TITLE" quiet period on the specified port.", "<interface>" "<quietPeriod>"},
  {dot1xPortQuietPeriodGet,        "Get "OPENAPI_TITLE" quiet period on the specified port.", "<interface>"},
  {dot1xPortTxPeriodSet,           "Set "OPENAPI_TITLE" transmit period on the specified port.", "<interface>" "<txPeriod>"},
  {dot1xPortTxPeriodGet,           "Get "OPENAPI_TITLE" transmit period on the specified port.", "<interface>"},
  {dot1xPortMaxReAuthReqIdentitySet,   "Set "OPENAPI_TITLE" max reauth request-identity retries count on the specified port.", "<interface>" "<maxReAuthReqIndentity>"},
  {dot1xPortMaxReAuthReqIdentityGet,   "Get "OPENAPI_TITLE" max reauth request-identity retries count on the specified port.", "<interface>"},
};
#define NUMBER_OF_FUNCTIONS sizeof(functionTable)/sizeof(functionTable[0])

void printAppMenu()
{
  int i;

  printf("\nUsage: %s <test#> <arg1> <arg2> ... \n\n", EXAMPLE_APP_NAME);
  for (i = 0; i < NUMBER_OF_FUNCTIONS; i++)
  {
    printf("\tTest %d: %s:  %s %d %s\n",
           i, functionTable[i].description, EXAMPLE_APP_NAME, i, functionTable[i].parmSyntax);
  }
  printf("\tTest %d: Run API sanity checks: %s %d \n",
         NUMBER_OF_FUNCTIONS, EXAMPLE_APP_NAME, NUMBER_OF_FUNCTIONS);
  printf("\tTest %d: Sets the dot1x max requests on the specified port: %s %d <intIfNum> <maxReq>\n",
         NUMBER_OF_FUNCTIONS+1, EXAMPLE_APP_NAME, NUMBER_OF_FUNCTIONS+1);
  printf("\tTest %d: Gets dot1x max requests on the specified port: %s %d <intIfNum>\n",
         NUMBER_OF_FUNCTIONS+2, EXAMPLE_APP_NAME, NUMBER_OF_FUNCTIONS+2);
  printf("\tTest %d: Sets the dot1x supplicant timeout on the specified port: %s %d <intIfNum> <suppTimeout>\n",
         NUMBER_OF_FUNCTIONS+3, EXAMPLE_APP_NAME, NUMBER_OF_FUNCTIONS+3);
  printf("\tTest %d: Gets dot1x supplicant timeout on the specified port: %s %d <intIfNum>\n",
         NUMBER_OF_FUNCTIONS+4, EXAMPLE_APP_NAME, NUMBER_OF_FUNCTIONS+4);
  printf("\tTest %d: Sets the dot1x Authentiation server timeout on the specified port: %s %d <intIfNum> <serverTimeout>\n",
         NUMBER_OF_FUNCTIONS+5, EXAMPLE_APP_NAME, NUMBER_OF_FUNCTIONS+5);
  printf("\tTest %d: Gets dot1x Authentiation Server timeout on the specified port: %s %d <intIfNum>\n",
         NUMBER_OF_FUNCTIONS+6, EXAMPLE_APP_NAME, NUMBER_OF_FUNCTIONS+6);
  printf("\tTest %d: Sets the port's dot1x capabilities (Supplicant or Authenticator): %s %d <intIfNum> <paeCapability>\n",
         NUMBER_OF_FUNCTIONS+7, EXAMPLE_APP_NAME, NUMBER_OF_FUNCTIONS+7);
  printf("\tTest %d: Gets dot1x PAE capabilities on the specified port: %s %d <intIfNum>\n",
         NUMBER_OF_FUNCTIONS+8, EXAMPLE_APP_NAME, NUMBER_OF_FUNCTIONS+8);
  printf("\tTest %d: Gets dot1x protocol version on the specified port: %s %d <intIfNum>\n",
         NUMBER_OF_FUNCTIONS+9, EXAMPLE_APP_NAME, NUMBER_OF_FUNCTIONS+9);
 
  printf("\n");

  return;
}
/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t maxReq;
  OPEN_DOT1X_PAE_PORT_PROTOCOL_VERSION_t protocolVersion;
  OPEN_DOT1X_PAE_PORT_CAPABILITY paeCapability;
  OPEN_AUTHMGR_PORT_DIRECTION_t cntrlDir;
  uint32_t suppTimeout;
  uint32_t serverTimeout;
  uint32_t maxReAuthReqIndentity = 10;
  uint32_t intIfNum = 10;

  printf("Testing  OpEN APIs sanity:\n");

  printf("Testing openapiDot1xPortMaxReAuthReqIdentitySet():\n");
  result = openapiDot1xPortMaxReAuthReqIdentitySet(NULL, intIfNum, maxReAuthReqIndentity);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1xPortMaxReAuthReqIdentitySet(clientHandle, 0, maxReAuthReqIndentity);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiDot1xPortMaxReAuthReqIdentitySet(clientHandle, intIfNum, 0);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiDot1xPortMaxReAuthReqIdentitySet(clientHandle, intIfNum, 21);
  printf("greater than maximum value argument 3:(result = %d)\n", result);
  printf("Testing openapiDot1xPortMaxReAuthReqIdentityGet():\n");
  result = openapiDot1xPortMaxReAuthReqIdentityGet(NULL, intIfNum, &maxReAuthReqIndentity);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1xPortMaxReAuthReqIdentityGet(clientHandle, 0, &maxReAuthReqIndentity);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiDot1xPortMaxReAuthReqIdentityGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  printf("Testing openapiDot1xPortMaxReqGet():\n");
  result = openapiDot1xPortMaxReqGet(NULL, intIfNum, &maxReq);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1xPortMaxReqGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1xPortSuppTimeoutGet():\n");
  result = openapiDot1xPortSuppTimeoutGet(NULL, intIfNum, &suppTimeout);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1xPortSuppTimeoutGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1xPortServerTimeoutGet():\n");
  result = openapiDot1xPortServerTimeoutGet(NULL, intIfNum, &serverTimeout);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1xPortServerTimeoutGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1xPortPaeCapabilitiesGet():\n");
  result = openapiDot1xPortPaeCapabilitiesGet(NULL, intIfNum, &paeCapability);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1xPortPaeCapabilitiesGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1xPortProtocolVersionGet():\n");
  result = openapiDot1xPortProtocolVersionGet(NULL, intIfNum, &protocolVersion);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1xPortProtocolVersionGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiAuthMgrPortAdminControlledDirectionsGet():\n");
  result = openapiAuthMgrPortAdminControlledDirectionsGet(NULL, intIfNum, &cntrlDir);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrPortAdminControlledDirectionsGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  return;
}


/*********************************************************************
* @purpose  Sets the dot1x max requests on the specified port.
*
* @param    clientHandle  @b{(input)}  client handle from registration API.
* @param    intIfNum       @b{(input)}  Internal interface number.
* @param    maxReq         @b{(input)}  Max requests value.

*
* @returns  none
*
* @end
*********************************************************************/
void
dot1xPortMaxReqSet(openapiClientHandle_t *clientHandle, uint32_t intIfNum,
                   uint32_t maxReq)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiDot1xPortMaxReqSet(clientHandle, intIfNum, maxReq)))
  {
    printf("Bad return code trying to sets the dot1x max requests on the "
        "specified port. (result = %d)\n", result);
  }
  else
  {
    printf("Dot1x max requests value of %d is successfully set on interface %d.\n",
           maxReq, intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Gets dot1x max requests on the specified port.
*
* @param    clientHandle  @b{(input)}  client handle from registration API.
* @param    intIfNum       @b{(input)}  Internal interface number.
* @param    pMaxReq        @b{(output)} Max requests value.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1xPortMaxReqGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t *pMaxReq)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiDot1xPortMaxReqGet(clientHandle, intIfNum, pMaxReq)))
  {
    printf("Bad return code trying to gets dot1x max requests on the "
        "specified port. (result = %d)\n", result);
  }
  else
  {
    printf("Dot1x max requests value of %d is successfully fetched on interface %d.\n",
           *pMaxReq, intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Sets the dot1x supplicant timeout on the specified port.
*
* @param    clientHandle  @b{(input)}  client handle from registration API.
* @param    intIfNum       @b{(input)}  Internal interface number.
* @param    suppTimeout    @b{(input)}  Supplicant timeout value.

*
* @returns  none
*
* @end
*********************************************************************/
void dot1xPortSuppTimeoutSet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t suppTimeout)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiDot1xPortSuppTimeoutSet(clientHandle, intIfNum, suppTimeout)))
  {
    printf("Bad return code trying to sets the dot1x supplicant timeout on "
        "the specified port. (result = %d)\n", result);
  }
  else
  {
    printf("Dot1x supplicant timeout value of %d is successfully set on interface %d.\n",
           suppTimeout, intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Gets dot1x supplicant timeout on the specified port.
*
* @param    clientHandle  @b{(input)}  client handle from registration API.
* @param    intIfNum       @b{(input)}  Internal interface number.
* @param    pSuppTimeout   @b{(output)} Supplicant timeout value.

*
* @returns  none
*
* @end
*********************************************************************/
void
dot1xPortSuppTimeoutGet(openapiClientHandle_t *clientHandle,
                        uint32_t intIfNum, uint32_t *pSuppTimeout)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiDot1xPortSuppTimeoutGet(clientHandle, intIfNum, pSuppTimeout)))
  {
    printf("Bad return code trying to gets dot1x supplicant timeout on the "
        "specified port. (result = %d)\n", result);
  }
  else
  {
    printf("Dot1x supplicant timeout value of %d is successfully fetched on interface %d.\n",
           *pSuppTimeout, intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Sets the dot1x Authentiation server timeout on the specified port.
*
* @param    clientHandle  @b{(input)}  client handle from registration API.
* @param    intIfNum       @b{(input)}  Internal interface number.
* @param    serverTimeout  @b{(input)}  Authentication Server timeout value.

*
* @returns  none
*
* @end
*********************************************************************/
void
dot1xPortServerTimeoutSet(openapiClientHandle_t *clientHandle,
                          uint32_t intIfNum, uint32_t serverTimeout)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiDot1xPortServerTimeoutSet(clientHandle, intIfNum, serverTimeout)))
  {
    printf("Bad return code trying to sets the dot1x Authentiation server "
        "timeout on the specified port. (result = %d)\n", result);
  }
  else
  {
    printf("Dot1x authentiation server timeout value of %d is successfully set on interface %d.\n",
           serverTimeout, intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Gets dot1x Authentiation Server timeout on the specified port.
*
* @param    clientHandle  @b{(input)}  client handle from registration API.
* @param    intIfNum       @b{(input)}  Internal interface number.
* @param    pServerTimeout @b{(output)} Authentication Server timeout value.

*
* @returns  none
*
* @end
*********************************************************************/
void
dot1xPortServerTimeoutGet(openapiClientHandle_t *clientHandle,
                          uint32_t intIfNum, uint32_t *pServerTimeout)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiDot1xPortServerTimeoutGet(clientHandle, intIfNum, pServerTimeout)))
  {
    printf("Bad return code trying to gets dot1x Authentiation Server timeout "
        "on the specified port. (result = %d)\n", result);
  }
  else
  {
    printf("Dot1x authentiation Server timeout value of %d is successfully fetched on interface %d.\n",
           *pServerTimeout, intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Sets the port's dot1x capabilities (Supplicant or Authenticator).
*
* @param    clientHandle  @b{(input)}  client handle from registration API.
* @param    intIfNum       @b{(input)}  Internal interface number.
* @param    paeCapability  @b{(input)}  Dot1x capabilities.

*
* @returns  none
*
* @end
*********************************************************************/
void 
dot1xPortPaeCapabilitiesSet(openapiClientHandle_t *clientHandle, uint32_t intIfNum,
                            OPEN_DOT1X_PAE_PORT_CAPABILITY paeCapability)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiDot1xPortPaeCapabilitiesSet(clientHandle, intIfNum, paeCapability)))
  {
    printf("Bad return code trying to sets the port's dot1x capabilities "
        "(Supplicant or Authenticator). (result = %d)\n", result);
  }
  else
  {
    printf("Dot1x port's dot1x capability value of %d is successfully set on interface %d.\n",
           paeCapability, intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Gets dot1x PAE capabilities on the specified port.
*
* @param    clientHandle       @b{(input)}  client handle from registration API.
* @param    intIfNum            @b{(input)}  Internal interface number.
* @param    pPaeCapability      @b{(output)} pae capabilities value.

*
* @returns  none
*
* @end
*********************************************************************/
void
dot1xPortPaeCapabilitiesGet(openapiClientHandle_t *clientHandle,
                            uint32_t intIfNum,
                            OPEN_DOT1X_PAE_PORT_CAPABILITY *pPaeCapability)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiDot1xPortPaeCapabilitiesGet(clientHandle, intIfNum, pPaeCapability)))
  {
    printf("Bad return code trying to gets dot1x PAE capabilities on the "
        "specified port. (result = %d)\n", result);
  }
  else
  {
    printf("Dot1x PAE capability value of %d is successfully fetched on interface %d.\n",
           *pPaeCapability, intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Gets dot1x protocol version on the specified port.
*
* @param    clientHandle       @b{(input)}  client handle from registration API.
* @param    intIfNum            @b{(input)}  Internal interface number.
* @param    pProtocolVersion    @b{(output)}   dot1x protocol version of specified port.

*
* @returns  none
*
* @end
*********************************************************************/
void
dot1xPortProtocolVersionGet(openapiClientHandle_t *clientHandle,
                            uint32_t intIfNum,
                            OPEN_DOT1X_PAE_PORT_PROTOCOL_VERSION_t *pProtocolVersion)
{
  open_error_t result;

  if (OPEN_E_NONE !=
      (result = openapiDot1xPortProtocolVersionGet(clientHandle, intIfNum, pProtocolVersion)))
  {
    printf("Bad return code trying to gets dot1x protocol version on the "
        "specified port. (result = %d)\n", result);
  }
  else
  {
    printf("Dot1x protocol version of %d is successfully fetched on interface %d.\n",
           *pProtocolVersion, intIfNum);
  }
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc buffDesc;
  char switch_os_revision_string[100];
  open_revision_data_t openApiVersion;
  int  show_help = 1;
  OPEN_DOT1X_PAE_PORT_CAPABILITY paeCapability;
  OPEN_DOT1X_PAE_PORT_PROTOCOL_VERSION_t protocolVersion;
  uint32_t suppTimeout;
  uint32_t serverTimeout;
  uint32_t maxReq;
  uint32_t intIfNum;


  if (argc < 2)
  {
    printAppMenu();
    exit(1);
  }

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister(""OPENAPI_TITLE" OpEN API example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting %s OpEN API example application", OPENAPI_TITLE);

  printf("\n");
  buffDesc.pstart = switch_os_revision_string;
  buffDesc.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &buffDesc) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  if (openapiApiVersionGet(&clientHandle, &openApiVersion) == OPEN_E_NONE)
    printf("OpEN API Version = %d.%d.%d.%d\n",
           openApiVersion.release,
           openApiVersion.version,
           openApiVersion.maint_level,
           openApiVersion.build_num);
  else
    printf("OpEN API Version retrieve error\n");

  printf("\n");

  errno = 0;
  testNum = strtol(argv[1], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid test number: (%s)\n", argv[1]);
    printAppMenu();
    exit(1);
  }

  if (testNum < NUMBER_OF_FUNCTIONS)
  {
    functionTable[testNum].function(&clientHandle, argc, argv);
    show_help = 0;
  }
  else
  {
    switch (testNum)
    {
      case NUMBER_OF_FUNCTIONS:
        if (argc == 2)
        {
          runSanity(&clientHandle);
          show_help = 0;
        }
        break;
      case NUMBER_OF_FUNCTIONS+1:
        if (argc == 4)
        {
          intIfNum = atoi(argv[2]);
          maxReq = atoi(argv[3]);
          dot1xPortMaxReqSet(&clientHandle, intIfNum, maxReq);
          show_help = 0;
        }
        break;
      case NUMBER_OF_FUNCTIONS+2:
        if (argc == 3)
        {
          intIfNum = atoi(argv[2]);
          dot1xPortMaxReqGet(&clientHandle, intIfNum, &maxReq);
          show_help = 0;
        }
        break;
      case NUMBER_OF_FUNCTIONS+3:
        if (argc == 4)
        {
          intIfNum = atoi(argv[2]);
          suppTimeout = atoi(argv[3]);
          dot1xPortSuppTimeoutSet(&clientHandle, intIfNum, suppTimeout);
          show_help = 0;
        }
        break;
      case NUMBER_OF_FUNCTIONS+4:
        if (argc == 3)
        {
          intIfNum = atoi(argv[2]);
          dot1xPortSuppTimeoutGet(&clientHandle, intIfNum, &suppTimeout);
          show_help = 0;
        }
        break;
      case NUMBER_OF_FUNCTIONS+5:
        if (argc == 4)
        {
          intIfNum = atoi(argv[2]);
          serverTimeout = atoi(argv[3]);
          dot1xPortServerTimeoutSet(&clientHandle, intIfNum, serverTimeout);
          show_help = 0;
        }
        break;
      case NUMBER_OF_FUNCTIONS+6:
        if (argc == 3)
        {
          intIfNum = atoi(argv[2]);
          dot1xPortServerTimeoutGet(&clientHandle, intIfNum, &serverTimeout);
          show_help = 0;
        }
        break;
      case NUMBER_OF_FUNCTIONS+7:
        if (argc == 4)
        {
          intIfNum = atoi(argv[2]);
          paeCapability = atoi(argv[3]);
          dot1xPortPaeCapabilitiesSet(&clientHandle, intIfNum, paeCapability);
          show_help = 0;
        }
        break;
      case NUMBER_OF_FUNCTIONS+8:
        if (argc == 3)
        {
          intIfNum = atoi(argv[2]);
          dot1xPortPaeCapabilitiesGet(&clientHandle, intIfNum, &paeCapability);
          show_help = 0;
        }
        break;
      case NUMBER_OF_FUNCTIONS+9:
        if (argc == 3)
        {
          intIfNum = atoi(argv[2]);
          dot1xPortProtocolVersionGet(&clientHandle, intIfNum, &protocolVersion);
          show_help = 0;
        }
        break;
      default:
        break;
    }
  }

  if (show_help == 1)
  {
    printAppMenu();
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping %s OpEN API example application", OPENAPI_TITLE);
        
  return 0;
}
