/*********************************************************************
*
*  Copyright 2024 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  event_notify_example.c
*
* @purpose   OpEN event_notify example.
*
* @component OpEN
*
* @create    04/11/2024
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>
#include <pthread.h>
#include <string.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_event_notify.h"

char clientDesc[64] = {'\0'};

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the event_notify Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <clentNum> <numOfNotifyEvents> <eventType1> <timeout1> <eventCount1> <eventType2> <timeout2> <eventCount2> ... &\n", name);
  printf("Usage: Only 1 or 2 can be used for clientNum.\n");
  printf("Usage: Mininum of one event and its data should be specified.\n");
  printf("Usage: It is preferrable to run this example in background using '&'\n");
  printf("Test 1: Run a sample code to simulate a event notification client : %s <clientNum> <numOfNotifyEvents> <eventType1> <timeout1> <eventCount1> &\n", name);
  return;
}

/*********************************************************************
* @purpose  Callback function to process event notification data received.
*
* @param    data       @b{(input)} event notification data

*
* @returns  none
*
* @end
*********************************************************************/
open_error_t dataReceive (OPEN_DEVICE_EVENT_t *data)
{
  int i = 0, j = 0;
  OPEN_FDB_EVENT_t *fdbEvent;
  OPEN_PORT_INFO_EVENT_t *portInfo;
  OPEN_LLDP_NEIGHBOR_EVENT_t *lldpNeighbor;
  OPEN_POE_PORT_STATUS_EVENT_t *poePortStatus;
  OPEN_ARP_EVENT_t             *arpEvent;
  OPEN_IP_INFO_EVENT_t         *ipInfo;
  OPEN_IPV4_ROUTE_EVENT_t      *ipv4Route;
  uint32_t ip;
  char ipAddrStr[24] = {0};
  unsigned char *mac;

  printf ("\n \"%s\" received the below event data:\n", clientDesc);
  printf ("\nEvent sequence  - %u", data->eventSeqNum);
  printf ("\nGlobal sequence - %u", data->globalSeqNum);
  printf ("\nEvent Type      - %u", data->eventType);
  printf ("\nEvent Count     - %u", data->eventCount);
  printf ("\nEvent Miss Flag - %u", data->eventMissFlag);
  for (i=0; i < data->eventCount; i++)
  {
    printf ("\r\n");
    switch (data->eventType)
    {
      case OPEN_NOTIFY_EVENT_IP_INFO:
           ipInfo = (OPEN_IP_INFO_EVENT_t *) &(data->eventData.pstart) + i;
           if (OPEN_MGMT_IPPORT != ipInfo->mgmtPortType)
           {
             printf ("IP info change on %s\n",
                     ((OPEN_MGMT_SERVICEPORT == ipInfo->mgmtPortType)? "Service port":"Network port"));
           }
           else
           {
             printf ("IP info change on IP port interface-%d \n", ipInfo->intIfNum);
           }
           printf ("Change type %u\n", ipInfo->eventChangeType);
           ip = htonl (ipInfo->ipAddr);
           if (0 == inet_ntop (AF_INET, (void*)&ip, ipAddrStr, sizeof(ipAddrStr)))
           {
             printf ("Bad return code trying to convert IP.\n");
           }
           else
           {
             printf ("IP Address : %s\n", ipAddrStr);
           }
           ip = htonl (ipInfo->netMask);
           if (0 == inet_ntop (AF_INET, (void*)&ip, ipAddrStr, sizeof(ipAddrStr)))
           {
             printf ("Bad return code trying to convert IP.\n");
           }
           else
           {
             printf ("Network mask : %s\n", ipAddrStr);
           }
           ip = htonl (ipInfo->gateway);
           if (0 == inet_ntop (AF_INET, (void*)&ip, ipAddrStr, sizeof(ipAddrStr)))
           {
             printf ("Bad return code trying to convert IP.\n");
           }
           else
           {
             printf ("Gateway : %s\n", ipAddrStr);
           }
           printf ("DNS servers : %s\n", ipAddrStr);
           for (j = 0;
                ((j < OPEN_DNS_NAME_SERVER_ENTRIES) &&
                 (0 != ipInfo->dnsServerIpAddr[j])); j++)
           {
             ip = htonl (ipInfo->dnsServerIpAddr[j]);
             if (0 == inet_ntop (AF_INET, (void*)&ip, ipAddrStr, sizeof(ipAddrStr)))
             {
               printf ("Bad return code trying to convert IP.\n");
             }
             else
             {
               printf ("    %s\n", ipAddrStr);
             }
           }
           printf ("Hostname : %s\n", ipInfo->hostName);
           break;
      case OPEN_NOTIFY_EVENT_ARP:
           arpEvent = (OPEN_ARP_EVENT_t *) &(data->eventData.pstart);
           printf ("ChangeType %u \r\n",(arpEvent+i)->eventChangeType);
           printf ("Arp entry Type, Age :");
           if (((arpEvent+i)->arpEntry.type & OPEN_ARP_LOCAL) != 0)
           {
             printf("%s, n/a\n", "Local");
           }
           else if (((arpEvent+i)->arpEntry.type & OPEN_ARP_GATEWAY) != 0)
           {
             printf("%s, %d\n", "Gateway", (arpEvent+i)->arpEntry.age);
           }
           else if (((arpEvent+i)->arpEntry.type & OPEN_ARP_STATIC) != 0)
           {
             printf("%s, n/a\n", "Static");
           }
           else if (((arpEvent+i)->arpEntry.type & OPEN_ARP_KERNEL) != 0)
           {
             printf("%s, n/a\n", "Kernel");
           }
           else
           {
             printf("%s, %d\n", "Dynamic", (arpEvent+i)->arpEntry.age);
           }
           ip = htonl ((arpEvent+i)->arpEntry.ipAddr.addr.ipv4);
           if (inet_ntop(AF_INET, (void*)&ip, ipAddrStr, sizeof(ipAddrStr)) == 0)
           {
             printf("Bad return code trying to convert IP.\n");
           }
           printf ("IP Address : %s \n",ipAddrStr);
           printf ("VRF ID %u\n",(arpEvent+i)->arpEntry.vrfId);
           mac = (arpEvent+i)->arpEntry.macAddr;
           printf ("MAC %02X:%02X:%02X:%02X:%02X:%02X \r\n",
                   mac[0],mac[1],mac[2],mac[3],mac[4],mac[5]);
           printf ("VlanId %u\n", (arpEvent+i)->arpEntry.vlanId);
           printf ("IfNum %u\n", (arpEvent+i)->arpEntry.intIfNum);
           printf ("Hits %u\n", (arpEvent+i)->arpEntry.hits);
           break;
      case OPEN_NOTIFY_EVENT_IPV4_ROUTE:
           ipv4Route = (OPEN_IPV4_ROUTE_EVENT_t *) &(data->eventData.pstart);
           printf ("ChangeType %u \r\n",(ipv4Route+i)->eventChangeType);
           printf ("VRF ID %u\n",(ipv4Route+i)->vrfId);

           ip = htonl ((ipv4Route+i)->routeDest);
           if (inet_ntop(AF_INET, (void*)&ip, ipAddrStr, sizeof(ipAddrStr)) == 0)
           {
             printf("Bad return code trying to convert IP.\n");
           }
           printf ("Route destination : %s \n",ipAddrStr);
 
           ip = htonl ((ipv4Route+i)->routeMask);
           if (inet_ntop(AF_INET, (void*)&ip, ipAddrStr, sizeof(ipAddrStr)) == 0)
           {
             printf("Bad return code trying to convert IP.\n");
           }
           printf ("Route mask : %s \n",ipAddrStr);
 
           printf ("Route protocol %u\n", (ipv4Route+i)->routeProto);
           printf ("Route preference %u\n", (ipv4Route+i)->routePref);
           break;
      case OPEN_NOTIFY_EVENT_PORT_INFO:
           portInfo = (OPEN_PORT_INFO_EVENT_t *) &(data->eventData.pstart);
           printf ("IfNum %u \r\n",(portInfo+i)->ifNum);
           printf ("PortState %u \r\n",(portInfo+i)->portState);
           printf ("LinkState %u \r\n",(portInfo+i)->linkState);
           printf ("FlowControl %u \r\n",(portInfo+i)->flowControl);
           printf ("DisableReason %u \r\n",(portInfo+i)->disableReason);
           printf ("IfSpeed %u \r\n",(portInfo+i)->ifSpeed);
           printf ("Autoneg %u \r\n",(portInfo+i)->autoneg);
           printf ("Duplex %u \r\n",(portInfo+i)->duplex);
           printf ("MdixMode %u \r\n",(portInfo+i)->mdixMode);
           printf ("Active medium %u \r\n",(portInfo+i)->medium);
           break;
      case OPEN_NOTIFY_EVENT_LLDP_NEIGHBORS:
           lldpNeighbor = (OPEN_LLDP_NEIGHBOR_EVENT_t *) &(data->eventData.pstart);
           printf ("ChangeType %u \r\n",(lldpNeighbor+i)->eventChangeType);
           printf ("IfNum %u \r\n",(lldpNeighbor+i)->ifNum);
           printf ("RemIndex %u \r\n",(lldpNeighbor+i)->remIndex);
           printf ("Timestamp %u \r\n",(lldpNeighbor+i)->timestamp);
           printf ("SrcMac %02X:%02X:%02X:%02X:%02X:%02X \r\n",(lldpNeighbor+i)->srcMac[0],
                   (lldpNeighbor+i)->srcMac[1],(lldpNeighbor+i)->srcMac[2],(lldpNeighbor+i)->srcMac[3],
                   (lldpNeighbor+i)->srcMac[4],(lldpNeighbor+i)->srcMac[5]);
           break;
      case OPEN_NOTIFY_EVENT_POE_STATUS:
           poePortStatus = (OPEN_POE_PORT_STATUS_EVENT_t *) &(data->eventData.pstart);
           printf ("IfNum %u \r\n",(poePortStatus+i)->ifNum);
           printf ("State %u \r\n",(poePortStatus+i)->state);
           printf ("DetectionStatus %u \r\n",(poePortStatus+i)->detectionStatus);
           printf ("Status %u \r\n",(poePortStatus+i)->status);
           printf ("PowerLimit %u \r\n",(poePortStatus+i)->powerLimit);
           printf ("MaxPower %u \r\n",(poePortStatus+i)->maxPower);
           printf ("PdClass %u \r\n",(poePortStatus+i)->pdClass);
           printf ("PdType %u \r\n",(poePortStatus+i)->pdType);
           break;
      case OPEN_NOTIFY_EVENT_FDB_ENTRY:
           fdbEvent = (OPEN_FDB_EVENT_t *) &(data->eventData.pstart);
           printf ("ChangeType %u \r\n",(fdbEvent+i)->eventChangeType);
           printf ("IfNum %u \r\n",(fdbEvent+i)->ifNum);
           printf ("VlanId %u \r\n",(fdbEvent+i)->vlanId);
           printf ("FDB event entry type %u \r\n",(fdbEvent+i)->entryType);
           printf ("Mac %02X:%02X:%02X:%02X:%02X:%02X \r\n",(fdbEvent+i)->mac[0],
                   (fdbEvent+i)->mac[1],(fdbEvent+i)->mac[2],(fdbEvent+i)->mac[3],
                   (fdbEvent+i)->mac[4],(fdbEvent+i)->mac[5]);
           break;
      default:
           printf ("Invalid event type %u.", data->eventType);
           return OPEN_E_NONE;
    }
    printf ("\r\n");
  }
  fflush(stdout);
  return OPEN_E_NONE;
}


/*******************************************************************
*
* @brief  This is the function that gets execunted when the thread
*          is invoded by the main function. This thread initializes
*          the socket and receives data.
*
* @returns none
*
*********************************************************************/
void *dataThread (void *p)
{
  openapiClientHandle_t *pClientHandle = (openapiClientHandle_t *)p;

  /* Receive notification data */
  if (OPEN_E_NONE != openapiEventNotifyDataReceive (pClientHandle, dataReceive))
  {
    printf ("\nFailed to receive event notifications\n");
    pthread_exit (NULL);
  }
  return NULL;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for event_notify
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  open_buffdesc clientDescBuf;
  openEventNotifyConfig_t eventNotifyCfg[OPEN_NOTIFY_EVENT_LAST];
  unsigned int numOfNotifyEvents = 0;
  open_buffdesc buf;
  int clientNum = 0;
  int argIndex = 0, i = 0;
  pthread_t tid;
  int errNum = 0;

  if (argc < 6)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  l7proc_crashlog_register();

  clientNum = atoi (argv[1]);

  if ((1 != clientNum) &&
      (2 != clientNum))
  {
    printf ("\nInvalid client number. Client number should be 1 or 2.\n");
    printAppMenu(argv[0]);
    return -1;
  }

  numOfNotifyEvents = atoi (argv[2]);

  if ((OPEN_NOTIFY_EVENT_IP_INFO > numOfNotifyEvents) ||
      (OPEN_NOTIFY_EVENT_LAST    < numOfNotifyEvents))
  {
    printf ("\nInvalid number of events. Supported range is 1-7.\n");
    printAppMenu(argv[0]);
    return -1;
  }

  if (((3*(numOfNotifyEvents)) + 3) != argc)
  {
    printf ("\nMissing some arguments.\n");
    printAppMenu(argv[0]);
    return -1;
  }

  argIndex = 3;
  for (i = 0; i < numOfNotifyEvents; i++)
  {
    eventNotifyCfg[i].eventType  = atoi (argv[argIndex]);
    eventNotifyCfg[i].timeout    = atoi (argv[argIndex + 1]);
    eventNotifyCfg[i].eventCount = atoi (argv[argIndex + 2]);
    argIndex += 3;
  }

  snprintf (clientDesc, sizeof (clientDesc),
            "event_notify_example_%d", clientNum);

  /* Register with OpEN */
  if ((result = openapiClientRegister (clientDesc, &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck (&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting event_notify API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  clientDescBuf.pstart = clientDesc;
  clientDescBuf.size = strlen (clientDesc);

  /* Register as client to receive event notifications */
  result = openapiEventNotifyClientRegister (&client_handle, &clientDescBuf);

  if (OPEN_E_NONE != result)
  {
    /* Log goodbye message with OpEN */
    printf ("\nFailed to register for event notifications\n");
    (void) openapiClientTearDown(&client_handle);
    return 0;
  }

  /* Create a thread to initialize socket and process data received */
  errNum = pthread_create (&tid, NULL, dataThread, (void *)&client_handle);
  if (0 != errNum)
  {
    printf ("Unable to create a thread for handling data with reason %s\n",
            strerror (errNum));
    (void) openapiEventNotifyEventDeregister (&client_handle);
    (void) openapiEventNotifyClientDeregister (&client_handle);
    (void) openapiClientTearDown (&client_handle);
    return 0;
  }

  buf.pstart = eventNotifyCfg;
  buf.size = numOfNotifyEvents * sizeof(openEventNotifyConfig_t);

  /* Register for required events */
  result = openapiEventNotifyEventRegister (&client_handle,
                                            numOfNotifyEvents, &buf);
  if (OPEN_E_NONE != result)
  {
    /* Log goodbye message with OpEN */
    printf ("\nFailed to register for events with client.\n");
    (void) openapiEventNotifyClientDeregister (&client_handle);
    (void) openapiClientTearDown (&client_handle);
    return 0;
  }

  errNum = pthread_join (tid, NULL);
  if (0 != errNum)
  {
    printf ("Unable to wait on thread with reason %s\n", strerror (errNum));
  }

  (void) openapiEventNotifyEventDeregister (&client_handle);
  (void) openapiEventNotifyClientDeregister (&client_handle);
  (void) openapiClientTearDown (&client_handle);
  return 0;
} 
