
/*********************************************************************
*
*  Copyright 2016-2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  config_status_example.c
*
* @purpose   Configuration Status Example
*
* @component OPEN
*
* @comments
*
* @create    12/02/2016
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>
#include <time.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_config.h"

uint32_t configChangeTracker;
uint32_t configSaveTracker;
uint32_t configClearTracker;

void config_event_monitor(openapiClientHandle_t *client,
                          uint32_t monitor_duration)
{
  struct timespec start_time, current_time;
  openapiEventList_t change_pending_event, purge_event;
  uint32_t newChangeTracker, newSaveTracker, newClearTracker;

  if (clock_gettime(CLOCK_MONOTONIC, &start_time))
  {
    perror("Can't get start time.\n");
    abort();
  }
  do
  {
    /* Wait for events. Time out every 5 seconds.
    */
    if (OPEN_E_NONE == openapiEventWait(client, 5))
    {
      if (OPEN_E_NONE == openapiPendingEventsGet(client, &change_pending_event, &purge_event))
      {
        /* Sometimes FASTPATH deletes entries from the table without
        ** telling the application. This can happen if the table gets full
        ** before the application has a chance to acknowledge
        ** the delete-pending events.
        ** If an applicaton gets a purge notification then it must
        ** check to see if related data is current.
        */
        if (openapiEventIsSet(&purge_event, OPEN_EVENT_CONFIG))
        {
          printf("Got Purge event for OPEN_EVENT_CONFIG\n");
          /* retrieve the current tracking values */
          openapiConfigChangeTrackerGet(client, &configChangeTracker);
          openapiConfigSaveTrackerGet(client, &configSaveTracker);
          openapiConfigClearTrackerGet(client, &configClearTracker);
        }
        else if (openapiEventIsSet(&change_pending_event, OPEN_EVENT_CONFIG))
        {
          printf("Got Change Pending event for OPEN_EVENT_CONFIG\n");

          /* retrieve and process the current tracking values */
          openapiConfigChangeTrackerGet(client, &newChangeTracker);
          if (newChangeTracker != configChangeTracker)
          {
            printf("\tRunning configuration has changed.\n");
            configChangeTracker = newChangeTracker;
          }

          openapiConfigSaveTrackerGet(client, &newSaveTracker);
          if (newSaveTracker != configSaveTracker)
          {
            printf("\tRunning configuration has been saved.\n");
            configSaveTracker = newSaveTracker;
          }

          openapiConfigClearTrackerGet(client, &newClearTracker);
          if (newClearTracker != configClearTracker)
          {
            printf("\tRunning configuration has been cleared.\n");
            configClearTracker = newClearTracker;
          }
        }
      }
    }
    if (clock_gettime(CLOCK_MONOTONIC, &current_time))
    {
      perror("Can't get current time.\n");
      abort();
    }
  } while ((current_time.tv_sec - start_time.tv_sec) < monitor_duration);
}

int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  uint32_t monitor_duration = 60;
  openapiEventList_t open_event;

  if ((argc != 1) && (argc != 2))
  {
    printf("Usage: %s [monitor-duration-seconds]\n", argv[0]);
    exit(1);
  }

  if (argc == 2)
  {
    monitor_duration = atoi(argv[1]);
  }
  printf("Monitoring Configuration Status for %d seconds.\n", monitor_duration);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("config_status_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Configuration Status Monitor example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  /* Register with FASTPATH to receive configuration events.
  ** The registration must be done before retrieving the tracker values
  ** in order not to miss any events.
  */
  openapiEventListClear(&open_event);
  openapiEventSet(&open_event, OPEN_EVENT_CONFIG);
  if (OPEN_E_NONE != openapiEventRegister(&clientHandle, &open_event))
  {
    printf("Failed to register for events.\n");
    abort();
  }

  /* Retrieve starting values of tracker variables. */
  openapiConfigChangeTrackerGet(&clientHandle, &configChangeTracker);
  openapiConfigSaveTrackerGet(&clientHandle, &configSaveTracker);

  config_event_monitor(&clientHandle, monitor_duration);

  /* Log goodbye message with OPEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Configuration Status Monitor example application");

  return 0;
}
